"""
Hospital Integration Configuration
This file contains configuration settings for hospital system integration.
"""

# Hospital API Key
# Single API key for all hospital integrations
HOSPITAL_API_KEY = "Z9xd3NikanhospitalQ8zYpR7G5wK2uHvM1R"

# Comment API Key
# API key for accessing comment system
COMMENT_API_KEY = "CommentAPIKey_ZiLuck2024_AccessToken_9pQ7mN3X"

# Hospital Information (for tracking purposes)
HOSPITAL_INFO = {
    "Z9xd3NikanhospitalQ8zYpR7G5wK2uHvM1R": {
        "name": "Nikan Hospital",
        "code": "NIKAN",
        "nikan_discount_group_code": "NIKAN_PROMOTION",
        "contact_email": "integration@nikanhospital.com",
        "enabled": True,
        "created_date": "2024-01-01",
        "max_users_per_day": 10000,
        "location": "Tehran, Iran",
        "integration_type": "payment_simulation"
    },
    "default": {
        "name": "Hospital Integration",
        "code": "HOSP",
        "contact_email": "integration@ziluck.com",
        "nikan_discount_group_code": "NIKAN_PROMOTION",
        "enabled": True,
        "created_date": "2024-01-01",
        "max_users_per_day": 10000,
        "location": "Unknown",
        "integration_type": "payment_simulation"
    }
}

# API Rate Limiting Settings
HOSPITAL_API_RATE_LIMITS = {
    "requests_per_minute": 60,
    "requests_per_hour": 1000,
    "requests_per_day": 10000
}

# Validation Settings
HOSPITAL_VALIDATION_SETTINGS = {
    "require_email": False,
    "require_national_code": False,
    "require_birth_date": False,
    "auto_assign_doctor": False,
    "default_diabetes_type": 1,  # 0: pregnancy, 1: type1, 2: type2
    "default_activity_level": 1.5,  # 1: very low, 1.25: low, 1.5: medium, 1.75: high, 2: very high
}

# Logging Settings
HOSPITAL_LOGGING = {
    "log_all_requests": True,
    "log_successful_creations": True,
    "log_failed_attempts": True,
    "log_api_key_usage": True,
    "retention_days": 90
}

def validate_hospital_api_key(api_key):
    """
    Validate if the provided API key is valid and enabled.
    Now checks the Hospital database model first, then falls back to hardcoded config.
    
    Args:
        api_key (str): The API key to validate
        
    Returns:
        dict: Hospital info if valid, None if invalid
    """
    # First, try to get from database (new method)
    try:
        from panel.models import Hospital
        hospital = Hospital.objects.filter(api_key=api_key, is_active=True).first()
        if hospital:
            # Return in the same format as HOSPITAL_INFO for compatibility
            return {
                "name": hospital.name,
                "code": hospital.hospital_code,
                "nikan_discount_group_code": f"{hospital.hospital_code}_PROMOTION",
                "contact_email": hospital.contact_email or "integration@hospital.com",
                "enabled": hospital.is_active,
                "created_date": hospital.created_at.strftime("%Y-%m-%d") if hospital.created_at else "2024-01-01",
                "max_users_per_day": 10000,
                "location": hospital.address or "Unknown",
                "integration_type": "payment_simulation"
            }
    except Exception as e:
        # If database query fails, fall back to hardcoded config
        print(f"Error querying Hospital model: {str(e)}")
    
    # Fallback to hardcoded HOSPITAL_INFO (for backward compatibility)
    if api_key in HOSPITAL_INFO:
        hospital_info = HOSPITAL_INFO[api_key]
        if hospital_info and hospital_info.get("enabled", False):
            return hospital_info
    
    # Fallback to default if not found
    if api_key == HOSPITAL_API_KEY:
        hospital_info = HOSPITAL_INFO.get("default")
        if hospital_info and hospital_info.get("enabled", False):
            return hospital_info
    
    return None

def get_hospital_name(api_key):
    """
    Get the hospital name for a given API key.
    Now checks the Hospital database model first.
    
    Args:
        api_key (str): The API key
        
    Returns:
        str: Hospital name or "Unknown Hospital"
    """
    hospital_info = validate_hospital_api_key(api_key)
    if hospital_info:
        return hospital_info.get("name", "Unknown Hospital")
    return "Unknown Hospital"

def get_hospital_code(api_key):
    """
    Get the hospital code for a given API key.
    Now checks the Hospital database model first.
    
    Args:
        api_key (str): The API key
        
    Returns:
        str: Hospital code or "UNKNOWN"
    """
    hospital_info = validate_hospital_api_key(api_key)
    if hospital_info:
        return hospital_info.get("code", "UNKNOWN")
    return "UNKNOWN"

def is_hospital_integration_enabled(api_key):
    """
    Check if hospital integration is enabled for the given API key.
    
    Args:
        api_key (str): The API key
        
    Returns:
        bool: True if enabled, False otherwise
    """
    hospital_info = validate_hospital_api_key(api_key)
    return hospital_info is not None and hospital_info.get("enabled", False)

def validate_comment_api_key(api_key):
    """
    Validate if the provided API key is valid for comment access.
    
    Args:
        api_key (str): The API key to validate
        
    Returns:
        bool: True if valid, False if invalid
    """
    return api_key == COMMENT_API_KEY 