from django.contrib.auth.models import Group
from django.shortcuts import render
from django.http import HttpResponse, JsonResponse
from django.shortcuts import redirect
import requests
import json
import urllib.parse
from APIs.models import User as AppUser, ZiLuckGroup
from APIs.views.views import RequsetChecker, myResponse
from APIs.views.Errors import Errors
from panel.models import TherapistSelection, Doctor
from payment.models import DigiPayPayment, PaymentPlans, ZarinPalPayment, Discount, ZarinPalPaymentCoin, HospitalUserUpgrade, HospitalUpgradeFailure
from rest_framework.decorators import api_view
from datetime import datetime, date, timedelta
import re
from panel.SMSNotification import SMS
from payment.email_handler import send_reciet_Email
from ziluck_project.constants import PRICES, MERCHANT, ZP_API_VERIFY, ZP_API_REQUEST, ZP_API_STARTPAY, CallbackURL
from panel.telegram_notification import TelegramNotification
from django.conf import settings
from django.views.decorators.csrf import csrf_exempt
from panel.models import Message
from payment.models import TimeSlotPayment
import jdatetime
from ziluck_project.hospital_config import validate_hospital_api_key, get_hospital_name, get_hospital_code
from decimal import Decimal

def calculate_paid_price(plan, discount_obj=None):
    """
    Calculate the final price paid by the user based on plan and discount.
    
    Args:
        plan (int): Plan number (1-8)
        discount_obj (Discount): Discount object if applicable
        
    Returns:
        Decimal: Final price after applying discounts and promotions
    """
    # Get base price from PRICES constant
    base_price = PRICES.get(plan, 0)
    
    # Try to get price from PaymentPlans model first
    try:
        payment_plan = PaymentPlans.objects.filter(plan_database_id=plan, is_active=True).first()
        if payment_plan:
            # Use promotion price if available and greater than 0, otherwise use regular price
            if payment_plan.price_with_promotion and payment_plan.price_with_promotion > 0:
                base_price = float(payment_plan.price_with_promotion)
            else:
                base_price = float(payment_plan.price)
    except Exception as e:
        print(f"Error getting price from PaymentPlans: {e}")
        # Fallback to PRICES constant
    
    # Apply discount if provided
    if discount_obj and discount_obj.is_valid():
        discount_percentage = discount_obj.percentage
        discounted_price = base_price * (100 - discount_percentage) / 100
        return Decimal(str(discounted_price))
    
    return Decimal(str(base_price))

try:
    from ziluck_project.constants import MERCHANT, ZP_API_VERIFY, ZP_API_REQUEST, ZP_API_STARTPAY, CallbackURL, DIGIPAY_CLIENT_ID, DIGIPAY_CLIENT_SECRET, DIGIPAY_AUTHORIZATION
except:
    MERCHANT, ZP_API_VERIFY, ZP_API_REQUEST, ZP_API_STARTPAY, CallbackURL = '', '', '', '', \
                                                                            'https://i-dia.ir/payment/verify/'

def is_valid_national_code(code: str):
    if not re.search(r'^\d{10}$', code): return False
    check = int(code[9])
    s = sum(int(code[x]) * (10 - x) for x in range(9)) % 11
    return check == s if s < 2 else check + s == 11


def get_plan_credit_days(plan):
    """Helper function to get credit days based on plan type"""
    if plan in [4, 3, 5, 7]:  # Monthly plans
        return 1 * 30
    elif plan in [2, 1, 6, 8]:  # Three month plans
        return 3 * 30
    return 0


def get_plan_name(plan):
    """Helper function to get Persian plan name based on plan ID"""
    plan_names = {
        1: "پلن نقره‌ای - ۳ ماهه",
        2: "پلن طلایی - ۳ ماهه",
        3: "پلن پلاتینیوم - ۳ ماهه",
        4: "پلن نقره‌ای - ۱ ماهه",
        5: "پلن طلایی - ۱ ماهه",
        6: "پلن پلاتینیوم - ۱ ماهه",
        7: "پلن برنزی - ۱ ماهه",
        8: "پلن برنزی - ۳ ماهه"
    }
    return plan_names.get(plan, f"پلن شماره {plan}")


@api_view(["GET"])
def verify(request):
    RequsetChecker(request.GET, [], request)
    t_status = request.GET.get('Status')
    t_authority = request.GET['Authority']
    print("this is the verify function {}".format(datetime.now()))
    print("GET {}".format(request.GET))
    print("t_status {}".format(t_status))
    print("t_authority {}".format(t_authority))
    print("Status {}".format(request.GET.get('Status')))
    if request.GET.get('Status') == 'OK':
        req_header = {"accept": "application/json",
                      "content-type": "application/json'"}
        zp: ZarinPalPayment = ZarinPalPayment.objects.filter(authority=t_authority)[0]
        req_data = {
            "merchant_id": MERCHANT,
            "amount": zp.amount_after_discount,
            "authority": t_authority
        }
        req = requests.post(url=ZP_API_VERIFY, data=json.dumps(req_data), headers=req_header)
        print("req.json()['errors']: {}".format(req.json()['errors']))
        print("len(req.json()['errors']): {}".format(len(req.json()['errors'])))
        if len(req.json()['errors']) == 0:
            t_status = req.json()['data']['code']
            if t_status == 100:
                zp.is_payed = True
                if zp.main_amount != zp.amount_after_discount:
                    zp.discount.used()
                    zp.assign_to_group()

                zp.save()

                ts = TherapistSelection(user=zp.user.django_user)
                print("new Therapist selection created")
                try:
                    previous_tss = TherapistSelection.objects.filter(user=zp.user.django_user)
                    print("checking previous therapist selections")
                    if previous_tss.exists():
                        print("previous therapist should be chosen!")
                        # if someone had already chosen a doctor, the same doctor should be chosen for her in the renewal.
                        ts.therapist = previous_tss.last().therapist
                        print("now therapist is", ts.therapist, " [ Django User ID ]")

                        # if the doctor was not selected for any reason, the default doctor of the system should be selected.
                        if previous_tss.last().therapist is None:
                            print("doctor was not selected for any reason!")
                            doctor = Doctor.objects.all()[0]
                            print("doctor[0] should be chosen!")
                            ts.therapist = doctor.django_user
                            print("and now therapist is", ts.therapist, "doctor:", doctor.first_name, doctor.last_name)
                    else:
                        print("dose not have any therapist so set the default doctor of the system")
                        doctor = Doctor.objects.all()[0]
                        ts.therapist = doctor.django_user
                        print("now therapist is", ts.therapist, "doctor:", doctor.first_name, doctor.last_name)

                    ts.save()
                    print("ts.save()")
                    zp.user.profile.therapist = ts.therapist
                    zp.user.profile.save()
                    print("zp.user.profile.save()")

                except Exception as e:
                    print("there is no doctor!")
                    pass

                print("plan:", zp.plan)
                # Add credit based on plan type
                credit_days = get_plan_credit_days(zp.plan)
                ts.add_credit(credit_days)
                ts.save()
                print("ts.save()")

                doctor = Doctor.objects.all()[0]
                first_name = zp.user.profile.first_name
                message_text = f"""سلام {first_name} عزیز

به جمع گرم لاکتو خوش اومدی! ما اینجا توی لاکتو دور هم جمع شدیم تا با هم سفری پرماجرا و لذت‌بخش به دنیای سلامتی رو آغاز کنیم. ‍

قبل از اینکه بریم سراغ طراحی یه برنامه غذایی اختصاصی و فوق‌العاده برای تو، می‌خوایم یه ملاقات کوتاه و صمیمی توی گوگل میت داشته باشیم.

تو این ملاقات دوستانه:
با هم حرف می‌زنیم و علایق، سبک زندگی و اهدافت رو توی مسیر سلامتی بهتر می‌شناسیم.
یه گپ دوستانه در مورد عادت‌های غذایی و فعالیت‌های روزمره‌ات می‌زنیم.
به تمام سوالات شما در مورد برنامه غذایی لاکتو و هر موضوع دیگه‌ای که ذهنت رو مشغول کرده جواب می‌دیم.

هدف ما توی لاکتو چیه؟
همراهی در مسیر سلامتی: مثل یه دوست و همراه توی این مسیر پرماجرا کنارت خواهیم بود و از تمام توانمون برای رسیدن تو به اهدافت استفاده می‌کنیم.
بهت کمک می‌کنیم تا به بهترین نتیجه‌ای که می‌خوای برسی و از یه زندگی پرنشاط و سالم لذت ببری. ‍
یه برنامه غذایی منحصر به فرد کاملا متناسب با نیازها، سلیقه و شرایط شما طراحی می‌کنیم که خوردن غذا رو برای تو به یه تجربه لذت‌بخش تبدیل کنه.

لطفا از قسمت رزرو جلسه تایمی که برات مناسبه انتخاب کن 

ما بی‌صبرانه منتظر صحبت کردن باهات و شروع این سفر هیجان‌انگیز با تو هستیم!"""

                if zp.plan in [7, 8]:
                    message_text = f"""سلام {first_name} عزیز

                    به جمع گرم لاکتو خوش اومدی! ما اینجا توی لاکتو دور هم جمع شدیم تا با هم سفری پرماجرا و لذت‌بخش به دنیای سلامتی رو آغاز کنیم. ‍
                    
                    رژیم غذایی شما تا ۲۴ ساعت آینده در دسترس قرار خواهد گرفت و از طریق قسمت مشاهده منو غذایی میتوانید آن را مشاهده بفرمایید.
                    """
                else:
                    try:
                        zp.user.profile.coins += 2000
                        zp.user.profile.save()
                    except Exception as e:
                        print("This is the exception: ", e)

                    
                Message.objects.create(
                    sender=doctor.django_user,
                    receiver=zp.user.django_user,
                    text=message_text,
                    time=datetime.now()
                ).save()

                try:
                    TelegramNotification().send_purchase_notif(zp.user, plan=zp.plan, discount_code=zp.discount)
                except Exception as e:
                    print("This is the telegram exception: ", e)

                # email to user
                try:
                    if len(str(zp.authority)) > 8:
                        reference_number = zp.authority[-8:]
                    else:
                        reference_number = ""
                    send_reciet_Email(zp.user.profile.email, zp.amount_after_discount, reference_number=reference_number)
                except Exception as e:
                    print("error on sending reciet email:", e)

                # Get user name
                user_name = f"{zp.user.profile.first_name} {zp.user.profile.last_name}" if zp.user and zp.user.profile and zp.user.profile.first_name else "کاربر"
                
                # Get plan information
                plan_name = get_plan_name(zp.plan)
                credit_days = get_plan_credit_days(zp.plan)
                
                # Get payment date in Jalali
                try:
                    jalali_date = jdatetime.datetime.fromgregorian(datetime=datetime.now())
                    payment_date = jalali_date.strftime('%Y/%m/%d - %H:%M')
                except:
                    payment_date = datetime.now().strftime('%Y-%m-%d %H:%M')
                
                # Check if discount was applied
                discount_applied = zp.main_amount != zp.amount_after_discount
                
                # Generate deep link for successful payment
                deep_link_url = f"lacto://payment/callback?status=success&plan_id={zp.plan}&transaction_id={req.json()['data']['ref_id']}&payment_id={zp.id}&amount={zp.amount_after_discount}&plan_name={urllib.parse.quote(plan_name)}&duration={credit_days}"
                
                context = dict(
                    success=True,
                    title='تراکنش با موفقیت انجام شد!',
                    user_name=user_name,
                    plan_name=plan_name,
                    plan_id=zp.plan,
                    duration=credit_days,
                    payment_date=payment_date,
                    ref_id=str(req.json()['data']['ref_id']),
                    transaction_id=str(req.json()['data']['ref_id']),
                    amount=f"{zp.amount_after_discount:,}",
                    discount_applied=discount_applied,
                    deep_link_url=deep_link_url,
                    payment_id=zp.id,
                    status='success',
                    debug=True  # Set to False in production
                )
                print("context:", context)
                return render(request, template_name='payment_success.html', context=context)
            else:
                # Generate deep link for verification error
                error_message_encoded = urllib.parse.quote('خطا در تایید نهایی تراکنش')
                deep_link_url = f"lacto://payment/callback?status=failed&error_message={error_message_encoded}&error_code={t_status}"
                
                context = dict(
                    success=False,
                    title='تراکنش با موفقیت انجام نشد!',
                    message='خطا در تایید نهایی تراکنش',
                    error_details=f'کد خطا: {t_status}',
                    deep_link_url=deep_link_url,
                    debug=True  # Set to False in production
                )
                return render(request, template_name='payment_success.html', context=context)
        else:
            # Generate deep link for API errors
            error_message_encoded = urllib.parse.quote('خطا در تایید نهایی تراکنش')
            deep_link_url = f"lacto://payment/callback?status=failed&error_message={error_message_encoded}"
            
            context = dict(
                success=False,
                title='تراکنش با موفقیت انجام نشد!',
                message='خطا در تایید نهایی تراکنش',
                error_details='خطا در ارتباط با درگاه پرداخت',
                deep_link_url=deep_link_url,
                debug=True  # Set to False in production
            )
            return render(request, template_name='payment_success.html', context=context)
    else:
        # Generate deep link for cancelled transaction
        error_message_encoded = urllib.parse.quote('تراکنش توسط کاربر کنسل شد و یا مشکلی در آن وجود داشت! مدتی بعد مجددا تلاش کنید')
        deep_link_url = f"lacto://payment/callback?status=cancelled&error_message={error_message_encoded}"
        
        context = dict(
            success=False,
            title='تراکنش با موفقیت انجام نشد!',
            message='تراکنش توسط کاربر کنسل شد و یا مشکلی در آن وجود داشت! مدتی بعد مجددا تلاش کنید',
            deep_link_url=deep_link_url,
            debug=True  # Set to False in production
        )
        return render(request, template_name='payment_success.html', context=context)


@api_view(["GET"])
def verify_timeslot_zarinpal(request):
    """Verify ZarinPal payment for timeslot purchase"""
    RequsetChecker(request.GET, [], request)
    t_status = request.GET.get('Status')
    t_authority = request.GET['Authority']
    print("this is the timeslot verify function {}".format(datetime.now()))
    print("GET {}".format(request.GET))
    print("t_status {}".format(t_status))
    print("t_authority {}".format(t_authority))
    
    if request.GET.get('Status') == 'OK':
        req_header = {"accept": "application/json",
                      "content-type": "application/json'"}
        
        # Find the timeslot payment by authority
        payment = TimeSlotPayment.objects.filter(authority=t_authority).first()
        
        if not payment:
            # Generate deep link for payment not found
            error_message_encoded = urllib.parse.quote('پرداخت یافت نشد')
            deep_link_url = f"lacto://payment/callback?status=failed&timeslot_id=0&error_message={error_message_encoded}"
            
            context = dict(
                success=False,
                title='تراکنش با موفقیت انجام نشد!',
                message='پرداخت یافت نشد',
                deep_link_url=deep_link_url,
                error_message='پرداخت یافت نشد',
                debug=True  # Set to False in production
            )
            return render(request, template_name='timeslot_success.html', context=context)
        
        req_data = {
            "merchant_id": MERCHANT,
            "amount": payment.money_amount,
            "authority": t_authority
        }
        req = requests.post(url=ZP_API_VERIFY, data=json.dumps(req_data), headers=req_header)
        print("req.json()['errors']: {}".format(req.json()['errors']))
        print("len(req.json()['errors']): {}".format(len(req.json()['errors'])))
        
        if len(req.json()['errors']) == 0:
            t_status = req.json()['data']['code']
            if t_status == 100:
                # Mark payment as successful
                payment.is_paid = True
                payment.payment_date = datetime.now()
                payment.save()
                
                # Update timeslot availability
                payment.time_slot.is_available = 2  # Purchased - Waiting Assignment
                payment.time_slot.save()
                
                # Send Telegram notification
                try:
                    user_name = f"{payment.user.profile.first_name} {payment.user.profile.last_name}" if payment.user and payment.user.profile else "کاربر"
                    jalali_date = jdatetime.datetime.fromgregorian(datetime=payment.time_slot.start_time)
                    meeting_info = jalali_date.strftime('%Y/%m/%d ساعت %H:%M')
                    TelegramNotification().send_timeslot_purchase_notif(
                        f"یک کاربر به نام {user_name} زمان مکالمه {meeting_info} را با پرداخت نقدی خریداری کرد"
                    )
                except Exception as e:
                    print("This is the telegram exception: ", e)
                
                # Format meeting information
                meeting_datetime = payment.time_slot.start_time
                if meeting_datetime:
                    # Convert to Jalali date
                    jalali_date = jdatetime.datetime.fromgregorian(datetime=meeting_datetime)
                    meeting_date = jalali_date.strftime('%Y/%m/%d')
                    meeting_time = jalali_date.strftime('%H:%M')
                else:
                    meeting_date = "تعیین نشده"
                    meeting_time = "تعیین نشده"
                
                # Get user name
                user_name = f"{payment.user.profile.first_name} {payment.user.profile.last_name}" if payment.user and payment.user.profile else "کاربر"
                
                # Format payment type
                payment_type_persian = "پرداخت نقدی" if payment.payment_type == 'money' else "پرداخت با سکه"
                
                # Generate deep link for successful payment
                deep_link_url = f"lacto://payment/callback?status=success&timeslot_id={payment.time_slot.id}&transaction_id={req.json()['data']['ref_id']}&payment_id={payment.id}"
                
                context = dict(
                    success=True,
                    title='تراکنش با موفقیت انجام شد!',
                    user_name=user_name,
                    meeting_date=meeting_date,
                    meeting_time=meeting_time,
                    duration=payment.time_slot.duration or 30,
                    payment_type=payment_type_persian,
                    ref_id=str(req.json()['data']['ref_id']),
                    amount=f"{payment.money_amount:,}" if payment.money_amount else "0",
                    deep_link_url=deep_link_url,
                    timeslot_id=payment.time_slot.id,
                    transaction_id=str(req.json()['data']['ref_id']),
                    debug=True  # Set to False in production
                )
                print("context:", context)
                return render(request, template_name='timeslot_success.html', context=context)
            else:
                # Generate deep link for verification error
                error_message_encoded = urllib.parse.quote('خطا در تایید نهایی تراکنش')
                deep_link_url = f"lacto://payment/callback?status=failed&timeslot_id={payment.time_slot.id if payment else 0}&error_message={error_message_encoded}"
                
                context = dict(
                    success=False,
                    title='تراکنش با موفقیت انجام نشد!',
                    message='خطا در تایید نهایی تراکنش',
                    deep_link_url=deep_link_url,
                    error_message='خطا در تایید نهایی تراکنش',
                    debug=True  # Set to False in production
                )
                return render(request, template_name='timeslot_success.html', context=context)
        else:
            # Generate deep link for API errors
            error_message_encoded = urllib.parse.quote('خطا در تایید نهایی تراکنش')
            deep_link_url = f"lacto://payment/callback?status=failed&timeslot_id={payment.time_slot.id if payment else 0}&error_message={error_message_encoded}"
            
            context = dict(
                success=False,
                title='تراکنش با موفقیت انجام نشد!',
                message='خطا در تایید نهایی تراکنش',
                deep_link_url=deep_link_url,
                error_message='خطا در تایید نهایی تراکنش',
                debug=True  # Set to False in production
            )
            return render(request, template_name='timeslot_success.html', context=context)
    else:
        # Generate deep link for cancelled transaction
        error_message_encoded = urllib.parse.quote('تراکنش توسط کاربر کنسل شد و یا مشکلی در آن وجود داشت! مدتی بعد مجددا تلاش کنید')
        deep_link_url = f"lacto://payment/callback?status=cancelled&timeslot_id=0&error_message={error_message_encoded}"
        
        context = dict(
            success=False,
            title='تراکنش با موفقیت انجام نشد!',
            message='تراکنش توسط کاربر کنسل شد و یا مشکلی در آن وجود داشت! مدتی بعد مجددا تلاش کنید',
            deep_link_url=deep_link_url,
            error_message='تراکنش توسط کاربر کنسل شد و یا مشکلی در آن وجود داشت! مدتی بعد مجددا تلاش کنید',
            debug=True  # Set to False in production
        )
        return render(request, template_name='timeslot_success.html', context=context)

@api_view(["POST"])
def verify_timeslot_digipay(request):
    """Verify DigiPay payment for timeslot purchase"""
    # Validate required POST parameters
    required_params = ['amount', 'providerId', 'trackingCode', 'result', 'type']
    for param in required_params:
        if param not in request.POST:
            return JsonResponse({
                'error': f'Missing required parameter: {param}'
            }, status=400)

    # Get all required fields from DigiPay response
    amount = request.POST.get('amount')
    provider_id = request.POST.get('providerId')
    tracking_code = request.POST.get('trackingCode')
    result = request.POST.get('result')
    type = request.POST.get('type')
    
    print("amount:", amount)
    print("provider_id:", provider_id)
    print("tracking_code:", tracking_code)
    print("result:", result)
    print("type:", type)    

    if result != 'SUCCESS':
        return render(request, template_name='message.html', context={
            'title': 'تراکنش انجام نشد!',
            'message': "تراکنش توسط کاربر کنسل شد و یا مشکلی در آن وجود داشت! مدتی بعد مجددا تلاش کنید"
        })

    try:
        # Find the timeslot payment by payment_id
        payment = TimeSlotPayment.objects.filter(payment_id=provider_id).first()
        
        if not payment:
            return render(request, template_name='message.html', context={
                'title': 'تراکنش با موفقیت انجام نشد!',
                'message': 'پرداخت یافت نشد'
            })
        
        # Verify the amount matches
        if int(amount) != payment.money_amount:
            return render(request, template_name='message.html', context={
                'title': 'تراکنش با موفقیت انجام نشد!',
                'message': 'مبلغ تراکنش با مبلغ درخواستی مطابقت ندارد'
            })
        
        # Call DigiPay verification API
        verify_headers = {
            "Authorization": f"Bearer {payment.digipay_token}",
            "Content-Type": "application/json; charset=UTF-8",
            "Agent": "WEB",
            "Digipay-Version": "2022-02-02",
            "Accept": "application/json"
        }
        
        verify_url = f"https://api.mydigipay.com/digipay/api/purchases/verify/{tracking_code}?type={type}"
        verify_response = requests.post(verify_url, headers=verify_headers)
        
        if verify_response.status_code != 200:
            return render(request, template_name='timeslot_success.html', context={
                'success': False,
                'title': 'تراکنش با موفقیت انجام نشد!',
                'message': 'خطا در تایید نهایی تراکنش'
            })
        
        print("verify_response:", verify_response.json())

        # Call DigiPay delivery API
        delivery_headers = {
            "Authorization": f"Bearer {payment.digipay_token}",
            "Content-Type": "application/json",
            "Agent": "WEB",
            "Digipay-Version": "2022-02-02"
        }
        
        delivery_url = f"https://api.mydigipay.com/digipay/api/purchases/deliver?type={type}"
        delivery_payload = {
            "deliveryDate": int(datetime.now().timestamp() * 1000),
            "invoiceNumber": tracking_code,
            "trackingCode": tracking_code,
            "products": ["timeslot-purchase"]
        }
        
        delivery_response = requests.post(delivery_url, headers=delivery_headers, json=delivery_payload)
        
        print("delivery_response:", delivery_response.json())
        
        if delivery_response.status_code != 200:
            return render(request, template_name='timeslot_success.html', context={
                'success': False,
                'title': 'تراکنش با موفقیت انجام نشد!',
                'message': 'خطا در ثبت اطلاعات تحویل'
            })
        
        # Process successful payment
        payment.is_paid = True
        payment.payment_date = datetime.now()
        payment.tracking_code = tracking_code
        payment.save()
        
        # Update timeslot availability
        payment.time_slot.is_available = 2  # Purchased - Waiting Assignment
        payment.time_slot.save()
        
        # Send Telegram notification
        try:
            user_name = f"{payment.user.profile.first_name} {payment.user.profile.last_name}" if payment.user and payment.user.profile else "کاربر"
            jalali_date = jdatetime.datetime.fromgregorian(datetime=payment.time_slot.start_time)
            meeting_info = jalali_date.strftime('%Y/%m/%d ساعت %H:%M')
            TelegramNotification().send_timeslot_purchase_notif(
                f"یک کاربر به نام {user_name} زمان مکالمه {meeting_info} را با پرداخت نقدی خریداری کرد"
            )
        except Exception as e:
            print("This is the telegram exception: ", e)
        
        meeting_datetime = payment.time_slot.start_time
        if meeting_datetime:
            # Convert to Jalali date
            jalali_date = jdatetime.datetime.fromgregorian(datetime=meeting_datetime)
            meeting_date = jalali_date.strftime('%Y/%m/%d')
            meeting_time = jalali_date.strftime('%H:%M')
        else:
            meeting_date = "تعیین نشده"
            meeting_time = "تعیین نشده"
        
        # Get user name
        user_name = f"{payment.user.profile.first_name} {payment.user.profile.last_name}" if payment.user and payment.user.profile else "کاربر"
        
        # Format payment type
        payment_type_persian = "پرداخت نقدی" if payment.payment_type == 'money' else "پرداخت با سکه"
        
        return render(request, template_name='timeslot_success.html', context={
            'success': True,
            'title': 'تراکنش با موفقیت انجام شد!',
            'user_name': user_name,
            'meeting_date': meeting_date,
            'meeting_time': meeting_time,
            'duration': payment.time_slot.duration or 30,
            'payment_type': payment_type_persian,
            'ref_id': tracking_code,
            'amount': f"{payment.money_amount:,}" if payment.money_amount else "0"
        })
        
    except Exception as e:
        print(f"Error in timeslot DigiPay verification: {e}")
        return render(request, template_name='timeslot_success.html', context={
            'success': False,
            'title': 'تراکنش با موفقیت انجام نشد!',
            'message': 'خطا در پردازش تراکنش'
        })


@api_view(["POST"])
def get_payment_link(request):
    # try:
        RequsetChecker(request.POST, [
            {
                "name": "token",
                "format": "^(\S)+$",
                "required": True,
            },
            {
                "name": "discount_code",
                "format": "",
                "required": False,
            },
            {
                "name": "plan_id",
                "format": "^(\S)+$",
                "required": True,
            },
        ], request)

        token = request.POST.get("token")
        if token == None:
            return myResponse.Error(Errors.InvalidArgument.message, Errors.InvalidArgument.code)

        temp = AppUser.objects.filter(token=token)
        if not temp.exists():
            return myResponse.Error(Errors.InvalidToken.message, Errors.InvalidToken.code)

        app_user = temp[0]
        
        if app_user.phone_number is None or not app_user.phone_number.startswith('09'):
            return myResponse.Error("This service is not currently available contact sales for more information", 1002)
        
        plan = int(request.POST.get("plan_id"))
        
        if plan == 3 or plan == 6:
            return myResponse.Error("ظرفیت سرویس دهی در پلن پلاتینیوم برای ماه جاری به اتمام رسیده است", -1)
        
        # Check if plan is valid
        if plan not in PRICES:
            return myResponse.Error("Invalid plan selected", -1)
            
        amount = PRICES[plan]
        discount_code = str(request.POST.get("discount_code")).upper()

        zp = ZarinPalPayment(user=app_user,
                           main_amount=amount,
                           amount_after_discount=amount,
                           plan=plan)
        zp.save()
        return zp.get_pay_link(discount_code, app_user.profile.user_code)

    # except Exception as e:
    #     return myResponse.Error(e.args[0], Errors.InvalidArgument.code)

@api_view(["POST"])
def check_for_discount(request):
    try:
        RequsetChecker(request.POST, [
            {
                "name": "token",  # required
                "format": "^(\S)+$",  # required
                "required": True,  # optional default is true
            },
            {
                "name": "discount_code",  # required
                "format": "",  # required
                "required": False,  # optional default is true
            },
        ], request)

        token = request.POST.get("token")
        if token == None:
            return myResponse.Error(Errors.InvalidArgument.message, Errors.InvalidArgument.code)

        discount_code = str(request.POST.get('discount_code')).upper()

        temp = Discount.objects.filter(code=discount_code)
        if temp.exists():
            dc: Discount = temp[0]
            if dc.is_valid():
                message = 'تبریک! خرید شما شامل ' + str(dc.percentage) + ' درصد تخفیف خواهد شد!'
            else:
                message = 'متاسفانه این کد تخفیف معتبر نیست!'
        else:
            message = 'متاسفانه این کد تخفیف معتبر نیست!'

        data = {
            'message': message
        }
        return myResponse.OK('discount code', data)

    except ValueError as e:
        return myResponse.Error(e.args[0], Errors.InvalidArgument.code)

from rest_framework.decorators import api_view

@api_view(["POST"])
def check_for_discount_hospital(request):
    try:
        # Check required parameters
        required_params = ['hospital_api_key', 'plan_id']
        for param in required_params:
            if param not in request.POST:
                return myResponse.Error(f"{param} is required", Errors.InvalidArgument.code)

        hospital_api_key = request.POST.get("hospital_api_key")
        discount_code = request.POST.get("discount_code", None)
        plan_id = request.POST.get("plan_id")

        # Validate hospital API key
        hospital_info = validate_hospital_api_key(hospital_api_key)
        if not hospital_info:
            return myResponse.Error(Errors.InvalidHospitalAPIKey.message, Errors.InvalidHospitalAPIKey.code)

        hospital_discount_group_code = hospital_info.get("nikan_discount_group_code")

        # Get the plan
        try:
            plan_id_int = int(plan_id)
        except Exception:
            return myResponse.Error(Errors.InvalidPlanID.message, Errors.InvalidPlanID.code)

        plan = PaymentPlans.objects.filter(plan_database_id=plan_id_int).first()
        if not plan:
            return myResponse.Error(Errors.InvalidPlanID.message, Errors.InvalidPlanID.code)

        # Get original price and promotion price
        original_price = float(plan.price)
        promotion_price = float(plan.price_with_promotion) if plan.price_with_promotion and plan.price_with_promotion > 0 else None

        # Determine base price for discount calculation: use promotion price if available, otherwise use regular price
        base_price = promotion_price if promotion_price is not None else original_price

        # If no discount code, just return the prices without discount
        if not discount_code:
            return myResponse.OK('discount code', {
                'message': 'کد تخفیفی وارد نشده است.',
                'original_price': original_price,
                'price_with_promotion': promotion_price,
                'price_after_discount': base_price
            })

        # Validate discount code
        discount = Discount.objects.filter(code=discount_code.upper()).first()
        if not discount:
            return myResponse.OK('متاسفانه این کد تخفیف معتبر نیست!', {
                'message': 'متاسفانه این کد تخفیف معتبر نیست!',
                'original_price': original_price,
                'price_with_promotion': promotion_price,
                'price_after_discount': base_price
            })

        # Check if discount is valid (not expired and has remaining uses)
        if not discount.is_valid():
            return myResponse.OK('متاسفانه این کد تخفیف معتبر نیست!', {
                'message': 'متاسفانه این کد تخفیف معتبر نیست!',
                'original_price': original_price,
                'price_with_promotion': promotion_price,
                'price_after_discount': base_price
            })

        # Check group match: group name must match hospital name
        ziluck_group = ZiLuckGroup.objects.filter(name=hospital_discount_group_code).first()
        
        
        if discount.ziluck_group != ziluck_group:
            return myResponse.Error(Errors.DiscountCodeNotDefinedForHospital.message, Errors.DiscountCodeNotDefinedForHospital.code)

        # Apply discount to the base price (promotion price if available, otherwise regular price)
        try:
            percentage = float(discount.percentage)
        except Exception:
            percentage = 0
        price_after_discount = base_price * (100 - percentage) / 100

        message = f'تبریک! خرید شما شامل {round(discount.percentage)} درصد تخفیف خواهد شد!'

        data = {
            'message': message,
            'original_price': original_price,
            'price_with_promotion': promotion_price,
            'price_after_discount': price_after_discount
        }
        return myResponse.OK('discount code', data)

    except Exception as e:
        return myResponse.Error("Error in check_for_discount_hospital:" + str(e), Errors.InternalError.code)


@api_view(["POST"])
def check_subscription(request):
    try:
        RequsetChecker(request.POST, [
            {
                "name": "token",
                "format": "^(\S)+$",
                "required": True,
            }
        ], request)

        token = request.POST.get("token")
        temp = AppUser.objects.filter(token=token)
        if not temp.exists():
            return myResponse.Error(Errors.InvalidToken.message, Errors.InvalidToken.code)

        app_user = temp[0]
        django_user = app_user.django_user

        temp = TherapistSelection.objects.filter(user=django_user)
        if not temp.exists():
            subscription_status = False
            plan = 0
            credit = 0
        else:
            ts = temp.last()
            zp: ZarinPalPayment = ZarinPalPayment.objects.filter(user=app_user).order_by("created_at").last()

            subscription_status = ts.has_credit()
            plan = zp.plan if zp else 0
            credit = ts.credit
        data = {
            "status": subscription_status,
            "plan": plan,
            "credit": credit
        }
        return myResponse.OK('subscription', data)

    except ValueError as e:
        return myResponse.Error(e.args[0], Errors.InvalidArgument.code)


@api_view(["POST"])
def get_payment_link_coin(request):
    try:
        RequsetChecker(request.POST, [
            {
                "name": "token",  # required
                "format": "^(\S)+$",  # required
                "required": True,  # optional default is true
            },
            {
                "name": "coin_amount",  # required
                "format": "^\d+$",  # required
                "required": True,  # optional default is true
            },
            {
                "name": "discount_code",  # optional
                "format": "",
                "required": False,  # optional default is true
            },
        ], request)

        token = request.POST.get("token")
        coin_amount = int(request.POST.get("coin_amount"))

        # Validate token
        if token is None:
            return myResponse.Error(Errors.InvalidArgument.message, Errors.InvalidArgument.code)

        temp = AppUser.objects.filter(token=token)
        if not temp.exists():
            return myResponse.Error(Errors.InvalidToken.message, Errors.InvalidToken.code)

        app_user = temp[0]
        discount_code = str(request.POST.get("discount_code")).upper() if request.POST.get("discount_code") else ""

        # Create a payment instance
        zp = ZarinPalPaymentCoin(user=app_user, coin_amount=coin_amount)
        zp.save()
        return zp.get_pay_link_coin(discount_code)

    except Exception as e:
        return myResponse.Error(str(e), Errors.InvalidArgument.code)


class MyDigiPayPayment:
    def __init__(self):
        self.api_key = settings.MYDIGIPAY_API_KEY
        self.base_url = "https://api.mydigipay.com/v1"
        self.headers = {
            "Authorization": f"Bearer {self.api_key}",
            "Content-Type": "application/json"
        }

    def create_payment(self, amount, callback_url, description=None):
        """
        Create a new payment request
        """
        endpoint = f"{self.base_url}/payments"
        payload = {
            "amount": amount,
            "callback_url": callback_url,
            "description": description or "Payment for Ziluck"
        }

        try:
            response = requests.post(endpoint, headers=self.headers, json=payload)
            response.raise_for_status()
            return response.json()
        except requests.exceptions.RequestException as e:
            return {"error": str(e)}

    def verify_payment(self, payment_id):
        """
        Verify a payment status
        """
        endpoint = f"{self.base_url}/payments/{payment_id}"
        try:
            response = requests.get(endpoint, headers=self.headers)
            response.raise_for_status()
            return response.json()
        except requests.exceptions.RequestException as e:
            return {"error": str(e)}


@api_view(["POST"])
def get_digipay_payment_link(request):
    try:
        RequsetChecker(request.POST, [
            {
                "name": "token",
                "format": "^(\S)+$",
                "required": True,
            },
            {
                "name": "discount_code",
                "format": "",
                "required": False,
            },
            {
                "name": "plan_id",
                "format": "^(\S)+$",
                "required": True,
            },
        ], request)

        token = request.POST.get("token")
        if token == None:
            return myResponse.Error(Errors.InvalidArgument.message, Errors.InvalidArgument.code)

        temp = AppUser.objects.filter(token=token)
        if not temp.exists():
            return myResponse.Error(Errors.InvalidToken.message, Errors.InvalidToken.code)

        app_user = temp[0]
        plan = int(request.POST.get("plan_id"))
        if plan == 3 or plan == 6:
             return myResponse.Error("ظرفیت سرویس دهی در پلن پلاتینیومf برای ماه جاری به اتمام رسیده است", -1)
        
        # Check if plan is valid
        if plan not in PRICES:
            return myResponse.Error("Invalid plan selected", -1)
        
        amount = PRICES[plan]
        discount_code = str(request.POST.get("discount_code")).upper()
        
        # Get DigiPay OAuth token
        oauth_url = "https://api.mydigipay.com/digipay/api/oauth/token"
        oauth_payload = {
            "username": DIGIPAY_CLIENT_ID,
            "password": DIGIPAY_CLIENT_SECRET,
            "grant_type": "password"
        }
        oauth_headers = {
            "Authorization": "Basic " + DIGIPAY_AUTHORIZATION
        }
        
        try:
            oauth_response = requests.post(oauth_url, data=oauth_payload, headers=oauth_headers)
            oauth_response.raise_for_status()
            digipay_token = oauth_response.json()["access_token"]
        except Exception as e:
            return myResponse.Error(f"Failed to get DigiPay token: {str(e)}", Errors.InvalidArgument.code)

        if digipay_token == None:
            return myResponse.Error(Errors.InvalidArgument.message, Errors.InvalidArgument.code)    
        
        provider_id = str(int(datetime.now().timestamp()))

        dp = DigiPayPayment(user=app_user,
                          main_amount=amount,
                          amount_after_discount=amount,
                          plan=plan,
                          payment_id=provider_id)
        dp.save()

        return dp.get_pay_link(discount_code, app_user.profile.user_code, digipay_token, provider_id)

    except Exception as e:
        return myResponse.Error(e.args[0], Errors.InvalidArgument.code)


@api_view(["POST"])
def verify_digipay(request):
    # Validate required POST parameters
    required_params = ['amount', 'providerId', 'trackingCode', 'result', 'type']
    for param in required_params:
        if param not in request.POST:
            return JsonResponse({
                'error': f'Missing required parameter: {param}'
            }, status=400)

    # Get all required fields from DigiPay response
    amount = request.POST.get('amount')
    provider_id = request.POST.get('providerId')
    tracking_code = request.POST.get('trackingCode')
    result = request.POST.get('result')
    type = request.POST.get('type')

    if result != 'SUCCESS':
        return render(request, template_name='message.html', context={
            'title': 'تراکنش انجام نشد!',
            'message': "تراکنش توسط کاربر کنسل شد و یا مشکلی در آن وجود داشت! مدتی بعد مجددا تلاش کنید"
        })

    try:
        dp: DigiPayPayment = DigiPayPayment.objects.filter(payment_id=provider_id)[0]
        
        # Verify the amount matches
        if int(amount) != dp.amount_after_discount:
            return render(request, template_name='message.html', context={
                'title': 'تراکنش با موفقیت انجام نشد!',
                'message': 'مبلغ تراکنش با مبلغ درخواستی مطابقت ندارد'
            })
        
        # Call DigiPay verification API
        verify_headers = {
            "Authorization": f"Bearer {dp.digipay_token}",
            "Content-Type": "application/json; charset=UTF-8",
            "Agent": "WEB",
            "Digipay-Version": "2022-02-02",
            "Accept": "application/json"
        }
        
        verify_url = f"https://api.mydigipay.com/digipay/api/purchases/verify/{tracking_code}?type={type}"
        verify_response = requests.post(verify_url, headers=verify_headers)
        
        if verify_response.status_code != 200:
            return render(request, template_name='message.html', context={
                'title': 'تراکنش با موفقیت انجام نشد!',
                'message': 'خطا در تایید نهایی تراکنش'
            })
        

        # Call DigiPay delivery API
        delivery_headers = {
            "Authorization": f"Bearer {dp.digipay_token}",
            "Content-Type": "application/json",
            "Agent": "WEB",
            "Digipay-Version": "2022-02-02"
        }
        
        delivery_url = f"https://api.mydigipay.com/digipay/api/purchases/deliver?type={type}"
        delivery_payload = {
            "deliveryDate": int(datetime.now().timestamp() * 1000),
            "invoiceNumber": tracking_code,
            "trackingCode": tracking_code,
            "products": [f"product-{dp.plan}"]
        }
        
        delivery_response = requests.post(delivery_url, headers=delivery_headers, json=delivery_payload)
        
        
        if delivery_response.status_code != 200:
            return render(request, template_name='message.html', context={
                'title': 'تراکنش با موفقیت انجام نشد!',
                'message': 'خطا در ثبت اطلاعات تحویل'
            })
        
        # Process successful payment
        dp.is_payed = True
        dp.tracking_code = tracking_code
        if dp.main_amount != dp.amount_after_discount:
            dp.discount.used()
            dp.assign_to_group()
        dp.save()

        # Handle therapist selection
        ts = TherapistSelection(user=dp.user.django_user)
        try:
            previous_tss = TherapistSelection.objects.filter(user=dp.user.django_user)
            if previous_tss.exists():
                ts.therapist = previous_tss.last().therapist
                if previous_tss.last().therapist is None:
                    doctor = Doctor.objects.all()[0]
                    ts.therapist = doctor.django_user
            else:
                doctor = Doctor.objects.all()[0]
                ts.therapist = doctor.django_user

            ts.save()
            dp.user.profile.therapist = ts.therapist
            dp.user.profile.save()
        except Exception:
            pass

        # Add credit based on plan type
        credit_days = get_plan_credit_days(dp.plan)
        ts.add_credit(credit_days)
        ts.save()

        doctor = Doctor.objects.all()[0]
        first_name = dp.user.profile.first_name
        message_text = f"""سلام {first_name} عزیز

به جمع گرم لاکتو خوش اومدی! ما اینجا توی لاکتو دور هم جمع شدیم تا با هم سفری پرماجرا و لذت‌بخش به دنیای سلامتی رو آغاز کنیم. ‍

قبل از اینکه بریم سراغ طراحی یه برنامه غذایی اختصاصی و فوق‌العاده برای تو، می‌خوایم یه ملاقات کوتاه و صمیمی توی گوگل میت داشته باشیم.

تو این ملاقات دوستانه:
با هم حرف می‌زنیم و علایق، سبک زندگی و اهدافت رو توی مسیر سلامتی بهتر می‌شناسیم.
یه گپ دوستانه در مورد عادت‌های غذایی و فعالیت‌های روزمره‌ات می‌زنیم.
به تمام سوالات شما در مورد برنامه غذایی لاکتو و هر موضوع دیگه‌ای که ذهنت رو مشغول کرده جواب می‌دیم.

هدف ما توی لاکتو چیه؟
همراهی در مسیر سلامتی: مثل یه دوست و همراه توی این مسیر پرماجرا کنارت خواهیم بود و از تمام توانمون برای رسیدن تو به اهدافت استفاده می‌کنیم.
بهت کمک می‌کنیم تا به بهترین نتیجه‌ای که می‌خوای برسی و از یه زندگی پرنشاط و سالم لذت ببری. ‍
یه برنامه غذایی منحصر به فرد کاملا متناسب با نیازها، سلیقه و شرایط شما طراحی می‌کنیم که خوردن غذا رو برای تو به یه تجربه لذت‌بخش تبدیل کنه.

لطفا برای امروز یه تایمی که برات مناسبه برای جلسه گوگل میت بهم بگو! ️

ما بی‌صبرانه منتظر صحبت کردن باهات و شروع این سفر هیجان‌انگیز با تو هستیم!"""



        if dp.plan in [7, 8]:
            message_text = f"""سلام {first_name} عزیز

            به جمع گرم لاکتو خوش اومدی! ما اینجا توی لاکتو دور هم جمع شدیم تا با هم سفری پرماجرا و لذت‌بخش به دنیای سلامتی رو آغاز کنیم. ‍
            
            رژیم غذایی شما تا ۲۴ ساعت آینده در دسترس قرار خواهد گرفت و از طریق قسمت مشاهده منو غذایی میتوانید آن را مشاهده بفرمایید.
            """
            


        Message.objects.create(
            sender=doctor.django_user,
            receiver=dp.user.django_user,
            text=message_text,
            time=datetime.now()
        ).save()

        # Send notifications
        try:
            TelegramNotification().send_purchase_notif(dp.user, plan=dp.plan, discount_code=dp.discount)
        except Exception:
            pass

        try:
            send_reciet_Email(dp.user.profile.email, dp.amount_after_discount, reference_number=tracking_code)
        except Exception:
            pass

        # Get user name
        user_name = f"{dp.user.profile.first_name} {dp.user.profile.last_name}" if dp.user and dp.user.profile and dp.user.profile.first_name else "کاربر"
        
        # Get plan information
        plan_name = get_plan_name(dp.plan)
        credit_days = get_plan_credit_days(dp.plan)
        
        # Get payment date in Jalali
        try:
            jalali_date = jdatetime.datetime.fromgregorian(datetime=datetime.now())
            payment_date = jalali_date.strftime('%Y/%m/%d - %H:%M')
        except:
            payment_date = datetime.now().strftime('%Y-%m-%d %H:%M')
        
        # Check if discount was applied
        discount_applied = dp.main_amount != dp.amount_after_discount
        
        # Generate deep link for successful payment
        deep_link_url = f"lacto://payment/callback?status=success&plan_id={dp.plan}&transaction_id={tracking_code}&payment_id={dp.id}&amount={dp.amount_after_discount}&plan_name={urllib.parse.quote(plan_name)}&duration={credit_days}"
        
        context = dict(
            success=True,
            title='تراکنش با موفقیت انجام شد!',
            user_name=user_name,
            plan_name=plan_name,
            plan_id=dp.plan,
            duration=credit_days,
            payment_date=payment_date,
            ref_id=tracking_code,
            transaction_id=tracking_code,
            amount=f"{dp.amount_after_discount:,}",
            discount_applied=discount_applied,
            deep_link_url=deep_link_url,
            payment_id=dp.id,
            status='success',
            debug=True  # Set to False in production
        )
        return render(request, template_name='payment_success.html', context=context)

    except DigiPayPayment.DoesNotExist:
        # Generate deep link for payment not found error
        error_message_encoded = urllib.parse.quote('تراکنش مورد نظر یافت نشد')
        deep_link_url = f"lacto://payment/callback?status=failed&error_message={error_message_encoded}"
        
        return render(request, template_name='payment_success.html', context={
            'success': False,
            'title': 'تراکنش با موفقیت انجام نشد!',
            'message': 'تراکنش مورد نظر یافت نشد',
            'error_details': 'شناسه پرداخت در سیستم یافت نشد',
            'deep_link_url': deep_link_url,
            'debug': True  # Set to False in production
        })
    except Exception as e:
        # Generate deep link for unexpected error
        error_message_encoded = urllib.parse.quote(f'خطای غیرمنتظره: {str(e)}')
        deep_link_url = f"lacto://payment/callback?status=failed&error_message={error_message_encoded}"
        
        return render(request, template_name='payment_success.html', context={
            'success': False,
            'title': 'تراکنش با موفقیت انجام نشد!',
            'message': 'خطای غیرمنتظره رخ داد',
            'error_details': str(e),
            'deep_link_url': deep_link_url,
            'debug': True  # Set to False in production
        })


@api_view(["POST"])
def successful_payment_from_hospital(request):
    """
    Hospital Integration API: Simulates successful payment actions
    This API performs the same actions as successful payment verification
    without requiring actual payment verification.
    
    Required parameters:
    - hospital_api_key: Valid hospital API key
    - phone_number: User's phone number
    - plan: Plan number (1-8)
    
    Optional parameters:
    - discount_code: Discount code to apply (if provided, will be validated and used)
    
    Returns:
    - Success: JSON response with action details
    - Error: JSON response with error message
    """
    
    # Validate required parameters
    required_params = ['hospital_api_key', 'phone_number', 'plan']
    for param in required_params:
        if param not in request.POST:
            return JsonResponse({
                'error': f'Missing required parameter: {param}',
                'status': 'error'
            }, status=400)
    
    # Get parameters
    hospital_api_key = request.POST.get('hospital_api_key')
    phone_number = request.POST.get('phone_number')
    plan = request.POST.get('plan')
    discount_code = request.POST.get('discount_code', '').strip().upper()  # Optional discount code
    
    # Validate hospital API key
    hospital_info = validate_hospital_api_key(hospital_api_key)
    if not hospital_info:
        return JsonResponse({
            'error': 'Invalid hospital API key',
            'status': 'error'
        }, status=401)
    
    # Validate plan number
    try:
        plan = int(plan)
        if plan not in [1, 2, 3, 4, 5, 6, 7, 8]:
            return JsonResponse({
                'error': 'Invalid plan number. Must be between 1-8',
                'status': 'error'
            }, status=400)
    except ValueError:
        return JsonResponse({
            'error': 'Plan must be a valid integer',
            'status': 'error'
        }, status=400)
    
    # Find user by phone number first (needed for failure tracking)
    try:
        user = AppUser.objects.filter(phone_number=phone_number).first()
        if not user:
            return JsonResponse({
                'error': 'User not found with the provided phone number',
                'status': 'error'
            }, status=404)
    except Exception as e:
        return JsonResponse({
            'error': f'Error finding user: {str(e)}',
            'status': 'error'
        }, status=500)
    
    # Validate discount code if provided
    discount_obj = None
    if discount_code:
        try:
            discount_obj = Discount.objects.filter(code=discount_code).first()
            if not discount_obj:
                return JsonResponse({
                    'error': f'Invalid discount code: {discount_code}',
                    'status': 'error'
                }, status=400)
            
            if not discount_obj.is_valid():
                return JsonResponse({
                    'error': f'Discount code {discount_code} is expired or has no remaining uses',
                    'status': 'error'
                }, status=400)
            
            # Validate discount group code matches hospital's expected group
            hospital_discount_group_code = hospital_info.get('nikan_discount_group_code')
            if hospital_discount_group_code and discount_obj.ziluck_group:
                # Get the discount group code from the discount's associated group
                discount_group_code = discount_obj.ziluck_group.name
                
                print("This is the discount group code: ", discount_group_code)
                print("This is the hospital discount group code: ", hospital_discount_group_code)
                
                if discount_group_code != hospital_discount_group_code:
                    # Create failure record for group mismatch
                    hospital_name = get_hospital_name(hospital_api_key)
                    hospital_code = get_hospital_code(hospital_api_key)
                    
                    HospitalUpgradeFailure.objects.create(
                        user=user,
                        hospital_name=hospital_name,
                        hospital_code=hospital_code,
                        plan=plan,
                        discount_code=discount_code,
                        discount_group_code=discount_group_code,
                        hospital_discount_group_code=hospital_discount_group_code,
                        failure_reason=f"Discount group mismatch: expected '{hospital_discount_group_code}', got '{discount_group_code}'",
                        failure_date=datetime.now(),
                        payment_reference=f"HOSPITAL_{hospital_code}_{datetime.now().strftime('%Y%m%d_%H%M%S')}",
                        notes=f"User attempted upgrade with discount code '{discount_code}' but group codes don't match. Hospital: {hospital_name}"
                    )
                    
                    return JsonResponse({
                        'error': f'Discount code group mismatch. Expected group: {hospital_discount_group_code}, but discount belongs to: {discount_group_code}',
                        'status': 'error'
                    }, status=400)
                
        except Exception as e:
            return JsonResponse({
                'error': f'Error validating discount code: {str(e)}',
                'status': 'error'
            }, status=500)
    
    try:
        # Handle therapist selection (same logic as payment verification)
        ts = TherapistSelection(user=user.django_user)
        try:
            previous_tss = TherapistSelection.objects.filter(user=user.django_user)
            if previous_tss.exists():
                # If user had a previous therapist, use the same one
                ts.therapist = previous_tss.last().therapist
                if previous_tss.last().therapist is None:
                    # If no therapist was assigned, use default doctor
                    doctor = Doctor.objects.all()[0]
                    ts.therapist = doctor.django_user
            else:
                # No previous therapist, assign default doctor
                doctor = Doctor.objects.all()[0]
                ts.therapist = doctor.django_user
            
            ts.save()
            user.profile.therapist = ts.therapist
            user.profile.save()
            
        except Exception as e:

            # Continue with default doctor assignment
            try:
                doctor = Doctor.objects.all()[0]
                ts.therapist = doctor.django_user
                ts.save()
                user.profile.therapist = doctor.django_user
                user.profile.save()
            except Exception as e2: 
                pass
        
        # Add credit based on plan type (same as payment verification)
        credit_days = get_plan_credit_days(plan)
        ts.add_credit(credit_days)
        ts.save()
        
        # Use discount code if provided
        if discount_obj:
            discount_obj.used()  # Reduce the number_of_discounts by 1
        
        # Calculate the final paid price
        paid_price = calculate_paid_price(plan, discount_obj)
        
        # Create hospital upgrade tracking record
        hospital_name = get_hospital_name(hospital_api_key)
        hospital_code = get_hospital_code(hospital_api_key)
        
        upgrade_notes = f"Upgrade processed through hospital integration API. Hospital: {hospital_name}"
        if discount_obj:
            upgrade_notes += f" | Discount code used: {discount_code} ({discount_obj.percentage}% off)"
        
        HospitalUserUpgrade.objects.create(
            user=user,
            hospital_name=hospital_name,
            hospital_code=hospital_code,
            plan=plan,
            credit_days_added=credit_days,
            upgrade_date=datetime.now(),
            payment_reference=f"HOSPITAL_{hospital_code}_{datetime.now().strftime('%Y%m%d_%H%M%S')}",
            discount_code=discount_obj,
            paid_price=paid_price,
            notes=upgrade_notes
        )
        
        # Send welcome message (same as payment verification)
        doctor = Doctor.objects.all()[0]
        first_name = user.profile.first_name if user.profile.first_name else "کاربر"
        
        # Default welcome message
        message_text = f"""سلام {first_name} عزیز

به جمع گرم لاکتو خوش اومدی! ما اینجا توی لاکتو دور هم جمع شدیم تا با هم سفری پرماجرا و لذت‌بخش به دنیای سلامتی رو آغاز کنیم. ‍

قبل از اینکه بریم سراغ طراحی یه برنامه غذایی اختصاصی و فوق‌العاده برای تو، می‌خوایم یه ملاقات کوتاه و صمیمی توی گوگل میت داشته باشیم.

تو این ملاقات دوستانه:
با هم حرف می‌زنیم و علایق، سبک زندگی و اهدافت رو توی مسیر سلامتی بهتر می‌شناسیم.
یه گپ دوستانه در مورد عادت‌های غذایی و فعالیت‌های روزمره‌ات می‌زنیم.
به تمام سوالات شما در مورد برنامه غذایی لاکتو و هر موضوع دیگه‌ای که ذهنت رو مشغول کرده جواب می‌دیم.

هدف ما توی لاکتو چیه؟
همراهی در مسیر سلامتی: مثل یه دوست و همراه توی این مسیر پرماجرا کنارت خواهیم بود و از تمام توانمون برای رسیدن تو به اهدافت استفاده می‌کنیم.
بهت کمک می‌کنیم تا به بهترین نتیجه‌ای که می‌خوای برسی و از یه زندگی پرنشاط و سالم لذت ببری. ‍
یه برنامه غذایی منحصر به فرد کاملا متناسب با نیازها، سلیقه و شرایط شما طراحی می‌کنیم که خوردن غذا رو برای تو به یه تجربه لذت‌بخش تبدیل کنه.

لطفا برای امروز یه تایمی که برات مناسبه برای جلسه گوگل میت بهم بگو! ️

ما بی‌صبرانه منتظر صحبت کردن باهات و شروع این سفر هیجان‌انگیز با تو هستیم!"""
        
        # Special message for plans 7 and 8 (same as payment verification)
        if plan in [7, 8]:
            message_text = f"""سلام {first_name} عزیز

به جمع گرم لاکتو خوش اومدی! ما اینجا توی لاکتو دور هم جمع شدیم تا با هم سفری پرماجرا و لذت‌بخش به دنیای سلامتی رو آغاز کنیم. ‍
            
رژیم غذایی شما تا ۲۴ ساعت آینده در دسترس قرار خواهد گرفت و از طریق قسمت مشاهده منو غذایی میتوانید آن را مشاهده بفرمایید.
"""
        
        # Create and save message
        Message.objects.create(
            sender=doctor.django_user,
            receiver=user.django_user,
            text=message_text,
            time=datetime.now()
        ).save()
        
        # Send notifications (same as payment verification)
        try:
            TelegramNotification().send_purchase_notif(user, plan=plan, discount_code=discount_obj)
        except Exception as e:
            print(f"Telegram notification error: {e}")
        
        # Send receipt email if user has email
        try:
            if user.profile.email:
                send_reciet_Email(user.profile.email, 0, reference_number="HOSPITAL_INTEGRATION")
        except Exception as e:
            print(f"Email sending error: {e}")
        
        # Return success response
        response_data = {
            'user_id': user.id,
            'phone_number': phone_number,
            'plan': plan,
            'credit_days_added': credit_days,
            'paid_price': float(paid_price),
            'therapist_assigned': ts.therapist.id if ts.therapist else None,
            'hospital_name': hospital_name,
            'hospital_code': hospital_code,
            'upgrade_tracking_id': f"HOSPITAL_{hospital_code}_{datetime.now().strftime('%Y%m%d_%H%M%S')}",
            'actions_performed': [
                'therapist_selection_created',
                'credit_added',
                'hospital_upgrade_tracked',
                'welcome_message_sent',
                'notifications_sent'
            ]
        }
        
        # Add discount information if used
        if discount_obj:
            response_data['discount_applied'] = {
                'code': discount_code,
                'percentage': discount_obj.percentage,
            }
            response_data['actions_performed'].append('discount_code_used')
        
        return JsonResponse({
            'status': 'success',
            'message': 'Payment simulation completed successfully',
            'data': response_data
        })
        
    except Exception as e:
        return JsonResponse({
            'error': f'Error processing payment simulation: {str(e)}',
            'status': 'error'
        }, status=500)


@api_view(["GET"])
def get_hospital_upgrade_statistics(request):
    """
    API endpoint to get statistics about hospital upgrades.
    This provides insights into which hospitals have upgraded which users.
    
    Query Parameters:
    - hospital_code: Optional filter for specific hospital
    - days: Number of days to look back (default: 30)
    - format: Response format ('summary' or 'detailed')
    """
    
    # Validate hospital API key if provided
    hospital_api_key = request.GET.get('hospital_api_key')
    if hospital_api_key:
        hospital_info = validate_hospital_api_key(hospital_api_key)
        if not hospital_info:
            return JsonResponse({
                'error': 'Invalid hospital API key',
                'status': 'error'
            }, status=401)
    
    # Get query parameters
    hospital_code = request.GET.get('hospital_code')
    days = int(request.GET.get('days', 30))
    format_type = request.GET.get('format', 'summary')
    
    try:
        if hospital_code:
            # Get statistics for specific hospital
            stats = HospitalUserUpgrade.get_hospital_statistics(hospital_code, days)
            hospital_name = HospitalUserUpgrade.objects.filter(
                hospital_code=hospital_code
            ).values_list('hospital_name', flat=True).first()
            
            data = {
                'hospital_code': hospital_code,
                'hospital_name': hospital_name or 'Unknown',
                'statistics': stats
            }
        else:
            # Get summary of all hospitals
            if format_type == 'detailed':
                # Detailed format with all upgrade records
                upgrades = HospitalUserUpgrade.objects.all().order_by('-upgrade_date')
                if days > 0:
                    from datetime import timedelta
                    start_date = datetime.now() - timedelta(days=days)
                    upgrades = upgrades.filter(upgrade_date__gte=start_date)
                
                data = {
                    'summary': list(HospitalUserUpgrade.get_hospitals_summary()),
                    'recent_upgrades': list(upgrades.values(
                        'user__phone_number', 'hospital_name', 'hospital_code', 
                        'plan', 'upgrade_date', 'credit_days_added'
                    )[:100])  # Limit to last 100 upgrades
                }
            else:
                # Summary format
                data = {
                    'summary': list(HospitalUserUpgrade.get_hospitals_summary()),
                    'total_hospitals': HospitalUserUpgrade.objects.values('hospital_code').distinct().count(),
                    'total_upgrades': HospitalUserUpgrade.objects.count(),
                    'total_users': HospitalUserUpgrade.objects.values('user').distinct().count()
                }
        
        return JsonResponse({
            'status': 'success',
            'data': data
        })
        
    except Exception as e:
        return JsonResponse({
            'error': f'Error retrieving hospital statistics: {str(e)}',
            'status': 'error'
        }, status=500)
