from django.db import models
from APIs.models import User as AppUser, CoinValue, AppointmentCredit
from APIs.models import ZiLuckGroup
from panel.telegram_notification import TelegramNotification
from ziluck_project.constants import MERCHANT, ZP_API_REQUEST, ZP_API_VERIFY, ZP_API_STARTPAY, CallbackURL
from datetime import datetime, timedelta
import requests
import json
from APIs.views.views import myResponse

from panel.SMSNotification import SMS
from panel.models import TherapistSelection, Doctor, Assistant


# Create your models here.


class Discount(models.Model):
    code = models.TextField(null=True, unique=True)
    created_at = models.DateTimeField(default=datetime.now)
    expire_date = models.DateTimeField(default=datetime.now)
    percentage = models.FloatField(default=0)  # percentage
    number_of_discounts = models.IntegerField(default=1)
    ziluck_group = models.ForeignKey(to=ZiLuckGroup, on_delete=models.SET_NULL, default=None, null=True, blank=True)

    def used(self):
        self.number_of_discounts -= 1
        self.save()

    def is_valid(self):
        if self.expire_date > datetime.now() and self.number_of_discounts > 0:
            return True
        else:
            return False


class ZarinPalPayment(models.Model):
    user = models.ForeignKey(to=AppUser, on_delete=models.SET_NULL, null=True)
    main_amount = models.IntegerField(null=True)
    amount_after_discount = models.IntegerField(null=True)
    discount = models.ForeignKey(to=Discount, on_delete=models.SET_NULL, null=True, default=None)
    response = models.JSONField(null=True, default=dict)
    created_at = models.DateTimeField(default=datetime.now)
    last_payment_try = models.DateTimeField(default=datetime.now)
    is_payed = models.BooleanField(default=False)
    authority = models.TextField(null=True)
    plan = models.IntegerField(choices=(('Silver', 1), ('Golden', 2), ('Platinum', 3), ('Bronze', 4)), default=1)

    def get_pay_link(self, discount_code, user_discount_code):
    # The user code is the discount code that is created based on the user's profile user_code.
        
        if len(user_discount_code) > 1 and (user_discount_code != 'NONE') and (user_discount_code != ""):
            user_discount = Discount.objects.filter(code=user_discount_code.upper())
            if user_discount.exists():
                user_discount = user_discount[0]
                if user_discount.number_of_discounts < 1000:
                    self.amount_after_discount = self.main_amount * 0.85
        
        if len(discount_code) > 1 and (discount_code != 'NONE') and (discount_code != ""):
            temp = Discount.objects.filter(code=discount_code)
            if temp.exists():
                self.discount = temp[0]
                if self.discount.is_valid():
                    self.amount_after_discount = (100 - self.discount.percentage) * self.main_amount / 100
                else:
                    self.amount_after_discount = self.main_amount
            else:
                self.amount_after_discount = self.main_amount
        else:
            self.amount_after_discount = self.main_amount

        if round(self.amount_after_discount) == 0:

            self.is_payed = True
            self.assign_to_group()
            if self.main_amount != self.amount_after_discount and self.discount.is_valid():
                self.discount.used()
            self.save()

            ts = TherapistSelection(user=self.user.django_user)
            try:
                # set a doctor for test
                doctor = Doctor.objects.all()[0]
                ts.therapist = doctor.django_user
                ts.save()

                # this is the current therapist
                self.user.profile.therapist = doctor.django_user
                self.user.profile.save()



            except Exception as e:
                print("there is no doctor!")
                pass

            if self.plan == 1:
                ts.add_credit(3 * 30)
                try:
                    AppointmentCredit.objects.create(
                        user=self.user,
                        total_credits=6
                    )
                except Exception as e:
                    print("AppointmentCredit 1:", e)
            elif self.plan == 2:
                ts.add_credit(3 * 30)
                try:
                    AppointmentCredit.objects.create(
                        user=self.user,
                        total_credits=8
                    )
                except Exception as e:
                    print("AppointmentCredit 2:", e)
            elif self.plan == 3:
                ts.add_credit(3 * 30)
                try:
                    AppointmentCredit.objects.create(
                        user=self.user,
                        total_credits=8
                    )
                except Exception as e:
                    print("AppointmentCredit 3:", e)
            elif self.plan == 4 or self.plan == 5:
                ts.add_credit(1 * 30)
                try:
                    AppointmentCredit.objects.create(
                        user=self.user,
                        total_credits=2
                    )
                except Exception as e:
                    print("AppointmentCredit 4,5:", e)
            elif self.plan == 7:
                ts.add_credit(1 * 30)
                try:
                    AppointmentCredit.objects.create(
                        user=self.user,
                        total_credits=2
                    )
                except Exception as e:
                    print("AppointmentCredit 7:", e)
            elif self.plan == 8:
                ts.add_credit(3 * 30)
                try:
                    AppointmentCredit.objects.create(
                        user=self.user,
                        total_credits=2
                    )  
                except Exception as e:
                    print("AppointmentCredit 8:", e)
                
            mghods = Assistant.objects.filter(first_name="مریم", last_name="قدس").first()
            mtabesh = Assistant.objects.filter(first_name="مهدیه", last_name="تابش").first()
                    
            print("This is the discount code: ", discount_code)
            if (discount_code == "EZMMANAGERS"):
                previous_ts = TherapistSelection.objects.all().order_by('-created_at').first()
                print("This is the previous therapist: ", previous_ts)
                if previous_ts:
                    if previous_ts.therapist.first_name == mghods.django_user.first_name:
                        self.user.profile.therapist = mtabesh.django_user
                        self.user.profile.save()
                        ts.therapist = mtabesh.django_user
                        print("This is the new therapist: ", ts.therapist)
                    elif previous_ts.therapist.first_name == mtabesh.first_name:
                        self.user.profile.therapist = mghods.django_user
                        self.user.profile.save()
                        ts.therapist = mghods.django_user
                        print("This is the new therapist: ", ts.therapist)
                    else:
                        self.user.profile.therapist = mghods.django_user
                        self.user.profile.save()
                        ts.therapist = mghods.django_user
                        print("The previous therapist is not one of the managers, so the default therapist is chosen: ", ts.therapist)

            ts.save()

            try:
                SMS().ziluck_welcome_sms(self.user)
            except Exception as e:
                print("This is the exception: ", e)

            try:
                TelegramNotification().send_purchase_notif(self.user, plan=self.plan, discount_code=self.discount)
            except Exception as e:
                print("This is the telegram exception: ", e)
                


            data = {
                'link': "is paid",
                'message': "پرداخت با موفقیت انجام شد"
            }

            return myResponse.OK('لینک پرداخت', data)

        data = {
            'amount': round(self.amount_after_discount),
            'callback_url': "https://lacto.ir/zarinpal/callback",
            'merchant_id': MERCHANT,
            'description': 'خرید پلن شماره {} لاکتو'.format(self.plan) + " برای کاربر {}".format(self.user),
        }
    
        
        req_header = {"accept": "application/json",
                      "content-type": "application/json"}
        response = requests.post(url="https://lacto.ir/wp-json/lacto/v1/zarinpal-request", json=data,
                                headers=req_header).json()
        self.last_payment_try = datetime.now()
        self.response = response
        self.save()
        print("This is the response: ", response)
        if response['errors'] == []:
            self.authority = response['data']['authority']
            self.save()
            data = {
                'link': ZP_API_STARTPAY.format(authority=self.authority),
                'main_amount': self.main_amount,
                'amount_after_discount': round(self.amount_after_discount)
            }
            return myResponse.OK('لینک پرداخت', data)
        else:
            return myResponse.Error("پرداخت با مشکل مواجه شد", 1001)

    def assign_to_group(self):
        if self.discount.ziluck_group is not None:
            self.discount.ziluck_group.user_set.add(self.user.django_user)


class ZarinPalPaymentCoin(models.Model):
    user = models.ForeignKey(to=AppUser, on_delete=models.SET_NULL, null=True)
    main_amount = models.FloatField(null=True)
    amount_after_discount = models.FloatField(null=True)
    discount = models.ForeignKey(to=Discount, on_delete=models.SET_NULL, null=True, default=None)
    response = models.JSONField(null=True, default=dict)
    created_at = models.DateTimeField(default=datetime.now)
    last_payment_try = models.DateTimeField(default=datetime.now)
    is_payed = models.BooleanField(default=False)
    authority = models.TextField(null=True)
    coin_amount = models.IntegerField(default=0)

    def calculate_main_amount(self):
        coin_value = CoinValue.objects.latest('created_at').price_of_coin
        self.main_amount = self.coin_amount * coin_value

    def get_pay_link_coin(self, discount_code):
        self.calculate_main_amount()

        print(f"Coin Amount: {self.coin_amount}, Main Amount: {self.main_amount}")

        # Discount application remains similar to the previous implementation
        if discount_code and discount_code != 'NONE':
            temp = Discount.objects.filter(code=discount_code)
            if temp.exists():
                self.discount = temp[0]
                if self.discount.is_valid():
                    self.amount_after_discount = (100 - self.discount.percentage) * self.main_amount / 100
                else:
                    self.amount_after_discount = self.main_amount
            else:
                self.amount_after_discount = self.main_amount
        else:
            self.amount_after_discount = self.main_amount

        if round(self.amount_after_discount) == 0:
            self.is_payed = True
            self.assign_to_group()
            if self.main_amount != self.amount_after_discount and self.discount.is_valid():
                self.discount.used()
            self.save()

            # Send a success message
            data = {
                'link': "is paid",
                'message': "پرداخت با موفقیت انجام شد"
            }
            return myResponse.OK('لینک پرداخت', data)

        # Prepare data for payment request
        data = {
            'amount': self.amount_after_discount,
            'callback_url': CallbackURL + 'zarinpal/',
            'merchant_id': MERCHANT,
            'description': 'خرید با کوین برای کاربر {}'.format(self.user),
        }

        print(f"Request Data: {data}")

        req_header = {"accept": "application/json", "content-type": "application/json"}
        response = requests.post(url=ZP_API_REQUEST, data=json.dumps(data), headers=req_header).json()

        self.last_payment_try = datetime.now()
        self.response = response
        self.save()

        if response['data']['message'] == 'Success':
            self.authority = response['data']['authority']
            self.save()
            data = {
                'link': ZP_API_STARTPAY.format(authority=self.authority),
                'main_amount': self.main_amount,
                'amount_after_discount': self.amount_after_discount
            }
            return myResponse.OK('لینک پرداخت', data)
        else:
            return myResponse.Error("پرداخت با مشکل مواجه شد", 1001)

    def assign_to_group(self):
        if self.discount and self.discount.ziluck_group is not None:
            self.discount.ziluck_group.user_set.add(self.user.django_user)



class DigiPayPayment(models.Model):
    user = models.ForeignKey(to=AppUser, on_delete=models.SET_NULL, null=True)
    main_amount = models.IntegerField(null=True)
    amount_after_discount = models.IntegerField(null=True)
    discount = models.ForeignKey(to=Discount, on_delete=models.SET_NULL, null=True, default=None)
    response = models.JSONField(null=True, default=dict)
    created_at = models.DateTimeField(default=datetime.now)
    last_payment_try = models.DateTimeField(default=datetime.now)
    is_payed = models.BooleanField(default=False)
    payment_id = models.TextField(null=True)
    plan = models.IntegerField(choices=(('Silver', 1), ('Golden', 2), ('Platinum', 3), ('Bronze', 4)), default=1)
    digipay_token = models.TextField(null=True)
    tracking_code = models.TextField(null=True)

    def get_pay_link(self, discount_code, user_discount_code, digipay_token, provider_id):
        # Apply discount logic similar to ZarinPalPayment
        if len(user_discount_code) > 1 and (user_discount_code != 'NONE') and (user_discount_code != ""):
            user_discount = Discount.objects.filter(code=user_discount_code.upper())
            if user_discount.exists():
                user_discount = user_discount[0]
                if user_discount.number_of_discounts < 1000:
                    self.amount_after_discount = self.main_amount * 0.85
        

        if len(discount_code) > 1 and (discount_code != 'NONE') and (discount_code != ""):
            temp = Discount.objects.filter(code=discount_code)
            if temp.exists():
                self.discount = temp[0]
                if self.discount.is_valid():
                    self.amount_after_discount = (100 - self.discount.percentage) * self.main_amount / 100
                else:
                    self.amount_after_discount = self.main_amount
            else:
                self.amount_after_discount = self.main_amount
        else:
            self.amount_after_discount = self.main_amount
        
        self.digipay_token = digipay_token
        self.save()

        # Get user's phone number
        if not self.user.phone_number:
            return myResponse.Error("شماره موبایل کاربر در دسترس نیست", 1002)

        # Map plan_id to plan name
        plan_names = {
            1: "Three Month Silver",
            2: "Three Month Gold",
            3: "Three Month Platinum",
            4: "Monthly Silver",
            5: "Monthly Gold",
            6: "Monthly Platinum",
            7: "Monthly Bronze",
            8: "Three Month Bronze"
        }
        
        plan_name = plan_names.get(self.plan, "Unknown Plan")

        print("This is the digipay token: ", digipay_token)

        # Prepare API request
        headers = {
            "Authorization": f"Bearer {digipay_token}",
            "Content-Type": "application/json",
            "Agent": "WEB",
            "Digipay-Version": "2022-02-02"
        }
        
        data = {
            "cellNumber": str(self.user.phone_number),
            "amount": self.amount_after_discount,
            "providerId": provider_id,
            "callbackUrl": "https://api.ziluck.info/payment/verify/digipay/payment/",
            "basketDetailsDto": {
                "items": [
                    {
                        "sellerId": "15",
                        "supplierId": "15",
                        "productCode": "465",
                        "brand": f"Lacto - {plan_name}",
                        "productType": 1,
                        "count": 1,
                        "categoryId": "1"
                    }
                ],
                "basketId": "1"
            }
        }

        print("This is the data: ", json.dumps(data, indent=2, ensure_ascii=False))

        # try:
        response = requests.post(
            "https://api.mydigipay.com/digipay/api/tickets/business?type=11",
            headers=headers,
            data=json.dumps(data, ensure_ascii=False),  # Convert data to JSON string
            json=None  # Don't use the json parameter since we're using data
        )
        
        print("Response Status Code:", response.status_code)
        print("Response Headers:", response.headers)
        print("Response Content:", response.text)
        
        response.raise_for_status()
        response_data = response.json()
        
        self.last_payment_try = datetime.now()
        self.response = response_data
        self.payment_id = provider_id  # Store the provider_id as payment_id
        self.tracking_code = response_data.get("trackingCode")
        self.save()

        if response_data.get("result", {}).get("status") == 0:  # Check for successful status
            data = {
                'link': response_data.get("redirectUrl"),  # Use the redirectUrl from response
                'main_amount': self.main_amount,
                'amount_after_discount': self.amount_after_discount
            }
            return myResponse.OK('لینک پرداخت', data)
        else:
            error_message = response_data.get("result", {}).get("message", "پرداخت با مشکل مواجه شد")
            return myResponse.Error(error_message, 1001)

    def assign_to_group(self):
        if self.discount and self.discount.ziluck_group is not None:
            self.discount.ziluck_group.user_set.add(self.user.django_user)


class TimeSlotPayment(models.Model):
    PAYMENT_TYPE_CHOICES = (
        ('coin', 'Coin Payment'),
        ('money', 'Money Payment'),
    )
    
    STATUS_CHOICES = (
        ('pending', 'Pending Assignment'),
        ('assigned', 'Assigned to Therapist'),
        ('completed', 'Completed'),
        ('cancelled', 'Cancelled'),
    )
    
    user = models.ForeignKey(to=AppUser, on_delete=models.SET_NULL, null=True)
    time_slot = models.ForeignKey('panel.TimeSlot', on_delete=models.CASCADE)
    payment_type = models.CharField(max_length=10, choices=PAYMENT_TYPE_CHOICES)
    coin_amount = models.IntegerField(default=0)  # 2000 coins per timeslot
    money_amount = models.IntegerField(default=0)  # 200,000 toman per timeslot
    is_paid = models.BooleanField(default=False)
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='pending')
    assigned_therapist = models.ForeignKey('panel.Assistant', on_delete=models.SET_NULL, null=True, blank=True)
    created_at = models.DateTimeField(default=datetime.now)
    payment_date = models.DateTimeField(null=True, blank=True)
    assigned_date = models.DateTimeField(null=True, blank=True)
    
    # For money payments - similar to existing payment models
    authority = models.TextField(null=True, blank=True)  # For ZarinPal
    payment_id = models.TextField(null=True, blank=True)  # For DigiPay
    response = models.JSONField(null=True, default=dict)
    
    def __str__(self):
        return f"{self.user} - {self.time_slot} - {self.payment_type}"
    
    def process_coin_payment(self):
        """Process coin payment for timeslot"""
        if self.user.profile.coins >= self.coin_amount:
            self.user.profile.coins -= self.coin_amount
            self.user.profile.save()
            
            # Mark timeslot as purchased but not yet assigned
            self.time_slot.is_available = 2  # 2 = purchased, waiting for assignment
            self.time_slot.save()
            
            self.is_paid = True
            self.payment_date = datetime.now()
            self.save()
            
            return True
        return False
    
    def process_money_payment(self):
        """Process money payment for timeslot"""
        # This would integrate with existing payment gateways
        # Similar to ZarinPalPayment but for timeslots
        pass
    
    def assign_to_therapist(self, therapist):
        """Assign the purchased timeslot to a therapist"""
        self.assigned_therapist = therapist
        self.status = 'assigned'
        self.assigned_date = datetime.now()
        
        # Update the timeslot
        self.time_slot.patient = self.user
        self.time_slot.is_available = 0  # 0 = booked
        self.time_slot.save()
        
        self.save()
        
        return True


class PaymentPlans(models.Model):
    name = models.TextField(default=None, null=True, verbose_name="Plan Name")
    plan_database_id = models.IntegerField(verbose_name="Plan Database ID", help_text="Database identifier for the plan", null=True, blank=True)
    price = models.DecimalField(max_digits=10, decimal_places=2, verbose_name="Price")
    items_included = models.TextField(verbose_name="Items Included", help_text="Separate items with & sign")
    items_excluded = models.TextField(verbose_name="Items Excluded", help_text="Separate items with & sign")
    promotion = models.TextField(verbose_name="Promotion", blank=True, null=True)
    price_with_promotion = models.DecimalField(max_digits=10, decimal_places=2, verbose_name="Price with Promotion", blank=True, null=True)
    subscription_period = models.CharField(max_length=50, verbose_name="Subscription Period", help_text="e.g., Monthly, Yearly, etc.")
    is_active = models.BooleanField(default=True, verbose_name="Is Active")
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        verbose_name = "Payment Plan"
        verbose_name_plural = "Payment Plans"
        ordering = ['price']

    def __str__(self):
        return f"{self.name} - {self.price}"

    def get_items_included_list(self):
        """Return items included as a list split by &"""
        if self.items_included:
            return [item.strip() for item in self.items_included.split('&') if item.strip()]
        return []

    def get_items_excluded_list(self):
        """Return items excluded as a list split by &"""
        if self.items_excluded:
            return [item.strip() for item in self.items_excluded.split('&') if item.strip()]
        return []


class HospitalUserUpgrade(models.Model):
    """
    Model to track users who received premium access through hospital payments.
    This provides a reliable way to associate premium users with the hospital that paid for them.
    """
    user = models.ForeignKey(to=AppUser, on_delete=models.CASCADE, related_name='hospital_upgrades')
    hospital_name = models.CharField(max_length=100, help_text="Name of the hospital that paid for the upgrade")
    hospital_code = models.CharField(max_length=20, help_text="Unique code identifying the hospital")
    plan = models.IntegerField(choices=(
        (1, 'Silver - 3 Months'),
        (2, 'Golden - 3 Months'), 
        (3, 'Platinum - 3 Months'),
        (4, 'Silver - 1 Month'),
        (5, 'Golden - 1 Month'),
        (6, 'Platinum - 1 Month'),
        (7, 'Bronze - 1 Month'),
        (8, 'Bronze - 3 Months')
    ), help_text="Plan number that was purchased")
    credit_days_added = models.IntegerField(help_text="Number of credit days added to user account")
    upgrade_date = models.DateTimeField(default=datetime.now, help_text="When the upgrade was processed")
    payment_reference = models.CharField(max_length=100, blank=True, null=True, help_text="Hospital's internal payment reference")
    discount_code = models.ForeignKey(to=Discount, on_delete=models.SET_NULL, null=True, blank=True, help_text="Discount code used for this upgrade")
    paid_price = models.DecimalField(max_digits=10, decimal_places=2, null=True, blank=True, help_text="Final price paid by the user after discounts and promotions")
    notes = models.TextField(blank=True, null=True, help_text="Additional notes about the upgrade")
    
    class Meta:
        verbose_name = "Hospital User Upgrade"
        verbose_name_plural = "Hospital User Upgrades"
        ordering = ['-upgrade_date']
        indexes = [
            models.Index(fields=['user', 'upgrade_date']),
            models.Index(fields=['hospital_code', 'upgrade_date']),
            models.Index(fields=['plan', 'upgrade_date']),
        ]
    
    def __str__(self):
        return f"{self.user} - {self.hospital_name} - Plan {self.plan} - {self.upgrade_date.strftime('%Y-%m-%d')}"
    
    @property
    def is_active(self):
        """Check if the upgrade is still active (within credit period)"""
        from datetime import timedelta
        expiry_date = self.upgrade_date + timedelta(days=self.credit_days_added)
        return datetime.now() < expiry_date
    
    @property
    def days_remaining(self):
        """Calculate remaining days of premium access"""
        from datetime import timedelta
        expiry_date = self.upgrade_date + timedelta(days=self.credit_days_added)
        remaining = (expiry_date - datetime.now()).days
        return max(0, remaining)
    
    @classmethod
    def get_active_upgrades_by_hospital(cls, hospital_code):
        """Get all active upgrades for a specific hospital"""
        return cls.objects.filter(
            hospital_code=hospital_code,
            upgrade_date__gte=datetime.now() - timedelta(days=365)  # Last year
        ).filter(
            upgrade_date__lte=datetime.now() + timedelta(days=365)  # Next year
        )
    
    @classmethod
    def get_hospital_statistics(cls, hospital_code, days=30):
        """Get statistics for a hospital over a specified period"""
        from datetime import timedelta
        
        start_date = datetime.now() - timedelta(days=days)
        upgrades = cls.objects.filter(
            hospital_code=hospital_code,
            upgrade_date__gte=start_date
        )
        
        return {
            'total_upgrades': upgrades.count(),
            'total_users': upgrades.values('user').distinct().count(),
            'plans_distribution': list(upgrades.values('plan').annotate(
                count=models.Count('plan')
            ).order_by('-count')),
            'total_credit_days': sum(upgrades.values_list('credit_days_added', flat=True)),
            'period_days': days
        }
    
    @classmethod
    def get_user_hospital_history(cls, user):
        """Get complete hospital upgrade history for a user"""
        return cls.objects.filter(user=user).order_by('-upgrade_date')
    
    @classmethod
    def get_hospitals_summary(cls):
        """Get summary of all hospitals and their upgrade counts"""
        return cls.objects.values('hospital_code', 'hospital_name').annotate(
            total_upgrades=models.Count('id'),
            unique_users=models.Count('user', distinct=True),
            latest_upgrade=models.Max('upgrade_date')
        ).order_by('-total_upgrades')


class HospitalUpgradeFailure(models.Model):
    """
    Model to track failed hospital upgrade attempts.
    This provides a way to monitor and analyze why upgrades failed.
    """
    user = models.ForeignKey(to=AppUser, on_delete=models.CASCADE, related_name='hospital_upgrade_failures')
    hospital_name = models.CharField(max_length=100, help_text="Name of the hospital that attempted the upgrade")
    hospital_code = models.CharField(max_length=20, help_text="Unique code identifying the hospital")
    plan = models.IntegerField(choices=(
        (1, 'Silver - 3 Months'),
        (2, 'Golden - 3 Months'), 
        (3, 'Platinum - 3 Months'),
        (4, 'Silver - 1 Month'),
        (5, 'Golden - 1 Month'),
        (6, 'Platinum - 1 Month'),
        (7, 'Bronze - 1 Month'),
        (8, 'Bronze - 3 Months')
    ), help_text="Plan number that was attempted")
    discount_code = models.CharField(max_length=50, blank=True, null=True, help_text="Discount code that was used")
    discount_group_code = models.CharField(max_length=50, blank=True, null=True, help_text="Discount group code from the discount")
    hospital_discount_group_code = models.CharField(max_length=50, blank=True, null=True, help_text="Hospital's expected discount group code")
    failure_reason = models.CharField(max_length=200, help_text="Reason why the upgrade failed")
    failure_date = models.DateTimeField(default=datetime.now, help_text="When the failure occurred")
    payment_reference = models.CharField(max_length=100, blank=True, null=True, help_text="Hospital's internal payment reference")
    notes = models.TextField(blank=True, null=True, help_text="Additional notes about the failure")
    
    class Meta:
        verbose_name = "Hospital Upgrade Failure"
        verbose_name_plural = "Hospital Upgrade Failures"
        ordering = ['-failure_date']
        indexes = [
            models.Index(fields=['user', 'failure_date']),
            models.Index(fields=['hospital_code', 'failure_date']),
            models.Index(fields=['failure_reason', 'failure_date']),
        ]
    
    def __str__(self):
        return f"{self.user} - {self.hospital_name} - Plan {self.plan} - {self.failure_reason} - {self.failure_date.strftime('%Y-%m-%d')}"
    
    @classmethod
    def get_failures_by_hospital(cls, hospital_code, days=30):
        """Get failures for a specific hospital over a specified period"""
        from datetime import timedelta
        
        start_date = datetime.now() - timedelta(days=days)
        failures = cls.objects.filter(
            hospital_code=hospital_code,
            failure_date__gte=start_date
        )
        
        return {
            'total_failures': failures.count(),
            'unique_users': failures.values('user').distinct().count(),
            'failure_reasons': list(failures.values('failure_reason').annotate(
                count=models.Count('failure_reason')
            ).order_by('-count')),
            'period_days': days
        }
    
    @classmethod
    def get_discount_group_mismatch_failures(cls, days=30):
        """Get failures specifically due to discount group mismatches"""
        from datetime import timedelta
        
        start_date = datetime.now() - timedelta(days=days)
        return cls.objects.filter(
            failure_reason__icontains='discount group',
            failure_date__gte=start_date
        ).order_by('-failure_date')
