from django.contrib import admin
from django.utils.timezone import now
from payment.models import ZarinPalPayment, Discount, ZarinPalPaymentCoin, DigiPayPayment, PaymentPlans, HospitalUserUpgrade, HospitalUpgradeFailure  # Explicit imports


# Custom filter for Payment Status
class PaymentStatusFilter(admin.SimpleListFilter):
    title = "Payment Status"
    parameter_name = "is_payed"

    def lookups(self, request, model_admin):
        return [("paid", "Paid"), ("unpaid", "Unpaid")]

    def queryset(self, request, queryset):
        if self.value() == "paid":
            return queryset.filter(is_payed=True)
        elif self.value() == "unpaid":
            return queryset.filter(is_payed=False)
        return queryset


# Payment Model Admin
class ZarinPalPaymentAdmin(admin.ModelAdmin):
    list_display = ("user", "plan", "main_amount", "amount_after_discount", 'discount', "is_payed", "created_at")
    list_filter = (PaymentStatusFilter, "plan", "discount")
    search_fields = ["user__username", "user__email"]


admin.site.register(ZarinPalPayment, ZarinPalPaymentAdmin)  # ✅ Registered once
# Payment Model Admin


class ZarinPalPaymentCoinAdmin(admin.ModelAdmin):
    list_display = ("user", "main_amount", "amount_after_discount", 'discount', "is_payed", "created_at",  'coin_amount')
    list_filter = (PaymentStatusFilter, "discount")
    search_fields = ["user__username", "user__email"]


admin.site.register(ZarinPalPaymentCoin, ZarinPalPaymentCoinAdmin)  # ✅ Registered once


# Custom filter for Expired Discounts
class ExpiredDiscountFilter(admin.SimpleListFilter):
    title = "Discount Validity"
    parameter_name = "validity"

    def lookups(self, request, model_admin):
        return [("valid", "Valid"), ("expired", "Expired")]

    def queryset(self, request, queryset):
        if self.value() == "valid":
            return queryset.filter(expire_date__gte=now(), number_of_discounts__gt=0)
        elif self.value() == "expired":
            return queryset.filter(expire_date__lt=now())
        return queryset


# Discount Model Admin
class DiscountAdmin(admin.ModelAdmin):
    list_display = ("code", "percentage", "number_of_discounts", "expire_date")
    list_filter = (ExpiredDiscountFilter,)


admin.site.register(Discount, DiscountAdmin)  # ✅ Registered once


# DigiPay Payment Model Admin
class DigiPayPaymentAdmin(admin.ModelAdmin):
    list_display = ("user", "plan", "main_amount", "amount_after_discount", "discount", "is_payed", "created_at", "tracking_code")
    list_filter = (PaymentStatusFilter, "plan", "discount")
    search_fields = ["user__username", "user__email", "tracking_code", "payment_id"]
    readonly_fields = ("created_at", "last_payment_try", "response", "digipay_token")
    
    def get_plan_display(self, obj):
        plan_choices = {
            1: "Silver",
            2: "Golden", 
            3: "Platinum",
            4: "Bronze"
        }
        return plan_choices.get(obj.plan, "Unknown")
    get_plan_display.short_description = "Plan"


admin.site.register(DigiPayPayment, DigiPayPaymentAdmin)  # Register DigiPayPayment model


@admin.register(PaymentPlans)
class PaymentPlansAdmin(admin.ModelAdmin):
    list_display = ['name', 'plan_database_id', 'price', 'price_with_promotion', 'subscription_period', 'is_active', 'created_at']
    list_filter = ['is_active', 'subscription_period', 'created_at']
    search_fields = ['name', 'plan_database_id', 'items_included', 'items_excluded']
    list_editable = ['is_active', 'price', 'price_with_promotion', 'plan_database_id']
    
    fieldsets = (
        ('Basic Information', {
            'fields': ('name', 'plan_database_id', 'price', 'subscription_period', 'is_active')
        }),
        ('Items', {
            'fields': ('items_included', 'items_excluded'),
            'description': 'Separate multiple items with & sign (e.g., Feature 1 & Feature 2 & Feature 3)'
        }),
        ('Promotion', {
            'fields': ('promotion', 'price_with_promotion'),
            'classes': ('collapse',)
        }),
        ('Timestamps', {
            'fields': ('created_at', 'updated_at'),
            'classes': ('collapse',),
            'readonly_fields': ('created_at', 'updated_at')
        })
    )
    
    readonly_fields = ('created_at', 'updated_at')
    
    def get_queryset(self, request):
        return super().get_queryset(request).order_by('price')


@admin.register(HospitalUserUpgrade)
class HospitalUserUpgradeAdmin(admin.ModelAdmin):
    list_display = ("user", "hospital_name", "hospital_code", "plan", "credit_days_added", "paid_price", "upgrade_date", "is_active", "days_remaining")
    list_filter = ("hospital_code", "plan", "upgrade_date", "hospital_name", "discount_code")
    search_fields = ["user__django_user__username", "user__phone_number", "user__profile__first_name", "user__profile__last_name", "hospital_name", "hospital_code", "payment_reference", "notes"]
    readonly_fields = ("upgrade_date", "is_active", "days_remaining")
    date_hierarchy = "upgrade_date"
    
    fieldsets = (
        ('User Information', {
            'fields': ('user', 'plan', 'credit_days_added')
        }),
        ('Hospital Information', {
            'fields': ('hospital_name', 'hospital_code', 'payment_reference')
        }),
        ('Payment Information', {
            'fields': ('discount_code', 'paid_price'),
            'classes': ('collapse',)
        }),
        ('Timing', {
            'fields': ('upgrade_date', 'is_active', 'days_remaining'),
            'classes': ('collapse',)
        }),
        ('Additional Information', {
            'fields': ('notes',),
            'classes': ('collapse',)
        })
    )
    
    def get_plan_display(self, obj):
        plan_choices = {
            1: "Silver",
            2: "Golden", 
            3: "Platinum",
            4: "Bronze",
            5: "Monthly",
            6: "Quarterly",
            7: "Special Monthly",
            8: "Special Quarterly"
        }
        return plan_choices.get(obj.plan, "Unknown")
    get_plan_display.short_description = "Plan"
    
    def has_add_permission(self, request):
        """Disable manual creation - only allow through API"""
        return False
    
    def has_change_permission(self, request, obj=None):
        """Allow editing notes and other fields"""
        return True
    
    def has_delete_permission(self, request, obj=None):
        """Allow deletion for audit purposes"""
        return True
    
    def get_queryset(self, request):
        """Order by most recent upgrades first"""
        return super().get_queryset(request).order_by('-upgrade_date')
    
    def get_search_results(self, request, queryset, search_term):
        """Enhanced search functionality for HospitalUserUpgrade"""
        queryset, use_distinct = super().get_search_results(request, queryset, search_term)
        
        # Additional search in related fields
        if search_term:
            # Search in discount code if it exists
            queryset |= self.model.objects.filter(
                discount_code__code__icontains=search_term
            )
            
            # Search in user profile email
            queryset |= self.model.objects.filter(
                user__profile__email__icontains=search_term
            )
            
            use_distinct = True
        
        return queryset, use_distinct


@admin.register(HospitalUpgradeFailure)
class HospitalUpgradeFailureAdmin(admin.ModelAdmin):
    list_display = ("user", "hospital_name", "hospital_code", "plan", "failure_reason", "failure_date", "discount_code")
    list_filter = ("hospital_code", "plan", "failure_reason", "failure_date", "hospital_name")
    search_fields = ["user__django_user__username", "user__phone_number", "user__profile__first_name", "user__profile__last_name", "hospital_name", "hospital_code", "discount_code", "discount_group_code", "hospital_discount_group_code", "failure_reason", "payment_reference", "notes"]
    readonly_fields = ("failure_date",)
    date_hierarchy = "failure_date"
    ordering = ["-failure_date"]
    
    fieldsets = (
        ('User Information', {
            'fields': ('user', 'plan')
        }),
        ('Hospital Information', {
            'fields': ('hospital_name', 'hospital_code', 'payment_reference')
        }),
        ('Discount Information', {
            'fields': ('discount_code', 'discount_group_code', 'hospital_discount_group_code'),
            'classes': ('collapse',)
        }),
        ('Failure Details', {
            'fields': ('failure_reason', 'failure_date', 'notes')
        })
    )
    
    def get_plan_display(self, obj):
        plan_choices = {
            1: "Silver - 3 Months",
            2: "Golden - 3 Months", 
            3: "Platinum - 3 Months",
            4: "Silver - 1 Month",
            5: "Golden - 1 Month",
            6: "Platinum - 1 Month",
            7: "Bronze - 1 Month",
            8: "Bronze - 3 Months"
        }
        return plan_choices.get(obj.plan, "Unknown")
    get_plan_display.short_description = "Plan"
    
    def has_add_permission(self, request):
        """Disable manual creation - only allow through API"""
        return False
    
    def has_change_permission(self, request, obj=None):
        """Allow editing notes for administrative purposes"""
        return True
    
    def has_delete_permission(self, request, obj=None):
        """Allow deletion for audit purposes"""
        return True
    
    def get_queryset(self, request):
        """Order by most recent failures first"""
        return super().get_queryset(request).order_by('-failure_date')
    
    def get_search_results(self, request, queryset, search_term):
        """Enhanced search functionality for HospitalUpgradeFailure"""
        queryset, use_distinct = super().get_search_results(request, queryset, search_term)
        
        # Additional search in related fields
        if search_term:
            # Search in user profile email
            queryset |= self.model.objects.filter(
                user__profile__email__icontains=search_term
            )
            
            # Search for partial matches in failure reason
            queryset |= self.model.objects.filter(
                failure_reason__icontains=search_term
            )
            
            use_distinct = True
        
        return queryset, use_distinct
