# assistant/tagging.py
import google.generativeai as palm
import json
import re
from datetime import datetime
import math
import google.generativeai as palm
from ziluck_project.constants import GOOGLE_API_KEY

palm.configure(api_key=GOOGLE_API_KEY)


# Helper function to round CR with threshold 50
def round_cr(cr):
    """
    Rounds the CR value to the nearest 100 with a threshold of 50.
    - Remainder < 50: round down
    - Remainder = 50: round down
    - Remainder > 50: round up
    """
    base = math.floor(cr / 100) * 100
    remainder = cr - base
    if remainder < 50:
        return base
    elif remainder > 50:
        return base + 100
    else:  # remainder == 50
        return base

all_alergic_tags = [
        'AL_Dairy_Butter', 'AL_Dairy_Cheese', 'AL_Dairy_Coconut', 'AL_Dairy_CoconutMilk', 'AL_Dairy_Cream',
        'AL_Dairy_CreamCheese', 'AL_Dairy_IceCream', 'AL_Dairy_Mayonnaise', 'AL_Dairy_Milk', 'AL_Dairy_Milk_Chocolate',
        'AL_Dairy_Whey', 'AL_Dairy_Yogurt', 'AL_Egg', 'AL_Eggs_Egg', 'AL_Eggs_Mayonnaise', 'AL_Fish_Crab',
        'AL_Fish_Fish', 'AL_Fish_Shrimp', 'AL_Fish_Tuna', 'AL_Grain_Barley', 'AL_Grain_Bean', 'AL_Grain_Buckwheat',
        'AL_Grain_Chickpea', 'AL_Grain_Corn', 'AL_Grain_Lentil', 'AL_Grain_Mung', 'AL_Grain_Oat', 'AL_Grain_Oats',
        'AL_Grain_Quinoa', 'AL_Grain_Rice', 'AL_Grain_Rye', 'AL_Grain_Soy', 'AL_Grain_Unspecified', 'AL_Grain_Wheat',
        'AL_Grains_Wheat', 'AL_Legume_Beans', 'AL_Legume_Lentil', 'AL_Legume_Pea', 'AL_Legume_RedBean', 'AL_Legume_Soy',
        'AL_Legume_Split_Peas', 'AL_Legume_SplitPea', 'AL_Meat_Bacon', 'AL_Meat_Beef', 'AL_Meat_Chicken', 'AL_Meat_Ham',
        'AL_Meat_Lamb', 'AL_Meat_Pepperoni', 'AL_Meat_Pork', 'AL_Meat_Processed', 'AL_Meat_Sausage', 'AL_Meat_Turkey',
        'AL_Meat_Unspecified', 'AL_Meat_Varies', 'AL_Meat_Various', 'AL_Nuts_Almond', 'AL_Nuts_Coconut',
        'AL_Nuts_Hazelnut',
        'AL_Nuts_Nuts', 'AL_Nuts_Peanut', 'AL_Nuts_Pine_Nuts', 'AL_Nuts_Pistachio', 'AL_Nuts_Sesame',
        'AL_Nuts_Unspecified',
        'AL_Nuts_Varies', 'AL_Nuts_Various', 'AL_Nuts_Walnut', 'AL_Poultry_Chicken', 'AL_Seeds_Chia',
        'AL_Shellfish_Shrimp',
        'AL_Dairy_Cream_Cheese', 'AL_Dairy_Creamer', 'AL_Dairy_Goat_Milk', 'AL_Dairy_Kashk', 'AL_Dairy_Kefir',
        'AL_Eggs_Whole', 'AL_Eggs_Whole_Eggs', 'AL_Eggs_Yolk', 'AL_Fish_Anchovy', 'AL_Fish_Carp', 'AL_Fish_Catfish',
        'AL_Fish_Caviar', 'AL_Fish_Croaker', 'AL_Fish_Flatfish', 'AL_Fish_Flounder', 'AL_Fish_Grouper',
        'AL_Fish_Herring',
        'AL_Fish_KingMackerel', 'AL_Fish_Mackerel', 'AL_Fish_Mullet', 'AL_Fish_Pike', 'AL_Fish_Porgy', 'AL_Fish_Roe',
        'AL_Fish_Salmon', 'AL_Fish_Sardine', 'AL_Fish_Shark', 'AL_Fish_Snapper', 'AL_Fish_Swordfish', 'AL_Fish_Tilapia',
        'AL_Fish_Trout', 'AL_Fish_Unspecified', 'AL_Fish_Whitefish', 'AL_Grain_Bulgur', 'AL_Grain_Cereal',
        'AL_Grain_Millet',
        'AL_Grain_Pea', 'AL_Grain_SplitPea', 'AL_Grains_Barley', 'AL_Grains_Brown_Rice', 'AL_Grains_Bulgur',
        'AL_Grains_Chickpea', 'AL_Grains_Corn', 'AL_Grains_Millet', 'AL_Grains_Oats', 'AL_Grains_Rice',
        'AL_Legume_Bean',
        'AL_Legume_Chickpea', 'AL_Legume_Fava', 'AL_Legume_FavaBean', 'AL_Legume_Peas', 'AL_Legume_Unspecified',
        'AL_Meat_Camel', 'AL_Meat_Duck', 'AL_Meat_Fat', 'AL_Meat_Fish', 'AL_Meat_Goose', 'AL_Meat_Ground',
        'AL_Meat_Kidney',
        'AL_Meat_Offal', 'AL_Meat_Organ', 'AL_Meat_Ostrich', 'AL_Meat_Pigeon', 'AL_Meat_Quail', 'AL_Meat_Sheep',
        'AL_Meat_Veal', 'AL_Nuts_Acorn', 'AL_Nuts_Almonds', 'AL_Nuts_Cashew', 'AL_Nuts_Chestnut', 'AL_Nuts_Hazelnuts',
        'AL_Nuts_Hemp', 'AL_Nuts_Mixed', 'AL_Nuts_Nutmeg', 'AL_Nuts_Peanuts', 'AL_Nuts_Pecan', 'AL_Other_Sesame',
        'AL_Poultry_Duck', 'AL_Poultry_Offal', 'AL_Seafood_Fish', 'AL_Seeds_Cottonseed', 'AL_Seeds_Flax',
        'AL_Seeds_Flaxseed',
        'AL_Seeds_Poppy', 'AL_Seeds_Pumpkin', 'AL_Seeds_Safflower', 'AL_Seeds_Sesame', 'AL_Seeds_Sunflower',
        'AL_Seeds_Watermelon', 'AL_Sesame_Sesame', 'AL_Soy_Soy', 'AL_Soy_SoySauce', 'AL_Soy_Soybean', 'AL_Soy_Soybeans',
        # Newly added allergen fields from Food model
        'Allergens_Eggplant', 'Allergens_Mushroom', 'Allergens_Dried_fruits',
    ]

cr_list = [1200, 1300, 1400, 1500, 1600, 1700, 1800, 1900, 2000, 2100, 2200, 2300, 2400, 2500, 2600]
calorie_distributions = {
    1200: {"B": 250, "L": 350, "D": 300, "S1": 150, "S2": 150},
    1300: {"B": 300, "L": 350, "D": 350, "S1": 150, "S2": 150},
    1400: {"B": 350, "L": 400, "D": 350, "S1": 150, "S2": 150},
    1500: {"B": 400, "L": 400, "D": 400, "S1": 150, "S2": 150},
    1600: {"B": 400, "L": 450, "D": 400, "S1": 150, "S2": 200},
    1700: {"B": 400, "L": 500, "D": 450, "S1": 150, "S2": 200},
    1800: {"B": 400, "L": 550, "D": 450, "S1": 200, "S2": 200},
    1900: {"B": 450, "L": 550, "D": 500, "S1": 200, "S2": 200},
    2000: {"B": 500, "L": 600, "D": 500, "S1": 200, "S2": 200},
    2100: {"B": 500, "L": 600, "D": 550, "S1": 200, "S2": 250},
    2200: {"B": 500, "L": 650, "D": 600, "S1": 200, "S2": 250},
    2300: {"B": 550, "L": 650, "D": 600, "S1": 250, "S2": 250},
    2400: {"B": 550, "L": 700, "D": 600, "S1": 250, "S2": 300},
    2500: {"B": 550, "L": 700, "D": 650, "S1": 300, "S2": 300},
    2600: {"B": 600, "L": 700, "D": 700, "S1": 300, "S2": 300},
}

tags = {
        "Allergens": {
            "AL_Dairy_Butter": 0, "AL_Dairy_Cheese": 0, "AL_Dairy_Coconut": 0, "AL_Dairy_CoconutMilk": 0,
            "AL_Dairy_Cream": 0, "AL_Dairy_CreamCheese": 0, "AL_Dairy_IceCream": 0, "AL_Dairy_Mayonnaise": 0,
            "AL_Dairy_Milk": 0, "AL_Dairy_Milk_Chocolate": 0, "AL_Dairy_Whey": 0, "AL_Dairy_Yogurt": 0,
            "AL_Egg": 0, "AL_Eggs_Egg": 0, "AL_Eggs_Mayonnaise": 0, "AL_Fish_Crab": 0, "AL_Fish_Fish": 0,
            "AL_Fish_Shrimp": 0, "AL_Fish_Tuna": 0, "AL_Grain_Barley": 0, "AL_Grain_Bean": 0,
            "AL_Grain_Buckwheat": 0, "AL_Grain_Chickpea": 0, "AL_Grain_Corn": 0, "AL_Grain_Lentil": 0,
            "AL_Grain_Mung": 0, "AL_Grain_Oat": 0, "AL_Grain_Oats": 0, "AL_Grain_Quinoa": 0,
            "AL_Grain_Rice": 0, "AL_Grain_Rye": 0, "AL_Grain_Soy": 0, "AL_Grain_Unspecified": 0,
            "AL_Grain_Wheat": 0, "AL_Grains_Wheat": 0, "AL_Legume_Beans": 0, "AL_Legume_Lentil": 0,
            "AL_Legume_Pea": 0, "AL_Legume_RedBean": 0, "AL_Legume_Soy": 0, "AL_Legume_Split_Peas": 0,
            "AL_Legume_SplitPea": 0, "AL_Meat_Bacon": 0, "AL_Meat_Beef": 0, "AL_Meat_Chicken": 0,
            "AL_Meat_Ham": 0, "AL_Meat_Lamb": 0, "AL_Meat_Pepperoni": 0, "AL_Meat_Pork": 0,
            "AL_Meat_Processed": 0, "AL_Meat_Sausage": 0, "AL_Meat_Turkey": 0, "AL_Meat_Unspecified": 0,
            "AL_Meat_Varies": 0, "AL_Meat_Various": 0, "AL_Nuts_Almond": 0, "AL_Nuts_Coconut": 0,
            "AL_Nuts_Hazelnut": 0, "AL_Nuts_Nuts": 0, "AL_Nuts_Peanut": 0, "AL_Nuts_Pine_Nuts": 0,
            "AL_Nuts_Pistachio": 0, "AL_Nuts_Sesame": 0, "AL_Nuts_Unspecified": 0, "AL_Nuts_Varies": 0,
            "AL_Nuts_Various": 0, "AL_Nuts_Walnut": 0, "AL_Poultry_Chicken": 0, "AL_Seeds_Chia": 0,
            "AL_Shellfish_Shrimp": 0, "AL_Dairy_Cream_Cheese": 0, "AL_Dairy_Creamer": 0, "AL_Dairy_Goat_Milk": 0,
            "AL_Dairy_Kashk": 0, "AL_Dairy_Kefir": 0, "AL_Eggs_Whole": 0, "AL_Eggs_Whole_Eggs": 0,
            "AL_Eggs_Yolk": 0, "AL_Fish_Anchovy": 0, "AL_Fish_Carp": 0, "AL_Fish_Catfish": 0, "AL_Fish_Caviar": 0,
            "AL_Fish_Croaker": 0, "AL_Fish_Flatfish": 0, "AL_Fish_Flounder": 0, "AL_Fish_Grouper": 0,
            "AL_Fish_Herring": 0, "AL_Fish_KingMackerel": 0, "AL_Fish_Mackerel": 0, "AL_Fish_Mullet": 0,
            "AL_Fish_Pike": 0, "AL_Fish_Porgy": 0, "AL_Fish_Roe": 0, "AL_Fish_Salmon": 0, "AL_Fish_Sardine": 0,
            "AL_Fish_Shark": 0, "AL_Fish_Snapper": 0, "AL_Fish_Swordfish": 0, "AL_Fish_Tilapia": 0,
            "AL_Fish_Trout": 0, "AL_Fish_Unspecified": 0, "AL_Fish_Whitefish": 0, "AL_Grain_Bulgur": 0,
            "AL_Grain_Cereal": 0, "AL_Grain_Millet": 0, "AL_Grain_Pea": 0, "AL_Grain_SplitPea": 0,
            "AL_Grains_Barley": 0, "AL_Grains_Brown_Rice": 0, "AL_Grains_Bulgur": 0, "AL_Grains_Chickpea": 0,
            "AL_Grains_Corn": 0, "AL_Grains_Millet": 0, "AL_Grains_Oats": 0, "AL_Grains_Rice": 0,
            "AL_Legume_Bean": 0, "AL_Legume_Chickpea": 0, "AL_Legume_Fava": 0, "AL_Legume_FavaBean": 0,
            "AL_Legume_Peas": 0, "AL_Legume_Unspecified": 0, "AL_Meat_Camel": 0, "AL_Meat_Duck": 0,
            "AL_Meat_Fat": 0, "AL_Meat_Fish": 0, "AL_Meat_Goose": 0, "AL_Meat_Ground": 0, "AL_Meat_Kidney": 0,
            "AL_Meat_Offal": 0, "AL_Meat_Organ": 0, "AL_Meat_Ostrich": 0, "AL_Meat_Pigeon": 0, "AL_Meat_Quail": 0,
            "AL_Meat_Sheep": 0, "AL_Meat_Veal": 0, "AL_Nuts_Acorn": 0, "AL_Nuts_Almonds": 0, "AL_Nuts_Cashew": 0,
            "AL_Nuts_Chestnut": 0, "AL_Nuts_Hazelnuts": 0, "AL_Nuts_Hemp": 0, "AL_Nuts_Mixed": 0,
            "AL_Nuts_Nutmeg": 0, "AL_Nuts_Peanuts": 0, "AL_Nuts_Pecan": 0, "AL_Other_Sesame": 0,
            "AL_Poultry_Duck": 0, "AL_Poultry_Offal": 0, "AL_Seafood_Fish": 0, "AL_Seeds_Cottonseed": 0,
            "AL_Seeds_Flax": 0, "AL_Seeds_Flaxseed": 0, "AL_Seeds_Poppy": 0, "AL_Seeds_Pumpkin": 0,
            "AL_Seeds_Safflower": 0, "AL_Seeds_Sesame": 0, "AL_Seeds_Sunflower": 0, "AL_Seeds_Watermelon": 0,
            "AL_Sesame_Sesame": 0, "AL_Soy_Soy": 0, "AL_Soy_SoySauce": 0, "AL_Soy_Soybean": 0, "AL_Soy_Soybeans": 0
        },
        "FoodCategories": {
            "Sweets": 0, "Dessert": 0, "Sweet": 0, "Fruits": 0, "Vegetables": 0, "Salad": 0,
            "Dairy": 0, "Eggs": 0, "Nuts_and_Seeds": 0, "Seafood": 0, "Poultry": 0, "Legumes": 0,
            "Processed_Meats": 0, "Grains": 0, "Beans_and_Lentils": 0
        },
        "DietaryPreferences": {
            "Diet_Low_Sugar": 0, "Diet_Low_Fat": 0, "Diet_Low_Carb": 0, "Diet_Vegan": 0,
            "Diet_Vegetarian": 0, "Diet_Gluten_Free": 0, "Diet_Plant_Based": 0, "Low_carb": 0,
            "Low_fat": 0, "Gluten_free": 0, "Lactose_free": 0, "Vegan": 0, "Vegetarian": 0
        },
        "Diseases": {
            "Disease_Diabetes": 0, "Disease_Hypertension": 0, "Disease_High_Cholesterol": 0, "Disease_CVD": 0,
            "Disease_Kidney": 0, "Disease_Celiac": 0, "Disease_IBS": 0, "Disease_IBD": 0, "Disease_Gout": 0,
            "Disease_Lactose_Intolerance": 0, "Disease_NAFLD": 0, "Disease_Anemia": 0, "Disease_Osteoporosis": 0,
            "Disease_Arthritis": 0, "Disease_Thyroid": 0, "Disease_Autoimmune": 0, "Disease_CancerPrevention": 0,
            "Disease_MentalHealth": 0, "Disease_PCOS": 0, "Disease_Obesity": 0, "Disease_Hypercholesterolemia": 0,
            "Disease_MS": 0, "Disease_Hypothyroidism": 0
        }
    }

def generate_health_tags(questionnaire_responses, profile=None):
    # Initialize tags in the required structure
    
    
    print(f"XOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXO")
    print("Generating health tags...")
    if profile is None:
        print("The profile is none.")
    else:
        print("Profile passed √√")
    print(f"XOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXOXO")

    # If questionnaire_responses is None, normalize to dict
    if questionnaire_responses is None:
        questionnaire_responses = {}

    if not questionnaire_responses and profile is None:
        return tags

    # Helper function to clean response (remove brackets and split if needed)
    def clean_response(response):
        if not response:
            return []
        if isinstance(response, (list, tuple, set)):
            return [item.strip() if isinstance(item, str) else item for item in response]
        if isinstance(response, str):
            cleaned = response.strip("[]")
            if not cleaned:
                return []
            return [item.strip() for item in cleaned.split(",")]
        return [response]

    def get_profile_diseases(profile_obj):
        if profile_obj is None:
            return []
        raw_codes = getattr(profile_obj, "_diseases", None)
        if not raw_codes:
            return []
        codes = [code for code in raw_codes.split("&") if code.strip()]
        disease_names = []
        disease_choices = getattr(profile_obj, "DISEASE_CHOICES", {})
        for code in codes:
            try:
                idx = int(code.strip())
            except (ValueError, TypeError):
                continue
            name = disease_choices.get(idx)
            if name:
                disease_names.append(name.strip())
        return disease_names

    # Process LIMITED_AVOIDED_FOODS, DIETARY_RESTRICTIONS_AVOIDANCES, or DIETARY_ALLERGY 
    avoided_food_keys = ("LIMITED_AVOIDED_FOODS", "DIETARY_RESTRICTIONS_AVOIDANCES", "DIETARY_ALLERGY")
    if any(key in questionnaire_responses for key in avoided_food_keys):
        avoided_foods = []
        for key in avoided_food_keys:
            avoided_foods.extend(clean_response(questionnaire_responses.get(key)))
        avoided_foods = [food for food in avoided_foods if food]
        for food in avoided_foods:
            if isinstance(food, str):
                food = food.strip().strip('"').strip("'")
            else:
                food = str(food).strip()
            if food == "لبنیات":
                allergen_tags = [
                    "AL_Dairy_Butter", "AL_Dairy_Coconut", "AL_Dairy_CoconutMilk",
                    "AL_Dairy_Cream", "AL_Dairy_IceCream", "AL_Dairy_Mayonnaise",
                    "AL_Dairy_Milk", "AL_Dairy_Milk_Chocolate", "AL_Dairy_Whey", "AL_Dairy_Yogurt", "AL_Dairy_Creamer", "AL_Dairy_Goat_Milk", "AL_Dairy_Kashk",
                    "AL_Dairy_Kefir"
                ]
                for tag in allergen_tags:
                    tags["Allergens"][tag] = 1
                tags["FoodCategories"]["Dairy"] = 1
                tags["DietaryPreferences"]["Lactose_free"] = 1
            elif food == "گلوتن":
                allergen_tags = [
                    "AL_Grain_Wheat", "AL_Grain_Barley", "AL_Grain_Rye", "AL_Grain_Bulgur",
                    "AL_Grains_Wheat", "AL_Grains_Barley", "AL_Grains_Bulgur"
                ]
                for tag in allergen_tags:
                    tags["Allergens"][tag] = 1
                tags["FoodCategories"]["Grains"] = 1
                tags["DietaryPreferences"]["Diet_Gluten_Free"] = 1
                tags["DietaryPreferences"]["Gluten_free"] = 1
            elif food == "تخم مرغ":
                allergen_tags = [
                    "AL_Egg", "AL_Eggs_Egg", "AL_Eggs_Mayonnaise", "AL_Eggs_Whole",
                    "AL_Eggs_Whole_Eggs", "AL_Eggs_Yolk"
                ]
                for tag in allergen_tags:
                    tags["Allergens"][tag] = 1
                tags["FoodCategories"]["Eggs"] = 1
            elif food == "سویا":
                allergen_tags = [
                    "AL_Grain_Soy", "AL_Legume_Soy", "AL_Soy_Soy", "AL_Soy_SoySauce",
                    "AL_Soy_Soybean", "AL_Soy_Soybeans"
                ]
                for tag in allergen_tags:
                    tags["Allergens"][tag] = 1
            elif food == "آجیل و دانه‌های روغنی" or food == "آجیل" or food == "آجیل و دانه های روغنی":
                allergen_tags = [
                    "AL_Nuts_Almond", "AL_Nuts_Coconut", "AL_Nuts_Hazelnut", "AL_Nuts_Nuts",
                    "AL_Nuts_Peanut", "AL_Nuts_Pine_Nuts", "AL_Nuts_Pistachio", "AL_Nuts_Sesame",
                    "AL_Nuts_Unspecified", "AL_Nuts_Varies", "AL_Nuts_Various", "AL_Nuts_Walnut",
                    "AL_Nuts_Acorn", "AL_Nuts_Almonds", "AL_Nuts_Cashew", "AL_Nuts_Chestnut",
                    "AL_Nuts_Hazelnuts", "AL_Nuts_Hemp", "AL_Nuts_Mixed", "AL_Nuts_Nutmeg",
                    "AL_Nuts_Peanuts", "AL_Nuts_Pecan", "AL_Seeds_Chia", "AL_Seeds_Cottonseed",
                    "AL_Seeds_Flax", "AL_Seeds_Flaxseed", "AL_Seeds_Poppy", "AL_Seeds_Pumpkin",
                    "AL_Seeds_Safflower", "AL_Seeds_Sesame", "AL_Seeds_Sunflower", "AL_Seeds_Watermelon",
                    "AL_Other_Sesame", "AL_Sesame_Sesame"
                ]
                for tag in allergen_tags:
                    tags["Allergens"][tag] = 1
                tags["FoodCategories"]["Nuts_and_Seeds"] = 1
            elif food == "گوشت قرمز":
                allergen_tags = [
                    "AL_Meat_Bacon", "AL_Meat_Beef", "AL_Meat_Ham", "AL_Meat_Lamb",
                    "AL_Meat_Pepperoni", "AL_Meat_Pork", "AL_Meat_Processed", "AL_Meat_Sausage",
                    "AL_Meat_Unspecified", "AL_Meat_Varies", "AL_Meat_Various", "AL_Meat_Camel",
                    "AL_Meat_Fat", "AL_Meat_Ground", "AL_Meat_Kidney", "AL_Meat_Offal",
                    "AL_Meat_Organ", "AL_Meat_Sheep", "AL_Meat_Veal"
                ]
                for tag in allergen_tags:
                    tags["Allergens"][tag] = 1
                tags["FoodCategories"]["Processed_Meats"] = 1
            elif food == "غذاهای فرآوری‌شده" or food == "غذاهای فرآوری شده":
                allergen_tags = [
                    "AL_Meat_Processed", "AL_Meat_Sausage", "AL_Meat_Bacon", "AL_Meat_Ham",
                    "AL_Meat_Ground"
                ]
                for tag in allergen_tags:
                    tags["Allergens"][tag] = 1
                tags["FoodCategories"]["Processed_Meats"] = 1
            elif food == "گوشت سفید" or food == "گوشت سفید/ماکیان":
                allergen_tags = [
                    "AL_Meat_Chicken", "AL_Meat_Turkey", "AL_Meat_Duck", "AL_Meat_Goose",
                    "AL_Meat_Pigeon", "AL_Meat_Quail", "AL_Meat_Ostrich",
                    "AL_Poultry_Chicken", "AL_Poultry_Duck", "AL_Poultry_Offal"
                ]
                for tag in allergen_tags:
                    tags["Allergens"][tag] = 1
                tags["FoodCategories"]["Poultry"] = 1
            elif food == "غذاهای دریایی" or food == "غذای دریایی" or food == "ماهی":
                allergen_tags = [
                    "AL_Fish_Crab", "AL_Fish_Fish", "AL_Fish_Shrimp", "AL_Fish_Tuna",
                    "AL_Fish_Anchovy", "AL_Fish_Carp", "AL_Fish_Catfish", "AL_Fish_Caviar",
                    "AL_Fish_Croaker", "AL_Fish_Flatfish", "AL_Fish_Flounder", "AL_Fish_Grouper",
                    "AL_Fish_Herring", "AL_Fish_KingMackerel", "AL_Fish_Mackerel", "AL_Fish_Mullet",
                    "AL_Fish_Pike", "AL_Fish_Porgy", "AL_Fish_Roe", "AL_Fish_Salmon",
                    "AL_Fish_Sardine", "AL_Fish_Shark", "AL_Fish_Snapper", "AL_Fish_Swordfish",
                    "AL_Fish_Tilapia", "AL_Fish_Trout", "AL_Fish_Unspecified", "AL_Fish_Whitefish",
                    "AL_Seafood_Fish", "AL_Shellfish_Shrimp"
                ]
                for tag in allergen_tags:
                    tags["Allergens"][tag] = 1
                tags["FoodCategories"]["Seafood"] = 1
            elif food == "حبوبات":
                allergen_tags = [
                    "AL_Grain_Bean", "AL_Grain_Chickpea", "AL_Grain_Lentil", "AL_Grain_Mung",
                    "AL_Grain_Pea", "AL_Grain_SplitPea", "AL_Legume_Bean", "AL_Legume_Beans",
                    "AL_Legume_Chickpea", "AL_Legume_Fava", "AL_Legume_FavaBean", "AL_Legume_Lentil",
                    "AL_Legume_Pea", "AL_Legume_Peas", "AL_Legume_RedBean", "AL_Legume_Split_Peas",
                    "AL_Legume_SplitPea", "AL_Legume_Unspecified"
                ]
                for tag in allergen_tags:
                    tags["Allergens"][tag] = 1
                tags["FoodCategories"]["Legumes"] = 1
                tags["FoodCategories"]["Beans_and_Lentils"] = 1
            elif food == "بادمجان":
                tags["Allergens"]["Allergens_Eggplant"] = 1
            elif food == "قارچ":
                tags["Allergens"]["Allergens_Mushroom"] = 1
            elif food == "میوه‌های خشک" or food == "میوه خشک":
                tags["Allergens"]["Allergens_Dried_fruits"] = 1
                tags["FoodCategories"]["Fruits"] = 1
            elif food == "قند" or food == "شیرینی جات" or food == "قند/شیرینی جات":
                tags["FoodCategories"]["Sweets"] = 1
                tags["FoodCategories"]["Dessert"] = 1
                tags["FoodCategories"]["Sweet"] = 1
                tags["DietaryPreferences"]["Diet_Low_Sugar"] = 1
            elif food == "کربوهیدرات کم" or food == "کم کربوهیدرات":
                tags["DietaryPreferences"]["Diet_Low_Carb"] = 1
                tags["DietaryPreferences"]["Low_carb"] = 1
            elif food == "کم چربی" or food == "کم چرب":
                tags["DietaryPreferences"]["Diet_Low_Fat"] = 1
                tags["DietaryPreferences"]["Low_fat"] = 1
            elif food == "کم سدیم" or food == "کم نمک":
                # No direct low-sodium tag; skipping as no suitable tag fits
                pass
            elif food == "غذاهایی با طبع سرد":
                tags["FoodCategories"]["Fruits"] = 1
                tags["FoodCategories"]["Vegetables"] = 1
            elif food == "میوه و سبزی":
                tags["FoodCategories"]["Fruits"] = 1
                tags["FoodCategories"]["Vegetables"] = 1
                tags["FoodCategories"]["Salad"] = 1
                tags["DietaryPreferences"]["Diet_Plant_Based"] = 1
            elif food == "گیاهخوار":
                allergen_tags = [
                    "AL_Meat_Bacon", "AL_Meat_Beef", "AL_Meat_Chicken", "AL_Meat_Ham",
                    "AL_Meat_Lamb", "AL_Meat_Pepperoni", "AL_Meat_Pork", "AL_Meat_Processed",
                    "AL_Meat_Sausage", "AL_Meat_Turkey", "AL_Meat_Unspecified", "AL_Meat_Varies",
                    "AL_Meat_Various", "AL_Meat_Camel", "AL_Meat_Duck", "AL_Meat_Fat",
                    "AL_Meat_Fish", "AL_Meat_Goose", "AL_Meat_Ground", "AL_Meat_Kidney",
                    "AL_Meat_Offal", "AL_Meat_Organ", "AL_Meat_Ostrich", "AL_Meat_Pigeon",
                    "AL_Meat_Quail", "AL_Meat_Sheep", "AL_Meat_Veal", "AL_Poultry_Chicken",
                    "AL_Poultry_Duck", "AL_Poultry_Offal", "AL_Fish_Crab", "AL_Fish_Fish",
                    "AL_Fish_Shrimp", "AL_Fish_Tuna", "AL_Fish_Anchovy", "AL_Fish_Carp",
                    "AL_Fish_Catfish", "AL_Fish_Caviar", "AL_Fish_Croaker", "AL_Fish_Flatfish",
                    "AL_Fish_Flounder", "AL_Fish_Grouper", "AL_Fish_Herring", "AL_Fish_KingMackerel",
                    "AL_Fish_Mackerel", "AL_Fish_Mullet", "AL_Fish_Pike", "AL_Fish_Porgy",
                    "AL_Fish_Roe", "AL_Fish_Salmon", "AL_Fish_Sardine", "AL_Fish_Shark",
                    "AL_Fish_Snapper", "AL_Fish_Swordfish", "AL_Fish_Tilapia", "AL_Fish_Trout",
                    "AL_Fish_Unspecified", "AL_Fish_Whitefish", "AL_Seafood_Fish", "AL_Shellfish_Shrimp"
                ]
                for tag in allergen_tags:
                    tags["Allergens"][tag] = 1
                tags["FoodCategories"]["Processed_Meats"] = 1
                tags["FoodCategories"]["Poultry"] = 1
                tags["FoodCategories"]["Seafood"] = 1
                tags["DietaryPreferences"]["Diet_Vegetarian"] = 1
                tags["DietaryPreferences"]["Vegetarian"] = 1
                tags["DietaryPreferences"]["Diet_Plant_Based"] = 1
            elif food == "وگان":
                allergen_tags = [
                    "AL_Dairy_Butter", "AL_Dairy_Cheese", "AL_Dairy_Coconut", "AL_Dairy_CoconutMilk",
                    "AL_Dairy_Cream", "AL_Dairy_CreamCheese", "AL_Dairy_IceCream", "AL_Dairy_Mayonnaise",
                    "AL_Dairy_Milk", "AL_Dairy_Milk_Chocolate", "AL_Dairy_Whey", "AL_Dairy_Yogurt",
                    "AL_Dairy_Cream_Cheese", "AL_Dairy_Creamer", "AL_Dairy_Goat_Milk", "AL_Dairy_Kashk",
                    "AL_Dairy_Kefir", "AL_Egg", "AL_Eggs_Egg", "AL_Eggs_Mayonnaise", "AL_Eggs_Whole",
                    "AL_Eggs_Whole_Eggs", "AL_Eggs_Yolk", "AL_Meat_Bacon", "AL_Meat_Beef",
                    "AL_Meat_Chicken", "AL_Meat_Ham", "AL_Meat_Lamb", "AL_Meat_Pepperoni",
                    "AL_Meat_Pork", "AL_Meat_Processed", "AL_Meat_Sausage", "AL_Meat_Turkey",
                    "AL_Meat_Unspecified", "AL_Meat_Varies", "AL_Meat_Various", "AL_Meat_Camel",
                    "AL_Meat_Duck", "AL_Meat_Fat", "AL_Meat_Fish", "AL_Meat_Goose", "AL_Meat_Ground",
                    "AL_Meat_Kidney", "AL_Meat_Offal", "AL_Meat_Organ", "AL_Meat_Ostrich",
                    "AL_Meat_Pigeon", "AL_Meat_Quail", "AL_Meat_Sheep", "AL_Meat_Veal",
                    "AL_Poultry_Chicken", "AL_Poultry_Duck", "AL_Poultry_Offal", "AL_Fish_Crab",
                    "AL_Fish_Fish", "AL_Fish_Shrimp", "AL_Fish_Tuna", "AL_Fish_Anchovy",
                    "AL_Fish_Carp", "AL_Fish_Catfish", "AL_Fish_Caviar", "AL_Fish_Croaker",
                    "AL_Fish_Flatfish", "AL_Fish_Flounder", "AL_Fish_Grouper", "AL_Fish_Herring",
                    "AL_Fish_KingMackerel", "AL_Fish_Mackerel", "AL_Fish_Mullet", "AL_Fish_Pike",
                    "AL_Fish_Porgy", "AL_Fish_Roe", "AL_Fish_Salmon", "AL_Fish_Sardine",
                    "AL_Fish_Shark", "AL_Fish_Snapper", "AL_Fish_Swordfish", "AL_Fish_Tilapia",
                    "AL_Fish_Trout", "AL_Fish_Unspecified", "AL_Fish_Whitefish", "AL_Seafood_Fish",
                    "AL_Shellfish_Shrimp"
                ]
                for tag in allergen_tags:
                    tags["Allergens"][tag] = 1
                tags["FoodCategories"]["Dairy"] = 1
                tags["FoodCategories"]["Eggs"] = 1
                tags["FoodCategories"]["Processed_Meats"] = 1
                tags["FoodCategories"]["Poultry"] = 1
                tags["FoodCategories"]["Seafood"] = 1
                tags["DietaryPreferences"]["Diet_Vegan"] = 1
                tags["DietaryPreferences"]["Vegan"] = 1
                tags["DietaryPreferences"]["Diet_Plant_Based"] = 1
                tags["DietaryPreferences"]["Lactose_free"] = 1
            elif food == "غیره" or food == "دیگر":
                # Other requires custom user input; skipping unless specific tags provided
                pass
            elif food == "هیچ‌کدام" or food == "هیچکدام":
                # None means no restrictions; no tags are set
                pass
            
            
    # Process diseases from questionnaire and profile
    illnesses = []
    if "CHRONIC_ILLNESSES" in questionnaire_responses:
        illnesses.extend(clean_response(questionnaire_responses["CHRONIC_ILLNESSES"]))
    illnesses.extend(get_profile_diseases(profile))

    for illness in illnesses:
        if not illness:
            continue
        normalized = illness.strip().replace("\u200c", "")
        normalized_lower = normalized.lower()

        if normalized in ("دیابت", "دیابت/پیش دیابت") or "diabetes" in normalized_lower:
            tags["Diseases"]["Disease_Diabetes"] = 1 # DM
        elif normalized in ("فشار خون بالا", "فشار خون") or "hypertension" in normalized_lower or "blood pressure" in normalized_lower:
            tags["Diseases"]["Disease_Hypertension"] = 1 #HTN
        elif normalized in ("کلسترول چربی و کمی دفع پروتئین", "هایپرلیپیدمی/چربی خون", "هایپرلیپیدمی/چربی", "چربی خون", "چربی خون بالا", "چربی") or "hyperlipidemia" in normalized_lower or "cholesterol" in normalized_lower:
            tags["Diseases"]["Disease_High_cholesterol"] = 1 #high cholesterol
            tags["Diseases"]["Disease_Hypercholesterolemia"] = 1 # high cholesterol
        elif normalized == "بیماری قلبی" or "cardiovascular" in normalized_lower or "heart disease" in normalized_lower:
            tags["Diseases"]["Disease_CVD"] = 1 #CVD
        elif normalized in ("IBS (سندروم روده تحریک پذیر)", "IBS", "سندروم روده تحریک پذیر", "نفخ") or "irritable bowel" in normalized_lower:
            tags["Diseases"]["Disease_IBS"] = 1
        elif normalized == "نقرس" or "gout" in normalized_lower:
            tags["Diseases"]["Disease_Gout"] = 1 #Gout
        elif normalized == "کبد چرب" or "fatty liver" in normalized_lower or "nafld" in normalized_lower:
            tags["Diseases"]["Disease_NAFLD"] = 1 #NAFLD
        elif normalized in ("PCOS (سندروم تخمدان پلی‌کیستیک)", "PCOS") or "polycystic" in normalized_lower:
            tags["Diseases"]["Disease_PCOS"] = 1 #PCOS
        elif normalized == "چاقی" or "obesity" in normalized_lower:
            tags["Diseases"]["Disease_Obesity"] = 1
        elif normalized == "IBD" or "inflammatory bowel disease" in normalized_lower:
            tags["Diseases"]["Disease_IBD"] = 1
        elif normalized == "Kidney Disease" or "kidney" in normalized_lower:
            tags["Diseases"]["Disease_Kidney"] = 1
        elif normalized in ("Celiac", "سیلیاک", "سلیاک") or "celiac" in normalized_lower:
            tags["Diseases"]["Disease_Celiac"] = 1
        elif normalized in ("Anemia", "کم خونی") or "anemia" in normalized_lower:
            tags["Diseases"]["Disease_Anemia"] = 1
        elif normalized in ("Hypothyroidism", "کم کاری تیروئید") or "hypothyroidism" in normalized_lower:
            tags["Diseases"]["Disease_Thyroid"] = 1
            tags["Diseases"]["Disease_Hypothyroidism"] = 1
        elif normalized == "MS" or "multiple sclerosis" in normalized_lower:
            tags["Diseases"]["Disease_MS"] = 1

    return tags

# arthritis, hypothyroidism, nafld, ibd, kidney disease
# Things that we don't have them in our questionnaire:
# celiac, anemia, ms, 

def calculate_cr(birth_date, weight, height, gender, activity_level, type_of_treatment):
    """
    Calculates calorie requirements (CR) based on the logic of Function 2,
    but only returns the CR value.

    Args:
        birth_date (datetime): User's birth date.
        weight (float): User's weight in kg.
        height (float): User's height in cm.
        gender (bool): False for male, True for female.
        activity_level (float): Activity level factor.
        type_of_treatment (int): Treatment type (0, 1, or other).  This is kept
            for API compatibility, but its value doesn't directly affect CR.

    Returns:
        float: The calculated calorie requirement (CR).
    """
    age = datetime.now().year - birth_date.year
    metabolism_energy = 1.0  # Default value
    cr_adjustment = 0.0

    boy = [15, 15, 15, 16, 16, 16, 15, 15, 15, 15, 14, 14, 14, 14, 13.5, 13.5, 13.5, 13.5]
    girl = [15, 15, 15, 16, 16, 16, 17, 17, 17, 17, 16, 16, 16, 16, 17, 17, 17, 17]

    if 1 < age < 19:
        if not gender:  # Male
            bmr = boy[age - 1] * height  # Use age - 1, like Function 2
        else:  # Female
            bmr = girl[age - 1] * height  # Use age - 1, like Function 2
        cr = bmr  # directly assign bmr to cr, instead of using separated CR variable
    else:
        bmi = weight / ((height / 100) ** 2)

        if bmi < 18.5:
            cr_adjustment = 500
            metabolism_energy = 1.1
        elif bmi < 25:
            cr_adjustment = 0
        elif bmi <= 30:
            cr_adjustment = -500
        else:  # BMI > 30
            cr_adjustment = -500

        if not gender:  # Male
            bmr = (10 * weight) + (6.25 * height) - (5 * age) + 5
        else:  # Female
            bmr = (10 * weight) + (6.25 * height) - (5 * age) - 161

        cr = cr_adjustment + round(bmr * activity_level * metabolism_energy, 2)

    return cr


def get_tag_values(questionnaire):
    """
    Generates allergen and disease tag values (0 or 1) based on the user's questionnaire.

    Args:
        questionnaire (dict): The user's questionnaire data as a dictionary.

    Returns:
        dict: A dictionary containing allergen and disease tags with their values (0 or 1).
              Returns None if an error occurs during processing.
    """
    base_prompt = """
You are an expert in nutrition, dietary restrictions, and data processing. Your task is to analyze a user's questionnaire response (provided as a JSON object) and generate a set of tag values—each being either 0 or 1—for both allergens and diseases. These tag values will later be used to filter a food dataset, ensuring that only food items meeting the user's dietary needs are shown.

The two groups of tags are defined as follows:

**Allergen Tags**
List: AL_Dairy_Butter, AL_Dairy_Cheese, AL_Dairy_Coconut, AL_Dairy_CoconutMilk, AL_Dairy_Cream, AL_Dairy_CreamCheese, AL_Dairy_IceCream, AL_Dairy_Mayonnaise, AL_Dairy_Milk, AL_Dairy_Milk_Chocolate, AL_Dairy_Whey, AL_Dairy_Yogurt, AL_Egg, AL_Eggs_Egg, AL_Eggs_Mayonnaise, AL_Fish_Crab, AL_Fish_Fish, AL_Fish_Shrimp, AL_Fish_Tuna, AL_Grain_Barley, AL_Grain_Bean, AL_Grain_Buckwheat, AL_Grain_Chickpea, AL_Grain_Corn, AL_Grain_Lentil, AL_Grain_Mung, AL_Grain_Oat, AL_Grain_Oats, AL_Grain_Quinoa, AL_Grain_Rice, AL_Grain_Rye, AL_Grain_Soy, AL_Grain_Unspecified, AL_Grain_Wheat, AL_Grains_Wheat, AL_Legume_Beans, AL_Legume_Lentil, AL_Legume_Pea, AL_Legume_RedBean, AL_Legume_Soy, AL_Legume_Split_Peas, AL_Legume_SplitPea, AL_Meat_Bacon, AL_Meat_Beef, AL_Meat_Chicken, AL_Meat_Ham, AL_Meat_Lamb, AL_Meat_Pepperoni, AL_Meat_Pork, AL_Meat_Processed, AL_Meat_Sausage, AL_Meat_Turkey, AL_Meat_Unspecified, AL_Meat_Varies, AL_Meat_Various, AL_Nuts_Almond, AL_Nuts_Coconut, AL_Nuts_Hazelnut, AL_Nuts_Nuts, AL_Nuts_Peanut, AL_Nuts_Pine_Nuts, AL_Nuts_Pistachio, AL_Nuts_Sesame, AL_Nuts_Unspecified, AL_Nuts_Varies, AL_Nuts_Various, AL_Nuts_Walnut, AL_Poultry_Chicken, AL_Seeds_Chia, AL_Shellfish_Shrimp, AL_Dairy_Cream_Cheese, AL_Dairy_Creamer, AL_Dairy_Goat_Milk, AL_Dairy_Kashk, AL_Dairy_Kefir, AL_Eggs_Whole, AL_Eggs_Whole_Eggs, AL_Eggs_Yolk, AL_Fish_Anchovy, AL_Fish_Carp, AL_Fish_Catfish, AL_Fish_Caviar, AL_Fish_Croaker, AL_Fish_Flatfish, AL_Fish_Flounder, AL_Fish_Grouper, AL_Fish_Herring, AL_Fish_KingMackerel, AL_Fish_Mackerel, AL_Fish_Mullet, AL_Fish_Pike, AL_Fish_Porgy, AL_Fish_Roe, AL_Fish_Salmon, AL_Fish_Sardine, AL_Fish_Shark, AL_Fish_Snapper, AL_Fish_Swordfish, AL_Fish_Tilapia, AL_Fish_Trout, AL_Fish_Unspecified, AL_Fish_Whitefish, AL_Grain_Bulgur, AL_Grain_Cereal, AL_Grain_Millet, AL_Grain_Pea, AL_Grain_SplitPea, AL_Grains_Barley, AL_Grains_Brown_Rice, AL_Grains_Bulgur, AL_Grains_Chickpea, AL_Grains_Corn, AL_Grains_Millet, AL_Grains_Oats, AL_Grains_Rice, AL_Legume_Bean, AL_Legume_Chickpea, AL_Legume_Fava, AL_Legume_FavaBean, AL_Legume_Peas, AL_Legume_Unspecified, AL_Meat_Camel, AL_Meat_Duck, AL_Meat_Fat, AL_Meat_Fish, AL_Meat_Goose, AL_Meat_Ground, AL_Meat_Kidney, AL_Meat_Offal, AL_Meat_Organ, AL_Meat_Ostrich, AL_Meat_Pigeon, AL_Meat_Quail, AL_Meat_Sheep, AL_Meat_Veal, AL_Nuts_Acorn, AL_Nuts_Almonds, AL_Nuts_Cashew, AL_Nuts_Chestnut, AL_Nuts_Hazelnuts, AL_Nuts_Hemp, AL_Nuts_Mixed, AL_Nuts_Nutmeg, AL_Nuts_Peanuts, AL_Nuts_Pecan, AL_Other_Sesame, AL_Poultry_Duck, AL_Poultry_Offal, AL_Seafood_Fish, AL_Seeds_Cottonseed, AL_Seeds_Flax, AL_Seeds_Flaxseed, AL_Seeds_Poppy, AL_Seeds_Pumpkin, AL_Seeds_Safflower, AL_Seeds_Sesame, AL_Seeds_Sunflower, AL_Seeds_Watermelon, AL_Sesame_Sesame, AL_Soy_Soy, AL_Soy_SoySauce, AL_Soy_Soybean, AL_Soy_Soybeans


- The user's questionnaire includes the field `FOOD_ALLERGIES_INTOLERANCES`.
- If the field's value is `"[هیچکدام]"` (meaning "none"), DO NOT FILTER ANY ALLERGENS (i.e., all values remain unchanged).
- Otherwise, if the field contains one or more allergens (by name), for each allergen in the list, check if its name appears in the field.
  - If found, assign that allergen tag a value of `1`.
  - If not mentioned, assign it a value of `0`.

**Disease Tags**
List: Disease_Diabetes, Disease_Hypertension, Disease_High_Cholesterol, Disease_CVD, Disease_Kidney, Disease_Celiac, Disease_IBS, Disease_IBD, Disease_Gout, Disease_Lactose_Intolerance, Disease_NAFLD, Disease_Anemia, Disease_Osteoporosis, Disease_Arthritis, Disease_Thyroid, Disease_Autoimmune, Disease_CancerPrevention, Disease_MentalHealth

- The user's questionnaire includes the field `CHRONIC_ILLNESSES`.
- If the field's value is `"[من سالمم]"` (meaning "I am healthy"), DO NOT FILTER ANY DISEASES (i.e., all values remain unchanged).
- Otherwise, if a chronic condition is indicated, for each disease tag in the list, check if the condition is mentioned in the response.
  - If mentioned, assign that disease tag a value of `1`.
  - If not mentioned, assign it a value of `0`.

### Step-by-Step Process:
1. **Parse the Input JSON:** Read and process the user's questionnaire.
2. **Process Allergen Tags:**
   - If `FOOD_ALLERGIES_INTOLERANCES` is `"[هیچکدام]"`, do not apply any allergen filtering.
   - Otherwise, check which allergens appear in the list and set values accordingly.
3. **Process Disease Tags:**
   - If `CHRONIC_ILLNESSES` is `"[من سالمم]"`, do not apply any disease filtering.
   - Otherwise, check which diseases appear in the list and set values accordingly.
4. **Return a JSON object** containing all tags and their respective values.

### Input JSON:
{json_questionnaire}

### Output JSON (Expected):
Return only a valid JSON object exactly in the following structure (with no additional commentary, markdown, or formatting):
{
  "Allergens": {
    "AL_Dairy_Butter": <0 or 1>,
    "AL_Dairy_Cheese": <0 or 1>,
    "AL_Dairy_Coconut": <0 or 1>,
    "AL_Dairy_CoconutMilk": <0 or 1>,
    "AL_Dairy_Cream": <0 or 1>,
    "AL_Dairy_CreamCheese": <0 or 1>,
    "AL_Dairy_IceCream": <0 or 1>,
    "AL_Dairy_Mayonnaise": <0 or 1>,
    "AL_Dairy_Milk": <0 or 1>,
    "AL_Dairy_Milk_Chocolate": <0 or 1>,
    "AL_Dairy_Whey": <0 or 1>,
    "AL_Dairy_Yogurt": <0 or 1>,
    "AL_Egg": <0 or 1>,
    "AL_Eggs_Egg": <0 or 1>,
    "AL_Eggs_Mayonnaise": <0 or 1>,
    "AL_Fish_Crab": <0 or 1>,
    "AL_Fish_Fish": <0 or 1>,
    "AL_Fish_Shrimp": <0 or 1>,
    "AL_Fish_Tuna": <0 or 1>,
    "AL_Grain_Barley": <0 or 1>,
    "AL_Grain_Bean": <0 or 1>,
    "AL_Grain_Buckwheat": <0 or 1>,
    "AL_Grain_Chickpea": <0 or 1>,
    "AL_Grain_Corn": <0 or 1>,
    "AL_Grain_Lentil": <0 or 1>,
    "AL_Grain_Mung": <0 or 1>,
    "AL_Grain_Oat": <0 or 1>,
    "AL_Grain_Oats": <0 or 1>,
    "AL_Grain_Quinoa": <0 or 1>,
    "AL_Grain_Rice": <0 or 1>,
    "AL_Grain_Rye": <0 or 1>,
    "AL_Grain_Soy": <0 or 1>,
    "AL_Grain_Unspecified": <0 or 1>,
    "AL_Grain_Wheat": <0 or 1>,
    "AL_Grains_Wheat": <0 or 1>,
    "AL_Legume_Beans": <0 or 1>,
    "AL_Legume_Lentil": <0 or 1>,
    "AL_Legume_Pea": <0 or 1>,
    "AL_Legume_RedBean": <0 or 1>,
    "AL_Legume_Soy": <0 or 1>,
    "AL_Legume_Split_Peas": <0 or 1>,
    "AL_Legume_SplitPea": <0 or 1>,
    "AL_Meat_Bacon": <0 or 1>,
    "AL_Meat_Beef": <0 or 1>,
    "AL_Meat_Chicken": <0 or 1>,
    "AL_Meat_Ham": <0 or 1>,
    "AL_Meat_Lamb": <0 or 1>,
    "AL_Meat_Pepperoni": <0 or 1>,
    "AL_Meat_Pork": <0 or 1>,
    "AL_Meat_Processed": <0 or 1>,
    "AL_Meat_Sausage": <0 or 1>,
    "AL_Meat_Turkey": <0 or 1>,
    "AL_Meat_Unspecified": <0 or 1>,
    "AL_Meat_Varies": <0 or 1>,
    "AL_Meat_Various": <0 or 1>,
    "AL_Nuts_Almond": <0 or 1>,
    "AL_Nuts_Coconut": <0 or 1>,
    "AL_Nuts_Hazelnut": <0 or 1>,
    "AL_Nuts_Nuts": <0 or 1>,
    "AL_Nuts_Peanut": <0 or 1>,
    "AL_Nuts_Pine_Nuts": <0 or 1>,
    "AL_Nuts_Pistachio": <0 or 1>,
    "AL_Nuts_Sesame": <0 or 1>,
    "AL_Nuts_Unspecified": <0 or 1>,
    "AL_Nuts_Varies": <0 or 1>,
    "AL_Nuts_Various": <0 or 1>,
    "AL_Nuts_Walnut": <0 or 1>,
    "AL_Poultry_Chicken": <0 or 1>,
    "AL_Seeds_Chia": <0 or 1>,
    "AL_Shellfish_Shrimp": <0 or 1>,
    "AL_Dairy_Cream_Cheese": <0 or 1>,
    "AL_Dairy_Creamer": <0 or 1>,
    "AL_Dairy_Goat_Milk": <0 or 1>,
    "AL_Dairy_Kashk": <0 or 1>,
    "AL_Dairy_Kefir": <0 or 1>,
    "AL_Eggs_Whole": <0 or 1>,
    "AL_Eggs_Whole_Eggs": <0 or 1>,
    "AL_Eggs_Yolk": <0 or 1>,
    "AL_Fish_Anchovy": <0 or 1>,
    "AL_Fish_Carp": <0 or 1>,
    "AL_Fish_Catfish": <0 or 1>,
    "AL_Fish_Caviar": <0 or 1>,
    "AL_Fish_Croaker": <0 or 1>,
    "AL_Fish_Flatfish": <0 or 1>,
    "AL_Fish_Flounder": <0 or 1>,
    "AL_Fish_Grouper": <0 or 1>,
    "AL_Fish_Herring": <0 or 1>,
    "AL_Fish_KingMackerel": <0 or 1>,
    "AL_Fish_Mackerel": <0 or 1>,
    "AL_Fish_Mullet": <0 or 1>,
    "AL_Fish_Pike": <0 or 1>,
    "AL_Fish_Porgy": <0 or 1>,
    "AL_Fish_Roe": <0 or 1>,
    "AL_Fish_Salmon": <0 or 1>,
    "AL_Fish_Sardine": <0 or 1>,
    "AL_Fish_Shark": <0 or 1>,
    "AL_Fish_Snapper": <0 or 1>,
    "AL_Fish_Swordfish": <0 or 1>,
    "AL_Fish_Tilapia": <0 or 1>,
    "AL_Fish_Trout": <0 or 1>,
    "AL_Fish_Unspecified": <0 or 1>,
    "AL_Fish_Whitefish": <0 or 1>,
    "AL_Grain_Bulgur": <0 or 1>,
    "AL_Grain_Cereal": <0 or 1>,
    "AL_Grain_Millet": <0 or 1>,
    "AL_Grain_Pea": <0 or 1>,
    "AL_Grain_SplitPea": <0 or 1>,
    "AL_Grains_Barley": <0 or 1>,
    "AL_Grains_Brown_Rice": <0 or 1>,
    "AL_Grains_Bulgur": <0 or 1>,
    "AL_Grains_Chickpea": <0 or 1>,
    "AL_Grains_Corn": <0 or 1>,
    "AL_Grains_Millet": <0 or 1>,
    "AL_Grains_Oats": <0 or 1>,
    "AL_Grains_Rice": <0 or 1>,
    "AL_Legume_Bean": <0 or 1>,
    "AL_Legume_Chickpea": <0 or 1>,
    "AL_Legume_Fava": <0 or 1>,
    "AL_Legume_FavaBean": <0 or 1>,
    "AL_Legume_Peas": <0 or 1>,
    "AL_Legume_Unspecified": <0 or 1>,
    "AL_Meat_Camel": <0 or 1>,
    "AL_Meat_Duck": <0 or 1>,
    "AL_Meat_Fat": <0 or 1>,
    "AL_Meat_Fish": <0 or 1>,
    "AL_Meat_Goose": <0 or 1>,
    "AL_Meat_Ground": <0 or 1>,
    "AL_Meat_Kidney": <0 or 1>,
    "AL_Meat_Offal": <0 or 1>,
    "AL_Meat_Organ": <0 or 1>,
    "AL_Meat_Ostrich": <0 or 1>,
    "AL_Meat_Pigeon": <0 or 1>,
    "AL_Meat_Quail": <0 or 1>,
    "AL_Meat_Sheep": <0 or 1>,
    "AL_Meat_Veal": <0 or 1>,
    "AL_Nuts_Acorn": <0 or 1>,
    "AL_Nuts_Almonds": <0 or 1>,
    "AL_Nuts_Cashew": <0 or 1>,
    "AL_Nuts_Chestnut": <0 or 1>,
    "AL_Nuts_Hazelnuts": <0 or 1>,
    "AL_Nuts_Hemp": <0 or 1>,
    "AL_Nuts_Mixed": <0 or 1>,
    "AL_Nuts_Nutmeg": <0 or 1>,
    "AL_Nuts_Peanuts": <0 or 1>,
    "AL_Nuts_Pecan": <0 or 1>,
    "AL_Other_Sesame": <0 or 1>,
    "AL_Poultry_Duck": <0 or 1>,
    "AL_Poultry_Offal": <0 or 1>,
    "AL_Seafood_Fish": <0 or 1>,
    "AL_Seeds_Cottonseed": <0 or 1>,
    "AL_Seeds_Flax": <0 or 1>,
    "AL_Seeds_Flaxseed": <0 or 1>,
    "AL_Seeds_Poppy": <0 or 1>,
    "AL_Seeds_Pumpkin": <0 or 1>,
    "AL_Seeds_Safflower": <0 or 1>,
    "AL_Seeds_Sesame": <0 or 1>,
    "AL_Seeds_Sunflower": <0 or 1>,
    "AL_Seeds_Watermelon": <0 or 1>,
    "AL_Sesame_Sesame": <0 or 1>,
    "AL_Soy_Soy": <0 or 1>,
    "AL_Soy_SoySauce": <0 or 1>,
    "AL_Soy_Soybean": <0 or 1>,
    "AL_Soy_Soybeans": <0 or 1>
  },
  "Diseases": {
    "Disease_Diabetes": <0 or 1>,
    "Disease_Hypertension": <0 or 1>,
    "Disease_High_Cholesterol": <0 or 1>,
    "Disease_CVD": <0 or 1>,
    "Disease_Kidney": <0 or 1>,
    "Disease_Celiac": <0 or 1>,
    "Disease_IBS": <0 or 1>,
    "Disease_IBD": <0 or 1>,
    "Disease_Gout": <0 or 1>,
    "Disease_Lactose_Intolerance": <0 or 1>,
    "Disease_NAFLD": <0 or 1>,
    "Disease_Anemia": <0 or 1>,
    "Disease_Osteoporosis": <0 or 1>,
    "Disease_Arthritis": <0 or 1>,
    "Disease_Thyroid": <0 or 1>,
    "Disease_Autoimmune": <0 or 1>,
    "Disease_CancerPrevention": <0 or 1>,
    "Disease_MentalHealth": <0 or 1>
  }
}
"""

    questionnaire_json_str = json.dumps(questionnaire, ensure_ascii=False, indent=2)
    complete_prompt = base_prompt.replace("{json_questionnaire}", questionnaire_json_str)

    try:
        model = palm.GenerativeModel(model_name="gemini-2.0-flash")
        response = model.generate_content(complete_prompt)

        if not response or not hasattr(response, "text") or not response.text:
            print("Error: Empty or invalid response from AI.")
            return None

        raw_text = response.text.strip()
        json_match = re.search(r'```json\n(.*?)\n```', raw_text, re.DOTALL)
        if json_match:
            cleaned_json = json_match.group(1)
        else:
            cleaned_json = raw_text

        try:
            tag_values = json.loads(cleaned_json)
            return tag_values
        except json.JSONDecodeError as e:
            print(f"Error: Failed to parse AI response as JSON: {e}")
            print(f"Raw Response: {raw_text}")
            return None

    except Exception as ex:
        print(f"Error: AI request failed: {ex}")
        return None


def process_questionnaire(questionnaire, calculatedCR, gender):
    """
    Processes a questionnaire dictionary to calculate CR, round it, generate tags,
    and provide calorie distributions.

    Args:
        questionnaire (dict): User's questionnaire data with fields:
            - "WEIGHT": str or float (weight in kg)
            - "HEIGHT": str or float (height in cm)
            - "BIRTHDAY": str or int (timestamp in seconds since epoch)
            - "GENDER": str (optional, "male" or "female", defaults to "male")
            - "EXERCISE_FREQUENCY": str (optional, e.g., "[1-2 بار]")
            - "FOOD_ALLERGIES_INTOLERANCES": str
            - "CHRONIC_ILLNESSES": str

    Returns:
        dict: Contains "CR" (rounded calorie requirement), "Tags" (allergen and disease tags),
              and calorie distributions, or None if an error occurs.
    """
    try:
        # Extract and convert data for CR calculation
        weight = float(questionnaire["WEIGHT"])
        height = float(questionnaire["HEIGHT"])
        birth_date = datetime.fromtimestamp(int(questionnaire["BIRTHDAY"]))

        # Handle gender (default to male if not specified or invalid)
        # gender_str = questionnaire.get("GENDER", "male").lower()
        # gender = True if gender_str == "female" else False

        # Determine activity level
        exercise_frequency = questionnaire.get("EXERCISE_FREQUENCY", "")
        if exercise_frequency == "[1-2 بار]":
            activity_level = 1.5
        elif exercise_frequency == "[3-4 بار]":
            activity_level = 1.75
        elif exercise_frequency == "[کمتر از 1 بار]":
            activity_level = 1.25
        else:
            activity_level = 2  # Default to very active

        type_of_treatment = 0  # Fixed default value

        # print(birth_date, weight, height, gender, activity_level, type_of_treatment)
        # Calculate CR
        # cr_value = calculate_cr(birth_date, weight, height, gender, activity_level, type_of_treatment)

        # Round CR using custom logic
        rounded_cr = round_cr(calculatedCR)

        # Find the closest CR in cr_list for calorie distribution
        selected_cr = min(cr_list, key=lambda cr: (abs(calculatedCR - cr), cr))
        distribution = calorie_distributions[selected_cr]

        # Generate tags
        tag_values = get_tag_values(questionnaire)
        if tag_values is None:
            print("Failed to generate tag values.")
            return None

        # Return combined result with rounded CR and calorie distributions
        return {
            "CR": rounded_cr,
            "Tags": tag_values,
            "breakfast_cal": distribution["B"],
            "lunch_cal": distribution["L"],
            "dinner_cal": distribution["D"],
            "snack1_cal": distribution["S1"],
            "snack2_cal": distribution["S2"]
        }

    except KeyError as e:
        print(f"Error: Missing required field: {e}")
        return None
    except ValueError as e:
        print(f"Error: Invalid value in field: {e}")
        return None
    except Exception as e:
        print(f"Error: An unexpected error occurred: {e}")
        return None