/**
 * Card Pagination Module
 * Handles all pagination functionality for cards and foods
 */

class CardPagination {
    constructor() {
        this.currentCardPage = 1;
        this.currentFoodPage = 1;
        this.initializeEventListeners();
    }

    initializeEventListeners() {
        // Initialize pagination event listeners on page load
        document.addEventListener('DOMContentLoaded', () => {
            this.attachPaginationEventListeners();
        });
    }

    attachPaginationEventListeners() {
        // Select all pagination links
        const paginationLinks = document.querySelectorAll('.pagination a');
        paginationLinks.forEach(link => {
            const pageNumber = link.getAttribute('data-page-number');
            if (pageNumber) {
                link.onclick = (e) => {
                    e.preventDefault();
                    this.fetchCardPage(pageNumber);
                };
            }
        });
    }

    fetchCardPage(pageNumber) {
        this.currentCardPage = pageNumber;
        const searchQuery = document.getElementById('foodSearchCard')?.value || '';

        // Get current filters from the URL or variables
        const urlParams = new URLSearchParams(window.location.search);
        const includeTags = urlParams.getAll('include_tags');
        const excludeTags = urlParams.getAll('exclude_tags');
        const minCalories = urlParams.get('min_calories');
        const maxCalories = urlParams.get('max_calories');
        const sortField = urlParams.get('sort_field');
        const sortOrder = urlParams.get('sort_order');

        let url = `/panel/assistant/cards/ajax/?card_page=${pageNumber}`;

        if (searchQuery) {
            url += `&card_search_query=${encodeURIComponent(searchQuery)}`;
        }

        // Include include_tags
        includeTags.forEach(tag => {
            url += `&include_tags=${encodeURIComponent(tag)}`;
        });

        // Include exclude_tags
        excludeTags.forEach(tag => {
            url += `&exclude_tags=${encodeURIComponent(tag)}`;
        });

        // Include calorie filters
        if (minCalories) {
            url += `&min_calories=${encodeURIComponent(minCalories)}`;
        }
        if (maxCalories) {
            url += `&max_calories=${encodeURIComponent(maxCalories)}`;
        }

        // Include sort_field and sort_order
        if (sortField) {
            url += `&sort_field=${encodeURIComponent(sortField)}`;
        }
        if (sortOrder) {
            url += `&sort_order=${encodeURIComponent(sortOrder)}`;
        }

        // Show loading indicator
        this.showLoadingIndicator('#cardList');

        // Make AJAX request
        $.ajax({
            url: url,
            type: 'GET',
            success: (response) => {
                this.handleCardPageSuccess(response);
            },
            error: (xhr, status, error) => {
                this.handleCardPageError(xhr, status, error);
            }
        });
    }

    handleCardPageSuccess(response) {
        // Update the card list with the new data
        const cardList = document.getElementById('cardList');
        if (cardList && response.cards_html) {
            cardList.innerHTML = response.cards_html;
        }

        // Update pagination controls if necessary
        const pagination = document.querySelector('.pagination');
        if (pagination && response.pagination_html) {
            pagination.innerHTML = response.pagination_html;
        }

        // Reattach event listeners to pagination links
        this.attachPaginationEventListeners();

        // Reattach event listeners to new card elements
        this.reattachCardEventListeners();

        // Hide loading indicator
        this.hideLoadingIndicator('#cardList');

        // Dispatch custom event for other modules
        const event = new CustomEvent('cardPageLoaded', {
            detail: { 
                pageNumber: this.currentCardPage,
                timestamp: Date.now() 
            }
        });
        document.dispatchEvent(event);
    }

    handleCardPageError(xhr, status, error) {
        console.error('Error fetching card page:', { xhr, status, error });
        this.hideLoadingIndicator('#cardList');
        
        // Show error message to user
        this.showErrorMessage('Error fetching cards. Please try again.');
    }

    fetchFoodPage(pageNumber) {
        this.currentFoodPage = pageNumber;
        const searchQuery = document.getElementById('foodSearch')?.value || '';
        
        let url = `/panel/assistant/foods/ajax/?food_page=${pageNumber}`;
    
        if (searchQuery) {
            url += `&food_search_query=${encodeURIComponent(searchQuery)}`;
        }

        // Show loading indicator
        this.showLoadingIndicator('#foodsList');
    
        $.ajax({
            url: url,
            type: 'GET',
            success: (response) => {
                this.handleFoodPageSuccess(response);
            },
            error: (xhr, status, error) => {
                this.handleFoodPageError(xhr, status, error);
            }
        });
    }

    handleFoodPageSuccess(response) {
        // Update the food list and pagination controls
        const foodsList = document.getElementById('foodsList');
        if (foodsList && response.foods_html) {
            foodsList.innerHTML = response.foods_html;
        }

        const foodsPagination = document.querySelector('.foods-pagination');
        if (foodsPagination && response.pagination_html) {
            foodsPagination.innerHTML = response.pagination_html;
        }

        // Re-initialize any event listeners on the new elements
        this.initializeFoodItemEventListeners();

        // Hide loading indicator
        this.hideLoadingIndicator('#foodsList');

        // Dispatch custom event for other modules
        const event = new CustomEvent('foodPageLoaded', {
            detail: { 
                pageNumber: this.currentFoodPage,
                timestamp: Date.now() 
            }
        });
        document.dispatchEvent(event);
    }

    handleFoodPageError(xhr, status, error) {
        console.error('Error fetching food page:', { xhr, status, error });
        this.hideLoadingIndicator('#foodsList');
        
        // Show error message to user
        this.showErrorMessage('خطا در بارگذاری مواد غذایی. لطفاً دوباره تلاش کنید.');
    }

    reattachCardEventListeners() {
        // Reattach event listeners to new card elements
        const cardItems = document.querySelectorAll('.card-item');
        cardItems.forEach(cardItem => {
            // Reattach toggle functionality
            const cardHeader = cardItem.querySelector('.card-header');
            if (cardHeader) {
                cardHeader.onclick = () => this.toggleFoodItems(cardHeader);
            }

            // Reattach allergy indicators
            this.addAllergyIndicatorsToCard(cardItem);
        });
    }

    initializeFoodItemEventListeners() {
        // Handle drag start for new food items
        const foodItems = document.querySelectorAll('.food-item');
        foodItems.forEach(item => {
            item.addEventListener('dragstart', (e) => {
                if (window.cardManager) {
                    window.cardManager.draggedElement = e.target;
                }
                e.dataTransfer.setData('text/plain', '');
            });
        });
    }

    toggleFoodItems(headerElement) {
        const cardBody = headerElement.nextElementSibling;
        if (cardBody.style.display === 'none' || cardBody.style.display === '') {
            cardBody.style.display = 'block';
        } else {
            cardBody.style.display = 'none';
        }
    }

    addAllergyIndicatorsToCard(cardItem) {
        const cardHeader = cardItem.querySelector('.card-header');
        if (!cardHeader) return;

        const foodItems = cardItem.querySelectorAll('.food-item');
        
        // Check if any food item has allergies
        let hasAllergies = false;
        foodItems.forEach(foodItem => {
            const alList = foodItem.getAttribute('data-al_list');
            if (alList && alList.trim() !== '') {
                hasAllergies = true;
            }
        });
        
        // Add allergy indicator if needed
        if (hasAllergies && !cardHeader.querySelector('.card-allergies-indicator')) {
            const allergyIndicator = document.createElement('div');
            allergyIndicator.className = 'card-allergies-indicator';
            allergyIndicator.style.cssText = 'margin-top: 5px;';
            allergyIndicator.innerHTML = '<span class="badge badge-danger" style="background-color: #dc3545; color: white; font-size: 10px; padding: 2px 6px;">⚠️ دارای آلرژی</span>';
            cardHeader.appendChild(allergyIndicator);
        }
    }

    addAllergyIndicatorsToCards() {
        const cardItems = document.querySelectorAll('.card-item');
        cardItems.forEach(cardItem => {
            this.addAllergyIndicatorsToCard(cardItem);
        });
    }

    showLoadingIndicator(selector) {
        const element = document.querySelector(selector);
        if (element) {
            element.innerHTML = '<div class="text-center p-4"><div class="spinner-border" role="status"><span class="sr-only">Loading...</span></div></div>';
        }
    }

    hideLoadingIndicator(selector) {
        // Loading indicator will be replaced by actual content
    }

    showErrorMessage(message) {
        // You can customize this to show a toast notification or alert
        alert(message);
    }

    // Method to get current page numbers
    getCurrentCardPage() {
        return this.currentCardPage;
    }

    getCurrentFoodPage() {
        return this.currentFoodPage;
    }

    // Method to go to specific page
    goToCardPage(pageNumber) {
        if (pageNumber > 0) {
            this.fetchCardPage(pageNumber);
        }
    }

    goToFoodPage(pageNumber) {
        if (pageNumber > 0) {
            this.fetchFoodPage(pageNumber);
        }
    }

    // Method to go to next page
    nextCardPage() {
        this.goToCardPage(this.currentCardPage + 1);
    }

    nextFoodPage() {
        this.goToFoodPage(this.currentFoodPage + 1);
    }

    // Method to go to previous page
    previousCardPage() {
        if (this.currentCardPage > 1) {
            this.goToCardPage(this.currentCardPage - 1);
        }
    }

    previousFoodPage() {
        if (this.currentFoodPage > 1) {
            this.goToFoodPage(this.currentFoodPage - 1);
        }
    }

    // Method to go to first page
    firstCardPage() {
        this.goToCardPage(1);
    }

    firstFoodPage() {
        this.goToFoodPage(1);
    }

    // Method to go to last page (if total pages is known)
    lastCardPage(totalPages) {
        if (totalPages && totalPages > 0) {
            this.goToCardPage(totalPages);
        }
    }

    lastFoodPage(totalPages) {
        if (totalPages && totalPages > 0) {
            this.goToFoodPage(totalPages);
        }
    }

    // Method to refresh current page
    refreshCurrentCardPage() {
        this.fetchCardPage(this.currentCardPage);
    }

    refreshCurrentFoodPage() {
        this.fetchFoodPage(this.currentFoodPage);
    }

    // Method to check if we're on first page
    isFirstCardPage() {
        return this.currentCardPage === 1;
    }

    isFirstFoodPage() {
        return this.currentFoodPage === 1;
    }

    // Method to get pagination info (useful for debugging)
    getPaginationInfo() {
        return {
            cardPage: this.currentCardPage,
            foodPage: this.currentFoodPage,
            timestamp: Date.now()
        };
    }
}

// Initialize card pagination when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.cardPagination = new CardPagination();
});

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = CardPagination;
} 