console.log('[SCRIPT_LOAD] 🚀 CardManager script is loading...');

class CardManager {
    constructor() {
        console.log('[CONSTRUCTOR] 🏗️🏗️🏗️ CardManager CONSTRUCTOR CALLED! 🏗️🏗️🏗️');
        console.log('[CONSTRUCTOR] 🏗️ this:', this);
        
        this.currentCard = null;
        console.log('[CONSTRUCTOR] 🏗️ this.currentCard set to null');
        
        this.initializeEventListeners();
        console.log('[CONSTRUCTOR] 🏗️ Event listeners initialized');
        
        console.log('[CONSTRUCTOR] ✅ CardManager initialized successfully');
        console.log('[CONSTRUCTOR] 🏗️ Final this object:', this);
    }

    initializeEventListeners() {
        // Initialize any global event listeners here
        console.log('[CONSTRUCTOR] Event listeners initialized');
    }

    setupDragAndDrop() {
        const cardDropArea = document.getElementById('cardDropArea');
        const editDropZone = document.getElementById('editDropZone');

        if (cardDropArea) {
            cardDropArea.addEventListener('dragover', (e) => this.handleDragOver(e));
            cardDropArea.addEventListener('drop', (e) => this.handleDrop(e));
        }

        if (editDropZone) {
            editDropZone.addEventListener('dragover', (e) => this.handleDragOver(e));
            editDropZone.addEventListener('drop', (e) => this.handleCardDrop(e));
        }
    }

    handleDragOver(e) {
        e.preventDefault();
        e.dataTransfer.dropEffect = 'copy';
    }

    handleDrop(e) {
        e.preventDefault();
        const foodElement = e.dataTransfer.getData('text/html');
        if (foodElement) {
            const tempDiv = document.createElement('div');
            tempDiv.innerHTML = foodElement;
            const foodItem = tempDiv.firstElementChild;
            if (foodItem) {
                this.addFoodItemToCard(this.currentCard, foodItem);
            }
        }
    }

    handleCardDrop(e) {
        e.preventDefault();
        const cardElement = e.dataTransfer.getData('text/html');
        if (cardElement) {
            const tempDiv = document.createElement('div');
            tempDiv.innerHTML = cardElement;
            const card = tempDiv.firstElementChild;
            if (card) {
                this.handleCardDrop(card);
            }
        }
    }

    openCreateCardModal(callback) {
        // Implementation for opening create card modal
        console.log('Opening create card modal');
    }

    createNewCard() {
        const cardTemplate = this.getCardTemplate();
        const cardElement = document.createElement('div');
        cardElement.innerHTML = cardTemplate;
        const card = cardElement.firstElementChild;
        
        this.currentCard = card;
        this.setupCardEventListeners(card);
        
        const cardDropArea = document.getElementById('cardDropArea');
        if (cardDropArea) {
            cardDropArea.appendChild(card);
        }
        
        console.log('New card created');
        return card;
    }

    getCardTemplate() {
        return `
            <div class="card" data-card-id="new">
                <div class="card-header">
                    <h5>کارت جدید</h5>
                </div>
                <div class="card-body">
                    <div class="nutrient-summary">
                        <span>کالری: <span class="total-calories">0</span></span>
                        <span>چربی: <span class="total-fat">0</span>g</span>
                        <span>پروتئین: <span class="total-protein">0</span>g</span>
                        <span>کربوهیدرات: <span class="total-carbs">0</span>g</span>
                    </div>
                    <ul class="card-items list-group">
                        <!-- Food items will be added here -->
                    </ul>
                </div>
            </div>
        `;
    }

    setupCardEventListeners(card) {
        // Setup event listeners for the card
        console.log('Card event listeners setup');
    }

    makeCardEditable(card) {
        // Implementation for making card editable
        console.log('Making card editable');
    }

    editCard(card) {
        // Implementation for editing card
        console.log('Editing card');
    }

    deleteCard(card) {
        // Implementation for deleting card
        if (confirm('آیا از حذف این کارت مطمئن هستید؟')) {
            card.remove();
            this.currentCard = null;
        }
    }

    addFoodItemToCard(card, foodElement) {
        console.log('➕➕➕ addFoodItemToCard METHOD CALLED! ➕➕➕');
        console.log('➕ card:', card);
        console.log('➕ foodElement:', foodElement);
        console.log('➕ this:', this);
        
        if (!card) {
            console.error('❌ No card provided to addFoodItemToCard!');
            return;
        }

        console.log('[ADD_FOOD_ITEM] 🔍 Extracting food item data...');
        const foodItemData = this.extractFoodItemData(foodElement);
        console.log('[ADD_FOOD_ITEM] 🔍 Extracted food item data:', foodItemData);
        
        console.log('[ADD_FOOD_ITEM] 🔨 Creating food item element...');
        const foodItemElement = this.createFoodItemElement(foodItemData);
        console.log('[ADD_FOOD_ITEM] 🔨 Created food item element:', foodItemElement);
        
        const cardItemsList = card.querySelector('.card-items');
        console.log('[ADD_FOOD_ITEM] 🔍 Found card items list:', cardItemsList);
        
        if (cardItemsList) {
            console.log('[ADD_FOOD_ITEM] 📝 Appending food item to card...');
            cardItemsList.appendChild(foodItemElement);
            console.log('[ADD_FOOD_ITEM] ✅ Food item appended to card');
            
            // Calculate initial nutrient values for this food item
            const initialAmount = parseFloat(foodItemElement.getAttribute('data-amount')) || 1;
            console.log('[ADD_FOOD_ITEM] 🧮 Initial amount for nutrient calculation:', initialAmount);
            
            console.log('[ADD_FOOD_ITEM] 🧮 Calling updateFoodItemNutrients...');
            this.updateFoodItemNutrients(foodItemElement, initialAmount);
            console.log('[ADD_FOOD_ITEM] ✅ updateFoodItemNutrients completed');
            
            // Update the overall card nutrients
            console.log('[ADD_FOOD_ITEM] 🔄 Calling updateCardNutrients...');
            this.updateCardNutrients();
            console.log('[ADD_FOOD_ITEM] ✅ updateCardNutrients completed');
            
            console.log('[ADD_FOOD_ITEM] 🎉 Food item added to card successfully!');
        } else {
            console.error('[ADD_FOOD_ITEM] ❌ No .card-items list found in card!');
        }
    }

    extractFoodItemData(foodElement) {
        return {
            foodCode: foodElement.getAttribute('data-food_code'),
            foodName: foodElement.getAttribute('data-food_name'),
            amountValue: foodElement.getAttribute('data-amount') || 1,
            selectedHomeUnit: foodElement.getAttribute('data-selected_home_unit'),
            weightOfHomeUnit: foodElement.getAttribute('data-selected_home_unit_weight') || 1,
            proteinPer100g: foodElement.getAttribute('data-protein_per100g') || 0,
            fatPer100g: foodElement.getAttribute('data-fat_per100g') || 0,
            carbsPer100g: foodElement.getAttribute('data-carbohydrate_per100g') || 0,
            caloriesPer100g: foodElement.getAttribute('data-energy_per100g') || 0,
            homeUnits: foodElement.getAttribute('data-home_units'),
            weightUnits: foodElement.getAttribute('data-weight_units'),
            alList: foodElement.getAttribute('data-al_list')
        };
    }

    createFoodItemElement(foodItemData) {
        console.log('🔨🔨🔨 createFoodItemElement METHOD CALLED! 🔨🔨🔨');
        console.log('🔨 foodItemData:', foodItemData);
        console.log('🔨 this:', this);
        
        const foodItemElement = document.createElement('li');
        console.log('🔨 Created li element:', foodItemElement);
        
        foodItemElement.classList.add('list-group-item', 'food-item');
        console.log('[CREATE_ELEMENT] 🔨 Added CSS classes');

        // Set data attributes
        console.log('[CREATE_ELEMENT] 🔨 Setting data attributes...');
        Object.entries(foodItemData).forEach(([key, value]) => {
            if (value !== null && value !== undefined) {
                const attrName = `data-${key}`;
                foodItemElement.setAttribute(attrName, value);
                console.log(`[CREATE_ELEMENT] 🔨 Set ${attrName} = ${value}`);
            }
        });

        console.log('[CREATE_ELEMENT] 🔨 Setting innerHTML...');
        foodItemElement.innerHTML = `
            <div class="food-item-content">
                <span class="food-name-display">${foodItemData.foodName} (${foodItemData.amountValue} ${foodItemData.selectedHomeUnit})</span>
                <div class="food-nutrient-display">
                    <small class="text-muted">محاسبه در حال انجام...</small>
                </div>
            </div>
            <span class="main-dish-indicator" style="display: none; color: #dc3545; font-weight: bold; margin-right: 8px; font-size: 16px;">👑</span>
            <div class="item-actions">
                <input type="number" class="item-amount-input" value="${foodItemData.amountValue}" min="0.5" step="0.5">
                <button class="set-main-dish-btn">تعیین غذای اصلی</button>
                <button class="delete-item-btn">حذف</button>
            </div>
        `;
        console.log('[CREATE_ELEMENT] 🔨 innerHTML set');

        console.log('[CREATE_ELEMENT] 🔨 About to attach event listeners...');
        this.attachFoodItemEventListeners(foodItemElement);
        console.log('[CREATE_ELEMENT] 🔨 Event listeners attached');
        
        console.log('[CREATE_ELEMENT] 🔨 Final food item element:', foodItemElement);
        console.log('[CREATE_ELEMENT] 🔨 Final HTML:', foodItemElement.innerHTML);
        
        return foodItemElement;
    }

    attachFoodItemEventListeners(foodItemElement) {
        console.log('[EVENT_LISTENERS] 🔧 Setting up event listeners for food item:', foodItemElement);
        
        const amountInput = foodItemElement.querySelector('.item-amount-input');
        const deleteBtn = foodItemElement.querySelector('.delete-item-btn');
        const setMainDishBtn = foodItemElement.querySelector('.set-main-dish-btn');

        console.log('[EVENT_LISTENERS] 🔍 Found elements:', {
            amountInput: amountInput ? 'YES' : 'NO',
            deleteBtn: deleteBtn ? 'YES' : 'NO',
            setMainDishBtn: setMainDishBtn ? 'YES' : 'NO'
        });

        // Store reference to this instance
        const self = this;
        console.log('[EVENT_LISTENERS] 🔗 Self reference stored:', self);
        console.log('[EVENT_LISTENERS] 🔗 Self.handleAmountChange exists:', typeof self.handleAmountChange);

        if (amountInput) {
            console.log('[EVENT_LISTENERS] 📝 Setting up amount input listener');
            
            // Handle real-time input changes
            amountInput.addEventListener('input', function(e) {
                console.log('[INPUT_EVENT] 🎯 INPUT EVENT FIRED!');
                console.log('[INPUT_EVENT] 🎯 Input value:', this.value);
                console.log('[INPUT_EVENT] 🎯 Event type:', e.type);
                console.log('[INPUT_EVENT] 🎯 Target element:', this);
                console.log('[INPUT_EVENT] 🎯 Self reference:', self);
                console.log('[INPUT_EVENT] 🎯 Self.handleAmountChange:', typeof self.handleAmountChange);
                
                try {
                    self.handleAmountChange(foodItemElement);
                    console.log('[INPUT_EVENT] ✅ handleAmountChange called successfully');
                } catch (error) {
                    console.error('[INPUT_EVENT] ❌ Error calling handleAmountChange:', error);
                }
            });
            
            // Also handle on blur for final validation
            amountInput.addEventListener('blur', function(e) {
                console.log('[BLUR_EVENT] 👁️ BLUR EVENT FIRED!');
                console.log('[BLUR_EVENT] 👁️ Input value:', this.value);
                console.log('[BLUR_EVENT] 👁️ Event type:', e.type);
                
                try {
                    self.validateAmountInput(foodItemElement);
                    console.log('[BLUR_EVENT] ✅ validateAmountInput called successfully');
                } catch (error) {
                    console.error('[BLUR_EVENT] ❌ Error calling validateAmountChange:', error);
                }
            });
            
            // Handle Enter key press
            amountInput.addEventListener('keypress', function(e) {
                if (e.key === 'Enter') {
                    console.log('[KEYPRESS_EVENT] ⌨️ ENTER KEY PRESSED!');
                    this.blur();
                }
            });
            
            console.log('[EVENT_LISTENERS] ✅ Amount input listeners attached');
        } else {
            console.error('[EVENT_LISTENERS] ❌ No amount input found!');
        }

        if (deleteBtn) {
            deleteBtn.addEventListener('click', function() {
                console.log('[EVENT_LISTENERS] 🗑️ Delete button clicked');
                self.deleteFoodItem(foodItemElement);
            });
        }

        if (setMainDishBtn) {
            setMainDishBtn.addEventListener('click', function() {
                console.log('[EVENT_LISTENERS] 👑 Set main dish button clicked');
                self.setAsMainDish(foodItemElement);
            });
        }
        
        // Add double-click to reset amount to 1
        if (amountInput) {
            amountInput.addEventListener('dblclick', function() {
                console.log('[EVENT_LISTENERS] 🖱️ Double-click on amount input');
                self.resetFoodItemAmount(foodItemElement);
            });
        }
        
        console.log('[EVENT_LISTENERS] ✅ All event listeners attached for food item');
    }

    handleAmountChange(foodItemElement) {
        console.log('[AMOUNT_CHANGE] 🚀🚀🚀 handleAmountChange METHOD CALLED! 🚀🚀🚀');
        console.log('[AMOUNT_CHANGE] 🚀 foodItemElement:', foodItemElement);
        console.log('[AMOUNT_CHANGE] 🚀 this:', this);
        console.log('[AMOUNT_CHANGE] 🚀 this.updateFoodItemNutrients:', typeof this.updateFoodItemNutrients);
        console.log('[AMOUNT_CHANGE] 🚀 this.updateCardNutrients:', typeof this.updateCardNutrients);
        
        if (!foodItemElement) {
            console.error('[AMOUNT_CHANGE] ❌ foodItemElement is null or undefined!');
            return;
        }
        
        const amountInput = foodItemElement.querySelector('.item-amount-input');
        console.log('[AMOUNT_CHANGE] 🔍 amountInput found:', amountInput);
        
        if (!amountInput) {
            console.error('[AMOUNT_CHANGE] ❌ No amount input found in foodItemElement!');
            console.log('[AMOUNT_CHANGE] 🔍 foodItemElement HTML:', foodItemElement.innerHTML);
            return;
        }
        
        const newAmount = parseFloat(amountInput.value) || 0;
        console.log('[AMOUNT_CHANGE] 📊 New amount parsed:', newAmount);
        console.log('[AMOUNT_CHANGE] 📊 Raw input value:', amountInput.value);
        
        foodItemElement.setAttribute('data-amount', newAmount);
        console.log('[AMOUNT_CHANGE] 💾 data-amount attribute set to:', newAmount);
        
        // Update the food name display
        const foodNameDisplay = foodItemElement.querySelector('.food-name-display');
        const foodName = foodItemElement.getAttribute('data-food_name');
        const homeUnit = foodItemElement.getAttribute('data-selected_home_unit');
        
        console.log('[AMOUNT_CHANGE] 🏷️ Food name:', foodName);
        console.log('[AMOUNT_CHANGE] 🏷️ Home unit:', homeUnit);
        console.log('[AMOUNT_CHANGE] 🏷️ Food name display element:', foodNameDisplay);
        
        if (foodNameDisplay) {
            foodNameDisplay.textContent = `${foodName} (${newAmount} ${homeUnit})`;
            console.log('[AMOUNT_CHANGE] ✅ Food name display updated');
        } else {
            console.warn('[AMOUNT_CHANGE] ⚠️ No food name display element found');
        }
        
        console.log('[AMOUNT_CHANGE] 🧮 About to update nutrients for amount:', newAmount);
        
        try {
            // Recalculate and update nutrients for this specific item
            this.updateFoodItemNutrients(foodItemElement, newAmount);
            console.log('[AMOUNT_CHANGE] ✅ updateFoodItemNutrients completed successfully');
        } catch (error) {
            console.error('[AMOUNT_CHANGE] ❌ Error in updateFoodItemNutrients:', error);
        }
        
        console.log('[AMOUNT_CHANGE] 🔄 About to update card nutrients');
        
        try {
            // Update the overall card nutrients
            this.updateCardNutrients();
            console.log('[AMOUNT_CHANGE] ✅ updateCardNutrients completed successfully');
        } catch (error) {
            console.error('[AMOUNT_CHANGE] ❌ Error in updateCardNutrients:', error);
        }
        
        console.log('[AMOUNT_CHANGE] 🎉 Amount change handling completed successfully!');
    }

    findElementByText(searchText) {
        console.log('[HELPER] 🔍 Searching for element with text:', searchText);
        
        // Search within the current card
        if (!this.currentCard) {
            console.log('[HELPER] ❌ No current card to search in');
            return null;
        }
        
        // Get all text nodes and elements within the card
        const walker = document.createTreeWalker(
            this.currentCard,
            NodeFilter.SHOW_TEXT | NodeFilter.SHOW_ELEMENT,
            null,
            false
        );
        
        let node;
        while (node = walker.nextNode()) {
            if (node.nodeType === Node.TEXT_NODE) {
                // Check if this text node contains our search text
                if (node.textContent && node.textContent.includes(searchText)) {
                    console.log('[HELPER] ✅ Found text node with:', searchText);
                    // Return the parent element that contains this text
                    const parentElement = node.parentElement;
                    console.log('[HELPER] 📍 Parent element:', parentElement);
                    return parentElement;
                }
            } else if (node.nodeType === Node.ELEMENT_NODE) {
                // Check if this element contains our search text
                if (node.textContent && node.textContent.includes(searchText)) {
                    console.log('[HELPER] ✅ Found element with text:', searchText);
                    console.log('[HELPER] 📍 Element:', node);
                    return node;
                }
            }
        }
        
        console.log('[HELPER] ❌ No element found with text:', searchText);
        return null;
    }

    deleteFoodItem(foodItemElement) {
        if (foodItemElement.classList.contains('main-dish-selected')) {
            this.currentCard.removeAttribute('data-main-dish-code');
        }
        foodItemElement.remove();
        this.updateCardNutrients();
    }

    setAsMainDish(selectedFoodItem) {
        if (!this.currentCard) return;
        
        // Remove main dish status from all other items
        const allFoodItems = this.currentCard.querySelectorAll('.card-items .list-group-item');
        allFoodItems.forEach(item => {
            const mainDishBtn = item.querySelector('.set-main-dish-btn');
            const mainDishIndicator = item.querySelector('.main-dish-indicator');
            if (mainDishBtn) {
                mainDishBtn.textContent = 'تعیین غذای اصلی';
                mainDishBtn.style.backgroundColor = '#28a745';
                item.classList.remove('main-dish-selected');
            }
            if (mainDishIndicator) {
                mainDishIndicator.style.display = 'none';
            }
        });
        
        // Set the selected item as main dish
        const mainDishBtn = selectedFoodItem.querySelector('.set-main-dish-btn');
        const mainDishIndicator = selectedFoodItem.querySelector('.main-dish-indicator');
        if (mainDishBtn) {
            mainDishBtn.textContent = '✓ غذای اصلی';
            mainDishBtn.style.backgroundColor = '#dc3545';
            selectedFoodItem.classList.add('main-dish-selected');
        }
        if (mainDishIndicator) {
            mainDishIndicator.style.display = 'inline';
        }
        
        // Store the main dish food code in the card
        const mainDishFoodCode = selectedFoodItem.getAttribute('data-food_code');
        this.currentCard.setAttribute('data-main-dish-code', mainDishFoodCode);
    }

    updateCardNutrients() {
        console.log('[NUTRIENTS_UPDATE] 🔄🔄🔄 updateCardNutrients METHOD CALLED! 🔄🔄🔄');
        console.log('[NUTRIENTS_UPDATE] 🔄 this:', this);
        console.log('[NUTRIENTS_UPDATE] 🔄 this.currentCard:', this.currentCard);
        
        if (!this.currentCard) {
            console.error('[NUTRIENTS_UPDATE] ❌ No current card found!');
            return;
        }

        const foodItems = this.currentCard.querySelectorAll('.card-items .list-group-item');
        console.log('[NUTRIENTS_UPDATE] 🔍 Found food items:', foodItems.length);
        console.log('[NUTRIENTS_UPDATE] 🔍 Food items:', foodItems);
        
        let totalCalories = 0;
        let totalFat = 0;
        let totalProtein = 0;
        let totalCarbs = 0;

        console.log('[NUTRIENTS_UPDATE] 📊 Starting nutrient calculation for', foodItems.length, 'food items');

        foodItems.forEach((item, index) => {
            // Check if this item has pre-calculated values (from amount changes)
            const hasPreCalculatedValues = item.hasAttribute('data-protein') && 
                                        item.hasAttribute('data-fat') && 
                                        item.hasAttribute('data-carbohydrates') && 
                                        item.hasAttribute('data-energy');
            
            if (hasPreCalculatedValues) {
                // Use the pre-calculated values (these are updated when amount changes)
                const protein = parseFloat(item.getAttribute('data-protein')) || 0;
                const fat = parseFloat(item.getAttribute('data-fat')) || 0;
                const carbs = parseFloat(item.getAttribute('data-carbohydrates')) || 0;
                const energy = parseFloat(item.getAttribute('data-energy')) || 0;
                
                totalProtein += protein;
                totalFat += fat;
                totalCarbs += carbs;
                totalCalories += energy;
                
                console.log(`[NUTRIENTS_UPDATE] Item ${index + 1} (${item.getAttribute('data-food_name')}): Using pre-calculated values:`, {
                    protein, fat, carbs, energy
                });
            } else {
                // Calculate based on amount and per-100g values (for newly added items)
                const amount = parseFloat(item.getAttribute('data-amount')) || 0;
                const weightUnit = this.getWeightUnit(item);
                const amountInGrams = amount * weightUnit;

                const proteinPer100g = parseFloat(item.getAttribute('data-protein_per100g')) || 0;
                const fatPer100g = parseFloat(item.getAttribute('data-fat_per100g')) || 0;
                const carbsPer100g = parseFloat(item.getAttribute('data-carbohydrate_per100g')) || 0;
                const caloriesPer100g = parseFloat(item.getAttribute('data-energy_per100g')) || 0;

                const itemProtein = (proteinPer100g * amountInGrams) / 100;
                const itemFat = (fatPer100g * amountInGrams) / 100;
                const itemCarbs = (carbsPer100g * amountInGrams) / 100;
                const itemCalories = (caloriesPer100g * amountInGrams) / 100;

                totalCalories += itemCalories;
                totalFat += itemFat;
                totalProtein += itemProtein;
                totalCarbs += itemCarbs;
                
                console.log(`[NUTRIENTS_UPDATE] Item ${index + 1} (${item.getAttribute('data-food_name')}): Calculated from amount:`, {
                    amount, amountInGrams, protein: itemProtein, fat: itemFat, carbs: itemCarbs, calories: itemCalories
                });
            }
        });

        console.log('Total nutrients calculated:', {
            totalCalories: this.roundToTwo(totalCalories),
            totalFat: this.roundToTwo(totalFat),
            totalProtein: this.roundToTwo(totalProtein),
            totalCarbs: this.roundToTwo(totalCarbs)
        });

        console.log('[NUTRIENTS_UPDATE] 📊 Final totals calculated:', {
            totalCalories: this.roundToTwo(totalCalories),
            totalFat: this.roundToTwo(totalFat),
            totalProtein: this.roundToTwo(totalProtein),
            totalCarbs: this.roundToTwo(totalCarbs)
        });

        // Update the UI - Look for the correct total nutrient containers
        console.log('[NUTRIENTS_UPDATE] 🔍 Looking for total nutrient containers...');
        
        // Try multiple selectors to find the total nutrient displays
        const caloriesElement = this.currentCard.querySelector('.total-calories') || 
                               this.currentCard.querySelector('[data-total-calories]') ||
                               this.findElementByText('کالری کارت');
        
        const fatElement = this.currentCard.querySelector('.total-fat') || 
                          this.currentCard.querySelector('[data-total-fat]') ||
                          this.findElementByText('چربی کارت');
        
        const proteinElement = this.currentCard.querySelector('.total-protein') || 
                              this.currentCard.querySelector('[data-total-protein]') ||
                              this.findElementByText('پروتئین کارت');
        
        const carbsElement = this.currentCard.querySelector('.total-carbs') || 
                            this.currentCard.querySelector('[data-total-carbs]') ||
                            this.findElementByText('کربوهیدرات کارت');

        console.log('[NUTRIENTS_UPDATE] 🔍 Found elements:', {
            calories: caloriesElement,
            fat: fatElement,
            protein: proteinElement,
            carbs: carbsElement
        });

        // Update the found elements
        if (caloriesElement) {
            caloriesElement.textContent = this.roundToTwo(totalCalories);
            console.log('[NUTRIENTS_UPDATE] ✅ Updated calories to:', this.roundToTwo(totalCalories));
        } else {
            console.error('[NUTRIENTS_UPDATE] ❌ No calories container found!');
        }
        
        if (fatElement) {
            fatElement.textContent = this.roundToTwo(totalFat);
            console.log('[NUTRIENTS_UPDATE] ✅ Updated fat to:', this.roundToTwo(totalFat));
        } else {
            console.error('[NUTRIENTS_UPDATE] ❌ No fat container found!');
        }
        
        if (proteinElement) {
            proteinElement.textContent = this.roundToTwo(totalProtein);
            console.log('[NUTRIENTS_UPDATE] ✅ Updated protein to:', this.roundToTwo(totalProtein));
        } else {
            console.error('[NUTRIENTS_UPDATE] ❌ No protein container found!');
        }
        
        if (carbsElement) {
            carbsElement.textContent = this.roundToTwo(totalCarbs);
            console.log('[NUTRIENTS_UPDATE] ✅ Updated carbs to:', this.roundToTwo(totalCarbs));
        } else {
            console.error('[NUTRIENTS_UPDATE] ❌ No carbs container found!');
        }
        
        // Update allergies
        this.updateCardAllergies();
        
        // Update individual food item nutrient displays if they exist
        this.updateFoodItemNutrientDisplays();
        
        // Log the nutrient update for debugging
        this.logNutrientUpdate();
        
        console.log('[NUTRIENTS_UPDATE] 🎉 Card nutrients update completed!');
    }

    getWeightUnit(item) {
        console.log('[WEIGHT_UNIT] ⚖️ getWeightUnit called for item:', item);
        
        const homeUnits = item.getAttribute('data-home_units');
        const weightUnits = item.getAttribute('data-weight_units');
        const selectedHomeUnit = item.getAttribute('data-selected_home_unit');
        
        console.log('[WEIGHT_UNIT] ⚖️ homeUnits:', homeUnits);
        console.log('[WEIGHT_UNIT] ⚖️ weightUnits:', weightUnits);
        console.log('[WEIGHT_UNIT] ⚖️ selectedHomeUnit:', selectedHomeUnit);
        
        if (homeUnits && weightUnits) {
            const homeUnitsArray = homeUnits.split('&');
            const weightUnitsArray = weightUnits.split('&');
            console.log('[WEIGHT_UNIT] ⚖️ homeUnitsArray:', homeUnitsArray);
            console.log('[WEIGHT_UNIT] ⚖️ weightUnitsArray:', weightUnitsArray);
            
            const index = homeUnitsArray.findIndex(unit => String(unit).trim() === String(selectedHomeUnit).trim());
            console.log('[WEIGHT_UNIT] ⚖️ Found index:', index);
            
            const result = parseFloat(weightUnitsArray[index]) || 1;
            console.log('[WEIGHT_UNIT] ⚖️ Returning weight unit:', result);
            return result;
        }
        
        console.log('[WEIGHT_UNIT] ⚖️ No home_units/weight_units found, returning 1');
        return 1;
    }

    roundToTwo(num) {
        return Math.round(num * 100) / 100;
    }

    updateFoodItemNutrients(foodItemElement, newAmount) {
        console.log('[FOOD_NUTRIENTS] 🧮🧮🧮 updateFoodItemNutrients METHOD CALLED! 🧮🧮🧮');
        console.log('[FOOD_NUTRIENTS] 🧮 foodItemElement:', foodItemElement);
        console.log('[FOOD_NUTRIENTS] 🧮 newAmount:', newAmount);
        
        if (!foodItemElement) {
            console.error('[FOOD_NUTRIENTS] ❌ foodItemElement is null in updateFoodItemNutrients!');
            return;
        }
        
        // Get the weight unit for this food item
        const weightUnit = this.getWeightUnit(foodItemElement);
        console.log('[FOOD_NUTRIENTS] ⚖️ Weight unit:', weightUnit);
        
        const amountInGrams = newAmount * weightUnit;
        console.log('[FOOD_NUTRIENTS] ⚖️ Amount in grams:', amountInGrams);

        // Get per 100g nutrient values
        const proteinPer100g = parseFloat(foodItemElement.getAttribute('data-protein_per100g')) || 0;
        const fatPer100g = parseFloat(foodItemElement.getAttribute('data-fat_per100g')) || 0;
        const carbsPer100g = parseFloat(foodItemElement.getAttribute('data-carbohydrate_per100g')) || 0;
        const caloriesPer100g = parseFloat(foodItemElement.getAttribute('data-energy_per100g')) || 0;
        
        console.log('[FOOD_NUTRIENTS] 📊 Per 100g values:', {
            protein: proteinPer100g,
            fat: fatPer100g,
            carbs: carbsPer100g,
            calories: caloriesPer100g
        });

        // Calculate new nutrient values based on amount
        const newProtein = (proteinPer100g * amountInGrams) / 100;
        const newFat = (fatPer100g * amountInGrams) / 100;
        const newCarbs = (carbsPer100g * amountInGrams) / 100;
        const newCalories = (caloriesPer100g * amountInGrams) / 100;
        
        console.log('[FOOD_NUTRIENTS] 🧮 Calculated new values:', {
            protein: newProtein,
            fat: newFat,
            carbs: newCarbs,
            calories: newCalories
        });

        // Update the data attributes with new calculated values
        foodItemElement.setAttribute('data-protein', this.roundToTwo(newProtein));
        foodItemElement.setAttribute('data-fat', this.roundToTwo(newFat));
        foodItemElement.setAttribute('data-carbohydrates', this.roundToTwo(newCarbs));
        foodItemElement.setAttribute('data-energy', this.roundToTwo(newCalories));
        
        console.log('[FOOD_NUTRIENTS] 💾 Data attributes updated');

        console.log(`[FOOD_NUTRIENTS] ✅ Updated nutrients for ${foodItemElement.getAttribute('data-food_name')}:`, {
            amount: newAmount,
            amountInGrams,
            protein: newProtein,
            fat: newFat,
            carbs: newCarbs,
            calories: newCalories
        });
    }

    validateAmountInput(foodItemElement) {
        const amountInput = foodItemElement.querySelector('.item-amount-input');
        let amount = parseFloat(amountInput.value) || 0;
        
        // Ensure minimum amount
        if (amount < 0.5) {
            amount = 0.5;
            amountInput.value = amount;
        }
        
        // Update the data attribute with validated amount
        foodItemElement.setAttribute('data-amount', amount);
        
        // Update nutrients with validated amount
        this.updateFoodItemNutrients(foodItemElement, amount);
        
        // Update the overall card nutrients
        this.updateCardNutrients();
        
        // Update the display
        const foodNameDisplay = foodItemElement.querySelector('.food-name-display');
        if (foodNameDisplay) {
            const foodName = foodItemElement.getAttribute('data-food_name');
            const homeUnit = foodItemElement.getAttribute('data-selected_home_unit');
            foodNameDisplay.textContent = `${foodName} (${amount} ${homeUnit})`;
        }
    }

    updateCardAllergies() {
        // Implementation for updating card allergies
        console.log('Updating card allergies');
    }

    updateFoodItemNutrientDisplays() {
        const foodItems = this.currentCard.querySelectorAll('.card-items .list-group-item');
        
        foodItems.forEach(item => {
            const nutrientDisplay = item.querySelector('.food-nutrient-display');
            if (nutrientDisplay) {
                const protein = parseFloat(item.getAttribute('data-protein')) || 0;
                const fat = parseFloat(item.getAttribute('data-fat')) || 0;
                const carbs = parseFloat(item.getAttribute('data-carbohydrates')) || 0;
                const calories = parseFloat(item.getAttribute('data-energy')) || 0;
                
                nutrientDisplay.innerHTML = `
                    <small class="text-muted">
                        پ: ${this.roundToTwo(protein)}g | 
                        چ: ${this.roundToTwo(fat)}g | 
                        ک: ${this.roundToTwo(carbs)}g | 
                        کالری: ${this.roundToTwo(calories)}
                    </small>
                `;
            }
        });
    }

    logNutrientUpdate() {
        const foodItems = this.currentCard.querySelectorAll('.card-items .list-group-item');
        console.log('=== Card Nutrient Summary ===');
        
        foodItems.forEach((item, index) => {
            const foodName = item.getAttribute('data-food_name');
            const amount = item.getAttribute('data-amount');
            const protein = item.getAttribute('data-protein');
            const fat = item.getAttribute('data-fat');
            const carbs = item.getAttribute('data-carbohydrates');
            const calories = item.getAttribute('data-energy');
            
            console.log(`${index + 1}. ${foodName} (${amount}):`, {
                protein: `${protein}g`,
                fat: `${fat}g`,
                carbs: `${carbs}g`,
                calories: `${calories}`
            });
        });
        
        const totalCalories = this.currentCard.querySelector('.total-calories')?.textContent;
        const totalFat = this.currentCard.querySelector('.total-fat')?.textContent;
        const totalProtein = this.currentCard.querySelector('.total-protein')?.textContent;
        const totalCarbs = this.currentCard.querySelector('.total-carbs')?.textContent;
        
        console.log('=== Totals ===', {
            calories: totalCalories,
            fat: totalFat,
            protein: totalProtein,
            carbs: totalCarbs
        });
        console.log('========================');
    }

    resetFoodItemAmount(foodItemElement) {
        const amountInput = foodItemElement.querySelector('.item-amount-input');
        if (amountInput) {
            amountInput.value = 1;
            this.handleAmountChange(foodItemElement);
            console.log(`Reset amount for ${foodItemElement.getAttribute('data-food_name')} to 1`);
        }
    }

    // Additional methods for card management
    loadCardDataAndAddItems(cardId) {
        // Implementation for loading card data
        console.log('Loading card data for ID:', cardId);
    }

    processCardFoodsQueue(foodsQueue) {
        // Implementation for processing foods queue
        console.log('Processing foods queue:', foodsQueue);
    }

    createFoodItemElementFromData(foodItemData) {
        const foodItemElement = document.createElement('li');
        foodItemElement.classList.add('list-group-item', 'food-item');

        // Set data attributes
        Object.entries(foodItemData).forEach(([key, value]) => {
            if (value !== null && value !== undefined) {
                foodItemElement.setAttribute(`data-${key}`, value);
            }
        });

        foodItemElement.innerHTML = `
            <div class="food-item-content">
                <span class="food-name-display">${foodItemData.food_name} (${foodItemData.home_unit_amount} ${foodItemData.home_unit_name})</span>
                <div class="food-nutrient-display">
                    <small class="text-muted">محاسبه در حال انجام...</small>
                </div>
            </div>
            <span class="main-dish-indicator" style="display: none;">👑</span>
            <div class="item-actions">
                <input type="number" class="item-amount-input" value="${foodItemData.home_unit_amount}" min="0.5" step="0.5">
                <button class="set-main-dish-btn">تعیین غذای اصلی</button>
                <button class="delete-item-btn">حذف</button>
            </div>
        `;

        this.attachFoodItemEventListeners(foodItemElement);
        return foodItemElement;
    }

    saveCard() {
        // Implementation for saving card
        console.log('Saving card');
    }

    collectCardData() {
        // Implementation for collecting card data
        console.log('Collecting card data');
    }

    getTotalCalories() {
        // Implementation for getting total calories
        console.log('Getting total calories');
    }

    saveCardToServer(cardData) {
        // Implementation for saving card to server
        console.log('Saving card to server:', cardData);
    }

    toggleElements() {
        // Implementation for toggling elements
        console.log('Toggling elements');
    }

    toggleElements2() {
        // Implementation for toggling elements 2
        console.log('Toggling elements 2');
    }
}

console.log('[SCRIPT_LOAD] ✅ CardManager class defined');

// Initialize when DOM is loaded
console.log('[SCRIPT_LOAD] 🔧 Setting up DOM ready listener...');

document.addEventListener('DOMContentLoaded', function() {
    console.log('[INIT] 🌐🌐🌐 DOM CONTENT LOADED! 🌐🌐🌐');
    console.log('[INIT] 🌐 Creating new CardManager instance...');
    
    try {
        window.cardManager = new CardManager();
        console.log('[INIT] ✅ CardManager created successfully');
        console.log('[INIT] 🌐 window.cardManager:', window.cardManager);
        console.log('[INIT] 🌐 window.cardManager.handleAmountChange:', typeof window.cardManager.handleAmountChange);
        console.log('[INIT] 🌐 window.cardManager.updateFoodItemNutrients:', typeof window.cardManager.updateFoodItemNutrients);
        console.log('[INIT] 🌐 window.cardManager.updateCardNutrients:', typeof window.cardManager.updateCardNutrients);
    } catch (error) {
        console.error('[INIT] ❌ Error creating CardManager:', error);
    }
    
    console.log('[INIT] 🎉 CardManager initialization completed!');
});

// Fallback initialization if DOM is already loaded
if (document.readyState === 'loading') {
    console.log('[SCRIPT_LOAD] 🔧 DOM still loading, waiting for DOMContentLoaded...');
} else {
    console.log('[SCRIPT_LOAD] 🔧 DOM already loaded, initializing immediately...');
    try {
        window.cardManager = new CardManager();
        console.log('[SCRIPT_LOAD] ✅ CardManager created immediately');
    } catch (error) {
        console.error('[SCRIPT_LOAD] ❌ Error creating CardManager immediately:', error);
    }
} 