/**
 * Card Filters Module
 * Handles all filtering, searching, and sorting functionality for cards
 */

class CardFilters {
    constructor() {
        this.includeTags = [];
        this.excludeTags = [];
        this.minCalories = null;
        this.maxCalories = null;
        this.currentAttribute = null;
        this.initializeEventListeners();
    }

    initializeEventListeners() {
        // Nutrition range modal
        this.setupNutritionRangeModal();
        
        // Calorie modal
        this.setupCalorieModal();
        
        // Search functionality
        this.setupSearch();
        
        // Initialize on page load
        document.addEventListener('DOMContentLoaded', () => {
            this.updateSelectedTags();
            this.attachPaginationEventListeners();
        });
    }

    setupNutritionRangeModal() {
        const nutritionRangeModal = document.getElementById('nutritionRangeModal');
        if (!nutritionRangeModal) return;

        const closeBtn = nutritionRangeModal.querySelector('.close');
        const applyFilterBtn = nutritionRangeModal.querySelector('#applyFilterBtn');

        if (closeBtn) {
            closeBtn.addEventListener('click', () => {
                nutritionRangeModal.style.display = 'none';
            });
        }

        if (applyFilterBtn) {
            applyFilterBtn.addEventListener('click', () => {
                this.applyNutritionFilter();
            });
        }
    }

    setupCalorieModal() {
        const calorieModal = document.getElementById('calorieModal');
        if (!calorieModal) return;

        // Close modal when clicking outside
        window.addEventListener('click', (event) => {
            if (event.target === calorieModal) {
                calorieModal.style.display = 'none';
            }
        });
    }

    setupSearch() {
        const foodSearchCard = document.getElementById('foodSearchCard');
        if (foodSearchCard) {
            foodSearchCard.addEventListener('input', () => {
                this.searchCards();
            });
        }
    }

    openNutritionRangeModal(attribute) {
        this.currentAttribute = attribute;
        const nutritionRangeModal = document.getElementById('nutritionRangeModal');
        if (nutritionRangeModal) {
            nutritionRangeModal.style.display = 'flex';
            document.getElementById('minValue').value = '';
            document.getElementById('maxValue').value = '';
        }
    }

    applyNutritionFilter() {
        const minValue = parseFloat(document.getElementById('minValue').value);
        const maxValue = parseFloat(document.getElementById('maxValue').value);
        
        if (!isNaN(minValue) && !isNaN(maxValue)) {
            this.filterFoodList(minValue, maxValue);
            document.getElementById('nutritionRangeModal').style.display = 'none';
        } else {
            alert('Please enter valid numbers for the range.');
        }
    }

    filterFoodList(minValue, maxValue) {
        const foodItems = document.querySelectorAll('#food-list .list-group-item');
        
        foodItems.forEach(item => {
            const attributeValue = parseFloat(item.getAttribute(`data-${this.currentAttribute}`));
            if (attributeValue >= minValue && attributeValue <= maxValue) {
                item.classList.remove('hidden');
            } else {
                item.classList.add('hidden');
            }
        });
    }

    openCalorieModal() {
        const calorieModal = document.getElementById('calorieModal');
        if (calorieModal) {
            calorieModal.style.display = 'flex';
        }
    }

    closeCalorieModal() {
        const calorieModal = document.getElementById('calorieModal');
        if (calorieModal) {
            calorieModal.style.display = 'none';
        }
    }

    filterByCalories() {
        const minCalories = document.getElementById('minCalories').value;
        const maxCalories = document.getElementById('maxCalories').value;

        const urlParams = new URLSearchParams(window.location.search);

        if (minCalories) {
            urlParams.set('min_calories', minCalories);
        } else {
            urlParams.delete('min_calories');
        }

        if (maxCalories) {
            urlParams.set('max_calories', maxCalories);
        } else {
            urlParams.delete('max_calories');
        }

        // Update the URL without reloading the page
        const newUrl = `${window.location.pathname}?${urlParams.toString()}`;
        window.history.replaceState(null, '', newUrl);

        // Now fetch the filtered cards
        this.fetchCardPage(1);
        this.closeCalorieModal();
    }

    filterCards(tag, action) {
        const urlParams = new URLSearchParams(window.location.search);
        const paramName = action === 'include' ? 'include_tags' : 'exclude_tags';
        let tags = urlParams.getAll(paramName);

        if (tags.includes(tag)) {
            tags = tags.filter(t => t !== tag);
        } else {
            tags.push(tag);
        }

        urlParams.delete(paramName);
        tags.forEach(t => urlParams.append(paramName, t));

        // Update the URL
        const newUrl = `${window.location.pathname}?${urlParams.toString()}`;
        window.history.replaceState(null, '', newUrl);

        // Update selected tags display
        this.updateSelectedTags();

        // Fetch filtered cards
        this.fetchCardPage(1);
    }

    searchCards() {
        const searchTerm = document.getElementById('foodSearchCard').value.toLowerCase();
        const cards = document.querySelectorAll('.card-item');

        cards.forEach(card => {
            const cardName = card.querySelector('.card-name').textContent.toLowerCase();

            // Apply search term and ensure it only affects currently visible (filtered) cards
            if (cardName.includes(searchTerm)) {
                // Keep the card visible only if it also passes the existing filters
                if (this.shouldDisplayCard(card)) {
                    card.style.display = '';
                }
            } else {
                card.style.display = 'none';
            }
        });
    }

    shouldDisplayCard(card) {
        const cardCalories = parseFloat(card.dataset.calories);
        let withinRange = true;

        if (this.minCalories !== null && this.maxCalories !== null) {
            withinRange = (!isNaN(this.minCalories) && cardCalories >= this.minCalories) && 
                         (!isNaN(this.maxCalories) && cardCalories <= this.maxCalories);
        }

        let showCard = withinRange;

        if (showCard) {
            for (const tag of this.includeTags) {
                if (card.dataset[tag] === "False") {
                    showCard = false;
                    break;
                }
            }

            for (const tag of this.excludeTags) {
                if (card.dataset[tag] === "True") {
                    showCard = false;
                    break;
                }
            }
        }

        return showCard;
    }

    updateSelectedTags() {
        const container = document.getElementById('selectedTagsContainer');
        if (!container) return;

        container.innerHTML = '';

        const urlParams = new URLSearchParams(window.location.search);
        const includeTags = urlParams.getAll('include_tags');
        const excludeTags = urlParams.getAll('exclude_tags');

        includeTags.forEach(tag => {
            const tagElement = document.createElement('span');
            tagElement.classList.add('selected-tag', 'include-tag');
            tagElement.textContent = `${tag} (شامل)`;
            tagElement.onclick = () => this.filterCards(tag, 'include');
            container.appendChild(tagElement);
        });

        excludeTags.forEach(tag => {
            const tagElement = document.createElement('span');
            tagElement.classList.add('selected-tag', 'exclude-tag');
            tagElement.textContent = `${tag} (حذف)`;
            tagElement.onclick = () => this.filterCards(tag, 'exclude');
            container.appendChild(tagElement);
        });
    }

    applyFilters() {
        const cards = document.querySelectorAll('.card-item');

        cards.forEach(card => {
            const cardCalories = parseFloat(card.dataset.calories);
            let withinRange = true;

            if (this.minCalories !== null && this.maxCalories !== null) {
                withinRange = (!isNaN(this.minCalories) && cardCalories >= this.minCalories) && 
                             (!isNaN(this.maxCalories) && cardCalories <= this.maxCalories);
            }

            let showCard = withinRange;

            if (showCard) {
                for (const tag of this.includeTags) {
                    if (card.dataset[tag] === "False") {
                        showCard = false;
                        break;
                    }
                }

                for (const tag of this.excludeTags) {
                    if (card.dataset[tag] === "True") {
                        showCard = false;
                        break;
                    }
                }
            }

            if (showCard) {
                card.style.display = "";
                const cardItems = card.querySelectorAll('.card-items .list-group-item');
                cardItems.forEach(item => item.style.display = "");
            } else {
                card.style.display = "none";
                const cardItems = card.querySelectorAll('.card-items .list-group-item');
                cardItems.forEach(item => item.style.display = "none");
            }
        });
    }

    resetFilters() {
        // Clear all filters and sorting parameters from URL
        const urlParams = new URLSearchParams();
        const newUrl = `${window.location.pathname}?${urlParams.toString()}`;
        window.history.replaceState(null, '', newUrl);

        // Reset variables
        this.includeTags = [];
        this.excludeTags = [];
        this.minCalories = null;
        this.maxCalories = null;

        // Remove 'active' class from buttons
        const buttons = document.querySelectorAll('.sort-btn');
        buttons.forEach(button => button.classList.remove('active'));

        // Clear inputs
        const minCaloriesInput = document.getElementById('minCalories');
        const maxCaloriesInput = document.getElementById('maxCalories');
        const foodSearchInput = document.getElementById('foodSearchCard');
        const selectedTagsContainer = document.getElementById('selectedTagsContainer');

        if (minCaloriesInput) minCaloriesInput.value = '';
        if (maxCaloriesInput) maxCaloriesInput.value = '';
        if (foodSearchInput) foodSearchInput.value = '';
        if (selectedTagsContainer) selectedTagsContainer.innerHTML = '';

        // Fetch the default card list
        this.fetchCardPage(1);
    }

    resetFoodFilters() {
        // Clear any input fields related to filtering
        const foodSearch = document.getElementById('foodSearch');
        const minValue = document.getElementById('minValue');
        const maxValue = document.getElementById('maxValue');

        if (foodSearch) foodSearch.value = '';
        if (minValue) minValue.value = '';
        if (maxValue) maxValue.value = '';
        
        this.currentAttribute = null;

        // Reset food items visibility
        const foodItems = document.querySelectorAll('#food-list .list-group-item');
        foodItems.forEach(item => {
            item.classList.remove('hidden');
        });
    }

    sortFoodList(attribute) {
        const foodList = document.getElementById('food-list');
        if (!foodList) return;

        const foods = Array.from(foodList.querySelectorAll('.list-group-item'));

        const sortAttribute = {
            fat: 'data-fat',
            calories: 'data-energy',
            protein: 'data-protein',
            carbohydrate: 'data-carbohydrate'
        }[attribute];

        if (!sortAttribute) return;

        const currentOrder = this.sortOrder[attribute];
        foods.sort((a, b) => {
            const aValue = parseFloat(a.getAttribute(sortAttribute));
            const bValue = parseFloat(b.getAttribute(sortAttribute));
            return currentOrder === 'asc' ? aValue - bValue : bValue - aValue;
        });

        this.sortOrder[attribute] = currentOrder === 'asc' ? 'desc' : 'asc';

        foods.forEach(food => foodList.appendChild(food));
    }

    fetchCardPage(pageNumber) {
        const searchQuery = document.getElementById('foodSearchCard')?.value || '';

        // Get current filters from the URL or variables
        const urlParams = new URLSearchParams(window.location.search);
        const includeTags = urlParams.getAll('include_tags');
        const excludeTags = urlParams.getAll('exclude_tags');
        const minCalories = urlParams.get('min_calories');
        const maxCalories = urlParams.get('max_calories');
        const sortField = urlParams.get('sort_field');
        const sortOrder = urlParams.get('sort_order');

        let url = `/panel/assistant/cards/ajax/?card_page=${pageNumber}`;

        if (searchQuery) {
            url += `&card_search_query=${encodeURIComponent(searchQuery)}`;
        }

        // Include include_tags
        includeTags.forEach(tag => {
            url += `&include_tags=${encodeURIComponent(tag)}`;
        });

        // Include exclude_tags
        excludeTags.forEach(tag => {
            url += `&exclude_tags=${encodeURIComponent(tag)}`;
        });

        // Include calorie filters
        if (minCalories) {
            url += `&min_calories=${encodeURIComponent(minCalories)}`;
        }
        if (maxCalories) {
            url += `&max_calories=${encodeURIComponent(maxCalories)}`;
        }

        // Include sort_field and sort_order
        if (sortField) {
            url += `&sort_field=${encodeURIComponent(sortField)}`;
        }
        if (sortOrder) {
            url += `&sort_order=${encodeURIComponent(sortOrder)}`;
        }

        // Use the built 'url' variable in the AJAX request
        $.ajax({
            url: url,
            type: 'GET',
            success: (response) => {
                // Update the card list with the new data
                $('#cardList').html(response.cards_html);
                // Update pagination controls if necessary
                $('.pagination').html(response.pagination_html);

                // Reattach event listeners to pagination links
                this.attachPaginationEventListeners();
            },
            error: (xhr, status, error) => {
                alert('Error fetching cards. Please try again.');
            }
        });
    }

    fetchFoodPage(pageNumber) {
        const searchQuery = document.getElementById('foodSearch')?.value || '';
        
        let url = `/panel/assistant/foods/ajax/?food_page=${pageNumber}`;
    
        if (searchQuery) {
            url += `&food_search_query=${encodeURIComponent(searchQuery)}`;
        }
    
        $.ajax({
            url: url,
            type: 'GET',
            success: (response) => {
                // Update the food list and pagination controls
                $('#foodsList').html(response.foods_html);
                $('.foods-pagination').html(response.pagination_html);
                // Re-initialize any event listeners on the new elements
                this.initializeFoodItemEventListeners();
            },
            error: (xhr, status, error) => {
                alert('خطا در بارگذاری مواد غذایی. لطفاً دوباره تلاش کنید.');
            }
        });
    }

    attachPaginationEventListeners() {
        // Select all pagination links
        const paginationLinks = document.querySelectorAll('.pagination a');
        paginationLinks.forEach(link => {
            const pageNumber = link.getAttribute('data-page-number');
            if (pageNumber) {
                link.onclick = () => {
                    this.fetchCardPage(pageNumber);
                };
            }
        });
    }

    initializeFoodItemEventListeners() {
        // Handle drag start for new food items
        const foodItems = document.querySelectorAll('.food-item');
        foodItems.forEach(item => {
            item.addEventListener('dragstart', (e) => {
                if (window.cardManager) {
                    window.cardManager.draggedElement = e.target;
                }
                e.dataTransfer.setData('text/plain', '');
            });
        });
    }

    // Initialize sort order tracking
    sortOrder = {
        fat: 'asc',
        calories: 'asc',
        protein: 'asc',
        carbohydrate: 'asc'
    };
}

// Initialize card filters when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.cardFilters = new CardFilters();
});

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = CardFilters;
} 