/**
 * Card Deletion Module
 * Handles all card deletion functionality
 */

class CardDeletion {
    constructor() {
        this.initializeEventListeners();
    }

    initializeEventListeners() {
        // Add event delegation for delete buttons
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('delete-card-btn')) {
                this.handleDeleteClick(e);
            }
        });
    }

    handleDeleteClick(event) {
        event.stopPropagation(); // Prevent the click event from toggling the card
        
        const cardElement = event.target.closest('.card-item');
        if (!cardElement) return;

        const cardId = cardElement.getAttribute('data-id') || 
                      cardElement.querySelector('[data-id]')?.getAttribute('data-id');
        
        if (cardId) {
            this.deleteCard(cardId, cardElement);
        } else {
            console.error('No card ID found for deletion');
        }
    }

    deleteCard(cardId, cardElement) {
        if (confirm('آیا از حذف این کارت مطمئن هستید؟')) {
            this.sendDeleteRequest(cardId, cardElement);
        }
    }

    sendDeleteRequest(cardId, cardElement) {
        // Get CSRF token
        const csrfToken = this.getCSRFToken();
        
        $.ajax({
            url: '/panel/assistant/delete/card/',
            type: 'POST',
            data: {
                'card_id': cardId,
                'csrfmiddlewaretoken': csrfToken
            },
            success: (response) => {
                this.handleDeleteSuccess(response, cardElement);
            },
            error: (xhr, status, error) => {
                this.handleDeleteError(xhr, status, error);
            }
        });
    }

    handleDeleteSuccess(response, cardElement) {
        if (response.status === 'success') {
            // Remove the card element from the DOM
            if (cardElement) {
                cardElement.remove();
            }
            this.showSuccessMessage('کارت با موفقیت حذف شد.');
            
            // Trigger any necessary UI updates
            this.triggerUIUpdate();
        } else {
            this.showErrorMessage('خطا در حذف کارت: ' + response.message);
        }
    }

    handleDeleteError(xhr, status, error) {
        console.error('Delete request failed:', { xhr, status, error });
        this.showErrorMessage('خطا در حذف کارت. لطفاً دوباره تلاش کنید.');
    }

    getCSRFToken() {
        // Try to get CSRF token from meta tag
        const metaTag = document.querySelector('meta[name="csrf-token"]');
        if (metaTag) {
            return metaTag.getAttribute('content');
        }

        // Try to get from window object
        if (window.CSRF_TOKEN) {
            return window.CSRF_TOKEN;
        }

        // Try to get from cookie
        return this.getCookie('csrftoken');
    }

    getCookie(name) {
        let cookieValue = null;
        if (document.cookie && document.cookie !== '') {
            const cookies = document.cookie.split(';');
            for (let i = 0; i < cookies.length; i++) {
                const cookie = cookies[i].trim();
                if (cookie.substring(0, name.length + 1) === (name + '=')) {
                    cookieValue = decodeURIComponent(cookie.substring(name.length + 1));
                    break;
                }
            }
        }
        return cookieValue;
    }

    showSuccessMessage(message) {
        // You can customize this to show a toast notification or alert
        alert(message);
    }

    showErrorMessage(message) {
        // You can customize this to show a toast notification or alert
        alert(message);
    }

    triggerUIUpdate() {
        // Trigger any necessary UI updates after deletion
        // For example, update card count, refresh lists, etc.
        
        // Dispatch custom event for other modules to listen to
        const event = new CustomEvent('cardDeleted', {
            detail: { timestamp: Date.now() }
        });
        document.dispatchEvent(event);
    }

    // Method to programmatically delete a card (useful for other modules)
    deleteCardById(cardId) {
        return new Promise((resolve, reject) => {
            const csrfToken = this.getCSRFToken();
            
            $.ajax({
                url: '/panel/assistant/delete/card/',
                type: 'POST',
                data: {
                    'card_id': cardId,
                    'csrfmiddlewaretoken': csrfToken
                },
                success: (response) => {
                    if (response.status === 'success') {
                        resolve(response);
                    } else {
                        reject(new Error(response.message));
                    }
                },
                error: (xhr, status, error) => {
                    reject(new Error(`Delete failed: ${error}`));
                }
            });
        });
    }

    // Method to delete multiple cards
    deleteMultipleCards(cardIds) {
        const deletePromises = cardIds.map(cardId => this.deleteCardById(cardId));
        
        return Promise.allSettled(deletePromises).then(results => {
            const successful = results.filter(result => result.status === 'fulfilled');
            const failed = results.filter(result => result.status === 'rejected');
            
            return {
                successful: successful.length,
                failed: failed.length,
                total: cardIds.length,
                results: results
            };
        });
    }

    // Method to check if a card can be deleted (permissions, etc.)
    canDeleteCard(cardId) {
        // Add any permission checks here
        // For now, return true (can be customized based on user roles, etc.)
        return true;
    }

    // Method to get deletion confirmation message
    getDeletionConfirmationMessage(cardCount = 1) {
        if (cardCount === 1) {
            return 'آیا از حذف این کارت مطمئن هستید؟';
        } else {
            return `آیا از حذف ${cardCount} کارت مطمئن هستید؟`;
        }
    }
}

// Initialize card deletion when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.cardDeletion = new CardDeletion();
});

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = CardDeletion;
} 