"""
Gemini AI Service for Body Analysis
This service handles image analysis using Google's Gemini AI API
"""

import json
import base64
import os
from typing import Dict, Any, Tuple
import google.generativeai as genai


# Analysis prompt from ANALYSIS_PROMP.txt
ANALYSIS_PROMPT = """شما یک تحلیلگر متخصص ترکیب بدنی هستید. تصویر آنالیز بدن پیوست شده را با دقت کامل و با دو بار چک کردن تحلیل کنید. سپس، یک گزارش دقیق فقط با فرمت JSON بر اساس ساختار زیر ایجاد کنید. پاسخ شما باید فقط و فقط یک شیء JSON معتبر باشد و هیچ متن اضافی، توضیحات یا قالب‌بندی Markdown مانند \`\`\`json ... \`\`\` در ابتدا یا انتهای آن وجود نداشته باشد.


You are a professional body analysis analyst, who is responsible for read the attached body analysis and check it twice to create the following JSON structure (keep in mind that I would share the data structure in TOON format and you must return its JSON format). Then, create a detailed report in JSON format only based on the following structure. Your response should be a valid JSON object only, with no other text, comments, or Markdown formatting like \`\`\`json ... \`\`\` at the beginning or end.

You must extract the following data from the body analysis and mention them in the report, if you didn’t find it then report it as the not found data: 
📝 Report Identifiers & Personal Info

* ID
* Gender
* Age
* Height
* Testing Time


📊 Body Composition Analysis

(Measured in kg, % of weight, and evaluated against a standard range)
* Weight
* Fat Mass (PBF)
* Protein Mass
* Mineral Mass
* Water Weight
* Muscle Mass
* Skeletal Muscle


⚖️ Muscle Fat & Obesity Analysis

* Skeletal Muscle (evaluation level: Low, Standard, High)
* Fat Mass (evaluation level: Low, Standard, High)
* BMI (kg/m2)
* Body Fat Rate (%)


🎯 Body Score & Weight Control

* Body Score (Points)
* Recommended Target Weight
* Weight Control (kg)
* Fat Control (kg)
* Muscle Control (kg)
* Obesity (Current weight/target weight %)


🏃 Segmental Analysis


Segmental Fat Analysis

(For Left Arm, Right Arm, Trunk, Left Leg, Right Leg)
* Fat Mass (kg)
* Fat Proportion (%)
* Fat Evaluation

Segmental Muscle Balance

(For Left Arm, Right Arm, Trunk, Left Leg, Right Leg)
* Muscle Mass (kg)
* Muscle Proportion (%)
* Muscle Evaluation


✨ Other Indicators & Body Evaluation

* Body Type Assessment (e.g., Slim, Muscular Fat, etc.)
* Visceral Fat Grade
* Basal Metabolic Rate (kcal)
* Fat-free Body Weight (kg)
* Subcutaneous Fat (%)
* SMI (kg/m2)
* Body Age
* WHR (Waist-Hip Ratio)
* Upper Body Age Evaluation (Extremely unbalanced, etc.)
* Lower Body Age Evaluation (Extremely unbalanced, etc.)
* Upper-Lower Body Age Evaluation (Extremely unbalanced, etc.)


⚡ Bioelectrical Impedance (Z Ω)

* Impedance at 20kHz for Right Arm
* Impedance at 20kHz for Left Arm
* Impedance at 20kHz for Trunk
* Impedance at 20kHz for Right Leg
* Impedance at 20kHz for Left Leg
* Impedance at 100kHz for Right Arm
* Impedance at 100kHz for Left Arm
* Impedance at 100kHz for Trunk
* Impedance at 100kHz for Right Leg
* Impedance at 100kHz for Left Leg

These data must be Structured in the following Format: report_title: گزارش تحلیل ترکیب بدنی
sections[10]:
  - section_id: 1
    title: شناسایی
    data:
      "نام_و_نام_خانوادگی": string
      "جنسیت": string
      "سن": number
      "قد_سانتیمتر": number
      "زمان_آزمایش": string
  - section_id: 2
    title: تحلیل ترکیب بدنی — امتیاز ترکیب بدنی
    data:
      "امتیاز_ترکیب_بدنی": string
      "توضیح_امتیاز": این امتیاز ارزش کلی ترکیب بدنی شما را نشان می‌دهد و افراد عضلانی‌تر ممکن است امتیازی بالاتر از ۱۰۰ کسب کنند.
      "توصیه_ها"[3]: string,string,string
  - section_id: 3
    title: کنترل وزن
    data:
      "جدول_کنترل_وزن"[4]{"عنوان","مقدار_کنونی","مقدار_هدف_پیشنهادی","تغییر_لازم"}:
        وزن فعلی,string,string,string
        وزن هدف پیشنهادی,—,string,—
        توده چربی,string,string,string
        توده عضله,string,string,string
      "اهمیت_شاخص": string
      "وضعیت_فعلی_شما": string
      "پیشنهادات_عملی_برای_بهبود"[3]: string,string,string
  - section_id: 4
    title: تحلیل اندازه‌گیری (شاخص‌های ترکیب)
    data[6]{"شاخص","مقدار","ارزیابی","توضیح_کوتاه","اقدام_پیشنهادی"}:
      چربی بدن (kg و %),string,string,string,string
      توده استخوانی (kg),string,string,string,string
      توده پروتئینی / پروتئین بدن (kg),string,string,string,string
      آب بدن (kg و %),string,string,string,string
      توده عضلانی (kg),string,string,string,string
      "عضلات اسکلتی (Skeletal muscle, kg)",string,string,string,string
  - section_id: 5
    title: تحلیل چربی و عضله (شاخص‌های چاقی)
    data[3]{"شاخص","مقدار","اهمیت","وضعیت_فعلی","اقدام_پیشنهادی"}:
      BMI,string,string,string,string
      درصد چربی بدن (%),string,string,string,string
      نسبت_وزن_فعلی_به_وزن_هدف,string,string,string,string
  - section_id: 6
    title: تحلیل توزیع چربی بدن (Segmental/Regional)
    data:
      "جدول_توزیع_چربی"[5]{"ناحیه","مقدار_چربی","ارزیابی","اثر_بر_ریسک_متابولیک"}:
        بازو چپ,string,string,string
        بازو راست,string,string,string
        تنه,string,string,string
        پا چپ,string,string,string
        پا راست,string,string,string
      "توضیح_کلی": string
      "راهکارهای_مشخص_برای_بهبود_توزیع_چربی"[3]: string,string,string
  - section_id: 7
    title: تعادل عضلانی (Muscle balance)
    data:
      "جدول_تعادل_عضلانی"[5]{"بخش","مقدار_عضله","ارزیابی_توازن","توصیه"}:
        بازو چپ,string,string,string
        بازو راست,string,string,string
        تنه,string,string,string
        پا چپ,string,string,string
        پا راست,string,string,string
      "توضیح_کلی": string
      "توصیه_های_تمرینی_مشخص_برای_بهبود_تعادل"[3]: string,string,string
  - section_id: 8
    title: مقاومت بیوالکتریک (Bioelectrical Impedance — BIA)
    data:
      "جدول_مقاومت_بیوالکتریک"[1]{"بخش","فرکانس","مقدار_مقاومت_اهم","ارزیابی","معنی"}:
        string,string,number,string,string
      "توضیح_کلی": string
      "توصیه_برای_بهبود_وضعیت_آب_و_الکترولیت"[3]: string,string,string
  - section_id: 9
    title: "پیشنهادات کلی (سه بخش: تغذیه، ورزش، سبک زندگی)"
    data:
      "تغذیه"[1]: string
      "ورزش"[1]: string
      "سبک_زندگی"[1]: string
  - section_id: 11
    title: خلاصه نهایی / جمع‌بندی
    data:
      "خلاصه": string
  - section_id: 12
    title: "Missed data"
    data:
      "Missed data": string
      "Missed data description": string
      
IMPORTANT DISCLAIMER: KEEP IN MIND THAT THE TEXT MUST BE WRITTEN IN FLUENT, HUMANIZED, NATIVE PERSIAN!

IMPORTANT NOTE: IN THE SECTION 4 WHERE WE ARE TALKING ABOUT WEIGHT CONTROL TABLE THERE ARE THREE VALUES: ,"مقدار_کنونی","مقدار_هدف_پیشنهادی","تغییر_لازم" THE مقدار_هدف_پیشنهادی MUST BE CALCULATED BY CALCULATING THE DIFFERENCE BETWEEN : مقدار_کنونی-تغییر_لازم


"""


class GeminiAnalysisService:
    """Service for analyzing body composition images using Gemini AI"""
    
    def __init__(self, api_key: str):
        """
        Initialize Gemini service with API key
        
        Args:
            api_key: Google Gemini API key
        """
        if not api_key:
            raise ValueError("Gemini API key is required")
        
        self.api_key = api_key
        genai.configure(api_key=api_key)
        # Use gemini-2.0-flash - stable, fast, and widely available
        self.model = genai.GenerativeModel('gemini-2.0-flash')
    
    def analyze_body_composition_image(self, image_path: str) -> Tuple[bool, Dict[str, Any], str]:
        """
        Analyze body composition image using Gemini AI
        
        Args:
            image_path: Path to the body composition image file
            
        Returns:
            Tuple of (success: bool, report_data: dict, error_message: str)
        """
        try:
            # Read and encode the image
            with open(image_path, 'rb') as img_file:
                image_data = img_file.read()
            
            # Determine mime type
            mime_type = self._get_mime_type(image_path)
            
            # Create the image part for Gemini
            image_part = {
                "mime_type": mime_type,
                "data": image_data
            }
            
            # Generate content with both image and prompt
            response = self.model.generate_content(
                [image_part, ANALYSIS_PROMPT],
                generation_config={
                    "temperature": 0.0,  # Low temperature for more consistent output
                    "top_p": 0.95,
                    "top_k": 40,
                    "max_output_tokens": 8192,
                }
            )
            
            # Extract text from response
            if not response or not response.text:
                return False, {}, "پاسخی از سرویس تحلیل دریافت نشد"
            
            response_text = response.text.strip()
            
            # Parse JSON response
            report_data = self._parse_json_response(response_text)
            
            if not report_data:
                return False, {}, "پاسخ دریافتی JSON معتبر نبود"
            
            # Apply CTA override (section_id 10)
            report_data = self._override_cta_section(report_data)
            print("--------------------------------")
            print(report_data)
            print("--------------------------------")
            return True, report_data, ""
            
        except FileNotFoundError:
            return False, {}, "فایل تصویر یافت نشد"
        except json.JSONDecodeError as e:
            return False, {}, f"خطا در تجزیه پاسخ JSON: {str(e)}"
        except Exception as e:
            error_message = str(e)
            print(f"Error in Gemini analysis: {error_message}")
            
            # Check for quota errors
            if "429" in error_message or "quota" in error_message.lower():
                return False, {}, "خطا: سهمیه API تمام شده است. لطفا کلید API جدید دریافت کنید یا بعدا امتحان کنید."
            elif "401" in error_message or "unauthorized" in error_message.lower():
                return False, {}, "خطا: کلید API نامعتبر است. لطفا کلید API را بررسی کنید."
            elif "403" in error_message or "forbidden" in error_message.lower():
                return False, {}, "خطا: دسترسی به API رد شد. لطفا تنظیمات API را بررسی کنید."
            else:
                return False, {}, f"خطایی در تحلیل تصویر رخ داد: {error_message}"
    
    def _get_mime_type(self, file_path: str) -> str:
        """Determine MIME type based on file extension"""
        ext = os.path.splitext(file_path)[1].lower()
        mime_types = {
            '.jpg': 'image/jpeg',
            '.jpeg': 'image/jpeg',
            '.png': 'image/png',
            '.gif': 'image/gif',
            '.webp': 'image/webp',
        }
        return mime_types.get(ext, 'image/jpeg')
    
    def _parse_json_response(self, response_text: str) -> Dict[str, Any]:
        """
        Parse JSON response, handling markdown code blocks if present
        
        Args:
            response_text: Raw response text from Gemini
            
        Returns:
            Parsed JSON data or None if parsing fails
        """
        # Remove markdown code blocks if present
        text = response_text.strip()
        
        # Remove ```json ... ``` wrappers
        if text.startswith('```json'):
            text = text[7:]  # Remove ```json
        elif text.startswith('```'):
            text = text[3:]  # Remove ```
        
        if text.endswith('```'):
            text = text[:-3]  # Remove trailing ```
        
        text = text.strip()
        
        try:
            return json.loads(text)
        except json.JSONDecodeError:
            return None
    
    def _override_cta_section(self, report_data: Dict[str, Any]) -> Dict[str, Any]:
        """
        Override the CTA section (section_id 10) with canonical content
        
        Args:
            report_data: The complete report data
            
        Returns:
            Updated report data with overridden CTA
        """
        canonical_cta = {
            "section_id": 10,
            "title": "کال تو اکشن (CTA) — نصب لاکتو و خرید پلن نقره‌ای",
            "data": {
                "عنوان_دعوت_به_عمل": "به جمع لاکتوها بپیوندید!",
                "توضیح": "برای دریافت برنامه‌های تغذیه‌ای شخصی‌سازی شده، پیگیری دقیق پیشرفت و مشاوره با متخصصان تغذیه، اپلیکیشن لاکتو را نصب کنید و از پلن نقره‌ای استفاده کنید.",
                "جمله_پایانی_دعوت_به_عمل": "سلامتی شما در اولویت ماست - همین حالا شروع کنید!",
                "لینک_ها": [
                    "https://lactoo.com/download",
                    "https://lactoo.com/plans/silver"
                ]
            }
        }
        
        # Find and replace section_id 10
        if 'sections' in report_data:
            for i, section in enumerate(report_data['sections']):
                if section.get('section_id') == 10:
                    report_data['sections'][i] = canonical_cta
                    break
        
        return report_data


def analyze_image(image_path: str, api_key: str = None) -> Tuple[bool, Dict[str, Any], str]:
    """
    Convenience function to analyze a body composition image
    
    Args:
        image_path: Path to the image file
        api_key: Optional API key (uses environment variable if not provided)
        
    Returns:
        Tuple of (success: bool, report_data: dict, error_message: str)
    """
    if not api_key:
        api_key = os.getenv('GEMINI_API_KEY', 'AIzaSyD7493IhDDt3wynFhxxdEKF0Ft_aa9K76E')
    
    service = GeminiAnalysisService(api_key=api_key)
    return service.analyze_body_composition_image(image_path)

