"""
Diet Generation Service

This module contains the core diet generation logic extracted from the assistant module.
It provides reusable functions for generating meal plans without requiring authentication.
"""

import random
import hashlib
from datetime import datetime, timedelta, date
from django.db.models import Q
from APIs.models import User as AppUser, FoodCard, UserQuestionnaire, Diet, LifestyleQuestionnaire
from panel.assistant import generate_health_tags, cr_list, calorie_distributions, all_alergic_tags
from panel.models import Assistant


def _get_user_random_seed(user, meal_type, additional_context="", use_timestamp=True):
    """
    Generate a user-specific random seed based on user ID, meal type, and additional context.
    This ensures different users get different results while maintaining consistency for the same user.
    
    Args:
        user: The user object
        meal_type: Type of meal being generated
        additional_context: Additional context for seed generation
        use_timestamp: If True, includes microsecond timestamp for maximum randomness
    """
    if use_timestamp:
        # Include timestamp with microseconds for unique results each time
        timestamp = datetime.now().isoformat()
        seed_string = f"{user.id}_{meal_type}_{additional_context}_{timestamp}"
    else:
        # Use only date for reproducible results on the same day
        seed_string = f"{user.id}_{meal_type}_{additional_context}_{date.today().isoformat()}"
    
    # Hash the string to get a consistent integer seed
    seed_hash = hashlib.md5(seed_string.encode()).hexdigest()
    seed_int = int(seed_hash[:8], 16)  # Use first 8 characters as integer
    
    return seed_int


def _get_questionnaire_value(user, key):
    """
    Get a questionnaire value from either UserQuestionnaire or LifestyleQuestionnaire.
    Checks UserQuestionnaire first (new version), then LifestyleQuestionnaire (old version).
    
    Args:
        user: The user object
        key: The questionnaire key to look for
        
    Returns:
        The value if found, None otherwise
    """
    # Check UserQuestionnaire first (new version)
    user_questionnaire = UserQuestionnaire.objects.filter(user=user).order_by('-id').first()
    if user_questionnaire and user_questionnaire.QandA:
        value = user_questionnaire.QandA.get(key)
        if value:
            return value
    
    # Check LifestyleQuestionnaire (old version)
    lifestyle_questionnaire = LifestyleQuestionnaire.objects.filter(user=user).order_by('-id').first()
    if lifestyle_questionnaire and lifestyle_questionnaire.QandA:
        value = lifestyle_questionnaire.QandA.get(key)
        if value:
            return value
    
    return None


def _get_excluded_meals(user):
    """
    Determine which meals should be excluded (set to 0 calories) based on questionnaire responses.
    
    Args:
        user: The user object
        
    Returns:
        dict: Dictionary with meal names as keys and boolean values indicating if excluded
              e.g., {'breakfast': True, 'lunch': False, 'dinner': False}
    """
    excluded_meals = {
        'breakfast': False,
        'lunch': False,
        'dinner': False
    }
    
    # Get MAIN_MEALS_PER_DAY value
    main_meals = _get_questionnaire_value(user, 'MAIN_MEALS_PER_DAY')
    
    print(f"User {user.id} - MAIN_MEALS_PER_DAY: {main_meals}")
    
    # If user eats 3 or more main meals per day, no exclusions
    if main_meals:
        try:
            # Handle different formats: "2", "[2]", etc.
            main_meals_str = str(main_meals).strip('[]"\' ')
            main_meals_count = int(main_meals_str)
            
            if main_meals_count >= 3:
                print(f"User {user.id} eats {main_meals_count} meals - no exclusions")
                return excluded_meals
        except (ValueError, TypeError):
            print(f"Could not parse MAIN_MEALS_PER_DAY value: {main_meals}")
            return excluded_meals
    
    # Check breakfast-related questions
    breakfast_options = _get_questionnaire_value(user, 'BREAKFAST_OPTIONS')
    breakfast_bread_type = _get_questionnaire_value(user, 'BREAKFAST_BREAD_TYPE')
    
    # Check if breakfast should be excluded
    # If any breakfast field contains "هیچ‌کدام" or "هیچکدام", exclude breakfast
    for value in [breakfast_options, breakfast_bread_type]:
        if value and ('هیچ‌کدام' in str(value) or 'هیچکدام' in str(value)):
            excluded_meals['breakfast'] = True
            print(f"User {user.id} - Breakfast excluded due to: {value}")
            break
    
    # Check lunch preference
    lunch_preference = _get_questionnaire_value(user, 'LUNCH_PREFERENCE')
    if lunch_preference:
        # If contains "هیچ‌کدام" or "قهوه", exclude lunch
        if 'هیچ‌کدام' in str(lunch_preference) or 'هیچکدام' in str(lunch_preference) or 'قهوه' in str(lunch_preference):
            excluded_meals['lunch'] = True
            print(f"User {user.id} - Lunch excluded due to: {lunch_preference}")
    
    # Check dinner preference
    dinner_preference = _get_questionnaire_value(user, 'DINNER_PREFERENCE')
    if dinner_preference:
        # If contains "هیچ‌کدام" or "قهوه", exclude dinner
        if 'هیچ‌کدام' in str(dinner_preference) or 'هیچکدام' in str(dinner_preference) or 'قهوه' in str(dinner_preference):
            excluded_meals['dinner'] = True
            print(f"User {user.id} - Dinner excluded due to: {dinner_preference}")
    
    print(f"User {user.id} - Final excluded meals: {excluded_meals}")
    return excluded_meals


def _calculate_adjusted_calorie_distribution(target_cr, excluded_meals):
    """
    Calculate the appropriate calorie distribution based on target CR and excluded meals.
    Finds the calorie tier where the sum of non-excluded meals is closest to target CR.
    
    Args:
        target_cr: Target calorie requirement
        excluded_meals: Dictionary indicating which meals are excluded
        
    Returns:
        dict: Calorie distribution for all meals (excluded meals will have 0 calories)
    """
    # If no meals are excluded, use standard logic
    if not any(excluded_meals.values()):
        selected_cr = min(cr_list, key=lambda cr: (abs(target_cr - cr), cr))
        return calorie_distributions[selected_cr]
    
    print(f"Calculating adjusted distribution - Target CR: {target_cr}, Excluded: {excluded_meals}")
    
    # Map excluded meals to distribution keys
    meal_key_map = {
        'breakfast': 'B',
        'lunch': 'L',
        'dinner': 'D'
    }
    
    # Calculate sum of non-excluded meals for each calorie tier
    best_cr = None
    best_diff = float('inf')
    
    # Start from calorie tiers at or above target CR
    for cr in cr_list:
        if cr < target_cr:
            continue
            
        distribution = calorie_distributions[cr]
        
        # Sum only non-excluded main meals (B, L, D)
        total = 0
        for meal_name, meal_key in meal_key_map.items():
            if not excluded_meals.get(meal_name, False):
                total += distribution[meal_key]
        
        # Add snacks (always included)
        total += distribution['S1'] + distribution['S2']
        
        print(f"CR {cr}: Total with non-excluded meals = {total}")
        
        # Check if this is close to target (within 5% tolerance)
        diff = abs(total - target_cr)
        tolerance = target_cr * 0.05
        
        if diff <= tolerance and diff < best_diff:
            best_cr = cr
            best_diff = diff
            print(f"CR {cr} is within tolerance (diff: {diff})")
            break
        
        # Keep track of closest even if not within tolerance
        if diff < best_diff:
            best_cr = cr
            best_diff = diff
    
    # If no suitable tier found, use the highest tier
    if best_cr is None:
        best_cr = cr_list[-1]
        print(f"No suitable tier found, using highest: {best_cr}")
    
    print(f"Selected CR tier: {best_cr} (difference: {best_diff})")
    
    # Get the distribution and set excluded meals to 0
    distribution = dict(calorie_distributions[best_cr])
    
    for meal_name, meal_key in meal_key_map.items():
        if excluded_meals.get(meal_name, False):
            distribution[meal_key] = 0
            print(f"Setting {meal_name} ({meal_key}) to 0 calories")
    
    return distribution


def generate_meal_plan(user, meal_type, selected_tags=None, count=14, months_filter=None, return_metadata=False, bypass_zero_calorie=False, use_default_distribution=False, bypassed_meals=None, main_dish_codes=None):
    """
    Generate a meal plan for a specific meal type and user.
    
    This function contains the core logic extracted from refresh_column_cards
    without the authentication requirement.
    
    Args:
        user (AppUser): The user to generate the meal plan for
        meal_type (str): Type of meal ('breakfast', 'lunch', 'dinner', 'snack1', 'snack2')
        selected_tags (dict): Optional tags for filtering food cards
        count (int): Number of meal cards to generate (default: 14)
        months_filter (list): Optional list of Persian month names for filtering
        return_metadata (bool): If True, returns dict with cards and metadata, else just cards list
        bypass_zero_calorie (bool): If True, bypass zero calorie limitation for this meal
        use_default_distribution (bool): If True, use default distribution for ALL meals (when ANY bypass is enabled)
        main_dish_codes (list): Optional list of main_dish_codes to match for same lunch/dinner feature
    
    Returns:
        If return_metadata=False: List of selected food cards with nutritional information
        If return_metadata=True: Dict with 'cards' and 'excluded_meals' keys
    """
    if selected_tags is None:
        selected_tags = {}
    if months_filter is None:
        months_filter = []
    if main_dish_codes is None:
        main_dish_codes = []
    
    try:
        user_profile = user.profile
    except Exception:
        try:
            from panel.telegram_notification import TelegramNotification
            TelegramNotification().send_debug_log(
                f"❌ Missing Profile\nUser ID: {user.id}\ngenerate_meal_plan could not access user.profile"
            )
        except Exception:
            pass
        return []
    
    # Get user's questionnaire data
    questionnaire = UserQuestionnaire.objects.filter(user=user).order_by('-id').first()
    if not questionnaire:
        diet_data = generate_health_tags({}, profile=user_profile)
    else:
        qanda = questionnaire.QandA
        diet_data = generate_health_tags(qanda, profile=user_profile)
    
    if diet_data is None:
        return []
    

    # Work on a copy to avoid mutating shared defaults
    diet_data = dict(diet_data)
    
    # Normalize meal type
    meal_normalized = meal_type if '-' in meal_type else meal_type.replace('snack1', 'snack-1').replace('snack2', 'snack-2')
    
    # Get user's calorie requirement
    patient_cr = getattr(user_profile, "CR", None)
    coach_cr = getattr(user_profile, "CR_coach", None)

    if patient_cr is None and coach_cr is None:
        default_cr = 2000
        try:
            from panel.telegram_notification import TelegramNotification
            TelegramNotification().send_debug_log(
                f"⚠️ Missing CR Values\nUser ID: {user.id}\nUsing default CR {default_cr}"
            )
        except Exception:
            pass
        target_cr = default_cr
    else:
        target_cr = coach_cr if coach_cr is not None else patient_cr

    # Get excluded meals based on questionnaire responses
    excluded_meals = _get_excluded_meals(user)
    
    # Calculate calorie distribution
    # If use_default_distribution is True (any bypass is enabled), use DEFAULT distribution for ALL meals
    # Otherwise, use adjusted distribution that accounts for excluded meals
    if use_default_distribution:
        print(f"⚠️ USE_DEFAULT_DISTRIBUTION is enabled - using DEFAULT distribution for ALL meals")
        selected_cr = min(cr_list, key=lambda cr: (abs(target_cr - cr), cr))
        distribution = calorie_distributions[selected_cr]
        print(f"Using DEFAULT distribution from CR tier {selected_cr}: B={distribution['B']}, L={distribution['L']}, D={distribution['D']}, S1={distribution['S1']}, S2={distribution['S2']}")
    else:
        # Use adjusted distribution that redistributes calories from excluded meals
        distribution = _calculate_adjusted_calorie_distribution(target_cr, excluded_meals)
    
    # Add calorie values to diet_data
    diet_data.update({
        "breakfast_cal": distribution["B"],
        "lunch_cal": distribution["L"],
        "dinner_cal": distribution["D"],
        "snack1_cal": distribution["S1"],
        "snack2_cal": distribution["S2"]
    })
    
    print(f"User {user.id} - Final calorie distribution: B={distribution['B']}, L={distribution['L']}, D={distribution['D']}, S1={distribution['S1']}, S2={distribution['S2']}")
    
    # Get calorie target for this meal type
    calorie_target_key = {
        'breakfast': 'breakfast_cal',
        'lunch': 'lunch_cal',
        'dinner': 'dinner_cal',
        'snack-1': 'snack1_cal',
        'snack-2': 'snack2_cal'
    }.get(meal_normalized, 'breakfast_cal')
    calorie_target = diet_data.get(calorie_target_key, 0)
    
    # Special handling for excluded meals (0 calories)
    # If bypass_zero_calorie is enabled, use DEFAULT calorie distribution (as if no meals were excluded)
    if bypass_zero_calorie and calorie_target == 0:
        print(f"⚠️ Bypass zero calorie enabled for {meal_normalized} - using DEFAULT calorie distribution")
        # Get the DEFAULT distribution directly from calorie_distributions (no exclusions)
        selected_cr = min(cr_list, key=lambda cr: (abs(target_cr - cr), cr))
        default_distribution = calorie_distributions[selected_cr]
        
        # Extract the calorie target for this specific meal from default distribution
        distribution_key = {
            'breakfast': 'B',
            'lunch': 'L',
            'dinner': 'D',
            'snack-1': 'S1',
            'snack-2': 'S2'
        }.get(meal_normalized, 'B')
        
        calorie_target = default_distribution[distribution_key]
        print(f"Using DEFAULT calorie target for {meal_normalized}: {calorie_target} (from CR tier {selected_cr})")
        is_excluded_meal = False
    else:
        is_excluded_meal = calorie_target == 0
    
    if is_excluded_meal:
        print(f"Meal {meal_normalized} is excluded (0 calories) - generating zero-calorie cards")
        # For excluded meals, we'll generate cards with 0 calories
        # This allows the diet page to show "no meal" cards instead of empty slots
    
    # Set user-specific random seed for unique results each time
    # Using timestamp ensures different users with same parameters get different results
    additional_context = f"{selected_tags}_{months_filter}_{count}"
    random.seed(_get_user_random_seed(user, meal_type, additional_context, use_timestamp=True))
    
    # Create allergen exclusion filters
    exclude_allergen_filters = {}
    for tag in all_alergic_tags:
        if diet_data.get('Allergens', {}).get(tag, 0) == 1:
            exclude_allergen_filters[tag] = False  # Exclude cards where allergen is present
    
    # Create exclude filters for diseases
    exclude_filters = {}
    tag_mapping = {
        'Disease_High_Cholesterol': 'Disease_High_cholesterol',
        'Disease_Hypertension': 'Disease_High_blood_pressure',
        'Disease_Thyroid': 'Disease_Hypothyroidism',
        'Allergens_Crustaceans': 'Allergens_Shrimp',
        'Allergens_TreeNuts': 'Allergens_Nuts',
    }
    for tag, value in diet_data.get('Diseases', {}).items():
        if value == 1:
            normalized_tag = tag_mapping.get(tag, tag)
            exclude_filters[normalized_tag] = True
    
    # Meal-specific filters
    meal_filters = {}
    
    if meal_normalized == 'breakfast':
        meal_filters['is_breakfast'] = True
        # Add breakfast-specific tag filters based on main_dish_code
        breakfast_codes = []
        if selected_tags.get('cheese_breakfast'):
            breakfast_codes.extend([1042, 23214, 14370])  # cheese cards
        if selected_tags.get('peanut_butter_breakfast'):
            breakfast_codes.append(16150)  # peanut butter card
        if selected_tags.get('tahini_breakfast'):
            breakfast_codes.append(12166)  # tahini card
        if selected_tags.get('oatmeal_breakfast'):
            breakfast_codes.extend([20038, 42236])  # oatmeal cards
            
        if breakfast_codes:  # Only add the filter if any breakfast codes are selected
            meal_filters['main_dish_code__in'] = breakfast_codes
    elif meal_normalized == 'lunch':
        meal_filters['is_lunch'] = True
    elif meal_normalized == 'dinner':
        meal_filters['is_dinner'] = True
    
    if meal_normalized in ['snack-1', 'snack-2']:
        meal_filters['is_snack'] = True
        # Add snack-specific tag filters
        if selected_tags.get('summer_snack'):
            meal_filters['summer'] = True
        if selected_tags.get('winter_snack'):
            meal_filters['winter'] = True
        if selected_tags.get('spring_snack'):
            meal_filters['spring'] = True
        if selected_tags.get('autumn_snack'):
            meal_filters['autumn'] = True
        # Apply coffee preference per snack column (uses FoodCard.is_coffee)
        if meal_normalized == 'snack-1' and selected_tags.get('coffee_snack1'):
            meal_filters['is_coffee'] = True
            print(f"☕ COFFEE FILTER APPLIED for snack-1: is_coffee=True")
        if meal_normalized == 'snack-2' and selected_tags.get('coffee_snack2'):
            meal_filters['is_coffee'] = True
            print(f"☕ COFFEE FILTER APPLIED for snack-2: is_coffee=True")
        # Backward compatibility for any legacy coffee_snack tag
        if selected_tags.get('coffee_snack'):
            meal_filters['is_coffee'] = True
            print(f"☕ COFFEE FILTER APPLIED (legacy): is_coffee=True")
    
    # Add general tag filters that apply to all meals
    if selected_tags.get("bread"):
        meal_filters['Bread'] = True
    if selected_tags.get("rice"):
        meal_filters['rice'] = True
    if selected_tags.get("salad_main_dish"):
        meal_filters['salad_main_dish'] = True
    if selected_tags.get("sport"):
        meal_filters['sport'] = True
    
    # Build the query
    print(f"🔍 MEAL FILTERS BEFORE QUERY: {meal_filters}")
    print(f"🔍 SELECTED TAGS: {selected_tags}")
    print(f"🔍 MEAL NORMALIZED: {meal_normalized}")
    cards_query = FoodCard.objects.filter(**meal_filters)
    print(f"📊 CARDS MATCHING FILTER: {cards_query.count()} cards")
    
    # Apply khorak filter (exclude cards with rice=True OR bread=True)
    if selected_tags.get("khorak"):
        cards_query = cards_query.filter(rice=False, Bread=False)
    
    # Apply month filtering if specified
    if months_filter and len(months_filter) > 0:
        month_q_objects = []
        for month_name in months_filter:
            # Normalize Persian month names and map to field names
            month_name = month_name.strip()
            # Normalize "اریبهشت" to "اردیبهشت"
            if month_name == "اریبهشت":
                month_name = "اردیبهشت"
            
            month_field_mapping = {
                "فروردین": "is_farvardin",
                "اردیبهشت": "is_ordibehesht",
                "خرداد": "is_khordad", 
                "تیر": "is_tir",
                "مرداد": "is_mordad",
                "شهریور": "is_shahrivar",
                "مهر": "is_mehr",
                "آبان": "is_aban",
                "آذر": "is_azar",
                "دی": "is_dey",
                "بهمن": "is_bahman",
                "اسفند": "is_esfand"
            }
            
            field_name = month_field_mapping.get(month_name)
            if field_name:
                month_q_objects.append(Q(**{field_name: True}))
        
        # Apply OR logic across selected month fields
        if month_q_objects:
            month_filter = month_q_objects[0]
            for q_obj in month_q_objects[1:]:
                month_filter |= q_obj
            cards_query = cards_query.filter(month_filter)
    
    # Apply allergen exclusions
    for field, value in exclude_allergen_filters.items():
        cards_query = cards_query.exclude(**{field: value})
    
    # Apply disease exclusions
    for field, value in exclude_filters.items():
        cards_query = cards_query.exclude(**{field: value})
    
    # Apply calorie filters
    if is_excluded_meal:
        # For excluded meals, get cards with 0 calories
        print(f"Filtering for 0-calorie cards for excluded meal: {meal_normalized}")
        min_calories = 0
        max_calories = 0
        cards_query = cards_query.filter(Calories=0)
    else:
        # Normal calorie filtering with 10% tolerance
        min_calories = calorie_target * 0.9
        max_calories = calorie_target * 1.1
        cards_query = cards_query.filter(Calories__gte=min_calories, Calories__lte=max_calories)
    
    total_cards = cards_query.count()
    
    if total_cards == 0:
        print(f"No cards found for {meal_normalized} with calorie target {calorie_target}")
        return []
    
    # Generate cards based on meal type
    selected_cards = []
    
    # Check if main_dish_codes filtering is needed (for same lunch/dinner feature)
    if main_dish_codes and len(main_dish_codes) > 0:
        print(f"🔄 Same lunch/dinner enabled - Position-based matching by main_dish_codes: {main_dish_codes}")
        selected_cards = []
        
        # Pre-calculate all seeds at once (optimization)
        position_seeds = []
        for idx, main_dish_code in enumerate(main_dish_codes):
            seed = _get_user_random_seed(user, f"{meal_type}_row_{idx}", str(main_dish_code), use_timestamp=True)
            position_seeds.append(seed)
        
        # Group main_dish_codes to batch database queries (optimization)
        valid_codes = {}  # {main_dish_code: [indices]}
        invalid_indices = []
        
        for idx, main_dish_code in enumerate(main_dish_codes):
            if main_dish_code is None or main_dish_code == -1 or main_dish_code == 0:
                invalid_indices.append(idx)
            else:
                if main_dish_code not in valid_codes:
                    valid_codes[main_dish_code] = []
                valid_codes[main_dish_code].append(idx)
        
        # Batch query for all valid main_dish_codes at once (major optimization!)
        if valid_codes:
            # Single query to get all matching cards grouped by main_dish_code
            all_matching_cards = {}
            for main_dish_code in valid_codes.keys():
                matching = list(cards_query.filter(main_dish_code=main_dish_code))
                if matching:
                    all_matching_cards[main_dish_code] = matching
            
            # Now assign cards to positions
            card_assignments = [None] * len(main_dish_codes)
            
            for main_dish_code, indices in valid_codes.items():
                if main_dish_code in all_matching_cards:
                    available_cards = all_matching_cards[main_dish_code]
                    for idx in indices:
                        # Use position-specific seed
                        random.seed(position_seeds[idx])
                        card = random.choice(available_cards)
                        card_assignments[idx] = card
                        print(f"✅ Row {idx+1}: Matched main_dish_code={main_dish_code} → Card: {card.FA_Name}")
                else:
                    # No matching cards found for this main_dish_code
                    for idx in indices:
                        print(f"⚠️ Row {idx+1}: No card found with main_dish_code={main_dish_code}, will generate random")
                        invalid_indices.append(idx)
            
            # Add successfully assigned cards
            for idx, card in enumerate(card_assignments):
                if card is not None:
                    selected_cards.append(card)
        
        # Generate random cards for invalid indices (batch operation)
        if invalid_indices:
            random.seed(position_seeds[0])  # Use first seed for random generation
            print(f"📍 Generating {len(invalid_indices)} random cards for invalid/missing main_dish_codes")
            random_cards = _generate_standard_meal_cards(user, cards_query, len(invalid_indices), meal_type=meal_normalized)
            
            # Insert random cards at their correct positions
            invalid_indices.sort()
            for i, idx in enumerate(invalid_indices):
                if i < len(random_cards):
                    # Insert at the correct position
                    selected_cards.insert(idx, random_cards[i])
                    print(f"📍 Row {idx+1}: Inserted random card")
    elif meal_normalized == 'breakfast':
        # For 0-calorie breakfast, use standard generation (not special 8+6 logic)
        if is_excluded_meal:
            print(f"Using standard generation for 0-calorie breakfast")
            selected_cards = _generate_standard_meal_cards(user, cards_query, count, meal_type=meal_normalized)
        else:
            # Special logic for breakfast: 8 specific + 6 general cards
            selected_cards = _generate_breakfast_cards(user, cards_query, meal_filters, exclude_allergen_filters, exclude_filters, min_calories, max_calories, months_filter)
    else:
        # Standard logic for other meals
        selected_cards = _generate_standard_meal_cards(user, cards_query, count, meal_type=meal_normalized)
    
    # Format response cards
    response_cards = []
    for card in selected_cards:
        response_card = {
            'FA_Name': card.FA_Name,
            'EN_Name': card.EN_Name,
            'Calories': float(card.Calories),
            'foods': card.foods,
            'fat': 0,
            'protein': 0,
            'carbohydrate': 0,
            'is_breakfast': card.is_breakfast,
            'is_lunch': card.is_lunch,
            'is_dinner': card.is_dinner,
            'is_snack': card.is_snack,
            'main_dish_code': card.main_dish_code
        }
        
        # Calculate nutritional values safely
        for food in card.foods:
            if isinstance(food, dict):
                response_card['fat'] += float(food.get('Fat', 0))
                response_card['protein'] += float(food.get('Protein', 0))
                response_card['carbohydrate'] += float(food.get('Carbohydrates', 0))
        
        response_cards.append(response_card)
    
    # Return with metadata if requested
    if return_metadata:
        result = {
            'cards': response_cards,
            'excluded_meals': excluded_meals
        }
        
        # Add biscuit info for snack meals
        if meal_normalized in ['snack-1', 'snack-2']:
            biscuit_count = _count_biscuit_cards(selected_cards)
            result['biscuit_info'] = {
                'count': biscuit_count,
                'max_allowed': 2,
                'is_at_limit': biscuit_count >= 2
            }
            print(f"🍪 BISCUIT INFO: {biscuit_count} biscuit cards in generated {meal_normalized} column")
        
        return result
    
    return response_cards


def _generate_breakfast_cards(user, base_query, meal_filters, exclude_allergen_filters, exclude_filters, min_calories, max_calories, months_filter=None):
    """
    Generate breakfast cards with special logic: 8 specific + 6 general.
    Optimized version using pre-fetched lists.
    """
    
    # Breakfast-specific card filtering
    breakfast_conditions = Q(
        Q(main_dish_code__in=[1042, 23214, 14370]) |  # cheese cards
        Q(main_dish_code__in=[12166, 16150]) |        # peanut/tahini cards
        Q(main_dish_code__in=[20038, 42236])          # oatmeal cards
    )
    
    # Pre-fetch cards to lists for faster in-memory operations (optimization)
    specific_breakfast_cards = list(base_query.filter(breakfast_conditions))
    general_breakfast_cards = list(base_query.filter(is_breakfast=True))
    
    selected_cards = []
    used_main_dish_codes = []
    used_fa_names = []
    
    def select_card_from_pool(card_pool, count):
        """Helper function to select cards with in-memory filtering."""
        for i in range(count):
            if not card_pool:
                break
                
            exclude_codes = [code for code in used_main_dish_codes if code not in [0, -1]]
            
            # Filter in memory (much faster than database queries)
            available_cards = [
                card for card in card_pool
                if card.main_dish_code not in exclude_codes 
                and card.FA_Name not in used_fa_names
            ]
            
            if not available_cards:
                # Fallback 1: Exclude only FA_Name
                available_cards = [
                    card for card in card_pool
                    if card.FA_Name not in used_fa_names
                ]
                
                if not available_cards:
                    # Fallback 2: Exclude only main_dish_code
                    available_cards = [
                        card for card in card_pool
                        if card.main_dish_code not in exclude_codes
                    ]
                    
                    if not available_cards:
                        # Fallback 3: Use all cards
                        available_cards = card_pool
            
            if available_cards:
                card = random.choice(available_cards)
                selected_cards.append(card)
                used_main_dish_codes.append(card.main_dish_code)
                used_fa_names.append(card.FA_Name)
    
    # First, select 8 cards from specific breakfast categories
    select_card_from_pool(specific_breakfast_cards, 8)
    
    # Then, select 6 cards from any breakfast cards
    select_card_from_pool(general_breakfast_cards, 6)
    
    return selected_cards


def _generate_standard_meal_cards(user, cards_query, count=14, meal_type=None):
    """
    Generate cards for non-breakfast meals with standard logic.
    Optimized version with biscuit card limitation for snacks.
    
    Args:
        user: User object
        cards_query: Filtered QuerySet of FoodCard objects
        count: Number of cards to generate
        meal_type: Type of meal (for biscuit restriction in snacks)
    """
    
    selected_cards = []
    used_main_dish_codes = []
    used_fa_names = []
    biscuit_count = 0  # Track biscuit cards for snack limitation
    
    # Pre-fetch all available cards to list for faster operations (optimization)
    all_available_cards = list(cards_query)
    
    if not all_available_cards:
        return selected_cards
    
    # Check if this is a snack meal (needs biscuit restriction)
    is_snack = meal_type in ['snack-1', 'snack-2', 'snack1', 'snack2'] if meal_type else False
    max_biscuits = 2  # Maximum biscuit cards allowed per snack
    
    for i in range(count):
        exclude_codes = [code for code in used_main_dish_codes if code not in [0, -1]]
        
        # Filter in Python instead of multiple database queries (optimization)
        available_cards = [
            card for card in all_available_cards 
            if card.main_dish_code not in exclude_codes 
            and card.FA_Name not in used_fa_names
        ]
        
        # CRITICAL: Apply biscuit restriction for snacks
        if is_snack and biscuit_count >= max_biscuits:
            # Filter out biscuit cards if we've reached the limit
            available_cards = [
                card for card in available_cards
                if not _is_biscuit_card(card)
            ]
            print(f"🍪 BISCUIT LIMIT REACHED ({biscuit_count}/{max_biscuits}) - Excluding biscuit cards from selection")
        
        if not available_cards:
            # Fallback 1: Try excluding only FA_Name
            available_cards = [
                card for card in all_available_cards 
                if card.FA_Name not in used_fa_names
            ]
            
            # Still apply biscuit restriction in fallback
            if is_snack and biscuit_count >= max_biscuits:
                available_cards = [
                    card for card in available_cards
                    if not _is_biscuit_card(card)
                ]
            
            if not available_cards:
                # Fallback 2: Try excluding only main_dish_code
                available_cards = [
                    card for card in all_available_cards 
                    if card.main_dish_code not in exclude_codes
                ]
                
                # Still apply biscuit restriction in fallback
                if is_snack and biscuit_count >= max_biscuits:
                    available_cards = [
                        card for card in available_cards
                        if not _is_biscuit_card(card)
                    ]
                
                if not available_cards:
                    # Fallback 3: Use all cards (but still exclude biscuits if limit reached)
                    if is_snack and biscuit_count >= max_biscuits:
                        available_cards = [
                            card for card in all_available_cards
                            if not _is_biscuit_card(card)
                        ]
                    else:
                        available_cards = all_available_cards
        
        if not available_cards:
            print(f"⚠️ WARNING: No cards available at position {i+1}, stopping generation")
            break
        
        # Select random card from filtered list
        card = random.choice(available_cards)
        selected_cards.append(card)
        used_main_dish_codes.append(card.main_dish_code)
        used_fa_names.append(card.FA_Name)
        
        # Track biscuit cards for snacks
        if is_snack and _is_biscuit_card(card):
            biscuit_count += 1
            print(f"🍪 Selected biscuit card #{biscuit_count} at position {i+1}: {card.FA_Name}")
    
    return selected_cards


def generate_complete_diet_for_user(user_id, questionnaire_event_id=None):
    """
    Generate a complete 14-day diet with 5 meals per day for a user.
    
    This function creates a full diet by calling generate_meal_plan for each meal type
    and assembling the results into a proper Diet instance.
    
    Args:
        user_id (int): ID of the user to generate diet for
        questionnaire_event_id (str): Optional unique identifier for idempotency
    
    Returns:
        tuple: (success: bool, diet_id: int or None, error_message: str or None)
    """
    from panel.telegram_notification import TelegramNotification
    telegram = TelegramNotification()
    
    try:
        # Get the user
        user = AppUser.objects.get(id=user_id)
        
        # Check for existing diet if questionnaire_event_id is provided (idempotency)
        if questionnaire_event_id:
            # Use a more compatible approach for SQLite
            # Check all recent diets for this user and look for the event ID in the JSON
            recent_diets = Diet.objects.filter(user=user).order_by('-created_at')[:10]  # Check last 10 diets
            existing_diet = None
            
            for diet in recent_diets:
                if diet.diet_json and isinstance(diet.diet_json, dict):
                    if diet.diet_json.get('questionnaire_event_id') == questionnaire_event_id:
                        existing_diet = diet
                        break
            
            if existing_diet:
                telegram.send_debug_log(
                    f"ℹ️ Diet Already Exists\nUser ID: {user_id}\nDiet ID: {existing_diet.id}\nSkipping duplicate generation"
                )
                return True, existing_diet.id, None
        
        # Create diet JSON structure
        diet_json = {
            "week1": {
                "breakfast": [],
                "lunch": [],
                "dinner": [],
                "snack1": [],
                "snack2": []
            },
            "type": "Auto-Generated Diet",
            "generated_at": datetime.now().isoformat(),
            "questionnaire_event_id": questionnaire_event_id
        }
        
        # Generate cards for each meal type
        meal_types = ['breakfast', 'lunch', 'dinner', 'snack1', 'snack2']
        generation_success = True
        
        telegram.send_debug_log(
            f"🍽️ Generating Meals\nUser ID: {user_id}\nMeal types: {len(meal_types)}\nGenerating 14 days of meals..."
        )
        
        for meal_type in meal_types:
            try:
                telegram.send_debug_log(
                    f"🥗 Generating {meal_type.title()}\nUser ID: {user_id}\nStatus: Started"
                )
                # Generate meal plan using the service function
                cards = generate_meal_plan(user, meal_type, selected_tags={}, count=14)
                
                if cards:
                    telegram.send_debug_log(
                        f"✅ {meal_type.title()} Ready\nUser ID: {user_id}\nCards generated: {len(cards)}"
                    )
                    # Convert cards to the expected format for diet_json
                    meal_entries = []
                    
                    for card in cards:
                        meal_entry = {
                            "title": card.get('FA_Name', 'Unknown'),
                            "ingredient": []
                        }
                        
                        # Add food items from the card
                        foods = card.get('foods', [])
                        for food in foods:
                            if isinstance(food, dict):
                                ingredient = {
                                    "name": food.get('food_name', food.get('title', 'Unknown')),
                                    "amount_in_home_unit": food.get('home_unit_amount', food.get('amount', '')),
                                    "home_unit": food.get('home_unit_name', food.get('home_unit', '')),
                                    "food_code": food.get('food_code', ''),
                                    "calories": food.get('Calories', 0),
                                    "protein": food.get('Protein', 0),
                                    "fat": food.get('Fat', 0),
                                    "carbohydrates": food.get('Carbohydrates', 0)
                                }
                                meal_entry["ingredient"].append(ingredient)
                        
                        meal_entries.append([meal_entry])
                    
                    # Add to diet_json
                    diet_json["week1"][meal_type] = meal_entries
                else:
                    telegram.send_debug_log(
                        f"⚠️ {meal_type.title()} Empty\nUser ID: {user_id}\nNo cards generated"
                    )
                    # Create empty entries if generation fails
                    diet_json["week1"][meal_type] = [[] for _ in range(14)]
                    generation_success = False
                    
            except Exception as e:
                telegram.send_debug_log(
                    f"❌ {meal_type.title()} Error\nUser ID: {user_id}\nError: {str(e)[:300]}"
                )
                # Create empty entries if generation fails
                diet_json["week1"][meal_type] = [[] for _ in range(14)]
                generation_success = False
        
        # Check if any meals were generated
        meals_generated = 0
        for meal_type in meal_types:
            if diet_json["week1"][meal_type] and any(day for day in diet_json["week1"][meal_type] if day):
                meals_generated += 1
        
        telegram.send_debug_log(
            f"📊 Meal Generation Complete\nUser ID: {user_id}\nSuccess: {meals_generated}/{len(meal_types)} meal types"
        )
        
        # Only create diet if at least some meals were generated successfully
        if meals_generated > 0:
            # Create the diet with start date 24 hours later
            start_date = date.today() + timedelta(days=1)
            
            try:
                assistant_creator = Assistant.objects.filter(django_user__isnull=False).order_by('id').first()
                creator_user = assistant_creator.django_user if assistant_creator else None
                if creator_user is None:
                    try:
                        from panel.telegram_notification import TelegramNotification
                        TelegramNotification().send_debug_log(
                            f"⚠️ No Assistant Creator Found\nUser ID: {user_id}\nDiet will be saved without creator"
                        )
                    except Exception:
                        pass
                
                diet = Diet(
                    user=user,
                    creator=creator_user,
                    diet_json=diet_json,
                    from_date=start_date
                )
                diet.save()
                telegram.send_debug_log(
                    f"💾 Diet Saved to Database\nUser ID: {user_id}\nDiet ID: {diet.id}\nStart Date: {start_date}"
                )
                
                return True, diet.id, None
            except Exception as e:
                error_msg = f"Error saving diet: {str(e)}"
                telegram.send_debug_log(
                    f"❌ Database Error\nUser ID: {user_id}\nError: {error_msg[:300]}"
                )
                return False, None, error_msg
        else:
            error_msg = f"Failed to generate any meal plans for user {user_id} (0/{len(meal_types)} meals generated)"
            telegram.send_debug_log(
                f"❌ No Meals Generated\nUser ID: {user_id}\nProblem: {error_msg[:300]}"
            )
            return False, None, error_msg
        
    except AppUser.DoesNotExist:
        error_msg = f"User {user_id} not found"
        telegram.send_debug_log(
            f"❌ User Not Found\nUser ID: {user_id}\nError: User does not exist in database"
        )
        return False, None, error_msg
    except Exception as e:
        error_msg = f"Error generating diet for user {user_id}: {str(e)}"
        telegram.send_debug_log(
            f"❌ Unexpected Error\nUser ID: {user_id}\nError: {str(e)[:300]}"
        )
        return False, None, error_msg


def start_diet_generation_background(user_id, questionnaire_event_id=None):
    """
    Start diet generation in a background thread.
    
    Args:
        user_id (int): ID of the user to generate diet for
        questionnaire_event_id (str): Optional unique identifier for idempotency
    
    Returns:
        bool: True if background task started successfully
    """
    from threading import Thread
    from panel.telegram_notification import TelegramNotification
    
    def background_task():
        telegram = TelegramNotification()
        try:
            telegram.send_debug_log(
                f"🔄 Background Task Started\nUser ID: {user_id}\nGenerating diet..."
            )
            
            success, diet_id, error = generate_complete_diet_for_user(user_id, questionnaire_event_id)
            
            if success:
                telegram.send_debug_log(
                    f"✅ SUCCESS!\nUser ID: {user_id}\nDiet ID: {diet_id}\nDiet generated successfully"
                )
            else:
                telegram.send_debug_log(
                    f"❌ FAILED\nUser ID: {user_id}\nError: {error}"
                )
        except Exception as e:
            error_msg = str(e)
            telegram.send_debug_log(
                f"❌ EXCEPTION in Background Task\nUser ID: {user_id}\nError: {error_msg[:500]}"
            )
    
    try:
        # Start the background thread
        thread = Thread(target=background_task)
        thread.daemon = True
        thread.start()
        
        return True
    except Exception as e:
        return False


# Biscuit card codes - list of main_dish_code values that represent biscuit cards
BISCUIT_FOOD_CODES = [
    18009, 18172, 14079, 187535, 9880117, 9980242, 9980248, 14347, 
    9980454, 14269, 14273, 14348, 14284, 14330, 20250208172400, 1748159354014
]


def _is_biscuit_code(code):
    """
    Check if a main_dish_code represents a biscuit card.
    
    Args:
        code: main_dish_code (int)
        
    Returns:
        bool: True if the code is a biscuit code, False otherwise
    """
    if not code:
        return False
    return code in BISCUIT_FOOD_CODES


def _is_biscuit_card(card):
    """
    Check if a FoodCard is a biscuit card by checking its main_dish_code.
    
    Args:
        card: FoodCard object
        
    Returns:
        bool: True if the card is a biscuit card, False otherwise
    """
    if not card:
        return False
    
    main_dish_code = getattr(card, 'main_dish_code', None)
    return _is_biscuit_code(main_dish_code)


def _count_biscuit_cards(cards):
    """
    Count the number of biscuit cards in a list of cards.
    
    Args:
        cards: List of FoodCard objects
        
    Returns:
        int: Number of biscuit cards in the list
    """
    if not cards:
        return 0
    
    print("cards: ", cards)
    
    count = 0
    for card in cards:
        if _is_biscuit_card(card):
            count += 1
    
    return count
