from django.db import models
from datetime import datetime, date, timedelta
from django.contrib.auth.models import User as DjangoUser
from APIs.models import User as AppUser, ZiLuckGroup
from django.utils import timezone
from django.db import transaction
from rest_framework import serializers

from typing import List


# from payment.models import ZarinPalPayment


class Field(models.Model):
    name = models.TextField(null=True)

    def __str__(self):
        return self.name


class Manager(models.Model):
    first_name = models.TextField(null=True)
    last_name = models.TextField(null=True)
    created_at = models.DateTimeField(default=datetime.now)
    django_user = models.ForeignKey(DjangoUser, related_name='manager', on_delete=models.CASCADE, null=True)
    image = models.ImageField(upload_to='panel/managers/', null=True)
    telegram_chat_id = models.TextField(default=None, null=True)


class Doctor(models.Model):
    first_name = models.TextField(null=True)
    last_name = models.TextField(null=True)
    gender = models.BooleanField(null=True, default=None)  # 0 --> woman   1 --> man
    field = models.ForeignKey(to=Field, on_delete=models.SET_NULL, null=True)
    manager = models.ForeignKey(to=Manager, on_delete=models.CASCADE)
    django_user = models.ForeignKey(to=DjangoUser, related_name='doctor', on_delete=models.CASCADE, null=True)
    created_at = models.DateTimeField(default=datetime.now)
    last_login = models.DateTimeField(default=datetime.now)
    image = models.ImageField(upload_to='panel/doctors/', null=True)
    is_active = models.BooleanField(default=False)
    phone = models.TextField(default=None, null=True)
    telegram_chat_id = models.TextField(default=None, null=True)

    def __str__(self):
        return "{} {} (therapist)".format(self.first_name, self.last_name)


class Assistant(models.Model):
    first_name = models.TextField(null=True)
    last_name = models.TextField(null=True)
    gender = models.BooleanField(null=True, default=None)  # 0 --> woman   1 --> man
    doctor = models.ForeignKey(to=Doctor, on_delete=models.SET_NULL, null=True)
    manager = models.ForeignKey(to=Manager, on_delete=models.CASCADE)
    django_user = models.ForeignKey(to=DjangoUser, related_name='assistant', on_delete=models.CASCADE, null=True)
    created_at = models.DateTimeField(default=datetime.now)
    last_login = models.DateTimeField(default=datetime.now)
    image = models.ImageField(upload_to='panel/assistants/', null=True)
    telegram_chat_id = models.TextField(default=None, null=True)
    view_chat_page = models.IntegerField(default=0, null=True, blank=True)
    view_chat_page_updated_at = models.DateTimeField(auto_now=True)

    @transaction.atomic
    def increment_view_chat_page(self):
        self.view_chat_page += 1
        self.save(update_fields=['view_chat_page', 'view_chat_page_updated_at'])
        AssistantActivityLog.objects.create(assistant=self, view_chat_page=self.view_chat_page)

    def __str__(self):
        return "{} {} (therapist)".format(self.first_name, self.last_name)


class AssistantActivityLog(models.Model):
    assistant = models.ForeignKey(Assistant, on_delete=models.CASCADE)
    view_chat_page = models.IntegerField()
    timestamp = models.DateTimeField(default=timezone.now)

    def __str__(self):
        return f"{self.assistant} - {self.view_chat_page} - {self.timestamp}"


class MedicalNote(models.Model):
    text = models.TextField(null=True)
    user = models.ForeignKey(to=DjangoUser, on_delete=models.CASCADE)
    writer_name = models.TextField(null=True, blank=True)
    time = models.DateTimeField(default=datetime.now, null=True)
    patient = models.ForeignKey(to=AppUser, on_delete=models.SET_NULL, null=True)


class Attachment(models.Model):
    content_type = models.TextField(null=True)
    is_diet = models.BooleanField(default=False)
    time = models.DateTimeField(default=datetime.now)
    name = models.TextField(default="file")
    path = models.TextField(default=None, null=True)
    size = models.IntegerField(default=None, null=True)
    owner = models.ForeignKey(to=DjangoUser, on_delete=models.SET_NULL, default=None, null=True)

    def get_size(self):
        if self.size is None:
            return 'None size!'
        elif self.size > (1000 * 1000):
            return str(round(self.size / 1000000)) + ' MB'
        elif self.size > 1000:
            return str(round(self.size / 1000)) + ' KB'
        else:
            return str(self.size) + ' B'


class Message(models.Model):
    time = models.DateTimeField(default=datetime.now)
    sender = models.ForeignKey(to=DjangoUser, related_name='sender', on_delete=models.SET_NULL, null=True)
    receiver = models.ForeignKey(to=DjangoUser, related_name='receiver', on_delete=models.SET_NULL, null=True)
    has_been_seen = models.BooleanField(default=False)
    text = models.TextField(default=None, null=True)
    # reply_text = models.TextField(default=None, null=True)
    # reply_message = models.ForeignKey(to=ReplyMessage, on_delete=models.SET_NULL, default=None, null=True)
    attachment = models.ForeignKey(to=Attachment, on_delete=models.SET_NULL, default=None, null=True)
    tags = models.JSONField(default=dict, null=True, blank=True)
    # tags  : {"tags": [extension]}
    reply_message = models.ForeignKey('self', related_name='replies', on_delete=models.SET_NULL, null=True, blank=True)
    is_stared = models.BooleanField(default=False)
    is_edited = models.BooleanField(default=False)
    edited_time = models.DateTimeField(null=True, blank=True)

    def __str__(self):
        return f'Message {self.id}'


class TherapistSelection(models.Model):
    created_at = models.DateTimeField(default=datetime.now)
    user = models.ForeignKey(to=DjangoUser, related_name='applicant', on_delete=models.SET_NULL, null=True)
    therapist = models.ForeignKey(to=DjangoUser, related_name='therapist', on_delete=models.SET_NULL, null=True)
    credit = models.DateTimeField(default=datetime.now)

    def credit_left(self):
        if self.credit.timestamp() < datetime.now().timestamp():  # expired account
            return True
        else:
            return False

    def has_credit(self):
        if self.credit.timestamp() > datetime.now().timestamp():
            return True
        else:
            return False

    def add_credit(self, days: int):
        if self.credit < datetime.now():
            self.credit = datetime.now() + timedelta(days=days)
            print("credit added", days, "days! current credit:", self.credit)
        else:
            self.credit += timedelta(days=days)
            print("Because this account already had some credit, a new credit of {} days was added to it. The current credit is until {}".format(days,self.credit))
        self.save()
        print("credit saved!")

    def remaining_credit(self) -> int:
        return (self.credit.date() - timezone.now().date()).days

    def is_new_subscription(self) -> bool:
        three_days_ago = timezone.now() - timedelta(days=3)
        return self.created_at >= three_days_ago


class DailyNote(models.Model):
    text = models.TextField(null=True, blank=True)
    user = models.ForeignKey(to=DjangoUser, on_delete=models.CASCADE)
    writer_name = models.TextField(null=True, blank=True)
    time = models.DateTimeField(default=datetime.now, null=True, blank=True)
    patient = models.ForeignKey(to=AppUser, on_delete=models.SET_NULL, null=True)


class FirebaseNotification(models.Model):
    user = models.ForeignKey(AppUser, on_delete=models.CASCADE, null=True, blank=True)
    title = models.CharField(max_length=255, default="No title")
    body = models.TextField(default="No body")
    datetime = models.DateTimeField(default=datetime.now)
    seen = models.BooleanField(default=False, null=True, blank=True)


class Challenge(models.Model):
    creator = models.ForeignKey(DjangoUser, on_delete=models.SET_NULL, null=True)
    name = models.TextField(default="")
    reward = models.IntegerField(default=10)
    fee = models.IntegerField(default=0)
    start_deadline = models.DateTimeField(null=True, default=None)
    end_deadline = models.DateTimeField(null=True, default=None)
    description = models.TextField(default="")
    audience = models.ManyToManyField(ZiLuckGroup, blank=True, related_name='challenges')
    audience_users = models.ManyToManyField(AppUser, blank=True, related_name='challenges_audience_users')
    category = models.IntegerField(null=False, default=-1)  # 0=food|1=B.S|2=Exercise|3=B.P|4=Sleep|5=water
    sub_category = models.IntegerField(null=False, default=-1)
    microbiome = models.IntegerField(null=True, default=None, blank=True)
    fat = models.IntegerField(null=True, default=None, blank=True)
    calorie = models.IntegerField(null=True, default=None, blank=True)
    protein = models.IntegerField(null=True, default=None, blank=True)
    carbohydrate = models.IntegerField(null=True, default=None, blank=True)
    more_than = models.IntegerField(null=True, default=None, blank=True)
    tedad_sabt = models.IntegerField(null=True, default=None, blank=True)
    steps = models.IntegerField(null=True, default=None, blank=True)
    calorie_burnt = models.IntegerField(null=True, default=None, blank=True)
    exercise_duration = models.IntegerField(null=True, default=None, blank=True)
    sleep_duration = models.IntegerField(null=True, default=None, blank=True)
    sleep_start_hour = models.TimeField(null=True, default=None, blank=True)
    sleep_end_hour = models.TimeField(null=True, default=None, blank=True)
    participants = models.ManyToManyField(AppUser, related_name='participated_challenges', blank=True)
    food_codes = models.JSONField(default=list, blank=True, null=True)

    def __str__(self):
        return self.name


class ChallengeSelection(models.Model):
    challenge = models.ForeignKey(Challenge, on_delete=models.SET_NULL, null=True)
    user = models.ForeignKey(AppUser, on_delete=models.SET_NULL, null=True)
    day_status = models.JSONField(default=dict, blank=True, null=True)
    # {
    #     "status":[True, True, False ...]
    # }


class TimeSlot(models.Model):
    AVAILABILITY_CHOICES = (
        (0, 'Booked'),
        (1, 'Available'),
        (2, 'Purchased - Waiting Assignment'),
        (3, 'Cancelled'),
    )
    created_at = models.DateTimeField(default=datetime.now)
    patient = models.ForeignKey(AppUser, related_name='patient', on_delete=models.CASCADE, null=True)
    therapist_django_user = models.ForeignKey(DjangoUser, related_name='therapist_django_user', on_delete=models.CASCADE, null=True)
    doctor_django_user = models.ForeignKey(DjangoUser, related_name='doctor_django_user', on_delete=models.CASCADE, null=True)  # Doctor who created the slot
    days_of_week = models.IntegerField(null=True, default=None)
    start_time = models.DateTimeField(null=True, default=None)
    duration = models.IntegerField(null=True, default=15)
    is_available = models.IntegerField(choices=AVAILABILITY_CHOICES, default=1)
    appointment_link = models.TextField(null=True, default=None)
    created_by_doctor = models.BooleanField(default=False)  # True if created by doctor for purchase


class MicrobiomeAnalysis(models.Model):
    user = models.ForeignKey(AppUser, on_delete=models.SET_NULL, null=True)
    created_at = models.DateTimeField(default=datetime.now)
    microbiome_age = models.IntegerField(null=True, default=None)
    microbiome_diversity = models.FloatField(null=True, default=None)
    carbohydrate_metabolism_current = models.IntegerField(null=True, default=None)
    carbohydrate_metabolism_avg = models.IntegerField(null=True, default=None)
    metabolic_score_current = models.IntegerField(null=True, default=None)
    metabolic_score_avg = models.IntegerField(null=True, default=None)
    protein_metabolism_current = models.IntegerField(null=True, default=None)
    protein_metabolism_avg = models.IntegerField(null=True, default=None)
    fat_metabolism_current = models.IntegerField(null=True, default=None)
    fat_metabolism_avg = models.IntegerField(null=True, default=None)
    sleep_quality_current = models.IntegerField(null=True, default=None)
    sleep_quality_avg = models.IntegerField(null=True, default=None)
    lactose_sensitivity_current = models.IntegerField(null=True, default=None)
    lactose_sensitivity_avg = models.IntegerField(null=True, default=None)
    vitamin_synthesis_current = models.IntegerField(null=True, default=None)
    vitamin_synthesis_avg = models.IntegerField(null=True, default=None)
    processed_food_index_current = models.IntegerField(null=True, default=None)
    processed_food_index_avg = models.IntegerField(null=True, default=None)
    sugar_index_current = models.IntegerField(null=True, default=None)
    sugar_index_avg = models.IntegerField(null=True, default=None)
    autoimmune_index_current = models.IntegerField(null=True, default=None)
    autoimmune_index_avg = models.IntegerField(null=True, default=None)
    bowel_mobility_current = models.IntegerField(null=True, default=None)
    bowel_mobility_avg = models.IntegerField(null=True, default=None)
    gluten_sensitivity_current = models.IntegerField(null=True, default=None)
    gluten_sensitivity_avg = models.IntegerField(null=True, default=None)
    antibiotic_damage_current = models.IntegerField(null=True, default=None)
    antibiotic_damage_avg = models.IntegerField(null=True, default=None)


# Manager Dashboard Models for B2B Management

class Lead(models.Model):
    """Model for sales pipeline management"""
    STATUS_CHOICES = (
        ('prospect', 'Prospect'),
        ('qualified', 'Qualified Lead'),
        ('proposal', 'Proposal Sent'),
        ('negotiation', 'In Negotiation'),
        ('closed_won', 'Closed Won'),
        ('closed_lost', 'Closed Lost'),
        ('on_hold', 'On Hold'),
    )
    
    PRIORITY_CHOICES = (
        ('low', 'Low'),
        ('medium', 'Medium'),
        ('high', 'High'),
        ('urgent', 'Urgent'),
    )
    
    company_name = models.CharField(max_length=200, verbose_name="Company Name")
    contact_name = models.CharField(max_length=100, verbose_name="Contact Person")
    contact_email = models.EmailField(verbose_name="Contact Email")
    contact_phone = models.CharField(max_length=20, verbose_name="Contact Phone")
    
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='prospect')
    priority = models.CharField(max_length=10, choices=PRIORITY_CHOICES, default='medium')
    
    estimated_licenses = models.PositiveIntegerField(default=0, verbose_name="Estimated Number of Licenses")
    estimated_value = models.DecimalField(max_digits=12, decimal_places=2, default=0, verbose_name="Estimated Deal Value")
    
    source = models.CharField(max_length=100, blank=True, null=True, verbose_name="Lead Source")
    industry = models.CharField(max_length=100, blank=True, null=True, verbose_name="Industry")
    
    notes = models.TextField(blank=True, null=True, verbose_name="Notes")
    next_action = models.TextField(blank=True, null=True, verbose_name="Next Action Required")
    next_follow_up_date = models.DateField(blank=True, null=True, verbose_name="Next Follow-up Date")
    
    # Tracking
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    created_by = models.ForeignKey(Manager, on_delete=models.SET_NULL, null=True, verbose_name="Created By")
    assigned_to = models.ForeignKey(Manager, on_delete=models.SET_NULL, null=True, related_name='assigned_leads', verbose_name="Assigned To")
    
    # If converted to customer - using ZiLuckGroup instead of Company
    converted_company = models.OneToOneField('APIs.ZiLuckGroup', on_delete=models.SET_NULL, null=True, blank=True, verbose_name="Converted to ZiLuckGroup")
    conversion_date = models.DateTimeField(null=True, blank=True, verbose_name="Conversion Date")
    
    class Meta:
        verbose_name = "Lead"
        verbose_name_plural = "Leads"
        ordering = ['-updated_at']
        indexes = [
            models.Index(fields=['status', 'priority']),
            models.Index(fields=['assigned_to', 'status']),
            models.Index(fields=['next_follow_up_date']),
        ]
    
    def __str__(self):
        return f"{self.company_name} - {self.contact_name}"
    
    @property
    def is_overdue(self):
        """Check if follow-up is overdue"""
        if self.next_follow_up_date:
            return self.next_follow_up_date < date.today()
        return False
    
    def convert_to_company(self):
        """Convert lead to company when deal is closed"""
        if self.status == 'closed_won' and not self.converted_company:
            company = Company.objects.create(
                name=self.company_name,
                primary_contact_name=self.contact_name,
                primary_contact_email=self.contact_email,
                primary_contact_phone=self.contact_phone,
                licenses_purchased=self.estimated_licenses,
                company_code=f"COMP_{self.id}_{self.company_name[:3].upper()}",
                contract_start_date=date.today(),
                contract_end_date=date.today() + timedelta(days=365),  # Default 1 year
                notes=f"Converted from lead #{self.id}"
            )
            self.converted_company = company
            self.conversion_date = timezone.now()
            self.save()
            return company
        return None


class SavedSegment(models.Model):
    """Model for saving user filter combinations"""
    name = models.CharField(max_length=100, verbose_name="Segment Name")
    description = models.TextField(blank=True, null=True, verbose_name="Description")
    
    # Filter criteria stored as JSON
    filter_criteria = models.JSONField(default=dict, verbose_name="Filter Criteria")
    
    # Metadata
    created_by = models.ForeignKey(Manager, on_delete=models.CASCADE, verbose_name="Created By")
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    is_shared = models.BooleanField(default=False, verbose_name="Shared with all managers")
    
    class Meta:
        verbose_name = "Saved Segment"
        verbose_name_plural = "Saved Segments"
        ordering = ['name']
        unique_together = ['name', 'created_by']
    
    def __str__(self):
        return f"{self.name} (by {self.created_by})"
    
    def get_users_count(self):
        """Get count of users matching this segment's criteria"""
        # This method would apply the filter criteria and return count
        # Implementation would depend on the actual filtering logic
        pass


class BodyAnalysisReport(models.Model):
    """Model for storing body composition analysis reports"""
    STATUS_CHOICES = (
        ('pending', 'Pending Review'),
        ('confirmed', 'Confirmed'),
        ('shared', 'Shared with Patient'),
    )
    
    patient = models.ForeignKey(to=AppUser, on_delete=models.CASCADE, related_name='body_analysis_reports')
    created_by = models.ForeignKey(to=DjangoUser, on_delete=models.SET_NULL, null=True, related_name='created_body_reports')
    
    # Image storage
    image = models.ImageField(upload_to='media/bia_uploads/', null=True, blank=True)
    
    # Report data as JSON
    report_data = models.JSONField(null=True, blank=True, verbose_name="Analysis Report Data")
    
    # Status tracking
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='pending')
    
    # Share token for public access
    share_token = models.CharField(max_length=64, unique=True, null=True, blank=True, db_index=True)
    
    # Error tracking
    error_message = models.TextField(null=True, blank=True)
    
    # Timestamps
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    shared_at = models.DateTimeField(null=True, blank=True)
    
    class Meta:
        verbose_name = "Body Analysis Report"
        verbose_name_plural = "Body Analysis Reports"
        ordering = ['-created_at']
    
    def __str__(self):
        return f"Body Analysis for {self.patient} - {self.created_at.strftime('%Y-%m-%d')}"
    
    def mark_as_shared(self):
        """Mark the report as shared with the patient"""
        self.status = 'shared'
        self.shared_at = timezone.now()
        self.save()
    
    def generate_share_token(self):
        """Generate a unique share token for public access"""
        import uuid
        import hashlib
        if not self.share_token:
            # Generate a unique token based on UUID and report ID
            unique_string = f"{uuid.uuid4()}-{self.id}-{self.created_at}"
            self.share_token = hashlib.sha256(unique_string.encode()).hexdigest()
            self.save()
        return self.share_token
    
    def get_public_share_url(self):
        """Get the public shareable URL for this report"""
        if not self.share_token:
            self.generate_share_token()
        from django.conf import settings
        domain = getattr(settings, 'SITE_DOMAIN', 'https://ziluck.probiofit.net')
        return f"{domain}/panel/bia-report/{self.share_token}/"


class Hospital(models.Model):
    """Model for hospital/clinic accounts that collaborate with the platform"""
    name = models.CharField(max_length=200, verbose_name="Hospital Name")
    hospital_code = models.CharField(max_length=50, unique=True, verbose_name="Hospital Code")
    
    # API Credentials
    api_key = models.CharField(max_length=100, unique=True, verbose_name="API Key")
    api_secret = models.CharField(max_length=100, verbose_name="API Secret")
    
    # Contact Information
    contact_email = models.EmailField(verbose_name="Contact Email")
    contact_phone = models.CharField(max_length=20, verbose_name="Contact Phone")
    contact_person = models.CharField(max_length=100, blank=True, null=True, verbose_name="Contact Person")
    
    # Additional Info
    address = models.TextField(blank=True, null=True, verbose_name="Address")
    description = models.TextField(blank=True, null=True, verbose_name="Description")
    
    # Status
    is_active = models.BooleanField(default=True, verbose_name="Is Active")
    
    # Linked Django User for login
    django_user = models.OneToOneField(DjangoUser, on_delete=models.CASCADE, null=True, blank=True, related_name='hospital')
    
    # Tracking
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    created_by = models.ForeignKey(Manager, on_delete=models.SET_NULL, null=True, verbose_name="Created By")
    
    class Meta:
        verbose_name = "Hospital"
        verbose_name_plural = "Hospitals"
        ordering = ['name']
        indexes = [
            models.Index(fields=['hospital_code']),
            models.Index(fields=['api_key']),
        ]
    
    def __str__(self):
        return f"{self.name} ({self.hospital_code})"
    
    def regenerate_api_credentials(self):
        """Regenerate API key and secret"""
        import secrets
        self.api_key = f"HK_{secrets.token_urlsafe(32)}"
        self.api_secret = f"HS_{secrets.token_urlsafe(48)}"
        self.save()
        return self.api_key, self.api_secret
    
    def get_users_count(self):
        """Get count of users belonging to this hospital"""
        from APIs.models import Profile
        return Profile.objects.filter(hospital_code=self.hospital_code).count()
    
    def get_active_upgrades_count(self):
        """Get count of active upgrades for this hospital"""
        from payment.models import HospitalUserUpgrade
        return HospitalUserUpgrade.objects.filter(
            hospital_code=self.hospital_code,
            upgrade_date__gte=datetime.now() - timedelta(days=365)
        ).count()
    
    def create_promotion_group_and_discount(self):
        """
        Create ZiLuckGroup and Discount code for this hospital
        Group name and discount code format: [HOSPITAL_CODE]_PROMOTION
        """
        from payment.models import Discount
        
        promotion_code = f"{self.hospital_code}_PROMOTION"
        
        # Create ZiLuckGroup for this hospital
        ziluck_group, group_created = ZiLuckGroup.objects.get_or_create(
            name=promotion_code,
            defaults={
                'is_public': False,
                'is_company': True,
                'group_id': promotion_code.lower(),
                'creator': self.django_user if self.django_user else None,
                'admin': self.django_user if self.django_user else None,
            }
        )
        
        if group_created:
            print(f"✓ Created ZiLuckGroup: {promotion_code}")
        else:
            print(f"⊘ ZiLuckGroup already exists: {promotion_code}")
        
        # Create Discount code linked to the ZiLuckGroup
        discount, discount_created = Discount.objects.get_or_create(
            code=promotion_code,
            defaults={
                'expire_date': datetime.now() + timedelta(days=3650),  # 10 years
                'percentage': 0,  # Can be set later by manager
                'number_of_discounts': 999999,  # Unlimited
                'ziluck_group': ziluck_group,
            }
        )
        
        if discount_created:
            print(f"✓ Created Discount code: {promotion_code}")
        else:
            print(f"⊘ Discount code already exists: {promotion_code}")
        
        return ziluck_group, discount
    
    def get_promotion_group(self):
        """Get the ZiLuckGroup for this hospital"""
        promotion_code = f"{self.hospital_code}_PROMOTION"
        try:
            return ZiLuckGroup.objects.get(name=promotion_code)
        except ZiLuckGroup.DoesNotExist:
            return None
    
    def get_promotion_discount(self):
        """Get the Discount code for this hospital"""
        promotion_code = f"{self.hospital_code}_PROMOTION"
        try:
            from payment.models import Discount
            return Discount.objects.get(code=promotion_code)
        except:
            return None
