"""
Manager Dashboard Views for B2B Client Management and User Analytics
"""

from django.shortcuts import render, redirect, get_object_or_404
from django.views.decorators.csrf import csrf_exempt
from rest_framework.decorators import api_view
from django.contrib.auth.decorators import login_required
from django.contrib.auth.models import Group
from django.contrib.auth.models import User as DjangoUser
from django.http import HttpResponse, JsonResponse
from datetime import timedelta, datetime, date
from django.contrib import messages
from django.core.cache import cache
from django.db.models import Q, Count, Avg, Sum, Case, When, IntegerField
from django.core.paginator import Paginator, EmptyPage, PageNotAnInteger
from django.utils import translation
from django.conf import settings
import json

from APIs.models import User as AppUser, Profile, Points, Weights, SugarMeasurement, AOneC, ZiLuckGroup, Activities_log, Diet
from panel.models import Manager, Lead, SavedSegment, TherapistSelection, Message
from payment.models import ZarinPalPayment, PaymentPlans
from panel.manager import generate_error_page, new_messages
from panel.PushNotification import PushNotification
import secrets
import string


def manager_required(view_func):
    """Decorator to ensure user has manager permissions"""
    def wrapper(request, *args, **kwargs):
        if not request.user.is_authenticated:
            return redirect('panel-login')
        
        if not (request.user.groups.filter(name='Manager').exists() or request.user.is_superuser):
            return generate_error_page(request, "Access denied. Manager permissions required.")
        
        return view_func(request, *args, **kwargs)
    return wrapper


@api_view(["GET"])
@manager_required
def dashboard_overview(request):
    """Main manager dashboard with key metrics and overview"""
    language = request.session.get(settings.LANGUAGE_COOKIE_NAME, 'en')
    translation.activate(language)
    
    try:
        # Get manager instance
        manager = Manager.objects.get(django_user=request.user) if not request.user.is_superuser else None
        
        # Key business metrics
        total_users = AppUser.objects.count()
        total_companies = ZiLuckGroup.objects.filter(is_company=True).count()
        total_leads = Lead.objects.count()
        
        # Today's metrics
        today = date.today()
        today_users = AppUser.objects.filter(profile__register_time__date=today).count()
        today_active_users = AppUser.objects.filter(profile__last_login__date=today).count()
        
        # Active users (last 30 days)
        thirty_days_ago = datetime.now() - timedelta(days=30)
        active_users = AppUser.objects.filter(profile__last_login__gte=thirty_days_ago).count()
        
        # Revenue metrics
        total_revenue = ZarinPalPayment.objects.filter(is_payed=True).aggregate(
            total=Sum('amount_after_discount')
        )['total'] or 0
        
        monthly_revenue = ZarinPalPayment.objects.filter(
            is_payed=True,
            created_at__gte=datetime.now() - timedelta(days=30)
        ).aggregate(total=Sum('amount_after_discount'))['total'] or 0
        
        # Subscription metrics
        active_subscriptions = TherapistSelection.objects.filter(
            credit__gt=datetime.now()
        ).count()
        
        # Churn risk analysis
        high_risk_users = []
        medium_risk_users = []
        for user in AppUser.objects.select_related('profile').all()[:100]:  # Sample for performance
            risk = user.churn_risk_score
            if risk == "High":
                high_risk_users.append(user)
            elif risk == "Medium":
                medium_risk_users.append(user)
        
        # User Demographics
        users_with_profile = AppUser.objects.select_related('profile').all()
        
        # Gender distribution
        male_users = users_with_profile.filter(profile__gender=0).count()
        female_users = users_with_profile.filter(profile__gender=1).count()
        
        # Age demographics (calculate age ranges)
        age_groups = {'18-25': 0, '26-35': 0, '36-45': 0, '46-60': 0, '60+': 0}
        diabetes_types = {'type1': 0, 'type2': 0, 'pregnancy': 0}
        market_sources = {'Bazaar': 0, 'Myket': 0, 'AppStore': 0}
        
        for user in users_with_profile:
            if not user.profile:
                continue
                
            # Age calculation
            try:
                age = user.profile.age if hasattr(user.profile, 'age') else 0
                if 18 <= age <= 25:
                    age_groups['18-25'] += 1
                elif 26 <= age <= 35:
                    age_groups['26-35'] += 1
                elif 36 <= age <= 45:
                    age_groups['36-45'] += 1
                elif 46 <= age <= 60:
                    age_groups['46-60'] += 1
                elif age > 60:
                    age_groups['60+'] += 1
            except:
                pass
            
            # Diabetes type
            try:
                market_source = user.profile.market
                if market_source == "Bazaar":
                    market_sources['Bazaar'] += 1
                elif market_source == "Myket":
                    market_sources['Myket'] += 1
                elif market_source == "AppStore":
                    market_sources['AppStore'] += 1
            except:
                pass
        
        # Company Demographics
        companies = ZiLuckGroup.objects.filter(is_company=True)
        company_stats = []
        total_company_users = 0
        
        for company in companies:
            company_users = AppUser.objects.filter(django_user__groups=company).count()
            total_company_users += company_users
            company_stats.append({
                'name': company.name,
                'users': company_users,
                'created_at': company.created_at
            })
        
        # Sort companies by user count
        company_stats.sort(key=lambda x: x['users'], reverse=True)
        top_companies = company_stats[:5]
        
        # Activity distribution (last 30 days)
        recent_activity_logs = Activities_log.objects.filter(
            start_date_time__gte=thirty_days_ago
        ).select_related('activity')
        
        activity_distribution = {}
        for log in recent_activity_logs:
            activity_name = getattr(log.activity, 'name', 'Unknown') if hasattr(log, 'activity') else 'Unknown'
            activity_distribution[activity_name] = activity_distribution.get(activity_name, 0) + 1
        
        # Get top 5 activities
        top_activities = sorted(activity_distribution.items(), key=lambda x: x[1], reverse=True)[:5]
        
        # Recent activity
        recent_companies = ZiLuckGroup.objects.filter(is_company=True).order_by('-created_at')[:5]
        
        # Get notifications
        zipped_data, total_new_messages, free_time = new_messages(request)
        total_notifications = len(free_time)
        
        # Convert zipped_data to list to avoid iteration issues
        message = list(zipped_data) if zipped_data else []
        
        print(market_sources)
        
        context = {
            'manager': manager,
            'total_users': total_users,
            'today_users': today_users,
            'today_active_users': today_active_users,
            'total_companies': total_companies,
            'active_users': active_users,
            'total_revenue': total_revenue,
            'monthly_revenue': monthly_revenue,
            'active_subscriptions': active_subscriptions,
            'high_risk_users_count': len(high_risk_users),
            'medium_risk_users_count': len(medium_risk_users),
            # Demographics data
            'male_users': male_users,
            'female_users': female_users,
            'age_groups': age_groups,
            'diabetes_types': diabetes_types,
            'market_source': market_sources,
            'top_companies': top_companies,
            'total_company_users': total_company_users,
            'top_activities': top_activities,
            'recent_companies': recent_companies,
            'message': message,
            'total_new_messages': total_new_messages,
            'free_time': free_time,
            'total_notifications': total_notifications,
        }
        
        return render(request, 'manager/dashboard_overview.html', context)
        
    except Exception as e:
        print(f"Dashboard error: {e}")
        return generate_error_page(request, f"Error loading dashboard: {str(e)}")


@api_view(["GET"])
@manager_required
def user_management(request):
    """User management and analytics view"""
    language = request.session.get(settings.LANGUAGE_COOKIE_NAME, 'en')
    translation.activate(language)
    
    try:
        # Get filter parameters
        search_query = request.GET.get('search', '')
        subscription_filter = request.GET.get('subscription', '')
        activity_filter = request.GET.get('activity', '')
        company_filter = request.GET.get('company', '')
        churn_risk_filter = request.GET.get('churn_risk', '')
        joined_date_filter = request.GET.get('joined_date', '')
        
        # Base queryset
        users = AppUser.objects.select_related('profile').all()
        
        # Apply filters
        if search_query:
            users = users.filter(
                Q(profile__first_name__icontains=search_query) |
                Q(profile__last_name__icontains=search_query) |
                Q(phone_number__icontains=search_query) |
                Q(profile__email__icontains=search_query)
            )
        
        if company_filter:
            # Filter by ZiluckGroup membership
            users = users.filter(django_user__groups__id=company_filter)
        
        if subscription_filter == 'active':
            # Users with active subscriptions
            active_subscription_users = TherapistSelection.objects.filter(
                credit__gt=datetime.now()
            ).values_list('user_id', flat=True)
            users = users.filter(django_user_id__in=active_subscription_users)
        elif subscription_filter == 'expired':
            # Users with expired subscriptions
            expired_subscription_users = TherapistSelection.objects.filter(
                credit__lte=datetime.now()
            ).values_list('user_id', flat=True)
            users = users.filter(django_user_id__in=expired_subscription_users)
        elif subscription_filter == 'free':
            # Users without any subscription
            subscription_users = TherapistSelection.objects.values_list('user_id', flat=True)
            users = users.exclude(django_user_id__in=subscription_users)
        
        if activity_filter == 'active':
            # Users active in last 30 days
            users = users.filter(profile__last_login__gte=datetime.now() - timedelta(days=30))
        elif activity_filter == 'inactive':
            # Users inactive for more than 30 days
            users = users.filter(
                Q(profile__last_login__lt=datetime.now() - timedelta(days=30)) |
                Q(profile__last_login__isnull=True)
            )
        
        # Add joined date filter
        if joined_date_filter:
            if joined_date_filter == 'today':
                today = date.today()
                users = users.filter(profile__register_time__date=today)
            elif joined_date_filter == 'this_week':
                week_ago = date.today() - timedelta(days=7)
                users = users.filter(profile__register_time__date__gte=week_ago)
            elif joined_date_filter == 'this_month':
                month_ago = date.today() - timedelta(days=30)
                users = users.filter(profile__register_time__date__gte=month_ago)
            elif joined_date_filter == 'last_month':
                two_months_ago = date.today() - timedelta(days=60)
                month_ago = date.today() - timedelta(days=30)
                users = users.filter(
                    profile__register_time__date__gte=two_months_ago,
                    profile__register_time__date__lt=month_ago
                )
        
        # Add churn risk filter - calculate actual churn risk for each user
        if churn_risk_filter:
            # We need to filter users based on their actual calculated churn risk
            # This is more accurate but potentially slower for large datasets
            filtered_user_ids = []
            for user in users:
                user_churn_risk = user.churn_risk_score
                if churn_risk_filter == 'high' and user_churn_risk == 'High':
                    filtered_user_ids.append(user.id)
                elif churn_risk_filter == 'medium' and user_churn_risk == 'Medium':
                    filtered_user_ids.append(user.id)
                elif churn_risk_filter == 'low' and user_churn_risk == 'Low':
                    filtered_user_ids.append(user.id)
            
            users = users.filter(id__in=filtered_user_ids)
        
        # Pagination
        paginator = Paginator(users, 25)
        page_number = request.GET.get('page')
        page_users = paginator.get_page(page_number)
        
        # Get available ZiluckGroups for filter dropdown (companies only)
        companies = ZiLuckGroup.objects.filter(is_company=True).order_by('name')
        
        # Get notifications
        zipped_data, total_new_messages, free_time = new_messages(request)
        total_notifications = len(free_time)
        
        # Convert zipped_data to list to avoid iteration issues
        message = list(zipped_data) if zipped_data else []
        
        context = {
            'users': page_users,
            'companies': companies,
            'search_query': search_query,
            'subscription_filter': subscription_filter,
            'activity_filter': activity_filter,
            'company_filter': company_filter,
            'churn_risk_filter': churn_risk_filter,
            'joined_date_filter': joined_date_filter,
            'message': message,
            'total_new_messages': total_new_messages,
            'free_time': free_time,
            'total_notifications': total_notifications,
        }
        
        return render(request, 'manager/user_management.html', context)
        
    except Exception as e:
        print(f"User management error: {e}")
        return generate_error_page(request, f"Error loading user management: {str(e)}")


@api_view(["GET", "POST"])
@manager_required
def user_profile_view(request, user_id):
    """Enhanced user profile view with graphs, analytics, and notification sending"""
    language = request.session.get(settings.LANGUAGE_COOKIE_NAME, 'en')
    translation.activate(language)
    
    try:
        user = get_object_or_404(AppUser, id=user_id)
        
        # Handle notification sending (POST request)
        if request.method == "POST":
            notification_title = request.POST.get('notification_title', '')
            notification_body = request.POST.get('notification_body', '')
            
            if notification_title and notification_body:
                push_notification = PushNotification(user)
                try:
                    success = push_notification.send_notification_to_user(user, notification_title, notification_body)
                    if success:
                        messages.success(request, f"Notification sent successfully to {user.profile.first_name or user.phone_number}")
                    else:
                        messages.error(request, "Failed to send notification")
                except Exception as e:
                    messages.error(request, f"Error sending notification: {str(e)}")
            else:
                messages.error(request, "Both title and body are required")
            
            return redirect('panel-manager-user-profile', user_id=user_id)
        
        # Get user's health metrics
        biometrics = user.latest_biometrics
        lacto_points = user.lacto_points_score
        churn_risk = user.churn_risk_score
        subscription_info = user.subscription_info
        
        # 1. Get 30-day points trend data
        thirty_days_ago = date.today() - timedelta(days=30)
        points_trend = Points.objects.filter(user=user, date__gte=thirty_days_ago).order_by('date')
        
        # Prepare data for chart
        trend_data = []
        for point in points_trend:
            trend_data.append({
                'date': point.date.strftime('%Y-%m-%d'),
                'points': point.total_points
            })
        
        # Calculate average lacto points (last 30 days)
        if trend_data:
            avg_lacto_points = sum(p['points'] for p in trend_data) / len(trend_data)
        else:
            avg_lacto_points = lacto_points
        
        # 2. Get activity logs data for bar chart
        activity_logs = Activities_log.objects.filter(
            user=user, 
            start_date_time__gte=datetime.now() - timedelta(days=30)
        )
        
        # Count activities by type
        activity_counts = {}
        for log in activity_logs:
            activity_name = getattr(log.activity, 'name', 'Unknown') if hasattr(log, 'activity') else 'Unknown'
            activity_counts[activity_name] = activity_counts.get(activity_name, 0) + 1
        
        # Sort by count and take top 10
        top_activities = sorted(activity_counts.items(), key=lambda x: x[1], reverse=True)[:10]
        
        # 3. Check if user is premium and get coach info
        coach_info = None
        is_premium = False
        if user.profile.therapist:
            is_premium = True
            coach_info = {
                'name': f"{user.profile.therapist.first_name} {user.profile.therapist.last_name}",
                'email': user.profile.therapist.email,
                'id': user.profile.therapist.id
            }
        
        # 4. Get diet history if premium user
        diet_history = []
        if is_premium:
            diets = Diet.objects.filter(user=user).order_by('-created_at')
            for diet in diets:
                diet_history.append({
                    'id': diet.id,
                    'created_at': diet.created_at,
                    'from_date': diet.from_date,
                    'creator': diet.creator.get_full_name() if diet.creator else 'System',
                    'has_data': bool(diet.diet_json)
                })
        
        # Get payment history
        payment_history = ZarinPalPayment.objects.filter(user=user, is_payed=True).order_by('-created_at')
        
        # Get notifications
        zipped_data, total_new_messages, free_time = new_messages(request)
        total_notifications = len(free_time)
        
        # Convert zipped_data to list to avoid iteration issues
        message = list(zipped_data) if zipped_data else []
        
        context = {
            'user': user,
            'biometrics': biometrics,
            'lacto_points': lacto_points,
            'avg_lacto_points': round(avg_lacto_points, 1),
            'churn_risk': churn_risk,
            'subscription_info': subscription_info,
            'recent_points': points_trend,
            'payment_history': payment_history,
            'trend_data': json.dumps(trend_data),  # For JavaScript chart
            'top_activities': top_activities,
            'is_premium': is_premium,
            'coach_info': coach_info,
            'diet_history': diet_history,
            'message': message,
            'total_new_messages': total_new_messages,
            'free_time': free_time,
            'total_notifications': total_notifications,
        }
        
        return render(request, 'manager/user_profile_view.html', context)
        
    except Exception as e:
        print(f"User profile view error: {e}")
        return generate_error_page(request, f"Error loading user profile: {str(e)}")


@api_view(["GET"])
@manager_required
def company_management(request):
    """B2B company management view"""
    language = request.session.get(settings.LANGUAGE_COOKIE_NAME, 'en')
    translation.activate(language)
    
    try:
        # Get filter parameters
        search_query = request.GET.get('search', '')
        company_type_filter = request.GET.get('company_type', '')
        
        # Base queryset
        companies = ZiLuckGroup.objects.filter(is_company=True)
        
        # Apply filters
        if search_query:
            companies = companies.filter(
                Q(name__icontains=search_query) |
                Q(company_code__icontains=search_query) |
                Q(primary_contact_name__icontains=search_query)
            )
        
        if company_type_filter:
            companies = companies.filter(company_type=company_type_filter)
        
        # Order by latest
        companies = companies.order_by('-created_at')
        
        # Add HR account info for each company
        companies_with_hr = []
        for company in companies:
            hr_username = f"hr_{company.group_id or company.id}"
            hr_user = DjangoUser.objects.filter(username=hr_username).first()
            company.has_hr_account = hr_user is not None
            company.hr_username = hr_username if company.has_hr_account else None
            companies_with_hr.append(company)
        
        # Pagination
        paginator = Paginator(companies_with_hr, 20)
        page_number = request.GET.get('page')
        page_companies = paginator.get_page(page_number)
        
        # Get notifications
        zipped_data, total_new_messages, free_time = new_messages(request)
        total_notifications = len(free_time)
        
        # Convert zipped_data to list to avoid iteration issues
        message = list(zipped_data) if zipped_data else []
        
        context = {
            'companies': page_companies,
            'search_query': search_query,
            'company_type_filter': company_type_filter,
            # Removed Company.COMPANY_TYPE_CHOICES - using ZiLuckGroup instead
            'message': message,
            'total_new_messages': total_new_messages,
            'free_time': free_time,
            'total_notifications': total_notifications,
        }
        
        return render(request, 'manager/company_management.html', context)
        
    except Exception as e:
        print(f"Company management error: {e}")
        return generate_error_page(request, f"Error loading company management: {str(e)}")


@api_view(["GET"])
@manager_required
def company_profile_view(request, company_id):
    """Detailed view of a specific ZiluckGroup company with comprehensive analytics"""
    language = request.session.get(settings.LANGUAGE_COOKIE_NAME, 'en')
    translation.activate(language)
    
    try:
        company = get_object_or_404(ZiLuckGroup, id=company_id, is_company=True)
        
        # Get company users (via group membership)
        company_users = AppUser.objects.filter(django_user__groups=company).select_related('profile').all()
        
        # Calculate aggregated metrics for the company
        total_users = company_users.count()
        
        # Weekly active users (last 7 days)
        week_ago = datetime.now() - timedelta(days=7)
        weekly_active_users = company_users.filter(
            profile__last_login__gte=week_ago
        ).count()
        
        # Monthly active users (last 30 days)
        month_ago = datetime.now() - timedelta(days=30)
        monthly_active_users = company_users.filter(
            profile__last_login__gte=month_ago
        ).count()
        
        # Average Lacto Points for company users
        user_points = []
        high_risk_count = 0
        medium_risk_count = 0
        total_protein = 0
        total_calories = 0
        total_fat = 0
        total_carbs = 0
        nutrition_users_count = 0
        
        for user in company_users:
            # Lacto points
            points = user.lacto_points_score
            user_points.append(points)
            
            # Churn risk analysis
            risk = user.churn_risk_score
            if risk == "High":
                high_risk_count += 1
            elif risk == "Medium":
                medium_risk_count += 1
            
            # Get latest nutrition data from Points model
            try:
                latest_points = Points.objects.filter(user=user).order_by('-date').first()
                if latest_points:
                    # Extract nutrition data from Points model - handle None values
                    protein_str = latest_points.this_day_Protein_list_g or "&0&"
                    carb_str = latest_points.this_day_Carbohydrates_g or "&0&"
                    fat_str = latest_points.this_day_Fat_list_g or "&0&"
                    
                    # Safe split operations
                    protein_data = protein_str.split('&')[1:-1] if protein_str and '&' in protein_str else []
                    carb_data = carb_str.split('&')[1:-1] if carb_str and '&' in carb_str else []
                    fat_data = fat_str.split('&')[1:-1] if fat_str and '&' in fat_str else []
                    
                    # Sum up daily nutrition with safe float conversion
                    if protein_data:
                        for x in protein_data:
                            try:
                                if x and x != '0':
                                    total_protein += float(x)
                            except (ValueError, TypeError):
                                continue
                    if carb_data:
                        for x in carb_data:
                            try:
                                if x and x != '0':
                                    total_carbs += float(x)
                            except (ValueError, TypeError):
                                continue
                    if fat_data:
                        for x in fat_data:
                            try:
                                if x and x != '0':
                                    total_fat += float(x)
                            except (ValueError, TypeError):
                                continue
                    
                    total_calories += latest_points.total_Calories_used or 0
                    nutrition_users_count += 1
            except Exception as e:
                print(f"Error getting nutrition for user {user.id}: {e}")
        
        # Calculate averages
        avg_lacto_points = sum(user_points) / len(user_points) if user_points else 0
        avg_protein = total_protein / nutrition_users_count if nutrition_users_count > 0 else 0
        avg_calories = total_calories / nutrition_users_count if nutrition_users_count > 0 else 0
        avg_fat = total_fat / nutrition_users_count if nutrition_users_count > 0 else 0
        avg_carbs = total_carbs / nutrition_users_count if nutrition_users_count > 0 else 0
        
        # Calculate derived values for template
        inactive_users = total_users - weekly_active_users
        low_risk_count = total_users - high_risk_count - medium_risk_count
        
        # Message statistics (sent and received)
        company_user_django_ids = [user.django_user.id for user in company_users if user.django_user]
        
        # Messages sent by company users
        messages_sent = Message.objects.filter(
            sender__id__in=company_user_django_ids
        ).count()
        
        # Messages received by company users
        messages_received = Message.objects.filter(
            receiver__id__in=company_user_django_ids
        ).count()
        
        # Recent messages (last 30 days)
        recent_messages_sent = Message.objects.filter(
            sender__id__in=company_user_django_ids,
            time__gte=month_ago
        ).count()
        
        recent_messages_received = Message.objects.filter(
            receiver__id__in=company_user_django_ids,
            time__gte=month_ago
        ).count()
        
        # Pagination for users
        paginator = Paginator(company_users, 20)
        page_number = request.GET.get('page')
        page_users = paginator.get_page(page_number)
        
        # Get notifications
        zipped_data, total_new_messages, free_time = new_messages(request)
        total_notifications = len(free_time)
        
        # Convert zipped_data to list to avoid iteration issues
        message = list(zipped_data) if zipped_data else []
        
        context = {
            'company': company,
            'users': page_users,
            'total_users': total_users,
            'weekly_active_users': weekly_active_users,
            'monthly_active_users': monthly_active_users,
            'inactive_users': inactive_users,
            'avg_lacto_points': round(avg_lacto_points, 1),
            'high_risk_count': high_risk_count,
            'medium_risk_count': medium_risk_count,
            'low_risk_count': low_risk_count,
            'avg_protein': round(avg_protein, 1),
            'avg_calories': round(avg_calories, 0),
            'avg_fat': round(avg_fat, 1),
            'avg_carbs': round(avg_carbs, 1),
            'messages_sent': messages_sent,
            'messages_received': messages_received,
            'recent_messages_sent': recent_messages_sent,
            'recent_messages_received': recent_messages_received,
            'total_messages': messages_sent + messages_received,
            'message': message,
            'total_new_messages': total_new_messages,
            'free_time': free_time,
            'total_notifications': total_notifications,
        }
        
        return render(request, 'manager/company_profile_view.html', context)
        
    except Exception as e:
        print(f"Company profile view error: {e}")
        return generate_error_page(request, f"Error loading company profile: {str(e)}")


@api_view(["GET"])
@manager_required
def sales_pipeline(request):
    """Sales pipeline management view"""
    language = request.session.get(settings.LANGUAGE_COOKIE_NAME, 'en')
    translation.activate(language)
    
    try:
        # Get filter parameters
        status_filter = request.GET.get('status', '')
        priority_filter = request.GET.get('priority', '')
        assigned_filter = request.GET.get('assigned', '')
        
        # Base queryset
        leads = Lead.objects.select_related('created_by', 'assigned_to').all()
        
        # Apply filters
        if status_filter:
            leads = leads.filter(status=status_filter)
        
        if priority_filter:
            leads = leads.filter(priority=priority_filter)
        
        if assigned_filter:
            leads = leads.filter(assigned_to_id=assigned_filter)
        
        # Order by priority and update date
        leads = leads.order_by('-priority', '-updated_at')
        
        # Get leads by status for Kanban view and counts
        leads_by_status = {}
        status_counts = {}
        for status_code, status_name in Lead.STATUS_CHOICES:
            status_leads = Lead.objects.filter(status=status_code).order_by('-updated_at')
            leads_by_status[status_code] = status_leads
            status_counts[status_code] = status_leads.count()
        
        # Get managers for assignment dropdown
        managers = Manager.objects.all()
        
        # Get notifications
        zipped_data, total_new_messages, free_time = new_messages(request)
        total_notifications = len(free_time)
        
        # Convert zipped_data to list to avoid iteration issues
        message = list(zipped_data) if zipped_data else []
        
        context = {
            'leads': leads,
            'leads_by_status': leads_by_status,
            'status_counts': status_counts,
            'managers': managers,
            'status_filter': status_filter,
            'priority_filter': priority_filter,
            'assigned_filter': assigned_filter,
            'status_choices': Lead.STATUS_CHOICES,
            'priority_choices': Lead.PRIORITY_CHOICES,
            'message': message,
            'total_new_messages': total_new_messages,
            'free_time': free_time,
            'total_notifications': total_notifications,
        }
        
        return render(request, 'manager/sales_pipeline.html', context)
        
    except Exception as e:
        print(f"Sales pipeline error: {e}")
        return generate_error_page(request, f"Error loading sales pipeline: {str(e)}")


@api_view(["GET", "POST"])
@manager_required
def user_segmentation(request):
    """User filtering and segmentation system"""
    language = request.session.get(settings.LANGUAGE_COOKIE_NAME, 'en')
    translation.activate(language)
    
    try:
        if request.method == 'POST':
            # Save a new segment
            segment_name = request.POST.get('segment_name')
            segment_description = request.POST.get('segment_description', '')
            filter_criteria = {
                'subscription_status': request.POST.get('subscription_status', ''),
                'activity_level': request.POST.get('activity_level', ''),
                'lacto_points_min': request.POST.get('lacto_points_min', ''),
                'lacto_points_max': request.POST.get('lacto_points_max', ''),
                'churn_risk': request.POST.get('churn_risk', ''),
                'company': request.POST.get('company', ''),
                'health_condition': request.POST.get('health_condition', ''),
                'last_active_days': request.POST.get('last_active_days', ''),
            }
            
            # Remove empty criteria
            filter_criteria = {k: v for k, v in filter_criteria.items() if v}
            
            if segment_name and filter_criteria:
                manager = Manager.objects.get(django_user=request.user) if not request.user.is_superuser else None
                SavedSegment.objects.create(
                    name=segment_name,
                    description=segment_description,
                    filter_criteria=filter_criteria,
                    created_by=manager
                )
                messages.success(request, f"Segment '{segment_name}' saved successfully!")
            else:
                messages.error(request, "Please provide a name and at least one filter criterion.")
        
        # Get saved segments
        saved_segments = SavedSegment.objects.filter(
            Q(created_by__django_user=request.user) | Q(is_shared=True)
        ).order_by('-created_at')
        
        # Get companies for filter dropdown
        companies = ZiLuckGroup.objects.filter(is_company=True).order_by('name')
        
        # Apply current filters to show results
        users = AppUser.objects.select_related('profile').all()
        
        # Apply filters based on GET parameters
        subscription_status = request.GET.get('subscription_status', '')
        activity_level = request.GET.get('activity_level', '')
        lacto_points_min = request.GET.get('lacto_points_min', '')
        lacto_points_max = request.GET.get('lacto_points_max', '')
        churn_risk = request.GET.get('churn_risk', '')
        company_filter = request.GET.get('company', '')
        health_condition = request.GET.get('health_condition', '')
        last_active_days = request.GET.get('last_active_days', '')
        
        # Apply filters (similar to user_management but more detailed)
        if subscription_status == 'active':
            active_subscription_users = TherapistSelection.objects.filter(
                credit__gt=datetime.now()
            ).values_list('user_id', flat=True)
            users = users.filter(django_user_id__in=active_subscription_users)
        elif subscription_status == 'expired':
            expired_subscription_users = TherapistSelection.objects.filter(
                credit__lte=datetime.now()
            ).values_list('user_id', flat=True)
            users = users.filter(django_user_id__in=expired_subscription_users)
        elif subscription_status == 'free':
            subscription_users = TherapistSelection.objects.values_list('user_id', flat=True)
            users = users.exclude(django_user_id__in=subscription_users)
        
        if company_filter:
            users = users.filter(company_id=company_filter)
        
        if last_active_days:
            cutoff_date = datetime.now() - timedelta(days=int(last_active_days))
            if activity_level == 'active':
                users = users.filter(profile__last_login__gte=cutoff_date)
            elif activity_level == 'inactive':
                users = users.filter(
                    Q(profile__last_login__lt=cutoff_date) | Q(profile__last_login__isnull=True)
                )
        
        # Pagination
        paginator = Paginator(users, 25)
        page_number = request.GET.get('page')
        page_users = paginator.get_page(page_number)
        
        # Get notifications
        zipped_data, total_new_messages, free_time = new_messages(request)
        total_notifications = len(free_time)
        
        # Convert zipped_data to list to avoid iteration issues
        message = list(zipped_data) if zipped_data else []
        
        context = {
            'users': page_users,
            'saved_segments': saved_segments,
            'companies': companies,
            'current_filters': {
                'subscription_status': subscription_status,
                'activity_level': activity_level,
                'lacto_points_min': lacto_points_min,
                'lacto_points_max': lacto_points_max,
                'churn_risk': churn_risk,
                'company': company_filter,
                'health_condition': health_condition,
                'last_active_days': last_active_days,
            },
            'message': message,
            'total_new_messages': total_new_messages,
            'free_time': free_time,
            'total_notifications': total_notifications,
        }
        
        return render(request, 'manager/user_segmentation.html', context)
        
    except Exception as e:
        print(f"User segmentation error: {e}")
        return generate_error_page(request, f"Error loading user segmentation: {str(e)}")


@api_view(["GET", "POST"])
@manager_required
def notification_conditions(request):
    """Notification conditions page with filtering and sending capabilities"""
    language = request.session.get(settings.LANGUAGE_COOKIE_NAME, 'en')
    translation.activate(language)
    
    try:
        if request.method == "POST":
            # Handle notification sending
            notification_title = request.POST.get('notification_title', '')
            notification_body = request.POST.get('notification_body', '')
            
            if not notification_title or not notification_body:
                messages.error(request, "Both title and body are required.")
                return redirect('panel-manager-notifications')
            
            # Get filter parameters to determine target users
            search_query = request.GET.get('search', '')
            subscription_filter = request.GET.get('subscription', '')
            activity_filter = request.GET.get('activity', '')
            company_filter = request.GET.get('company', '')
            churn_risk_filter = request.GET.get('churn_risk', '')
            joined_date_filter = request.GET.get('joined_date', '')
            
            # Apply the same filtering logic as user_management
            users = AppUser.objects.select_related('profile').all()
            
            # Apply filters (same as user_management function)
            if search_query:
                users = users.filter(
                    Q(profile__first_name__icontains=search_query) |
                    Q(profile__last_name__icontains=search_query) |
                    Q(phone_number__icontains=search_query) |
                    Q(profile__email__icontains=search_query)
                )
            
            if company_filter:
                users = users.filter(django_user__groups__id=company_filter)
            
            if subscription_filter == 'active':
                active_subscription_users = TherapistSelection.objects.filter(
                    credit__gt=datetime.now()
                ).values_list('user_id', flat=True)
                users = users.filter(django_user_id__in=active_subscription_users)
            elif subscription_filter == 'expired':
                expired_subscription_users = TherapistSelection.objects.filter(
                    credit__lte=datetime.now()
                ).values_list('user_id', flat=True)
                users = users.filter(django_user_id__in=expired_subscription_users)
            elif subscription_filter == 'free':
                subscription_users = TherapistSelection.objects.values_list('user_id', flat=True)
                users = users.exclude(django_user_id__in=subscription_users)
            
            if activity_filter == 'active':
                users = users.filter(profile__last_login__gte=datetime.now() - timedelta(days=30))
            elif activity_filter == 'inactive':
                users = users.filter(
                    Q(profile__last_login__lt=datetime.now() - timedelta(days=30)) |
                    Q(profile__last_login__isnull=True)
                )
            
            if joined_date_filter:
                if joined_date_filter == 'today':
                    today = date.today()
                    users = users.filter(profile__register_time__date=today)
                elif joined_date_filter == 'this_week':
                    week_ago = date.today() - timedelta(days=7)
                    users = users.filter(profile__register_time__date__gte=week_ago)
                elif joined_date_filter == 'this_month':
                    month_ago = date.today() - timedelta(days=30)
                    users = users.filter(profile__register_time__date__gte=month_ago)
                elif joined_date_filter == 'last_month':
                    two_months_ago = date.today() - timedelta(days=60)
                    month_ago = date.today() - timedelta(days=30)
                    users = users.filter(
                        profile__register_time__date__gte=two_months_ago,
                        profile__register_time__date__lt=month_ago
                    )
            
            if churn_risk_filter:
                # Filter based on actual churn risk calculation
                filtered_user_ids = []
                for user in users:
                    user_churn_risk = user.churn_risk_score
                    if churn_risk_filter == 'high' and user_churn_risk == 'High':
                        filtered_user_ids.append(user.id)
                    elif churn_risk_filter == 'medium' and user_churn_risk == 'Medium':
                        filtered_user_ids.append(user.id)
                    elif churn_risk_filter == 'low' and user_churn_risk == 'Low':
                        filtered_user_ids.append(user.id)
                
                users = users.filter(id__in=filtered_user_ids)
            
            # Send notifications to filtered users
            push_notification = PushNotification()
            success_count = 0
            failed_count = 0
            
            for user in users:
                try:
                    if push_notification.send_notification_to_user(user, notification_title, notification_body):
                        success_count += 1
                    else:
                        failed_count += 1
                except Exception as e:
                    print(f"Failed to send notification to {user.identifier}: {e}")
                    failed_count += 1
            
            messages.success(request, f"Notification sent successfully to {success_count} users. {failed_count} failed.")
            return redirect('panel-manager-notifications')
        
        # GET request - show the form
        # Get filter parameters for preview
        search_query = request.GET.get('search', '')
        subscription_filter = request.GET.get('subscription', '')
        activity_filter = request.GET.get('activity', '')
        company_filter = request.GET.get('company', '')
        churn_risk_filter = request.GET.get('churn_risk', '')
        joined_date_filter = request.GET.get('joined_date', '')
        
        # Base queryset for preview
        users = AppUser.objects.select_related('profile').all()
        
        # Apply same filters to show preview count
        if search_query:
            users = users.filter(
                Q(profile__first_name__icontains=search_query) |
                Q(profile__last_name__icontains=search_query) |
                Q(phone_number__icontains=search_query) |
                Q(profile__email__icontains=search_query)
            )
        
        if company_filter:
            users = users.filter(django_user__groups__id=company_filter)
        
        if subscription_filter == 'active':
            active_subscription_users = TherapistSelection.objects.filter(
                credit__gt=datetime.now()
            ).values_list('user_id', flat=True)
            users = users.filter(django_user_id__in=active_subscription_users)
        elif subscription_filter == 'expired':
            expired_subscription_users = TherapistSelection.objects.filter(
                credit__lte=datetime.now()
            ).values_list('user_id', flat=True)
            users = users.filter(django_user_id__in=expired_subscription_users)
        elif subscription_filter == 'free':
            subscription_users = TherapistSelection.objects.values_list('user_id', flat=True)
            users = users.exclude(django_user_id__in=subscription_users)
        
        if activity_filter == 'active':
            users = users.filter(profile__last_login__gte=datetime.now() - timedelta(days=30))
        elif activity_filter == 'inactive':
            users = users.filter(
                Q(profile__last_login__lt=datetime.now() - timedelta(days=30)) |
                Q(profile__last_login__isnull=True)
            )
        
        if joined_date_filter:
            if joined_date_filter == 'today':
                today = date.today()
                users = users.filter(profile__register_time__date=today)
            elif joined_date_filter == 'this_week':
                week_ago = date.today() - timedelta(days=7)
                users = users.filter(profile__register_time__date__gte=week_ago)
            elif joined_date_filter == 'this_month':
                month_ago = date.today() - timedelta(days=30)
                users = users.filter(profile__register_time__date__gte=month_ago)
            elif joined_date_filter == 'last_month':
                two_months_ago = date.today() - timedelta(days=60)
                month_ago = date.today() - timedelta(days=30)
                users = users.filter(
                    profile__register_time__date__gte=two_months_ago,
                    profile__register_time__date__lt=month_ago
                )
        
        if churn_risk_filter:
            # Filter based on actual churn risk calculation
            filtered_user_ids = []
            for user in users:
                user_churn_risk = user.churn_risk_score
                if churn_risk_filter == 'high' and user_churn_risk == 'High':
                    filtered_user_ids.append(user.id)
                elif churn_risk_filter == 'medium' and user_churn_risk == 'Medium':
                    filtered_user_ids.append(user.id)
                elif churn_risk_filter == 'low' and user_churn_risk == 'Low':
                    filtered_user_ids.append(user.id)
            
            users = users.filter(id__in=filtered_user_ids)
        
        # Get available ZiluckGroups for filter dropdown (companies only)
        companies = ZiLuckGroup.objects.filter(is_company=True).order_by('name')
        
        # Get notifications
        zipped_data, total_new_messages, free_time = new_messages(request)
        total_notifications = len(free_time)
        
        # Convert zipped_data to list to avoid iteration issues
        message = list(zipped_data) if zipped_data else []
        
        context = {
            'target_users_count': users.count(),
            'companies': companies,
            'search_query': search_query,
            'subscription_filter': subscription_filter,
            'activity_filter': activity_filter,
            'company_filter': company_filter,
            'churn_risk_filter': churn_risk_filter,
            'joined_date_filter': joined_date_filter,
            'message': message,
            'total_new_messages': total_new_messages,
            'free_time': free_time,
            'total_notifications': total_notifications,
        }
        
        return render(request, 'manager/notification_conditions.html', context)
        
    except Exception as e:
        print(f"Notification conditions error: {e}")
        return generate_error_page(request, f"Error loading notification conditions: {str(e)}")


# API endpoints for AJAX functionality

def create_hr_account_for_company(company):
    """
    Create an HR user account for a company.
    Returns: (django_user, password) or (None, None) if account already exists
    """
    from django.contrib.auth.models import User as DjangoUser, Group
    import secrets
    import string
    
    # Check if HR account already exists for this company
    # HR username format: hr_{company_id} or hr_{company.group_id}
    hr_username = f"hr_{company.group_id or company.id}"
    
    # Check if user already exists
    if DjangoUser.objects.filter(username=hr_username).exists():
        existing_user = DjangoUser.objects.get(username=hr_username)
        return existing_user, None  # Account exists, return user but no password
    
    # Generate a secure random password
    password_length = 12
    alphabet = string.ascii_letters + string.digits + "!@#$%^&*"
    password = ''.join(secrets.choice(alphabet) for i in range(password_length))
    
    # Create Django user
    hr_user = DjangoUser.objects.create_user(
        username=hr_username,
        password=password,
        email=f"{hr_username}@ziluck.com",
        is_staff=True,  # Allow access to admin if needed
    )
    
    # Add to HR group (separate from Manager, Doctor, Assistant, etc.)
    hr_group, _ = Group.objects.get_or_create(name='HR')
    hr_user.groups.add(hr_group)
    
    # Add HR user to the company group (so they can see their company's employees)
    company.user_set.add(hr_user)
    
    # Set HR user as admin of the company (optional, but useful)
    company.admin = hr_user
    company.save()
    
    return hr_user, password


@api_view(["POST"])
@manager_required
def create_company_ajax(request):
    """AJAX endpoint to create a new ZiluckGroup company"""
    try:
        # Use request.data if available (DRF), otherwise parse body
        if hasattr(request, 'data'):
            data = request.data
        else:
            # Read body once
            body = request.body.decode('utf-8') if isinstance(request.body, bytes) else request.body
            data = json.loads(body)
        
        # Create a ZiluckGroup for the company
        company = ZiLuckGroup.objects.create(
            name=data['name'],
            group_id=data.get('company_code', f"company_{data['name'].lower().replace(' ', '_')}"),
            is_company=True,
            is_public=False,
            creator=request.user,
            admin=request.user
        )
        
        # Automatically create HR account for the new company
        hr_user, hr_password = create_hr_account_for_company(company)
        
        response_data = {
            'success': True,
            'message': 'Company created successfully',
            'company_id': company.id
        }
        
        if hr_user and hr_password:
            response_data['hr_account'] = {
                'username': hr_user.username,
                'password': hr_password,
                'message': 'HR account created successfully'
            }
        elif hr_user:
            response_data['hr_account'] = {
                'username': hr_user.username,
                'message': 'HR account already exists'
            }
        
        return JsonResponse(response_data)
        
    except Exception as e:
        return JsonResponse({
            'success': False,
            'message': str(e)
        })


@api_view(["POST"])
@manager_required
def create_hr_account_ajax(request):
    """AJAX endpoint to create HR account for an existing company"""
    try:
        # Use request.data if available (DRF), otherwise parse body
        if hasattr(request, 'data'):
            company_id = request.data.get('company_id')
        else:
            # Read body once
            body = request.body.decode('utf-8') if isinstance(request.body, bytes) else request.body
            data = json.loads(body)
            company_id = data.get('company_id')
        
        if not company_id:
            return JsonResponse({
                'success': False,
                'message': 'Company ID is required'
            })
        
        try:
            company = ZiLuckGroup.objects.get(id=company_id, is_company=True)
        except ZiLuckGroup.DoesNotExist:
            return JsonResponse({
                'success': False,
                'message': 'Company not found'
            })
        
        # Create HR account
        hr_user, hr_password = create_hr_account_for_company(company)
        
        if hr_user and hr_password:
            return JsonResponse({
                'success': True,
                'message': 'HR account created successfully',
                'hr_account': {
                    'username': hr_user.username,
                    'password': hr_password
                }
            })
        elif hr_user:
            return JsonResponse({
                'success': False,
                'message': 'HR account already exists for this company',
                'hr_account': {
                    'username': hr_user.username
                }
            })
        else:
            return JsonResponse({
                'success': False,
                'message': 'Failed to create HR account'
            })
        
    except Exception as e:
        return JsonResponse({
            'success': False,
            'message': str(e)
        })


@api_view(["POST"])
@manager_required
def create_lead_ajax(request):
    """AJAX endpoint to create a new lead"""
    try:
        data = json.loads(request.body)
        manager = Manager.objects.get(django_user=request.user) if not request.user.is_superuser else None
        
        lead = Lead.objects.create(
            company_name=data['company_name'],
            contact_name=data['contact_name'],
            contact_email=data['contact_email'],
            contact_phone=data['contact_phone'],
            status=data.get('status', 'prospect'),
            priority=data.get('priority', 'medium'),
            estimated_licenses=data.get('estimated_licenses', 0),
            estimated_value=data.get('estimated_value', 0),
            source=data.get('source', ''),
            industry=data.get('industry', ''),
            notes=data.get('notes', ''),
            next_action=data.get('next_action', ''),
            next_follow_up_date=data.get('next_follow_up_date'),
            created_by=manager,
            assigned_to=manager
        )
        
        return JsonResponse({
            'success': True,
            'message': 'Lead created successfully',
            'lead_id': lead.id
        })
        
    except Exception as e:
        return JsonResponse({
            'success': False,
            'message': str(e)
        })


@api_view(["POST"])
@manager_required
def update_lead_status_ajax(request):
    """AJAX endpoint to update lead status"""
    try:
        data = json.loads(request.body)
        lead = get_object_or_404(Lead, id=data['lead_id'])
        
        lead.status = data['status']
        if 'notes' in data:
            lead.notes = data['notes']
        lead.save()
        
        # If status is closed_won, offer to convert to company
        converted_company = None
        if lead.status == 'closed_won' and not lead.converted_company:
            converted_company = lead.convert_to_company()
        
        return JsonResponse({
            'success': True,
            'message': 'Lead status updated successfully',
            'converted_company_id': converted_company.id if converted_company else None
        })
        
    except Exception as e:
        return JsonResponse({
            'success': False,
            'message': str(e)
        })


@api_view(["GET"])
@manager_required
def acquisition_analytics(request):
    """Analytics dashboard for user acquisition sources"""
    language = request.session.get(settings.LANGUAGE_COOKIE_NAME, 'en')
    translation.activate(language)
    
    try:
        # Get manager instance
        manager = Manager.objects.get(django_user=request.user) if not request.user.is_superuser else None
        
        # Get time range filter (default: all time)
        time_range = request.GET.get('time_range', 'all')
        
        # Calculate date filter
        date_filter = Q()
        if time_range == 'today':
            date_filter = Q(acquisition_date__date=date.today())
        elif time_range == 'week':
            date_filter = Q(acquisition_date__gte=datetime.now() - timedelta(days=7))
        elif time_range == 'month':
            date_filter = Q(acquisition_date__gte=datetime.now() - timedelta(days=30))
        elif time_range == '3months':
            date_filter = Q(acquisition_date__gte=datetime.now() - timedelta(days=90))
        
        # Overall acquisition statistics
        total_tracked = Profile.objects.filter(acquisition_source__isnull=False).filter(date_filter).count()
        total_users = Profile.objects.filter(register_time__isnull=False).count()
        tracking_rate = (total_tracked / total_users * 100) if total_users > 0 else 0
        
        # Acquisition source breakdown
        acquisition_breakdown = Profile.objects.filter(
            acquisition_source__isnull=False
        ).filter(date_filter).values('acquisition_source').annotate(
            count=Count('id')
        ).order_by('-count')
        
        # Get Persian names for sources
        source_choices_dict = dict(Profile.ACQUISITION_CHOICES)
        acquisition_data = []
        total_count = sum([item['count'] for item in acquisition_breakdown])
        
        for item in acquisition_breakdown:
            source_code = item['acquisition_source']
            count = item['count']
            percentage = (count / total_count * 100) if total_count > 0 else 0
            
            acquisition_data.append({
                'code': source_code,
                'name': source_choices_dict.get(source_code, source_code),
                'count': count,
                'percentage': round(percentage, 2)
            })
        
        # "Other" source details - get the actual text users entered
        other_sources = Profile.objects.filter(
            acquisition_source='other',
            acquisition_details__isnull=False
        ).filter(date_filter).exclude(
            acquisition_details=''
        ).values('acquisition_details').annotate(
            count=Count('id')
        ).order_by('-count')[:20]
        
        # Time-based trend analysis (last 30 days)
        trend_data = []
        if time_range in ['all', 'month', '3months']:
            for i in range(30):
                day = date.today() - timedelta(days=29-i)
                day_count = Profile.objects.filter(
                    acquisition_source__isnull=False,
                    acquisition_date__date=day
                ).count()
                trend_data.append({
                    'date': day.strftime('%Y-%m-%d'),
                    'count': day_count
                })
        
        # Top sources by active users
        thirty_days_ago = datetime.now() - timedelta(days=30)
        source_engagement = []
        
        for source_item in acquisition_breakdown:
            source = source_item['acquisition_source']
            total_from_source = source_item['count']
            
            active_from_source = Profile.objects.filter(
                acquisition_source=source,
                last_login__gte=thirty_days_ago
            ).filter(date_filter).count()
            
            engagement_rate = (active_from_source / total_from_source * 100) if total_from_source > 0 else 0
            
            source_engagement.append({
                'source': source_choices_dict.get(source, source),
                'total_users': total_from_source,
                'active_users': active_from_source,
                'engagement_rate': round(engagement_rate, 2)
            })
        
        source_engagement.sort(key=lambda x: x['engagement_rate'], reverse=True)
        
        # Revenue by acquisition source
        source_revenue = []
        for source_item in acquisition_breakdown:
            source = source_item['acquisition_source']
            user_profiles = Profile.objects.filter(acquisition_source=source).filter(date_filter)
            user_ids = [p.user_set.first().id for p in user_profiles if p.user_set.exists()]
            
            revenue = ZarinPalPayment.objects.filter(
                user_id__in=user_ids,
                is_payed=True
            ).aggregate(total=Sum('amount_after_discount'))['total'] or 0
            
            paying_users = ZarinPalPayment.objects.filter(
                user_id__in=user_ids,
                is_payed=True
            ).values('user').distinct().count()
            
            conversion_rate = (paying_users / len(user_ids) * 100) if len(user_ids) > 0 else 0
            
            source_revenue.append({
                'source': source_choices_dict.get(source, source),
                'revenue': revenue,
                'paying_users': paying_users,
                'total_users': len(user_ids),
                'conversion_rate': round(conversion_rate, 2)
            })
        
        source_revenue.sort(key=lambda x: x['revenue'], reverse=True)
        
        untracked_users = Profile.objects.filter(
            Q(acquisition_source__isnull=True) | Q(acquisition_source='')
        ).count()
        
        # Recent acquisitions
        recent_acquisitions = Profile.objects.filter(
            acquisition_source__isnull=False,
            acquisition_date__isnull=False
        ).order_by('-acquisition_date')[:10]
        
        recent_list = []
        for profile in recent_acquisitions:
            user = profile.user_set.first()
            if user:
                recent_list.append({
                    'user_id': user.id,
                    'name': f"{profile.first_name} {profile.last_name}",
                    'phone': user.phone_number,
                    'source': source_choices_dict.get(profile.acquisition_source, profile.acquisition_source),
                    'details': profile.acquisition_details or '-',
                    'date': profile.acquisition_date
                })
        
        zipped_data, total_new_messages, free_time = new_messages(request)
        total_notifications = len(free_time)
        
        context = {
            'manager': manager,
            'total_tracked': total_tracked,
            'total_users': total_users,
            'tracking_rate': round(tracking_rate, 2),
            'untracked_users': untracked_users,
            'acquisition_data': acquisition_data,
            'other_sources': other_sources,
            'trend_data': trend_data,
            'source_engagement': source_engagement,
            'source_revenue': source_revenue,
            'recent_acquisitions': recent_list,
            'time_range': time_range,
            'message': zipped_data,
            'total_new_messages': total_new_messages,
            'free_time': free_time,
            'total_notifications': total_notifications,
        }
        
        return render(request, 'manager/acquisition_analytics.html', context)
        
    except Exception as e:
        print(f"Acquisition analytics error: {e}")
        import traceback
        traceback.print_exc()
        return generate_error_page(request, f"Error loading acquisition analytics: {str(e)}")
