import hashlib

from django.shortcuts import render, redirect, get_object_or_404
from django.views.decorators.csrf import csrf_exempt
from rest_framework.decorators import api_view
from django.contrib.auth.decorators import login_required
from django.contrib.auth.models import Group
from django.contrib.auth.models import User as DjangoUser
from django.contrib.auth import logout, authenticate
from django.contrib.auth import login as django_login
from django.http import HttpResponse, JsonResponse
from datetime import timedelta, datetime, time, date
from django.contrib import messages
from django.core.cache import cache

from APIs.views.Errors import Errors
from APIs.views.firebase import send_notification_note
from panel.models import *
from payment.models import ZarinPalPayment
from ziluck_project.settings import STATIC_ROOT
import os
from APIs.models import User as AppUser, Sleep, Walking, UserQuestionnaire, Profile, Eating, User, ZiLuckGroup, Points, \
    DietTemplate, FoodCard, RequestedFoods, Activities_log, Water, Activity
from APIs.models import Drug, SugarMeasurement, Weights, Diet
from APIs.models import InsulinTypeSelection, Category, Resource
from django.db.models import Q
import jdatetime
from .utility import *
from panel.SMSNotification import *
from panel.PushNotification import PushNotification
from django.utils.safestring import mark_safe
from django.utils import translation
from django.conf import settings
from django.urls import reverse
from rest_framework.response import Response
from django.core.paginator import Paginator, EmptyPage, PageNotAnInteger
from django.template.loader import render_to_string
from APIs.forms import FoodForm, ActivityForm
import traceback
from collections import defaultdict
from django.forms.models import model_to_dict


def generate_error_page(request, error_message):
    # Get notifications for base template
    try:
        zipped_data, total_new_messages, free_time = new_messages(request)
        message = list(zipped_data) if zipped_data else []
        total_notifications = len(free_time)
    except:
        message = []
        total_new_messages = 0
        free_time = []
        total_notifications = 0
    
    context = {
        'message': message,  # For base template notifications
        'error_message': error_message,  # The actual error message
        'total_new_messages': total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }
    return render(request=request, template_name='manager/error.html', context=context)


def new_messages(request):
    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)
    user = request.user
    patients_list = AppUser.objects.filter(profile__therapist=user)
    managers = Manager.objects.filter(django_user=user)

    today = date.today()
    t1 = datetime(year=today.year, month=today.month, day=today.day, hour=0, minute=0, second=0)
    t2 = datetime(year=today.year, month=today.month, day=today.day, hour=23, minute=59, second=59)
    t3 = t1 + timedelta(hours=47, minutes=59, seconds=59)
    free_time = TimeSlot.objects.filter(therapist_django_user=user, start_time__range=(t1, t3), is_available=0)

    messages = []

    qset = AppUser

    for manager in managers:
        qset = AppUser.objects.filter(profile__therapist=manager.django_user)
        patients_list = patients_list.union(qset)
    patients_list = list(patients_list)
    number_of_new_messages = []

    # print(f"This is the patient list element one: {patients_list[2]}")

    for p in patients_list:
        temp = Message.objects.filter((Q(sender=p.django_user) & Q(has_been_seen=False)))
        number_of_new_messages.append(len(temp))
        for i in range(len(temp)):
            sender_name = User.objects.filter(django_user=temp[i].sender)[0].profile.first_name
            sender_id = User.objects.filter(django_user=temp[i].sender)[0].id

            # print(f"this is the message sender id 1: {temp[i].sender}")

            messages.append(
                {
                    'text': temp[i].text,
                    'sender': temp[i].sender,
                    'time': temp[i].time,
                    "sender_name": sender_name,
                    "sender_id": sender_id
                }
            )
    max_length = max(len(messages), len(number_of_new_messages))
    patients_list += [None] * (max_length - len(patients_list))
    number_of_new_messages += [None] * (max_length - len(number_of_new_messages))
    zipped_data = zip(patients_list, messages, number_of_new_messages)
    total_new_messages = len(messages)

    return zipped_data, total_new_messages, free_time


@api_view(["GET"])
def patients(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)

    patients_list = AppUser.objects.filter(
        profile__therapist__isnull=True,
    )

    # For example, to show those registered in the last 7 days:
    # from datetime import timedelta, datetime
    # seven_days_ago = datetime.now() - timedelta(days=7)
    # patients_list = patients_list.filter(created_at__gte=seven_days_ago)

    number_of_new_messages = []
    total_of_messages = []
    for patient in patients_list:
        # Count new (unseen) messages for the patient
        new_msgs = Message.objects.filter(
            Q(sender=patient.django_user) & Q(has_been_seen=False)
        ).count()
        number_of_new_messages.append(new_msgs)

        # Count all messages for the patient
        total_msgs = Message.objects.filter(sender=patient.django_user).count()
        total_of_messages.append(total_msgs)

    # Get any extra data for notifications (if needed)
    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        # We're zipping patients with their message counts; adjust your template accordingly.
        'patients_list': zip(patients_list, number_of_new_messages, total_of_messages),
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }

    return render(request, template_name='manager/patients.html', context=context)


# Category Management Views
@api_view(["GET"])
def categories(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)

    # Get all categories with their subcategories
    categories_list = Category.objects.filter(parent=None, is_active=True).order_by('ordering', 'name')
    
    # Get any extra data for notifications (if needed)
    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'categories_list': categories_list,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }

    return render(request, template_name='manager/categories.html', context=context)


@api_view(["GET", "POST"])
def create_category(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)

    if request.method == "POST":
        try:
            name = request.POST.get("name")
            slug = request.POST.get("slug")
            description = request.POST.get("description", "")
            parent_id = request.POST.get("parent_id")
            ordering = int(request.POST.get("ordering", 0))

            # Validate slug format
            if not slug or not slug.strip():
                messages.error(request, "نامک (Slug) الزامی است")
                return redirect('panel-manager-create-category')
            
            # Check if slug is unique
            if Category.objects.filter(slug=slug).exists():
                messages.error(request, "این نامک قبلاً استفاده شده است. لطفاً نامک دیگری انتخاب کنید")
                return redirect('panel-manager-create-category')

            # Create category
            category_data = {
                "name": name,
                "slug": slug,
                "description": description,
                "ordering": ordering,
                "creator": request.user
            }

            if parent_id:
                try:
                    parent = Category.objects.get(id=parent_id)
                    category_data["parent"] = parent
                except Category.DoesNotExist:
                    messages.error(request, "دسته‌بندی والد یافت نشد")
                    return redirect('panel-manager-create-category')

            category = Category.objects.create(**category_data)
            messages.success(request, "دسته‌بندی با موفقیت ایجاد شد")
            return redirect('panel-manager-categories')

        except Exception as e:
            messages.error(request, f"خطا در ایجاد دسته‌بندی: {str(e)}")
            return redirect('panel-manager-create-category')

    # Get parent categories for dropdown
    parent_categories = Category.objects.filter(parent=None, is_active=True).order_by('name')
    
    # Get any extra data for notifications (if needed)
    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'parent_categories': parent_categories,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }

    return render(request, template_name='manager/create_category.html', context=context)


@api_view(["GET", "POST"])
def edit_category(request, category_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)

    category = get_object_or_404(Category, id=category_id)

    if request.method == "POST":
        try:
            name = request.POST.get("name")
            slug = request.POST.get("slug")
            description = request.POST.get("description", "")
            parent_id = request.POST.get("parent_id")
            ordering = int(request.POST.get("ordering", 0))
            is_active = request.POST.get("is_active") == "on"

            # Validate slug format
            if not slug or not slug.strip():
                messages.error(request, "نامک (Slug) الزامی است")
                return redirect('panel-manager-edit-category', category_id=category_id)
            
            # Check if slug is unique (excluding current category)
            if Category.objects.filter(slug=slug).exclude(id=category_id).exists():
                messages.error(request, "این نامک قبلاً استفاده شده است. لطفاً نامک دیگری انتخاب کنید")
                return redirect('panel-manager-edit-category', category_id=category_id)

            # Update category
            category.name = name
            category.slug = slug
            category.description = description
            category.ordering = ordering
            category.is_active = is_active

            if parent_id:
                try:
                    parent = Category.objects.get(id=parent_id)
                    category.parent = parent
                except Category.DoesNotExist:
                    messages.error(request, "دسته‌بندی والد یافت نشد")
                    return redirect('panel-manager-edit-category', category_id=category_id)
            else:
                category.parent = None

            category.save()
            messages.success(request, "دسته‌بندی با موفقیت ویرایش شد")
            return redirect('panel-manager-categories')

        except Exception as e:
            messages.error(request, f"خطا در ویرایش دسته‌بندی: {str(e)}")
            return redirect('panel-manager-edit-category', category_id=category_id)

    # Get parent categories for dropdown (exclude current category and its descendants)
    parent_categories = Category.objects.filter(parent=None, is_active=True).exclude(id=category_id).order_by('name')
    
    # Get any extra data for notifications (if needed)
    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'category': category,
        'parent_categories': parent_categories,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }

    return render(request, template_name='manager/edit_category.html', context=context)


@api_view(["POST"])
def delete_category(request, category_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    try:
        category = get_object_or_404(Category, id=category_id)
        category.delete()
        messages.success(request, "دسته‌بندی با موفقیت حذف شد")
    except Exception as e:
        messages.error(request, f"خطا در حذف دسته‌بندی: {str(e)}")

    return redirect('panel-manager-categories')


# Resource Management Views
@api_view(["GET"])
def resources(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)

    # Get all resources with their categories
    resources_list = Resource.objects.all().order_by('category__name', 'ordering', 'title')
    
    # Get any extra data for notifications (if needed)
    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'resources_list': resources_list,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }

    return render(request, template_name='manager/resources.html', context=context)


@api_view(["GET", "POST"])
def create_resource(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)

    if request.method == "POST":
        try:
            title = request.POST.get("title")
            description = request.POST.get("description", "")
            category_id = request.POST.get("category_id")
            resource_type = request.POST.get("type")
            ordering = int(request.POST.get("ordering", 0))
            is_published = request.POST.get("is_published") == "on"
            is_featured = request.POST.get("is_featured") == "on"

            # Check if category exists
            try:
                category = Category.objects.get(id=category_id)
            except Category.DoesNotExist:
                messages.error(request, "دسته‌بندی یافت نشد")
                return redirect('panel-manager-create-resource')

            # Create resource
            resource = Resource.objects.create(
                title=title,
                description=description,
                category=category,
                type=resource_type,
                ordering=ordering,
                is_published=is_published,
                is_featured=is_featured,
                creator=request.user
            )

            # Handle file uploads based on resource type
            if resource_type == 'video':
                video_url = request.POST.get("video_url")
                duration_minutes = int(request.POST.get("duration_minutes", 0))
                thumbnail = request.FILES.get("thumbnail")
                
                from APIs.models import VideoResource
                video_resource = VideoResource.objects.create(
                    resource=resource,
                    video_url=video_url,
                    duration_minutes=duration_minutes
                )
                if thumbnail:
                    video_resource.thumbnail = thumbnail
                    video_resource.save()

            elif resource_type == 'pdf':
                pdf_file = request.FILES.get("pdf_file")
                if pdf_file:
                    from APIs.models import PDFResource
                    pdf_resource = PDFResource.objects.create(
                        resource=resource,
                        file=pdf_file,
                        file_size=pdf_file.size,
                        page_count=1  # You might want to calculate this
                    )

            elif resource_type == 'image_series':
                from APIs.models import ImageSeriesResource
                image_series_resource = ImageSeriesResource.objects.create(resource=resource)
                
                # Handle multiple image uploads
                images = request.FILES.getlist("images")
                for i, image in enumerate(images):
                    caption = request.POST.get(f"caption_{i}", "")
                    from APIs.models import ImageSeriesItem
                    ImageSeriesItem.objects.create(
                        image_series=image_series_resource,
                        image=image,
                        caption=caption,
                        ordering=i
                    )

            messages.success(request, "محتوا با موفقیت ایجاد شد")
            return redirect('panel-manager-resources')

        except Exception as e:
            messages.error(request, f"خطا در ایجاد محتوا: {str(e)}")
            return redirect('panel-manager-create-resource')

    # Get categories for dropdown
    categories_list = Category.objects.filter(is_active=True).order_by('name')
    
    # Get any extra data for notifications (if needed)
    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'categories_list': categories_list,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }

    return render(request, template_name='manager/create_resource.html', context=context)


@api_view(["GET", "POST"])
def edit_resource(request, resource_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)

    resource = get_object_or_404(Resource, id=resource_id)

    if request.method == "POST":
        try:
            title = request.POST.get("title")
            description = request.POST.get("description", "")
            category_id = request.POST.get("category_id")
            ordering = int(request.POST.get("ordering", 0))
            is_published = request.POST.get("is_published") == "on"
            is_featured = request.POST.get("is_featured") == "on"

            # Check if category exists
            try:
                category = Category.objects.get(id=category_id)
            except Category.DoesNotExist:
                messages.error(request, "دسته‌بندی یافت نشد")
                return redirect('panel-manager-edit-resource', resource_id=resource_id)

            # Update resource
            resource.title = title
            resource.description = description
            resource.category = category
            resource.ordering = ordering
            resource.is_published = is_published
            resource.is_featured = is_featured
            resource.save()

            # Handle file updates based on resource type
            if resource.type == 'video':
                video_url = request.POST.get("video_url")
                duration_minutes = int(request.POST.get("duration_minutes", 0))
                thumbnail = request.FILES.get("thumbnail")
                
                from APIs.models import VideoResource
                video_resource, created = VideoResource.objects.get_or_create(resource=resource)
                video_resource.video_url = video_url
                video_resource.duration_minutes = duration_minutes
                if thumbnail:
                    video_resource.thumbnail = thumbnail
                video_resource.save()

            elif resource.type == 'pdf':
                pdf_file = request.FILES.get("pdf_file")
                if pdf_file:
                    from APIs.models import PDFResource
                    pdf_resource, created = PDFResource.objects.get_or_create(resource=resource)
                    pdf_resource.file = pdf_file
                    pdf_resource.file_size = pdf_file.size
                    pdf_resource.save()

            messages.success(request, "محتوا با موفقیت ویرایش شد")
            return redirect('panel-manager-resources')

        except Exception as e:
            messages.error(request, f"خطا در ویرایش محتوا: {str(e)}")
            return redirect('panel-manager-edit-resource', resource_id=resource_id)

    # Get categories for dropdown
    categories_list = Category.objects.filter(is_active=True).order_by('name')
    
    # Get any extra data for notifications (if needed)
    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'resource': resource,
        'categories_list': categories_list,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }

    return render(request, template_name='manager/edit_resource.html', context=context)


@api_view(["POST"])
def delete_resource(request, resource_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    try:
        resource = get_object_or_404(Resource, id=resource_id)
        resource.delete()
        messages.success(request, "محتوا با موفقیت حذف شد")
    except Exception as e:
        messages.error(request, f"خطا در حذف محتوا: {str(e)}")

    return redirect('panel-manager-resources')


# Payment Plans Management Functions
def payment_plans(request):
    """Display all payment plans for managers"""
    if not request.user.is_authenticated:
        return redirect('panel-login')
    
    # Check if user is a manager
    managers = Manager.objects.filter(django_user=request.user)
    if not managers.exists():
        return generate_error_page(request, "Access denied. Manager privileges required.")
    
    try:
        from payment.models import PaymentPlans
        plans = PaymentPlans.objects.all().order_by('price')
        
        context = {
            'plans': plans,
            'user': request.user,
            'is_manager': True
        }
        return render(request, 'manager/payment_plans.html', context)
        
    except Exception as e:
        print(f"Error in payment_plans: {str(e)}")
        return generate_error_page(request, f"Error loading payment plans: {str(e)}")


def create_payment_plan(request):
    """Create a new payment plan"""
    if not request.user.is_authenticated:
        return redirect('panel-login')
    
    # Check if user is a manager
    managers = Manager.objects.filter(django_user=request.user)
    if not managers.exists():
        return generate_error_page(request, "Access denied. Manager privileges required.")
    
    if request.method == 'POST':
        try:
            from payment.models import PaymentPlans
            
            name = request.POST.get('name')
            plan_database_id = request.POST.get('plan_database_id')
            price = request.POST.get('price')
            items_included = request.POST.get('items_included', '')
            items_excluded = request.POST.get('items_excluded', '')
            promotion = request.POST.get('promotion', '')
            price_with_promotion = request.POST.get('price_with_promotion', '')
            subscription_period = request.POST.get('subscription_period')
            
            if not all([name, price, subscription_period]):
                messages.error(request, "Name, price, and subscription period are required.")
                return redirect('panel-manager-payment-plans')
            
            # Create the plan
            plan = PaymentPlans.objects.create(
                name=name,
                plan_database_id=plan_database_id,
                price=price,
                items_included=items_included,
                items_excluded=items_excluded,
                promotion=promotion if promotion else None,
                price_with_promotion=price_with_promotion if price_with_promotion else None,
                subscription_period=subscription_period
            )
            
            messages.success(request, f"Payment plan '{plan.name}' created successfully!")
            return redirect('panel-manager-payment-plans')
            
        except Exception as e:
            print(f"Error creating payment plan: {str(e)}")
            messages.error(request, f"Error creating payment plan: {str(e)}")
            return redirect('panel-manager-payment-plans')
    
    # GET request - show create form
    context = {
        'user': request.user,
        'is_manager': True,
        'action': 'create'
    }
    return render(request, 'manager/payment_plan_form.html', context)


def edit_payment_plan(request, plan_id):
    """Edit an existing payment plan"""
    if not request.user.is_authenticated:
        return redirect('panel-login')
    
    # Check if user is a manager
    managers = Manager.objects.filter(django_user=request.user)
    if not managers.exists():
        return generate_error_page(request, "Access denied. Manager privileges required.")
    
    try:
        from payment.models import PaymentPlans
        plan = get_object_or_404(PaymentPlans, id=plan_id)
        
        if request.method == 'POST':
            name = request.POST.get('name')
            plan_database_id = request.POST.get('plan_database_id')
            price = request.POST.get('price')
            items_included = request.POST.get('items_included', '')
            items_excluded = request.POST.get('items_excluded', '')
            promotion = request.POST.get('promotion', '')
            price_with_promotion = request.POST.get('price_with_promotion', '')
            subscription_period = request.POST.get('subscription_period')
            is_active = request.POST.get('is_active') == 'on'
            
            if not all([name, price, subscription_period]):
                messages.error(request, "Name, price, and subscription period are required.")
                return redirect('panel-manager-edit-payment-plan', plan_id=plan_id)
            
            # Update the plan
            plan.name = name
            plan.plan_database_id = plan_database_id
            plan.price = price
            plan.items_included = items_included
            plan.items_excluded = items_excluded
            plan.promotion = promotion if promotion else None
            plan.price_with_promotion = price_with_promotion if price_with_promotion else None
            plan.subscription_period = subscription_period
            plan.is_active = is_active
            plan.save()
            
            messages.success(request, f"Payment plan '{plan.name}' updated successfully!")
            return redirect('panel-manager-payment-plans')
        
        # GET request - show edit form
        context = {
            'plan': plan,
            'user': request.user,
            'is_manager': True,
            'action': 'edit'
        }
        return render(request, 'manager/payment_plan_form.html', context)
        
    except Exception as e:
        print(f"Error editing payment plan: {str(e)}")
        messages.error(request, f"Error editing payment plan: {str(e)}")
        return redirect('panel-manager-payment-plans')


def delete_payment_plan(request, plan_id):
    """Delete a payment plan"""
    if not request.user.is_authenticated:
        return redirect('panel-login')
    
    # Check if user is a manager
    managers = Manager.objects.filter(django_user=request.user)
    if not managers.exists():
        return generate_error_page(request, "Access denied. Manager privileges required.")
    
    try:
        from payment.models import PaymentPlans
        plan = get_object_or_404(PaymentPlans, id=plan_id)
        plan_name = plan.name
        
        plan.delete()
        messages.success(request, f"Payment plan '{plan_name}' deleted successfully!")
        
    except Exception as e:
        print(f"Error deleting payment plan: {str(e)}")
        messages.error(request, f"Error deleting payment plan: {str(e)}")
    
    return redirect('panel-manager-payment-plans')


def toggle_payment_plan(request, plan_id):
    """Toggle the active status of a payment plan"""
    if not request.user.is_authenticated:
        return redirect('panel-login')
    
    # Check if user is a manager
    managers = Manager.objects.filter(django_user=request.user)
    if not managers.exists():
        return generate_error_page(request, "Access denied. Manager privileges required.")
    
    try:
        from payment.models import PaymentPlans
        plan = get_object_or_404(PaymentPlans, id=plan_id)
        
        plan.is_active = not plan.is_active
        plan.save()
        
        status = "activated" if plan.is_active else "deactivated"
        messages.success(request, f"Payment plan '{plan.name}' {status} successfully!")
        
    except Exception as e:
        print(f"Error toggling payment plan: {str(e)}")
        messages.error(request, f"Error toggling payment plan: {str(e)}")
    
    return redirect('panel-manager-payment-plans')


# Hospital Management Views for Managers

@api_view(["GET"])
def hospitals(request):
    """Display all hospitals for manager"""
    if not request.user.is_authenticated:
        return redirect('panel-login')
    
    managers = Manager.objects.filter(django_user=request.user)
    if not managers.exists() and not request.user.is_superuser:
        return generate_error_page(request, "Access denied. Manager privileges required.")
    
    manager = managers.first() if managers.exists() else None
    
    from panel.models import Hospital
    hospitals_list = Hospital.objects.all().order_by('-created_at')
    
    # Get notification data
    zipped_data, total_new_messages, free_time = new_messages(request)
    message = list(zipped_data) if zipped_data else []
    total_notifications = len(free_time)
    
    context = {
        'user': manager,
        'hospitals': hospitals_list,
        'message': message,
        'total_new_messages': total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }
    
    return render(request, template_name='manager/hospitals.html', context=context)


@api_view(["GET", "POST"])
@csrf_exempt
def create_hospital(request):
    """Create a new hospital"""
    if not request.user.is_authenticated:
        return redirect('panel-login')
    
    managers = Manager.objects.filter(django_user=request.user)
    if not managers.exists() and not request.user.is_superuser:
        return generate_error_page(request, "Access denied. Manager privileges required.")
    
    manager = managers.first() if managers.exists() else None
    
    if request.method == "POST":
        try:
            from panel.models import Hospital
            import secrets
            
            # Get form data
            name = request.POST.get('name')
            hospital_code = request.POST.get('hospital_code')
            contact_email = request.POST.get('contact_email')
            contact_phone = request.POST.get('contact_phone')
            contact_person = request.POST.get('contact_person', '')
            address = request.POST.get('address', '')
            description = request.POST.get('description', '')
            username = request.POST.get('username')
            password = request.POST.get('password')
            
            # Validate required fields
            if not all([name, hospital_code, contact_email, contact_phone, username, password]):
                messages.error(request, "All required fields must be filled!")
                return redirect('panel-manager-create-hospital')
            
            # Check if hospital_code already exists
            if Hospital.objects.filter(hospital_code=hospital_code).exists():
                messages.error(request, "Hospital code already exists!")
                return redirect('panel-manager-create-hospital')
            
            # Create Django user
            if DjangoUser.objects.filter(username=username).exists():
                messages.error(request, "Username already exists!")
                return redirect('panel-manager-create-hospital')
            
            django_user = DjangoUser.objects.create_user(
                username=username,
                password=password,
                email=contact_email
            )
            
            # Add to Hospital group
            hospital_group, created = Group.objects.get_or_create(name='Hospital')
            django_user.groups.add(hospital_group)
            
            # Generate API credentials
            api_key = f"HK_{secrets.token_urlsafe(32)}"
            api_secret = f"HS_{secrets.token_urlsafe(48)}"
            
            # Create hospital
            hospital = Hospital.objects.create(
                name=name,
                hospital_code=hospital_code,
                api_key=api_key,
                api_secret=api_secret,
                contact_email=contact_email,
                contact_phone=contact_phone,
                contact_person=contact_person,
                address=address,
                description=description,
                django_user=django_user,
                created_by=manager,
                is_active=True
            )
            
            # Create promotion group and discount for this hospital
            try:
                ziluck_group, discount = hospital.create_promotion_group_and_discount()
                promotion_code = f"{hospital_code}_PROMOTION"
                messages.success(
                    request, 
                    f"Hospital '{name}' created successfully! "
                    f"Promotion code '{promotion_code}' and group created."
                )
            except Exception as e:
                print(f"Error creating promotion group/discount: {str(e)}")
                messages.warning(
                    request,
                    f"Hospital '{name}' created but there was an error creating promotion group: {str(e)}"
                )
            
            return redirect('panel-manager-hospitals')
            
        except Exception as e:
            print(f"Error creating hospital: {str(e)}")
            messages.error(request, f"Error creating hospital: {str(e)}")
            return redirect('panel-manager-create-hospital')
    
    # GET request - show form
    zipped_data, total_new_messages, free_time = new_messages(request)
    message = list(zipped_data) if zipped_data else []
    total_notifications = len(free_time)
    
    context = {
        'user': manager,
        'message': message,
        'total_new_messages': total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }
    
    return render(request, template_name='manager/create_hospital.html', context=context)


@api_view(["GET", "POST"])
@csrf_exempt
def edit_hospital(request, hospital_id):
    """Edit an existing hospital"""
    if not request.user.is_authenticated:
        return redirect('panel-login')
    
    managers = Manager.objects.filter(django_user=request.user)
    if not managers.exists() and not request.user.is_superuser:
        return generate_error_page(request, "Access denied. Manager privileges required.")
    
    manager = managers.first() if managers.exists() else None
    
    from panel.models import Hospital
    hospital = get_object_or_404(Hospital, id=hospital_id)
    
    if request.method == "POST":
        try:
            hospital.name = request.POST.get('name')
            hospital.contact_email = request.POST.get('contact_email')
            hospital.contact_phone = request.POST.get('contact_phone')
            hospital.contact_person = request.POST.get('contact_person', '')
            hospital.address = request.POST.get('address', '')
            hospital.description = request.POST.get('description', '')
            hospital.is_active = request.POST.get('is_active') == 'on'
            
            hospital.save()
            
            messages.success(request, f"Hospital '{hospital.name}' updated successfully!")
            return redirect('panel-manager-hospitals')
            
        except Exception as e:
            print(f"Error updating hospital: {str(e)}")
            messages.error(request, f"Error updating hospital: {str(e)}")
    
    # GET request - show form
    zipped_data, total_new_messages, free_time = new_messages(request)
    message = list(zipped_data) if zipped_data else []
    total_notifications = len(free_time)
    
    context = {
        'user': manager,
        'hospital': hospital,
        'message': message,
        'total_new_messages': total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }
    
    return render(request, template_name='manager/edit_hospital.html', context=context)


@api_view(["POST"])
@csrf_exempt
def delete_hospital(request, hospital_id):
    """Delete a hospital"""
    if not request.user.is_authenticated:
        return redirect('panel-login')
    
    managers = Manager.objects.filter(django_user=request.user)
    if not managers.exists() and not request.user.is_superuser:
        return generate_error_page(request, "Access denied. Manager privileges required.")
    
    try:
        from panel.models import Hospital
        hospital = get_object_or_404(Hospital, id=hospital_id)
        hospital_name = hospital.name
        
        # Delete associated django user
        if hospital.django_user:
            hospital.django_user.delete()
        
        # Delete hospital
        hospital.delete()
        
        messages.success(request, f"Hospital '{hospital_name}' deleted successfully!")
        
    except Exception as e:
        print(f"Error deleting hospital: {str(e)}")
        messages.error(request, f"Error deleting hospital: {str(e)}")
    
    return redirect('panel-manager-hospitals')


@api_view(["POST"])
@csrf_exempt
def regenerate_hospital_credentials(request, hospital_id):
    """Regenerate API credentials for a hospital"""
    if not request.user.is_authenticated:
        return redirect('panel-login')
    
    managers = Manager.objects.filter(django_user=request.user)
    if not managers.exists() and not request.user.is_superuser:
        return generate_error_page(request, "Access denied. Manager privileges required.")
    
    try:
        from panel.models import Hospital
        hospital = get_object_or_404(Hospital, id=hospital_id)
        
        api_key, api_secret = hospital.regenerate_api_credentials()
        
        messages.success(request, f"API credentials for '{hospital.name}' regenerated successfully!")
        
    except Exception as e:
        print(f"Error regenerating credentials: {str(e)}")
        messages.error(request, f"Error regenerating credentials: {str(e)}")
    
    return redirect('panel-manager-hospitals')
