"""
HR Lab Test Management Views
Allows HR to view employee lab test results with aggregate statistics
"""

from django.shortcuts import render, redirect, get_object_or_404
from django.views.decorators.csrf import csrf_exempt
from rest_framework.decorators import api_view
from django.contrib.auth.decorators import login_required
from django.contrib.auth.models import Group
from django.contrib.auth.models import User as DjangoUser
from django.http import HttpResponse, JsonResponse
from datetime import timedelta, datetime, date
from django.contrib import messages
from django.db.models import Q, Count, Avg, Sum, Case, When, IntegerField, Max, Min
from django.db import models
from django.core.paginator import Paginator, EmptyPage, PageNotAnInteger
from django.utils import translation
from django.conf import settings
from django.db.models.functions import Coalesce
import json

from APIs.models import User as AppUser, Profile, LabTest, LabTestResult, BiomarkerDefinition, ZiLuckGroup, Eating, Food
from panel.models import Manager
from panel.manager import generate_error_page, new_messages


def hr_required(view_func):
    """Decorator to ensure user has HR permissions"""
    def wrapper(request, *args, **kwargs):
        if not request.user.is_authenticated:
            return redirect('panel-login')
        
        # Allow HR group members and superusers
        if not (request.user.groups.filter(name='HR').exists() or request.user.is_superuser):
            return generate_error_page(request, "Access denied. HR permissions required.")
        
        return view_func(request, *args, **kwargs)
    return wrapper


@api_view(["GET"])
@hr_required
def employee_lab_tests_dashboard(request):
    """
    HR Dashboard showing aggregate lab test statistics for all employees in their company.
    Shows compound numbers and health insights.
    """
    language = request.session.get(settings.LANGUAGE_COOKIE_NAME, 'en')
    translation.activate(language)
    
    try:
        # Get company groups that the HR user has access to
        # HR users belong to their company group and are admin of it
        if request.user.is_superuser:
            company_groups = ZiLuckGroup.objects.filter(is_company=True)
        else:
            # Get companies where HR user is a member (via group membership) or admin
            company_groups = ZiLuckGroup.objects.filter(
                Q(is_company=True) & 
                (Q(admin=request.user) | Q(id__in=request.user.groups.values_list('id', flat=True)))
            ).distinct()
        
        # Get selected company from query params
        company_id = request.GET.get('company_id')
        if company_id:
            try:
                selected_company = ZiLuckGroup.objects.get(id=company_id, is_company=True)
            except ZiLuckGroup.DoesNotExist:
                selected_company = None
        else:
            selected_company = company_groups.first() if company_groups.exists() else None
        
        # Get all employees (users) in the selected company
        employees = AppUser.objects.none()
        if selected_company:
            employees = AppUser.objects.filter(
                django_user__groups=selected_company
            ).select_related('profile').prefetch_related('lab_tests')
        
        # Get all lab tests for these employees
        lab_tests = LabTest.objects.none()
        if selected_company:
            lab_tests = LabTest.objects.filter(
                user__django_user__groups=selected_company
            ).select_related('user', 'user__profile').order_by('-test_date')
        
        # Calculate aggregate statistics
        aggregate_stats = calculate_aggregate_statistics(lab_tests)
        
        # Get employee-level statistics
        employee_stats = []
        for employee in employees:
            emp_lab_tests = LabTest.objects.filter(user=employee).order_by('-test_date')
            if emp_lab_tests.exists():
                latest_test = emp_lab_tests.first()
                employee_stats.append({
                    'employee': employee,
                    'latest_test': latest_test,
                    'total_tests': emp_lab_tests.count(),
                    'overall_score': latest_test.overall_health_score,
                    'test_date': latest_test.test_date,
                })
        
        # Sort by overall score (lowest first - most concerning)
        employee_stats.sort(key=lambda x: x['overall_score'] if x['overall_score'] else 999)
        
        # Get notifications
        zipped_data, total_new_messages, free_time = new_messages(request)
        total_notifications = len(free_time)
        message = list(zipped_data) if zipped_data else []
        
        context = {
            'selected_company': selected_company,
            'company_groups': company_groups,
            'employees': employees,
            'employee_stats': employee_stats,
            'aggregate_stats': aggregate_stats,
            'total_employees': employees.count(),
            'total_lab_tests': lab_tests.count(),
            'employees_with_tests': len(employee_stats),
            'message': message,
            'total_new_messages': total_new_messages,
            'free_time': free_time,
            'total_notifications': total_notifications,
        }
        
        return render(request, 'hr/employee_lab_tests_dashboard.html', context)
        
    except Exception as e:
        print(f"[HR] Error in employee_lab_tests_dashboard: {str(e)}")
        import traceback
        traceback.print_exc()
        return generate_error_page(request, f"Error loading HR dashboard: {str(e)}")


def calculate_aggregate_statistics(lab_tests):
    """
    Calculate aggregate statistics from lab tests.
    Returns compound numbers and health insights.
    """
    if not lab_tests.exists():
        return {
            'avg_overall_score': None,
            'health_distribution': {},
            'index_averages': {},
            'risk_summary': {},
            'biomarker_statistics': {},
            'concerning_biomarkers': [],
        }
    
    # Overall health score statistics
    scores = [lt.overall_health_score for lt in lab_tests if lt.overall_health_score is not None]
    avg_overall_score = sum(scores) / len(scores) if scores else None
    
    # Health distribution (Excellent, Good, Fair, Poor)
    health_distribution = {
        'excellent': len([s for s in scores if s >= 90]),
        'good': len([s for s in scores if 70 <= s < 90]),
        'fair': len([s for s in scores if 50 <= s < 70]),
        'poor': len([s for s in scores if s < 50]),
    }
    
    # Index averages
    index_averages = {}
    index_names = ['Cardiometabolic', 'Hematology', 'Organ Function', 'Thyroid', 'Hormonal']
    for index_name in index_names:
        index_scores = []
        for lt in lab_tests:
            if lt.indices and index_name in lt.indices:
                score = lt.indices[index_name]
                if score is not None:
                    index_scores.append(score)
        if index_scores:
            index_averages[index_name] = {
                'average': sum(index_scores) / len(index_scores),
                'count': len(index_scores),
                'min': min(index_scores),
                'max': max(index_scores),
            }
    
    # Risk summary
    risk_summary = {
        'cvd': {'low': 0, 'medium': 0, 'high': 0},
        'diabetes': {'low': 0, 'medium': 0, 'high': 0},
        'liver': {'low': 0, 'medium': 0, 'high': 0},
    }
    
    for lt in lab_tests:
        if lt.risks:
            # CVD Risk
            if 'CVD' in lt.risks or 'cardio' in lt.risks:
                cvd_risk = lt.risks.get('CVD') or lt.risks.get('cardio', {})
                if isinstance(cvd_risk, dict):
                    label = cvd_risk.get('label', '').lower()
                    if 'low' in label or 'کم' in label:
                        risk_summary['cvd']['low'] += 1
                    elif 'medium' in label or 'متوسط' in label:
                        risk_summary['cvd']['medium'] += 1
                    elif 'high' in label or 'بالا' in label:
                        risk_summary['cvd']['high'] += 1
            
            # Diabetes Risk
            if 'Diabetes' in lt.risks or 'diabetes' in lt.risks:
                diabetes_risk = lt.risks.get('Diabetes') or lt.risks.get('diabetes', {})
                if isinstance(diabetes_risk, dict):
                    label = diabetes_risk.get('label', '').lower()
                    if 'low' in label or 'کم' in label:
                        risk_summary['diabetes']['low'] += 1
                    elif 'medium' in label or 'متوسط' in label:
                        risk_summary['diabetes']['medium'] += 1
                    elif 'high' in label or 'بالا' in label:
                        risk_summary['diabetes']['high'] += 1
            
            # Liver Risk
            if 'Liver' in lt.risks or 'liver' in lt.risks:
                liver_risk = lt.risks.get('Liver') or lt.risks.get('liver', {})
                if isinstance(liver_risk, dict):
                    label = liver_risk.get('label', '').lower()
                    if 'low' in label or 'کم' in label or 'سالم' in label:
                        risk_summary['liver']['low'] += 1
                    elif 'medium' in label or 'متوسط' in label or 'بینابینی' in label:
                        risk_summary['liver']['medium'] += 1
                    elif 'high' in label or 'بالا' in label or 'کبد چرب' in label:
                        risk_summary['liver']['high'] += 1
    
    # Biomarker statistics (most common concerning biomarkers)
    biomarker_stats = {}
    concerning_biomarkers = []
    
    # Get all lab test results
    lab_test_ids = [lt.id for lt in lab_tests]
    lab_test_results = LabTestResult.objects.filter(
        lab_test_id__in=lab_test_ids
    ).select_related('biomarker', 'lab_test')
    
    # Group by biomarker
    biomarker_groups = {}
    for result in lab_test_results:
        biomarker_name = result.biomarker_name
        if biomarker_name not in biomarker_groups:
            biomarker_groups[biomarker_name] = {
                'values': [],
                'scores': [],
                'statuses': [],
            }
        
        if result.value is not None:
            biomarker_groups[biomarker_name]['values'].append(result.value)
        if result.fuzzy_score is not None:
            biomarker_groups[biomarker_name]['scores'].append(result.fuzzy_score)
        if result.status:
            biomarker_groups[biomarker_name]['statuses'].append(result.status)
    
    # Calculate statistics for each biomarker
    for biomarker_name, data in biomarker_groups.items():
        if data['values']:
            avg_value = sum(data['values']) / len(data['values'])
            min_value = min(data['values'])
            max_value = max(data['values'])
            
            avg_score = sum(data['scores']) / len(data['scores']) if data['scores'] else None
            
            # Count concerning statuses
            concerning_count = len([s for s in data['statuses'] if s in ['نیازمند بررسی', 'کمبود']])
            
            biomarker_stats[biomarker_name] = {
                'average': avg_value,
                'min': min_value,
                'max': max_value,
                'count': len(data['values']),
                'average_score': avg_score,
                'concerning_count': concerning_count,
                'concerning_percentage': (concerning_count / len(data['statuses']) * 100) if data['statuses'] else 0,
            }
            
            # Add to concerning list if >30% are concerning
            if biomarker_stats[biomarker_name]['concerning_percentage'] > 30:
                concerning_biomarkers.append({
                    'name': biomarker_name,
                    'concerning_percentage': biomarker_stats[biomarker_name]['concerning_percentage'],
                    'average_score': avg_score,
                })
    
    # Sort concerning biomarkers by percentage
    concerning_biomarkers.sort(key=lambda x: x['concerning_percentage'], reverse=True)
    
    return {
        'avg_overall_score': round(avg_overall_score) if avg_overall_score else None,
        'health_distribution': health_distribution,
        'index_averages': index_averages,
        'risk_summary': risk_summary,
        'biomarker_statistics': biomarker_stats,
        'concerning_biomarkers': concerning_biomarkers[:10],  # Top 10 most concerning
    }


@api_view(["GET"])
@hr_required
def employee_lab_test_detail(request, employee_id):
    """View detailed lab test results for a specific employee"""
    language = request.session.get(settings.LANGUAGE_COOKIE_NAME, 'en')
    translation.activate(language)
    
    try:
        employee = get_object_or_404(AppUser, id=employee_id)
        
        # Verify employee belongs to a company the HR has access to
        if not request.user.is_superuser:
            employee_companies = ZiLuckGroup.objects.filter(
                is_company=True,
                user=employee.django_user
            ).filter(
                Q(admin=request.user) | Q(creator=request.user)
            )
            if not employee_companies.exists():
                return generate_error_page(request, "Access denied. Employee not in your company.")
        
        # Get all lab tests for this employee
        lab_tests = LabTest.objects.filter(user=employee).order_by('-test_date')
        
        # Get notifications
        zipped_data, total_new_messages, free_time = new_messages(request)
        total_notifications = len(free_time)
        message = list(zipped_data) if zipped_data else []
        
        context = {
            'employee': employee,
            'lab_tests': lab_tests,
            'message': message,
            'total_new_messages': total_new_messages,
            'free_time': free_time,
            'total_notifications': total_notifications,
        }
        
        return render(request, 'hr/employee_lab_test_detail.html', context)
        
    except Exception as e:
        print(f"[HR] Error in employee_lab_test_detail: {str(e)}")
        import traceback
        traceback.print_exc()
        return generate_error_page(request, f"Error loading employee details: {str(e)}")


@api_view(["GET"])
@hr_required
def employee_eating_habits(request):
    """
    HR Dashboard showing employee eating habits and nutrition statistics.
    Shows average calories, carbs, protein, fat, and most used foods.
    """
    language = request.session.get(settings.LANGUAGE_COOKIE_NAME, 'en')
    translation.activate(language)
    
    try:
        # Get company groups that the HR user has access to
        if request.user.is_superuser:
            company_groups = ZiLuckGroup.objects.filter(is_company=True)
        else:
            # Get companies where HR user is a member (via group membership) or admin
            company_groups = ZiLuckGroup.objects.filter(
                Q(is_company=True) & 
                (Q(admin=request.user) | Q(id__in=request.user.groups.values_list('id', flat=True)))
            ).distinct()
        
        # Get selected company from query params
        company_id = request.GET.get('company_id')
        if company_id:
            try:
                selected_company = ZiLuckGroup.objects.get(id=company_id, is_company=True)
            except ZiLuckGroup.DoesNotExist:
                selected_company = None
        else:
            selected_company = company_groups.first() if company_groups.exists() else None
        
        # Get all employees (users) in the selected company
        employees = AppUser.objects.none()
        if selected_company:
            employees = AppUser.objects.filter(
                django_user__groups=selected_company
            ).select_related('profile')
        
        # Get all eating logs for these employees
        eating_logs = Eating.objects.none()
        if selected_company:
            eating_logs = Eating.objects.filter(
                user__django_user__groups=selected_company
            ).select_related('user', 'user__profile', 'food')
        
        # Calculate average nutrition values
        avg_calories = 0
        avg_carbs = 0
        avg_protein = 0
        avg_fat = 0
        total_eating_logs = 0
        
        if eating_logs.exists():
            # Calculate totals for all eating logs
            total_calories = 0
            total_carbs = 0
            total_protein = 0
            total_fat = 0
            
            for eating_log in eating_logs:
                if eating_log.food:
                    # Calculate nutrition based on amount (amount is in grams)
                    amount_g = eating_log.amount
                    if amount_g > 0:
                        # Nutrition per 100g * (amount_g / 100)
                        total_calories += (amount_g / 100) * eating_log.food.Calories
                        total_carbs += (amount_g / 100) * eating_log.food.Carbohydrates_g
                        total_protein += (amount_g / 100) * eating_log.food.Protein_g
                        total_fat += (amount_g / 100) * eating_log.food.Fat_g
                        total_eating_logs += 1
            
            if total_eating_logs > 0:
                # Calculate average nutrition per employee
                # Get unique employees count
                unique_employees = employees.count()
                if unique_employees > 0:
                    # Average per employee (total nutrition / number of employees)
                    avg_calories = total_calories / unique_employees
                    avg_carbs = total_carbs / unique_employees
                    avg_protein = total_protein / unique_employees
                    avg_fat = total_fat / unique_employees
        
        # Get most used foods with frequency
        most_used_foods = []
        if eating_logs.exists():
            # Count food occurrences
            from collections import Counter
            food_counter = Counter()
            
            for eating_log in eating_logs:
                if eating_log.food:
                    food_name = eating_log.food.FA_Name or eating_log.food.EN_Name or "Unknown"
                    food_counter[food_name] += 1
            
            # Get top 20 most used foods
            most_used_foods = food_counter.most_common(20)
        
        # Get notifications (minimal for HR)
        zipped_data, total_new_messages, free_time = new_messages(request)
        total_notifications = len(free_time)
        message = list(zipped_data) if zipped_data else []
        
        context = {
            'selected_company': selected_company,
            'company_groups': company_groups,
            'employees': employees,
            'total_employees': employees.count(),
            'avg_calories': round(avg_calories, 1),
            'avg_carbs': round(avg_carbs, 1),
            'avg_protein': round(avg_protein, 1),
            'avg_fat': round(avg_fat, 1),
            'most_used_foods': most_used_foods,
            'total_eating_logs': total_eating_logs,
            'message': message,
            'total_new_messages': total_new_messages,
            'free_time': free_time,
            'total_notifications': total_notifications,
        }
        
        return render(request, 'hr/employee_eating_habits.html', context)
        
    except Exception as e:
        print(f"[HR] Error in employee_eating_habits: {str(e)}")
        import traceback
        traceback.print_exc()
        return generate_error_page(request, f"Error loading eating habits: {str(e)}")


@api_view(["GET"])
@hr_required
def critical_cases(request):
    """
    HR Dashboard showing critical health cases (employees with health score < 60).
    Shows statistics and list of critical cases.
    """
    language = request.session.get(settings.LANGUAGE_COOKIE_NAME, 'en')
    translation.activate(language)
    
    try:
        # Get company groups that the HR user has access to
        if request.user.is_superuser:
            company_groups = ZiLuckGroup.objects.filter(is_company=True)
        else:
            # Get companies where HR user is a member (via group membership) or admin
            company_groups = ZiLuckGroup.objects.filter(
                Q(is_company=True) & 
                (Q(admin=request.user) | Q(id__in=request.user.groups.values_list('id', flat=True)))
            ).distinct()
        
        # Get selected company from query params
        company_id = request.GET.get('company_id')
        if company_id:
            try:
                selected_company = ZiLuckGroup.objects.get(id=company_id, is_company=True)
            except ZiLuckGroup.DoesNotExist:
                selected_company = None
        else:
            selected_company = company_groups.first() if company_groups.exists() else None
        
        # Get all employees (users) in the selected company
        employees = AppUser.objects.none()
        if selected_company:
            employees = AppUser.objects.filter(
                django_user__groups=selected_company
            ).select_related('profile')
        
        # Get all lab tests for these employees
        lab_tests = LabTest.objects.none()
        if selected_company:
            lab_tests = LabTest.objects.filter(
                user__django_user__groups=selected_company,
                overall_health_score__isnull=False
            ).select_related('user', 'user__profile').order_by('-test_date')
        
        # Calculate statistics
        total_employees = employees.count()
        critical_cases_list = []
        all_health_scores = []
        total_health_score = 0
        health_score_count = 0
        
        # Get latest lab test for each employee
        employee_latest_tests = {}
        for lab_test in lab_tests:
            user_id = lab_test.user.id
            if user_id not in employee_latest_tests:
                employee_latest_tests[user_id] = lab_test
        
        # Process each employee's latest test
        for user_id, latest_test in employee_latest_tests.items():
            health_score = latest_test.overall_health_score
            if health_score is not None:
                all_health_scores.append(health_score)
                total_health_score += health_score
                health_score_count += 1
                
                # Check if critical (health score < 60)
                if health_score < 60:
                    employee = latest_test.user
                    critical_cases_list.append({
                        'employee': employee,
                        'health_score': health_score,
                        'test_date': latest_test.test_date,
                        'lab_test': latest_test,
                    })
        
        # Calculate averages
        avg_health_score = (total_health_score / health_score_count) if health_score_count > 0 else 0
        overall_health_percentage = round(avg_health_score, 0) if avg_health_score > 0 else 0
        
        # Critical cases count
        critical_cases_count = len(critical_cases_list)
        
        # Physical productivity (employees with health score >= 70, or non-critical)
        physical_productivity_count = health_score_count - critical_cases_count
        
        # Sort critical cases by health score (lowest first - most critical)
        critical_cases_list.sort(key=lambda x: x['health_score'] if x['health_score'] else 999)
        
        # Get notifications (minimal for HR)
        zipped_data, total_new_messages, free_time = new_messages(request)
        total_notifications = len(free_time)
        message = list(zipped_data) if zipped_data else []
        
        context = {
            'selected_company': selected_company,
            'company_groups': company_groups,
            'total_employees': total_employees,
            'overall_health_percentage': overall_health_percentage,
            'critical_cases_count': critical_cases_count,
            'physical_productivity_count': physical_productivity_count,
            'avg_health_score': round(avg_health_score, 1),
            'critical_cases_list': critical_cases_list,
            'message': message,
            'total_new_messages': total_new_messages,
            'free_time': free_time,
            'total_notifications': total_notifications,
        }
        
        return render(request, 'hr/critical_cases.html', context)
        
    except Exception as e:
        print(f"[HR] Error in critical_cases: {str(e)}")
        import traceback
        traceback.print_exc()
        return generate_error_page(request, f"Error loading critical cases: {str(e)}")


@api_view(["GET"])
@hr_required
def green_employees(request):
    """
    HR Dashboard showing green employees (employees with health score > 70).
    Shows statistics and list of healthy employees.
    """
    language = request.session.get(settings.LANGUAGE_COOKIE_NAME, 'en')
    translation.activate(language)
    
    try:
        # Get company groups that the HR user has access to
        if request.user.is_superuser:
            company_groups = ZiLuckGroup.objects.filter(is_company=True)
        else:
            # Get companies where HR user is a member (via group membership) or admin
            company_groups = ZiLuckGroup.objects.filter(
                Q(is_company=True) & 
                (Q(admin=request.user) | Q(id__in=request.user.groups.values_list('id', flat=True)))
            ).distinct()
        
        # Get selected company from query params
        company_id = request.GET.get('company_id')
        if company_id:
            try:
                selected_company = ZiLuckGroup.objects.get(id=company_id, is_company=True)
            except ZiLuckGroup.DoesNotExist:
                selected_company = None
        else:
            selected_company = company_groups.first() if company_groups.exists() else None
        
        # Get all employees (users) in the selected company
        employees = AppUser.objects.none()
        if selected_company:
            employees = AppUser.objects.filter(
                django_user__groups=selected_company
            ).select_related('profile')
        
        # Get all lab tests for these employees
        lab_tests = LabTest.objects.none()
        if selected_company:
            lab_tests = LabTest.objects.filter(
                user__django_user__groups=selected_company,
                overall_health_score__isnull=False
            ).select_related('user', 'user__profile').order_by('-test_date')
        
        # Calculate statistics
        total_employees = employees.count()
        green_employees_list = []
        all_health_scores = []
        total_health_score = 0
        health_score_count = 0
        
        # Get latest lab test for each employee
        employee_latest_tests = {}
        for lab_test in lab_tests:
            user_id = lab_test.user.id
            if user_id not in employee_latest_tests:
                employee_latest_tests[user_id] = lab_test
        
        # Process each employee's latest test
        for user_id, latest_test in employee_latest_tests.items():
            health_score = latest_test.overall_health_score
            if health_score is not None:
                all_health_scores.append(health_score)
                total_health_score += health_score
                health_score_count += 1
                
                # Check if green employee (health score > 70)
                if health_score > 70:
                    employee = latest_test.user
                    green_employees_list.append({
                        'employee': employee,
                        'health_score': health_score,
                        'test_date': latest_test.test_date,
                        'lab_test': latest_test,
                    })
        
        # Calculate averages
        avg_health_score = (total_health_score / health_score_count) if health_score_count > 0 else 0
        overall_health_percentage = round(avg_health_score, 0) if avg_health_score > 0 else 0
        
        # Green employees count
        green_employees_count = len(green_employees_list)
        
        # Critical cases (employees with health score < 60)
        critical_cases_count = sum(1 for score in all_health_scores if score < 60)
        
        # Sort green employees by health score (highest first - best health)
        green_employees_list.sort(key=lambda x: x['health_score'] if x['health_score'] else 0, reverse=True)
        
        # Get notifications (minimal for HR)
        zipped_data, total_new_messages, free_time = new_messages(request)
        total_notifications = len(free_time)
        message = list(zipped_data) if zipped_data else []
        
        context = {
            'selected_company': selected_company,
            'company_groups': company_groups,
            'total_employees': total_employees,
            'overall_health_percentage': overall_health_percentage,
            'green_employees_count': green_employees_count,
            'critical_cases_count': critical_cases_count,
            'avg_health_score': round(avg_health_score, 1),
            'green_employees_list': green_employees_list,
            'message': message,
            'total_new_messages': total_new_messages,
            'free_time': free_time,
            'total_notifications': total_notifications,
        }
        
        return render(request, 'hr/green_employees.html', context)
        
    except Exception as e:
        print(f"[HR] Error in green_employees: {str(e)}")
        import traceback
        traceback.print_exc()
        return generate_error_page(request, f"Error loading green employees: {str(e)}")


@api_view(["GET"])
@hr_required
def company_risks(request):
    """
    HR Dashboard showing company risks with 5 pie charts for different health categories.
    Shows average scores for: Metabolic & Inflammation, Hematology & Nutrition,
    Liver & Kidney Function, Thyroid, Hormonal & Minerals.
    """
    language = request.session.get(settings.LANGUAGE_COOKIE_NAME, 'en')
    translation.activate(language)
    
    try:
        # Get company groups that the HR user has access to
        if request.user.is_superuser:
            company_groups = ZiLuckGroup.objects.filter(is_company=True)
        else:
            # Get companies where HR user is a member (via group membership) or admin
            company_groups = ZiLuckGroup.objects.filter(
                Q(is_company=True) & 
                (Q(admin=request.user) | Q(id__in=request.user.groups.values_list('id', flat=True)))
            ).distinct()
        
        # Get selected company from query params
        company_id = request.GET.get('company_id')
        if company_id:
            try:
                selected_company = ZiLuckGroup.objects.get(id=company_id, is_company=True)
            except ZiLuckGroup.DoesNotExist:
                selected_company = None
        else:
            selected_company = company_groups.first() if company_groups.exists() else None
        
        # Get gender filter
        gender_filter = request.GET.get('gender', '')  # 'male', 'female', or '' for all
        
        # Get all lab tests for employees in the selected company
        # Only include tests with uploaded reports (source_file exists)
        lab_tests = LabTest.objects.none()
        if selected_company:
            lab_tests = LabTest.objects.filter(
                user__django_user__groups=selected_company,
                source_file__isnull=False,  # Only uploaded reports
                indices__isnull=False  # Must have indices calculated
            ).select_related('user', 'user__profile').order_by('user', '-test_date', '-created_at')
            
            # Apply gender filter if specified
            if gender_filter == 'male':
                lab_tests = lab_tests.filter(user__profile__gender=0)
            elif gender_filter == 'female':
                lab_tests = lab_tests.filter(user__profile__gender=1)
        
        # Get latest test per employee (only one test per person)
        employee_latest_tests = {}
        for lab_test in lab_tests:
            user_id = lab_test.user.id
            if user_id not in employee_latest_tests:
                employee_latest_tests[user_id] = lab_test
        
        # Category mapping: English index name -> Persian display name
        category_mapping = {
            'Cardiometabolic': {
                'name': 'متابولیک و التهاب',
                'name_en': 'Metabolic & Inflammation',
                'key': 'Cardiometabolic'
            },
            'Hematology': {
                'name': 'هماتولوژی و تغذیه',
                'name_en': 'Hematology & Nutrition',
                'key': 'Hematology'
            },
            'Organ Function': {
                'name': 'عملکرد کبد و کلیه',
                'name_en': 'Liver & Kidney Function',
                'key': 'Organ Function'
            },
            'Thyroid': {
                'name': 'تیروئید',
                'name_en': 'Thyroid',
                'key': 'Thyroid'
            },
            'Hormonal': {
                'name': 'هورمونی و مواد معدنی',
                'name_en': 'Hormonal & Minerals',
                'key': 'Hormonal'
            }
        }
        
        # Calculate average scores for each category
        category_stats = {}
        for category_key, category_info in category_mapping.items():
            scores = []
            for user_id, lab_test in employee_latest_tests.items():
                if lab_test.indices and category_key in lab_test.indices:
                    score = lab_test.indices[category_key]
                    if score is not None:
                        scores.append(score)
            
            if scores:
                avg_score = sum(scores) / len(scores)
                category_stats[category_key] = {
                    'name': category_info['name'],
                    'name_en': category_info['name_en'],
                    'average': round(avg_score, 1),
                    'percentage': round(avg_score, 0),
                    'count': len(scores)
                }
            else:
                category_stats[category_key] = {
                    'name': category_info['name'],
                    'name_en': category_info['name_en'],
                    'average': 0,
                    'percentage': 0,
                    'count': 0
                }
        
        # Get total employees count
        total_employees = 0
        if selected_company:
            employees = AppUser.objects.filter(
                django_user__groups=selected_company
            )
            if gender_filter == 'male':
                employees = employees.filter(profile__gender=0)
            elif gender_filter == 'female':
                employees = employees.filter(profile__gender=1)
            total_employees = employees.count()
        
        # Get notifications (minimal for HR)
        zipped_data, total_new_messages, free_time = new_messages(request)
        total_notifications = len(free_time)
        message = list(zipped_data) if zipped_data else []
        
        context = {
            'selected_company': selected_company,
            'company_groups': company_groups,
            'gender_filter': gender_filter,
            'total_employees': total_employees,
            'category_stats': category_stats,
            'message': message,
            'total_new_messages': total_new_messages,
            'free_time': free_time,
            'total_notifications': total_notifications,
        }
        
        return render(request, 'hr/company_risks.html', context)
        
    except Exception as e:
        print(f"[HR] Error in company_risks: {str(e)}")
        import traceback
        traceback.print_exc()
        return generate_error_page(request, f"Error loading company risks: {str(e)}")

