"""
Hospital Panel Views
Handles all views for hospital user accounts including:
- Credentials page
- Users page (filtered by hospital_code)
- Upgrades page (subscription monitoring)
"""

from django.shortcuts import render, redirect, get_object_or_404
from django.views.decorators.csrf import csrf_exempt
from rest_framework.decorators import api_view
from django.contrib.auth.decorators import login_required
from django.contrib.auth.models import Group
from django.contrib.auth.models import User as DjangoUser
from django.http import HttpResponse, JsonResponse
from datetime import timedelta, datetime, date
from django.contrib import messages
from django.core.cache import cache
from django.db.models import Q, Count, Avg, Sum, Case, When, IntegerField
from django.core.paginator import Paginator, EmptyPage, PageNotAnInteger
from django.utils import translation
from django.conf import settings
import json

from APIs.models import User as AppUser, Profile
from panel.models import Hospital
from payment.models import HospitalUserUpgrade
from panel.manager import generate_error_page


def hospital_required(view_func):
    """Decorator to ensure user has hospital permissions"""
    def wrapper(request, *args, **kwargs):
        if not request.user.is_authenticated:
            return redirect('panel-login')
        
        if not request.user.groups.filter(name='Hospital').exists():
            return generate_error_page(request, "Access denied. Hospital permissions required.")
        
        # Ensure hospital account exists
        try:
            hospital = Hospital.objects.get(django_user=request.user)
            request.hospital = hospital
        except Hospital.DoesNotExist:
            return generate_error_page(request, "Hospital account not found.")
        
        return view_func(request, *args, **kwargs)
    return wrapper


@api_view(["GET"])
@hospital_required
def credentials(request):
    """
    Hospital Credentials Page
    Displays API key, secret, and other hospital information
    """
    hospital = request.hospital
    
    context = {
        'hospital': hospital,
        'page_title': 'Credentials',
    }
    
    return render(request, template_name='hospital/credentials.html', context=context)


@api_view(["POST"])
@hospital_required
def regenerate_credentials(request):
    """
    Regenerate API credentials for hospital
    """
    try:
        hospital = request.hospital
        api_key, api_secret = hospital.regenerate_api_credentials()
        
        messages.success(request, 'API credentials have been regenerated successfully.')
        return redirect('panel-hospital-credentials')
    
    except Exception as e:
        messages.error(request, f'Error regenerating credentials: {str(e)}')
        return redirect('panel-hospital-credentials')


@api_view(["GET"])
@hospital_required
def users(request):
    """
    Hospital Users Page
    Lists all users filtered by hospital_code
    """
    hospital = request.hospital
    
    # Get all users with this hospital code
    users_list = AppUser.objects.filter(
        profile__hospital_code=hospital.hospital_code
    ).select_related('profile', 'django_user').order_by('-created_at')
    
    # Search functionality
    search_query = request.GET.get('search', '')
    if search_query:
        users_list = users_list.filter(
            Q(profile__first_name__icontains=search_query) |
            Q(profile__last_name__icontains=search_query) |
            Q(phone_number__icontains=search_query)
        )
    
    # Pagination
    page = request.GET.get('page', 1)
    paginator = Paginator(users_list, 25)  # 25 users per page
    
    try:
        users_page = paginator.page(page)
    except PageNotAnInteger:
        users_page = paginator.page(1)
    except EmptyPage:
        users_page = paginator.page(paginator.num_pages)
    
    context = {
        'hospital': hospital,
        'users': users_page,
        'total_users': users_list.count(),
        'search_query': search_query,
        'page_title': 'Users',
    }
    
    return render(request, template_name='hospital/users.html', context=context)


@api_view(["GET"])
@hospital_required
def upgrades(request):
    """
    Hospital Upgrades Page
    Displays subscription monitoring grouped into:
    - Expiring Soon (< 7 days remaining)
    - Recently Upgraded (within last 7 days)
    - Active Subscriptions (all others)
    """
    hospital = request.hospital
    
    # Get all upgrades for this hospital
    all_upgrades = HospitalUserUpgrade.objects.filter(
        hospital_code=hospital.hospital_code
    ).select_related('user', 'user__profile').order_by('-upgrade_date')
    
    # Calculate today's date
    today = datetime.now()
    seven_days_ago = today - timedelta(days=7)
    
    # Group upgrades
    expiring_soon = []
    recently_upgraded = []
    active_subscriptions = []
    
    for upgrade in all_upgrades:
        days_remaining = upgrade.days_remaining
        
        # Check if expiring soon (< 7 days)
        if 0 < days_remaining < 7:
            expiring_soon.append(upgrade)
        # Check if recently upgraded (within last 7 days)
        elif upgrade.upgrade_date >= seven_days_ago:
            recently_upgraded.append(upgrade)
        # Active subscriptions (still has time)
        elif days_remaining > 0:
            active_subscriptions.append(upgrade)
    
    # Calculate summary counts
    total_active = len([u for u in all_upgrades if u.days_remaining > 0])
    
    context = {
        'hospital': hospital,
        'expiring_soon': expiring_soon,
        'expiring_soon_count': len(expiring_soon),
        'recently_upgraded': recently_upgraded,
        'recently_upgraded_count': len(recently_upgraded),
        'active_subscriptions': active_subscriptions,
        'active_subscriptions_count': len(active_subscriptions),
        'total_upgrades': all_upgrades.count(),
        'total_active': total_active,
        'page_title': 'Subscription Upgrades',
    }
    
    return render(request, template_name='hospital/upgrades.html', context=context)


@api_view(["GET"])
@hospital_required
def dashboard(request):
    """
    Hospital Dashboard - Overview page
    """
    hospital = request.hospital
    
    # Get statistics
    total_users = Profile.objects.filter(hospital_code=hospital.hospital_code).count()
    total_upgrades = HospitalUserUpgrade.objects.filter(hospital_code=hospital.hospital_code).count()
    active_upgrades = HospitalUserUpgrade.objects.filter(
        hospital_code=hospital.hospital_code,
        upgrade_date__gte=datetime.now() - timedelta(days=365)
    ).filter(
        upgrade_date__lte=datetime.now()
    ).count()
    
    # Recent upgrades (last 30 days)
    thirty_days_ago = datetime.now() - timedelta(days=30)
    recent_upgrades = HospitalUserUpgrade.objects.filter(
        hospital_code=hospital.hospital_code,
        upgrade_date__gte=thirty_days_ago
    ).select_related('user', 'user__profile').order_by('-upgrade_date')[:10]
    
    context = {
        'hospital': hospital,
        'total_users': total_users,
        'total_upgrades': total_upgrades,
        'active_upgrades': active_upgrades,
        'recent_upgrades': recent_upgrades,
        'page_title': 'Dashboard',
    }
    
    return render(request, template_name='hospital/dashboard.html', context=context)


@api_view(["GET"])
@hospital_required
def show_patient(request, patient_id):
    """
    Hospital Show Patient - View basic patient details
    Shows patient information for hospital users
    """
    hospital = request.hospital
    
    # Get patient and verify they exist
    patient = AppUser.objects.filter(id=patient_id).select_related('profile').first()
    if not patient:
        return redirect('panel-hospital-users')
    
    # Verify patient belongs to this hospital
    if patient.profile.hospital_code != hospital.hospital_code:
        return redirect('panel-hospital-users')
    
    # Get patient's upgrade history
    upgrades = HospitalUserUpgrade.objects.filter(
        user=patient,
        hospital_code=hospital.hospital_code
    ).order_by('-upgrade_date')
    
    # Get current active upgrade
    active_upgrade = None
    for upgrade in upgrades:
        if upgrade.days_remaining > 0:
            active_upgrade = upgrade
            break
    
    # Calculate BMI if height and weight available
    bmi = None
    if patient.profile.height and patient.profile.weight and patient.profile.height > 0:
        height_m = patient.profile.height / 100
        bmi = round(patient.profile.weight / (height_m ** 2), 1)
    
    context = {
        'hospital': hospital,
        'patient': patient,
        'upgrades': upgrades,
        'active_upgrade': active_upgrade,
        'bmi': bmi,
        'page_title': f'Patient: {patient.profile.first_name} {patient.profile.last_name}',
    }
    
    return render(request, template_name='hospital/patient_detail.html', context=context)

