import logging

from django.shortcuts import render, redirect, get_object_or_404
from django.views.decorators.csrf import csrf_exempt
from rest_framework.decorators import api_view
from rest_framework.response import Response
from django.contrib.auth.decorators import login_required
from django.contrib.auth.models import Group
from django.contrib.auth.models import User as DjangoUser
from django.contrib.auth import logout, authenticate
from django.contrib.auth import login as django_login
from django.http import HttpResponse, JsonResponse, HttpResponseBadRequest, Http404
from datetime import timedelta, datetime, time
from django.utils import timezone

from APIs.views.firebase import send_notification_note
from panel.models import *
from panel.telegram_notification import TelegramNotification
from ziluck_project.settings import STATIC_ROOT, MEDIA_ROOT
import os

from APIs.models import User as AppUser, Sleep, Walking, Profile, User, Points, Shop, RequestedFoods, ContentService, \
    Video, PDF, LabTest, LabTestResult, BiomarkerDefinition
from APIs.models import Drug, SugarMeasurement, Weights, Diet, Food, UserQuestionnaire, Eating
from payment.models import ZarinPalPayment

from django.http import HttpResponseRedirect
from APIs.models import InsulinTypeSelection
from django.db.models import Q, Max
import jdatetime
from .utility import *
from panel.SMSNotification import *
from panel.PushNotification import PushNotification
import pandas as pd
from pandas import read_excel
from django.utils.safestring import mark_safe
from collections import defaultdict
from django.utils import translation
from django.conf import settings
from django.core.paginator import Paginator, EmptyPage, PageNotAnInteger
import traceback
import tempfile
import shutil
from panel.blood_test_analyzer import BloodHealthAnalyzer

logger = logging.getLogger(__name__)


def generate_error_page(request, error_message):
    # Get notifications for base template
    try:
        zipped_data, total_new_messages, free_time = new_messages(request)
        message = list(zipped_data) if zipped_data else []
        total_notifications = len(free_time)
    except:
        message = []
        total_new_messages = 0
        free_time = []
        total_notifications = 0
    
    context = {
        'message': message,  # For base template notifications
        'error_message': error_message,  # The actual error message
        'total_new_messages': total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }
    return render(request=request, template_name='doctor/error.html', context=context)


@api_view(["GET", "POST"])
def file_upload_page(request):
    """
    Doctor panel page for uploading and analyzing blood test Excel files.
    Supports novo_lab_test.xlsx format with grouping by patient name.
    """
    if not request.user.is_authenticated:
        return redirect('panel-login')

    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)

    user = request.user
    doctor = Doctor.objects.get(django_user=user)

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)
    
    # Get company groups that the doctor has access to
    # For doctors, show all company groups (they can assign users to any company)
    from APIs.models import ZiLuckGroup
    from django.db.models import Q
    available_company_groups = ZiLuckGroup.objects.filter(is_company=True).order_by('name')

    context = {
        "user": doctor,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        "free_time": free_time,
        "total_notifications": total_notifications,
        "patients_analysis": [],
        "error_message": None,
        "available_company_groups": available_company_groups,
    }

    # Handle POST: file upload and analysis
    if request.method == "POST":
        if 'file' not in request.FILES:
            context["error_message"] = "لطفا فایل Excel را انتخاب کنید."
            return render(request, template_name='doctor/file_upload.html', context=context)
        
        uploaded_file = request.FILES['file']
        file_extension = os.path.splitext(uploaded_file.name)[1].lower()
        
        if file_extension not in ['.xlsx', '.xls']:
            context["error_message"] = "فرمت فایل باید Excel (.xlsx یا .xls) باشد."
            return render(request, template_name='doctor/file_upload.html', context=context)
        
        # Create temp directory
        temp_dir = tempfile.mkdtemp(prefix="doctor_blood_test_")
        try:
            # Save uploaded file
            input_path = os.path.join(temp_dir, f"input{file_extension}")
            with open(input_path, "wb+") as destination:
                for chunk in uploaded_file.chunks():
                    destination.write(chunk)
            
            # Read Excel file
            try:
                df = read_excel(input_path)
            except Exception as e:
                context["error_message"] = f"خطا در خواندن فایل Excel: {str(e)}"
                return render(request, template_name='doctor/file_upload.html', context=context)
            
            # Map Persian column names (handle variations)
            column_mapping = {
                'کد پذیرش': 'code',
                'نام': 'first_name',
                'نام خانوادگی': 'last_name',
                'کد ملی': 'national_id',
                'کد آزمایش': 'test_code',
                'نام آزمایش': 'test_name',
                'متد': 'method',
                'واحد': 'unit',
                'از محدوده نرمال': 'ref_min',
                'تا محدوده نرمال': 'ref_max',
                'تاریخ پذیرش': 'date',
                'جواب': 'result',
                'مرکز': 'center',
                'سن': 'age',  # Age column
                'تاریخ تولد': 'birth_date',  # Birth date column
            }
            
            # Rename columns if they exist
            df.columns = [column_mapping.get(col, col) for col in df.columns]
            
            # Check required columns
            required_cols = ['first_name', 'last_name', 'test_name', 'result']
            missing_cols = [col for col in required_cols if col not in df.columns]
            if missing_cols:
                context["error_message"] = f"ستون‌های مورد نیاز یافت نشد: {', '.join(missing_cols)}"
                return render(request, template_name='doctor/file_upload.html', context=context)
            
            # Group by patient (first_name + last_name)
            patients_analysis = []
            
            # Test name mapping from Persian/Excel format to English analyzer format
            # Based on actual novo_lab_test.xlsx column names
            test_name_mapping = {
                # Liver/Kidney
                'ALT': 'ALT',
                'AST': 'AST',
                'GGT': 'GGT',
                'ALP': 'ALP',
                'Bili-T': 'Bilirubin Total',
                'Bili-D': 'Bilirubin Direct',
                'Cr': 'Creatinine',
                'Urea': 'Urea',
                'eGFR': 'eGFR',
                # Metabolic
                'FBS': 'Fasting Glucose',
                'A1C-E': 'HbA1c',
                'HbA1c': 'HbA1c',
                'TG': 'Triglycerides',
                'Triglycerides': 'Triglycerides',
                'Chol': 'Total Cholesterol',
                'Total Cholesterol': 'Total Cholesterol',
                'LDL': 'LDL',
                'HDL': 'HDL',
                # Hematology
                'Hgb': 'Hb',
                'Hb': 'Hb',
                'HCT': 'HCT',
                'Hct': 'HCT',
                'RBC': 'RBC',
                'MCV': 'MCV',
                'MCH': 'MCH',
                'MCHC': 'MCHC',
                'WBC': 'WBC',
                'Platelet': 'Platelet',
                # Thyroid
                'TSH': 'TSH',
                'T3 Free': 'T3 Free',
                'T4 Free': 'T4 Free',
                # Vitamins/Minerals
                'FTL': 'Ferritin',
                'Ferritin': 'Ferritin',
                'Fe': 'Iron',
                'Iron': 'Iron',
                'Vit D3': 'Vitamin D3',
                'Vitamin D3': 'Vitamin D3',
                'Vit B12': 'Vitamin B12',
                'Vitamin B12': 'Vitamin B12',
                # Inflammation/Hormones
                'CRP': 'CRP',
                'Calcium': 'Calcium',
                'Magnesium': 'Magnesium',
                'Cortisol': 'Cortisol',
                'Estradiol': 'Estradiol',
                'Testosterone': 'Testosterone',
                'FSH': 'FSH',
            }
            
            # Get selected company group from form (if any)
            selected_company_group = None
            company_group_id = request.POST.get('company_group', '').strip()
            if company_group_id:
                try:
                    from APIs.models import ZiLuckGroup
                    selected_company_group = ZiLuckGroup.objects.get(
                        id=int(company_group_id),
                        is_company=True
                    )
                    print(f"[LAB_TEST] Doctor selected company group: {selected_company_group.name} (ID: {selected_company_group.id})")
                except (ValueError, ZiLuckGroup.DoesNotExist) as e:
                    print(f"[LAB_TEST] Invalid company group ID: {company_group_id}, error: {str(e)}")
                    selected_company_group = None
            
            # Track the first user's company groups (only if no company group was selected)
            first_user_company_groups = None
            
            # Group by patient
            for idx, ((first_name, last_name), group_df) in enumerate(df.groupby(['first_name', 'last_name'])):
                patient_name = f"{first_name} {last_name}".strip()
                
                # Extract patient info (first row of group)
                patient_row = group_df.iloc[0]
                national_id = str(patient_row.get('national_id', '')).strip() if 'national_id' in patient_row else ''
                test_date_str = str(patient_row.get('date', '')).strip() if 'date' in patient_row else ''
                lab_center = str(patient_row.get('center', '')).strip() if 'center' in patient_row else ''
                registration_number = str(patient_row.get('code', '')).strip() if 'code' in patient_row else ''
                
                # Extract age from Excel (read from test)
                age = None
                if 'age' in patient_row and not pd.isna(patient_row.get('age')):
                    try:
                        age = int(float(str(patient_row.get('age')).strip()))
                    except (ValueError, TypeError):
                        pass
                
                # If age not found, try to calculate from birth_date
                if age is None and 'birth_date' in patient_row:
                    birth_date_str = str(patient_row.get('birth_date', '')).strip()
                    if birth_date_str and not pd.isna(patient_row.get('birth_date')):
                        try:
                            from datetime import datetime, date
                            # Try to parse birth date
                            if '/' in birth_date_str:
                                parts = birth_date_str.split('/')
                                if len(parts) == 3:
                                    # Assume Persian/Jalali date - convert if needed
                                    # For now, try direct parsing
                                    try:
                                        birth_date = datetime.strptime(birth_date_str.replace('/', '-'), '%Y-%m-%d').date()
                                        today = date.today()
                                        age = today.year - birth_date.year - ((today.month, today.day) < (birth_date.month, birth_date.day))
                                    except:
                                        pass
                        except Exception:
                            pass
                
                # Parse test date (handle Persian date format if needed)
                test_date = None
                if test_date_str:
                    try:
                        # Try to parse date (assuming format like "1404/08/18")
                        if '/' in test_date_str:
                            parts = test_date_str.split('/')
                            if len(parts) == 3:
                                # Convert Persian/Jalali date to Gregorian if needed
                                # For now, assume it's already in a parseable format
                                from datetime import datetime
                                # Simple parsing - may need jdatetime conversion
                                test_date = datetime.strptime(test_date_str.replace('/', '-'), '%Y-%m-%d').date()
                    except Exception:
                        pass
                
                # Find or create user profile (no duplicates - uses existing profile)
                # first_name, last_name, and age are read from the lab test Excel file
                try:
                    user, user_created = LabTest.find_or_create_user_from_lab_data(
                        first_name=first_name,  # Read from Excel
                        last_name=last_name,  # Read from Excel
                        national_id=national_id if national_id else None,
                        gender='unknown',  # Could be extracted from data if available
                        age=age,  # Read from Excel (age column or calculated from birth_date)
                        created_by=doctor.django_user  # Current doctor
                    )
                    
                    # Assign users to company group
                    # Priority: 1) Selected company group, 2) First user's company groups
                    company_groups_to_assign = None
                    
                    if selected_company_group:
                        # Use selected company group (assign to ALL users - new and existing)
                        company_groups_to_assign = [selected_company_group]
                        try:
                            selected_company_group.user_set.add(user.django_user)
                            print(f"[LAB_TEST] Assigned user {user.id} ({first_name} {last_name}) to selected company group: {selected_company_group.name}")
                        except Exception as e:
                            print(f"[LAB_TEST] Error assigning user to selected company group: {str(e)}")
                            import traceback
                            traceback.print_exc()
                    else:
                        # No company group selected - use first user's company groups logic
                        if idx == 0:
                            # For the FIRST user: capture their company groups
                            try:
                                from APIs.models import ZiLuckGroup
                                first_user_company_groups = ZiLuckGroup.objects.filter(
                                    is_company=True,
                                    id__in=user.django_user.groups.values_list('id', flat=True)
                                ).distinct()
                                
                                if first_user_company_groups.exists():
                                    print(f"[LAB_TEST] First user {user.id} ({first_name} {last_name}) is in {first_user_company_groups.count()} company group(s) - will assign subsequent new users to same groups")
                                else:
                                    print(f"[LAB_TEST] First user {user.id} ({first_name} {last_name}) is not in any company groups - subsequent new users will not be assigned to companies")
                            except Exception as e:
                                print(f"[LAB_TEST] Error getting first user's company groups: {str(e)}")
                                first_user_company_groups = None
                        
                        # For subsequent NEW users: assign to first user's company groups
                        if idx > 0 and user_created and first_user_company_groups and first_user_company_groups.exists():
                            try:
                                # Assign new user to the same company groups as the first user
                                for company_group in first_user_company_groups:
                                    company_group.user_set.add(user.django_user)
                                print(f"[LAB_TEST] Assigned new user {user.id} ({first_name} {last_name}) to {first_user_company_groups.count()} company group(s) (same as first user)")
                            except Exception as e:
                                print(f"[LAB_TEST] Error assigning new user to first user's company groups: {str(e)}")
                                import traceback
                                traceback.print_exc()
                except Exception as e:
                    print(f"[DOCTOR] Error finding/creating user for {patient_name}: {str(e)}")
                    traceback.print_exc()
                    continue
                
                # Get gender and age from user profile for analysis
                gender = 'female' if user.profile.gender else 'male'
                age = LabTest.calculate_age_from_birth_date(user.profile.birth_date) if user.profile.birth_date else 30
                
                # Build raw_data dict for analyzer and collect results for database
                raw_data = {}
                test_results_data = []  # Store for LabTestResult creation
                
                for _, row in group_df.iterrows():
                    test_name_persian = str(row.get('test_name', '')).strip()
                    result_value = row.get('result', '')
                    ref_min = row.get('ref_min', None)
                    ref_max = row.get('ref_max', None)
                    unit = str(row.get('unit', '')).strip() if 'unit' in row else ''
                    method = str(row.get('method', '')).strip() if 'method' in row else ''
                    test_code = str(row.get('test_code', '')).strip() if 'test_code' in row else ''
                    
                    # Skip empty test names or results
                    if not test_name_persian or pd.isna(result_value):
                        continue
                    
                    # Skip non-test rows (like "S.N پذيرش", "S.Vg خونگيري خلا", etc.)
                    if any(skip in test_name_persian for skip in ['پذيرش', 'خونگيري', 'Attach', 'See the']):
                        continue
                    
                    # Map test name to English
                    test_name_english = test_name_mapping.get(test_name_persian, test_name_persian)
                    
                    # Try to find biomarker definition - improved matching
                    biomarker_def = None
                    try:
                        # First try exact match (case-insensitive)
                        biomarker_def = BiomarkerDefinition.objects.filter(
                            name__iexact=test_name_english
                        ).first()
                        
                        # If not found, try matching by persian_name
                        if not biomarker_def:
                            biomarker_def = BiomarkerDefinition.objects.filter(
                                persian_name__icontains=test_name_persian
                            ).first()
                        
                        # If still not found, try matching by full_name
                        if not biomarker_def:
                            biomarker_def = BiomarkerDefinition.objects.filter(
                                full_name__iexact=test_name_english
                            ).first()
                        
                        # If still not found, try partial match on name
                        if not biomarker_def:
                            biomarker_def = BiomarkerDefinition.objects.filter(
                                name__icontains=test_name_english
                            ).first()
                    except Exception as e:
                        print(f"[DOCTOR] Error finding biomarker definition for {test_name_english}: {str(e)}")
                        pass
                    
                    # Handle special cases
                    parsed_value = None
                    value_string = str(result_value).strip()
                    
                    if test_name_english == 'CRP':
                        # CRP might be "Negative" or numeric
                        if 'negative' in value_string.lower() or value_string == '0' or value_string == '':
                            parsed_value = 0
                            raw_data[test_name_english] = 0
                        else:
                            # Try to parse as number
                            try:
                                parsed_value = float(value_string)
                                raw_data[test_name_english] = parsed_value
                            except (ValueError, TypeError):
                                parsed_value = 0
                                raw_data[test_name_english] = 0
                    else:
                        # Handle values with asterisk (e.g., "93.0*")
                        if isinstance(result_value, str) and '*' in result_value:
                            value_string = result_value.replace('*', '').strip()
                            result_value = value_string
                        
                        # Try to parse numeric value
                        try:
                            parsed_value = float(value_string)
                            raw_data[test_name_english] = parsed_value
                        except (ValueError, TypeError):
                            parsed_value = None
                            raw_data[test_name_english] = result_value
                    
                    # Store for LabTestResult creation
                    test_results_data.append({
                        'biomarker': biomarker_def,
                        'biomarker_name': test_name_english,
                        'value': parsed_value,
                        'value_string': value_string if parsed_value is None else '',
                        'unit': unit,
                        'method': method,
                        'ref_min': float(ref_min) if ref_min and not pd.isna(ref_min) else None,
                        'ref_max': float(ref_max) if ref_max and not pd.isna(ref_max) else None,
                        'test_code': test_code,
                    })
                
                if not raw_data:
                    continue
                
                # Create profile for risk engine (use actual user profile data)
                profile = {
                    'gender': gender,
                    'age': age,
                    'height': user.profile.height if user.profile.height and user.profile.height > 0 else 170,
                    'weight': user.profile.weight if user.profile.weight and user.profile.weight > 0 else 70,
                    'sbp': 120,  # Could be extracted from data if available
                    'smoker': False,  # Could be extracted from data if available
                    'diabetesHistory': False,  # Could be extracted from data if available
                    'bpMeds': False,  # Could be extracted from data if available
                }
                
                # Run analysis
                try:
                    analyzer = BloodHealthAnalyzer(gender=gender, age=age)
                    analysis = analyzer.analyze(raw_data, profile=profile)
                    
                    # Create LabTest record in database
                    lab_test = LabTest.objects.create(
                        user=user,
                        test_date=test_date,
                        lab_center=lab_center,
                        registration_number=registration_number,
                        source_file=uploaded_file,
                        source_type='excel',
                        overall_health_score=analysis.get('Overall_Health_Score'),
                        indices=analysis.get('Indices', {}),
                        index_details=analysis.get('Index_Details', {}),
                        risks=analysis.get('Risks', {}),
                        raw_data=raw_data,
                        is_analyzed=True,
                        analyzed_at=timezone.now(),
                        created_by=doctor.django_user
                    )
                    
                    # Create LabTestResult records for each biomarker
                    for result_data in test_results_data:
                        # Calculate fuzzy score and status if biomarker definition exists
                        fuzzy_score = None
                        status = 'یافت نشد'
                        lifestyle_feedback = ''
                        
                        if result_data['biomarker'] and result_data['value'] is not None:
                            try:
                                temp_analyzer = BloodHealthAnalyzer(gender=gender, age=age)
                                fuzzy_score = temp_analyzer._calculate_fuzzy_score(
                                    result_data['biomarker_name'],
                                    result_data['value']
                                )
                                if fuzzy_score is not None:
                                    if fuzzy_score == 100:
                                        status = 'ایده‌آل'
                                    elif fuzzy_score >= 70:
                                        status = 'طبیعی'
                                    else:
                                        status = 'نیازمند بررسی'
                                    
                                    # Get lifestyle feedback
                                    lifestyle_feedback = temp_analyzer._get_lifestyle_feedback(
                                        result_data['biomarker_name'],
                                        result_data['value'],
                                        fuzzy_score
                                    )
                            except Exception:
                                pass
                        
                        LabTestResult.objects.create(
                            lab_test=lab_test,
                            biomarker=result_data['biomarker'],
                            biomarker_name=result_data['biomarker_name'],
                            value=result_data['value'],
                            value_string=result_data['value_string'],
                            unit=result_data['unit'],
                            method=result_data['method'],
                            ref_min=result_data['ref_min'],
                            ref_max=result_data['ref_max'],
                            test_code=result_data['test_code'],
                            fuzzy_score=round(fuzzy_score) if fuzzy_score is not None else None,
                            status=status,
                            lifestyle_feedback=lifestyle_feedback
                        )
                    
                    # Add to patients_analysis for display (get info from user profile)
                    patients_analysis.append({
                        'user_id': user.id,
                        'name': user.profile.first_name + ' ' + user.profile.last_name if user.profile.first_name and user.profile.last_name else patient_name,
                        'national_id': user.profile.national_id or national_id,
                        'test_date': test_date_str,
                        'lab_test_id': lab_test.id,
                        'analysis': analysis,
                        'raw_data': raw_data
                    })
                except Exception as e:
                    print(f"[DOCTOR] Error analyzing patient {patient_name}: {str(e)}")
                    traceback.print_exc()
                    continue
            
            context["patients_analysis"] = patients_analysis
            
            if not patients_analysis:
                context["error_message"] = "هیچ داده آزمایشی معتبری در فایل یافت نشد."
        
        finally:
            # Clean up temp directory
            try:
                shutil.rmtree(temp_dir)
            except Exception:
                pass
    
    return render(request, template_name='doctor/file_upload.html', context=context)


def new_messages(request):
    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)
    user = request.user
    patients_list = AppUser.objects.filter(profile__therapist=user)
    doctor = Doctor.objects.get(django_user=user)
    assistants = Assistant.objects.filter(doctor=doctor)

    today = date.today()
    t1 = datetime(year=today.year, month=today.month, day=today.day, hour=0, minute=0, second=0)
    t2 = datetime(year=today.year, month=today.month, day=today.day, hour=23, minute=59, second=59)
    t3 = t1 + timedelta(hours=47, minutes=59, seconds=59)
    free_time = TimeSlot.objects.filter(therapist_django_user=user, start_time__range=(t1, t3), is_available=0)

    messages = []

    qset = AppUser

    for assistant in assistants:
        qset = AppUser.objects.filter(profile__therapist=assistant.django_user)
        patients_list = patients_list.union(qset)
    patients_list = list(patients_list)
    number_of_new_messages = []

    # print(f"This is the patient list element one: {patients_list[2]}")

    for p in patients_list:
        temp = Message.objects.filter((Q(sender=p.django_user) & Q(has_been_seen=False)))
        number_of_new_messages.append(len(temp))
        for i in range(len(temp)):
            sender_name = User.objects.filter(django_user=temp[i].sender)[0].profile.first_name
            sender_id = User.objects.filter(django_user=temp[i].sender)[0].id

            # print(f"this is the message sender id 1: {temp[i].sender}")

            messages.append(
                {
                    'text': temp[i].text,
                    'sender': temp[i].sender,
                    'time': temp[i].time,
                    "sender_name": sender_name,
                    "sender_id": sender_id
                }
            )
    max_length = max(len(messages), len(number_of_new_messages))
    patients_list += [None] * (max_length - len(patients_list))
    number_of_new_messages += [None] * (max_length - len(number_of_new_messages))
    zipped_data = zip(patients_list, messages, number_of_new_messages)
    total_new_messages = len(messages)

    return zipped_data, total_new_messages, free_time


@api_view(["GET"])
def patients_archive(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)

    user = request.user
    patients_list = AppUser.objects.filter(profile__therapist=user)
    doctor = Doctor.objects.get(django_user=user)
    assistants = Assistant.objects.filter(doctor=doctor)

    now = datetime.now().date()

    three_days_ago = (datetime.now() - timedelta(days=3)).date()
    new_sub = []
    qset = AppUser

    for assistant in assistants:
        qset = AppUser.objects.filter(profile__therapist=assistant.django_user)
        patients_list = patients_list.union(qset)
    patients_list = list(patients_list)
    number_of_new_messages = []
    total_of_messages = []
    subscription_status = []
    point_user = []
    user_groups = []

    # Prepare data structure to hold group information for each patient
    django_groups = request.user.groups.filter(~Q(name='Assistant'))
    group_map = {}
    for group in django_groups:
        for user in group.user_set.all():
            if user.id not in group_map:
                group_map[user.id] = []
            group_map[user.id].append(group.name)

    non_premium_patients = []
    for p in patients_list:
        point = Points.objects.filter(user=p, date=now)
        if not point.exists():
            point_user.append(0)
        else:
            point_user.append(round(point[0].total_points))

        temp = Message.objects.filter((Q(sender=p.django_user) & Q(has_been_seen=False)))
        total_messages = Message.objects.filter(sender=p.django_user)
        number_of_new_messages.append(len(temp))
        total_of_messages.append(len(total_messages))

        # Collect group information for each patient
        user_groups.append(", ".join(group_map.get(p.django_user.id, ["---"])))

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'patients_list': zip(non_premium_patients, number_of_new_messages, subscription_status, new_sub, point_user,
                             total_of_messages, user_groups),
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
        'user': doctor,
        'number_of_new_messages': number_of_new_messages,
        'assistants': assistants

    }

    return render(request, template_name='doctor/patients_archive.html', context=context)


@api_view(["GET"])
def patients(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)

    user = request.user
    patients_list = AppUser.objects.filter(profile__therapist=user)
    doctor = Doctor.objects.get(django_user=user)
    assistants = Assistant.objects.filter(doctor=doctor)

    now = datetime.now().date()
    three_days_ago = (datetime.now() - timedelta(days=3)).date()
    new_sub = []
    qset = AppUser

    for assistant in assistants:
        qset = AppUser.objects.filter(profile__therapist=assistant.django_user)
        patients_list = patients_list.union(qset)
    patients_list = list(patients_list)
    number_of_new_messages = []
    total_of_messages = []
    subscription_status = []
    point_user = []
    user_groups = []
    diet_date = []
    weight = []
    discounts = []
    plan_types = []

    # Define groups to exclude
    excluded_groups = ["Patient", "Doctor", "Doctors", "Manager", "Assistant"]

    # Prepare data structure to hold group information for each patient
    django_groups = request.user.groups.filter(~Q(name='Assistant'))
    group_map = {}
    for patient in patients_list:
        # Exclude the unwanted groups
        patient_groups = patient.django_user.groups.exclude(name__in=excluded_groups)
        group_map[patient.django_user.id] = [group.name for group in patient_groups]

    # Filter patients with premium subscription
    premium_patients = []
    for p in patients_list:
        app_user = p
        django_user = app_user.django_user
        temp2 = TherapistSelection.objects.filter(user=django_user)

        # Check if the patient has a premium subscription
        if temp2.exists() and temp2.last().has_credit():
            # Handle first_weight (from profile)
            first_weight = p.profile.weight if p.profile and hasattr(p.profile, 'weight') else 0
            # Handle last_weight (from Weights model)
            last_weight_obj = Weights.objects.filter(user=p).last()
            last_weight = last_weight_obj.weight if last_weight_obj else first_weight  # Fallback to first_weight or 0

            # Calculate weight difference based on goal
            if p.profile and p.profile.goal_weight is not None and p.profile.weight is not None:
                if p.profile.goal_weight > p.profile.weight:
                    weight.append(last_weight - first_weight)
                else:
                    weight.append(first_weight - last_weight)
            else:
                weight.append(0)  # Default to 0 if profile or weights are missing
            # Handle discounts and plan types
            try:
                # Define plan type mapping
                plan_type_map = {
                    1: 'Silver', 
                    2: 'Gold', 
                    3: 'Platinum', 
                    4: 'Silver',
                    5: 'Gold',
                    6: 'Bronze',
                    7: 'Bronze',
                    8: 'Bronze'
                }
                
                plan_type_found = False
                
                # Debug output
                print(f"\n=== DEBUG for user: {app_user.profile.first_name} {app_user.profile.last_name} (ID: {app_user.id}) ===")
                
                # First, check ZarinPalPayment for plan type
                payment_qs = ZarinPalPayment.objects.filter(user=app_user)
                print(f"ZarinPalPayment exists: {payment_qs.exists()}")
                if payment_qs.exists():
                    payment = payment_qs.last()
                    print(f"Payment.plan value: {payment.plan}, type: {type(payment.plan)}")
                    if payment.discount:
                        discounts.append(payment.discount.code)
                    else:
                        discounts.append('None')
                    if payment.plan:
                        plan_types.append(plan_type_map.get(payment.plan, 'Unknown'))
                        plan_type_found = True
                        print(f"Plan type found in ZarinPalPayment: {plan_type_map.get(payment.plan, 'Unknown')}")
                else:
                    discounts.append('None')
                
                # If not found in ZarinPalPayment, check Purchase/Shop model
                if not plan_type_found:
                    from APIs.models import Purchase, Shop
                    purchase_qs = Purchase.objects.filter(
                        user=app_user, 
                        shop_item__service_type='plan'
                    ).select_related('shop_item').order_by('-purchase_date')
                    
                    print(f"Purchase with plan exists: {purchase_qs.exists()}")
                    if purchase_qs.exists():
                        purchase = purchase_qs.first()
                        print(f"Purchase.shop_item.plan_type: {purchase.shop_item.plan_type}")
                        if purchase.shop_item.plan_type:
                            plan_types.append(plan_type_map.get(purchase.shop_item.plan_type, 'Unknown'))
                            plan_type_found = True
                            print(f"Plan type found in Purchase: {plan_type_map.get(purchase.shop_item.plan_type, 'Unknown')}")
                
                # If still not found, append default
                if not plan_type_found:
                    plan_types.append('---')
                    print("No plan type found, using '---'")
                    
            except Exception as e:
                print(f"Exception occurred: {str(e)}")
                discounts.append('Error')
                plan_types.append('---')

            # Append to premium_patients and other lists
            premium_patients.append(p)
            ts = temp2.last()
            subscription_status.append(ts.remaining_credit())

            if ts.created_at.date() >= three_days_ago:
                new_sub.append(True)
            else:
                new_sub.append(False)

            # Collect points for the patient
            point = Points.objects.filter(user=p, date=now)
            if not point.exists():
                point_user.append(0)
            else:
                point_user.append(round(point[0].total_points))

            diet = Diet.objects.filter(user=p)
            if diet.exists():
                diet_date.append(diet.last().from_date)
            else:
                diet_date.append('None')
            # Collect message data for the patient
            temp = Message.objects.filter(Q(sender=p.django_user) & Q(has_been_seen=False))
            total_messages = Message.objects.filter(sender=p.django_user)
            number_of_new_messages.append(len(temp))
            total_of_messages.append(len(total_messages))

            # Collect group information for each patient
            user_groups.append(", ".join(group_map.get(p.django_user.id, ["---"])))

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'patients_list': zip(premium_patients, weight, number_of_new_messages, subscription_status, new_sub, point_user,
                             total_of_messages, user_groups, diet_date, plan_types),
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
        'user': doctor,
        'number_of_new_messages': number_of_new_messages,
        'assistants': assistants
    }

    return render(request, template_name='doctor/patients.html', context=context)


@api_view(["GET"])
def assign_patient(request, patient_id, assistant_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    user = request.user

    patient = AppUser.objects.filter(id=patient_id).first()
    if patient is None:
        return generate_error_page(request, "patient not found!")

    if patient.profile.therapist is None:
        return generate_error_page(request, "patient has no therapist")
    access = False
    doctor = Doctor.objects.get(django_user=user)
    if int(assistant_id) == 0:
        assistants = []
        access = True
    else:
        assistants = Assistant.objects.filter(doctor=doctor)

    if patient.profile.therapist == doctor.django_user:
        access = True
    if not access:
        # check doctor's assistants
        for assistant in assistants:
            if patient.profile.therapist == assistant.django_user:
                access = True
                break
    if not access:
        return generate_error_page(request, "user have not access to this patient")
    if int(assistant_id) != 0:
        new_therapist = Assistant.objects.filter(id=assistant_id).first()
    else:
        new_therapist = doctor
    if new_therapist is None:
        return generate_error_page(request, "there is no such a therapist!")

    patient_name = patient.profile.first_name
    TelegramNotification().send_assign_patient_notif(username=patient_name, chat_id=new_therapist.telegram_chat_id)

    patient.profile.therapist = new_therapist.django_user
    patient.profile.save()

    return redirect('panel-doctor-patients')


@api_view(["GET"])
def assistants_page(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    doctor = Doctor.objects.get(django_user=request.user)
    assistants = Assistant.objects.filter(doctor=doctor)

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        "assistants": assistants,
        'user': doctor,
        "message": zipped_data,
        'free_time': free_time,
        'total_notifications': total_notifications,
        "total_new_messages": total_new_messages,
    }
    return render(request, 'doctor/assistants.html', context=context)


@api_view(["GET"])
def add_assistant_page(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    doctor = Doctor.objects.get(django_user=django_user)
    if doctor is None:
        return generate_error_page(request, "doctor not found!")
    context = {
        'doctor': doctor,
        'user': doctor,
    }
    return render(request, 'doctor/add-assistant.html', context=context)


@api_view(["POST"])
def create_assistant(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Doctor').exists():
        return generate_error_page(request, "you have not access")

    doctor = Doctor.objects.get(id=request.POST.get("doctor_id"))
    manager = doctor.manager

    if doctor is None:
        return generate_error_page(request, "doctor not selected!")
    first_name = request.POST.get("first_name")
    last_name = request.POST.get("last_name")
    if len(first_name) < 3 or len(last_name) < 3:
        return generate_error_page(request, "نام یا نام خانوادگی اشتباه است")
    password = request.POST.get("password")
    confirm = request.POST.get("confirm")
    if len(password) < 5 or len(confirm) < 5:
        return generate_error_page(request, "رمز (های) عبور کوتاه هستند، حداقل باید 5 رقم وارد کنید")
    if confirm != password:
        return generate_error_page(request, "password not equal to confirmation")
    username = request.POST.get("username")
    if DjangoUser.objects.filter(username=username).exists():
        return generate_error_page(request, "این نام کاربری قبلا انتخاب شده است")

    gender = str(request.POST.get("gender"))
    if gender == "Female":
        gender = 0
    elif gender == "Male":
        gender = 1
    else:
        gender = None

    image = request.FILES.get("image")
    if not (image == None):
        if os.path.exists(MEDIA_ROOT + '/panel/assistants/name.png'):
            os.remove(MEDIA_ROOT + '/panel/assistants/name.png')
        with open(MEDIA_ROOT + '/panel/assistants/name.png', 'wb+') as destination:
            for chunk in image.chunks():
                destination.write(chunk)
        destination.close()
    try:
        du = DjangoUser(username=username, first_name=first_name, last_name=last_name)
        du.set_password(password)
        du.save()
        gr = Group.objects.get(name='Assistant')
        gr.user_set.add(du)
        assistant = Assistant(manager=manager, doctor=doctor, first_name=first_name, last_name=last_name, gender=gender,
                              django_user=du)
        assistant.save()

        if not (request.FILES.get("image") == None):
            new_image_name = create_image_name(du.id) + ".png"
            new_image_path = MEDIA_ROOT + '/panel/assistants/' + new_image_name
            os.rename(MEDIA_ROOT + '/panel/assistants/name.png',
                      new_image_path)
            assistant.image = 'panel/assistants/' + new_image_name
            assistant.save()

    except Exception as e:
        return generate_error_page(request, 'can not create assistant with this username!: {}'.format(e))

    return redirect('panel-doctor-assistants')


@api_view(["GET"])
def delete_assistant(request, assistant_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Doctor').exists():
        return generate_error_page(request, "you have not access")

    assistant = Assistant.objects.get(id=assistant_id)
    doctor = Doctor.objects.get(django_user=django_user)
    if assistant is not None:
        if assistant.doctor == doctor:
            assistant.django_user.delete()
            assistant.delete()
        else:
            return generate_error_page(request, "you have not access!")
    return redirect('panel-doctor-assistants')


@api_view(["GET"])
def add_food(request):
    food_list = []
    diet_list = []
    eat = Food.objects.all()
    # diet = Diet.objects.all()

    for i in range(len(eat)):
        food_list.append(
            {
                'food': eat[i].FA_Name,
            }
        )
    # for i in range(len(diet)):
    #     diet_list.append(
    #         {
    #             'diets': diet[i].diet_json,
    #         }
    #     )
    context = {
        'food': food_list,
        # 'diets': diet_list,
    }
    return render(request, template_name='doctor/add-food.html', context=context)


@api_view(["GET"])
def show_patient(request, patient_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Doctor').exists():
        return generate_error_page(request, "you have not access")

    temp = AppUser.objects.filter(id=patient_id)
    if not temp.exists():
        return generate_error_page(request, 'invalid request')
    patient = temp[0]
    doctor = Doctor.objects.get(django_user=django_user)
    if patient.profile.therapist != doctor.django_user:
        du = patient.profile.therapist
        temp = Assistant.objects.filter(django_user=du, doctor=doctor)
        if not temp.exists():
            return generate_error_page(request, "you have not access to this patient!")

    drugs = []
    drugs_ids = getIntList(patient.profile.pills)
    for id in drugs_ids:
        temp = Drug.objects.filter(DataBaseNumber=id)
        if temp.exists():
            drugs.append(temp[0])

    # notification = send_notification_note(patient, "khubi")
    # Q & A -------------------------------------------------------------------
    user_questionnaire = UserQuestionnaire.objects.filter(user=patient)
    QandA_list = []
    if user_questionnaire.exists():
        questions_list = []
        answers_list = []
        for questionnaire in user_questionnaire:
            QandA_list.append(questionnaire.QandA)
        for qanda_dict in QandA_list:
            questions = qanda_dict.get('questions', [])
            answers = qanda_dict.get('answers', [])
            questions_list.extend(questions)  # Extend the list with questions
            answers_list.extend(answers)  # Extend the list with answers
    else:
        questions_list = []
        answers_list = []

    # Eating log -------------------------------------------------------------------
    today = date.today()
    y1 = datetime.now() - timedelta(days=1)
    y2 = datetime.now() - timedelta(days=2)
    y3 = datetime.now() - timedelta(days=3)
    y4 = datetime.now() - timedelta(days=4)
    y5 = datetime.now() - timedelta(days=5)
    y6 = datetime.now() - timedelta(days=6)

    t1 = datetime(year=today.year, month=today.month, day=today.day, hour=0, minute=0, second=0)
    t2 = datetime(year=today.year, month=today.month, day=today.day, hour=23, minute=59, second=59)
    d1 = t1.date()
    d2 = t2.date()
    end_date = datetime.now().date()
    start_date = end_date - timedelta(days=6)

    eaten_list = []
    eaten_list_week = []
    total_protein = 0
    total_fat = 0
    total_calorie = 0

    days = (d2 - d1).days + 1
    data = patient.calculate_cr_macro_nutrients_distribution()
    meal = 0
    meal_soboone = True
    meal_nahar = True
    meal_sham = True
    meal_mian_vade = True

    y01 = True
    y11 = True
    y21 = True
    y31 = True
    y41 = True
    y51 = True
    y61 = True

    def getList(strList):
        S = strList.split("&")
        last = len(S) - 1
        F = []
        for i in range(1, last):
            if not S[i] == "":
                F.append(float(S[i]))
        return F

    def listToStr(list):
        S = "&"
        for item in list:
            S += str(item) + "&"
        return S

    Carbohydrates_distribution_str = listToStr(data["carbohydrate_distribution_g"])
    patient.profile.Carbohydrates_distribution_list_g = Carbohydrates_distribution_str
    Carbohydrates_list_to_use = getList(patient.profile.Carbohydrates_distribution_list_g)
    Carbohydrates_list_to_use = list(
        map(lambda x: days * x, getList(patient.profile.Carbohydrates_distribution_list_g)))
    Carbohydrates = sum(Carbohydrates_list_to_use)
    totalcarb = 0

    for i in range(5):
        temp = Eating.objects.filter(user=patient, date_time__range=(t1, t2), meal=i)

        for ii in range(temp.count()):
            totalcarb += round((temp[ii].food.Carbohydrates_g * temp[ii].amount) / 100, 2)
    Protein_to_use = patient.profile.Protein_g * days
    Fat_to_use = patient.profile.Fat_g * days

    eat = Eating.objects.filter(user=patient, date_time__range=(t1, t2))
    eat_foods = Eating.objects.filter(user=patient, date_time__range=(start_date, end_date)).order_by('-date_time')

    for i in range(len(eat)):
        # Calculate the protein, fat, and calorie for each food item
        protein = (eat[i].food.Protein_g * eat[i].amount) / 100
        fat = (eat[i].food.Fat_g * eat[i].amount) / 100
        calorie = (eat[i].food.Calories * eat[i].amount) / 100

        # Add the calculated values to the totals
        total_protein += protein
        total_fat += fat
        total_calorie += calorie

        if eat[i].meal == 0 and meal_soboone:
            meal = 0
            meal_soboone = False
            eaten_list.append({'meal': meal})
        elif eat[i].meal == 2 and meal_nahar:
            meal = 2
            meal_nahar = False
            eaten_list.append({'meal': meal})
        elif eat[i].meal == 4 and meal_sham:
            meal = 4
            meal_sham = False
            eaten_list.append({'meal': meal})
        elif (eat[i].meal == 1 or eat[i].meal == 3 or eat[i].meal == 5) and meal_mian_vade:
            meal = 6
            meal_mian_vade = False
            eaten_list.append({'meal': meal})

        # Append information about each food item to the list
        eaten_list.append(
            {
                'food': eat[i].food,
                'protein': round(protein, 2),
                'fat': round(fat, 2),
                'calorie': round(calorie, 2),
                # 'meal': eat[i].meal
            }
        )
    # protein / fat / carbohydrate Score
    try:
        protein_score = (total_protein * 100) / Protein_to_use
    except Exception as e:
        protein_score = -1
    if total_protein > Protein_to_use:
        protein_score = 100

    try:
        fat_score = (total_fat * 100) / Fat_to_use
    except Exception as e:
        fat_score = -1
    if total_fat > Fat_to_use:
        fat_score = 100

    try:
        carbohydrate_score = (totalcarb * 100) / Carbohydrates
    except Exception as e:
        carbohydrate_score = -1
    if totalcarb > Carbohydrates:
        carbohydrate_score = 100

    # Eaten foods for a week
    for i in range(len(eat_foods)):

        food_item = {}

        if eat_foods[i].date_time.date() == today and y01:
            y01 = False
            food_item['day'] = 0
        elif eat_foods[i].date_time.date() == y1.date() and y11:
            y11 = False
            food_item['day'] = 1
        elif eat_foods[i].date_time.date() == y2.date() and y21:
            y21 = False
            food_item['day'] = 2
        elif eat_foods[i].date_time.date() == y3.date() and y31:
            y31 = False
            food_item['day'] = 3
        elif eat_foods[i].date_time.date() == y4.date() and y41:
            y41 = False
            food_item['day'] = 4
        elif eat_foods[i].date_time.date() == y5.date() and y51:
            y51 = False
            food_item['day'] = 5
        elif eat_foods[i].date_time.date() == y6.date() and y61:
            y61 = False
            food_item['day'] = 6

        if eat_foods[i].meal == 0:
            meal_soboone_week = False
            food_item['meal'] = 0
        elif eat_foods[i].meal == 2:
            meal_nahar_week = False
            food_item['meal'] = 2
        elif eat_foods[i].meal == 4:
            meal_sham_week = False
            food_item['meal'] = 4
        elif eat_foods[i].meal == 1 or eat_foods[i].meal == 3 or eat_foods[i].meal == 5:
            meal_mian_vade_week = False
            food_item['meal'] = 6

        food_item['food'] = eat_foods[i].food
        food_item['time'] = eat_foods[i].date_time

        # Append the food item to the list
        eaten_list_week.append(food_item)

    # Sleep_Duration log -------------------------------------------------------------------
    # Get the date range for the past 7 days
    end_date = datetime.now().date()
    start_date = end_date - timedelta(days=6)  # Calculate 6 days back for a total of 7 days
    h = 0
    low = 0
    sleep_list = []
    sleep = Sleep.objects.filter(user=patient, time__date__range=(start_date, end_date))

    for i in range(len(sleep)):
        sleep_list.append(
            {
                'sleep': sleep[i].sleeped,
            }
        )

        if sleep[i].sleeped != "0.0":
            sleep_hour = int(sleep[i].sleeped.split(":")[0])
            sleep_minute = int(sleep[i].sleeped.split(":")[1])

            sleep_duration_in_minutes = (sleep_hour * 60) + sleep_minute

            if sleep_duration_in_minutes >= (8 * 60):
                h = h + 1
            else:
                low = low + 1

    # diseases_list ---------------------------------------------------------------
    diseases_list = patient.profile.diseases

    # Medical_notes -------------------------------------------------------------------
    medical_notes = MedicalNote.objects.filter(patient=patient).order_by('-time')
    daily_notes = DailyNote.objects.filter(patient=patient).order_by('-time')

    # Steps -------------------------------------------------------------------
    steps_list = []
    steps = Walking.objects.filter(user=patient).order_by("date")

    # max_steps_record = max(steps, key=lambda x: x.steps)
    step_index = 0
    if steps.exists():
        steps_list.append({'steps': steps.last().steps, })
    else:
        steps_list.append({'steps': 0})

    # blood sugar lists
    duration = 100
    fasting_sugar_list = []
    temp = SugarMeasurement.objects.filter(time__range=(datetime.combine((datetime.now() - timedelta(days=duration)).
                                                                         date(), time(0, 0, 0)),
                                                        datetime.now()), meal=0, user=patient)
    for i in range(duration):
        s = temp.filter(time__range=(datetime.combine((datetime.now() - timedelta(days=i)).
                                                      date(), time(0, 0, 0)),
                                     datetime.combine((datetime.now() - timedelta(days=i)).
                                                      date(), time(23, 59, 59))))
        if s.exists():
            fasting_sugar_list.append(s[0].amount)
        else:
            fasting_sugar_list.append("null")

    after_breakfast_sugar_list = []
    temp = SugarMeasurement.objects.filter(time__range=(datetime.combine((datetime.now() - timedelta(days=duration)).
                                                                         date(), time(0, 0, 0)),
                                                        datetime.now()), meal=1, user=patient)
    for i in range(duration):
        s = temp.filter(time__range=(datetime.combine((datetime.now() - timedelta(days=i)).
                                                      date(), time(0, 0, 0)),
                                     datetime.combine((datetime.now() - timedelta(days=i)).
                                                      date(), time(23, 59, 59))))
        if s.exists():
            after_breakfast_sugar_list.append(s[0].amount)
        else:
            after_breakfast_sugar_list.append("null")

    after_lunch_sugar_list = []
    temp = SugarMeasurement.objects.filter(time__range=(datetime.combine((datetime.now() - timedelta(days=duration)).
                                                                         date(), time(0, 0, 0)),
                                                        datetime.now()), meal=3, user=patient)
    for i in range(duration):
        s = temp.filter(time__range=(datetime.combine((datetime.now() - timedelta(days=i)).
                                                      date(), time(0, 0, 0)),
                                     datetime.combine((datetime.now() - timedelta(days=i)).
                                                      date(), time(23, 59, 59))))
        if s.exists():
            after_lunch_sugar_list.append(s[0].amount)
        else:
            after_lunch_sugar_list.append("null")

    after_dinner_sugar_list = []
    temp = SugarMeasurement.objects.filter(time__range=(datetime.combine((datetime.now() - timedelta(days=duration)).
                                                                         date(), time(0, 0, 0)),
                                                        datetime.now()), meal=5, user=patient)
    for i in range(duration):
        s = temp.filter(time__range=(datetime.combine((datetime.now() - timedelta(days=i)).
                                                      date(), time(0, 0, 0)),
                                     datetime.combine((datetime.now() - timedelta(days=i)).
                                                      date(), time(23, 59, 59))))
        if s.exists():
            after_dinner_sugar_list.append(s[0].amount)
        else:
            after_dinner_sugar_list.append("null")

    if patient.profile.insulin_mixed_effect is None:
        if patient.profile.insulin_sensitivity != 0:
            insulin_dosage = round(1800 / patient.profile.insulin_sensitivity)
            insulin_dosage_per_weight_ratio = round(insulin_dosage / patient.profile.weight, 1)
        else:
            insulin_dosage = 'نا معین'
            insulin_dosage_per_weight_ratio = 'نا معین'
    else:
        if patient.profile.insulin_sensitivity != 0:
            insulin_dosage = round(1500 / patient.profile.insulin_sensitivity)
            insulin_dosage_per_weight_ratio = round(insulin_dosage / patient.profile.weight, 1)
        else:
            insulin_dosage = 'نا معین'
            insulin_dosage_per_weight_ratio = 'نا معین'

    short_act_insulin_selections = InsulinTypeSelection.objects.filter(user=patient, insulin__EffectType=1).order_by(
        'start')
    long_act_insulin_selections = InsulinTypeSelection.objects.filter(user=patient, insulin__EffectType=2).order_by(
        'start')

    weights = Weights.objects.filter(user=patient).order_by('time')

    weight = patient.profile.weight
    goal_weight = patient.profile.goal_weight
    weight_score = (goal_weight / weight) * 100

    diets = Diet.objects.filter(user=patient).order_by('-created_at')

    challenges = patient.participated_challenges.all()

    # assistant appointments -------------------------------------------------------------------

    now = datetime.now()
    assistant_times_upcoming = TimeSlot.objects.filter(patient=patient, start_time__gt=now, is_available=0)
    assistant_times_expired = TimeSlot.objects.filter(patient=patient, start_time__lt=now, is_available=0)
    assistant_times_canceled = TimeSlot.objects.filter(patient=patient, is_available=2)

    upcoming_assistant = []
    expired_assistant = []
    canceled_assistant = []

    for times in assistant_times_upcoming:
        item = {
            "id": times.id,
            "days_of_week": times.days_of_week,
            "start_time": times.start_time,
            "duration": times.duration,
            "is_available": times.is_available,
            "therapist": times.therapist_django_user,
            "appointment_link": times.appointment_link,
        }
        upcoming_assistant.append(item)

    for times in assistant_times_expired:
        item = {
            "id": times.id,
            "days_of_week": times.days_of_week,
            "start_time": times.start_time,
            "duration": times.duration,
            "is_available": times.is_available,
            "therapist": times.therapist_django_user,
        }
        expired_assistant.append(item)

    for times in assistant_times_canceled:
        item = {
            "id": times.id,
            "days_of_week": times.days_of_week,
            "start_time": times.start_time,
            "duration": times.duration,
            "is_available": times.is_available,
            "therapist": times.therapist_django_user,
        }
        canceled_assistant.append(item)

    # doctor appointments -------------------------------------------------------------------

    doctor_times_upcoming = TimeSlot.objects.filter(therapist_django_user=django_user, patient=patient,
                                                    start_time__gt=now, is_available=0)
    doctor_times_expired = TimeSlot.objects.filter(therapist_django_user=django_user, patient=patient,
                                                   start_time__lt=now, is_available=0)
    doctor_times_canceled = TimeSlot.objects.filter(therapist_django_user=django_user, patient=patient, is_available=2)

    upcoming = []
    expired = []
    canceled = []

    for times in doctor_times_upcoming:
        item = {
            "id": times.id,
            "days_of_week": times.days_of_week,
            "start_time": times.start_time,
            "duration": times.duration,
            "is_available": times.is_available,
            "appointment_link": times.appointment_link,
        }
        upcoming.append(item)

    for times in doctor_times_expired:
        item = {
            "id": times.id,
            "days_of_week": times.days_of_week,
            "start_time": times.start_time,
            "duration": times.duration,
            "is_available": times.is_available,
        }
        expired.append(item)

    for times in doctor_times_canceled:
        item = {
            "id": times.id,
            "days_of_week": times.days_of_week,
            "start_time": times.start_time,
            "duration": times.duration,
            "is_available": times.is_available,
        }
        canceled.append(item)

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'user': doctor,
        'patient': patient,
        'drugs': drugs,
        'insulin_dosage': insulin_dosage,
        'insulin_dosage_per_weight_ratio': insulin_dosage_per_weight_ratio,
        'medical_notes': medical_notes,
        'daily_notes': daily_notes,
        'age': patient.profile.age,
        'detection_year': jdatetime.datetime.fromgregorian(datetime=patient.profile.birth_date).year,
        'BMI': round(patient.profile.weight / ((patient.profile.height / 100) ** 2), 1),
        'fasting_sugar_list': fasting_sugar_list[::-1],
        'after_breakfast_sugar_list': after_breakfast_sugar_list[::-1],
        'after_lunch_sugar_list': after_lunch_sugar_list[::-1],
        'after_dinner_sugar_list': after_dinner_sugar_list[::-1],
        'short_act_insulin_selections': short_act_insulin_selections,
        'long_act_insulin_selections': long_act_insulin_selections,
        'weights': weights,
        'weight': round(weight, 2),
        'goal_weight': goal_weight,
        'weight_score': round(weight_score),
        'sleep_high': h,
        'sleep_low': low,
        'steps': steps_list,
        "qanda": zip(questions_list, answers_list),
        'eating': eaten_list,
        'eating_week': eaten_list_week,
        'total_protein': round(total_protein, 2),
        'Protein_to_use': round(Protein_to_use, 2),
        'protein_score': round(protein_score),
        'total_fat': round(total_fat, 2),
        'Fat_to_use': round(Fat_to_use, 2),
        'fat_score': round(fat_score),
        'total_calorie': round(total_calorie, 2),
        'Carbohydrates_to_use': round(Carbohydrates, 2),
        'total_carbohydrate': round(totalcarb, 2),
        'carbohydrate_score': round(carbohydrate_score),
        'diseases_list': diseases_list,
        'diets': diets,
        'challenges': challenges,
        'upcoming_assistant': upcoming_assistant,
        'expired_assistant': expired_assistant,
        'canceled_assistant': canceled_assistant,
        'upcoming': upcoming,
        'expired': expired,
        'canceled': canceled,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }
    return render(request, template_name='doctor/patient_profile.html', context=context)


@api_view(["POST"])
def add_note(request, patient_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Doctor').exists():
        return generate_error_page(request, "you have not access")

    temp = AppUser.objects.filter(id=patient_id)
    if not temp.exists():
        return generate_error_page(request, 'invalid request')
    patient = temp[0]
    doctor = Doctor.objects.get(django_user=django_user)
    if patient.profile.therapist != doctor.django_user:
        du = patient.profile.therapist
        temp = Assistant.objects.filter(django_user=du, doctor=doctor)
        if not temp.exists():
            return generate_error_page(request, "you have not access to this patient!")

    text = request.POST.get("text")
    text_with_br_tags = text.replace('\n', '<br>')
    if text is not None:
        note = MedicalNote(user=django_user, patient=patient, text=text_with_br_tags, writer_name=str(doctor))
        note.save()

    return redirect('panel-doctor-show-patient', patient_id=patient.id)


@api_view(["POST"])
def edit_note(request, note_id):
    # Retrieve the note object or return a 404 error if it doesn't exist
    note = get_object_or_404(MedicalNote, id=note_id)

    if request.method == 'POST':
        # Get the new text from the form data
        new_text = request.POST.get('editnote')

        # Update the note text if new text is provided
        if new_text:
            new_text = new_text.replace('<br>', '')
            text_with_br_tags = new_text.replace('\n', '<br>')
            note.text = text_with_br_tags
            note.save()

        # Redirect back to the patient's page after editing the note
        return redirect('panel-doctor-show-patient', patient_id=note.patient.id)

    # If the request method is not POST, redirect back to the patient's page without making any changes
    return redirect('panel-doctor-show-patient', patient_id=note.patient.id)


def delete_note(request, note_id):
    note = get_object_or_404(MedicalNote, id=note_id)
    patient_id = note.patient.id
    note.delete()
    return redirect('panel-doctor-show-patient', patient_id=patient_id)


@api_view(["GET"])
def chat_with(request, patient_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Doctor').exists():
        return generate_error_page(request, "you have not access")

    temp = AppUser.objects.filter(id=patient_id)
    if not temp.exists():
        return generate_error_page(request, 'invalid request')
    patient = temp[0]
    doctor = Doctor.objects.get(django_user=django_user)
    now = datetime.now()
    upcoming = []
    expired = []

    if patient.profile.therapist != doctor.django_user:
        du = patient.profile.therapist
        temp = Assistant.objects.filter(django_user=du, doctor=doctor)
        if not temp.exists():
            return generate_error_page(request, "you have not access to this patient!")

    chats = Message.objects.filter(Q(sender=patient.django_user) | Q(receiver=patient.django_user)).order_by("time")
    for chat in chats:
        chat.text = chat.text.replace('\n', '<br>')
        chat.text = mark_safe(chat.text)

    doctor_times_upcoming = TimeSlot.objects.filter(therapist_django_user=django_user, patient=patient,
                                                    start_time__gt=now, is_available=0)
    doctor_times_expired = TimeSlot.objects.filter(therapist_django_user=django_user, patient=patient,
                                                   start_time__lt=now, is_available=0)

    for times in doctor_times_upcoming:
        item = {
            "id": times.id,
            "days_of_week": times.days_of_week,
            "start_time": times.start_time,
            "duration": times.duration,
            "is_available": times.is_available,
            "appointment_link": times.appointment_link,
        }
        upcoming.append(item)

    for times in doctor_times_expired:
        item = {
            "id": times.id,
            "days_of_week": times.days_of_week,
            "start_time": times.start_time,
            "duration": times.duration,
            "is_available": times.is_available,
        }
        expired.append(item)

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'user': doctor,
        'patient': patient,
        'chats': chats,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
        'upcoming': upcoming,
        'expired': expired,
    }

    for chat in chats:
        if chat.receiver == django_user:
            chat.has_been_seen = True
            chat.save()

    return render(request, template_name='doctor/chat.html', context=context)


@api_view(["POST"])
def send_message(request, patient_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Doctor').exists():
        return generate_error_page(request, "you have not access")

    temp = AppUser.objects.filter(id=patient_id)
    if not temp.exists():
        return generate_error_page(request, 'invalid request')
    patient = temp[0]
    doctor = Doctor.objects.get(django_user=django_user)
    if patient.profile.therapist != doctor.django_user:
        du = patient.profile.therapist
        temp = Assistant.objects.filter(django_user=du, doctor=doctor)
        if not temp.exists():
            return generate_error_page(request, "you have not access to this patient!")

    replied_message_id = request.POST.get("message_id")

    if replied_message_id is None:
        text = request.POST.get("text")
        if text is not None:
            message = Message(sender=django_user, receiver=patient.django_user, text=text)
            message.save()
            # SMS().notify_user(user=patient)

            # ***********************************************************
            send_notification_note(patient, text)
            # ***********************************************************

            return myResponse.OK("sent", [])

        return myResponse.Error("error on saving the message", [])

    else:
        text_reply = request.POST.get("reply_text")
        past_message = get_object_or_404(Message, id=replied_message_id)
        if text_reply is not None:
            # Creating a reply message directly in the same Message model
            message = Message(sender=django_user,
                              receiver=patient.django_user,
                              text=text_reply,
                              reply_message=past_message  # Assigning the replied message
                              )
            message.save()

            # SMS().notify_user(user=patient)

            # ***********************************************************
            send_notification_note(patient, text_reply)
            # ***********************************************************

            return myResponse.OK("sent", [])

        return myResponse.Error("error on saving the message", [])


@api_view(["POST"])
def delete_message(request):
    if not request.user.is_authenticated:
        return JsonResponse({"error": "User not authenticated"}, status=401)

    if request.method == "POST":
        message_id = request.POST.get("message_id")
        message = get_object_or_404(Message, id=message_id)

        # Check if the user has permission to delete the message
        if request.user == message.sender or request.user == message.receiver:
            # Delete the message
            message.delete()
            # print("delete message")
            return JsonResponse({"message": "Message deleted successfully"})
        else:
            return JsonResponse({"error": "You do not have permission to delete this message"}, status=403)

    return JsonResponse({"error": "Invalid request method"}, status=405)


@api_view(["POST"])
def edit_message(request):
    if not request.user.is_authenticated:
        return JsonResponse({"error": "User not authenticated"}, status=401)

    if request.method == "POST":
        message_id = request.POST.get("message_id")
        edited_text = request.POST.get("edited_text")

        message = get_object_or_404(Message, id=message_id)

        # Check if the user has permission to edit the message
        if request.user == message.sender or request.user == message.receiver:
            # Update the message text
            message.text = edited_text
            message.save()
            # print("Edit message")
            return JsonResponse({"message": "Message edited successfully"})
        else:
            return JsonResponse({"error": "You do not have permission to edit this message"}, status=403)

    return JsonResponse({"error": "Invalid request method"}, status=405)


@api_view(["POST"])
def send_file(request, patient_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Doctor').exists():
        return generate_error_page(request, "you have not access")

    temp = AppUser.objects.filter(id=patient_id)
    if not temp.exists():
        return generate_error_page(request, 'invalid request')
    patient = temp[0]
    doctor = Doctor.objects.get(django_user=django_user)
    if patient.profile.therapist != doctor.django_user:
        du = patient.profile.therapist
        temp = Assistant.objects.filter(django_user=du, doctor=doctor)
        if not temp.exists():
            return generate_error_page(request, "you have not access to this patient!")

    file = request.FILES["file"]
    extension = file.name.split(".")[-1]
    if extension == "blob":
        extension = 'mp3'
    hashed_name = generate_file_name(extension)
    path_from_static_root = "panel/attachments/" + hashed_name
    try:
        with open(STATIC_ROOT + "/" + path_from_static_root, 'wb+') as destination:
            for chunk in file.chunks():
                destination.write(chunk)
            destination.close()

        file_size = os.path.getsize(STATIC_ROOT + "/" + path_from_static_root)
        is_diet = str(request.POST.get("is_diet"))
        if is_diet == "0" or is_diet == "None":
            is_diet = False
        else:
            is_diet = True
        attachment = Attachment(owner=django_user, name=file.name, path=path_from_static_root,
                                size=file_size, content_type=extension, is_diet=is_diet)
        attachment.save()
        message = Message(sender=django_user, receiver=patient.django_user, text="", attachment=attachment)
        message.save()
        data = {
            "url_uploaded": "/static/" + path_from_static_root,
            "is_diet": is_diet,
            "content_type": extension,
            "name": file.name,
            "time": "%s:%s" % (message.time.hour, message.time.minute),
            'size': attachment.get_size(),  # changing byte to KB
        }
        SMS().notify_user(user=patient)
        return myResponse.OK("sent", data)
    except Exception as e:
        return myResponse.Error("error on saving the message")


@api_view(["POST"])
def check_new_message_from(request, patient_id):
    try:
        if not request.user.is_authenticated:
            return redirect('panel-login')
        django_user = request.user
        if not django_user.groups.filter(name='Doctor').exists():
            return generate_error_page(request, "you have not access")

        temp = AppUser.objects.filter(id=patient_id)
        if not temp.exists():
            return generate_error_page(request, 'invalid request')

        patient = temp[0]
        doctor = Doctor.objects.get(django_user=django_user)

        if patient.profile.therapist != doctor.django_user:
            du = patient.profile.therapist
            temp = Assistant.objects.filter(django_user=du, doctor=doctor)
            if not temp.exists():
                return generate_error_page(request, "you have not access to this patient!")

        temp = Message.objects.filter(Q(sender=patient.django_user) & Q(has_been_seen=False)).order_by("time")
        chats_list = []
        for chat in temp:
            item = {
                'time': str(chat.time.hour) + ":" + str(chat.time.minute),
                'text': chat.text,
                'attachment': None,
                'sender': 'other',
            }
            if chat.sender == doctor:
                item['sender'] = 'you'
            else:
                item['sender'] = 'other'
                chat.has_been_seen = True
                chat.save()

            if chat.attachment is not None:
                item['attachment'] = {
                    'name': chat.attachment.name,
                    'time': int(chat.attachment.time.timestamp()),
                    'content_type': chat.attachment.content_type,
                    'path': 'https://i-dia.ir/static/' + chat.attachment.path,
                    'size': chat.attachment.get_size(),
                    'owner': chat.attachment.owner.id,
                }

            chats_list.append(item)
        data = {
            'new_chats': chats_list,
            'number_of_new_messages': len(chats_list)
        }
        return myResponse.OK('new messages', data=data)
    except Exception as e:
        return myResponse.Error(str(e), e.args)


@api_view(["POST"])
def send_message_to_all_patients(request):
    if not request.user.is_authenticated:
        return HttpResponse("login")
    message = request.POST.get('message')
    message = message.replace('"', '')
    message = message.replace('\\n', '\n')
    doctor = Doctor.objects.filter(django_user=request.user)[0]
    if PushNotification().send_message_to_all_patients(message, doctor):
        return HttpResponse("ok")
    else:
        return HttpResponse("no")


@api_view(["POST"])
def send_notification_to_all_patients(request):
    if not request.user.is_authenticated:
        return HttpResponse("login")
    message = request.POST.get('message')
    message = message.replace('"', '')
    message = message.replace('\\n', '\n')
    doctor = Doctor.objects.filter(django_user=request.user)[0]
    if PushNotification().send_notification_to_all_patients(message, doctor):
        return HttpResponse("ok")
    else:
        return HttpResponse("no")


@api_view(["POST"])
def add_diet_with_excel(request, patient_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    # # todo update django user
    # django_user = Doctor.objects.all()[0].django_user

    if not django_user.groups.filter(name='Doctor').exists():
        return generate_error_page(request, "you have not access")

    temp = AppUser.objects.filter(id=patient_id)
    if not temp.exists():
        return generate_error_page(request, 'invalid request')
    patient = temp[0]
    doctor = Doctor.objects.get(django_user=django_user)
    if patient.profile.therapist != doctor.django_user:
        du = patient.profile.therapist
        temp = Assistant.objects.filter(django_user=du, doctor=doctor)
        if not temp.exists():
            return generate_error_page(request, "you have not access to this patient!")

    file = request.FILES["file"]
    extension = file.name.split(".")[-1]
    hashed_name = generate_file_name(extension)
    path_from_static_root = "panel/diets/" + hashed_name
    try:
        diet_excel_path = STATIC_ROOT + "/" + path_from_static_root
        with open(diet_excel_path, 'wb+') as destination:
            for chunk in file.chunks():
                destination.write(chunk)
            destination.close()

        diet_json = convert_excel_to_diet(diet_excel_path)
        diet = Diet(diet_json=diet_json,
                    user=patient,
                    creator=django_user,
                    )
        diet.adjust_starting_date()
        diet.save()
        data = {
            'df': diet_json
        }
        # SMS().notify_user(user=patient)
        return myResponse.OK("Diet was successfully registered", data)
    except FileExistsError as e:
        return myResponse.Error("error on saving the diet: {}".format(e), 1001)


@api_view(["POST"])
def add_daily_note(request, patient_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Doctor').exists():
        return generate_error_page(request, "you have not access")

    temp = AppUser.objects.filter(id=patient_id)
    if not temp.exists():
        return generate_error_page(request, 'invalid request')
    patient = temp[0]
    doctor = Doctor.objects.get(django_user=django_user)
    if patient.profile.therapist != doctor.django_user:
        return generate_error_page(request, "you have not access to this patient!")
    try:
        note = request.POST.get("note")
        daily_note = DailyNote(user=django_user, patient=patient,
                               text=note, writer_name=str(doctor))
        daily_note.save()
        send_notification_note(patient, note)
        return redirect('panel-doctor-show-patient', patient_id=patient.id)

    except Exception as e:
        return myResponse.Error("error on saving the daily note: {}".format(e), 1001)


@api_view(["GET"])
def challenge(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    user = request.user
    if not user.groups.filter(name='Doctor').exists():
        return generate_error_page(request, "you have not access")
    doctor = Doctor.objects.get(django_user=user)
    temp = AppUser.objects.all()
    if not temp.exists():
        return generate_error_page(request, 'invalid request')
    challenges = Challenge.objects.all()
    users = []
    for c in challenges:
        participants_data = []
        for user in c.participants.all():
            profile = user  # Assuming profile is a OneToOneField related to AppUser
            participants_data.append(profile)
        users.append(participants_data)

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'user': doctor,
        'challenges': zip(challenges, users),
        "participants_data": users,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }
    return render(request, template_name='doctor/challenge.html', context=context)


@api_view(["POST"])
def add_challenge(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Doctor').exists():
        return generate_error_page(request, "you have not access")

    try:
        name = request.POST.get("name")
        reward = int(request.POST.get("reward"))
        fee = int(request.POST.get("fee")) if request.POST.get("fee") else 0
        start_deadline = datetime.strptime(request.POST.get("start_deadline"), '%Y-%m-%dT%H:%M') if request.POST.get(
            "start_deadline") else None
        end_deadline = datetime.strptime(request.POST.get("end_deadline"), '%Y-%m-%dT%H:%M') if request.POST.get(
            "end_deadline") else None
        description = request.POST.get("description")

        category_mapping = {
            "Food": 0,
            "BS": 1,
            "Exercise": 2,
            "BP": 3,
            "Sleep": 4,
            "Water": 5,
        }

        category_str = request.POST.get("category")
        category = category_mapping.get(category_str, -1)  # Default to -1 if not found

        sub_category_mapping = {
            "مدت زمان خواب": 0, "میکروبیوم": 0, "بیشتر از": 0, "قدم": 0, "تعداد لیوان": 0,
            "ساعت شروع خواب": 1, "چربی": 1, "تعداد ثبت": 1, "سوزاندن کالری": 1,
            "ساعت پایان خواب": 2, "کالری": 2, "مدت زمان": 2,
            "ساعت شروع خواب و مدت زمان خواب": 3, "پروتئین": 3,
            "کربوهیدرات": 4,
            "انتخاب کارت": 5,
        }

        sub_category_str = request.POST.get("sub_category")
        sub_category = sub_category_mapping.get(sub_category_str, -1)
        # sub = Challenge.objects.all()
        # sub2 = len(sub)

        if category == 0:  # Food
            if sub_category == 0:  # microbiome
                microbiome = request.POST.get("microbiome")
                challenges = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                       start_deadline=start_deadline, end_deadline=end_deadline,
                                       description=description,
                                       category=category, sub_category=sub_category, microbiome=microbiome)
                challenges.save()
            elif sub_category == 1:  # fat
                fat = request.POST.get("fat")
                challenges = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                       start_deadline=start_deadline, end_deadline=end_deadline,
                                       description=description,
                                       category=category, sub_category=sub_category, fat=fat)
                challenges.save()
            elif sub_category == 2:  # calorie
                calorie = request.POST.get("calorie")
                challenges = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                       start_deadline=start_deadline, end_deadline=end_deadline,
                                       description=description,
                                       category=category, sub_category=sub_category, calorie=calorie)
                challenges.save()
            elif sub_category == 3:  # protein
                protein = request.POST.get("protein")
                challenges = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                       start_deadline=start_deadline, end_deadline=end_deadline,
                                       description=description,
                                       category=category, sub_category=sub_category, protein=protein)
                challenges.save()
            elif sub_category == 4:  # carbohydrate
                carbohydrate = request.POST.get("carbohydrate")
                challenges = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                       start_deadline=start_deadline, end_deadline=end_deadline,
                                       description=description,
                                       category=category, sub_category=sub_category, carbohydrate=carbohydrate)
                challenges.save()
            else:  # cards
                challenges = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                       start_deadline=start_deadline, end_deadline=end_deadline,
                                       description=description,
                                       category=category, sub_category=sub_category)
                challenges.save()
        elif category == 1:  # BS
            if sub_category == 0:  # MoreThan-BS
                moreThan = request.POST.get("moreThan")
                challenges = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                       start_deadline=start_deadline, end_deadline=end_deadline,
                                       description=description,
                                       category=category, sub_category=sub_category, more_than=moreThan)
                challenges.save()
            elif sub_category == 1:  # TedadSabt-BS
                tedadSabt = request.POST.get("tedadSabt")
                challenges = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                       start_deadline=start_deadline, end_deadline=end_deadline,
                                       description=description,
                                       category=category, sub_category=sub_category, tedad_sabt=tedadSabt)
                challenges.save()
        elif category == 2:  # Exercise
            if sub_category == 0:  # Steps
                steps = request.POST.get("steps")
                challenges = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                       start_deadline=start_deadline, end_deadline=end_deadline,
                                       description=description,
                                       category=category, sub_category=sub_category, steps=steps)
                challenges.save()
            elif sub_category == 1:  # calorie_burnt
                calorieBurnt = request.POST.get("calorieBurnt")
                challenges = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                       start_deadline=start_deadline, end_deadline=end_deadline,
                                       description=description,
                                       category=category, sub_category=sub_category, calorie_burnt=calorieBurnt)
                challenges.save()
            elif sub_category == 2:  # exercise_duration
                durationExercise = request.POST.get("durationExercise")
                challenges = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                       start_deadline=start_deadline, end_deadline=end_deadline,
                                       description=description,
                                       category=category, sub_category=sub_category, exercise_duration=durationExercise)
                challenges.save()
        elif category == 3:  # BP
            if sub_category == 0:  # MoreThan-BP
                moreThan = request.POST.get("moreThan")
                challenges = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                       start_deadline=start_deadline, end_deadline=end_deadline,
                                       description=description,
                                       category=category, sub_category=sub_category, more_than=moreThan)
                challenges.save()
            elif sub_category == 1:  # TedadSabt-BP
                tedadSabt = request.POST.get("tedadSabt")
                challenges = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                       start_deadline=start_deadline, end_deadline=end_deadline,
                                       description=description,
                                       category=category, sub_category=sub_category, tedad_sabt=tedadSabt)
                challenges.save()
        elif category == 4:  # Sleep
            if sub_category == 0:  # Sleep-Duration
                durationField = request.POST.get("duration")
                challenges = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                       start_deadline=start_deadline, end_deadline=end_deadline,
                                       description=description,
                                       category=category, sub_category=sub_category, sleep_duration=durationField)
                challenges.save()
            elif sub_category == 1:  # sleep_start_hour
                startHourField = request.POST.get("startHour")
                challenges = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                       start_deadline=start_deadline, end_deadline=end_deadline,
                                       description=description,
                                       category=category, sub_category=sub_category, sleep_start_hour=startHourField)
                challenges.save()
            elif sub_category == 2:  # sleep_end_hour
                endHourField = request.POST.get("endHour")
                challenges = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                       start_deadline=start_deadline, end_deadline=end_deadline,
                                       description=description,
                                       category=category, sub_category=sub_category, sleep_end_hour=endHourField)
                challenges.save()
            elif sub_category == 3:  # sleep_start_hour & Duration
                durationField = request.POST.get("duration")
                startHourField = request.POST.get("startHour")
                challenges = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                       start_deadline=start_deadline, end_deadline=end_deadline,
                                       description=description,
                                       category=category, sub_category=sub_category, sleep_duration=durationField,
                                       sleep_start_hour=startHourField)
                challenges.save()
        elif category == 5:  # Water
            water = request.POST.get("tedadSabt")
            challenges = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                   start_deadline=start_deadline, end_deadline=end_deadline,
                                   description=description,
                                   category=category, sub_category=sub_category, tedad_sabt=water)
            challenges.save()

        return redirect('panel-doctor-challenge')

    except Exception as e:
        return myResponse.Error("error on saving the challenge: {}".format(e), 1001)


def delete_challenge(request, challenge_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    challenge = get_object_or_404(Challenge, id=challenge_id)
    challenge.delete()
    return redirect('panel-doctor-challenge')


@api_view(["GET"])
def appointments(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    user = request.user
    patients_list = AppUser.objects.filter(profile__therapist=user)
    doctor = Doctor.objects.get(django_user=user)
    free_times = TimeSlot.objects.filter(therapist_django_user=user)
    # TimeSlot.objects.all().delete()

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'user': doctor,
        'free_times': free_times,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }

    return render(request, template_name='doctor/appointments.html', context=context)


@api_view(['POST'])
def submit_free_time(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Doctor').exists():
        return generate_error_page(request, "you have not access")

    try:
        selected_ids = request.POST.getlist("selectedIds")  # Get the selectedIds list from POST data
        day_name = selected_ids[0]  # Assuming the first element in the list is the day name
        split = day_name.strip('[]').split(',')
        day_names = int(day_name[2])
        if day_names == 0:
            day_names_def = day_names + 5
        elif day_names == 1:
            day_names_def = day_names + 5
        else:
            day_names_def = day_names - 2

        days_times = [slot.strip('"') for slot in split]
        start_times = [slot.split('-')[0] for slot in days_times]
        modified_list = [item[1:] for item in start_times]
        time_objects = [datetime.strptime(time_str, '%H%M').time() for time_str in modified_list]
        # TimeSlot.objects.all().delete()
        dup = False
        doctor_free_times = TimeSlot.objects.filter(therapist_django_user=django_user)
        for i in range(len(days_times)):
            for times in doctor_free_times:
                if convert_to_datetime(day_names_def, modified_list[i]) == times.start_time:
                    dup = True
                    break
            if dup:
                break
            free_time = TimeSlot(therapist_django_user=django_user,
                                 start_time=convert_to_datetime(day_names_def, modified_list[i]),
                                 days_of_week=day_names)
            free_time.save()

        return redirect('panel-doctor-appointments')
    # Delete existing free time slots for the doctor

    except Exception as e:
        return myResponse.Error("error on saving the free time: {}".format(e), 1001)


def convert_to_datetime(weekday, time_str):
    # Get today's date
    today = datetime.now().date()

    # Calculate the number of days to add to reach the desired weekday
    days_to_add = (weekday - today.weekday() + 7) % 7

    # Parse the time string to get the hours and minutes
    hours = int(time_str[:2])
    minutes = int(time_str[2:])

    # Create a timedelta object for the time
    time_delta = timedelta(hours=hours, minutes=minutes)

    # Calculate the target date by adding the days_to_add to today's date
    target_date = today + timedelta(days=days_to_add)

    # Combine the target date and time to create the datetime object
    target_datetime = datetime.combine(target_date, datetime.min.time()) + time_delta

    return target_datetime


@api_view(['POST'])
def add_meeting_link(request, session_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Doctor').exists():
        return generate_error_page(request, "you have not access")
    try:
        session = get_object_or_404(TimeSlot, id=session_id)
        if request.method == 'POST':
            meeting_link = request.POST.get("appointment_link")
            if meeting_link:
                session.appointment_link = meeting_link
                session.save()

                # Send notification to the user using the same system as message sending
                try:
                    from panel.PushNotification import PushNotification
                    from panel.models import FirebaseNotification
                    
                    # Get the patient from the session
                    patient = session.patient
                    if patient:
                        # Prepare notification message
                        session_time = session.start_time.strftime("%Y/%m/%d ساعت %H:%M")
                        doctor_name = f"{django_user.first_name} {django_user.last_name}"
                        
                        notification_title = "لینک تماس تنظیم شد"
                        notification_body = f"لینک تماس برای جلسه شما در تاریخ {session_time} توسط {doctor_name} تنظیم شد."
                        
                        # Send notification using the same system as message sending
                        push_notification = PushNotification(patient)
                        notification_sent = push_notification.send_notification_to_user(
                            patient, 
                            notification_title, 
                            notification_body
                        )
                        
                        if notification_sent:
                            print("Notification sent successfully to patient")
                        else:
                            print("Notification failed to send to patient")
                    
                except Exception as e:
                    print(f"Debug: Notification system failed: {e}")

            return redirect('panel-doctor-show-patient')

    except Exception as e:
        return myResponse.Error("error on saving the meeting link: {}".format(e), 1001)


def delete_appointment(request, appointment_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    appointment = get_object_or_404(TimeSlot, id=appointment_id)
    if appointment.is_available == 1:
        appointment.delete()
    return redirect('panel-doctor-appointments')


@api_view(["GET"])
def delete_diet(request, diet_id):
    try:
        if not request.user.is_authenticated:
            return redirect('panel-login')
        django_user = request.user
        if not django_user.groups.filter(name='Doctor').exists():
            return generate_error_page(request, "you have not access")

        temp = Diet.objects.filter(id=diet_id)
        if not temp.exists():
            return generate_error_page(request, 'invalid request')
        diet = temp[0]
        patient = diet.user

        doctor = Doctor.objects.get(django_user=django_user)
        if diet.creator != doctor.django_user:
            return generate_error_page(request, "you have not access to this patient!")
        diet.delete()
        return redirect('/panel/doctor/show/patient/%s/' % patient.id)
    except Exception as e:
        return generate_error_page(request, e.args[0])


@api_view(["GET"])
def shop(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    user = request.user
    patients_list = AppUser.objects.filter(profile__therapist=user)
    doctor = Doctor.objects.get(django_user=user)
    items = Shop.objects.all()

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'user': doctor,
        'items': items,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }

    return render(request, template_name='doctor/shop.html', context=context)


@api_view(["POST"])
def add_item(request):
    django_user = request.user
    if not django_user.groups.filter(name='Doctor').exists():
        return generate_error_page(request, "You do not have access.")

    try:
        name = request.POST.get("name")
        if not name:
            return Response({"error": "Name is required."}, status=400)

        tag = request.POST.get("tag")
        if not tag or ' ' in tag or Shop.objects.filter(tag=tag).exists():
            return Response({"error": "Invalid or duplicate tag."}, status=400)

        image = request.FILES.get("image")
        price = int(request.POST.get("price", 0))
        if price < 0:
            return Response({"error": "Price must be non-negative."}, status=400)

        is_coin = request.POST.get("is_coin") in ['1', 'کوین', 'داخلی']
        service_type = request.POST.get('service_type')
        if service_type not in ['coaching', 'content', 'plan']:
            return Response({"error": "Invalid service type."}, status=400)

        description = request.POST.get('description', '')  # For plans, this includes services
        promotion_percentage = int(request.POST.get('promotion_percentage', 0))
        promotion_end_date_str = request.POST.get('promotion_end_date')
        promotion_end_date = datetime.strptime(promotion_end_date_str,
                                               '%Y-%m-%dT%H:%M') if promotion_end_date_str else None

        # Plan-specific fields
        if service_type == 'plan':
            duration_months = int(request.POST.get('duration_months', 0))
            plan_type = int(request.POST.get('plan_type', 0))
            if duration_months <= 0 or plan_type not in [1, 2, 3]:
                return Response({"error": "Invalid duration or plan type."}, status=400)
        else:
            duration_months = 0
            plan_type = None

        shop_item = Shop(
            name=name, tag=tag, image=image, price=price, is_coin=is_coin,
            description=description, service_type=service_type,
            promotion_percentage=promotion_percentage, promotion_end_date=promotion_end_date,
            duration_months=duration_months, plan_type=plan_type
        )
        shop_item.save()

        # Handle content files (unchanged)
        if service_type == 'content':
            content_service = ContentService.objects.create(shop_item=shop_item)
            for pdf_file in request.FILES.getlist('pdf_files'):
                pdf = PDF.objects.create(file_path=pdf_file)
                content_service.pdf_files.add(pdf)
            for video_file in request.FILES.getlist('videos'):
                video = Video.objects.create(file_path=video_file)
                content_service.videos.add(video)

        return redirect('panel-doctor-shop')

    except Exception as e:
        return Response({"error": str(e)}, status=400)


def delete_item(request, item_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    item = get_object_or_404(Shop, id=item_id)
    item.delete()
    return redirect('panel-doctor-shop')


@api_view(["GET"])
def get_existing_tags(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    tags = Shop.objects.values_list('tag', flat=True)
    return JsonResponse({"tags": list(tags)})


@api_view(["GET"])
def analyze(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    user = request.user
    patients_list = AppUser.objects.filter(profile__therapist=user)
    doctor = Doctor.objects.get(django_user=user)
    assistant = Assistant.objects.filter(doctor=doctor)
    # TimeSlot.objects.all().delete()

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    now = datetime.now()
    today = datetime.now().date()
    assistant_times_upcoming = TimeSlot.objects.filter(start_time__gt=now, is_available=0)
    upcoming = []
    for times in assistant_times_upcoming:
        item = {
            "id": times.id,
            "days_of_week": times.days_of_week,
            "start_time": times.start_time,
            "duration": times.duration,
            "is_available": times.is_available,
            "therapist": times.therapist_django_user,
            "patient": times.patient,
            "appointment_link": times.appointment_link,
        }
        upcoming.append(item)

    three_days_ago = (datetime.now() - timedelta(days=3)).date()
    new_sub = []
    patients_list = AppUser.objects.all()
    patients_list = list(patients_list)
    for p in patients_list:
        app_user = p
        django_user = app_user.django_user
        temp2 = TherapistSelection.objects.filter(user=django_user)
        try:
            if temp2.last().created_at.date() >= three_days_ago:
                new_sub.append(True)
            else:
                new_sub.append(False)
        except Exception as e:
            new_sub.append(False)

    context = {
        'user': doctor,
        'assistant': assistant,
        'number_of_assistant': assistant.count(),
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
        'upcoming': upcoming,
        'patients_list': zip(patients_list, new_sub),
    }

    return render(request, template_name='doctor/analyze.html', context=context)


@api_view(["GET"])
def get_assistant_activity(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    today = timezone.now().date()
    seven_days_ago = today - timedelta(days=7)
    assistants = Assistant.objects.all()
    activity_data = {}

    # Initialize the data structure for the past 7 days
    date_labels = [seven_days_ago + timedelta(days=i) for i in range(8)]

    for assistant in assistants:
        name = f"{assistant.first_name} {assistant.last_name}"
        activity_logs = AssistantActivityLog.objects.filter(
            assistant=assistant,
            timestamp__date__gte=seven_days_ago
        )

        # Initialize daily counts
        daily_counts = {date: 0 for date in date_labels}

        for log in activity_logs:
            log_date = log.timestamp.date()
            if log_date in daily_counts:
                daily_counts[log_date] += 1

        # Prepare data for JSON response
        activity_data[name] = [{'date': date.strftime('%Y-%m-%d'), 'viewChatPage': daily_counts[date]} for date in
                               date_labels]

    return JsonResponse(activity_data, safe=False)


@api_view(["GET"])
def get_assistant_weight_changes(request):
    try:
        if not request.user.is_authenticated:
            logger.error("User not authenticated")
            return JsonResponse({"error": "User not authenticated"}, status=401)

        doctor = Doctor.objects.get(django_user=request.user)
        assistants = Assistant.objects.filter(doctor=doctor)
        weight_changes = {}

        for assistant in assistants:
            name = f"{assistant.first_name} {assistant.last_name}"
            profiles = Profile.objects.filter(therapist=assistant.django_user)
            gain_total = 0
            gain_count = 0
            loss_total = 0
            loss_count = 0

            for profile in profiles:
                app_user = User.objects.filter(profile=profile).first()
                if not app_user:
                    continue

                initial_weight = profile.weight
                goal_weight = profile.goal_weight
                latest_weight_entry = Weights.objects.filter(user=app_user).order_by('-time').first()

                if not latest_weight_entry:
                    continue

                latest_weight = latest_weight_entry.weight
                weight_change = latest_weight - initial_weight  # Raw change, not distance from goal

                if goal_weight > initial_weight:  # Gain goal
                    gain_total += weight_change
                    gain_count += 1
                elif goal_weight < initial_weight:  # Loss goal
                    loss_total += weight_change
                    loss_count += 1

            avg_gain = round(gain_total / gain_count, 2) if gain_count > 0 else 0
            avg_loss = round(loss_total / loss_count, 2) if loss_count > 0 else 0
            weight_changes[name] = {"gain": avg_gain, "loss": avg_loss}

        return JsonResponse(weight_changes)
    except Doctor.DoesNotExist:
        logger.error(f"No Doctor found for user {request.user.username}")
        return JsonResponse({"error": "Doctor not found for this user"}, status=404)
    except Exception as e:
        logger.error(f"Unexpected error: {str(e)}")
        return JsonResponse({"error": str(e)}, status=500)


@api_view(["GET"])
def get_assistants_points_data(request):
    if not request.user.is_authenticated:
        return JsonResponse({'error': 'User not authenticated'}, status=401)

    user = request.user
    doctor = Doctor.objects.get(django_user=user)
    assistants = Assistant.objects.filter(doctor=doctor)

    end_date = datetime.now().date()
    start_date = end_date - timedelta(days=30)

    assistants_data = []

    for assistant in assistants:
        patients_list = AppUser.objects.filter(profile__therapist=assistant.django_user)

        daily_points = defaultdict(int)
        for patient in patients_list:
            points = Points.objects.filter(user=patient, date__range=[start_date, end_date])
            for point in points:
                daily_points[point.date] += round(point.total_points)

        # Convert daily_points to a list of points sorted by date for the past 30 days
        points_last_30_days = []
        for i in range(30):
            day = start_date + timedelta(days=i)
            points_last_30_days.append({
                'date': day.strftime('%Y-%m-%d'),
                'points': daily_points[day]
            })

        assistants_data.append({
            'name': f"{assistant.first_name} {assistant.last_name}",
            'points_last_30_days': points_last_30_days
        })

    return JsonResponse({'assistants_data': assistants_data})


@api_view(["GET"])
def get_assistant_messages(request):
    try:
        if not request.user.is_authenticated:
            logger.error("User not authenticated")
            return JsonResponse({'error': 'User not authenticated'}, status=401)

        # Get the doctor
        doctor = Doctor.objects.get(django_user=request.user)
        assistants = Assistant.objects.filter(doctor=doctor)

        # Define the 7-day range
        today = timezone.now().date()
        seven_days_ago = today - timedelta(days=6)  # Last 7 days inclusive

        # Initialize data structure
        message_data = {}

        for assistant in assistants:
            assistant_name = f"{assistant.first_name} {assistant.last_name}"
            # Filter out automatic messages with 'https://' in content
            sent_messages = Message.objects.filter(
                sender=assistant.django_user,
                time__date__range=(seven_days_ago, today)
            ).exclude(text__contains='https://')
            received_messages = Message.objects.filter(
                receiver=assistant.django_user,
                time__date__range=(seven_days_ago, today)
            ).exclude(text__contains='https://')

            # Count messages per day
            sent_counts = defaultdict(int)
            received_counts = defaultdict(int)

            for msg in sent_messages:
                msg_date = msg.time.date()
                sent_counts[msg_date] += 1

            for msg in received_messages:
                msg_date = msg.time.date()
                received_counts[msg_date] += 1

            # Prepare data for the last 7 days
            daily_data = []
            for i in range(7):
                date = seven_days_ago + timedelta(days=i)
                persian_date = jdatetime.date.fromgregorian(date=date).strftime('%Y/%m/%d')
                daily_data.append({
                    'date': persian_date,
                    'sent': sent_counts[date],
                    'received': received_counts[date]
                })

            message_data[assistant_name] = daily_data

        return JsonResponse(message_data, safe=False)
    except Doctor.DoesNotExist:
        logger.error(f"No Doctor found for user {request.user.username}")
        return JsonResponse({'error': 'Doctor not found for this user'}, status=404)
    except Exception as e:
        logger.error(f"Unexpected error: {str(e)}")
        return JsonResponse({'error': str(e)}, status=500)


@api_view(["GET"])
def requested_foods(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    user = request.user
    patients_list = AppUser.objects.all()
    doctor = Doctor.objects.get(django_user=user)
    foods = RequestedFoods.objects.all()

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'user': doctor,
        "message": zipped_data,
        "foods": foods,
        "patients": patients_list,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }

    return render(request, template_name='doctor/foods.html', context=context)


@api_view(["GET"])
def get_diet_activity(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    today = timezone.now().date()
    start_of_week = today - timedelta(days=(today.weekday() + 2) % 7)  # Start of the week: Saturday
    end_of_week = start_of_week + timedelta(days=6)  # End of the week: Friday

    diets = Diet.objects.filter(created_at__date__gte=start_of_week, created_at__date__lte=end_of_week)
    activity_data = defaultdict(
        lambda: {day: 0 for day in ["Saturday", "Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday"]})

    for diet in diets:
        creator_name = diet.creator.username if diet.creator else 'Unknown'
        day_of_week = diet.created_at.strftime('%A')
        activity_data[creator_name][day_of_week] += 1

    return JsonResponse(activity_data, safe=False)


@api_view(["GET"])
def chat_inbox(request):
    user = request.user
    if not user.is_authenticated:
        return redirect('panel-login')

    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)

    # Step 1. Get all patients (AppUser instances) and map by their django_user.id.
    patients = AppUser.objects.select_related('django_user').all()
    patient_map = {p.django_user.id: p for p in patients}

    # Step 2. Get all messages where either the sender or receiver is one of our patients.
    messages = (Message.objects
                .filter(Q(sender__id__in=patient_map.keys()) | Q(receiver__id__in=patient_map.keys()))
                .select_related('sender', 'receiver')
                .order_by('-time'))

    # Step 3. Group messages by conversation partner (the patient)
    conversations = {}
    for msg in messages:
        # Skip if sender or receiver is None
        if not msg.sender or not msg.receiver:
            continue  # ignore this message, move to the next
        # Determine which party is the patient.
        if msg.sender.id in patient_map:
            patient_id = msg.sender.id
        elif msg.receiver.id in patient_map:
            patient_id = msg.receiver.id
        else:
            continue  # Skip messages not involving a patient (should not happen)

        if patient_id not in conversations:
            conversations[patient_id] = msg

    # Step 4. Build the enriched conversation list for display.
    enriched_conversations = []
    now = timezone.now()
    for patient_id, last_msg in conversations.items():
        partner = patient_map.get(patient_id)
        if not partner:
            continue

        is_from_patient = (last_msg.sender.id == partner.django_user.id)
        prefix = f"{partner.profile.first_name}: " if is_from_patient else f"مربی ({partner.profile.therapist}): "
        message_author_class = "from-patient" if is_from_patient else "from-doctor"

        # Count unread messages
        unread_count = Message.objects.filter(
            sender=partner.django_user,
            has_been_seen=False
        ).count()

        # Format the message time:
        message_time = last_msg.time
        if message_time.date() == now.date():
            time_display = "Today"
        elif message_time.date() == (now - timedelta(days=1)).date():
            time_display = "Yesterday"
        else:
            time_display = message_time.strftime("%d/%m/%Y")
        # Combine date and time (hour:minute)
        time_with_hour = f"{time_display} {message_time.strftime('%H:%M')}"

        enriched_conversations.append({
            'app_user': partner,  # the patient AppUser record
            'last_message': last_msg,  # the latest message in that conversation
            'prefix': prefix,  # e.g. "John:" or "You:"
            'message_author_class': message_author_class,
            'formatted_time': time_with_hour,
            'unread_count': unread_count,
        })

    # (Optional) Implement search if needed.
    search_query = request.GET.get('search', '')
    if search_query:
        enriched_conversations = [
            conv for conv in enriched_conversations
            if (search_query.lower() in conv['app_user'].django_user.get_full_name().lower()
                or search_query.lower() in conv['last_message'].text.lower())
        ]

    # (Optional) Implement pagination.
    from django.core.paginator import Paginator
    paginator = Paginator(enriched_conversations, 20)  # 20 conversations per page
    page_number = request.GET.get('page')
    page_obj = paginator.get_page(page_number)

    context = {
        'conversations': page_obj,
        'search_query': search_query,
    }
    return render(request, 'doctor/chat_inbox.html', context)


@api_view(["GET"])
def recent_points_view(request):
    # 1) Ensure user is authenticated
    if not request.user.is_authenticated:
        return redirect('panel-login')

    # 2) Language handling (if needed)
    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)

    django_user = request.user
    if not django_user.groups.filter(name='Doctor').exists():
        return generate_error_page(request, "you have not access")

    # 4) Define the date range for the recent 7 days
    end_date = date.today()
    start_date = end_date - timedelta(days=6)

    # 5) Get the search term for filtering by patient name
    search_query = request.GET.get('search', '').strip()

    # 6) Base queryset of Points limited to the coach's patients (via user.profile.therapist)
    points_qs = Points.objects.select_related('user', 'user__profile').filter(
        date__gte=start_date,
        date__lte=end_date
    )

    # 7) If a search term is provided, filter by first name/last name/phone
    if search_query:
        points_qs = points_qs.filter(
            Q(user__profile__first_name__icontains=search_query) |
            Q(user__profile__last_name__icontains=search_query) |
            Q(user__phone_number__icontains=search_query)
        )

    # 8) Order by date descending
    points_qs = points_qs.order_by('-date')

    # 9) Paginate the results
    paginator = Paginator(points_qs, 20)
    page_number = request.GET.get('page')
    try:
        points_page = paginator.page(page_number)
    except PageNotAnInteger:
        points_page = paginator.page(1)
    except EmptyPage:
        points_page = paginator.page(paginator.num_pages)

    # 10) Render the template
    context = {
        'points_page': points_page,
        'search_query': search_query,
        'start_date': start_date,
        'end_date': end_date,
    }
    return render(request, 'doctor/recent_points.html', context)


@api_view(["GET"])
def timeslot_management(request):
    """Doctor interface for managing timeslot purchases"""
    if not request.user.is_authenticated:
        return redirect('panel-login')
    user = request.user
    if not user.groups.filter(name='Doctor').exists():
        return generate_error_page(request, "you have not access")

    doctor = Doctor.objects.get(django_user=user)
    
    # Get all purchased timeslots (not just pending ones)
    from payment.models import TimeSlotPayment
    purchased_timeslots = TimeSlotPayment.objects.filter(
        time_slot__doctor_django_user=user,
        is_paid=True
    ).order_by('time_slot__start_time')

    # Get statistics
    total_timeslots = TimeSlot.objects.filter(
        doctor_django_user=user,
        created_by_doctor=True
    ).count()
    
    available_timeslots = TimeSlot.objects.filter(
        doctor_django_user=user,
        created_by_doctor=True,
        is_available=1
    ).count()
    
    purchased_timeslots_count = TimeSlotPayment.objects.filter(
        time_slot__doctor_django_user=user,
        is_paid=True
    ).count()
    
    assigned_timeslots = TimeSlotPayment.objects.filter(
        time_slot__doctor_django_user=user,
        is_paid=True,
        status='assigned'
    ).count()

    # Get available therapists
    therapists = Assistant.objects.filter(doctor=doctor)

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'user': doctor,
        'purchased_timeslots': purchased_timeslots,
        'therapists': therapists,
        'total_timeslots': total_timeslots,
        'available_timeslots': available_timeslots,
        'purchased_timeslots_count': purchased_timeslots_count,
        'assigned_timeslots': assigned_timeslots,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }

    return render(request, template_name='doctor/timeslot_management.html', context=context)


@api_view(["GET"])
def create_timeslot_for_purchase(request):
    """Doctor interface for creating timeslots for purchase"""
    if not request.user.is_authenticated:
        return redirect('panel-login')
    user = request.user
    if not user.groups.filter(name='Doctor').exists():
        return generate_error_page(request, "you have not access")

    doctor = Doctor.objects.get(django_user=user)
    
    # Get existing timeslots created by this doctor
    existing_timeslots = TimeSlot.objects.filter(
        doctor_django_user=user,
        created_by_doctor=True
    ).order_by('-start_time')

    # Convert to JSON for JavaScript
    import json
    existing_timeslots_json = []
    for timeslot in existing_timeslots:
        existing_timeslots_json.append({
            'id': timeslot.id,
            'start_time': timeslot.start_time.isoformat(),
            'duration': timeslot.duration,
            'is_available': timeslot.is_available,
        })

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'user': doctor,
        'existing_timeslots': existing_timeslots,
        'existing_timeslots_json': json.dumps(existing_timeslots_json),
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }

    return render(request, template_name='doctor/create_timeslot.html', context=context)


@api_view(['POST'])
def submit_timeslot_for_purchase(request):
    """Submit new timeslot for purchase"""
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Doctor').exists():
        return generate_error_page(request, "you have not access")

    try:
        selected_ids = request.POST.getlist("selectedIds")
        selected_year = request.POST.get("selectedYear", "2025")
        selected_month = request.POST.get("selectedMonth", "1")
        selected_day = request.POST.get("selectedDay", "1")
        
        print(f"Debug: selected_ids = {selected_ids}")
        print(f"Debug: selected_year = {selected_year}")
        print(f"Debug: selected_month = {selected_month}")
        print(f"Debug: selected_day = {selected_day}")
        
        # Create timeslots for purchase
        for selected_id in selected_ids:
            print(f"Debug: Processing selected_id = {selected_id}")
            
            # Parse the selected ID format: "YYYY-MM-DD_HHMM_HHMM"
            parts = selected_id.split('_')
            print(f"Debug: parts = {parts}")
            
            if len(parts) >= 3:
                date_str = parts[0]  # YYYY-MM-DD
                print(f"Debug: date_str = {date_str}")
                
                # Parse date
                year, month, day = map(int, date_str.split('-'))
                print(f"Debug: year={year}, month={month}, day={day}")
                
                # Parse time
                start_time_str = parts[1]  # HHMM
                end_time_str = parts[2]    # HHMM
                print(f"Debug: start_time_str = {start_time_str}, end_time_str = {end_time_str}")
                
                start_hour = int(start_time_str[:2])
                start_minute = int(start_time_str[2:])
                end_hour = int(end_time_str[:2])
                end_minute = int(end_time_str[2:])
                print(f"Debug: start_hour={start_hour}, start_minute={start_minute}, end_hour={end_hour}, end_minute={end_minute}")
                
                # Create datetime objects
                start_datetime = datetime(year, month, day, start_hour, start_minute)
                end_datetime = datetime(year, month, day, end_hour, end_minute)
                print(f"Debug: start_datetime = {start_datetime}, end_datetime = {end_datetime}")
                
                # Calculate duration in minutes
                duration = int((end_datetime - start_datetime).total_seconds() / 60)
                print(f"Debug: duration = {duration} minutes")
                
                # Calculate day of week (0 = Saturday, 1 = Sunday, etc.)
                day_of_week = (start_datetime.weekday() + 2) % 7
                print(f"Debug: day_of_week = {day_of_week}")
                
                # Create timeslot
                timeslot = TimeSlot(
                    doctor_django_user=django_user,
                    start_time=start_datetime,
                    duration=duration,
                    days_of_week=day_of_week,
                    is_available=1,
                    created_by_doctor=True
                )
                timeslot.save()
                print(f"Debug: Timeslot saved with ID = {timeslot.id}")
            else:
                print(f"Debug: Invalid selected_id format: {selected_id}")

        print(f"Debug: Successfully created {len(selected_ids)} timeslots")
        from django.contrib import messages
        messages.success(request, f"تعداد {len(selected_ids)} زمان با موفقیت ایجاد شد")
        return redirect('panel-doctor-create-timeslot')
    except Exception as e:
        return myResponse.Error("error on saving the timeslot: {}".format(e), 1001)


@api_view(['POST'])
def assign_timeslot_to_therapist_panel(request):
    """Assign purchased timeslot to therapist from panel"""
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Doctor').exists():
        return generate_error_page(request, "you have not access")

    try:
        payment_id = request.POST.get("payment_id")
        therapist_id = request.POST.get("therapist_id")

        from payment.models import TimeSlotPayment
        payment = TimeSlotPayment.objects.filter(
            id=payment_id,
            time_slot__doctor_django_user=django_user,
            is_paid=True,
            status='pending'
        ).first()

        if not payment:
            return myResponse.Error("Payment not found", Errors.DataNotFound.code)

        therapist = Assistant.objects.filter(id=therapist_id).first()
        if not therapist:
            return myResponse.Error("Therapist not found", Errors.DataNotFound.code)

        # Check if therapist belongs to this doctor
        if therapist.doctor.django_user != django_user:
            return myResponse.Error("Therapist does not belong to this doctor", Errors.InvalidArgument.code)

        # Assign the timeslot
        if payment.assign_to_therapist(therapist):
            # Send Telegram notification to the therapist
            try:
                from panel.telegram_notification import TelegramNotification
                
                # Prepare notification data
                therapist_name = f"{therapist.first_name} {therapist.last_name}"
                patient_name = f"{payment.user.profile.first_name} {payment.user.profile.last_name}"
                session_time = payment.time_slot.start_time.strftime("%Y/%m/%d ساعت %H:%M")
                
                # Send notification if therapist has telegram_chat_id
                if therapist.telegram_chat_id:
                    TelegramNotification().send_timeslot_assignment_notif(
                        therapist_name=therapist_name,
                        patient_name=patient_name,
                        session_time=session_time,
                        chat_id=therapist.telegram_chat_id
                    )
                    print(f"Telegram notification sent to {therapist_name}")
                else:
                    print(f"No telegram_chat_id found for therapist {therapist_name}")
                    
            except Exception as e:
                print(f"Error sending Telegram notification: {e}")
            
            from django.contrib import messages
            messages.success(request, f"زمان با موفقیت به {therapist.first_name} {therapist.last_name} تخصیص داده شد")
            return redirect('panel-doctor-timeslot-management')
        else:
            return myResponse.Error("Failed to assign timeslot", Errors.InvalidArgument.code)

    except Exception as e:
        return myResponse.Error("error on assigning timeslot: {}".format(e), 1001)


@api_view(['POST'])
def change_timeslot_assignment(request):
    """Doctor changes the assignment of an already assigned timeslot"""
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Doctor').exists():
        return generate_error_page(request, "you have not access")

    try:
        payment_id = int(request.POST.get('payment_id'))
        therapist_id = int(request.POST.get('therapist_id'))
        current_therapist_id = int(request.POST.get('current_therapist_id'))

        # Get the payment and new therapist
        from payment.models import TimeSlotPayment
        payment = TimeSlotPayment.objects.filter(
            id=payment_id,
            time_slot__doctor_django_user=django_user,
            is_paid=True,
            status='assigned'
        ).first()

        if not payment:
            from django.contrib import messages
            messages.error(request, "Payment not found or not eligible for reassignment")
            return redirect('panel-doctor-timeslot-management')

        # Verify current therapist matches
        if payment.assigned_therapist.id != current_therapist_id:
            from django.contrib import messages
            messages.error(request, "Current therapist mismatch")
            return redirect('panel-doctor-timeslot-management')

        new_therapist = Assistant.objects.filter(id=therapist_id).first()
        if not new_therapist:
            from django.contrib import messages
            messages.error(request, "New therapist not found")
            return redirect('panel-doctor-timeslot-management')

        # Don't allow assigning to the same therapist
        if new_therapist.id == current_therapist_id:
            from django.contrib import messages
            messages.error(request, "Cannot assign to the same therapist")
            return redirect('panel-doctor-timeslot-management')

        # Get the old therapist name for the message
        old_therapist_name = f"{payment.assigned_therapist.first_name} {payment.assigned_therapist.last_name}"
        new_therapist_name = f"{new_therapist.first_name} {new_therapist.last_name}"

        # Change the assignment
        payment.assigned_therapist = new_therapist
        payment.assigned_date = datetime.now()  # Update assignment date
        payment.save()

        # Send Telegram notification to the new therapist
        try:
            from panel.telegram_notification import TelegramNotification
            
            # Prepare notification data
            new_therapist_name = f"{new_therapist.first_name} {new_therapist.last_name}"
            patient_name = f"{payment.user.profile.first_name} {payment.user.profile.last_name}"
            session_time = payment.time_slot.start_time.strftime("%Y/%m/%d ساعت %H:%M")
            
            # Send notification if new therapist has telegram_chat_id
            if new_therapist.telegram_chat_id:
                TelegramNotification().send_timeslot_assignment_notif(
                    therapist_name=new_therapist_name,
                    patient_name=patient_name,
                    session_time=session_time,
                    chat_id=new_therapist.telegram_chat_id
                )
                print(f"Telegram notification sent to {new_therapist_name}")
            else:
                print(f"No telegram_chat_id found for therapist {new_therapist_name}")
                
        except Exception as e:
            print(f"Error sending Telegram notification: {e}")

        from django.contrib import messages
        messages.success(request, f"تخصیص از {old_therapist_name} به {new_therapist_name} تغییر یافت")
        return redirect('panel-doctor-timeslot-management')

    except Exception as e:
        from django.contrib import messages
        messages.error(request, f"خطا در تغییر تخصیص: {str(e)}")
        return redirect('panel-doctor-timeslot-management')
