"""
Blood Test Health Analyzer - Python implementation based on TypeScript analysisEngine.ts and riskEngine.ts
"""
import re
import math
from typing import Dict, List, Any, Optional


class BloodHealthAnalyzer:
    """
    Analyzes blood test results and calculates health indices.
    Based on the TypeScript analysisEngine.ts implementation.
    """
    
    def __init__(self, gender: str = 'unknown', age: int = 30):
        self.gender = gender.lower() if gender else 'unknown'
        self.age = age if age else 30
        self.ranges = self._build_dynamic_ranges()
    
    def _build_dynamic_ranges(self) -> Dict[str, Dict[str, float]]:
        """Build dynamic reference ranges based on gender and age."""
        r = {}
        
        # 1. Hematology
        if self.gender == 'male':
            r['Hb'] = {'norm_min': 13.5, 'norm_max': 17.5, 'opt_min': 14.5, 'opt_max': 16.5}
            r['HCT'] = {'norm_min': 40.0, 'norm_max': 54.0, 'opt_min': 42.0, 'opt_max': 50.0}
            r['RBC'] = {'norm_min': 4.5, 'norm_max': 6.0, 'opt_min': 4.7, 'opt_max': 5.5}
            r['Ferritin'] = {'norm_min': 30, 'norm_max': 400, 'opt_min': 50, 'opt_max': 200}
        else:
            r['Hb'] = {'norm_min': 12.0, 'norm_max': 16.0, 'opt_min': 13.0, 'opt_max': 14.5}
            r['HCT'] = {'norm_min': 33.0, 'norm_max': 48.0, 'opt_min': 37.0, 'opt_max': 44.0}
            r['RBC'] = {'norm_min': 4.0, 'norm_max': 5.2, 'opt_min': 4.2, 'opt_max': 4.9}
            
            if self.age > 50:
                r['Ferritin'] = {'norm_min': 20, 'norm_max': 200, 'opt_min': 40, 'opt_max': 150}
            else:
                r['Ferritin'] = {'norm_min': 10, 'norm_max': 150, 'opt_min': 30, 'opt_max': 100}
        
        r['WBC'] = {'norm_min': 4.5, 'norm_max': 11.0, 'opt_min': 5.0, 'opt_max': 8.0}
        r['Platelet'] = {'norm_min': 150, 'norm_max': 450, 'opt_min': 200, 'opt_max': 350}
        r['MCV'] = {'norm_min': 80, 'norm_max': 100, 'opt_min': 85, 'opt_max': 92}
        
        # 2. Kidney
        if self.gender == 'male':
            r['Creatinine'] = {'norm_min': 0.7, 'norm_max': 1.3, 'opt_min': 0.8, 'opt_max': 1.1}
        else:
            r['Creatinine'] = {'norm_min': 0.7, 'norm_max': 1.4, 'opt_min': 0.7, 'opt_max': 1.0}
        
        base_optimal = 100
        age_penalty = max(0, (self.age - 30) * 0.5)
        optimal_min_gfr = max(60, base_optimal - age_penalty)
        
        r['eGFR'] = {'norm_min': 60, 'norm_max': 140, 'opt_min': optimal_min_gfr, 'opt_max': 130}
        r['Urea'] = {'norm_min': 15, 'norm_max': 40, 'opt_min': 18, 'opt_max': 30}
        
        # 3. Hormonal
        if self.gender == 'male':
            r['Estradiol'] = {'norm_min': 10, 'norm_max': 50, 'opt_min': 20, 'opt_max': 40}
            r['Testosterone'] = {'norm_min': 250, 'norm_max': 1100, 'opt_min': 400, 'opt_max': 900}
            r['FSH'] = {'norm_min': 1.5, 'norm_max': 12.4, 'opt_min': 2.0, 'opt_max': 10.0}
        else:
            if self.age > 50:
                r['Estradiol'] = {'norm_min': 0, 'norm_max': 35, 'opt_min': 0, 'opt_max': 20}
                r['FSH'] = {'norm_min': 25, 'norm_max': 135, 'opt_min': 30, 'opt_max': 100}
            else:
                r['Estradiol'] = {'norm_min': 12.5, 'norm_max': 350, 'opt_min': 40, 'opt_max': 200}
                r['FSH'] = {'norm_min': 2.5, 'norm_max': 20, 'opt_min': 3.0, 'opt_max': 9.0}
        
        tsh_max_opt = 2.5
        if self.age > 70:
            tsh_max_opt = 4.0
        r['TSH'] = {'norm_min': 0.5, 'norm_max': 4.7, 'opt_min': 1.0, 'opt_max': tsh_max_opt}
        r['T3 Free'] = {'norm_min': 2.3, 'norm_max': 4.2, 'opt_min': 3.0, 'opt_max': 3.8}
        r['T4 Free'] = {'norm_min': 0.9, 'norm_max': 1.8, 'opt_min': 1.1, 'opt_max': 1.5}
        
        # 4. Metabolic
        r['Fasting Glucose'] = {'norm_min': 65, 'norm_max': 108, 'opt_min': 75, 'opt_max': 90}
        r['HbA1c'] = {'norm_min': 0, 'norm_max': 6.0, 'opt_min': 4.5, 'opt_max': 5.3}
        r['Triglycerides'] = {'norm_min': 0, 'norm_max': 150, 'opt_min': 40, 'opt_max': 100}
        r['Total Cholesterol'] = {'norm_min': 0, 'norm_max': 200, 'opt_min': 140, 'opt_max': 190}
        r['LDL'] = {'norm_min': 0, 'norm_max': 130, 'opt_min': 50, 'opt_max': 100}
        r['HDL'] = {'norm_min': 40, 'norm_max': 100, 'opt_min': 60, 'opt_max': 90}
        
        r['ALT'] = {'norm_min': 7, 'norm_max': 35, 'opt_min': 10, 'opt_max': 25}
        r['AST'] = {'norm_min': 8, 'norm_max': 37, 'opt_min': 10, 'opt_max': 28}
        r['GGT'] = {'norm_min': 5, 'norm_max': 36, 'opt_min': 10, 'opt_max': 25}
        r['Bilirubin Total'] = {'norm_min': 0.1, 'norm_max': 1.2, 'opt_min': 0.3, 'opt_max': 0.9}
        
        r['Vitamin D3'] = {'norm_min': 21, 'norm_max': 100, 'opt_min': 40, 'opt_max': 70}
        r['Vitamin B12'] = {'norm_min': 211, 'norm_max': 911, 'opt_min': 450, 'opt_max': 900}
        r['Iron'] = {'norm_min': 25, 'norm_max': 145, 'opt_min': 60, 'opt_max': 120}
        r['Calcium'] = {'norm_min': 8.5, 'norm_max': 10.5, 'opt_min': 9.2, 'opt_max': 10.0}
        r['Magnesium'] = {'norm_min': 1.9, 'norm_max': 2.5, 'opt_min': 2.1, 'opt_max': 2.4}
        r['Cortisol'] = {'norm_min': 5.27, 'norm_max': 22.45, 'opt_min': 10, 'opt_max': 18}
        r['CRP'] = {'norm_min': 0, 'norm_max': 6.0, 'opt_min': 0, 'opt_max': 1.0}
        
        return r
    
    def _parse_value(self, value: Any) -> Optional[float]:
        """Parse a value to float, handling strings with Persian digits and < > operators."""
        if isinstance(value, (int, float)):
            return float(value)
        if isinstance(value, str):
            # Replace Persian digits
            persian_digits = '۰۱۲۳۴۵۶۷۸۹'
            for i, d in enumerate(persian_digits):
                value = value.replace(d, str(i))
            
            clean = value.strip().replace('*', '')
            if not clean:
                return None
            
            if clean.startswith('<'):
                nums = re.search(r'[\d.]+', clean)
                return float(nums.group()) - 0.1 if nums else None
            
            if clean.startswith('>'):
                nums = re.search(r'[\d.]+', clean)
                return float(nums.group()) + 0.1 if nums else None
            
            # Extract first number
            nums = re.search(r'[\d.]+', clean)
            if nums:
                return float(nums.group())
        
        return None
    
    def _calculate_fuzzy_score(self, test_name: str, raw_val: Any) -> Optional[float]:
        """Calculate fuzzy score for a test value."""
        val = self._parse_value(raw_val)
        if val is None or test_name not in self.ranges:
            return None
        
        r = self.ranges[test_name]
        norm_min, norm_max = r['norm_min'], r['norm_max']
        opt_min, opt_max = r['opt_min'], r['opt_max']
        
        if opt_min <= val <= opt_max:
            return 100.0
        
        if norm_min <= val <= norm_max:
            if val < opt_min:
                dist = (opt_min - val) / (opt_min - norm_min + 0.00001)
                return max(70, 100 - (dist * 30))
            else:
                dist = (val - opt_max) / (norm_max - opt_max + 0.00001)
                return max(70, 100 - (dist * 30))
        
        if val < norm_min:
            denom = norm_min if norm_min > 0 else 1
            dist = (norm_min - val) / denom
            return max(20, 70 - (dist * 100))
        else:
            denom = norm_max if norm_max > 0 else 1
            dist = (val - norm_max) / denom
            return max(20, 70 - (dist * 100))
    
    def _get_lifestyle_feedback(self, test_name: str, value: float, score: float) -> str:
        """Get lifestyle feedback for a test result."""
        if test_name not in self.ranges:
            return ""
        
        ranges = self.ranges[test_name]
        is_low = value < ranges['opt_min']
        is_high = value > ranges['opt_max']
        is_optimal = score == 100
        
        if is_optimal:
            if test_name == 'Vitamin D3':
                return 'سطح ویتامین D شما عالی است. این برای ایمنی و سلامت استخوان حیاتی است. همین روند را حفظ کنید.'
            elif test_name == 'HbA1c':
                return 'کنترل قند خون شما عالی است که نشان‌دهنده حساسیت خوب به انسولین می‌باشد.'
            elif test_name == 'CRP':
                return 'سطح التهاب بدن شما بسیار پایین و ایده‌آل است.'
            elif test_name == 'Testosterone':
                return 'سطح تستوسترون شما در محدوده بهینه برای سلامت و انرژی قرار دارد.'
            elif test_name == 'Ferritin':
                return 'ذخایر آهن شما در وضعیت مطلوبی قرار دارد.'
            else:
                return 'این شاخص در محدوده عملکردی بهینه قرار دارد. سبک زندگی فعلی شما از این نظر پشتیبان سلامت شماست.'
        
        if test_name == 'Vitamin D3' and is_low:
            return 'سطح ویتامین D پایین است. مصرف ماهی‌های چرب (سالمون)، زرده تخم‌مرغ و قارچ را افزایش دهید. روزانه ۲۰ دقیقه در معرض نور خورشید باشید. مکمل ویتامین D3 (همراه با K2) تحت نظر پزشک توصیه می‌شود.'
        elif test_name == 'Vitamin B12' and is_low:
            return 'کمبود B12 می‌تواند باعث خستگی شود. منابع حیوانی مانند گوشت قرمز، جگر، تخم‌مرغ و لبنیات را افزایش دهید. در صورت گیاهخواری، مکمل متیل‌کوبالامین مصرف کنید.'
        elif test_name == 'Ferritin':
            if is_low:
                return 'ذخایر آهن کم است. گوشت قرمز، عدس، اسفناج مصرف کنید و برای جذب بهتر، آن‌ها را با منابع ویتامین C (لیمو، فلفل دلمه‌ای) همراه کنید. از نوشیدن چای بلافاصله بعد غذا پرهیز کنید.'
            elif is_high:
                return 'فرتین بالا می‌تواند نشانگر التهاب یا کبد چرب باشد. مصرف گوشت قرمز و الکل را کاهش دهید و اهدای خون را (در صورت تایید پزشک) در نظر بگیرید.'
        elif test_name in ['Hb', 'HCT', 'RBC']:
            if is_low:
                return 'احتمال کم‌خونی وجود دارد. مصرف منابع آهن و ویتامین‌های گروه B را افزایش دهید. وضعیت گوارش و جذب مواد مغذی را بررسی کنید.'
            elif is_high:
                return 'غلظت خون بالاست. هیدراتاسیون (نوشیدن آب) را افزایش دهید. سیگار (در صورت مصرف) را ترک کنید و احتمال آپنه خواب را بررسی کنید.'
        elif test_name in ['Fasting Glucose', 'HbA1c'] and is_high:
            return 'متابولیسم قند نیاز به اصلاح دارد. مصرف کربوهیدرات‌های تصفیه شده (نان سفید، شیرینی) را کاهش دهید. پیاده‌روی ۱۰ دقیقه‌ای بعد از هر وعده غذا و مصرف فیبر بیشتر معجزه می‌کند.'
        elif test_name == 'Triglycerides' and is_high:
            return 'تری‌گلیسرید بالا معمولاً ناشی از مصرف زیاد قند و نشاسته است، نه چربی. شیرینی‌جات، میوه‌های خیلی شیرین و الکل را محدود کنید. مصرف امگا-۳ (ماهی، گردو) را افزایش دهید.'
        elif test_name == 'HDL' and is_low:
            return 'برای افزایش HDL (چربی خوب)، چربی‌های سالم (روغن زیتون، آووکادو، آجیل) مصرف کنید و فعالیت بدنی، به‌ویژه تمرینات قدرتی یا اینتروال با شدت بالا، انجام دهید.'
        elif test_name == 'LDL' and is_high:
            return 'کیفیت چربی‌های مصرفی مهم است. از روغن‌های نباتی صنعتی (آفتابگردان، ذرت) پرهیز کنید. فیبر محلول (جو دوسر، حبوبات) را افزایش دهید تا دفع کلسترول بهبود یابد.'
        elif test_name in ['ALT', 'AST', 'GGT'] and is_high:
            return 'نشان‌دهنده فشار روی کبد است. فروکتوز (قند میوه و شربت‌ها) و الکل را حذف کنید. سبزیجات چلیپایی (کلم بروکلی، کلم پیچ) و مکمل‌هایی مثل خار مریم (Milk Thistle) برای حمایت کبد مفیدند.'
        elif test_name == 'TSH':
            if is_high:
                return 'عملکرد تیروئید کمی کند است. سلنیوم (روزانه ۲ عدد بادام برزیلی) و ید (غذاهای دریایی) کافی دریافت کنید. مدیریت استرس و خواب کافی برای سلامت تیروئید حیاتی است.'
            elif is_low:
                return 'عملکرد تیروئید ممکن است بیش از حد فعال باشد. مدیریت استرس و پرهیز از محرک‌ها (کافئین زیاد) توصیه می‌شود.'
        elif test_name == 'Magnesium' and is_low:
            return 'منیزیم برای آرامش و خواب ضروری است. سبزیجات برگ سبز تیره، تخم کدو و شکلات تلخ مصرف کنید. حمام نمک اپسوم یا مکمل منیزیم گلیسینات قبل خواب کمک‌کننده است.'
        elif test_name == 'Cortisol' and is_high:
            return 'استرس مزمن مشهود است. تمرینات تنفسی، مدیتیشن و کاهش مصرف کافئین ضروری است. خواب شبانه منظم (قبل از ۱۱ شب) اولویت باشد.'
        elif test_name == 'CRP' and is_high:
            return 'التهاب سیستمیک بالاست. رژیم ضدالتهابی (زردچوبه، زنجبیل، ماهی‌های چرب، روغن زیتون) را پیش بگیرید و غذاهای فرآوری شده و روغن‌های سرخ‌کردنی را حذف کنید.'
        elif test_name in ['eGFR', 'Creatinine'] and not is_optimal:
            return 'برای حمایت از کلیه، نوشیدن آب کافی را فراموش نکنید. مصرف پروتئین را متعادل کرده و فشار خون را کنترل نمایید.'
        
        if is_high:
            return f'مقدار این شاخص بالاتر از حد بهینه است. لطفاً رژیم غذایی و سبک زندگی مرتبط با {test_name} را بازنگری کنید.'
        elif is_low:
            return f'مقدار این شاخص پایین‌تر از حد بهینه است. لطفاً دریافت مواد مغذی مرتبط با {test_name} را افزایش دهید.'
        
        return ""
    
    def _weighted_avg(self, items: List[Dict[str, Any]]) -> Dict[str, Any]:
        """Calculate weighted average score from test items."""
        total_w = 0
        total_s = 0
        details = []
        
        for item in items:
            test_name = item['name']
            val = item['val']
            weight = item['w']
            
            score = self._calculate_fuzzy_score(test_name, val)
            parsed_val = self._parse_value(val)
            
            status = 'یافت نشد'
            feedback = ''
            
            if score is not None:
                if score == 100:
                    status = 'ایده‌آل'
                elif score >= 70:
                    status = 'طبیعی'
                else:
                    status = 'نیازمند بررسی'
                
                if parsed_val is not None:
                    feedback = self._get_lifestyle_feedback(test_name, parsed_val, score)
                
                total_s += score * weight
                total_w += weight
            
            details.append({
                'test': test_name,
                'value': parsed_val,
                'score': round(score) if score is not None else None,
                'status': status,
                'feedback': feedback
            })
        
        return {
            'score': round(total_s / total_w) if total_w > 0 else 0,
            'details': details
        }
    
    def idx_cardiometabolic(self, data: Dict[str, Any]) -> Dict[str, Any]:
        """Calculate cardiometabolic index."""
        tg = self._parse_value(data.get('Triglycerides'))
        hdl = self._parse_value(data.get('HDL'))
        ratio_score = 80
        warnings = []
        
        if tg is not None and hdl is not None and hdl != 0:
            ratio = tg / hdl
            if ratio < 2:
                ratio_score = 100
            elif ratio < 4:
                ratio_score = 75
            else:
                ratio_score = 40
                warnings.append(f'نسبت تری‌گلیسرید به HDL بالا است: {ratio:.1f}')
        
        tests = [
            {'name': 'HbA1c', 'val': data.get('HbA1c'), 'w': 0.25},
            {'name': 'Fasting Glucose', 'val': data.get('Fasting Glucose'), 'w': 0.15},
            {'name': 'LDL', 'val': data.get('LDL'), 'w': 0.15},
            {'name': 'CRP', 'val': data.get('CRP'), 'w': 0.20},
            {'name': 'Total Cholesterol', 'val': data.get('Total Cholesterol'), 'w': 0.05},
            {'name': 'Triglycerides', 'val': data.get('Triglycerides'), 'w': 0.10},
            {'name': 'HDL', 'val': data.get('HDL'), 'w': 0.10},
        ]
        
        result = self._weighted_avg(tests)
        base_score = result['score']
        final_score = round((base_score * 0.8) + (ratio_score * 0.2))
        
        if final_score >= 95:
            status = 'عالی'
        elif final_score >= 80:
            status = 'خوب'
        elif final_score >= 60:
            status = 'متوسط'
        else:
            status = 'نیازمند توجه'
        
        return {
            'score': final_score,
            'status': status,
            'details': result['details'],
            'warnings': warnings
        }
    
    def idx_hematology(self, data: Dict[str, Any]) -> Dict[str, Any]:
        """Calculate hematology index."""
        tests = [
            {'name': 'Hb', 'val': data.get('Hb'), 'w': 0.20},
            {'name': 'MCV', 'val': data.get('MCV'), 'w': 0.10},
            {'name': 'Ferritin', 'val': data.get('Ferritin'), 'w': 0.20},
            {'name': 'Iron', 'val': data.get('Iron'), 'w': 0.10},
            {'name': 'Vitamin D3', 'val': data.get('Vitamin D3'), 'w': 0.25},
            {'name': 'Vitamin B12', 'val': data.get('Vitamin B12'), 'w': 0.15},
        ]
        result = self._weighted_avg(tests)
        
        if result['score'] >= 90:
            status = 'ایده‌آل'
        elif result['score'] < 70:
            status = 'کمبود'
        else:
            status = 'خوب'
        
        return {
            'score': result['score'],
            'status': status,
            'details': result['details']
        }
    
    def idx_organ_function(self, data: Dict[str, Any]) -> Dict[str, Any]:
        """Calculate organ function index."""
        tests = [
            {'name': 'ALT', 'val': data.get('ALT'), 'w': 0.20},
            {'name': 'AST', 'val': data.get('AST'), 'w': 0.15},
            {'name': 'Bilirubin Total', 'val': data.get('Bilirubin Total'), 'w': 0.10},
            {'name': 'eGFR', 'val': data.get('eGFR'), 'w': 0.35},
            {'name': 'Creatinine', 'val': data.get('Creatinine'), 'w': 0.15},
            {'name': 'Urea', 'val': data.get('Urea'), 'w': 0.05},
        ]
        result = self._weighted_avg(tests)
        
        status = 'خوب' if result['score'] >= 75 else 'تحت فشار'
        
        return {
            'score': result['score'],
            'status': status,
            'details': result['details']
        }
    
    def idx_thyroid(self, data: Dict[str, Any]) -> Dict[str, Any]:
        """Calculate thyroid index."""
        tests = [
            {'name': 'TSH', 'val': data.get('TSH'), 'w': 0.60},
            {'name': 'T4 Free', 'val': data.get('T4 Free'), 'w': 0.20},
            {'name': 'T3 Free', 'val': data.get('T3 Free'), 'w': 0.20},
        ]
        result = self._weighted_avg(tests)
        
        status = 'متعادل' if result['score'] > 85 else 'نامتعادل'
        
        return {
            'score': result['score'],
            'status': status,
            'details': result['details']
        }
    
    def idx_hormonal(self, data: Dict[str, Any]) -> Dict[str, Any]:
        """Calculate hormonal balance index."""
        tests = [
            {'name': 'Magnesium', 'val': data.get('Magnesium'), 'w': 0.20},
            {'name': 'Calcium', 'val': data.get('Calcium'), 'w': 0.10},
            {'name': 'Cortisol', 'val': data.get('Cortisol'), 'w': 0.30},
            {'name': 'Estradiol', 'val': data.get('Estradiol'), 'w': 0.20},
            {'name': 'FSH', 'val': data.get('FSH'), 'w': 0.20},
            {'name': 'Testosterone', 'val': data.get('Testosterone'), 'w': 0.20},
        ]
        # Remove duplicates
        unique_tests = {t['name']: t for t in tests}.values()
        result = self._weighted_avg(list(unique_tests))
        
        status = 'پایدار' if result['score'] > 80 else 'بررسی تعادل'
        
        return {
            'score': result['score'],
            'status': status,
            'details': result['details']
        }
    
    def analyze(self, raw_data: Dict[str, Any], profile: Optional[Dict[str, Any]] = None) -> Dict[str, Any]:
        """
        Main analysis method that calculates all indices, overall score, and risks.
        
        Args:
            raw_data: Dictionary mapping test names to values (e.g., {'Hb': 14.5, 'ALT': 20})
            profile: Optional patient profile with gender, age, height, weight, etc.
        
        Returns:
            Dictionary with overall_score, indices, risks, and patient info
        """
        s1 = self.idx_cardiometabolic(raw_data)
        s2 = self.idx_hematology(raw_data)
        s3 = self.idx_organ_function(raw_data)
        s4 = self.idx_thyroid(raw_data)
        s5 = self.idx_hormonal(raw_data)
        
        # Initialize Risk Engine
        if profile is None:
            profile = {'gender': self.gender, 'age': self.age}
        risk_engine = HealthRiskEngine(profile, raw_data)
        risks = risk_engine.get_report()
        
        overall = (
            (s1['score'] * 0.30) +
            (s2['score'] * 0.20) +
            (s3['score'] * 0.20) +
            (s4['score'] * 0.15) +
            (s5['score'] * 0.15)
        )
        
        return {
            'Overall_Health_Score': round(overall),
            'Indices': {
                'Cardiometabolic': s1['score'],
                'Hematology': s2['score'],
                'Organ Function': s3['score'],
                'Thyroid': s4['score'],
                'Hormonal': s5['score']
            },
            'Index_Details': {
                'Cardiometabolic': s1,
                'Hematology': s2,
                'Organ Function': s3,
                'Thyroid': s4,
                'Hormonal': s5
            },
            'Risks': risks
        }


class HealthRiskEngine:
    """
    Calculates health risks (CVD, Diabetes, Liver) based on lab data and patient profile.
    Based on the TypeScript riskEngine.ts implementation.
    """
    
    def __init__(self, profile: Dict[str, Any], lab_data: Dict[str, Any]):
        self.age = profile.get('age', 30)
        self.gender = 0 if profile.get('gender', 'unknown').lower() == 'male' else 1
        
        # Biometrics with defaults
        height_m = (profile.get('height', 170) or 170) / 100
        weight = profile.get('weight', 70) or 70
        self.bmi = weight / (height_m * height_m) if height_m > 0 else 25
        self.sbp = profile.get('sbp', 120) or 120
        self.smoker = 1 if profile.get('smoker', False) else 0
        self.family_hist = 1 if profile.get('diabetesHistory', False) else 0
        self.bp_meds = 1 if profile.get('bpMeds', False) else 0
        
        # Waist estimation
        if profile.get('waist'):
            self.waist = profile['waist']
        else:
            self.waist = self.bmi * 3.2 if self.gender == 0 else self.bmi * 3.0
        
        # Parse lab data
        self.fbs = self._parse(lab_data.get('Fasting Glucose'))
        self.a1c = self._parse(lab_data.get('HbA1c'))
        self.chol_total = self._parse(lab_data.get('Total Cholesterol'))
        self.hdl = self._parse(lab_data.get('HDL'))
        self.tg = self._parse(lab_data.get('Triglycerides'))
        self.ggt = self._parse(lab_data.get('GGT'))
    
    def _parse(self, val: Any) -> Optional[float]:
        """Parse value to float."""
        if isinstance(val, (int, float)):
            return float(val)
        if isinstance(val, str):
            clean = re.sub(r'[<>*]', '', val).strip()
            try:
                return float(clean)
            except (ValueError, TypeError):
                return None
        return None
    
    def calculate_cvd_risk(self) -> Dict[str, Any]:
        """Calculate CVD risk (WHO EMR Region B - Adapted)."""
        if not self.chol_total:
            return self._missing_result("اطلاعات کلسترول یافت نشد")
        
        # Convert to mmol/L
        chol_mmol = self.chol_total / 38.67
        score = 0
        
        # Age
        if self.age < 50:
            score += 1
        elif self.age < 60:
            score += 2
        elif self.age < 70:
            score += 3
        else:
            score += 4
        
        # Gender (Male higher risk)
        if self.gender == 0:
            score += 1
        
        # Smoker
        if self.smoker:
            score += 2
        
        # SBP
        if self.sbp >= 140:
            score += 1
        if self.sbp >= 160:
            score += 1
        if self.sbp >= 180:
            score += 1
        
        # Diabetes Status
        is_diabetic = (self.fbs and self.fbs >= 126) or (self.a1c and self.a1c >= 6.5)
        if is_diabetic:
            score += 2
        
        # Cholesterol
        if chol_mmol < 4.5:
            score -= 1
        elif chol_mmol > 6.5:
            score += 1
        elif chol_mmol > 8.0:
            score += 2
        
        # Mapping Score to %
        risk_map = {
            -1: 1, 0: 2, 1: 3, 2: 4, 3: 6,
            4: 10, 5: 15, 6: 22, 7: 30, 8: 45, 9: 60
        }
        
        final_risk = risk_map.get(score, 70 if score > 9 else 1)
        
        if final_risk >= 20:
            label = "خطر بالا"
            color = "badge-danger"
        elif final_risk >= 10:
            label = "خطر متوسط"
            color = "badge-warning"
        else:
            label = "کم خطر"
            color = "badge-success"
        
        return {
            'value': final_risk,
            'label': label,
            'color': color,
            'description': f'احتمال {final_risk}% بروز سکته قلبی یا مغزی در ۱۰ سال آینده.'
        }
    
    def calculate_diabetes_risk(self) -> Dict[str, Any]:
        """Calculate Diabetes Risk (Hybrid Model)."""
        # A) Clinical Status
        if (self.a1c and self.a1c >= 6.5) or (self.fbs and self.fbs >= 126):
            return {
                'value': 100,
                'label': "دیابت تشخیص داده شده",
                'color': "badge-danger",
                'description': "نتایج آزمایش نشان‌دهنده دیابت است."
            }
        
        prob = None
        
        # B) Framingham Offspring Simplified (Using A1C)
        if self.a1c:
            intercept = -6.322
            logit = intercept + (0.584 * self.a1c) + (0.015 * self.age) + (0.009 * self.bmi) + (0.495 * self.family_hist)
            prob = 1 / (1 + math.exp(-logit))
        # C) Stern Model (Using FBS & HDL)
        elif self.fbs and self.hdl:
            intercept = -13.415
            logit = intercept + (0.028 * self.age) + (0.661 * self.gender) + (0.412 * self.family_hist) + (0.058 * self.bmi) + (0.007 * self.sbp) + (0.098 * self.fbs) + (-0.018 * self.hdl)
            prob = 1 / (1 + math.exp(-logit))
        else:
            return self._missing_result("داده‌های قند خون (FBS/A1C) کافی نیست")
        
        percentage = round(prob * 1000) / 10
        
        if percentage >= 30:
            label = "خطر بالا"
            color = "badge-danger"
        elif percentage >= 15:
            label = "خطر متوسط (پیش‌دیابت)"
            color = "badge-warning"
        else:
            label = "کم خطر"
            color = "badge-success"
        
        return {
            'value': percentage,
            'label': label,
            'color': color,
            'description': f'احتمال {percentage}% ابتلا به دیابت نوع ۲ در ۸ سال آینده.'
        }
    
    def calculate_liver_risk(self) -> Dict[str, Any]:
        """Calculate Liver Risk (Fatty Liver Index - FLI)."""
        if not self.tg or not self.ggt:
            return self._missing_result("نیاز به تری‌گلیسرید و آنزیم GGT")
        
        ln_tg = math.log(self.tg) if self.tg > 0 else 0
        ln_ggt = math.log(self.ggt) if self.ggt > 0 else 0
        
        L = (0.953 * ln_tg) + (0.139 * self.bmi) + (0.718 * ln_ggt) + (0.053 * self.waist) - 15.745
        
        # Use numerically stable sigmoid function to avoid overflow
        # For large positive L: use 1 / (1 + exp(-L))
        # For large negative L: use exp(L) / (1 + exp(L))
        if L > 700:  # exp(700) would overflow, so result is essentially 1
            fli_score = 100.0
        elif L < -700:  # exp(-700) is essentially 0, so result is essentially 0
            fli_score = 0.0
        elif L > 0:
            # For positive L, use 1 / (1 + exp(-L)) which is more stable
            fli_score = (1.0 / (1.0 + math.exp(-L))) * 100
        else:
            # For negative L, use exp(L) / (1 + exp(L))
            exp_L = math.exp(L)
            fli_score = (exp_L / (1.0 + exp_L)) * 100
        
        final_score = round(fli_score * 10) / 10
        
        if final_score >= 60:
            label = "کبد چرب (احتمال بالا)"
            color = "badge-danger"
        elif final_score >= 30:
            label = "هشدار (بینابینی)"
            color = "badge-warning"
        else:
            label = "سالم"
            color = "badge-success"
        
        return {
            'value': final_score,
            'label': label,
            'color': color,
            'description': f'شاخص FLI شما {final_score} است. نمره بالای ۶۰ نشان‌دهنده احتمال بالای کبد چرب است.'
        }
    
    def _missing_result(self, msg: str) -> Dict[str, Any]:
        return {
            'value': None,
            'label': "داده ناکافی",
            'color': "badge-secondary",
            'description': msg
        }
    
    def get_report(self) -> Dict[str, Any]:
        """Get all risk calculations."""
        return {
            'cardio': self.calculate_cvd_risk(),
            'diabetes': self.calculate_diabetes_risk(),
            'liver': self.calculate_liver_risk()
        }

