import hashlib
import random
from functools import reduce

from django.shortcuts import render, redirect, get_object_or_404
from django.views.decorators.csrf import csrf_exempt
from rest_framework.decorators import api_view
from django.contrib.auth.decorators import login_required
from django.contrib.auth.models import Group
from django.contrib.auth.models import User as DjangoUser
from django.contrib.auth import logout, authenticate
from django.contrib.auth import login as django_login
from django.http import HttpResponse, JsonResponse
from datetime import timedelta, datetime, time
from django.contrib import messages
from django.core.cache import cache
from django.db import transaction

from APIs.views.Errors import Errors
from APIs.views.firebase import send_notification_note
from panel.models import *
from payment.models import ZarinPalPayment
from ziluck_project.settings import STATIC_ROOT
import os
from APIs.models import User as AppUser, Sleep, Walking, UserQuestionnaire, MicrobiomeQuestionnaire, Profile, Eating, User, ZiLuckGroup, Points, \
    DietTemplate, FoodCard, RequestedFoods, Activities_log, Water, Activity, LifestyleQuestionnaire, Food
from APIs.models import Drug, SugarMeasurement, Weights, Diet
from APIs.models import InsulinTypeSelection
from django.db.models import Q
import jdatetime

from .tagging import process_questionnaire, generate_health_tags
from .utility import *
from panel.SMSNotification import *
from panel.PushNotification import PushNotification
from django.utils.safestring import mark_safe
from django.utils import translation
from django.conf import settings
from django.urls import reverse
from rest_framework.response import Response
from django.core.paginator import Paginator, EmptyPage, PageNotAnInteger
from django.template.loader import render_to_string
from APIs.forms import FoodForm, ActivityForm
import traceback
from collections import defaultdict
from django.forms.models import model_to_dict
from panel.tagging import all_alergic_tags
from django.db.models import Q

# Enhanced Column-level uniqueness for Refresh Cell
import re
import unicodedata
import string as _string
import logging
import hashlib
from datetime import datetime

# ============================================================================
# HELPER FUNCTIONS FOR REFRESH CELL FUNCTIONALITY
# ============================================================================

def _normalize_text(value):
    """Normalize Persian/Arabic text for consistent comparison."""
    if value is None:
        return ''
    
    s = str(value).strip()
    if not s:
        return ''
    
    # Persian/Arabic character normalization
    persian_arabic_map = {
        'ك': 'ک', 'ي': 'ی', 'ء': '', 'أ': 'ا', 'إ': 'ا', 'آ': 'ا', 'ة': 'ه'
    }
    
    for arabic, persian in persian_arabic_map.items():
        s = s.replace(arabic, persian)
    
    # Remove directional marks
    s = re.sub(r'[\u200C-\u200F\u202A-\u202E\u2066-\u2069\u061C]', '', s)
    
    # Unicode normalization and diacritic removal
    s = ''.join(ch for ch in unicodedata.normalize('NFD', s) 
                if unicodedata.category(ch) != 'Mn')
    
    # Normalize whitespace
    s = re.sub(r'\s+', ' ', s).strip()
    
    # Convert to lowercase
    s = s.lower()
    
    # Strip common punctuation
    punctuation_chars = _string.punctuation + '،؛؟«»""''·—–-٪٫٬'
    s = s.strip(punctuation_chars)
    
    return s


def _card_identities(card):
    """Extract normalized FA and EN identities from a card."""
    fa_norm = _normalize_text(card.FA_Name) if card.FA_Name else ''
    en_norm = _normalize_text(card.EN_Name) if card.EN_Name else 'unknown'
    return fa_norm, en_norm


def _create_identity_slug(fa_name, en_name):
    """Create a unique slug combining FA and EN names."""
    fa_slug = fa_name.replace(' ', '_') if fa_name else ''
    en_slug = en_name.replace(' ', '_') if en_name else 'unknown'
    return f"{fa_slug}_{en_slug}"


def _filter_candidates(cards, strict=True, debug_name="candidates", 
                       used_identities=None, used_slugs=None, 
                       current_fa_norm=None, current_en_norm=None, 
                       current_slug=None, current_code=None):
    """
    Filter candidate cards to remove exact duplicates.
    
    Args:
        cards: QuerySet or list of FoodCard objects
        strict: If True, apply all uniqueness checks
        debug_name: Name for debugging purposes
        used_identities: Set of used identity strings
        used_slugs: Set of used slug strings  
        current_fa_norm: Current card's normalized FA name
        current_en_norm: Current card's normalized EN name
        current_slug: Current card's slug
        current_code: Current card's main_dish_code
    """
    logger = logging.getLogger(__name__)
    
    # Handle both QuerySet and list
    cards_list = list(cards) if hasattr(cards, 'exists') else cards
    if not cards_list:
        return []
    
    result = []
    duplicates_found = []
    same_as_current_found = []
    
    # Initialize sets if not provided
    used_identities = used_identities or set()
    used_slugs = used_slugs or set()
    
    for candidate_card in cards_list:
        fa, en = _card_identities(candidate_card)
        slug = _create_identity_slug(fa, en)
        
        # STRICT DUPLICATE CHECKING: Check if this candidate duplicates any card in column
        if strict:
            is_duplicate = (
                (fa and fa in used_identities) or 
                (en and en in used_identities) or
                (slug and slug in used_slugs)
            )
            if is_duplicate:
                duplicates_found.append({
                    'code': candidate_card.main_dish_code,
                    'fa': fa, 'en': en, 'slug': slug,
                    'original_fa': candidate_card.FA_Name,
                    'original_en': candidate_card.EN_Name
                })
                continue
        
        # CRITICAL: Always prevent selecting the exact same card as current
        is_same_as_current = (
            (current_fa_norm and fa == current_fa_norm) or 
            (current_en_norm and en == current_en_norm) or
            (current_slug and slug == current_slug) or
            (current_code and candidate_card.main_dish_code == current_code)
        )
        if is_same_as_current:
            same_as_current_found.append({
                'code': candidate_card.main_dish_code,
                'fa': fa, 'en': en, 'slug': slug,
                'original_fa': candidate_card.FA_Name,
                'original_en': candidate_card.EN_Name
            })
            continue
            
        result.append(candidate_card)
    
    # DEBUG LOGGING
    if duplicates_found:
        logger.debug(f"Filtered out {len(duplicates_found)} duplicates from {debug_name}: {duplicates_found[:3]}")
    if same_as_current_found:
        logger.debug(f"Filtered out {len(same_as_current_found)} same-as-current from {debug_name}: {same_as_current_found}")
        
    return result


def _select_card_with_variety(cards, context_data=None):
    """Select a card with controlled variety using hash-based selection."""
    logger = logging.getLogger(__name__)
    
    try:
        if not cards:
            return None
        if len(cards) == 1:
            return cards[0]
        
        import random
        
        # Build seed from context and current time
        seed_parts = []
        if context_data:
            seed_parts.extend([
                str(context_data.get('patient_id', 0)),
                str(context_data.get('meal_normalized', 'unknown')),
                str(context_data.get('cell_index', 0))
            ])
        
        current_time = datetime.now().strftime("%Y%m%d%H%M%S")
        seed_parts.append(current_time)
        
        # Generate hash-based index
        seed_string = '_'.join(seed_parts)
        hash_digest = hashlib.md5(seed_string.encode()).hexdigest()
        hash_int = int(hash_digest[:8], 16)
        
        sorted_cards = sorted(cards, key=lambda c: getattr(c, 'id', 0) or 0)
        selected_index = hash_int % len(sorted_cards)
        selected_card = sorted_cards[selected_index]
        
        logger.info(f"🎲 CARD SELECTION: index {selected_index}/{len(cards)}, "
                   f"card {selected_card.main_dish_code} - '{selected_card.FA_Name}'")
        
        return selected_card
        
    except Exception as e:
        logger.error(f"Error in card selection: {e}")
        try:
            import random
            return random.choice(cards)
        except Exception:
            return cards[0] if cards else None


def generate_error_page(request, error_message):
    # Get notifications for base template
    try:
        zipped_data, total_new_messages, free_time = new_messages(request)
        message = list(zipped_data) if zipped_data else []
        total_notifications = len(free_time)
    except:
        message = []
        total_new_messages = 0
        free_time = []
        total_notifications = 0
    
    context = {
        'message': message,  # For base template notifications
        'error_message': error_message,  # The actual error message
        'total_new_messages': total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }
    return render(request=request, template_name='assistant/error.html', context=context)


def new_messages(request):
    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)
    user = request.user
    patients_list = AppUser.objects.filter(profile__therapist=user)
    assistants = Assistant.objects.filter(django_user=user)

    today = date.today()
    t1 = datetime(year=today.year, month=today.month, day=today.day, hour=0, minute=0, second=0)
    t2 = datetime(year=today.year, month=today.month, day=today.day, hour=23, minute=59, second=59)
    t3 = t1 + timedelta(hours=47, minutes=59, seconds=59)
    free_time = TimeSlot.objects.filter(therapist_django_user=user, start_time__range=(t1, t3), is_available=0)

    messages = []

    qset = AppUser

    for assistant in assistants:
        qset = AppUser.objects.filter(profile__therapist=assistant.django_user)
        patients_list = patients_list.union(qset)
    patients_list = list(patients_list)
    number_of_new_messages = []

    # print(f"This is the patient list element one: {patients_list[2]}")

    for p in patients_list:
        temp = Message.objects.filter((Q(sender=p.django_user) & Q(has_been_seen=False)))
        number_of_new_messages.append(len(temp))
        for i in range(len(temp)):
            sender_name = User.objects.filter(django_user=temp[i].sender)[0].profile.first_name
            sender_id = User.objects.filter(django_user=temp[i].sender)[0].id

            # print(f"this is the message sender id 1: {temp[i].sender}")

            messages.append(
                {
                    'text': temp[i].text,
                    'sender': temp[i].sender,
                    'time': temp[i].time,
                    "sender_name": sender_name,
                    "sender_id": sender_id
                }
            )
    max_length = max(len(messages), len(number_of_new_messages))
    patients_list += [None] * (max_length - len(patients_list))
    number_of_new_messages += [None] * (max_length - len(number_of_new_messages))
    zipped_data = zip(patients_list, messages, number_of_new_messages)
    total_new_messages = len(messages)

    return zipped_data, total_new_messages, free_time


@api_view(["GET"])
def patients_archive(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)

    user = request.user
    patients_list = AppUser.objects.filter(profile__therapist=user)
    assistant = Assistant.objects.get(django_user=user)
    time_login = datetime.now()
    assistant.last_login = time_login
    assistant.save()

    now = datetime.now().date()
    three_days_ago = (datetime.now() - timedelta(days=3)).date()
    new_sub = []
    number_of_new_messages = []
    total_of_messages = []
    subscription_status = []
    point_user = []
    user_groups = []
    discounts = []

    # Define groups to exclude
    excluded_groups = ["Patient", "Doctor", "Doctors", "Manager", "Assistant"]

    # Prepare data structure to hold group information for each patient
    django_groups = request.user.groups.filter(~Q(name='Assistant'))
    group_map = {}
    for patient in patients_list:
        # Exclude the unwanted groups
        patient_groups = patient.django_user.groups.exclude(name__in=excluded_groups)
        group_map[patient.django_user.id] = [group.name for group in patient_groups]

    # Filter patients without premium subscription
    non_premium_patients = []
    for p in patients_list:
        app_user = p
        django_user = app_user.django_user
        temp2 = TherapistSelection.objects.filter(user=django_user)
        try:
            payment_qs = ZarinPalPayment.objects.filter(user=app_user)
            if payment_qs.exists():
                payment = payment_qs.last()
                if payment.discount:
                    discounts.append(payment.discount.code)
                else:
                    discounts.append('None')
            else:
                discounts.append('None')
        except Exception as e:
            # logger.error(f"Error fetching discount for user {app_user.id}: {e}")
            discounts.append('Error')

        # Check if the patient does not have a premium subscription
        if not temp2.exists() or not temp2.last().has_credit():
            non_premium_patients.append(p)
            subscription_status.append(False)
            new_sub.append(False)

            # Collect points for the patient
            point = Points.objects.filter(user=p, date=now)
            if not point.exists():
                point_user.append(0)
            else:
                point_user.append(round(point[0].total_points))

            # Collect message data for the patient
            temp = Message.objects.filter((Q(sender=p.django_user) & Q(has_been_seen=False)))
            total_messages = Message.objects.filter(sender=p.django_user)
            number_of_new_messages.append(len(temp))
            total_of_messages.append(len(total_messages))

            # Collect group information for each patient
            user_groups.append(", ".join(group_map.get(p.django_user.id, ["---"])))

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'patients_list': zip(non_premium_patients, number_of_new_messages, subscription_status, new_sub, point_user,
                             total_of_messages, user_groups, discounts),
        'user': assistant,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
        'number_of_new_messages': number_of_new_messages,
    }

    return render(request, template_name='assistant/patients_archive.html', context=context)


@api_view(["GET"])
def patients(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)

    user = request.user
    patients_list = AppUser.objects.filter(profile__therapist=user)
    assistant = Assistant.objects.get(django_user=user)
    time_login = datetime.now()
    assistant.last_login = time_login
    assistant.save()

    now = datetime.now().date()
    three_days_ago = (datetime.now() - timedelta(days=3)).date()

    new_sub = []
    number_of_new_messages = []
    total_of_messages = []
    subscription_status = []
    point_user = []
    user_groups = []
    discounts = []
    diet_date = []
    plan_types = []

    # Define groups to exclude
    excluded_groups = ["Patient", "Doctor", "Doctors", "Manager", "Assistant"]

    # Prepare data structure to hold group information for each patient
    django_groups = request.user.groups.filter(~Q(name='Assistant'))
    group_map = {}
    for patient in patients_list:
        # Exclude the unwanted groups
        patient_groups = patient.django_user.groups.exclude(name__in=excluded_groups)
        group_map[patient.django_user.id] = [group.name for group in patient_groups]

    # Filter patients with premium subscription
    premium_patients = []
    for p in patients_list:
        app_user = p
        django_user = app_user.django_user
        temp2 = TherapistSelection.objects.filter(user=django_user)

        # Check if the patient has a premium subscription
        if temp2.exists() and temp2.last().has_credit():
            # Handle discounts and plan types
            try:
                # Define plan type mapping
                plan_type_map = {
                    1: 'Silver', 
                    2: 'Gold', 
                    3: 'Platinum', 
                    4: 'Silver',
                    5: 'Gold',
                    6: 'Bronze',
                    7: 'Bronze',
                    8: 'Bronze'
                }
                
                plan_type_found = False
                
                # First, check ZarinPalPayment for plan type
                payment_qs = ZarinPalPayment.objects.filter(user=app_user)
                if payment_qs.exists():
                    payment = payment_qs.last()
                    if payment.discount:
                        discounts.append(payment.discount.code)
                    else:
                        discounts.append('None')
                    if payment.plan:
                        plan_types.append(plan_type_map.get(payment.plan, 'Unknown'))
                        plan_type_found = True
                else:
                    discounts.append('None')
                
                # If not found in ZarinPalPayment, check Purchase/Shop model
                if not plan_type_found:
                    from APIs.models import Purchase, Shop
                    purchase_qs = Purchase.objects.filter(
                        user=app_user, 
                        shop_item__service_type='plan'
                    ).select_related('shop_item').order_by('-purchase_date')
                    
                    if purchase_qs.exists():
                        purchase = purchase_qs.first()
                        if purchase.shop_item.plan_type:
                            plan_types.append(plan_type_map.get(purchase.shop_item.plan_type, 'Unknown'))
                            plan_type_found = True
                
                # If still not found, append default
                if not plan_type_found:
                    plan_types.append('---')
                    
            except Exception as e:
                # Log the exception and append a fallback value
                # logger.error(f"Error fetching discount for user {app_user.id}: {e}")
                discounts.append('Error')
                plan_types.append('---')

            # Append to premium_patients and other lists
            premium_patients.append(p)
            ts = temp2.last()
            subscription_status.append(ts.remaining_credit())

            if ts.created_at.date() >= three_days_ago:
                new_sub.append(True)
            else:
                new_sub.append(False)

            # Collect points for the patient
            point = Points.objects.filter(user=p, date=now)
            if not point.exists():
                point_user.append(0)
            else:
                point_user.append(round(point[0].total_points))

            diet = Diet.objects.filter(user=p)
            if diet.exists():
                diet_date.append(diet.last().from_date)
            else:
                diet_date.append('None')
            # Collect message data for the patient
            temp = Message.objects.filter(Q(sender=p.django_user) & Q(has_been_seen=False))
            total_messages = Message.objects.filter(sender=p.django_user)
            number_of_new_messages.append(len(temp))
            total_of_messages.append(len(total_messages))

            # Collect group information for each patient
            user_groups.append(", ".join(group_map.get(p.django_user.id, ["---"])))

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'patients_list': zip(premium_patients, number_of_new_messages, subscription_status, new_sub, point_user,
                             total_of_messages, user_groups, discounts, diet_date, plan_types),
        'user': assistant,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
        'number_of_new_messages': number_of_new_messages,
    }

    return render(request, template_name='assistant/patients.html', context=context)


@api_view(["GET"])
def toggle_favorite_user(request, user_id):
    try:
        if not request.user.is_authenticated:
            return redirect('panel-login')

        user = get_object_or_404(User, id=user_id)
        if user.is_favorite:
            user.is_favorite = False
            user.save()
            return redirect('panel-assistant-patients')
        else:
            user.is_favorite = True
            user.save()
            return redirect('panel-assistant-patients-favorites')
    except Exception as e:
        return redirect('panel-assistant-patients')


@api_view(["GET"])
def patients_favorites(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)

    user = request.user
    # Filter only favorite patients
    patients_list = AppUser.objects.filter(profile__therapist=user, django_user__user__is_favorite=True)
    assistant = Assistant.objects.get(django_user=user)
    time_login = datetime.now()
    assistant.last_login = time_login
    assistant.save()

    now = datetime.now().date()
    three_days_ago = (datetime.now() - timedelta(days=3)).date()
    five_days_ago = (datetime.now() - timedelta(days=5)).date()
    six_day_ago = (datetime.now() - timedelta(days=6)).date()
    seven_day_ago = (datetime.now() - timedelta(days=7)).date()
    new_sub = []
    number_of_new_messages = []
    total_of_messages = []
    subscription_status = []
    point_user = []
    user_groups = []
    diet_date = []
    color = []

    # Define groups to exclude
    excluded_groups = ["Patient", "Doctor", "Doctors", "Manager", "Assistant"]

    # Prepare data structure to hold group information for each patient
    django_groups = request.user.groups.filter(~Q(name='Assistant'))
    group_map = {}
    for patient in patients_list:
        # Exclude the unwanted groups
        patient_groups = patient.django_user.groups.exclude(name__in=excluded_groups)
        group_map[patient.django_user.id] = [group.name for group in patient_groups]

    premium_patients = []
    non_premium_patients = []
    for p in patients_list:
        app_user = p
        django_user = app_user.django_user
        temp2 = TherapistSelection.objects.filter(user=django_user)

        # Check if the patient has a premium subscription
        if temp2.exists() and temp2.last().has_credit():
            premium_patients.append(p)
            ts = temp2.last()
            subscription_status.append(ts.remaining_credit())

            if ts.created_at.date() >= three_days_ago:
                new_sub.append(True)
            else:
                new_sub.append(False)

            # Collect points for the patient
            point = Points.objects.filter(user=p, date=now)
            if not point.exists():
                point_user.append(0)
            else:
                point_user.append(round(point[0].total_points))

            diet = Diet.objects.filter(user=p)
            if diet.exists():
                diet_date.append(diet.last().from_date)
                if diet.last().from_date > five_days_ago:
                    color.append('green')
                elif six_day_ago < diet.last().from_date <= five_days_ago:
                    color.append('yellow')
                elif seven_day_ago < diet.last().from_date <= six_day_ago:
                    color.append('yellow')
                else:
                    color.append('red')
            else:
                diet_date.append('None')
                color.append('gray')

            # Collect message data for the patient
            temp = Message.objects.filter((Q(sender=p.django_user) & Q(has_been_seen=False)))
            total_messages = Message.objects.filter(sender=p.django_user)
            number_of_new_messages.append(len(temp))
            total_of_messages.append(len(total_messages))

            # Collect group information for each patient
            user_groups.append(", ".join(group_map.get(p.django_user.id, ["---"])))

        elif not temp2.exists() or not temp2.last().has_credit():
            non_premium_patients.append(p)
            subscription_status.append(False)
            new_sub.append(False)

            # Collect points for the patient
            point = Points.objects.filter(user=p, date=now)
            if not point.exists():
                point_user.append(0)
            else:
                point_user.append(round(point[0].total_points))

            diet_date.append('None')
            color.append('gray')
            # Collect message data for the patient
            temp = Message.objects.filter((Q(sender=p.django_user) & Q(has_been_seen=False)))
            total_messages = Message.objects.filter(sender=p.django_user)
            number_of_new_messages.append(len(temp))
            total_of_messages.append(len(total_messages))

            # Collect group information for each patient
            user_groups.append(", ".join(group_map.get(p.django_user.id, ["---"])))

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'patients_list': zip(patients_list, number_of_new_messages, subscription_status, new_sub, point_user,
                             total_of_messages, user_groups, diet_date, color),
        'user': assistant,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
        'number_of_new_messages': number_of_new_messages,
    }

    return render(request, template_name='assistant/patients_favorites.html', context=context)


def get_day_label(delta_days):
    """
    Return a Farsi label (e.g. 'امروز', 'دیروز', etc.) based on the number of days ago.
    Adjust or extend as needed up to 6+ days.
    """
    if delta_days == 0:
        return "امروز"
    elif delta_days == 1:
        return "دیروز"
    elif delta_days == 2:
        return "دو روز پیش"
    elif delta_days == 3:
        return "سه روز پیش"
    elif delta_days == 4:
        return "چهار روز پیش"
    elif delta_days == 5:
        return "پنج روز پیش"
    elif delta_days == 6:
        return "شش روز پیش"
    # Fallback for day 7 or more
    return f"{delta_days} روز پیش"


# For mapping meal integers to Persian labels:
meal_labels = {
    0: "صبحانه",
    1: "ناهار",
    2: "شام",
    3: "میان وعده ۱",
    4: "میان وعده ۲",
}


@api_view(["GET"])
def show_patient(request, patient_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")

    temp = AppUser.objects.filter(id=patient_id)
    if not temp.exists():
        return generate_error_page(request, 'invalid request')
    patient = temp[0]
    assistant = Assistant.objects.get(django_user=django_user)
    if patient.profile.therapist != assistant.django_user:
        return generate_error_page(request, "you have not access to this patient!")

    # drugs = []
    # drugs_ids = getIntList(patient.profile.pills)
    # for id in drugs_ids:
    #     temp = Drug.objects.filter(DataBaseNumber=id)
    #     if temp.exists():
    #         drugs.append(temp[0])

    medical_notes = MedicalNote.objects.filter(patient=patient).order_by('-time')
    daily_notes = DailyNote.objects.filter(patient=patient).order_by('-time')
    microbiome_analysis = MicrobiomeAnalysis.objects.filter(user=patient).last()
    if microbiome_analysis:
        analysis_fields = {
            'Carbohydrate Metabolism': {
                'current': microbiome_analysis.carbohydrate_metabolism_current,
                'avg': microbiome_analysis.carbohydrate_metabolism_avg
            },
            'Metabolic Score': {
                'current': microbiome_analysis.metabolic_score_current,
                'avg': microbiome_analysis.metabolic_score_avg
            },
            'Protein Metabolism': {
                'current': microbiome_analysis.protein_metabolism_current,
                'avg': microbiome_analysis.protein_metabolism_avg
            },
            'Fat Metabolism': {
                'current': microbiome_analysis.fat_metabolism_current,
                'avg': microbiome_analysis.fat_metabolism_avg
            },
            'Sleep Quality': {
                'current': microbiome_analysis.sleep_quality_current,
                'avg': microbiome_analysis.sleep_quality_avg
            },
            'Lactose Sensitivity': {
                'current': microbiome_analysis.lactose_sensitivity_current,
                'avg': microbiome_analysis.lactose_sensitivity_avg
            },
            'Vitamin Synthesis': {
                'current': microbiome_analysis.vitamin_synthesis_current,
                'avg': microbiome_analysis.vitamin_synthesis_avg
            },
            'Processed Food Index': {
                'current': microbiome_analysis.processed_food_index_current,
                'avg': microbiome_analysis.processed_food_index_avg
            },
            'Sugar Index': {
                'current': microbiome_analysis.sugar_index_current,
                'avg': microbiome_analysis.sugar_index_avg
            },
            'Autoimmune Index': {
                'current': microbiome_analysis.autoimmune_index_current,
                'avg': microbiome_analysis.autoimmune_index_avg
            },
            'Bowel Mobility': {
                'current': microbiome_analysis.bowel_mobility_current,
                'avg': microbiome_analysis.bowel_mobility_avg
            },
            'Gluten Sensitivity': {
                'current': microbiome_analysis.gluten_sensitivity_current,
                'avg': microbiome_analysis.gluten_sensitivity_avg
            },
            'Antibiotic Damage': {
                'current': microbiome_analysis.antibiotic_damage_current,
                'avg': microbiome_analysis.antibiotic_damage_avg
            }
        }

        # Categorize the fields into Excellent, Moderate, and Poor based on the 'current' score
        excellent_scores = {name: values for name, values in analysis_fields.items() if values['current'] > 70}
        moderate_scores = {name: values for name, values in analysis_fields.items() if 30 <= values['current'] <= 70}
        poor_scores = {name: values for name, values in analysis_fields.items() if values['current'] < 30}
    else:
        # If no data exists, set empty scores or default values
        excellent_scores = {}
        moderate_scores = {}
        poor_scores = {}

    # Eating log -------------------------------------------------------------------
    today = date.today()
    y1 = datetime.now() - timedelta(days=1)
    y2 = datetime.now() - timedelta(days=2)
    y3 = datetime.now() - timedelta(days=3)
    y4 = datetime.now() - timedelta(days=4)
    y5 = datetime.now() - timedelta(days=5)
    y6 = datetime.now() - timedelta(days=6)
    y7 = datetime.now() + timedelta(days=1)

    t1 = datetime(year=today.year, month=today.month, day=today.day, hour=0, minute=0, second=0)
    t2 = datetime(year=today.year, month=today.month, day=today.day, hour=23, minute=59, second=59)
    d1 = t1.date()
    d2 = t2.date()
    end_date = datetime.now().date()
    start_date = end_date - timedelta(days=6)  # Calculate 6 days back for a total of 7 days

    eaten_list = []
    eaten_list_week = []
    total_protein = 0
    total_fat = 0
    total_calorie = 0

    days = (d2 - d1).days + 1
    data = patient.calculate_cr_macro_nutrients_distribution()
    meal = 0
    meal_soboone = True
    meal_nahar = True
    meal_sham = True
    meal_snack1 = True
    meal_snack2 = True

    meal_soboone_week = True
    meal_nahar_week = True
    meal_sham_week = True
    meal_snack1_week = True
    meal_snack2_week = True

    y01 = True
    y11 = True
    y21 = True
    y31 = True
    y41 = True
    y51 = True
    y61 = True

    def getList(strList):
        S = strList.split("&")
        last = len(S) - 1
        F = []
        for i in range(1, last):
            if not S[i] == "":
                F.append(float(S[i]))
        return F

    def listToStr(list):
        S = "&"
        for item in list:
            S += str(item) + "&"
        return S

    Carbohydrates_distribution_str = listToStr(data["carbohydrate_distribution_g"])
    patient.profile.Carbohydrates_distribution_list_g = Carbohydrates_distribution_str
    Carbohydrates_list_to_use = getList(patient.profile.Carbohydrates_distribution_list_g)
    Carbohydrates_list_to_use = list(
        map(lambda x: days * x, getList(patient.profile.Carbohydrates_distribution_list_g)))
    Carbohydrates = sum(Carbohydrates_list_to_use)
    totalcarb = 0

    for i in range(5):
        temp = Eating.objects.filter(user=patient, date_time__range=(t1, t2), meal=i)

        for ii in range(temp.count()):
            totalcarb += round((temp[ii].food.Carbohydrates_g * temp[ii].amount) / 100, 2)
    Protein_to_use = patient.profile.Protein_g * days
    Fat_to_use = patient.profile.Fat_g * days

    eat = Eating.objects.filter(user=patient, date_time__range=(t1, t2))
    eat_foods = Eating.objects.filter(user=patient, date_time__range=(start_date, t2)).order_by('-date_time')
    day_map = defaultdict(lambda: {
        'date': None,
        'foods_by_meal': defaultdict(list),
    })

    for record in eat_foods:
        food_date = record.date_time.date()
        meal_key = record.meal
        day_data = day_map[food_date]
        if day_data['date'] is None:
            day_data['date'] = food_date
        day_data['foods_by_meal'][meal_key].append(record)

    day_list = list(day_map.values())
    day_list.sort(key=lambda x: x['date'], reverse=True)
    
    # Convert defaultdict to regular dict for template compatibility
    for day_item in day_list:
        day_item['foods_by_meal'] = dict(day_item['foods_by_meal'])

    # Calculate nutritional totals for each day and add Persian date
    for item in day_list:
        delta = (today - item['date']).days
        item['day_label'] = get_day_label(delta)
        item['jalali_date'] = jdatetime.date.fromgregorian(date=item['date']).strftime('%Y/%m/%d')
        
        # Calculate daily totals
        daily_protein = 0
        daily_fat = 0
        daily_carbs = 0
        daily_calories = 0
        
        for meal_records in item['foods_by_meal'].values():
            for record in meal_records:
                if record.food:
                    daily_protein += (record.food.Protein_g * record.amount) / 100
                    daily_fat += (record.food.Fat_g * record.amount) / 100
                    daily_carbs += (record.food.Carbohydrates_g * record.amount) / 100
                    daily_calories += (record.food.Calories * record.amount) / 100
        
        item['daily_protein'] = round(daily_protein, 1)
        item['daily_fat'] = round(daily_fat, 1)
        item['daily_carbs'] = round(daily_carbs, 1)
        item['daily_calories'] = round(daily_calories, 1)
    
    # Prepare chart data for 7 days and 30 days
    def get_nutrition_chart_data(days):
        """Get nutritional data for chart visualization"""
        chart_start_date = end_date - timedelta(days=days-1)
        chart_data = []
        
        for i in range(days):
            current_date = chart_start_date + timedelta(days=i)
            day_protein = 0
            day_fat = 0
            day_carbs = 0
            day_calories = 0
            
            # Find eating records for this date
            day_records = Eating.objects.filter(
                user=patient,
                date_time__date=current_date
            )
            
            for record in day_records:
                if record.food:
                    day_protein += (record.food.Protein_g * record.amount) / 100
                    day_fat += (record.food.Fat_g * record.amount) / 100
                    day_carbs += (record.food.Carbohydrates_g * record.amount) / 100
                    day_calories += (record.food.Calories * record.amount) / 100
            
            jalali_date = jdatetime.date.fromgregorian(date=current_date).strftime('%m/%d')
            
            chart_data.append({
                'date': current_date.strftime('%Y-%m-%d'),
                'jalali_date': jalali_date,
                'protein': round(day_protein, 1),
                'fat': round(day_fat, 1),
                'carbs': round(day_carbs, 1),
                'calories': round(day_calories, 1)
            })
        
        return chart_data
    
    # Get chart data for both periods
    chart_data_7days = get_nutrition_chart_data(7)
    chart_data_30days = get_nutrition_chart_data(30)
    
    print(f"[Chart Data] 7 days: {len(chart_data_7days)} days")
    print(f"[Chart Data] 30 days: {len(chart_data_30days)} days")
    if chart_data_7days:
        print(f"[Chart Data] Sample day: {chart_data_7days[0]}")
    
    # Prepare JSON for template
    import json
    chart_data_7days_json = json.dumps(chart_data_7days)
    chart_data_30days_json = json.dumps(chart_data_30days)
    
    print(f"[Chart Data] JSON lengths - 7d: {len(chart_data_7days_json)}, 30d: {len(chart_data_30days_json)}")

    for i in range(len(eat)):
        # Calculate the protein, fat, and calorie for each food item
        protein = (eat[i].food.Protein_g * eat[i].amount) / 100
        fat = (eat[i].food.Fat_g * eat[i].amount) / 100
        calorie = (eat[i].food.Calories * eat[i].amount) / 100

        # Add the calculated values to the totals
        total_protein += protein
        total_fat += fat
        total_calorie += calorie

        if eat[i].meal == 0 and meal_soboone:
            meal = 0
            meal_soboone = False
            eaten_list.append({'meal': meal})
        elif eat[i].meal == 1 and meal_nahar:
            meal = 1
            meal_nahar = False
            eaten_list.append({'meal': meal})
        elif eat[i].meal == 2 and meal_sham:
            meal = 2
            meal_sham = False
            eaten_list.append({'meal': meal})
        elif eat[i].meal == 3 and meal_snack1:
            meal = 3
            meal_snack1 = False
            eaten_list.append({'meal': meal})
        elif eat[i].meal == 4 and meal_snack2:
            meal = 4
            meal_snack2 = False
            eaten_list.append({'meal': meal})
        # Append information about each food item to the list
        eaten_list.append(
            {
                'food': eat[i].food,
                'protein': round(protein, 2),
                'fat': round(fat, 2),
                'calorie': round(calorie, 2),
                # 'meal': meal
            }
        )
    # protein / fat / carbohydrate Score
    try:
        protein_score = (total_protein * 100) / Protein_to_use
    except Exception as e:
        protein_score = -1
    if total_protein > Protein_to_use:
        protein_score = 100
    try:
        fat_score = (total_fat * 100) / Fat_to_use
    except Exception as e:
        fat_score = -1
    if total_fat > Fat_to_use:
        fat_score = 100
    try:
        carbohydrate_score = (totalcarb * 100) / Carbohydrates
    except Exception as e:
        carbohydrate_score = -1
    if totalcarb > Carbohydrates:
        carbohydrate_score = 100

    # Eaten foods for a week
    # for i in range(len(eat_foods)):
    #
    #     food_item = {}
    #
    #     if eat_foods[i].date_time.date() == today and y01:
    #         y01 = False
    #         food_item['day'] = 0
    #     elif eat_foods[i].date_time.date() == y1.date() and y11:
    #         y11 = False
    #         food_item['day'] = 1
    #     elif eat_foods[i].date_time.date() == y2.date() and y21:
    #         y21 = False
    #         food_item['day'] = 2
    #     elif eat_foods[i].date_time.date() == y3.date() and y31:
    #         y31 = False
    #         food_item['day'] = 3
    #     elif eat_foods[i].date_time.date() == y4.date() and y41:
    #         y41 = False
    #         food_item['day'] = 4
    #     elif eat_foods[i].date_time.date() == y5.date() and y51:
    #         y51 = False
    #         food_item['day'] = 5
    #     elif eat_foods[i].date_time.date() == y6.date() and y61:
    #         y61 = False
    #         food_item['day'] = 6
    #
    #     if eat_foods[i].meal == 0:
    #         meal_soboone_week = False
    #         food_item['meal'] = 0
    #     elif eat_foods[i].meal == 2:
    #         meal_nahar_week = False
    #         food_item['meal'] = 2
    #     elif eat_foods[i].meal == 4:
    #         meal_sham_week = False
    #         food_item['meal'] = 4
    #     elif eat_foods[i].meal == 1 or eat_foods[i].meal == 3 or eat_foods[i].meal == 5:
    #         meal_mian_vade_week = False
    #         food_item['meal'] = 6
    #
    #     food_item['food'] = eat_foods[i].food
    #     food_item['time'] = eat_foods[i].date_time
    #
    #     # Append the food item to the list
    #     eaten_list_week.append(food_item)

    # Sleep_Duration log -------------------------------------------------------------------
    # Dictionary to map English day names to Persian
    day_of_week_persian = {
        "Monday": "دوشنبه",
        "Tuesday": "سه‌شنبه",
        "Wednesday": "چهارشنبه",
        "Thursday": "پنج‌شنبه",
        "Friday": "جمعه",
        "Saturday": "شنبه",
        "Sunday": "یک‌شنبه",
    }

    sleep_list = []

    # Fetch the sleep logs for the last 7 days, ordered by time descending (newest first)
    start_date = datetime.now().date() - timedelta(days=6)  # Last 7 days
    end_date = datetime.now().date()
    sleep = Sleep.objects.filter(
        user=patient,
        time__date__range=(start_date, end_date)
    ).order_by('-time')  # Changed to descending order

    h = 0  # Counter for sleep >= 8 hours
    low = 0  # Counter for sleep < 8 hours

    for s in sleep:
        # Convert Unix timestamps (start, end) to human-readable format
        sleep_start = datetime.fromtimestamp(s.start).strftime("%H:%M")  # HH:MM
        sleep_end = datetime.fromtimestamp(s.end).strftime("%H:%M")  # HH:MM

        # Calculate the sleep duration in minutes
        sleep_start_time = datetime.fromtimestamp(s.start)
        sleep_end_time = datetime.fromtimestamp(s.end)

        # Handle cases where sleep crosses midnight (next day)
        if sleep_end_time > sleep_start_time:
            sleep_duration_in_minutes = (sleep_end_time - sleep_start_time).total_seconds() // 60
        else:
            # If end time is before start time, it means the sleep crosses midnight
            sleep_duration_in_minutes = ((sleep_end_time + timedelta(days=1)) - sleep_start_time).total_seconds() // 60

        sleep_duration_hours = sleep_duration_in_minutes // 60
        sleep_duration_minutes = sleep_duration_in_minutes % 60
        sleep_duration = f"{int(sleep_duration_hours)}:{int(sleep_duration_minutes):02}"  # Format as HH:MM

        # Use the day from the start time for the sleep log
        sleep_day_english = sleep_start_time.strftime("%A")
        sleep_day_persian = day_of_week_persian[sleep_day_english]

        # Get the Persian date from s.time
        sleep_date = jdatetime.date.fromgregorian(date=s.time.date()).strftime('%Y/%m/%d')  # e.g., "1403/12/20"

        sleep_list.append({
            'start_time': sleep_start,
            'end_time': sleep_end,
            'duration': sleep_duration,
            'day': sleep_day_persian,
            'date': sleep_date
        })

        # Track sleep hours greater or less than 8 hours
        if sleep_duration_in_minutes >= (8 * 60):
            h += 1
        else:
            low += 1

    # Steps -------------------------------------------------------------------
    steps_list = []
    steps = Walking.objects.filter(user=patient).order_by("date")

    # max_steps_record = max(steps, key=lambda x: x.steps)
    step_index = 0
    if len(steps) > 0:
        step_index = len(steps) - 1
    if step_index != 0:
        steps_list.append({'steps': steps[step_index].steps, })
    else:
        steps_list.append({'steps': 0, })

    # Exercise -------------------------------------------------------------------
    exercise_logs = Activities_log.objects.filter(user=patient, start_date_time__range=(y6, y7))
    exercise = []
    for log in exercise_logs:
        start = log.start_date_time
        end = log.end_date_time
        duration_seconds = (end - start).total_seconds()
        hours = int(duration_seconds // 3600)
        minutes = int((duration_seconds % 3600) // 60)

        exercise.append({
            'activity': log.activity,
            'start_date_time': start,
            'end_date_time': end,
            'amount': log.amount,
            'point': log.point,
            'energy': log.energy,
            'hours': hours,
            'minutes': minutes,
        })

    # diseases_list ---------------------------------------------------------------
    diseases_list = patient.profile.diseases

    # if patient.profile.insulin_mixed_effect is None:
    #     if patient.profile.insulin_sensitivity != 0:
    #         insulin_dosage = round(1800 / patient.profile.insulin_sensitivity)
    #         insulin_dosage_per_weight_ratio = round(insulin_dosage / patient.profile.weight, 1)
    #     else:
    #         insulin_dosage = 'نا معین'
    #         insulin_dosage_per_weight_ratio = 'نا معین'
    # else:
    #     if patient.profile.insulin_sensitivity != 0:
    #         insulin_dosage = round(1500 / patient.profile.insulin_sensitivity)
    #         insulin_dosage_per_weight_ratio = round(insulin_dosage / patient.profile.weight, 1)
    #     else:
    #         insulin_dosage = 'نا معین'
    #         insulin_dosage_per_weight_ratio = 'نا معین'
    #
    # short_act_insulin_selections = InsulinTypeSelection.objects.filter(user=patient, insulin__EffectType=1).order_by(
    #     'start')
    # long_act_insulin_selections = InsulinTypeSelection.objects.filter(user=patient, insulin__EffectType=2).order_by(
    #     'start')

    weights = Weights.objects.filter(user=patient).order_by('time')
    weight = patient.profile.weight
    goal_weight = patient.profile.goal_weight
    weight_score = (goal_weight / weight) * 100

    diets = Diet.objects.filter(user=patient).order_by('-created_at')

    try:
        user_goal_weight = UserQuestionnaire.objects.filter(user=patient).last()
        if user_goal_weight and isinstance(user_goal_weight.QandA, dict):
            # Safely access the "IDEAL_WEIGHT" key
            ideal_weight = user_goal_weight.QandA.get("IDEAL_WEIGHT")
            if ideal_weight:
                # Ensure the value is a string and strip the brackets
                user_goal_weight = str(ideal_weight).strip("[]")
            else:
                user_goal_weight = None  # Handle missing or invalid "IDEAL_WEIGHT" gracefully
        else:
            user_goal_weight = None  # No valid UserQuestionnaire or invalid QandA
    except Exception as e:
        # Log the exception for debugging purposes
        user_goal_weight = None  # Default to None in case of error

    waters = Water.objects.filter(user=patient, time__range=(y6, y7))
    water = Water.objects.filter(user=patient, time__range=(t1, t2))
    water_logs = []
    water_log_today = 0
    for w in waters:
        item = {
            "time": w.time,
            "amount": w.amount,
            "point": w.point,
        }
        water_logs.append(item)
    for w in water:
        water_log_today = water_log_today + w.amount

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'user': assistant,
        'patient': patient,
        # 'drugs': drugs,
        # 'insulin_dosage': insulin_dosage,
        # 'insulin_dosage_per_weight_ratio': insulin_dosage_per_weight_ratio,
        'medical_notes': medical_notes,
        'daily_notes': daily_notes,
        'age': patient.profile.age,
        # 'detection_year': jdatetime.datetime.fromgregorian(datetime=patient.profile.birth_date).year,
        'BMI': round(patient.profile.weight / ((patient.profile.height / 100) ** 2), 1),
        # 'short_act_insulin_selections': short_act_insulin_selections,
        # 'long_act_insulin_selections': long_act_insulin_selections,
        'weights': weights,
        'weight': round(weight),
        'goal_weight': goal_weight,
        'weight_score': round(weight_score),
        'sleep_list': sleep_list,
        'sleep_high': h,
        'sleep_low': low,
        'steps': steps_list,
        'eating': eaten_list,
        'eating_week': eaten_list_week,
        'total_protein': round(total_protein, 2),
        'Protein_to_use': round(Protein_to_use, 2),
        'protein_score': round(protein_score),
        'total_fat': round(total_fat, 2),
        'Fat_to_use': round(Fat_to_use, 2),
        'fat_score': round(fat_score),
        'total_calorie': round(total_calorie, 2),
        'Carbohydrates_to_use': round(Carbohydrates, 2),
        'total_carbohydrate': round(totalcarb, 2),
        'carbohydrate_score': round(carbohydrate_score),
        'diseases_list': diseases_list,
        'diets': diets,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
        'exercise': exercise,
        'water_logs': water_logs,
        'water': water_log_today,
        'microbiome_analysis': microbiome_analysis,
        'excellent_scores': excellent_scores,
        'moderate_scores': moderate_scores,
        'poor_scores': poor_scores,
        'has_data': microbiome_analysis is not None,
        'user_goal_weight': user_goal_weight,
        "day_list": day_list,
        'meal_labels': meal_labels,
        'chart_data_7days': chart_data_7days_json,
        'chart_data_30days': chart_data_30days_json,
    }
    return render(request, template_name='assistant/patient_profile.html', context=context)


@api_view(["POST"])
def update_cr_coach(request, patient_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    django_user = request.user
    # Check if the user is an Assistant
    if not django_user.groups.filter(name='Assistant').exists():
        return Response({"success": False, "error": "You do not have access."}, status=403)

    # Validate the patient ID
    patient_qs = AppUser.objects.filter(id=patient_id)
    if not patient_qs.exists():
        return Response({"success": False, "error": "Invalid patient ID."}, status=400)

    patient = patient_qs.first()
    assistant = Assistant.objects.get(django_user=django_user)
    # Ensure the assistant has access to this patient
    if patient.profile.therapist != assistant.django_user:
        return Response({"success": False, "error": "You do not have access to this patient."}, status=403)

    # Get the new CR_coach value from the request
    cr_coach = request.POST.get("cr_coach")
    if not cr_coach:
        return Response({"success": False, "error": "No CR_coach value provided."}, status=400)

    try:
        cr_coach = float(cr_coach)  # Ensure the value is a valid float
    except ValueError:
        return Response({"success": False, "error": "Invalid CR_coach value."}, status=400)

    # Update the CR_coach field
    patient.profile.CR_coach = cr_coach
    patient.profile.save()

    return Response({"success": True, "new_cr_coach": cr_coach}, status=200)


@api_view(["GET"])
def show_patient_additional_information(request, patient_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")

    temp = AppUser.objects.filter(id=patient_id)
    if not temp.exists():
        return generate_error_page(request, 'invalid request')
    patient = temp[0]
    assistant = Assistant.objects.get(django_user=django_user)
    if patient.profile.therapist != assistant.django_user:
        return generate_error_page(request, "you have not access to this patient!")

    # Q & A -------------------------------------------------------------------
    QUESTION_ID_MAP = {
        "WATER_INTAKE": "آب مصرفی",
        "ALCOHOL_CONSUMPTION": "مصرف الکل",
        "TOBACCO_USAGE": "استفاده از تنباکو",
        "SWEETS_FREQUENCY": "تناوب استفاده شیرینی",
        "EXERCISE_FREQUENCY": "فعالیت بدنی",
        "IDEAL_WEIGHT": "وزن ایده‌آل",
        "HEALTH_GOAL": "هدف سلامتی",
        "ANTIBIOTICS_NSAIDS_USAGE": "مصرف آنتی‌بیوتیک‌ها و NSAID",
        "FOOD_LIMITING_MEDICATION": "داروهای محدودکننده غذا",
        "SUPPLEMENTS_MEDICATION": "مکمل‌ها و داروها",
        "PROBIOTICS_PREBIOTICS": "پروبیوتیک‌ها و پریبیوتیک‌ها",
        "HEALTH_CHANGES": "تغییرات سلامتی",
        "CONTACT_PREFERENCE": "ترجیح ارتباط",
        "CONTENT_FORMAT": "فرمت محتوا",
        "MICROBIOME_KNOWLEDGE": "دانش میکروبیوم",
        "CHRONIC_ILLNESSES": "بیماری‌های مزمن",
        "RESTRICTIVE_MEDICATIONS": "داروهای محدودکننده",
        "FOOD_RELATIONSHIP": "رابطه با غذا",
        "FERMENTED_FOODS_FREQUENCY": "تناوب مصرف غذاهای تخمیری",
        "HIGH_FIBER_PLANT_FOODS": "غذاهای گیاهی پرفیبر",
        "PROCESSED_FOODS_FREQUENCY": "تناوب مصرف غذاهای فرآوری‌شده",
        "MEDICAL_DIETARY_RESTRICTIONS": "محدودیت‌های غذایی پزشکی",
        "MAIN_MEALS_PER_DAY": "وعده‌های اصلی روزانه",
        "HOME_MADE_MEALS": "وعده‌های خانگی",
        "LIMITED_AVOIDED_FOODS": "غذاهای محدود یا اجتنابی",
        "SNACK_TYPE_PREFERENCE": "ترجیح نوع میان‌وعده",
        "SNACKS_PER_DAY": "تعداد میان‌وعده در روز",
        "MEAT_CONSUMPTION_FREQUENCY": "تناوب مصرف گوشت",
        "MEAT_PORTION_SIZE": "اندازه وعده گوشت",
        "EATING_PATTERNS": "الگوهای غذایی",
        "FOOD_ALLERGIES_INTOLERANCES": "حساسیت‌ها و عدم تحمل غذایی",
        "COOK_METHODS": "روش‌های پخت",
        "BREAKFAST_TIME": "زمان صبحانه",
        "BREAKFAST_OPTIONS": "گزینه‌های صبحانه",
        "BREAKFAST_BREAD_TYPE": "نوع نان صبحانه",
        "LUNCH_TIME": "زمان ناهار",
        "WORK_LUNCH_HABITS": "عادات ناهار در محل کار",
        "LUNCH_PREFERENCE": "ترجیح ناهار",
        "LUNCH_SIDE_DISH": "غذای جانبی ناهار",
        "DINNER_TIME": "زمان شام",
        "DINNER_PREFERENCE": "ترجیح شام",
        "DINNER_SIDE_DISH": "غذای جانبی شام",
        "MEAL_VARIETY": "تنوع غذایی",
        "BREAST_FEEDING_DURATION": "مدت شیردهی"
    }

    # Helper function to process questionnaire data
    def process_questionnaire_data(questionnaire_queryset):
        questions_list = []
        answers_list = []
        
        if questionnaire_queryset.exists():
            for questionnaire in questionnaire_queryset:
                qanda_dict = questionnaire.QandA
                if qanda_dict:
                    for key, value in qanda_dict.items():
                        translated_key = QUESTION_ID_MAP.get(key, key)
                        questions_list.append(translated_key)
                        answers_list.append(value)
        
        return list(zip(questions_list, answers_list))

    # Fetch all three types of questionnaires
    user_questionnaire = UserQuestionnaire.objects.filter(user=patient)
    microbiome_questionnaire = MicrobiomeQuestionnaire.objects.filter(user=patient)
    lifestyle_questionnaire = LifestyleQuestionnaire.objects.filter(user=patient)

    # Process each questionnaire type
    user_qanda_pairs = process_questionnaire_data(user_questionnaire)
    microbiome_qanda_pairs = process_questionnaire_data(microbiome_questionnaire)
    lifestyle_qanda_pairs = process_questionnaire_data(lifestyle_questionnaire)

    # For backward compatibility, keep the combined qanda_pairs
    qanda_pairs = user_qanda_pairs + microbiome_qanda_pairs + lifestyle_qanda_pairs

    # blood sugar lists --------------------------------------------------------------
    duration = 100
    fasting_sugar_list = []
    temp = SugarMeasurement.objects.filter(
        time__range=(datetime.combine((datetime.now() - timedelta(days=duration)).
                                      date(), time(0, 0, 0)),
                     datetime.now()), meal=0, user=patient)
    for i in range(duration):
        s = temp.filter(time__range=(datetime.combine((datetime.now() - timedelta(days=i)).
                                                      date(), time(0, 0, 0)),
                                     datetime.combine((datetime.now() - timedelta(days=i)).
                                                      date(), time(23, 59, 59))))
        if s.exists():
            fasting_sugar_list.append(s[0].amount)
        else:
            fasting_sugar_list.append("null")

    after_breakfast_sugar_list = []
    temp = SugarMeasurement.objects.filter(
        time__range=(datetime.combine((datetime.now() - timedelta(days=duration)).
                                      date(), time(0, 0, 0)),
                     datetime.now()), meal=1, user=patient)
    for i in range(duration):
        s = temp.filter(time__range=(datetime.combine((datetime.now() - timedelta(days=i)).
                                                      date(), time(0, 0, 0)),
                                     datetime.combine((datetime.now() - timedelta(days=i)).
                                                      date(), time(23, 59, 59))))
        if s.exists():
            after_breakfast_sugar_list.append(s[0].amount)
        else:
            after_breakfast_sugar_list.append("null")

    after_lunch_sugar_list = []
    temp = SugarMeasurement.objects.filter(
        time__range=(datetime.combine((datetime.now() - timedelta(days=duration)).
                                      date(), time(0, 0, 0)),
                     datetime.now()), meal=3, user=patient)
    for i in range(duration):
        s = temp.filter(time__range=(datetime.combine((datetime.now() - timedelta(days=i)).
                                                      date(), time(0, 0, 0)),
                                     datetime.combine((datetime.now() - timedelta(days=i)).
                                                      date(), time(23, 59, 59))))
        if s.exists():
            after_lunch_sugar_list.append(s[0].amount)
        else:
            after_lunch_sugar_list.append("null")

    after_dinner_sugar_list = []
    temp = SugarMeasurement.objects.filter(
        time__range=(datetime.combine((datetime.now() - timedelta(days=duration)).
                                      date(), time(0, 0, 0)),
                     datetime.now()), meal=5, user=patient)
    for i in range(duration):
        s = temp.filter(time__range=(datetime.combine((datetime.now() - timedelta(days=i)).
                                                      date(), time(0, 0, 0)),
                                     datetime.combine((datetime.now() - timedelta(days=i)).
                                                      date(), time(23, 59, 59))))
        if s.exists():
            after_dinner_sugar_list.append(s[0].amount)
        else:
            after_dinner_sugar_list.append("null")

    # assistant appointments -------------------------------------------------------------------

    now = datetime.now()
    first_day_of_month = now.replace(day=1)
    last_day_of_month = (now.replace(month=now.month % 12 + 1, day=1) - timedelta(days=1))
    thirty_days_ago = now - timedelta(days=30)

    assistant_times_upcoming = TimeSlot.objects.filter(therapist_django_user=django_user, patient=patient,
                                                       start_time__gt=now, is_available=0)
    assistant_times_expired = TimeSlot.objects.filter(therapist_django_user=django_user, patient=patient,
                                                      start_time__lt=now, start_time__gte=thirty_days_ago,
                                                      is_available=0)
    assistant_times_canceled = TimeSlot.objects.filter(therapist_django_user=django_user, patient=patient,
                                                       is_available=2, start_time__gte=first_day_of_month,
                                                       start_time__lte=last_day_of_month)

    upcoming_assistant = []
    expired_assistant = []
    canceled_assistant = []

    for times in assistant_times_upcoming:
        item = {
            "id": times.id,
            "days_of_week": times.days_of_week,
            "start_time": times.start_time,
            "duration": times.duration,
            "is_available": times.is_available,
            "therapist": times.therapist_django_user,
            "appointment_link": times.appointment_link,
        }
        upcoming_assistant.append(item)

    for times in assistant_times_expired:
        item = {
            "id": times.id,
            "days_of_week": times.days_of_week,
            "start_time": times.start_time,
            "duration": times.duration,
            "is_available": times.is_available,
            "therapist": times.therapist_django_user,
        }
        expired_assistant.append(item)

    for times in assistant_times_canceled:
        item = {
            "id": times.id,
            "days_of_week": times.days_of_week,
            "start_time": times.start_time,
            "duration": times.duration,
            "is_available": times.is_available,
            "therapist": times.therapist_django_user,
        }
        canceled_assistant.append(item)

    # Eating log and chart data (same as show_patient view) -------------------------
    from collections import defaultdict
    today = date.today()
    y7 = datetime.now() + timedelta(days=1)
    t2 = datetime(year=today.year, month=today.month, day=today.day, hour=23, minute=59, second=59)
    end_date = datetime.now().date()
    start_date = end_date - timedelta(days=6)  # Calculate 6 days back for a total of 7 days

    eat_foods = Eating.objects.filter(user=patient, date_time__range=(start_date, t2)).order_by('-date_time')
    
    day_map = defaultdict(lambda: {
        'date': None,
        'foods_by_meal': defaultdict(list),
    })

    for record in eat_foods:
        food_date = record.date_time.date()
        meal_key = record.meal
        day_data = day_map[food_date]
        if day_data['date'] is None:
            day_data['date'] = food_date
        day_data['foods_by_meal'][meal_key].append(record)

    day_list = list(day_map.values())
    day_list.sort(key=lambda x: x['date'], reverse=True)
    
    # Convert defaultdict to regular dict for template compatibility
    for day_item in day_list:
        day_item['foods_by_meal'] = dict(day_item['foods_by_meal'])

    # Calculate nutritional totals for each day and add Persian date
    for item in day_list:
        delta = (today - item['date']).days
        item['day_label'] = get_day_label(delta)
        item['jalali_date'] = jdatetime.date.fromgregorian(date=item['date']).strftime('%Y/%m/%d')
        
        # Calculate daily totals
        daily_protein = 0
        daily_fat = 0
        daily_carbs = 0
        daily_calories = 0
        
        for meal_records in item['foods_by_meal'].values():
            for record in meal_records:
                if record.food:
                    daily_protein += (record.food.Protein_g * record.amount) / 100
                    daily_fat += (record.food.Fat_g * record.amount) / 100
                    daily_carbs += (record.food.Carbohydrates_g * record.amount) / 100
                    daily_calories += (record.food.Calories * record.amount) / 100
        
        item['daily_protein'] = round(daily_protein, 1)
        item['daily_fat'] = round(daily_fat, 1)
        item['daily_carbs'] = round(daily_carbs, 1)
        item['daily_calories'] = round(daily_calories, 1)
    
    # Prepare chart data for 7 days and 30 days
    def get_nutrition_chart_data(days):
        """Get nutritional data for chart visualization"""
        chart_start_date = end_date - timedelta(days=days-1)
        chart_data = []
        
        for i in range(days):
            current_date = chart_start_date + timedelta(days=i)
            day_protein = 0
            day_fat = 0
            day_carbs = 0
            day_calories = 0
            
            # Find eating records for this date
            day_records = Eating.objects.filter(
                user=patient,
                date_time__date=current_date
            )
            
            for record in day_records:
                if record.food:
                    day_protein += (record.food.Protein_g * record.amount) / 100
                    day_fat += (record.food.Fat_g * record.amount) / 100
                    day_carbs += (record.food.Carbohydrates_g * record.amount) / 100
                    day_calories += (record.food.Calories * record.amount) / 100
            
            jalali_date = jdatetime.date.fromgregorian(date=current_date).strftime('%m/%d')
            
            chart_data.append({
                'date': current_date.strftime('%Y-%m-%d'),
                'jalali_date': jalali_date,
                'protein': round(day_protein, 1),
                'fat': round(day_fat, 1),
                'carbs': round(day_carbs, 1),
                'calories': round(day_calories, 1)
            })
        
        return chart_data
    
    # Get chart data for both periods
    chart_data_7days = get_nutrition_chart_data(7)
    chart_data_30days = get_nutrition_chart_data(30)
    
    # Prepare JSON for template
    import json
    chart_data_7days_json = json.dumps(chart_data_7days)
    chart_data_30days_json = json.dumps(chart_data_30days)

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'user': assistant,
        'patient': patient,
        'fasting_sugar_list': fasting_sugar_list[::-1],
        'after_breakfast_sugar_list': after_breakfast_sugar_list[::-1],
        'after_lunch_sugar_list': after_lunch_sugar_list[::-1],
        'after_dinner_sugar_list': after_dinner_sugar_list[::-1],
        "qanda": qanda_pairs,
        "user_qanda": user_qanda_pairs,
        "microbiome_qanda": microbiome_qanda_pairs,
        "lifestyle_qanda": lifestyle_qanda_pairs,
        'upcoming_assistant': upcoming_assistant,
        'expired_assistant': expired_assistant,
        'canceled_assistant': canceled_assistant,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
        "day_list": day_list,
        'chart_data_7days': chart_data_7days_json,
        'chart_data_30days': chart_data_30days_json,
        'meal_labels': meal_labels,
    }
    return render(request, template_name='assistant/patient_additional_information.html', context=context)


@api_view(["POST"])
def add_note(request, patient_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")

    temp = AppUser.objects.filter(id=patient_id)
    if not temp.exists():
        return generate_error_page(request, 'invalid request')
    patient = temp[0]
    assistant = Assistant.objects.get(django_user=django_user)
    if patient.profile.therapist != assistant.django_user:
        return generate_error_page(request, "you have not access to this patient!")

    text = request.POST.get("text")
    text_with_br_tags = text.replace('\n', '<br>')
    if text is not None:
        note = MedicalNote(user=django_user, patient=patient, text=text_with_br_tags, writer_name=str(assistant))
        note.save()
    return redirect('panel-assistant-show-patient', patient_id=patient.id)


@api_view(["POST"])
def edit_note(request, note_id):
    # Retrieve the note object or return a 404 error if it doesn't exist
    note = get_object_or_404(MedicalNote, id=note_id)

    if request.method == 'POST':
        # Get the new text from the form data
        new_text = request.POST.get('editnote')

        # Update the note text if new text is provided
        if new_text:
            new_text = new_text.replace('<br>', '')
            text_with_br_tags = new_text.replace('\n', '<br>')
            note.text = text_with_br_tags
            note.save()

        # Redirect back to the patient's page after editing the note
        return redirect('panel-assistant-show-patient', patient_id=note.patient.id)

    # If the request method is not POST, redirect back to the patient's page without making any changes
    return redirect('panel-assistant-show-patient', patient_id=note.patient.id)


def delete_note(request, note_id):
    note = get_object_or_404(MedicalNote, id=note_id)
    patient_id = note.patient.id
    note.delete()
    return redirect('panel-assistant-show-patient', patient_id=patient_id)


@api_view(["GET"])
def chat_with(request, patient_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")

    temp = AppUser.objects.filter(id=patient_id)
    if not temp.exists():
        return generate_error_page(request, 'invalid request')
    patient = temp[0]
    assistant = Assistant.objects.get(django_user=django_user)
    now = datetime.now()
    upcoming_assistant = []
    expired_assistant = []

    if patient.profile.therapist != assistant.django_user:
        return generate_error_page(request, "you have not access to this patient!")

    chats = Message.objects.filter(Q(sender=patient.django_user) | Q(receiver=patient.django_user)).order_by("time")
    for chat in chats:
        chat.text = chat.text.replace('\n', '<br>')
        chat.text = mark_safe(chat.text)

    assistant_times_upcoming = TimeSlot.objects.filter(patient=patient, start_time__gt=now, is_available=0)
    assistant_times_expired = TimeSlot.objects.filter(patient=patient, start_time__lt=now, is_available=0)

    assistant.increment_view_chat_page()

    for times in assistant_times_upcoming:
        item = {
            "id": times.id,
            "days_of_week": times.days_of_week,
            "start_time": times.start_time,
            "duration": times.duration,
            "is_available": times.is_available,
            "therapist": times.therapist_django_user,
            "appointment_link": times.appointment_link,
        }
        upcoming_assistant.append(item)

    for times in assistant_times_expired:
        item = {
            "id": times.id,
            "days_of_week": times.days_of_week,
            "start_time": times.start_time,
            "duration": times.duration,
            "is_available": times.is_available,
            "therapist": times.therapist_django_user,
        }
        expired_assistant.append(item)

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'user': assistant,
        'patient': patient,
        'chats': chats,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'upcoming_assistant': upcoming_assistant,
        'expired_assistant': expired_assistant,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }

    for chat in chats:
        if chat.receiver == django_user:
            chat.has_been_seen = True
            chat.save()

    return render(request, template_name='assistant/chat.html', context=context)


@api_view(["POST"])
def send_message(request, patient_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")

    temp = AppUser.objects.filter(id=patient_id)
    if not temp.exists():
        return generate_error_page(request, 'invalid request')
    patient = temp[0]
    assistant = Assistant.objects.get(django_user=django_user)
    if patient.profile.therapist != assistant.django_user:
        return generate_error_page(request, "you have not access to this patient!")

    replied_message_id = request.POST.get("message_id")

    if replied_message_id is None:
        text = request.POST.get("text")
        if text is not None:
            message = Message(sender=django_user, receiver=patient.django_user, text=text)
            message.save()
            # SMS().notify_user(user=patient)

            # ***********************************************************
            send_notification_note(patient, text)
            # ***********************************************************

            return myResponse.OK("sent", [])

        return myResponse.Error("error on saving the message", [])

    else:
        text_reply = request.POST.get("reply_text")
        past_message = get_object_or_404(Message, id=replied_message_id)
        if text_reply is not None:
            # Creating a reply message directly in the same Message model
            message = Message(sender=django_user,
                              receiver=patient.django_user,
                              text=text_reply,
                              reply_message=past_message  # Assigning the replied message
                              )
            message.save()

            # SMS().notify_user(user=patient)

            # ***********************************************************
            send_notification_note(patient, text_reply)
            # ***********************************************************

            return myResponse.OK("sent", [])

        return myResponse.Error("error on saving the message", [])


@api_view(["POST"])
def delete_message(request):
    if not request.user.is_authenticated:
        return JsonResponse({"error": "User not authenticated"}, status=401)

    if request.method == "POST":
        message_id = request.POST.get("message_id")
        message = get_object_or_404(Message, id=message_id)

        # Check if the user has permission to delete the message
        if request.user == message.sender or request.user == message.receiver:
            # Delete the message
            message.delete()
            # print("delete message")
            return JsonResponse({"message": "Message deleted successfully"})
        else:
            return JsonResponse({"error": "You do not have permission to delete this message"}, status=403)

    return JsonResponse({"error": "Invalid request method"}, status=405)


@api_view(["POST"])
def edit_message(request):
    if not request.user.is_authenticated:
        return JsonResponse({"error": "User not authenticated"}, status=401)

    if request.method == "POST":
        message_id = request.POST.get("message_id")
        edited_text = request.POST.get("edited_text")

        message = get_object_or_404(Message, id=message_id)

        # Check if the user has permission to edit the message
        if request.user == message.sender or request.user == message.receiver:
            # Update the message text
            message.text = edited_text
            message.save()
            # print("Edit message")
            return JsonResponse({"message": "Message edited successfully"})
        else:
            return JsonResponse({"error": "You do not have permission to edit this message"}, status=403)

    return JsonResponse({"error": "Invalid request method"}, status=405)


@api_view(["POST"])
def send_file(request, patient_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")

    temp = AppUser.objects.filter(id=patient_id)
    if not temp.exists():
        return generate_error_page(request, 'invalid request')
    patient = temp[0]
    assistant = Assistant.objects.get(django_user=django_user)
    if patient.profile.therapist != assistant.django_user:
        return generate_error_page(request, "you have not access to this patient!")

    file = request.FILES["file"]
    extension = file.name.split(".")[-1]
    if extension == "blob":
        extension = 'mp3'
    hashed_name = generate_file_name(extension)
    path_from_static_root = "panel/attachments/" + hashed_name
    try:
        with open(STATIC_ROOT + "/" + path_from_static_root, 'wb+') as destination:
            for chunk in file.chunks():
                destination.write(chunk)
            destination.close()

        file_size = os.path.getsize(STATIC_ROOT + "/" + path_from_static_root)
        is_diet = str(request.POST.get("is_diet"))
        if is_diet == "0" or is_diet == "None":
            is_diet = False
        else:
            is_diet = True
        attachment = Attachment(owner=django_user, name=file.name, path=path_from_static_root,
                                size=file_size, content_type=extension, is_diet=is_diet)
        attachment.save()
        message = Message(sender=django_user, receiver=patient.django_user, text="", attachment=attachment)
        message.save()
        data = {
            "url_uploaded": "/static/" + path_from_static_root,
            "is_diet": is_diet,
            "content_type": extension,
            "name": file.name,
            "time": "%s:%s" % (message.time.hour, message.time.minute),
            'size': attachment.get_size(),  # changing byte to KB
        }
        SMS().notify_user(user=patient)
        return myResponse.OK("sent", data)
    except Exception as e:
        return myResponse.Error("error on saving the message")


@api_view(["POST"])
def check_new_message_from(request, patient_id):
    try:
        if not request.user.is_authenticated:
            return redirect('panel-login')
        django_user = request.user
        if not django_user.groups.filter(name='Assistant').exists():
            return generate_error_page(request, "you have not access")

        temp = AppUser.objects.filter(id=patient_id)
        if not temp.exists():
            return generate_error_page(request, 'invalid request')

        patient = temp[0]
        assistant = Assistant.objects.get(django_user=django_user)

        if patient.profile.therapist != assistant.django_user:
            return generate_error_page(request, "you have not access to this patient!")

        temp = Message.objects.filter(Q(sender=patient.django_user) & Q(has_been_seen=False)).order_by("time")
        chats_list = []
        for chat in temp:
            item = {
                'time': str(chat.time.hour) + ":" + str(chat.time.minute),
                'text': chat.text,
                'attachment': None,
                'sender': 'other',
            }
            if chat.sender == assistant:
                item['sender'] = 'you'
            else:
                item['sender'] = 'other'
                chat.has_been_seen = True
                chat.save()

            if chat.attachment is not None:
                item['attachment'] = {
                    'name': chat.attachment.name,
                    'time': int(chat.attachment.time.timestamp()),
                    'content_type': chat.attachment.content_type,
                    'path': 'https://ziluck.probiofit.net/static/' + chat.attachment.path,
                    'size': chat.attachment.get_size(),
                    'owner': chat.attachment.owner.id,
                }

            chats_list.append(item)
        data = {
            'new_chats': chats_list,
            'number_of_new_messages': len(chats_list)
        }
        return myResponse.OK('new messages', data=data)
    except Exception as e:
        return myResponse.Error(str(e), e.args)


@api_view(["POST"])
def send_message_to_my_patients(request):
    # 1. Check authentication
    if not request.user.is_authenticated:
        return JsonResponse({"status": "login"})

    # 2. Get and sanitize message
    message = request.POST.get('message', '')
    message = message.replace('"', '')
    message = message.replace('\\n', '\n')

    # 3. Parse checked_items
    checked_items_json = request.POST.get('checked_items', '[]')
    checked_items = json.loads(checked_items_json)

    # 4. Distinguish group IDs from user IDs based on prefix
    user_ids = []
    group_ids = []
    for item in checked_items:
        if item.startswith("user-"):
            # "user-912" → "912"
            user_id_str = item.split("-", 1)[1]  # split only once
            user_ids.append(user_id_str)
        elif item.startswith("group-"):
            # "group-54" → "54"
            group_id_str = item.split("-", 1)[1]
            group_ids.append(group_id_str)

    # Convert user_ids to integers
    user_ids = [int(uid) for uid in user_ids]

    # 5. Fetch group memberships: gather *all* user IDs from selected groups
    django_user_ids = []
    groups = ZiLuckGroup.objects.filter(group_id__in=group_ids)
    for group in groups:
        # group.user_set.all() -> the set of Django "User" objects in this group
        for user_obj in group.user_set.all():
            django_user_ids.append(user_obj.id)

    # Now add the direct user_ids
    django_user_ids += user_ids

    # Remove duplicates if necessary
    django_user_ids = list(set(django_user_ids))

    # 6. Get the assistant object
    assistant = Assistant.objects.filter(django_user=request.user).first()
    if not assistant:
        return JsonResponse({"status": "no", "error": "No matching assistant found"})

    # 7. Loop over each ID to send the message
    STATUS = False
    for user_id in django_user_ids:
        # find the AppUser
        app_user = AppUser.objects.filter(django_user__id=user_id).first()
        if not app_user:
            # skip if no AppUser
            continue

        # check if the assistant is the therapist
        if app_user.profile.therapist == assistant.django_user:
            # create the message
            try:
                msg = Message(
                    sender=app_user.profile.therapist,
                    receiver=app_user.django_user,
                    text=message
                )
                msg.save()
            except Exception as e:
                traceback.print_exc()
                continue

            # Attempt to send push notification
            try:
                if app_user.profile.firebase_token and not app_user.profile.version:
                    PushNotification(app_user).send_notification_to(
                        registration_id=app_user.profile.firebase_token,
                        message_title="پیام جدید",
                        message=message
                    )
                    STATUS = True
            except Exception as e:
                traceback.print_exc()
                # We won't fail the entire loop, but won't mark success
                continue

    # 8. Final response
    if STATUS:
        return JsonResponse({"status": "ok"})
    else:
        return JsonResponse({"status": "no"})


@api_view(["POST"])
def send_notification_to_my_patients(request):
    # 1. Check authentication
    if not request.user.is_authenticated:
        return JsonResponse({"status": "login"})

    # 2. Get and sanitize message
    message = request.POST.get('message', '')
    message = message.replace('"', '')
    message = message.replace('\\n', '\n')

    # 3. Parse checked_items_notif
    checked_items_json = request.POST.get('checked_items_notif', '[]')
    checked_items = json.loads(checked_items_json)

    # 4. Distinguish group IDs from user IDs
    user_ids = []
    group_ids = []
    for item in checked_items:
        if item.startswith("user-"):
            user_id_str = item.split("-", 1)[1]
            user_ids.append(user_id_str)
        elif item.startswith("group-"):
            group_id_str = item.split("-", 1)[1]
            group_ids.append(group_id_str)

    # Convert user_ids to integer
    user_ids = [int(uid) for uid in user_ids]

    # 5. Gather all user IDs from the selected groups
    django_user_ids = []
    groups = ZiLuckGroup.objects.filter(group_id__in=group_ids)
    for group in groups:
        for user_obj in group.user_set.all():
            django_user_ids.append(user_obj.id)

    django_user_ids += user_ids
    django_user_ids = list(set(django_user_ids))

    # 6. Assistant
    assistant = Assistant.objects.filter(django_user=request.user).first()
    if not assistant:
        return JsonResponse({"status": "no", "error": "No matching assistant found"})

    # 7. Send notifications
    STATUS = False
    for user_id in django_user_ids:
        app_user = AppUser.objects.filter(django_user__id=user_id).first()
        if not app_user:
            continue

        if app_user.profile.therapist == assistant.django_user:
            try:
                if app_user.profile.firebase_token and not app_user.profile.version:
                    PushNotification(app_user).send_notification_to(
                        registration_id=app_user.profile.firebase_token,
                        message_title="نوتیفیکشن جدید",
                        message=message
                    )
                    STATUS = True
            except Exception as e:
                traceback.print_exc()
                return JsonResponse({"status": "error", "error": str(e)})

    if STATUS:
        return JsonResponse({"status": "ok"})
    else:
        return JsonResponse({"status": "no"})


@api_view(["POST"])
def add_diet_with_excel(request, patient_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")

    # # django_user for test
    # django_user = Assistant.objects.filter(django_user__username="dastyar11")[0].django_user

    temp = AppUser.objects.filter(id=patient_id)
    # temp = AppUser.objects.all()
    if not temp.exists():
        return generate_error_page(request, 'invalid request')
    patient = temp[0]
    assistant = Assistant.objects.get(django_user=django_user)
    if patient.profile.therapist != assistant.django_user:
        return generate_error_page(request, "you have not access to this patient!")

    file = request.FILES["file"]
    extension = file.name.split(".")[-1]
    hashed_name = generate_file_name(extension)
    path_from_static_root = "panel/diets/" + hashed_name
    try:
        diet_excel_path = STATIC_ROOT + "/" + path_from_static_root
        with open(diet_excel_path, 'wb+') as destination:
            for chunk in file.chunks():
                destination.write(chunk)
            destination.close()

        diet_json = convert_excel_to_diet(diet_excel_path)
        diet = Diet(diet_json=diet_json,
                    user=patient,
                    creator=django_user,
                    )
        diet.adjust_starting_date()
        diet.save()
        data = {
            'df': diet_json
        }
        # SMS().notify_user(user=patient)
        return myResponse.OK("Diet was successfully registered", data)
    except Exception as e:
        return myResponse.Error("error on saving the diet: {}".format(e), 1001)


@api_view(["POST"])
def upload_last_diet(request, patient_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")

    # Ensure patient exists
    temp = AppUser.objects.filter(id=patient_id)
    if not temp.exists():
        return generate_error_page(request, 'invalid request')
    patient = temp[0]
    assistant = Assistant.objects.get(django_user=django_user)
    if patient.profile.therapist != assistant.django_user:
        return generate_error_page(request, "you have not access to this patient!")

    try:
        # Fetch the last diet
        last_diet = Diet.objects.filter(user=patient).order_by('-created_at').first()
        if not last_diet:
            return myResponse.Error("No previous diet found", 1002)

        # Create a new diet entry with the last diet's JSON data
        diet_json = last_diet.diet_json
        new_diet = Diet(diet_json=diet_json, user=patient, creator=django_user)
        new_diet.adjust_starting_date()
        new_diet.save()

        data = {'df': diet_json}
        return myResponse.OK("Last diet was successfully uploaded", data)

    except Exception as e:
        return myResponse.Error("error on uploading the last diet: {}".format(e), 1001)


@api_view(["POST"])
def add_daily_note(request, patient_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")

    temp = AppUser.objects.filter(id=patient_id)
    if not temp.exists():
        return generate_error_page(request, 'invalid request')
    patient = temp[0]
    assistant = Assistant.objects.get(django_user=django_user)
    if patient.profile.therapist != assistant.django_user:
        return generate_error_page(request, "you have not access to this patient!")
    try:
        note = request.POST.get("note")
        daily_note = DailyNote(user=django_user, patient=patient,
                               text=note, writer_name=str(assistant))
        daily_note.save()
        send_notification_note(patient, note)
        return redirect('panel-assistant-show-patient', patient_id=patient.id)

    except Exception as e:
        return myResponse.Error("error on saving the daily note: {}".format(e), 1001)


@api_view(["GET"])
def appointments(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    user = request.user
    end_date = datetime.now().date() + timedelta(days=8)
    start_date = end_date - timedelta(days=15)
    patients_list = AppUser.objects.filter(profile__therapist=user)
    assistant = Assistant.objects.get(django_user=user)
    free_times = TimeSlot.objects.filter(therapist_django_user=user, start_time__range=(start_date, end_date))
    # TimeSlot.objects.all().delete()

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'user': assistant,
        'free_times': free_times,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }

    return render(request, template_name='assistant/appointments.html', context=context)


@api_view(['POST'])
def submit_free_time(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")

    try:
        selected_ids = request.POST.getlist("selectedIds")  # Get the selectedIds list from POST data
        day_name = selected_ids[0]  # Assuming the first element in the list is the day name
        split = day_name.strip('[]').split(',')
        day_names = int(day_name[2])
        if day_names == 0:
            day_names_def = day_names + 5
        elif day_names == 1:
            day_names_def = day_names + 5
        else:
            day_names_def = day_names - 2

        days_times = [slot.strip('"') for slot in split]
        start_times = [slot.split('-')[0] for slot in days_times]
        modified_list = [item[1:] for item in start_times]
        time_objects = [datetime.strptime(time_str, '%H%M').time() for time_str in modified_list]
        # TimeSlot.objects.all().delete()
        dup = False
        assistant_free_times = TimeSlot.objects.filter(therapist_django_user=django_user)
        for i in range(len(days_times)):
            for times in assistant_free_times:
                if convert_to_datetime(day_names_def, modified_list[i]) == times.start_time:
                    dup = True
                    break
            if dup:
                break
            free_time = TimeSlot(therapist_django_user=django_user,
                                 start_time=convert_to_datetime(day_names_def, modified_list[i]),
                                 days_of_week=day_names)
            free_time.save()

        return redirect('panel-assistant-appointments')
    # Delete existing free time slots for the assistant

    except Exception as e:
        return myResponse.Error("error on saving the free time: {}".format(e), 1001)


def convert_to_datetime(weekday, time_str):
    # Get today's date
    today = datetime.now().date()

    # Calculate the number of days to add to reach the desired weekday
    days_to_add = (weekday - today.weekday() + 7) % 7

    # Parse the time string to get the hours and minutes
    hours = int(time_str[:2])
    minutes = int(time_str[2:])

    # Create a timedelta object for the time
    time_delta = timedelta(hours=hours, minutes=minutes)

    # Calculate the target date by adding the days_to_add to today's date
    target_date = today + timedelta(days=days_to_add)

    # Combine the target date and time to create the datetime object
    target_datetime = datetime.combine(target_date, datetime.min.time()) + time_delta

    return target_datetime


@api_view(['POST'])
def add_meeting_link(request, session_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")
    try:
        session = get_object_or_404(TimeSlot, id=session_id)
        if request.method == 'POST':
            meeting_link = request.POST.get("appointment_link")
            if meeting_link:
                session.appointment_link = meeting_link
                session.save()

            return redirect('panel-assistant-show-patient')

    except Exception as e:
        return myResponse.Error("error on saving the meeting link: {}".format(e), 1001)


def delete_appointment(request, appointment_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    appointment = get_object_or_404(TimeSlot, id=appointment_id)
    if appointment.is_available == 1:
        appointment.delete()
    return redirect('panel-assistant-appointments')


@api_view(["GET"])
def delete_diet(request, diet_id):
    try:
        if not request.user.is_authenticated:
            return redirect('panel-login')
        django_user = request.user
        if not django_user.groups.filter(name='Assistant').exists():
            return generate_error_page(request, "you have not access")

        temp = Diet.objects.filter(id=diet_id)
        if not temp.exists():
            return generate_error_page(request, 'invalid request')
        diet = temp[0]
        patient = diet.user
        assistant = Assistant.objects.get(django_user=django_user)
        if diet.creator != assistant.django_user:
            return generate_error_page(request, "you have not access to this patient!")
        diet.delete()
        return redirect('/panel/assistant/show/patient/%s/' % patient.id)
    except Exception as e:
        return generate_error_page(request, e.args[0])


def edit_diet(request, patient_id, diet_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "You do not have access")

    diet = get_object_or_404(Diet, id=diet_id, user__id=patient_id)

    context = {
        'diet': diet,
        'diet_json': diet.diet_json,  # Assuming the diet has a JSON field with the diet data
        # Add any other necessary context variables
    }

    # return render(request, 'your_diet_editing_template.html', context)
    return


@api_view(["GET"])
def diets(request, patient_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")
    temp = AppUser.objects.filter(id=patient_id)
    if not temp.exists():
        return generate_error_page(request, 'invalid request')
    patient = temp[0]
    assistant = Assistant.objects.get(django_user=django_user)
    free_times = TimeSlot.objects.filter(therapist_django_user=django_user)

    medical_notes = MedicalNote.objects.filter(patient=patient).order_by('-time')
    diseases_list = patient.profile.diseases

    DISPLAY_KEYS = ["WEIGHT", "HEIGHT", "SWEETS_FREQUENCY", "EXERCISE_FREQUENCY", "IDEAL_WEIGHT",
                    "HEALTH_GOAL", "FOOD_LIMITING_MEDICATION", "SUPPLEMENTS_MEDICATION", "CHRONIC_ILLNESSES",
                    "HIGH_FIBER_PLANT_FOODS", "PROCESSED_FOODS_FREQUENCY", "MEDICAL_DIETARY_RESTRICTIONS",
                    "MAIN_MEALS_PER_DAY", "HOME_MADE_MEALS", "LIMITED_AVOIDED_FOODS", "SNACK_TYPE_PREFERENCE",
                    "SNACKS_PER_DAY", "MEAT_CONSUMPTION_FREQUENCY", "MEAT_PORTION_SIZE", "EATING_PATTERNS",
                    "FOOD_ALLERGIES_INTOLERANCES", "BREAKFAST_OPTIONS", "BREAKFAST_BREAD_TYPE", "WORK_LUNCH_HABITS",
                    "LUNCH_PREFERENCE", "LUNCH_SIDE_DISH", "DINNER_PREFERENCE", "DINNER_SIDE_DISH", "MEAL_VARIETY"]

    user_questionnaire = UserQuestionnaire.objects.filter(user=patient)
    QandA_filtered = []

    if user_questionnaire.exists():
        for questionnaire in user_questionnaire:
            qanda_dict = questionnaire.QandA
            # Filter the QandA dictionary to include only keys in DISPLAY_KEYS
            filtered_qanda = {key: qanda_dict[key] for key in DISPLAY_KEYS if key in qanda_dict}
            QandA_filtered.append(filtered_qanda)

    # Flatten the filtered data for rendering
    qanda_pairs = []
    for qanda in QandA_filtered:
        qanda_pairs.extend(qanda.items())

    QUESTION_ID_MAP = {
        "WATER_INTAKE": "آب مصرفی",
        "ALCOHOL_CONSUMPTION": "مصرف الکل",
        "TOBACCO_USAGE": "استفاده از تنباکو",
        "SWEETS_FREQUENCY": "تناوب استفاده شیرینی",
        "EXERCISE_FREQUENCY": "فعالیت بدنی",
        "IDEAL_WEIGHT": "وزن ایده‌آل",
        "HEALTH_GOAL": "هدف سلامتی",
        "ANTIBIOTICS_NSAIDS_USAGE": "مصرف آنتی‌بیوتیک‌ها و NSAID",
        "FOOD_LIMITING_MEDICATION": "داروهای محدودکننده غذا",
        "SUPPLEMENTS_MEDICATION": "مکمل‌ها و داروها",
        "PROBIOTICS_PREBIOTICS": "پروبیوتیک‌ها و پریبیوتیک‌ها",
        "HEALTH_CHANGES": "تغییرات سلامتی",
        "CONTACT_PREFERENCE": "ترجیح ارتباط",
        "CONTENT_FORMAT": "فرمت محتوا",
        "MICROBIOME_KNOWLEDGE": "دانش میکروبیوم",
        "CHRONIC_ILLNESSES": "بیماری‌های مزمن",
        "RESTRICTIVE_MEDICATIONS": "داروهای محدودکننده",
        "FOOD_RELATIONSHIP": "رابطه با غذا",
        "FERMENTED_FOODS_FREQUENCY": "تناوب مصرف غذاهای تخمیری",
        "HIGH_FIBER_PLANT_FOODS": "غذاهای گیاهی پرفیبر",
        "PROCESSED_FOODS_FREQUENCY": "تناوب مصرف غذاهای فرآوری‌شده",
        "MEDICAL_DIETARY_RESTRICTIONS": "محدودیت‌های غذایی پزشکی",
        "MAIN_MEALS_PER_DAY": "وعده‌های اصلی روزانه",
        "HOME_MADE_MEALS": "وعده‌های خانگی",
        "LIMITED_AVOIDED_FOODS": "غذاهای محدود یا اجتنابی",
        "SNACK_TYPE_PREFERENCE": "ترجیح نوع میان‌وعده",
        "SNACKS_PER_DAY": "تعداد میان‌وعده در روز",
        "MEAT_CONSUMPTION_FREQUENCY": "تناوب مصرف گوشت",
        "MEAT_PORTION_SIZE": "اندازه وعده گوشت",
        "EATING_PATTERNS": "الگوهای غذایی",
        "FOOD_ALLERGIES_INTOLERANCES": "حساسیت‌ها و عدم تحمل غذایی",
        "COOK_METHODS": "روش‌های پخت",
        "BREAKFAST_TIME": "زمان صبحانه",
        "BREAKFAST_OPTIONS": "گزینه‌های صبحانه",
        "BREAKFAST_BREAD_TYPE": "نوع نان صبحانه",
        "LUNCH_TIME": "زمان ناهار",
        "WORK_LUNCH_HABITS": "عادات ناهار در محل کار",
        "LUNCH_PREFERENCE": "ترجیح ناهار",
        "LUNCH_SIDE_DISH": "غذای جانبی ناهار",
        "DINNER_TIME": "زمان شام",
        "DINNER_PREFERENCE": "ترجیح شام",
        "DINNER_SIDE_DISH": "غذای جانبی شام",
        "MEAL_VARIETY": "تنوع غذایی",
        "BREAST_FEEDING_DURATION": "مدت شیردهی"
    }

    qanda_pairs_translated = [(QUESTION_ID_MAP.get(q, q), a) for q, a in qanda_pairs]

    diet_id = request.GET.get('diet_id')  # Get the diet_id from the query parameters
    diet = None

    if diet_id:
        diet = get_object_or_404(Diet, id=diet_id, user__id=patient_id)

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'user': assistant,
        'diet': diet,
        'diet_json': json.dumps(diet.diet_json) if diet else None,
        'patient': patient,
        'free_times': free_times,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
        'medical_notes': medical_notes,
        'diseases_list': diseases_list,
        "qanda": qanda_pairs_translated,
    }

    return render(request, template_name='assistant/diets.html', context=context)


@api_view(["GET"])
def diets2(request, patient_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")
    temp = AppUser.objects.filter(id=patient_id)
    if not temp.exists():
        return generate_error_page(request, 'invalid request')
    patient = temp[0]
    assistant = Assistant.objects.get(django_user=django_user)
    free_times = TimeSlot.objects.filter(therapist_django_user=django_user)

    medical_notes = MedicalNote.objects.filter(patient=patient).order_by('-time')
    diseases_list = patient.profile.diseases

    DISPLAY_KEYS = ["WEIGHT", "HEIGHT", "SWEETS_FREQUENCY", "EXERCISE_FREQUENCY", "IDEAL_WEIGHT",
                    "HEALTH_GOAL", "FOOD_LIMITING_MEDICATION", "SUPPLEMENTS_MEDICATION", "CHRONIC_ILLNESSES",
                    "HIGH_FIBER_PLANT_FOODS", "PROCESSED_FOODS_FREQUENCY", "MEDICAL_DIETARY_RESTRICTIONS",
                    "MAIN_MEALS_PER_DAY", "HOME_MADE_MEALS", "LIMITED_AVOIDED_FOODS", "SNACK_TYPE_PREFERENCE",
                    "SNACKS_PER_DAY", "MEAT_CONSUMPTION_FREQUENCY", "MEAT_PORTION_SIZE", "EATING_PATTERNS",
                    "FOOD_ALLERGIES_INTOLERANCES", "BREAKFAST_OPTIONS", "BREAKFAST_BREAD_TYPE", "WORK_LUNCH_HABITS",
                    "LUNCH_PREFERENCE", "LUNCH_SIDE_DISH", "DINNER_PREFERENCE", "DINNER_SIDE_DISH", "MEAL_VARIETY"]

    user_questionnaire = UserQuestionnaire.objects.filter(user=patient)
    QandA_filtered = []

    if user_questionnaire.exists():
        for questionnaire in user_questionnaire:
            qanda_dict = questionnaire.QandA
            # Filter the QandA dictionary to include only keys in DISPLAY_KEYS
            filtered_qanda = {key: qanda_dict[key] for key in DISPLAY_KEYS if key in qanda_dict}
            QandA_filtered.append(filtered_qanda)

    # Flatten the filtered data for rendering
    qanda_pairs = []
    for qanda in QandA_filtered:
        qanda_pairs.extend(qanda.items())

    QUESTION_ID_MAP = {
        "WATER_INTAKE": "آب مصرفی",
        "ALCOHOL_CONSUMPTION": "مصرف الکل",
        "TOBACCO_USAGE": "استفاده از تنباکو",
        "SWEETS_FREQUENCY": "تناوب استفاده شیرینی",
        "EXERCISE_FREQUENCY": "فعالیت بدنی",
        "IDEAL_WEIGHT": "وزن ایده‌آل",
        "HEALTH_GOAL": "هدف سلامتی",
        "ANTIBIOTICS_NSAIDS_USAGE": "مصرف آنتی‌بیوتیک‌ها و NSAID",
        "FOOD_LIMITING_MEDICATION": "داروهای محدودکننده غذا",
        "SUPPLEMENTS_MEDICATION": "مکمل‌ها و داروها",
        "PROBIOTICS_PREBIOTICS": "پروبیوتیک‌ها و پریبیوتیک‌ها",
        "HEALTH_CHANGES": "تغییرات سلامتی",
        "CONTACT_PREFERENCE": "ترجیح ارتباط",
        "CONTENT_FORMAT": "فرمت محتوا",
        "MICROBIOME_KNOWLEDGE": "دانش میکروبیوم",
        "CHRONIC_ILLNESSES": "بیماری‌های مزمن",
        "RESTRICTIVE_MEDICATIONS": "داروهای محدودکننده",
        "FOOD_RELATIONSHIP": "رابطه با غذا",
        "FERMENTED_FOODS_FREQUENCY": "تناوب مصرف غذاهای تخمیری",
        "HIGH_FIBER_PLANT_FOODS": "غذاهای گیاهی پرفیبر",
        "PROCESSED_FOODS_FREQUENCY": "تناوب مصرف غذاهای فرآوری‌شده",
        "MEDICAL_DIETARY_RESTRICTIONS": "محدودیت‌های غذایی پزشکی",
        "MAIN_MEALS_PER_DAY": "وعده‌های اصلی روزانه",
        "HOME_MADE_MEALS": "وعده‌های خانگی",
        "LIMITED_AVOIDED_FOODS": "غذاهای محدود یا اجتنابی",
        "SNACK_TYPE_PREFERENCE": "ترجیح نوع میان‌وعده",
        "SNACKS_PER_DAY": "تعداد میان‌وعده در روز",
        "MEAT_CONSUMPTION_FREQUENCY": "تناوب مصرف گوشت",
        "MEAT_PORTION_SIZE": "اندازه وعده گوشت",
        "EATING_PATTERNS": "الگوهای غذایی",
        "FOOD_ALLERGIES_INTOLERANCES": "حساسیت‌ها و عدم تحمل غذایی",
        "COOK_METHODS": "روش‌های پخت",
        "BREAKFAST_TIME": "زمان صبحانه",
        "BREAKFAST_OPTIONS": "گزینه‌های صبحانه",
        "BREAKFAST_BREAD_TYPE": "نوع نان صبحانه",
        "LUNCH_TIME": "زمان ناهار",
        "WORK_LUNCH_HABITS": "عادات ناهار در محل کار",
        "LUNCH_PREFERENCE": "ترجیح ناهار",
        "LUNCH_SIDE_DISH": "غذای جانبی ناهار",
        "DINNER_TIME": "زمان شام",
        "DINNER_PREFERENCE": "ترجیح شام",
        "DINNER_SIDE_DISH": "غذای جانبی شام",
        "MEAL_VARIETY": "تنوع غذایی",
        "BREAST_FEEDING_DURATION": "مدت شیردهی"
    }

    qanda_pairs_translated = [(QUESTION_ID_MAP.get(q, q), a) for q, a in qanda_pairs]

    diet_id = request.GET.get('diet_id')  # Get the diet_id from the query parameters
    diet = None

    if diet_id:
        diet = get_object_or_404(Diet, id=diet_id, user__id=patient_id)

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    # Get raw questionnaire data for JavaScript processing
    raw_questionnaire = {}
    if user_questionnaire.exists():
        latest_questionnaire = user_questionnaire.order_by('-id').first()
        raw_questionnaire = latest_questionnaire.QandA if latest_questionnaire else {}
    
    # Get lifestyle questionnaire data
    raw_lifestyle_questionnaire = {}
    lifestyle_questionnaire = LifestyleQuestionnaire.objects.filter(user=patient).order_by('-id').first()
    if lifestyle_questionnaire and lifestyle_questionnaire.QandA:
        raw_lifestyle_questionnaire = lifestyle_questionnaire.QandA

    context = {
        'user': assistant,
        'diet': diet,
        'diet_json': json.dumps(diet.diet_json) if diet else None,
        'patient': patient,
        'free_times': free_times,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
        'medical_notes': medical_notes,
        'diseases_list': diseases_list,
        "qanda": qanda_pairs_translated,
        "raw_questionnaire": json.dumps(raw_questionnaire),
        "raw_lifestyle_questionnaire": json.dumps(raw_lifestyle_questionnaire),
    }

    return render(request, template_name='assistant/diets2.html', context=context)


@api_view(['POST'])
def get_user_diet_tags(request):
    if not request.user.is_authenticated or not request.user.groups.filter(name='Assistant').exists():
        return Response({"error": "Unauthorized"}, status=403)

    patient_id = request.data.get('patient_id')
    if not patient_id:
        return Response({"error": "Patient ID is required"}, status=400)

    try:
        patient = AppUser.objects.get(id=patient_id)
        diet_tags = patient.profile.diet_tags
        if not diet_tags:
            return Response({"error": "No diet tags found for this patient"}, status=404)
        return Response(diet_tags)
    except AppUser.DoesNotExist:
        return Response({"error": "Invalid patient ID"}, status=400)
    except AttributeError:
        return Response({"error": "Profile or diet tags not set for this patient"}, status=500)


# Define the calorie distributions dictionary directly in this file
calorie_distributions = {
    1200: {"B": 250, "L": 350, "D": 300, "S1": 150, "S2": 150},
    1300: {"B": 300, "L": 350, "D": 350, "S1": 150, "S2": 150},
    1400: {"B": 350, "L": 400, "D": 350, "S1": 150, "S2": 150},
    1500: {"B": 400, "L": 400, "D": 400, "S1": 150, "S2": 150},
    1600: {"B": 400, "L": 450, "D": 400, "S1": 150, "S2": 200},
    1700: {"B": 400, "L": 500, "D": 450, "S1": 150, "S2": 200},
    1800: {"B": 400, "L": 550, "D": 450, "S1": 200, "S2": 200},
    1900: {"B": 450, "L": 550, "D": 500, "S1": 200, "S2": 200},
    2000: {"B": 500, "L": 600, "D": 500, "S1": 200, "S2": 200},
    2100: {"B": 500, "L": 600, "D": 550, "S1": 200, "S2": 250},
    2200: {"B": 500, "L": 650, "D": 600, "S1": 200, "S2": 250},
    2300: {"B": 550, "L": 650, "D": 600, "S1": 250, "S2": 250},
    2400: {"B": 550, "L": 700, "D": 600, "S1": 250, "S2": 300},
}

# List of valid CR values for selecting the closest match
cr_list = [1200, 1300, 1400, 1500, 1600, 1700, 1800, 1900, 2000, 2100, 2200, 2300, 2400]

from django.db.models import Q  # Move import to top of function
@api_view(['POST'])
def refresh_column_cards(request):

    if not request.user.is_authenticated or not request.user.groups.filter(name='Assistant').exists():
        return Response({"error": "Unauthorized"}, status=403)

    patient_id = request.data.get('patient_id')
    meal_type = request.data.get('meal_type')
    selected_tags = request.data.get('tags')
    count = int(request.data.get('count', 1))  # Should be 14 from refreshMeal
    months_filter = request.data.get('months', [])  # Get months filter
    bypass_zero_calorie = request.data.get('bypass_zero_calorie', False)  # Get bypass parameter
    bypassed_meals = request.data.get('bypassed_meals', {'breakfast': False, 'lunch': False, 'dinner': False})
    main_dish_codes = request.data.get('main_dish_codes', None)  # Get main_dish_codes for same lunch/dinner
    
    # Ensure selected_tags is a dictionary, not None
    if selected_tags is None:
        selected_tags = {}
    
    # Ensure months_filter is a list
    if months_filter is None:
        months_filter = []
    
    # Ensure bypassed_meals is a dict with all required keys
    if bypassed_meals is None or not isinstance(bypassed_meals, dict):
        bypassed_meals = {'breakfast': False, 'lunch': False, 'dinner': False}
    
    print(f"This is the selected_tags: {selected_tags}")
    print(f"Bypass zero calorie: {bypass_zero_calorie}")
    print(f"Bypassed meals: {bypassed_meals}")
    print(f"Main dish codes: {main_dish_codes}")

    if not patient_id or not meal_type or count != 14:
        print(f"Invalid input: patient_id={patient_id}, meal_type={meal_type}, count={count}")
        return Response({"error": "Invalid patient_id, meal_type, or count"}, status=400)

    try:
        patient = AppUser.objects.get(id=patient_id)
        print(f"Found patient: {patient.id}")
    except AppUser.DoesNotExist:
        print("Patient not found")
        return Response({"error": "Invalid patient ID"}, status=400)

    # Use the new service function for meal generation
    from panel.services.diet_generation import generate_meal_plan
    
    try:
        result = generate_meal_plan(
            patient, 
            meal_type, 
            selected_tags, 
            count, 
            months_filter, 
            return_metadata=True, 
            bypass_zero_calorie=bypass_zero_calorie,
            bypassed_meals=bypassed_meals,
            main_dish_codes=main_dish_codes
        )
        return Response({
            "cards": result['cards'],
            "excluded_meals": result['excluded_meals']
        })
    except Exception as e:
        print(f"Error generating meal plan: {str(e)}")
        import traceback
        traceback.print_exc()
        return Response({"error": "Failed to generate meal plan"}, status=500)



@api_view(['POST'])
def generate_cells_card(request):
    # Set up logging early
    logger = logging.getLogger(__name__)
    
    # Import excluded meals helper
    from panel.services.diet_generation import _get_excluded_meals
    
    try:
        if not request.user.is_authenticated or not request.user.groups.filter(name='Assistant').exists():
            logger.warning("Unauthorized user attempted refresh cell")
            return Response({"error": "Unauthorized"}, status=403)
    except Exception as e:
        logger.error(f"Error in authentication check: {e}")
        return Response({"error": "Authentication error"}, status=500)

    try:
        # Extract request data with error handling
        patient_id = request.data.get('patient_id')
        cell_id = request.data.get('cell_id')
        meal_type = request.data.get('meal_type')
        try:
            count = int(request.data.get('count', 1))
        except (ValueError, TypeError):
            count = 1
        selected_tags = request.data.get('tags', {})
        months_filter = request.data.get('months', [])  # Get months filter
        bypass_zero_calorie = request.data.get('bypass_zero_calorie', False)  # Get bypass parameter
        bypassed_meals = request.data.get('bypassed_meals', {'breakfast': False, 'lunch': False, 'dinner': False})
        column_data = request.data.get('column_data', [])  # List of existing main_dish_code values
        exclude_food_codes = request.data.get('exclude_food_codes', [])  # List of food_codes to exclude for duplicate avoidance
        cell_index = request.data.get('cell_index')  # Optional: index of the cell to replace
        
        # Ensure bypassed_meals is a dict with all required keys
        if bypassed_meals is None or not isinstance(bypassed_meals, dict):
            bypassed_meals = {'breakfast': False, 'lunch': False, 'dinner': False}
        
        logger.info(f"🔄 REFRESH CELL REQUEST: patient_id={patient_id}, meal_type={meal_type}, "
                   f"cell_index={cell_index}, column_data_length={len(column_data) if column_data else 0}, "
                   f"bypass_zero_calorie={bypass_zero_calorie}, bypassed_meals={bypassed_meals}")

        if not patient_id or not meal_type:
            logger.error(f"Invalid input: patient_id={patient_id}, meal_type={meal_type}")
            return Response({"error": "Invalid patient_id or meal_type"}, status=400)
            
    except Exception as e:
        logger.error(f"Error extracting request data: {e}")
        return Response({"error": "Invalid request data"}, status=400)

    try:
        patient = AppUser.objects.get(id=patient_id)
        print(f"Found patient: {patient.id}")
    except AppUser.DoesNotExist:
        print("Patient not found")
        return Response({"error": "Invalid patient ID"}, status=400)

    # Get excluded meals information
    excluded_meals = _get_excluded_meals(patient)
    
    patient_profile = patient.profile
    questionnaire = UserQuestionnaire.objects.filter(user=patient).order_by('-id').first()
    if not questionnaire:
        diet_data = generate_health_tags({}, profile=patient_profile)
    else:
        qanda = questionnaire.QandA
        diet_data = generate_health_tags(qanda, profile=patient_profile)
        
        
    if diet_data is None:
        print("Failed to process questionnaire")
        return Response({"error": "Failed to process questionnaire"}, status=500)

    # Ensure selected_tags is a dictionary
    if selected_tags is None:
        selected_tags = {}

    # Normalize meal type
    meal_normalized = meal_type if '-' in meal_type else meal_type.replace('snack1', 'snack-1').replace('snack2', 'snack-2')
    print(f"Refreshing column for meal: {meal_normalized}")

    patient_cr = patient_profile.CR
    if patient_cr is None:
        print("No CR found in patient profile")
        return Response({"error": "No calorie requirement found for this patient"}, status=500)

    # Here is the place where the coach CR should be used if there was any
    coach_cr = patient_profile.CR_coach
    target_cr_for_distribution = coach_cr if coach_cr is not None else patient_cr
    selected_cr = min(cr_list, key=lambda cr: (abs(target_cr_for_distribution - cr), cr))

    # Determine if we should use default distribution
    # If bypass_zero_calorie is set OR this specific meal is bypassed, use default distribution (remove zero-calorie limitation)
    meal_bypassed = False
    try:
        meal_bypassed = bool(bypassed_meals.get(meal_normalized, False))
    except Exception:
        meal_bypassed = False

    use_default_distribution = bypass_zero_calorie or meal_bypassed
    
    # Use DEFAULT distribution if use_default_distribution is True, otherwise use adjusted distribution
    if use_default_distribution:
        print(f"⚠️ USE_DEFAULT_DISTRIBUTION is enabled in generate_cells_card - using DEFAULT distribution for ALL meals")
        distribution = calorie_distributions[selected_cr]
        print(f"Using DEFAULT distribution from CR tier {selected_cr}: B={distribution['B']}, L={distribution['L']}, D={distribution['D']}, S1={distribution['S1']}, S2={distribution['S2']}")
    else:
        # Use adjusted distribution that accounts for excluded meals
        from panel.services.diet_generation import _calculate_adjusted_calorie_distribution
        distribution = _calculate_adjusted_calorie_distribution(target_cr_for_distribution, excluded_meals)

    # Add calorie values to diet_data
    diet_data.update({
        "breakfast_cal": distribution["B"],
        "lunch_cal": distribution["L"],
        "dinner_cal": distribution["D"],
        "snack1_cal": distribution["S1"],
        "snack2_cal": distribution["S2"]
    })

    # Calorie target
    calorie_target_key = {
        'breakfast': 'breakfast_cal',
        'lunch': 'lunch_cal',
        'dinner': 'dinner_cal',
        'snack-1': 'snack1_cal',
        'snack-2': 'snack2_cal'
    }.get(meal_normalized, 'breakfast_cal')
    calorie_target = diet_data.get(calorie_target_key, 0)
    
    print(f"Calorie target for {meal_normalized}: {calorie_target}")

    # Create allergen exclusion filters
    exclude_allergen_filters = {}
    for tag in all_alergic_tags:
        if diet_data.get('Allergens', {}).get(tag, 0) == 1:
            exclude_allergen_filters[tag] = False  # Exclude cards where allergen is present
    print(f"Exclude allergen filters: {exclude_allergen_filters}")

    # Create exclude filters for diseases
    exclude_filters = {}
    tag_mapping = {
        'Disease_High_Cholesterol': 'Disease_High_cholesterol',
        'Disease_Hypertension': 'Disease_High_blood_pressure',
        'Disease_Thyroid': 'Disease_Hypothyroidism',
        'Allergens_Crustaceans': 'Allergens_Shrimp',
        'Allergens_TreeNuts': 'Allergens_Nuts',
    }
    
    for tag, value in diet_data.get('Diseases', {}).items():
        if value == 1:
            normalized_tag = tag_mapping.get(tag, tag)
            # This tag shows what should be excluded, the code below means that the food cards
            # with disease tag of true will be excluded 
            # as an example if a card has disease high blood pressure tag of true, it will be excluded
            # and would not be shown to the user who has disease high blood pressure
            exclude_filters[normalized_tag] = True

    # Meal filters
    meal_filters = {}
    if meal_normalized == 'breakfast':
        meal_filters['is_breakfast'] = True
        # Add breakfast-specific tag filters based on main_dish_code
        breakfast_codes = []
        if selected_tags.get('cheese_breakfast'):
            breakfast_codes.extend([1042, 23214, 14370])  # cheese cards
        if selected_tags.get('peanut_butter_breakfast'):
            breakfast_codes.append(16150)  # peanut butter card
        if selected_tags.get('tahini_breakfast'):
            breakfast_codes.append(12166)  # tahini card
        if selected_tags.get('oatmeal_breakfast'):
            breakfast_codes.extend([20038, 42236])  # oatmeal cards
        if selected_tags.get('coffee_breakfast'):
            breakfast_codes.extend([55556, 55558, 55559, 14209, 55563, 14210, 9990442, 9990444, 1748159354020, 1748159354021, 14227, 23252])  # coffee cards
        if selected_tags.get('omelette_breakfast'):
            breakfast_codes.extend([14352, 14161, 14111, 14110, 14106, 14105, 14084, 14083, 9881004, 9980409, 9980221, 9880094])  # omelette cards
            
            
        if breakfast_codes:  # Only add the filter if any breakfast codes are selected
            meal_filters['main_dish_code__in'] = breakfast_codes
            
    elif meal_normalized == 'lunch':
        meal_filters['is_lunch'] = True
    elif meal_normalized == 'dinner':
        meal_filters['is_dinner'] = True
    
    if meal_normalized in ['snack-1', 'snack-2']:
        meal_filters['is_snack'] = True
        # Add snack-specific tag filters
        snack_codes = []
        if selected_tags.get('summer_snack'):
            meal_filters['summer'] = True  # This is correct as per model
        if selected_tags.get('winter_snack'):
            meal_filters['winter'] = True  # This is correct as per model
        if selected_tags.get('spring_snack'):
            meal_filters['spring'] = True  # This is correct as per model
        if selected_tags.get('autumn_snack'):
            meal_filters['autumn'] = True  # This is correct as per model
        # Apply coffee preference per snack column (uses FoodCard.is_coffee)
        if meal_normalized == 'snack-1' and selected_tags.get('coffee_snack1'):
            meal_filters['is_coffee'] = True
            logger.info(f"☕ COFFEE FILTER APPLIED for snack-1: is_coffee=True")
            print(f"☕ COFFEE FILTER APPLIED for snack-1: is_coffee=True")
        if meal_normalized == 'snack-2' and selected_tags.get('coffee_snack2'):
            meal_filters['is_coffee'] = True
            logger.info(f"☕ COFFEE FILTER APPLIED for snack-2: is_coffee=True")
            print(f"☕ COFFEE FILTER APPLIED for snack-2: is_coffee=True")
        # Backward compatibility for any legacy coffee_snack tag
        if selected_tags.get('coffee_snack'):
            meal_filters['is_coffee'] = True
            logger.info(f"☕ COFFEE FILTER APPLIED (legacy): is_coffee=True")
            print(f"☕ COFFEE FILTER APPLIED (legacy): is_coffee=True")
        
        if snack_codes:  # Only add the filter if any snack codes are selected
            meal_filters['main_dish_code__in'] = snack_codes
    
    # Add general tag filters that apply to all meals
    if selected_tags.get("bread"):
        meal_filters['Bread'] = True
    if selected_tags.get("rice"):
        meal_filters['rice'] = True
    if selected_tags.get("salad_main_dish"):
        meal_filters['salad_main_dish'] = True
    if selected_tags.get("sport"):
        meal_filters['sport'] = True
    
    logger.info(f"🔍 MEAL FILTERS APPLIED: {meal_filters}")
    
    # CRITICAL WARNING: Check for overly restrictive filters
    if 'main_dish_code__in' in meal_filters:
        codes_count = len(meal_filters['main_dish_code__in'])
        logger.warning(f"⚠️ RESTRICTIVE FILTER: Limited to {codes_count} specific codes: {meal_filters['main_dish_code__in']}")
        if codes_count < 10:
            logger.warning("   This may cause the same cards to be returned repeatedly!")

    # OPTIMIZATION: Build all filters at once using Q objects
    
    # Start with meal filters
    query = Q(**meal_filters)
    
    # Determine if this meal should be treated as excluded (0 calorie)
    is_excluded_meal = False
    if not use_default_distribution and calorie_target == 0:
        is_excluded_meal = True
        print(f"Meal {meal_normalized} is excluded (0 calories) - applying zero-calorie filter")

    # Add calorie range filter
    if is_excluded_meal:
        min_calories = 0
        max_calories = 0
    else:
        min_calories = calorie_target * 0.9
        max_calories = calorie_target * 1.1
    query &= Q(Calories__gte=min_calories, Calories__lte=max_calories)
    
    # Add allergen exclusions
    for field, value in exclude_allergen_filters.items():
        query &= ~Q(**{field: value})
    
    # Add disease exclusions
    for field, value in exclude_filters.items():
        query &= ~Q(**{field: value})
    
    # Apply month filtering if specified
    if months_filter and len(months_filter) > 0:
        logger.info(f"🗓️ APPLYING MONTH FILTER TO INITIAL QUERY: {months_filter}")
        month_q_objects = []
        for month_name in months_filter:
            # Normalize Persian month names and map to field names
            month_name = month_name.strip()
            # Normalize "اریبهشت" to "اردیبهشت"
            if month_name == "اریبهشت":
                month_name = "اردیبهشت"
            
            month_field_mapping = {
                "فروردین": "is_farvardin",
                "اردیبهشت": "is_ordibehesht",
                "خرداد": "is_khordad", 
                "تیر": "is_tir",
                "مرداد": "is_mordad",
                "شهریور": "is_shahrivar",
                "مهر": "is_mehr",
                "آبان": "is_aban",
                "آذر": "is_azar",
                "دی": "is_dey",
                "بهمن": "is_bahman",
                "اسفند": "is_esfand"
            }
            
            field_name = month_field_mapping.get(month_name)
            if field_name:
                month_q_objects.append(Q(**{field_name: True}))
        
        # Apply OR logic across selected month fields
        if month_q_objects:
            month_filter = month_q_objects[0]
            for q_obj in month_q_objects[1:]:
                month_filter |= q_obj
            query &= month_filter
    
    # Apply third layer exclusion for duplicate avoidance (cell-based session exclusion)
    duplicate_fallback = False
    
    # SERVER-SIDE CELL-SPECIFIC TRACKING: Track previously suggested food cards for this cell
    from django.core.cache import cache
    cell_session_key = f"cell_suggestions_{patient_id}_{cell_id}"
    previously_suggested_cards = cache.get(cell_session_key, [])
    
    if previously_suggested_cards:
        logger.info(f"🚫 CELL-SPECIFIC EXCLUSION: Found {len(previously_suggested_cards)} previously suggested cards for cell {cell_id}")
        logger.info(f"🚫 PREVIOUSLY SUGGESTED: {previously_suggested_cards}")
        
        # Create exclusion conditions based on previously suggested cards
        exclusion_conditions = Q()
        
        for suggested_card_data in previously_suggested_cards:
            # Exclude cards with the same main_dish_code OR FA_Name
            if 'main_dish_code' in suggested_card_data:
                exclusion_conditions |= Q(main_dish_code=suggested_card_data['main_dish_code'])
            if 'FA_Name' in suggested_card_data:
                exclusion_conditions |= Q(FA_Name=suggested_card_data['FA_Name'])
        
        if exclusion_conditions:
            query &= ~exclusion_conditions
            logger.info(f"🚫 APPLIED THIRD LAYER EXCLUSION: Excluding previously suggested cards for cell {cell_id}")
    
    # Also handle the legacy exclude_food_codes parameter (for backward compatibility)
    if exclude_food_codes and len(exclude_food_codes) > 0:
        logger.info(f"🚫 LEGACY EXCLUSION: Also excluding column food codes: {exclude_food_codes}")
        
        # Add legacy exclusion conditions
        legacy_exclusion_conditions = Q()
        for code in exclude_food_codes:
            try:
                food_code = int(code)
                legacy_exclusion_conditions |= Q(main_dish_code=food_code)
            except (ValueError, TypeError):
                continue
        
        if legacy_exclusion_conditions:
            query &= ~legacy_exclusion_conditions
    
    # OPTIMIZATION: Single database query with all filters
    cards_query = FoodCard.objects.filter(query)
    print(f"Total available cards: {cards_query.count()}")

    # OPTIMIZATION: Pre-fetch all cards into memory for faster processing
    all_cards = list(cards_query)
    logger.info(f"📦 INITIAL CARD POOL: Fetched {len(all_cards)} cards into memory")
    
    # DUPLICATE FALLBACK: If no cards found after excluding duplicates, try again without exclusion
    if len(all_cards) == 0 and exclude_food_codes and len(exclude_food_codes) > 0:
        logger.warning(f"⚠️ NO CARDS FOUND AFTER DUPLICATE EXCLUSION, TRYING FALLBACK...")
        duplicate_fallback = True
        # Remove the third layer exclusion and try again
        # Since we can't easily remove a condition from a Q object, we'll rebuild from scratch
        logger.info(f"🔄 REBUILDING QUERY FOR DUPLICATE FALLBACK")
        
        # Recreate the query without the third layer exclusion
        fallback_query = Q(**meal_filters)
        fallback_query &= Q(Calories__gte=min_calories, Calories__lte=max_calories)
        
        # Add allergen exclusions
        for field, value in exclude_allergen_filters.items():
            fallback_query &= ~Q(**{field: value})
        
        # Add disease exclusions
        for field, value in exclude_filters.items():
            fallback_query &= ~Q(**{field: value})
        
        # Re-apply month filtering if specified (same logic as before)
        if months_filter and len(months_filter) > 0:
            month_q_objects = []
            for month_name in months_filter:
                month_name = month_name.strip()
                if month_name == "اریبهشت":
                    month_name = "اردیبهشت"
                
                month_field_mapping = {
                    "فروردین": "is_farvardin", "اردیبهشت": "is_ordibehesht", "خرداد": "is_khordad",
                    "تیر": "is_tir", "مرداد": "is_mordad", "شهریور": "is_shahrivar",
                    "مهر": "is_mehr", "آبان": "is_aban", "آذر": "is_azar",
                    "دی": "is_dey", "بهمن": "is_bahman", "اسفند": "is_esfand"
                }
                field_name = month_field_mapping.get(month_name)
                if field_name:
                    month_q_objects.append(Q(**{field_name: True}))
            
            if month_q_objects:
                month_filter = month_q_objects[0]
                for q_obj in month_q_objects[1:]:
                    month_filter |= q_obj
                fallback_query &= month_filter
        
        all_cards = list(FoodCard.objects.filter(fallback_query))
        logger.info(f"📦 FALLBACK CARD POOL: Fetched {len(all_cards)} cards (allowing duplicates)")
    
    # CRITICAL FIX: If the pool is too small, broaden the search immediately
    if len(all_cards) < 5:
        logger.warning(f"⚠️ SMALL CARD POOL: Only {len(all_cards)} cards found, broadening search...")
        
        # Remove restrictive main_dish_code filter and try again with just meal type
        broadened_meal_filters = {}
        if meal_normalized == 'breakfast':
            broadened_meal_filters['is_breakfast'] = True
        elif meal_normalized == 'lunch':
            broadened_meal_filters['is_lunch'] = True
        elif meal_normalized == 'dinner':
            broadened_meal_filters['is_dinner'] = True
        elif meal_normalized in ['snack-1', 'snack-2']:
            broadened_meal_filters['is_snack'] = True
        
        # Keep general tag filters but remove specific code restrictions
        if selected_tags.get("bread"):
            broadened_meal_filters['Bread'] = True
        if selected_tags.get("rice"):
            broadened_meal_filters['rice'] = True
        if selected_tags.get("salad_main_dish"):
            broadened_meal_filters['salad_main_dish'] = True
        if selected_tags.get("sport"):
            broadened_meal_filters['sport'] = True
        # Preserve coffee filter for snacks
        if meal_normalized in ['snack-1', 'snack-2']:
            if meal_normalized == 'snack-1' and selected_tags.get('coffee_snack1'):
                broadened_meal_filters['is_coffee'] = True
                logger.info(f"☕ PRESERVING COFFEE FILTER in broadened search for snack-1")
            if meal_normalized == 'snack-2' and selected_tags.get('coffee_snack2'):
                broadened_meal_filters['is_coffee'] = True
                logger.info(f"☕ PRESERVING COFFEE FILTER in broadened search for snack-2")
            # Backward compatibility
            if selected_tags.get('coffee_snack'):
                broadened_meal_filters['is_coffee'] = True
                logger.info(f"☕ PRESERVING COFFEE FILTER in broadened search (legacy)")
            
        # Rebuild query with broadened filters
        broadened_query = Q(**broadened_meal_filters)
        broadened_query &= Q(Calories__gte=min_calories, Calories__lte=max_calories)
        
        for field, value in exclude_allergen_filters.items():
            broadened_query &= ~Q(**{field: value})
        for field, value in exclude_filters.items():
            broadened_query &= ~Q(**{field: value})
            
        # RE-APPLY MONTH FILTERING TO BROADENED QUERY
        if months_filter and len(months_filter) > 0:
            logger.info(f"🗓️ APPLYING MONTH FILTER TO BROADENED QUERY: {months_filter}")
            month_q_objects = []
            for month_name in months_filter:
                # Normalize Persian month names and map to field names
                month_name = month_name.strip()
                # Normalize "اریبهشت" to "اردیبهشت"
                if month_name == "اریبهشت":
                    month_name = "اردیبهشت"
                
                month_field_mapping = {
                    "فروردین": "is_farvardin",
                    "اردیبهشت": "is_ordibehesht",
                    "خرداد": "is_khordad", 
                    "تیر": "is_tir",
                    "مرداد": "is_mordad",
                    "شهریور": "is_shahrivar",
                    "مهر": "is_mehr",
                    "آبان": "is_aban",
                    "آذر": "is_azar",
                    "دی": "is_dey",
                    "بهمن": "is_bahman",
                    "اسفند": "is_esfand"
                }
                
                field_name = month_field_mapping.get(month_name)
                if field_name:
                    month_q_objects.append(Q(**{field_name: True}))
            
            # Apply OR logic across selected month fields
            if month_q_objects:
                month_filter = month_q_objects[0]
                for q_obj in month_q_objects[1:]:
                    month_filter |= q_obj
                broadened_query &= month_filter
            
        broadened_cards_query = FoodCard.objects.filter(broadened_query)
        broadened_cards = list(broadened_cards_query)
        
        if len(broadened_cards) > len(all_cards):
            logger.info(f"✅ BROADENED SEARCH: Expanded from {len(all_cards)} to {len(broadened_cards)} cards")
            all_cards = broadened_cards
        else:
            logger.warning(f"❌ BROADENING FAILED: Still only {len(broadened_cards)} cards available")


    # Set up logging for refresh cell operations
    logger = logging.getLogger(__name__)
    


    # OPTIMIZED: Build used identities for column-level uniqueness
    used_identities = set()
    used_slugs = set()
    current_fa_norm = None
    current_en_norm = None
    current_slug = None

    # Log refresh cell operation details
    logger.info(f"Refresh cell operation - patient_id={patient_id}, meal_type={meal_normalized}, "
                f"cell_index={cell_index}, column_data_length={len(column_data) if column_data else 0}")

    # Initialize variables to avoid NameError
    valid_codes_in_column = []
    current_code = None
    
    try:
        # CRITICAL FIX: Improved column data parsing
        
        if isinstance(column_data, list) and column_data:
            logger.debug(f"Raw column_data: {column_data}")
            
            for idx, raw_code in enumerate(column_data):
                try:
                    # Handle different possible formats
                    if isinstance(raw_code, dict):
                        code_val = int(raw_code.get('main_dish_code', -1))
                    elif isinstance(raw_code, (int, float)):
                        code_val = int(raw_code)
                    elif isinstance(raw_code, str):
                        # Try to parse string as int
                        code_val = int(raw_code) if raw_code.strip().isdigit() else -1
                    else:
                        logger.warning(f"Unexpected column_data format at index {idx}: {type(raw_code)} = {raw_code}")
                        continue
                        
                    # Skip invalid codes (negative, zero, or invalid)
                    if code_val <= 0:
                        continue
                        
                    # CRITICAL: Check if this is the cell being replaced
                    if cell_index is not None and idx == cell_index:
                        current_code = code_val
                        logger.debug(f"Found current cell code: {current_code} at index {cell_index}")
                    else:
                        valid_codes_in_column.append(code_val)
                        
                except (ValueError, TypeError) as e:
                    logger.warning(f"Failed to parse code at index {idx}: {raw_code} - {e}")
                    continue

        # OPTIMIZATION: Get used cards from already fetched all_cards instead of new query
        used_cards_dict = {}
        if valid_codes_in_column:
            # PERFORMANCE: Filter from in-memory all_cards instead of database query
            used_cards = [card for card in all_cards if card.main_dish_code in valid_codes_in_column]
            
            # If not found in memory (edge case), query database
            if not used_cards and len(valid_codes_in_column) > 0:
                logger.info(f"  - Used cards not in memory, querying database for {len(valid_codes_in_column)} codes")
                used_cards = FoodCard.objects.filter(main_dish_code__in=valid_codes_in_column).only(
                    'main_dish_code', 'FA_Name', 'EN_Name'
                )
            else:
                logger.info(f"  - Found {len(used_cards)} used cards in memory (no database query needed)")
            
            for used_card in used_cards:
                fa, en = _card_identities(used_card)
                slug = _create_identity_slug(fa, en)
                
                # Store for debugging
                used_cards_dict[used_card.main_dish_code] = {
                    'fa': fa, 'en': en, 'slug': slug,
                    'original_fa': used_card.FA_Name, 'original_en': used_card.EN_Name
                }
                
                # Add to exclusion sets
                if fa:
                    used_identities.add(fa)
                if en:
                    used_identities.add(en)
                if slug:
                    used_slugs.add(slug)

        # Get current card info (to avoid selecting same card) - from memory first (OPTIMIZATION!)
        current_card_info = None
        if current_code is not None and current_code > 0:
            try:
                # PERFORMANCE: Look in already fetched all_cards first
                current_card = next((card for card in all_cards if card.main_dish_code == current_code), None)
                
                # If not found in memory, query database
                if not current_card:
                    logger.info(f"  - Current card (code={current_code}) not in memory, querying database")
                    current_card = FoodCard.objects.filter(main_dish_code=current_code).only(
                        'main_dish_code', 'FA_Name', 'EN_Name'
                    ).first()
                else:
                    logger.info(f"  - Found current card in memory (no database query needed)")
                
                if current_card:
                    current_fa_norm, current_en_norm = _card_identities(current_card)
                    current_slug = _create_identity_slug(current_fa_norm, current_en_norm)
                    current_card_info = {
                        'code': current_code,
                        'fa': current_fa_norm, 'en': current_en_norm, 'slug': current_slug,
                        'original_fa': current_card.FA_Name, 'original_en': current_card.EN_Name
                    }
            except Exception as e:
                logger.error(f"Failed to get current card info for code {current_code}: {e}")

        # DETAILED LOGGING for debugging
        logger.info(f"Column analysis complete:")
        logger.info(f"  - Valid codes in column: {valid_codes_in_column}")
        logger.info(f"  - Used identities: {len(used_identities)}")
        logger.info(f"  - Used slugs: {len(used_slugs)}")
        logger.info(f"  - Current card code: {current_code}")
        logger.debug(f"  - Used cards details: {used_cards_dict}")
        logger.debug(f"  - Current card info: {current_card_info}")

    except Exception as e:
        logger.error(f"CRITICAL: Failed to build used identities: {e}", exc_info=True)
        # Continue with empty sets to avoid total failure
        used_identities = set()
        used_slugs = set()
        current_fa_norm = None
        current_en_norm = None
        current_slug = None

    # OPTIMIZED: Enhanced candidate filtering with strict duplicate detection - using module-level function

    # PERFORMANCE: Initial candidate filtering with strict uniqueness
    candidate_pool_size_before_filter = len(all_cards)
    candidates = _filter_candidates(all_cards, strict=True, debug_name="initial_pool", 
                                     used_identities=used_identities, used_slugs=used_slugs,
                                     current_fa_norm=current_fa_norm, current_en_norm=current_en_norm,
                                     current_slug=current_slug, current_code=current_code)
    candidate_pool_size_after_filter = len(candidates)
    
    logger.info(f"FILTERING RESULTS:")
    logger.info(f"  - Before filtering: {candidate_pool_size_before_filter} cards")
    logger.info(f"  - After filtering: {candidate_pool_size_after_filter} cards")
    logger.info(f"  - Filtered out: {candidate_pool_size_before_filter - candidate_pool_size_after_filter} cards")
    
    # CRITICAL DEBUGGING: Check if we have variety in candidates
    if candidates:
        candidate_codes = [c.main_dish_code for c in candidates[:10]]  # First 10 codes
        candidate_names = [c.FA_Name for c in candidates[:5]]  # First 5 names
        logger.info(f"  - Available candidate codes: {candidate_codes}")
        logger.info(f"  - Available candidate names: {candidate_names}")
        logger.info(f"  - Total candidates available for selection: {len(candidates)}")
    else:
        logger.warning("  - NO CANDIDATES AVAILABLE AFTER FILTERING!")

    # OPTIMIZED: Fallback strategy implementation with in-memory filtering
    fallback_used = None
    degraded_selection = False
    
    # Fallback 1: Broaden calorie range using already fetched cards (OPTIMIZATION!)
    if not candidates:
        fallback_used = "broadened_calories"
        logger.info("FALLBACK 1: Broadening calorie range (80%-120%) in memory")
        
        broaden_min = calorie_target * 0.8
        broaden_max = calorie_target * 1.2
        
        # OPTIMIZATION: Filter in memory instead of new database query
        broaden_cards = [
            card for card in all_cards
            if broaden_min <= card.Calories <= broaden_max
        ]
        
        logger.info(f"  - In-memory filtering: {len(all_cards)} cards → {len(broaden_cards)} within broadened range")
        
        # If we still need more cards, then query database
        if len(broaden_cards) < 5:
            logger.info("  - Pool still too small, querying database for more cards")
            broaden_query = Q(**meal_filters)
            broaden_query &= Q(Calories__gte=broaden_min, Calories__lte=broaden_max)
            
            for field, value in exclude_allergen_filters.items():
                broaden_query &= ~Q(**{field: value})
            for field, value in exclude_filters.items():
                broaden_query &= ~Q(**{field: value})
                
            broaden_cards = list(FoodCard.objects.filter(broaden_query).only(
                'main_dish_code', 'FA_Name', 'EN_Name', 'Calories', 'foods',
                'is_breakfast', 'is_lunch', 'is_dinner', 'is_snack'
            ))
        
        candidates = _filter_candidates(broaden_cards, strict=True, debug_name="broadened_calories",
                                         used_identities=used_identities, used_slugs=used_slugs,
                                         current_fa_norm=current_fa_norm, current_en_norm=current_en_norm,
                                         current_slug=current_slug, current_code=current_code)
        
        logger.info(f"Fallback 1 results: {len(candidates)} candidates")

    # Fallback 2: Very broad calorie range - try in-memory first (OPTIMIZATION!)
    if not candidates:
        fallback_used = "very_broad_calories"
        logger.info("FALLBACK 2: Very broad calorie range (70%-130%) in memory first")
        
        very_broad_min = calorie_target * 0.7  
        very_broad_max = calorie_target * 1.3
        
        # OPTIMIZATION: Try filtering already fetched cards first
        very_broad_cards = [
            card for card in all_cards
            if very_broad_min <= card.Calories <= very_broad_max
        ]
        
        logger.info(f"  - In-memory filtering: {len(all_cards)} cards → {len(very_broad_cards)} within very broad range")
        
        # If pool is still too small, query database with relaxed filters
        if len(very_broad_cards) < 5:
            logger.info("  - Pool still too small, querying database with relaxed meal preferences")
            
            basic_meal_filters = {}
            if meal_normalized == 'breakfast':
                basic_meal_filters['is_breakfast'] = True
            elif meal_normalized == 'lunch':
                basic_meal_filters['is_lunch'] = True
            elif meal_normalized == 'dinner':
                basic_meal_filters['is_dinner'] = True
            elif meal_normalized in ['snack-1', 'snack-2']:
                basic_meal_filters['is_snack'] = True
            
            very_broad_query = Q(**basic_meal_filters)
            very_broad_query &= Q(Calories__gte=very_broad_min, Calories__lte=very_broad_max)
            
            # STILL keep critical health restrictions
            for field, value in exclude_allergen_filters.items():
                very_broad_query &= ~Q(**{field: value})
            for field, value in exclude_filters.items():
                very_broad_query &= ~Q(**{field: value})
                
            very_broad_cards = list(FoodCard.objects.filter(very_broad_query).only(
                'main_dish_code', 'FA_Name', 'EN_Name', 'Calories', 'foods',
                'is_breakfast', 'is_lunch', 'is_dinner', 'is_snack'
            ))
        
        candidates = _filter_candidates(very_broad_cards, strict=True, debug_name="very_broad_calories",
                                         used_identities=used_identities, used_slugs=used_slugs,
                                         current_fa_norm=current_fa_norm, current_en_norm=current_en_norm,
                                         current_slug=current_slug, current_code=current_code)
        
        logger.info(f"Fallback 2 results: {len(candidates)} candidates")

    # Fallback 3: Allow column duplicates (but never the current card) - already in memory (OPTIMIZED!)
    if not candidates:
        fallback_used = "allow_column_duplicates"
        degraded_selection = True
        logger.warning("FALLBACK 3: Allowing column duplicates due to extreme scarcity (using in-memory cards)")
        
        # OPTIMIZATION: Use already fetched all_cards - no new database query needed
        candidates = _filter_candidates(all_cards, strict=False, debug_name="allow_duplicates_original",
                                         used_identities=used_identities, used_slugs=used_slugs,
                                         current_fa_norm=current_fa_norm, current_en_norm=current_en_norm,
                                         current_slug=current_slug, current_code=current_code)
        
        # If still empty, try with very broad calorie range + non-strict - in memory first! (OPTIMIZATION!)
        if not candidates:
            logger.warning("FALLBACK 3.1: Using emergency broadened search with duplicates allowed (in-memory first)")
            
            emergency_min = calorie_target * 0.6
            emergency_max = calorie_target * 1.4
            
            # OPTIMIZATION: Try in-memory filtering first
            emergency_cards = [
                card for card in all_cards
                if emergency_min <= card.Calories <= emergency_max
            ]
            
            logger.info(f"  - In-memory emergency filter: {len(all_cards)} cards → {len(emergency_cards)} within emergency range")
            
            # Only query database if in-memory pool is insufficient
            if len(emergency_cards) < 3:
                logger.warning("  - Emergency pool too small, querying database with most permissive filters")
                
                emergency_filters = {}
                if meal_normalized == 'breakfast':
                    emergency_filters['is_breakfast'] = True
                elif meal_normalized == 'lunch':
                    emergency_filters['is_lunch'] = True
                elif meal_normalized == 'dinner':
                    emergency_filters['is_dinner'] = True
                elif meal_normalized in ['snack-1', 'snack-2']:
                    emergency_filters['is_snack'] = True
                    
                emergency_query = Q(**emergency_filters)
                emergency_query &= Q(Calories__gte=emergency_min, Calories__lte=emergency_max)
                
                # Only exclude allergens (most critical)
                for field, value in exclude_allergen_filters.items():
                    emergency_query &= ~Q(**{field: value})
                    
                emergency_cards = list(FoodCard.objects.filter(emergency_query).only(
                    'main_dish_code', 'FA_Name', 'EN_Name', 'Calories', 'foods',
                    'is_breakfast', 'is_lunch', 'is_dinner', 'is_snack'
                ))
                
            candidates = _filter_candidates(emergency_cards, strict=False, debug_name="emergency_pool",
                                             used_identities=used_identities, used_slugs=used_slugs,
                                             current_fa_norm=current_fa_norm, current_en_norm=current_en_norm,
                                             current_slug=current_slug, current_code=current_code)
        
        logger.warning(f"Fallback 3 results: {len(candidates)} candidates (degraded=True)")

    # SIMPLIFIED: Using module-level _select_card_with_variety function

    # BISCUIT CARD RESTRICTION FOR SNACKS
    # Import biscuit card helper functions from diet_generation
    from panel.services.diet_generation import _is_biscuit_card, _is_biscuit_code
    
    # Check if this is a snack meal that needs biscuit restriction
    is_snack_meal = meal_normalized in ['snack-1', 'snack-2', 'snack1', 'snack2']
    
    # Initialize biscuit info for snacks
    biscuit_info = None
    existing_biscuit_count = 0
    candidates_debug_info = None
    
    if is_snack_meal:
        # Count biscuit cards already in the column
        if valid_codes_in_column:
            # Count how many of the 14 displayed cards are biscuits
            # Simply check if each code is in the predefined biscuit codes list
            existing_biscuit_count = 0
            biscuit_codes_found = []
            
            for code in valid_codes_in_column:
                if code and _is_biscuit_code(code):
                    existing_biscuit_count += 1
                    if code not in biscuit_codes_found:
                        biscuit_codes_found.append(code)
            
            # Log which biscuit cards were found
            if existing_biscuit_count > 0:
                logger.info(f"🍪 Found {existing_biscuit_count} biscuit card(s) in column with codes: {biscuit_codes_found}")
        
        # Store biscuit info to return in response
        biscuit_info = {
            'count': existing_biscuit_count,
            'max_allowed': 2,
            'is_at_limit': existing_biscuit_count >= 2
        }
        
        logger.info(f"🍪 BISCUIT CARD CHECK: {existing_biscuit_count} biscuit cards already in {meal_normalized} column")
        
        # Capture candidates before filter for debug
        candidates_before_filter = []
        if candidates:
            for card in candidates:
                candidates_before_filter.append({
                    'id': card.id,
                    'fa_name': card.FA_Name,
                    'en_name': card.EN_Name,
                    'main_dish_code': card.main_dish_code,
                    'calories': float(card.Calories),
                    'is_biscuit': _is_biscuit_card(card)
                })
        
        # If we already have 2 or more biscuit cards, filter them out from candidates
        if existing_biscuit_count >= 2 and candidates:
            candidates_before_biscuit_filter = len(candidates)
            candidates = [card for card in candidates if not _is_biscuit_card(card)]
            candidates_after_biscuit_filter = len(candidates)
            
            # Capture candidates after filter for debug
            candidates_after_filter = []
            for card in candidates:
                candidates_after_filter.append({
                    'id': card.id,
                    'fa_name': card.FA_Name,
                    'en_name': card.EN_Name,
                    'main_dish_code': card.main_dish_code,
                    'calories': float(card.Calories),
                    'is_biscuit': _is_biscuit_card(card)
                })
            
            # Determine if this was applied to fallback candidates
            fallback_context = f" (from {fallback_used})" if fallback_used else ""
            
            candidates_debug_info = {
                'restriction_applied': True,
                'before_count': candidates_before_biscuit_filter,
                'after_count': candidates_after_biscuit_filter,
                'filtered_out': candidates_before_biscuit_filter - candidates_after_biscuit_filter,
                'before_sample': candidates_before_filter,
                'after_sample': candidates_after_filter,
                'fallback_used': fallback_used
            }
            
            print(f"🚫 BISCUIT RESTRICTION APPLIED{fallback_context}: Maximum biscuit cards (2) reached")
            print(f"   - Candidates before biscuit filter: {candidates_before_biscuit_filter}")
            print(f"   - Candidates after biscuit filter: {candidates_after_biscuit_filter}")
            print(f"   - Biscuit cards filtered out: {candidates_before_biscuit_filter - candidates_after_biscuit_filter}")
        else:
            # No restriction applied
            candidates_debug_info = {
                'restriction_applied': False,
                'before_count': len(candidates),
                'before_sample': candidates_before_filter,
                'fallback_used': fallback_used
            }

    # OPTIMIZED: Final selection and comprehensive telemetry
    selected_cards = []
    selected_card = None
    
    # Capture final candidates for debug (after ALL filtering)
    final_candidates_debug = []
    selected_card_info = None
    
    if candidates:
        for card in candidates:
            final_candidates_debug.append({
                'id': card.id,
                'fa_name': card.FA_Name,
                'en_name': card.EN_Name,
                'main_dish_code': card.main_dish_code,
                'calories': float(card.Calories),
                'is_biscuit': _is_biscuit_card(card)
            })
    
    if candidates:
        # Prepare context data for variety selection
        selection_context = {
            'patient_id': patient_id,
            'meal_normalized': meal_normalized,
            'cell_index': cell_index,
            'used_identities_count': len(used_identities),
            'candidate_pool_size_after_filter': candidate_pool_size_after_filter
        }
        selected_card = _select_card_with_variety(candidates, selection_context)
        
    if selected_card:
        selected_cards.append(selected_card)
        
        # DETAILED SUCCESS LOGGING
        selected_fa, selected_en = _card_identities(selected_card)
        selected_slug = _create_identity_slug(selected_fa, selected_en)
        
        # Capture selected card info for debug
        selected_card_info = {
            'id': selected_card.id,
            'fa_name': selected_card.FA_Name,
            'en_name': selected_card.EN_Name,
            'main_dish_code': selected_card.main_dish_code,
            'calories': float(selected_card.Calories),
            'is_biscuit': _is_biscuit_card(selected_card),
            'total_candidates': len(final_candidates_debug)
        }
        
        logger.info(f"✅ CARD SELECTED SUCCESSFULLY:")
        logger.info(f"   - ID: {selected_card.id}, Code: {selected_card.main_dish_code}")
        logger.info(f"   - FA: '{selected_card.FA_Name}' → '{selected_fa}'")
        logger.info(f"   - EN: '{selected_card.EN_Name}' → '{selected_en}'") 
        logger.info(f"   - Slug: '{selected_slug}'")
        logger.info(f"   - Calories: {selected_card.Calories}")
        logger.info(f"   - Fallback: {fallback_used or 'none'}, Degraded: {degraded_selection}")
        
        # CELL-SPECIFIC TRACKING: Store this card for future exclusion in the same cell
        try:
            from django.core.cache import cache
            cell_session_key = f"cell_suggestions_{patient_id}_{cell_id}"
            
            # Get existing suggestions for this cell
            previously_suggested_cards = cache.get(cell_session_key, [])
            
            # Create card data for tracking
            card_tracking_data = {
                'main_dish_code': selected_card.main_dish_code,
                'FA_Name': selected_card.FA_Name,
                'EN_Name': selected_card.EN_Name,
                'selected_at': datetime.now().isoformat(),
                'card_id': selected_card.id
            }
            
            # Add this card to the exclusion list for this cell
            previously_suggested_cards.append(card_tracking_data)
            
            # Store in cache for 24 hours (86400 seconds)
            cache.set(cell_session_key, previously_suggested_cards, 86400)
            
            logger.info(f"💾 CELL TRACKING: Stored card {selected_card.main_dish_code} - '{selected_card.FA_Name}' for cell {cell_id}")
            logger.info(f"💾 CELL TRACKING: Total tracked cards for cell {cell_id}: {len(previously_suggested_cards)}")
            
        except Exception as cache_error:
            logger.error(f"❌ CELL TRACKING ERROR: Failed to store card for exclusion: {cache_error}")
        
        # VERIFICATION: Double-check this card isn't a duplicate
        is_duplicate_check = (
            (selected_fa and selected_fa in used_identities) or
            (selected_en and selected_en in used_identities) or
            (selected_slug and selected_slug in used_slugs)
        )
        is_same_as_current_check = (
            (current_fa_norm and selected_fa == current_fa_norm) or
            (current_en_norm and selected_en == current_en_norm) or
            (current_slug and selected_slug == current_slug) or
            (current_code and selected_card.main_dish_code == current_code)
        )
        
        if is_duplicate_check and not degraded_selection:
            logger.error(f"🚨 CRITICAL ERROR: Selected card is a duplicate! This should not happen.")
            logger.error(f"   - Selected FA '{selected_fa}' in used_identities: {selected_fa in used_identities}")
            logger.error(f"   - Selected EN '{selected_en}' in used_identities: {selected_en in used_identities}")
            logger.error(f"   - Selected slug '{selected_slug}' in used_slugs: {selected_slug in used_slugs}")
            
        if is_same_as_current_check:
            logger.error(f"🚨 CRITICAL ERROR: Selected card is same as current! This should NEVER happen.")
            logger.error(f"   - Same FA: {selected_fa == current_fa_norm}")
            logger.error(f"   - Same EN: {selected_en == current_en_norm}")
            logger.error(f"   - Same slug: {selected_slug == current_slug}")
            logger.error(f"   - Same code: {selected_card.main_dish_code == current_code}")
        
    else:
        # EMERGENCY FALLBACK: Try to return ANY valid card for this meal type
        logger.warning(f"🚨 EMERGENCY FALLBACK: No candidates found, trying basic meal query")
        
        try:
            emergency_filters = {}
            if meal_normalized == 'breakfast':
                emergency_filters['is_breakfast'] = True
            elif meal_normalized == 'lunch':
                emergency_filters['is_lunch'] = True
            elif meal_normalized == 'dinner':
                emergency_filters['is_dinner'] = True
            elif meal_normalized in ['snack-1', 'snack-2']:
                emergency_filters['is_snack'] = True
            
            # Very basic query - just meal type and reasonable calorie range
            emergency_query = FoodCard.objects.filter(**emergency_filters)
            emergency_query = emergency_query.filter(
                Calories__gte=calorie_target * 0.5,
                Calories__lte=calorie_target * 1.5
            )
            
            emergency_cards = list(emergency_query[:10])  # Get first 10 cards
            
            # Apply biscuit restriction for snacks in emergency fallback too
            if is_snack_meal and emergency_cards and existing_biscuit_count >= 2:
                logger.warning(f"🍪 EMERGENCY FALLBACK: Applying biscuit restriction ({existing_biscuit_count}/2 biscuits already in column)")
                emergency_cards_before = len(emergency_cards)
                emergency_cards = [card for card in emergency_cards if not _is_biscuit_card(card)]
                logger.warning(f"🍪 EMERGENCY FALLBACK: Filtered {emergency_cards_before - len(emergency_cards)} biscuit cards")
            
            if emergency_cards:
                # Use the simple selection method
                emergency_card = _select_card_with_variety(emergency_cards, {
                    'patient_id': patient_id,
                    'meal_normalized': meal_normalized,
                    'cell_index': cell_index
                })
                
                if emergency_card:
                    logger.warning(f"✅ EMERGENCY CARD SELECTED: {emergency_card.main_dish_code} - '{emergency_card.FA_Name}'")
                    selected_cards = [emergency_card]
                    fallback_used = "emergency_any_card"
                    degraded_selection = True
                    
                    # CELL-SPECIFIC TRACKING: Store emergency card for future exclusion
                    try:
                        from django.core.cache import cache
                        cell_session_key = f"cell_suggestions_{patient_id}_{cell_id}"
                        
                        previously_suggested_cards = cache.get(cell_session_key, [])
                        
                        card_tracking_data = {
                            'main_dish_code': emergency_card.main_dish_code,
                            'FA_Name': emergency_card.FA_Name,
                            'EN_Name': emergency_card.EN_Name,
                            'selected_at': datetime.now().isoformat(),
                            'card_id': emergency_card.id
                        }
                        
                        previously_suggested_cards.append(card_tracking_data)
                        cache.set(cell_session_key, previously_suggested_cards, 86400)
                        
                        logger.info(f"💾 EMERGENCY CELL TRACKING: Stored emergency card {emergency_card.main_dish_code} for cell {cell_id}")
                        
                    except Exception as cache_error:
                        logger.error(f"❌ EMERGENCY CELL TRACKING ERROR: {cache_error}")
                    
        except Exception as emergency_error:
            logger.error(f"Emergency fallback also failed: {emergency_error}")
        
        # If we still have no cards, return an error
        if not selected_cards:
            logger.error(f"❌ NO CARDS FOUND EVEN WITH EMERGENCY FALLBACK:")
            logger.error(f"   - Patient: {patient_id}, Meal: {meal_normalized}")
            logger.error(f"   - Calorie target: {calorie_target}")
            
            return Response({
                "error": "No suitable food options available",
                "message": "Please contact support - this indicates a data issue",
                "details": {
                    "meal_type": meal_normalized,
                    "calorie_target": calorie_target,
                    "fallback_attempted": fallback_used
                }
            }, status=422)

    # Format response
    response_cards = []
    for card in selected_cards:
        response_card = {
            'FA_Name': card.FA_Name,
            'EN_Name': card.EN_Name,
            'Calories': float(card.Calories),
            'foods': card.foods,
            'fat': 0,
            'protein': 0,
            'carbohydrate': 0,
            'is_breakfast': card.is_breakfast,
            'is_lunch': card.is_lunch,
            'is_dinner': card.is_dinner,
            'is_snack': card.is_snack,
            'main_dish_code': card.main_dish_code
        }
        
        # Calculate nutritional values safely
        for food in card.foods:
            if isinstance(food, dict):
                try:
                    response_card['fat'] += float(food.get('Fat', 0))
                    response_card['protein'] += float(food.get('Protein', 0))
                    response_card['carbohydrate'] += float(food.get('Carbohydrates', 0))
                except (ValueError, TypeError):
                    logger.warning(f"Invalid nutritional value in food: {food}")
                    continue
        
        response_cards.append(response_card)

    # COMPREHENSIVE FINAL TELEMETRY
    operation_summary = {
        "action": "refresh_cell",
        "status": "success",
        "meal_type": meal_normalized,
        "patient_id": patient_id,
        "cell_index": cell_index,
        "selected_card_count": len(response_cards),
        "candidate_pool_before_filter": candidate_pool_size_before_filter,
        "candidate_pool_after_filter": candidate_pool_size_after_filter,
        "filtered_out_count": candidate_pool_size_before_filter - candidate_pool_size_after_filter,
        "used_identities_count": len(used_identities),
        "used_slugs_count": len(used_slugs),
        "fallback_strategy": fallback_used or "none",
        "degraded_selection": degraded_selection,
        "calorie_target": calorie_target,
        "current_card_code": current_code
    }
    
    if selected_cards:
        selected_card = selected_cards[0]
        operation_summary.update({
            "selected_card_id": selected_card.id,
            "selected_card_code": selected_card.main_dish_code,
            "selected_card_fa": selected_card.FA_Name,
            "selected_card_en": selected_card.EN_Name,
            "selected_card_calories": float(selected_card.Calories)
        })
    
    logger.info(f"🎯 REFRESH CELL OPERATION COMPLETED:")
    logger.info(f"   - Status: {'SUCCESS' if response_cards else 'FAILED'}")
    logger.info(f"   - Cards generated: {len(response_cards)}")
    logger.info(f"   - Fallback strategy: {fallback_used or 'none'}")
    logger.info(f"   - Degraded selection: {degraded_selection}")
    logger.info(f"   - Performance: {candidate_pool_size_before_filter}→{candidate_pool_size_after_filter} candidates")
    
    # Build response with biscuit info if applicable
    response_data = {
        "cards": response_cards,
        "duplicate_fallback": duplicate_fallback,
        "excluded_meals": excluded_meals,
        "telemetry": operation_summary,
        "debug_info": {
            "column_data_received": column_data,
            "valid_codes_in_column": valid_codes_in_column,
            "normalization_test": {
                "معجون با موز": _normalize_text("معجون با موز"),
                "معجون با موز ": _normalize_text("معجون با موز "),
                "معجون  با موز": _normalize_text("معجون  با موز")
            }
        } if logger.isEnabledFor(logging.DEBUG) else {}
    }
    
    # Add biscuit info for snack meals
    if biscuit_info is not None:
        response_data["biscuit_info"] = biscuit_info
        logger.info(f"🍪 BISCUIT INFO ADDED TO RESPONSE: {biscuit_info}")
    
    # Add candidates debug info for snack meals
    if candidates_debug_info is not None:
        response_data["candidates_debug"] = candidates_debug_info
        print(f"🔍 CANDIDATES DEBUG INFO ADDED TO RESPONSE")
    
    # Add final candidates list (after ALL filtering)
    if final_candidates_debug:
        response_data["final_candidates"] = final_candidates_debug
        print(f"🎯 FINAL CANDIDATES ({len(final_candidates_debug)} cards) ADDED TO RESPONSE")
    
    # Add selected card info
    if selected_card_info:
        response_data["selected_card_info"] = selected_card_info
        print(f"✨ SELECTED CARD INFO ADDED TO RESPONSE: {selected_card_info['fa_name']}")
    
    return Response(response_data)

# Add a final catch-all exception handler for the entire function
generate_cells_card_original = generate_cells_card

def generate_cells_card_with_error_handling(request):
    """Wrapper function to provide comprehensive error handling"""
    import logging
    logger = logging.getLogger(__name__)
    
    try:
        return generate_cells_card_original(request)
    except Exception as e:
        logger.error(f"🚨 CRITICAL ERROR in generate_cells_card: {e}", exc_info=True)
        return Response({
            "error": "An unexpected error occurred while generating the card",
            "message": "Please try again or contact support if the problem persists",
            "debug": str(e) if logger.isEnabledFor(logging.DEBUG) else None
        }, status=500)

# Replace the original function
generate_cells_card = generate_cells_card_with_error_handling

def diet_cards_ajax(request):
    # Retrieve filter parameters from GET
    search_query = request.GET.get('card_search_query', '').strip()
    include_tags = request.GET.getlist('include_tags')
    exclude_tags = request.GET.getlist('exclude_tags')
    min_calories = request.GET.get('min_calories', '').strip()
    max_calories = request.GET.get('max_calories', '').strip()
    sort_field = request.GET.get('sort_field', 'id')  # Default sort field
    sort_order = request.GET.get('sort_order', 'desc')
    card_page_number = request.GET.get('card_page', 1)

    # Determine if any meaningful filter is active.
    filter_active = any([
        search_query,
        include_tags,  # non-empty list is truthy
        exclude_tags,
        min_calories,
        max_calories
    ])

    if not filter_active:
        # No filter applied: return an empty queryset to avoid scanning all records.
        card_list = FoodCard.objects.none()
    else:
        # Initialize a Q object for complex conditions and a dictionary for simple lookups.
        q_filter = Q()
        filter_dict = {}

        # For text search: we want to search in FA_Name OR EN_Name.
        if search_query:
            q_filter &= (Q(FA_Name__icontains=search_query) | Q(EN_Name__icontains=search_query))

        # For numeric calorie filters, add them to the dictionary.
        if min_calories:
            try:
                filter_dict['Calories__gte'] = int(min_calories)
            except ValueError:
                pass
        if max_calories:
            try:
                filter_dict['Calories__lte'] = int(max_calories)
            except ValueError:
                pass

        # For include tags (assume these GET parameter names match Boolean field names).
        for tag in include_tags:
            filter_dict[tag] = True

        # Build the queryset from both the Q object and the filter dictionary.
        card_list = FoodCard.objects.filter(q_filter, **filter_dict)

        # Apply exclusion for any tags.
        for tag in exclude_tags:
            # Skip invalid tags like 'undefined' or empty strings
            if tag and tag != 'undefined' and hasattr(FoodCard, tag):
                card_list = card_list.exclude(**{tag: True})

        # Apply sorting
        if sort_order == 'desc':
            sort_field = '-' + sort_field
        card_list = card_list.order_by(sort_field)

    # Pagination: 20 items per page
    paginator = Paginator(card_list, 20)
    try:
        card_page = paginator.page(card_page_number)
    except PageNotAnInteger:
        card_page = paginator.page(1)
    except EmptyPage:
        card_page = paginator.page(paginator.num_pages)

    # Render the partial templates
    cards_html = render_to_string('assistant/diet_cards_list.html', {'card_page': card_page})
    pagination_html = render_to_string('assistant/diet_cards_pagination.html', {'card_page': card_page})

    return JsonResponse({'cards_html': cards_html, 'pagination_html': pagination_html})


def diet_foods_ajax(request):
    search_query = request.GET.get('food_search_query', '').strip()
    food_page_number = request.GET.get('food_page', 1)

    # If no search query is provided, return an empty queryset.
    if search_query:
        food_list = Food.objects.filter(FA_Name__icontains=search_query)
    else:
        food_list = Food.objects.none()

    # Optional: Cache the response based on GET parameters.
    params = request.GET.dict()
    params_json = json.dumps(params, sort_keys=True)
    cache_key = "diet_foods_ajax_" + hashlib.md5(params_json.encode('utf-8')).hexdigest()
    cached_response = cache.get(cache_key)
    if cached_response:
        return JsonResponse(cached_response)

    food_paginator = Paginator(food_list, 20)
    try:
        food_page = food_paginator.page(food_page_number)
    except (PageNotAnInteger, EmptyPage):
        food_page = food_paginator.page(1)

    foods_html = render_to_string('assistant/diet_foods_list.html', {'food_page': food_page})
    pagination_html = render_to_string('assistant/diet_foods_pagination.html', {'food_page': food_page})
    response_data = {'foods_html': foods_html, 'pagination_html': pagination_html}
    cache.set(cache_key, response_data, 300)  # Cache for 5 minutes
    return JsonResponse(response_data)


@csrf_exempt
def get_card_nutrients(request, card_id):
    """
    AJAX endpoint to fetch full nutrient data for all foods in a card.
    Calculates total nutrients based on the amounts in the card.
    """
    try:
        card = FoodCard.objects.get(id=card_id)
        food_items = card.foods  # This is a JSONField containing list of food items
        
        if not food_items or not isinstance(food_items, list):
            return JsonResponse({'error': 'No foods found in card'}, status=400)
        
        # Extract food codes and get Food objects
        food_codes = [str(item.get('food_code', '')) for item in food_items if item.get('food_code')]
        foods = Food.objects.filter(Data_Base_Number__in=food_codes)
        food_map = {str(food.Data_Base_Number): food for food in foods}
        
        # Initialize totals
        totals = {
            'Protein_g': 0, 'Fat_g': 0, 'Carbohydrates_g': 0, 'Calories': 0,
            'Fiber_g': 0, 'Saturated_Fat_g': 0, 'Sugar_g': 0, 'Sodium_mg': 0,
            'Calcium_mg': 0, 'Iron_mg': 0, 'Magnesium_mg': 0, 'Phosphorus_mg': 0,
            'Potasssium_mg': 0, 'Zinc_mg': 0, 'Cupper_mg': 0, 'Manganese_mg': 0,
            'Selenium_mcg': 0, 'Vitamin_A_IU': 0, 'Vitamin_C_mg': 0, 'Vitamin_D_mcg': 0,
            'Vitamin_E_mg': 0, 'Vitamin_B1_mg': 0, 'Vitamin_B2_mg': 0, 'Vitamin_B3_mg': 0,
            'Vitamin_B5_mg': 0, 'Vitamin_B6_mg': 0, 'Vitamin_B9_mg': 0, 'Vitamin_B12': 0,
            'Choline_mg': 0, 'Cholesterol_mg': 0, 'Water_g': 0, 'Ash_g': 0,
            'Starch_g': 0, 'Sucrose_g': 0, 'Glucose_g': 0, 'Fructose_g': 0,
            'Lactose_g': 0, 'Maltose_g': 0, 'Alcohol_g': 0, 'Caffeine_mg': 0,
            'MUFA': 0, 'PUFA': 0
        }
        
        # Calculate nutrients for each food item
        for food_item in food_items:
            food_code = str(food_item.get('food_code', ''))
            amount_g = float(food_item.get('amount', 0))  # Amount in grams
            
            if not food_code or amount_g <= 0:
                continue
                
            food = food_map.get(food_code)
            if not food:
                continue
            
            # Calculate nutrients based on amount (per 100g values * amount / 100)
            ratio = amount_g / 100.0
            
            totals['Protein_g'] += food.Protein_g * ratio
            totals['Fat_g'] += food.Fat_g * ratio
            totals['Carbohydrates_g'] += food.Carbohydrates_g * ratio
            totals['Calories'] += food.Calories * ratio
            totals['Fiber_g'] += food.Fiber_g * ratio
            totals['Saturated_Fat_g'] += food.Saturated_Fat_g * ratio
            totals['Sugar_g'] += food.Sugar_g * ratio
            totals['Sodium_mg'] += food.Sodium_mg * ratio
            totals['Calcium_mg'] += food.Calcium_mg * ratio
            totals['Iron_mg'] += food.Iron_mg * ratio
            totals['Magnesium_mg'] += food.Magnesium_mg * ratio
            totals['Phosphorus_mg'] += food.Phosphorus_mg * ratio
            totals['Potasssium_mg'] += food.Potasssium_mg * ratio
            totals['Zinc_mg'] += food.Zinc_mg * ratio
            totals['Cupper_mg'] += food.Cupper_mg * ratio
            totals['Manganese_mg'] += food.Manganese_mg * ratio
            totals['Selenium_mcg'] += food.Selenium_mcg * ratio
            totals['Vitamin_A_IU'] += food.Vitamin_A_IU * ratio
            totals['Vitamin_C_mg'] += food.Vitamin_C_mg * ratio
            totals['Vitamin_D_mcg'] += food.Vitamin_D_mcg * ratio
            totals['Vitamin_E_mg'] += food.Vitamin_E_mg * ratio
            totals['Vitamin_B1_mg'] += food.Vitamin_B1_mg * ratio
            totals['Vitamin_B2_mg'] += food.Vitamin_B2_mg * ratio
            totals['Vitamin_B3_mg'] += food.Vitamin_B3_mg * ratio
            totals['Vitamin_B5_mg'] += food.Vitamin_B5_mg * ratio
            totals['Vitamin_B6_mg'] += food.Vitamin_B6_mg * ratio
            totals['Vitamin_B9_mg'] += food.Vitamin_B9_mg * ratio
            totals['Vitamin_B12'] += food.Vitamin_B12 * ratio
            totals['Choline_mg'] += food.Choline_mg * ratio
            totals['Cholesterol_mg'] += food.Cholesterol_mg * ratio
            totals['Water_g'] += food.Water_g * ratio
            totals['Ash_g'] += food.Ash_g * ratio
            totals['Starch_g'] += food.Starch_g * ratio
            totals['Sucrose_g'] += food.Sucrose_g * ratio
            totals['Glucose_g'] += food.Glucose_g * ratio
            totals['Fructose_g'] += food.Fructose_g * ratio
            totals['Lactose_g'] += food.Lactose_g * ratio
            totals['Maltose_g'] += food.Maltose_g * ratio
            totals['Alcohol_g'] += food.Alcohol_g * ratio
            totals['Caffeine_mg'] += food.Caffeine_mg * ratio
            totals['MUFA'] += food.MUFA * ratio
            totals['PUFA'] += food.PUFA * ratio
        
        # Round all values to 2 decimal places
        for key in totals:
            totals[key] = round(totals[key], 2)
        
        return JsonResponse({'nutrients': totals})
        
    except FoodCard.DoesNotExist:
        return JsonResponse({'error': 'Card not found'}, status=404)
    except Exception as e:
        return JsonResponse({'error': str(e)}, status=500)


@api_view(["GET"])
def diet_templates(request):
    # --- Authentication & Permissions ---
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")
    assistant = Assistant.objects.get(django_user=django_user)

    # --- 1. Retrieve Filter Parameters ---
    company_name = request.GET.get('company_name')
    city = request.GET.get('city')
    bacteria_type = request.GET.get('bacteria_type')
    diet_type = request.GET.get('diet_type')
    diseases_type = request.GET.get('diseases_type')
    allergic_type = request.GET.get('allergic_type')
    other_tags = request.GET.get('other_tags')
    is_user_template = request.GET.get('is_user_template')

    # Search parameters (for text inputs)
    name_search = request.GET.get('name_search', '').strip()
    calorie_search = request.GET.get('calorie_search', '').strip()
    lunch_stews = request.GET.get('lunch_Stews')
    dinner_stews = request.GET.get('dinner_Stews')
    lunch_mixed_rice = request.GET.get('lunch_Mixed_Rice')
    dinner_mixed_rice = request.GET.get('dinner_Mixed_Rice')

    # --- 2. Check if any filtering parameter is provided ---
    # We consider a filter "active" if any of the above values is non-empty (or truthy).
    filter_active = any([
        company_name, city, bacteria_type, diet_type, diseases_type, allergic_type, other_tags,
        is_user_template,
        name_search, calorie_search,
        lunch_stews, dinner_stews, lunch_mixed_rice, dinner_mixed_rice
    ])

    # --- 3. Build the QuerySet ---
    if not filter_active:
        # If no filters are provided, return an empty queryset.
        queryset = DietTemplate.objects.none()
    else:
        # Build a dictionary for the "base" filters
        filters = {}
        if company_name:
            filters['company_name__iexact'] = company_name
        if city:
            filters['city__iexact'] = city
        if is_user_template:
            filters['is_user_template'] = True
        if bacteria_type:
            filters[bacteria_type] = True
        if diet_type:
            filters[diet_type] = True
        if diseases_type:
            filters[diseases_type] = True
        if allergic_type:
            filters[allergic_type] = True
        if other_tags:
            filters[other_tags] = True

        queryset = DietTemplate.objects.filter(**filters).select_related('creator')

        # Apply additional search filters:
        if name_search:
            queryset = queryset.filter(
                Q(diet_name__icontains=name_search) |
                Q(creator__first_name__icontains=name_search) |
                Q(creator__last_name__icontains=name_search)
            )
        if calorie_search:
            try:
                calorie_val = int(calorie_search)
                queryset = queryset.filter(calorie_range=calorie_val)
            except ValueError:
                pass  # Ignore if invalid
        if lunch_stews:
            queryset = queryset.filter(lunch_Stews__icontains=lunch_stews)
        if dinner_stews:
            queryset = queryset.filter(dinner_Stews__icontains=dinner_stews)
        if lunch_mixed_rice:
            queryset = queryset.filter(lunch_Mixed_Rice__icontains=lunch_mixed_rice)
        if dinner_mixed_rice:
            queryset = queryset.filter(dinner_Mixed_Rice__icontains=dinner_mixed_rice)

    # --- 4. Sorting ---
    sort_field = request.GET.get('sort_field', 'created_at')
    sort_order = request.GET.get('sort_order', 'desc')
    if sort_order == 'desc':
        sort_field = '-' + sort_field
    queryset = queryset.order_by(sort_field)

    # --- 5. Pagination ---
    paginator = Paginator(queryset, 20)
    page_number = request.GET.get('page')
    page_obj = paginator.get_page(page_number)

    # --- 6. Build the Context and Cache It (only if filters are active) ---
    # We exclude the 'request' object from the cache.
    params = request.GET.dict()
    params_json = json.dumps(params, sort_keys=True)
    cache_key = "diet_templates_" + hashlib.md5(params_json.encode('utf-8')).hexdigest()
    cached_context = cache.get(cache_key)
    if not cached_context:
        zipped_data, total_new_messages, free_time = new_messages(request)
        total_notifications = len(free_time)
        cached_context = {
            'user': assistant,
            'message': zipped_data,
            'total_new_messages': total_new_messages,
            'free_time': free_time,
            'total_notifications': total_notifications,
            'page_obj': page_obj,
            'name_search': name_search,
            'calorie_search': calorie_search,
            'sort_field': request.GET.get('sort_field', ''),
            'sort_order': request.GET.get('sort_order', ''),
            'no_filters': not filter_active  # flag to indicate no filters were applied
        }
        cache.set(cache_key, cached_context, 300)  # Cache for 5 minutes

    # Always add the request object back to the context for template rendering.
    cached_context['request'] = request

    return render(request, 'assistant/diet_templates.html', cached_context)


@api_view(["POST"])
def add_diet_from_table(request, patient_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return myResponse.Error("you have not access", -1)

    temp = AppUser.objects.filter(id=patient_id)
    if not temp.exists():
        return myResponse.Error('invalid request', -2)
    patient = temp[0]
    assistant = Assistant.objects.get(django_user=django_user)
    if patient.profile.therapist != assistant.django_user:
        return myResponse.Error("you have not access to this patient!", -3)

    try:
        diet_json = request.data.get("diet_json")
        diet_id = request.data.get("diet_id")  # Get diet_id from the request data
        if not diet_json:
            return myResponse.Error("No JSON data provided", -4)

        # Validate JSON format
        try:
            diet_data = json.loads(diet_json)
        except json.JSONDecodeError:
            return myResponse.Error("Invalid JSON format", -5)

        if diet_id:
            # If diet_id is provided, update the existing diet
            diet = Diet.objects.get(id=diet_id, user=patient)
            diet.diet_json = diet_data
            diet.creator = django_user
            # diet.adjust_starting_date()
            diet.save()
        else:
            diet = Diet(diet_json=diet_data,
                        user=patient,
                        creator=django_user,
                        )
            diet.adjust_starting_date()
            diet.save()

        diet_start_date = request.data.get("start_date")
        if diet_start_date is not None and str(diet_start_date) != "":
            diet.from_date = date.fromtimestamp(int(diet_start_date))
            diet.save()

        data = {
            'df': diet_data
        }
        # SMS().notify_user(user=patient)
        return myResponse.OK("Diet was successfully registered", data)
    except Exception as e:
        return myResponse.Error("error on saving the diet: {}".format(e), 1001)


@api_view(["POST"])
def save_diet_template(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    django_user = request.user

    try:
        diet_json = request.data.get("diet_json")
        diet_name = request.data.get("diet_name")
        calorie_range = request.data.get("calorie_range")
        company_name = request.data.get("company_name")
        city = request.data.get("city")
        bacteria_types = request.data.get("bacteria_types", [])
        diet_types = request.data.get("diet_types", [])
        diseases = request.data.get("diseases", [])
        allergic_type = request.data.get("allergic", [])

        # Validating required fields
        if not diet_json:
            return JsonResponse({'error': 'No JSON data provided'}, status=400)
        if not diet_name:
            return JsonResponse({'error': 'No diet name provided'}, status=400)
        if calorie_range is None:
            return JsonResponse({'error': 'No calorie range provided'}, status=400)

        # Validate JSON format for diet data
        try:
            diet_data = json.loads(diet_json)
        except json.JSONDecodeError:
            return JsonResponse({'error': 'Invalid JSON format'}, status=400)

        is_user_template = request.data.get("is_user_template", False)
        # Create the DietTemplate instance
        diet_template = DietTemplate(
            creator=django_user,
            diet_json=diet_data,
            diet_name=diet_name,
            calorie_range=calorie_range,
            company_name=company_name,
            city=city,
            is_user_template=is_user_template,
        )

        # Set the boolean fields for the tags (bacteria, diet types, diseases)
        diet_template.akkermansia = 'akkermansia' in bacteria_types
        diet_template.bacteroides = 'bacteroides' in bacteria_types
        diet_template.bifidobacterium = 'bifidobacterium' in bacteria_types
        diet_template.faecalibacterium = 'faecalibacterium' in bacteria_types
        diet_template.alistipes = 'alistipes' in bacteria_types
        diet_template.eubacterium = 'eubacterium' in bacteria_types
        diet_template.lactobacillus = 'lactobacillus' in bacteria_types
        diet_template.blautia = 'blautia' in bacteria_types

        diet_template.Allergens_Dairy = 'dairy' in allergic_type
        diet_template.Allergens_Red_meat = 'red-meat' in allergic_type
        diet_template.Allergens_Eggplant = 'eggplant' in allergic_type
        diet_template.Allergens_Tomato = 'tomato' in allergic_type
        diet_template.Allergens_Eggs = 'egg' in allergic_type
        diet_template.Allergens_Nuts = 'nut' in allergic_type
        diet_template.Allergens_Sea_food = 'sea-food' in allergic_type
        diet_template.Allergens_Fava = 'fava' in allergic_type
        diet_template.Allergens_Milk = 'milk' in allergic_type
        diet_template.Allergens_Coffee = 'coffee' in allergic_type

        diet_template.med = 'med' in diet_types
        diet_template.low_gi = 'low-gi' in diet_types
        diet_template.fasting = 'fasting' in diet_types
        diet_template.keto = 'keto' in diet_types
        diet_template.mind = 'mind' in diet_types
        diet_template.sport = 'sport' in diet_types
        diet_template.liquid_or_pureed_diet = 'liquid' in diet_types
        diet_template.eliminate_diet = 'eliminate' in diet_types
        diet_template.low_fodmap = 'low-fodmap' in diet_types
        diet_template.pregnant = 'pregnant' in diet_types
        diet_template.lactating = 'lactating' in diet_types
        diet_template.Diet_Low_Carb = 'low-carb' in diet_types
        diet_template.Diet_Vegetarian = 'vegetarian' in diet_types

        diet_template.fatty_liver = 'fatty-liver' in diseases
        diet_template.diabetes = 'diabetes' in diseases
        diet_template.ibs = 'ibs' in diseases
        diet_template.pcos = 'pcos' in diseases
        diet_template.neghres = 'neghres' in diseases
        diet_template.ms = 'ms' in diseases
        diet_template.obesity_surgery = 'obesity-surgery' in diseases
        diet_template.renal = 'renal' in diseases
        diet_template.cvd = 'cvd' in diseases
        diet_template.migraine = 'migraine' in diseases
        diet_template.cancer = 'cancer' in diseases
        diet_template.nervous_disease = 'nervous-disease' in diseases
        diet_template.gi_disease = 'gi-disease' in diseases
        diet_template.healthy = 'healthy' in diseases
        diet_template.Disease_Hypothyroidism = 'Hypothyroidism' in diseases
        diet_template.Disease_Celiac = 'celiac' in diseases
        diet_template.Disease_Acid_reflux = 'Reflux' in diseases
        diet_template.Disease_Diarrhea = 'Diarrhea' in diseases
        diet_template.Disease_Constipation = 'Constipation' in diseases
        diet_template.Disease_Cyst = 'Cyst' in diseases

        # Save the template
        diet_template.save()

        data = {
            'template_id': diet_template.id,
            'message': "Diet template saved successfully"
        }
        return JsonResponse({'message': 'Diet template was successfully saved', 'data': data}, status=200)

    except Exception as e:
        return JsonResponse({'error': 'Error on saving the diet template: {}'.format(e)}, status=500)


@api_view(["GET"])
def get_diet_template(request, template_id):
    if not request.user.is_authenticated:
        return JsonResponse({'error': 'User not authenticated'}, status=401)

    django_user = request.user

    try:
        template = DietTemplate.objects.get(id=template_id)
        response_data = {
            'template_id': template.id,
            'created_at': template.created_at.isoformat(),
            'diet_json': template.diet_json,
        }
        return JsonResponse(response_data)
    except DietTemplate.DoesNotExist:
        return JsonResponse({'error': 'Template not found'}, status=404)


def edit_diet_template(request, template_id):
    if request.method == 'POST':
        try:
            template = DietTemplate.objects.get(pk=template_id)
            data = json.loads(request.body)
            template.diet_name = data.get('diet_name', template.diet_name)
            template.save()
            return JsonResponse({'status': 'success'})
        except DietTemplate.DoesNotExist:
            return JsonResponse({'status': 'error', 'message': 'Template not found'}, status=404)
        except Exception as e:
            return JsonResponse({'status': 'error', 'message': str(e)}, status=500)


def delete_diet_template(request, template_id):
    if request.method == 'DELETE':
        try:
            template = DietTemplate.objects.get(pk=template_id)
            template.delete()
            return JsonResponse({'status': 'success'})
        except DietTemplate.DoesNotExist:
            return JsonResponse({'status': 'error', 'message': 'Template not found'}, status=404)
        except Exception as e:
            return JsonResponse({'status': 'error', 'message': str(e)}, status=500)


@api_view(["GET"])
def requested_foods(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    user = request.user
    patients_list = AppUser.objects.all()
    assistant = Assistant.objects.get(django_user=user)
    foods = RequestedFoods.objects.all()

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'user': assistant,
        "message": zipped_data,
        "foods": foods,
        "patients": patients_list,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }

    return render(request, template_name='assistant/foods.html', context=context)


@api_view(["POST"])
def check_food(request, food_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    user = request.user

    food = RequestedFoods.objects.get(id=food_id)
    food.Checked = True
    food.save()

    return redirect('panel-assistant-requested-foods')


@api_view(["POST"])
def delete_food(request, food_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    food = RequestedFoods.objects.get(id=food_id)
    food.delete()

    return redirect('panel-assistant-requested-foods')


@api_view(["GET"])
def challenge(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    user = request.user
    if not user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")
    assistant = Assistant.objects.get(django_user=user)
    temp = AppUser.objects.all()
    if not temp.exists():
        return generate_error_page(request, 'invalid request')
    challenges = Challenge.objects.filter(creator=user)
    users = []
    groups_per_challenge = []
    audience_users_per_challenge = []
    challenge_display_data = []

    for c in challenges:
        participants_data = []
        for user in c.participants.all():
            profile = user  # Assuming profile is a OneToOneField related to AppUser
            participants_data.append(profile)
        users.append(participants_data)

        # Collect groups assigned to this challenge
        challenge_groups = c.audience.all()
        groups_per_challenge.append(challenge_groups)

        # Collect audience users assigned to this challenge
        challenge_audience_users = c.audience_users.all()
        audience_users_per_challenge.append(challenge_audience_users)

        matched_foods = []
        if c.sub_category == 6 and c.food_codes:
            # We'll preserve the order of c.food_codes in the final list
            for code in c.food_codes:
                try:
                    f = Food.objects.get(Data_Base_Number=code)
                    matched_foods.append(f)
                except Food.DoesNotExist:
                    pass

        # Append everything as a single tuple
        challenge_display_data.append(
            (c, participants_data, challenge_groups, challenge_audience_users, matched_foods)
        )

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    # Fetch groups managed by the assistant
    groups = ZiLuckGroup.objects.filter(Q(admin=assistant.django_user) | Q(creator=assistant.django_user))

    # Fetch all users managed by the assistant
    all_users = User.objects.filter(profile__therapist=assistant.django_user)

    # Fetch DjangoUsers in the assistant's groups
    django_users_in_groups = DjangoUser.objects.filter(groups__in=groups).distinct()

    # Map DjangoUsers to your User instances
    users_in_groups = User.objects.filter(django_user__in=django_users_in_groups)

    # Get users not in any of these groups
    users_not_in_groups = all_users.exclude(id__in=users_in_groups)

    context = {
        'user': assistant,
        'challenges': challenge_display_data,
        "participants_data": users,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
        'groups': groups,
        'users_not_in_groups': users_not_in_groups,
    }
    return render(request, template_name='assistant/challenge.html', context=context)


@api_view(["POST"])
def add_challenge(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")

    try:
        name = request.POST.get("name")
        reward = int(request.POST.get("reward"))
        fee = int(request.POST.get("fee")) if request.POST.get("fee") else None
        start_deadline = datetime.strptime(request.POST.get("start_deadline"), '%Y-%m-%dT%H:%M') if request.POST.get(
            "start_deadline") else None
        end_deadline = datetime.strptime(request.POST.get("end_deadline"), '%Y-%m-%dT%H:%M') if request.POST.get(
            "end_deadline") else None
        description = request.POST.get("description")

        # Get audience groups from the request
        audience_group_ids = request.POST.getlist('audience_groups')  # Expecting a list of group IDs
        audience_groups = ZiLuckGroup.objects.filter(id__in=audience_group_ids)

        # Get audience users from the request
        audience_user_ids = request.POST.getlist('audience_users')
        audience_users = User.objects.filter(id__in=audience_user_ids)

        category_mapping = {
            "Food": 0,
            "BS": 1,
            "Exercise": 2,
            "BP": 3,
            "Sleep": 4,
            "Water": 5,
        }

        category_str = request.POST.get("category")
        category = category_mapping.get(category_str, -1)  # Default to -1 if not found

        sub_category_mapping = {
            "مدت زمان خواب": 0, "میکروبیوم": 0, "بیشتر از": 0, "قدم": 0, "تعداد لیوان": 0,
            "ساعت شروع خواب": 1, "چربی": 1, "تعداد ثبت": 1, "سوزاندن کالری": 1,
            "ساعت پایان خواب": 2, "کالری": 2, "مدت زمان": 2,
            "ساعت شروع خواب و مدت زمان خواب": 3, "پروتئین": 3,
            "کربوهیدرات": 4,
            "انتخاب کارت": 5,
            "کدهای غذا": 6,
        }

        sub_category_str = request.POST.get("sub_category")
        sub_category = sub_category_mapping.get(sub_category_str, -1)

        new_challenge = None
        # sub = Challenge.objects.all()
        # sub2 = len(sub)

        if category == 0:  # Food
            if sub_category == 0:  # microbiome
                microbiome = request.POST.get("microbiome")
                new_challenge = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                          start_deadline=start_deadline, end_deadline=end_deadline,
                                          description=description,
                                          category=category, sub_category=sub_category, microbiome=microbiome)
                new_challenge.save()
            elif sub_category == 1:  # fat
                fat = request.POST.get("fat")
                new_challenge = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                          start_deadline=start_deadline, end_deadline=end_deadline,
                                          description=description,
                                          category=category, sub_category=sub_category, fat=fat)
                new_challenge.save()
            elif sub_category == 2:  # calorie
                calorie = request.POST.get("calorie")
                new_challenge = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                          start_deadline=start_deadline, end_deadline=end_deadline,
                                          description=description,
                                          category=category, sub_category=sub_category, calorie=calorie)
                new_challenge.save()
            elif sub_category == 3:  # protein
                protein = request.POST.get("protein")
                new_challenge = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                          start_deadline=start_deadline, end_deadline=end_deadline,
                                          description=description,
                                          category=category, sub_category=sub_category, protein=protein)
                new_challenge.save()
            elif sub_category == 4:  # carbohydrate
                carbohydrate = request.POST.get("carbohydrate")
                new_challenge = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                          start_deadline=start_deadline, end_deadline=end_deadline,
                                          description=description,
                                          category=category, sub_category=sub_category, carbohydrate=carbohydrate)
                new_challenge.save()
            elif sub_category == 6:  # Food Codes
                food_codes_list = request.POST.getlist("food_codes[]")  # list of entered codes
                new_challenge = Challenge(
                    creator=django_user,
                    name=name,
                    reward=reward,
                    fee=fee,
                    start_deadline=start_deadline,
                    end_deadline=end_deadline,
                    description=description,
                    category=category,
                    sub_category=sub_category,
                    food_codes=food_codes_list,  # store the list here
                )
                new_challenge.save()
            else:  # cards
                new_challenge = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                          start_deadline=start_deadline, end_deadline=end_deadline,
                                          description=description,
                                          category=category, sub_category=sub_category)
                new_challenge.save()
        elif category == 1:  # BS
            if sub_category == 0:  # MoreThan-BS
                moreThan = request.POST.get("moreThan")
                new_challenge = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                          start_deadline=start_deadline, end_deadline=end_deadline,
                                          description=description,
                                          category=category, sub_category=sub_category, more_than=moreThan)
                new_challenge.save()
            elif sub_category == 1:  # TedadSabt-BS
                tedadSabt = request.POST.get("tedadSabt")
                new_challenge = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                          start_deadline=start_deadline, end_deadline=end_deadline,
                                          description=description,
                                          category=category, sub_category=sub_category, tedad_sabt=tedadSabt)
                new_challenge.save()
        elif category == 2:  # Exercise
            if sub_category == 0:  # Steps
                steps = request.POST.get("steps")
                new_challenge = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                          start_deadline=start_deadline, end_deadline=end_deadline,
                                          description=description,
                                          category=category, sub_category=sub_category, steps=steps)
                new_challenge.save()
            elif sub_category == 1:  # calorie_burnt
                calorieBurnt = request.POST.get("calorieBurnt")
                new_challenge = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                          start_deadline=start_deadline, end_deadline=end_deadline,
                                          description=description,
                                          category=category, sub_category=sub_category, calorie_burnt=calorieBurnt)
                new_challenge.save()
            elif sub_category == 2:  # exercise_duration
                durationExercise = request.POST.get("durationExercise")
                new_challenge = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                          start_deadline=start_deadline, end_deadline=end_deadline,
                                          description=description,
                                          category=category, sub_category=sub_category,
                                          exercise_duration=durationExercise)
                new_challenge.save()
        elif category == 3:  # BP
            if sub_category == 0:  # MoreThan-BP
                moreThan = request.POST.get("moreThan")
                new_challenge = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                          start_deadline=start_deadline, end_deadline=end_deadline,
                                          description=description,
                                          category=category, sub_category=sub_category, more_than=moreThan)
                new_challenge.save()
            elif sub_category == 1:  # TedadSabt-BP
                tedadSabt = request.POST.get("tedadSabt")
                new_challenge = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                          start_deadline=start_deadline, end_deadline=end_deadline,
                                          description=description,
                                          category=category, sub_category=sub_category, tedad_sabt=tedadSabt)
                new_challenge.save()
        elif category == 4:  # Sleep
            if sub_category == 0:  # Sleep-Duration
                durationField = request.POST.get("duration")
                new_challenge = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                          start_deadline=start_deadline, end_deadline=end_deadline,
                                          description=description,
                                          category=category, sub_category=sub_category, sleep_duration=durationField)
                new_challenge.save()
            elif sub_category == 1:  # sleep_start_hour
                startHourField = request.POST.get("startHour")
                new_challenge = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                          start_deadline=start_deadline, end_deadline=end_deadline,
                                          description=description,
                                          category=category, sub_category=sub_category, sleep_start_hour=startHourField)
                new_challenge.save()
            elif sub_category == 2:  # sleep_end_hour
                endHourField = request.POST.get("endHour")
                new_challenge = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                          start_deadline=start_deadline, end_deadline=end_deadline,
                                          description=description,
                                          category=category, sub_category=sub_category, sleep_end_hour=endHourField)
                new_challenge.save()
            elif sub_category == 3:  # sleep_start_hour & Duration
                durationField = request.POST.get("duration")
                startHourField = request.POST.get("startHour")
                new_challenge = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                          start_deadline=start_deadline, end_deadline=end_deadline,
                                          description=description,
                                          category=category, sub_category=sub_category, sleep_duration=durationField,
                                          sleep_start_hour=startHourField)
                new_challenge.save()
        elif category == 5:  # Water
            water = request.POST.get("tedadSabt")
            new_challenge = Challenge(creator=django_user, name=name, reward=reward, fee=fee,
                                      start_deadline=start_deadline, end_deadline=end_deadline,
                                      description=description,
                                      category=category, sub_category=sub_category, tedad_sabt=water)
            new_challenge.save()

        if new_challenge:
            new_challenge.audience.set(audience_groups)
            new_challenge.audience_users.set(audience_users)
            new_challenge.save()

        return redirect('panel-assistant-challenge')

    except Exception as e:
        return myResponse.Error("error on saving the challenge: {}".format(e), 1001)


def delete_challenge(request, challenge_id):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    challenge = get_object_or_404(Challenge, id=challenge_id)
    challenge.delete()
    return redirect('panel-assistant-challenge')


def food_autocomplete(request):
    # 'term' is what jQuery UI Autocomplete sends by default
    query = request.GET.get('term', '')
    # Filter foods whose code starts with or contains 'query'
    # (you can decide if you want startswith, contains, etc.)
    foods = Food.objects.filter(FA_Name__icontains=query)[:20]

    # Build a simple list of dicts that jQuery UI can read
    results = []
    for food in foods:
        results.append({
            "label": f"{food.FA_Name} (کد: {food.Data_Base_Number})",
            "value": food.Data_Base_Number,  # Once selected, the input will be set to this
            # You can add more fields here if needed
        })

    return JsonResponse(results, safe=False)


def cards(request):
    """
    Renders the cards and food list pages for the assistant panel.

    Handles user authentication, data filtering, and pagination for both
    FoodCard and Food items to ensure a performant and responsive experience.
    """

    # 1. Authentication and Authorization (Initial Checks)
    if not request.user.is_authenticated:
        return redirect('panel-login')

    user = request.user
    if not user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")

    try:
        assistant = user.assistant # Assuming a one-to-one relationship
    except Assistant.DoesNotExist:
        # Handle the case where a user doesn't have an associated Assistant profile
        return generate_error_page(request, "Assistant profile not found.")
    
    # 2. Get and Normalize Query Parameters
    card_search_query = request.GET.get('card_search_query', '')
    food_search_query = request.GET.get('food_search_query', '')
    include_tags = request.GET.getlist('include_tags')
    exclude_tags = request.GET.getlist('exclude_tags')
    min_calories = request.GET.get('min_calories')
    max_calories = request.GET.get('max_calories')
    
    card_page_number = request.GET.get('card_page', 1)
    food_page_number = request.GET.get('food_page', 1)

    # 3. Build Optimized Database Queries (Lazy QuerySets)

    # Start with a base QuerySet for both models
    # This is where the magic happens - filters are applied on the QuerySet
    # but no database queries are executed yet.
    card_list = FoodCard.objects.all()
    food_list = Food.objects.all()

    # Apply all search filters to the QuerySets
    if card_search_query:
        card_list = card_list.filter(
            Q(FA_Name__icontains=card_search_query) | 
            Q(EN_Name__icontains=card_search_query)
        )

    if food_search_query:
        food_list = food_list.filter(
            Q(FA_Name__icontains=food_search_query) | 
            Q(english_Name__icontains=food_search_query)
        )

    # Apply tag filters using dynamic dictionary unpacking
    include_conditions = {tag: True for tag in include_tags if hasattr(FoodCard, tag)}
    exclude_conditions = {tag: True for tag in exclude_tags if hasattr(FoodCard, tag)}
    
    if include_conditions:
        card_list = card_list.filter(**include_conditions)

    if exclude_conditions:
        card_list = card_list.exclude(**exclude_conditions)

    # Apply calorie filters with error handling
    if min_calories:
        try:
            min_calories_float = float(min_calories)
            card_list = card_list.filter(Calories__gte=min_calories_float)
        except (ValueError, TypeError):
            pass

    if max_calories:
        try:
            max_calories_float = float(max_calories)
            card_list = card_list.filter(Calories__lte=max_calories_float)
        except (ValueError, TypeError):
            pass

    # 4. Paginate the Optimized QuerySets
    card_per_page = 20
    food_per_page = 20
    
    card_paginator = Paginator(card_list, card_per_page)
    food_paginator = Paginator(food_list, food_per_page)

    # Now the QuerySets are evaluated and the database is hit
    try:
        card_page = card_paginator.get_page(card_page_number)
        food_page = food_paginator.get_page(food_page_number)
    except (PageNotAnInteger, EmptyPage):
        card_page = card_paginator.get_page(1)
        food_page = food_paginator.get_page(1)

    # 5. Caching for Expensive Function Calls
    cache_key = f"new_messages_{user.id}"
    cached_data = cache.get(cache_key)
    
    if cached_data is None:
        zipped_data, total_new_messages, free_time = new_messages(request)
        cache.set(cache_key, (zipped_data, total_new_messages, free_time), 300) # 5 minutes
    else:
        zipped_data, total_new_messages, free_time = cached_data
    
    total_notifications = len(free_time)

    # 6. Prepare Context for Rendering
    context = {
        'user': assistant,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
        'card_page': card_page,
        'food_page': food_page,
        'card_search_query': card_search_query,
        'include_tags': include_tags,
        'exclude_tags': exclude_tags,
        'min_calories': min_calories,
        'max_calories': max_calories,
        'food_search_query': food_search_query,
    }

    return render(request, template_name='assistant/cards.html', context=context)



def cards_ajax(request):
    # Get parameters from GET request
    search_query = request.GET.get('card_search_query', '')
    include_tags = request.GET.getlist('include_tags')
    exclude_tags = request.GET.getlist('exclude_tags')
    min_calories = request.GET.get('min_calories')
    max_calories = request.GET.get('max_calories')
    sort_field = request.GET.get('sort_field', 'id')  # Default sort field
    sort_order = request.GET.get('sort_order', 'desc')
    card_page_number = request.GET.get('card_page', 1)

    # Create cache key for this search
    cache_key_parts = [
        'cards_search',
        search_query or 'empty',
        str(sorted(include_tags or [])),
        str(sorted(exclude_tags or [])),
        str(min_calories or '0'),
        str(max_calories or '9999'),
        str(sort_field or 'id'),
        str(sort_order or 'desc'),
        str(card_page_number or '1')
    ]
    cache_key = '_'.join(cache_key_parts)
    
    # Try to get cached result first
    try:
        cached_result = cache.get(cache_key)
        if cached_result:
            print(f"[SEARCH_PERFORMANCE] Cache HIT for key: {cache_key}")
            return JsonResponse(cached_result)
        
        print(f"[SEARCH_PERFORMANCE] Cache MISS for key: {cache_key}")
    except Exception as e:
        print(f"[SEARCH_PERFORMANCE] Cache error: {e}")
        # Continue without cache if there's an error

    # Optimize database queries - fetch only necessary fields
    card_list = FoodCard.objects.only(
        'id', 'FA_Name', 'EN_Name', 'Calories', 'created_at', 'edited_at',
        'is_breakfast', 'is_lunch', 'is_dinner', 'is_snack'
    )

    # Apply filters efficiently with optimized search using indexes
    if search_query:
        print(f"[SEARCH_PERFORMANCE] Processing search query: '{search_query}'")
        
        # Use optimized search that leverages database indexes
        # Start with exact matches for better performance
        card_list = card_list.filter(
            Q(FA_Name__icontains=search_query) | 
            Q(EN_Name__icontains=search_query)
        ).distinct()
        
        # Add debug logging for performance monitoring
        result_count = card_list.count()
        print(f"[SEARCH_PERFORMANCE] Search query: '{search_query}' - Found {result_count} results")
        
        # Debug: Check if the queryset is empty before count
        print(f"[SEARCH_PERFORMANCE] Queryset empty: {card_list.exists()}")
        
        # Log some sample results for debugging
        if result_count > 0:
            sample_results = card_list[:3]  # Get first 3 results
            for i, card in enumerate(sample_results):
                print(f"[SEARCH_PERFORMANCE] Sample result {i+1}: FA_Name='{card.FA_Name}', EN_Name='{card.EN_Name}'")
        else:
            print(f"[SEARCH_PERFORMANCE] No results found for query: '{search_query}'")
            
            # Debug: Check what's in the database
            all_cards = FoodCard.objects.all()
            print(f"[SEARCH_PERFORMANCE] Total cards in database: {all_cards.count()}")
            
            # Check for partial matches
            partial_matches = FoodCard.objects.filter(
                Q(FA_Name__icontains=search_query[:3]) | 
                Q(EN_Name__icontains=search_query[:3])
            )
            print(f"[SEARCH_PERFORMANCE] Partial matches (first 3 chars): {partial_matches.count()}")
            
            if partial_matches.count() > 0:
                for i, card in enumerate(partial_matches[:3]):
                    print(f"[SEARCH_PERFORMANCE] Partial match {i+1}: FA_Name='{card.FA_Name}', EN_Name='{card.EN_Name}'")
            
    else:
        print(f"[SEARCH_PERFORMANCE] No search query provided")

    # Build filter conditions dynamically to avoid multiple queries
    filter_conditions = {}
    for tag in include_tags:
        if hasattr(FoodCard, tag):
            filter_conditions[tag] = True

    if filter_conditions:
        card_list = card_list.filter(**filter_conditions)

    exclude_conditions = {}
    for tag in exclude_tags:
        if hasattr(FoodCard, tag):
            exclude_conditions[tag] = True

    if exclude_conditions:
        card_list = card_list.exclude(**exclude_conditions)

    if min_calories:
        try:
            min_calories_float = float(min_calories)
            card_list = card_list.filter(Calories__gte=min_calories_float)
        except (ValueError, TypeError):
            pass

    if max_calories:
        try:
            max_calories_float = float(max_calories)
            card_list = card_list.filter(Calories__lte=max_calories_float)
        except (ValueError, TypeError):
            pass

    # Handle sorting safely
    if sort_order == 'asc':
        sort_field = f'-{sort_field}'

    # Validate sort field to prevent injection
    allowed_sort_fields = ['id', 'FA_Name', 'EN_Name', 'Calories', 'created_at', 'edited_at']
    if sort_field.lstrip('-') not in allowed_sort_fields:
        sort_field = 'id'

    card_list = card_list.order_by(sort_field)

    # Pagination
    card_paginator = Paginator(card_list, 20)
    try:
        card_page = card_paginator.page(card_page_number)
    except PageNotAnInteger:
        card_page = card_paginator.page(1)
    except EmptyPage:
        card_page = card_paginator.page(card_paginator.num_pages)

    # Render partial templates
    print(f"[SEARCH_PERFORMANCE] Rendering template with {card_page.object_list.count()} cards")
    cards_html = render_to_string('assistant/cards_list.html', {'card_page': card_page})
    pagination_html = render_to_string('assistant/cards_pagination.html', {'card_page': card_page})
    print(f"[SEARCH_PERFORMANCE] Template rendering completed. Cards HTML length: {len(cards_html)}")

    # Prepare response
    response_data = {'cards_html': cards_html, 'pagination_html': pagination_html}
    
    # Cache the result for 5 minutes (300 seconds)
    try:
        cache.set(cache_key, response_data, 300)
        print(f"[SEARCH_PERFORMANCE] Cached result for key: {cache_key}")
    except Exception as e:
        print(f"[SEARCH_PERFORMANCE] Cache storage error: {e}")
        # Continue without caching if there's an error
    
    return JsonResponse(response_data)


def foods_ajax(request):
    # Get parameters from GET request
    search_query = request.GET.get('food_search_query', '')
    food_page_number = request.GET.get('food_page', 1)

    # Optimize database queries with only necessary fields
    food_list = Food.objects.all()

    # Apply filters - prioritize Persian names (FA_Name)
    if search_query:
        food_list = food_list.filter(
            Q(FA_Name__icontains=search_query) | Q(english_Name__icontains=search_query)
        )

    # Pagination
    food_paginator = Paginator(food_list, 20)
    try:
        food_page = food_paginator.page(food_page_number)
    except PageNotAnInteger:
        food_page = food_paginator.page(1)
    except EmptyPage:
        food_page = food_paginator.page(food_paginator.num_pages)

    # Render partial templates
    foods_html = render_to_string('assistant/foods_list.html', {'food_page': food_page})
    pagination_html = render_to_string('assistant/foods_pagination.html', {'food_page': food_page})

    return JsonResponse({'foods_html': foods_html, 'pagination_html': pagination_html})


@api_view(["POST"])
def save_card(request):
    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")
    # try:
        # Access the parsed data from request.data
    data = request.data
    card_data = data.get('card_data')
    patient_id = data.get('patient_id')  # If needed

    if not card_data:
        return Response({'error': 'No card data provided'}, status=400)
    
    print("This is the card_data: ", card_data);

    # Extract card data
    card_name = card_data.get('FA_Name')
    carbRatio = card_data.get('carbRatio')
    fatRatio = card_data.get('fatRatio')
    proteinRatio = card_data.get('proteinRatio')
    calories = card_data.get('Calories')
    foods_data = card_data.get('foods')
    main_dish_code = card_data.get('main_dish_code', '')  # Extract main_dish_code

    # Extract flags
    flags = {key: card_data.get(key) for key in [
        'is_keto', 'is_med', 'is_conditional_med', 'is_lowcarb',
        'is_mediumcarb', 'is_highcarb', 'is_highprotein', 'is_mediumprotein', 'is_lowprotein', 'is_mind',
        'is_low_fodmap', 'is_lowfat', 'is_mediumfat', 'is_highfat', 'is_breakfast', 'is_lunch',
        'is_dinner', 'is_snack'
    ] if key in card_data}

    # Extract allergen fields - all should be True by default, then set found ones to False
    allergen_fields = {key: card_data.get(key) for key in all_alergic_tags if key in card_data}

    # Use transaction to ensure atomicity
    with transaction.atomic():
        # Always create a new FoodCard instance
        current_card = FoodCard(FA_Name=card_name)

        # Set flags on the FoodCard
        for key, value in flags.items():
            setattr(current_card, key, value)

        # Initialize the foods list
        current_card.foods = []

        # Initialize total calories
        total_calories = 0

        # Process and save food items
        for food_item_data in foods_data:
            food_code = food_item_data.get('food_code')

            # Get existing Food item by Data_Base_Number
            try:
                food = Food.objects.get(Data_Base_Number=food_code)
            except Food.DoesNotExist:
                return Response({'error': f'Food with Data_Base_Number {food_code} does not exist'}, status=400)

            # Calculate per 100g values if needed
            amount = food_item_data.get('amount') or 0
            protein = food_item_data.get('Protein') or 0
            fat = food_item_data.get('Fat') or 0
            carbohydrates = food_item_data.get('Carbohydrates') or 0
            energy = food_item_data.get('Energy') or 0
            
            # Validate nutrient values
            if not all(isinstance(x, (int, float)) for x in [protein, fat, carbohydrates, energy]):
                print(f"Warning: Invalid nutrient values for food {food_code}")
                protein = fat = carbohydrates = energy = 0

            if amount > 0:
                protein_g = (protein * 100) / amount
                fat_g = (fat * 100) / amount
                carbs_g = (carbohydrates * 100) / amount
                calories_g = (energy * 100) / amount
            else:
                protein_g = fat_g = carbs_g = calories_g = 0

            # Update the food item data with calculated values
            food_item_data.update({
                'Protein_g': round(protein_g, 2),
                'Fat_g': round(fat_g, 2),
                'Carbohydrates_g': round(carbs_g, 2),
                'Energy': round(energy, 2)  # Use Energy instead of Calories
            })

            # Append food item to the current card's foods list
            current_card.foods.append(food_item_data)

            # Update total calories
            total_calories += energy

        # Update the Calories field of the FoodCard
        current_card.Calories = round(total_calories, 2)
        
        # Ensure ratios are provided with fallback values
        if carbRatio is None or carbRatio == '':
            carbRatio = 0.4
        if fatRatio is None or fatRatio == '':
            fatRatio = 0.4
        if proteinRatio is None or proteinRatio == '':
            proteinRatio = 0.2
            
        current_card.carb_ratio = float(carbRatio)
        current_card.fat_ratio = float(fatRatio)
        current_card.protein_ratio = float(proteinRatio)
        current_card.creator = django_user
        
        # Set main_dish_code if provided
        if main_dish_code:
            try:
                current_card.main_dish_code = int(main_dish_code)
            except (ValueError, TypeError):
                current_card.main_dish_code = -1  # Default value if conversion fails
        else:
            current_card.main_dish_code = -1  # Default value if not provided

        # Handle allergen fields - Set all to True by default, then set found ones to False
        for allergen_tag in all_alergic_tags:
            setattr(current_card, allergen_tag, True)  # Default to True (unsafe)
        
        # Now set the specific allergens found in this card to False (safe)
        for allergen_tag, value in allergen_fields.items():
            if value is False:  # Only set to False if explicitly set to False
                setattr(current_card, allergen_tag, False)
                print(f"Setting {allergen_tag} to False (safe for this card)")

        # Save current card state
        current_card.save()

    # Return a success response with the new card data
    return Response({
        'status': 'success',
        'message': 'Card saved successfully',
        'card': {
            'id': current_card.id,
            'FA_Name': current_card.FA_Name,
            'Calories': current_card.Calories,
            'foods': current_card.foods,
            'main_dish_code': current_card.main_dish_code,
            **flags,  # Include the flags in the response
            **allergen_fields  # Include the allergen fields in the response
        }
    })

    # except Exception as e:
    #     return Response({'error': str(e)}, status=500)


@api_view(["POST"])
def delete_card(request):
    if request.method == 'POST':
        card_id = request.POST.get('card_id')
        try:
            card = FoodCard.objects.get(id=card_id)
            card.delete()
            return JsonResponse({'status': 'success'})
        except FoodCard.DoesNotExist:
            return JsonResponse({'status': 'error', 'message': 'Card not found'}, status=404)
    else:
        return JsonResponse({'status': 'error', 'message': 'Invalid request'}, status=400)


@api_view(["POST"])
def delete_multiple_cards(request):
    if request.method == 'POST':
        card_ids = request.POST.getlist('card_ids')
        try:
            FoodCard.objects.filter(id__in=card_ids).delete()
            return JsonResponse({'status': 'success'})
        except Exception as e:
            return JsonResponse({'status': 'error', 'message': str(e)}, status=500)
    else:
        return JsonResponse({'status': 'error', 'message': 'Invalid request'}, status=400)


def get_card_data(request, card_id):
    try:
        card = FoodCard.objects.get(id=card_id)
        food_items = card.foods  # Assuming this is a list of dictionaries

        food_codes = [item.get('food_code') for item in food_items]
        foods = Food.objects.filter(Data_Base_Number__in=food_codes)
        food_map = {str(food.Data_Base_Number): food for food in foods}

        foods_data = []
        for food_item in food_items:
            food_code = str(food_item.get('food_code'))
            home_unit_name = food_item.get('home_unit_name')
            home_unit_amount = food_item.get('home_unit_amount')
            amount = food_item.get('amount')
            Protein = food_item.get('Protein')
            Fat = food_item.get('Fat')
            Carbohydrates = food_item.get('Carbohydrates')
            Energy = food_item.get('Energy')

            food = food_map.get(food_code)

            if food:
                # Extract home_units and weight_units from the Food model
                home_units = food.Weight_of_home_units.split('&')[1:-1] if food.Weight_of_home_units else []
                weight_units = food.Name_of_home_units.split('&')[1:-1] if food.Index_of_home_units else []
                
                # Create food data with all necessary information including AL_List
                food_data = {
                    'food_code': food_code,
                    'food_name': food.FA_Name,
                    'home_unit_name': home_unit_name,
                    'home_unit_amount': home_unit_amount,
                    'amount': amount,
                    'Protein_g': food.Protein_g,
                    'Fat_g': food.Fat_g,
                    'Carbohydrates_g': food.Carbohydrates_g,
                    'Calories': food.Calories,
                    'Protein': Protein,
                    'Fat': Fat,
                    'Carbohydrates': Carbohydrates,
                    'Energy': Energy,
                    'home_units': weight_units,
                    'weight_units': home_units,
                    'AL_List': food.AL_List  # Include allergy information
                }
                
                foods_data.append(food_data)

            else:
                # Food not found, provide default values
                foods_data.append({
                    'food_code': food_code,
                    'food_name': food_item.get('food_name', 'Unknown'),
                    'home_unit_name': home_unit_name,
                    'home_unit_amount': home_unit_amount,
                    'amount': amount,
                    'Protein_g': 0,
                    'Fat_g': 0,
                    'Carbohydrates_g': 0,
                    'Calories': 0,
                    'Protein': 0,
                    'Fat': 0,
                    'Carbohydrates': 0,
                    'Energy': 0,
                    'home_units': [],
                    'weight_units': [],
                })

        # Build the response data
        card_data = {
            'id': card.id,
            'FA_Name': card.FA_Name,
            'EN_Name': card.EN_Name,
            'Calories': card.Calories,
            'carb_ratio': card.carb_ratio,
            'fat_ratio': card.fat_ratio,
            'protein_ratio': card.protein_ratio,
            'recipe_description': card.recipe_description,
            'recipe_image_link': card.recipe_image_link,
            'created_at': card.created_at.strftime('%Y-%m-%d %H:%M:%S') if card.created_at else None,
            'edited_at': card.edited_at.strftime('%Y-%m-%d %H:%M:%S') if card.edited_at else None,
            'foods': foods_data,
            # Include all tag fields
            'is_keto': card.is_keto,
            'is_med': card.is_med,
            'is_conditional_med': card.is_conditional_med,
            'is_lowcarb': card.is_lowcarb,
            'is_mediumcarb': card.is_mediumcarb,
            'is_highcarb': card.is_highcarb,
            'is_highprotein': card.is_highprotein,
            'is_lowprotein': card.is_lowprotein,
            'is_mediumprotein': card.is_mediumprotein,
            'is_mind': card.is_mind,
            'is_lowfat': card.is_lowfat,
            'is_mediumfat': card.is_mediumfat,
            'is_highfat': card.is_highfat,
            'is_low_fodmap': card.is_low_fodmap,
            'is_breakfast': card.is_breakfast,
            'is_lunch': card.is_lunch,
            'is_dinner': card.is_dinner,
            'is_snack': card.is_snack,
            'Dessert': card.Dessert,
            'Added_Meat': card.Added_Meat,
            'Ing_Main': card.Ing_Main,
            'Beverage': card.Beverage,
            'Sandwich': card.Sandwich,
            'Condiment': card.Condiment,
            'Lunch': card.Lunch,
            'Snack': card.Snack,
            'Bread': card.Bread,
            'Entree': card.Entree,
            'Soup': card.Soup,
            'Side_Dish': card.Side_Dish,
            'Appetizer': card.Appetizer,
            'Dinner': card.Dinner,
            'Pasta': card.Pasta,
            'Breakfast': card.Breakfast,
            'Salad': card.Salad,
            'Ingredient': card.Ingredient,
            'Bacteria_Acidaminococcus': card.Bacteria_Acidaminococcus,
            'Bacteria_Bacteroidetes': card.Bacteria_Bacteroidetes,
            'Bacteria_Alistipes': card.Bacteria_Alistipes,
            'Bacteria_Bifidobacterium': card.Bacteria_Bifidobacterium,
            'Bacteria_Akkermansia': card.Bacteria_Akkermansia,
            'Bacteria_Blautia': card.Bacteria_Blautia,
            'Bacteria_Fecalibacterium': card.Bacteria_Fecalibacterium,
            'Bacteria_Lawsonibacter_asaccharolyticus': card.Bacteria_Lawsonibacter_asaccharolyticus,
            'Bacteria_Lactobacillus': card.Bacteria_Lactobacillus,
            'Vegan': card.Vegan,
            'Umami': card.Umami,
            'Slow_cooked': card.Slow_cooked,
            'Spicy': card.Spicy,
            'Baked': card.Baked,
            'Pickled': card.Pickled,
            'Low_calorie': card.Low_calorie,
            'Grilled': card.Grilled,
            'Roasted': card.Roasted,
            'High_protein': card.High_protein,
            'Fried': card.Fried,
            'Sweet': card.Sweet,
            'Salty': card.Salty,
            'Lactose_free': card.Lactose_free,
            'Low_carb': card.Low_carb,
            'Cold': card.Cold,
            'Vegetarian': card.Vegetarian,
            'Sour': card.Sour,
            'Sweet_Sour': card.Sweet_Sour,
            'Raw': card.Raw,
            'Gluten_free': card.Gluten_free,
            'Boiled': card.Boiled,
            'Steamed': card.Steamed,
            'Low_fat': card.Low_fat,
            'Prepared_Meals': card.Prepared_Meals,
            'Meats': card.Meats,
            'Sweets': card.Sweets,
            'Beans_and_Lentils': card.Beans_and_Lentils,
            'Snacks': card.Snacks,
            'Grains_and_Pasta': card.Grains_and_Pasta,
            'Nuts_and_Seeds': card.Nuts_and_Seeds,
            'Fruits': card.Fruits,
            'Beverages': card.Beverages,
            'Fish': card.Fish,
            'Baked_Foods': card.Baked_Foods,
            'Breakfast_Cereals': card.Breakfast_Cereals,
            'Vegetables': card.Vegetables,
            'Spices_and_Herbs': card.Spices_and_Herbs,
            'Soups_and_Sauces': card.Soups_and_Sauces,
            'Dairy_and_Egg_Products': card.Dairy_and_Egg_Products,
            'Fats_and_Oils': card.Fats_and_Oils,
            'Lactose_Intolerance': card.Lactose_Intolerance,
            'stones': card.stones,
            'reflux': card.reflux,
            'Spices': card.Spices,
            'Seafood': card.Seafood,
            'Poultry': card.Poultry,
            'Herbs': card.Herbs,
            'Processed_Meats': card.Processed_Meats,
            'Legumes': card.Legumes,
            'Dairy': card.Dairy,
            'Nuts': card.Nuts,
            'Mixed_Dishes': card.Mixed_Dishes,
            'Sauces': card.Sauces,
            'Oils': card.Oils,
            'Grains': card.Grains,
            'Seeds': card.Seeds,
            'Pastries': card.Pastries,
            'Spreads': card.Spreads,
            'Eggs': card.Eggs,
            'Prebiotic': card.Prebiotic,
            'Probiotic': card.Probiotic,
            'Low_Inflammatory': card.Low_Inflammatory,
            'Diverse': card.Diverse,
            'Polyphenol_Rich': card.Polyphenol_Rich,
            'Stews': card.Stews,
            'Mixed_Rice': card.Mixed_Rice,
            'Soups_and_Aashes': card.Soups_and_Aashes,
            'Traditional_Foods': card.Traditional_Foods,
            'International_Foods': card.International_Foods,
            'Vegetarian_Foods': card.Vegetarian_Foods,
            'Simple_Dishes': card.Simple_Dishes,
            'Main_Salads': card.Main_Salads,
            'Fast_Foods': card.Fast_Foods,
            'FODMAP_Classification': card.FODMAP_Classification,
            'AL_Dairy_Butter': card.AL_Dairy_Butter,
            'AL_Dairy_Cheese': card.AL_Dairy_Cheese,
            'AL_Dairy_Coconut': card.AL_Dairy_Coconut,
            'AL_Dairy_CoconutMilk': card.AL_Dairy_CoconutMilk,
            'AL_Dairy_Cream': card.AL_Dairy_Cream,
            'AL_Dairy_CreamCheese': card.AL_Dairy_CreamCheese,
            'AL_Dairy_IceCream': card.AL_Dairy_IceCream,
            'AL_Dairy_Mayonnaise': card.AL_Dairy_Mayonnaise,
            'AL_Dairy_Milk': card.AL_Dairy_Milk,
            'AL_Dairy_Milk_Chocolate': card.AL_Dairy_Milk_Chocolate,
            'AL_Dairy_Whey': card.AL_Dairy_Whey,
            'AL_Dairy_Yogurt': card.AL_Dairy_Yogurt,
            'AL_Egg': card.AL_Egg,
            'AL_Eggs_Egg': card.AL_Eggs_Egg,
            'AL_Eggs_Mayonnaise': card.AL_Eggs_Mayonnaise,
            'AL_Fish_Crab': card.AL_Fish_Crab,
            'AL_Fish_Fish': card.AL_Fish_Fish,
            'AL_Fish_Shrimp': card.AL_Fish_Shrimp,
            'AL_Fish_Tuna': card.AL_Fish_Tuna,
            'AL_Grain_Barley': card.AL_Grain_Barley,
            'AL_Grain_Bean': card.AL_Grain_Bean,
            'AL_Grain_Buckwheat': card.AL_Grain_Buckwheat,
            'AL_Grain_Chickpea': card.AL_Grain_Chickpea,
            'AL_Grain_Corn': card.AL_Grain_Corn,
            'AL_Grain_Lentil': card.AL_Grain_Lentil,
            'AL_Grain_Mung': card.AL_Grain_Mung,
            'AL_Grain_Oat': card.AL_Grain_Oat,
            'AL_Grain_Oats': card.AL_Grain_Oats,
            'AL_Grain_Quinoa': card.AL_Grain_Quinoa,
            'AL_Grain_Rice': card.AL_Grain_Rice,
            'AL_Grain_Rye': card.AL_Grain_Rye,
            'AL_Grain_Soy': card.AL_Grain_Soy,
            'AL_Grain_Unspecified': card.AL_Grain_Unspecified,
            'AL_Grain_Wheat': card.AL_Grain_Wheat,
            'AL_Grains_Wheat': card.AL_Grains_Wheat,
            'AL_Legume_Beans': card.AL_Legume_Beans,
            'AL_Legume_Lentil': card.AL_Legume_Lentil,
            'AL_Legume_Pea': card.AL_Legume_Pea,
            'AL_Legume_RedBean': card.AL_Legume_RedBean,
            'AL_Legume_Soy': card.AL_Legume_Soy,
            'AL_Legume_Split_Peas': card.AL_Legume_Split_Peas,
            'AL_Legume_SplitPea': card.AL_Legume_SplitPea,
            'AL_Meat_Bacon': card.AL_Meat_Bacon,
            'AL_Meat_Beef': card.AL_Meat_Beef,
            'AL_Meat_Chicken': card.AL_Meat_Chicken,
            'AL_Meat_Ham': card.AL_Meat_Ham,
            'AL_Meat_Lamb': card.AL_Meat_Lamb,
            'AL_Meat_Pepperoni': card.AL_Meat_Pepperoni,
            'AL_Meat_Pork': card.AL_Meat_Pork,
            'AL_Meat_Processed': card.AL_Meat_Processed,
            'AL_Meat_Sausage': card.AL_Meat_Sausage,
            'AL_Meat_Turkey': card.AL_Meat_Turkey,
            'AL_Meat_Unspecified': card.AL_Meat_Unspecified,
            'AL_Meat_Varies': card.AL_Meat_Varies,
            'AL_Meat_Various': card.AL_Meat_Various,
            'AL_Nuts_Almond': card.AL_Nuts_Almond,
            'AL_Nuts_Coconut': card.AL_Nuts_Coconut,
            'AL_Nuts_Hazelnut': card.AL_Nuts_Hazelnut,
            'AL_Nuts_Nuts': card.AL_Nuts_Nuts,
            'AL_Nuts_Peanut': card.AL_Nuts_Peanut,
            'AL_Nuts_Pine_Nuts': card.AL_Nuts_Pine_Nuts,
            'AL_Nuts_Pistachio': card.AL_Nuts_Pistachio,
            'AL_Nuts_Sesame': card.AL_Nuts_Sesame,
            'AL_Nuts_Unspecified': card.AL_Nuts_Unspecified,
            'AL_Nuts_Varies': card.AL_Nuts_Varies,
            'AL_Nuts_Various': card.AL_Nuts_Various,
            'AL_Nuts_Walnut': card.AL_Nuts_Walnut,
            'AL_Poultry_Chicken': card.AL_Poultry_Chicken,
            'AL_Seeds_Chia': card.AL_Seeds_Chia,
            'AL_Shellfish_Shrimp': card.AL_Shellfish_Shrimp,
            'AL_Dairy_Cream_Cheese': card.AL_Dairy_Cream_Cheese,
            'AL_Dairy_Creamer': card.AL_Dairy_Creamer,
            'AL_Dairy_Goat_Milk': card.AL_Dairy_Goat_Milk,
            'AL_Dairy_Kashk': card.AL_Dairy_Kashk,
            'AL_Dairy_Kefir': card.AL_Dairy_Kefir,
            'AL_Eggs_Whole': card.AL_Eggs_Whole,
            'AL_Eggs_Whole_Eggs': card.AL_Eggs_Whole_Eggs,
            'AL_Eggs_Yolk': card.AL_Eggs_Yolk,
            'AL_Fish_Anchovy': card.AL_Fish_Anchovy,
            'AL_Fish_Carp': card.AL_Fish_Carp,
            'AL_Fish_Catfish': card.AL_Fish_Catfish,
            'AL_Fish_Caviar': card.AL_Fish_Caviar,
            'AL_Fish_Croaker': card.AL_Fish_Croaker,
            'AL_Fish_Flatfish': card.AL_Fish_Flatfish,
            'AL_Fish_Flounder': card.AL_Fish_Flounder,
            'AL_Fish_Grouper': card.AL_Fish_Grouper,
            'AL_Fish_Herring': card.AL_Fish_Herring,
            'AL_Fish_KingMackerel': card.AL_Fish_KingMackerel,
            'AL_Fish_Mackerel': card.AL_Fish_Mackerel,
            'AL_Fish_Mullet': card.AL_Fish_Mullet,
            'AL_Fish_Pike': card.AL_Fish_Pike,
            'AL_Fish_Porgy': card.AL_Fish_Porgy,
            'AL_Fish_Roe': card.AL_Fish_Roe,
            'AL_Fish_Salmon': card.AL_Fish_Salmon,
            'AL_Fish_Sardine': card.AL_Fish_Sardine,
            'AL_Fish_Shark': card.AL_Fish_Shark,
            'AL_Fish_Snapper': card.AL_Fish_Snapper,
            'AL_Fish_Swordfish': card.AL_Fish_Swordfish,
            'AL_Fish_Tilapia': card.AL_Fish_Tilapia,
            'AL_Fish_Trout': card.AL_Fish_Trout,
            'AL_Fish_Unspecified': card.AL_Fish_Unspecified,
            'AL_Fish_Whitefish': card.AL_Fish_Whitefish,
            'AL_Grain_Bulgur': card.AL_Grain_Bulgur,
            'AL_Grain_Cereal': card.AL_Grain_Cereal,
            'AL_Grain_Millet': card.AL_Grain_Millet,
            'AL_Grain_Pea': card.AL_Grain_Pea,
            'AL_Grain_SplitPea': card.AL_Grain_SplitPea,
            'AL_Grains_Barley': card.AL_Grains_Barley,
            'AL_Grains_Brown_Rice': card.AL_Grains_Brown_Rice,
            'AL_Grains_Bulgur': card.AL_Grains_Bulgur,
            'AL_Grains_Chickpea': card.AL_Grains_Chickpea,
            'AL_Grains_Corn': card.AL_Grains_Corn,
            'AL_Grains_Millet': card.AL_Grains_Millet,
            'AL_Grains_Oats': card.AL_Grains_Oats,
            'AL_Grains_Rice': card.AL_Grains_Rice,
            'AL_Legume_Bean': card.AL_Legume_Bean,
            'AL_Legume_Chickpea': card.AL_Legume_Chickpea,
            'AL_Legume_Fava': card.AL_Legume_Fava,
            'AL_Legume_FavaBean': card.AL_Legume_FavaBean,
            'AL_Legume_Peas': card.AL_Legume_Peas,
            'AL_Legume_Unspecified': card.AL_Legume_Unspecified,
            'AL_Meat_Camel': card.AL_Meat_Camel,
            'AL_Meat_Duck': card.AL_Meat_Duck,
            'AL_Meat_Fat': card.AL_Meat_Fat,
            'AL_Meat_Fish': card.AL_Meat_Fish,
            'AL_Meat_Goose': card.AL_Meat_Goose,
            'AL_Meat_Ground': card.AL_Meat_Ground,
            'AL_Meat_Kidney': card.AL_Meat_Kidney,
            'AL_Meat_Offal': card.AL_Meat_Offal,
            'AL_Meat_Organ': card.AL_Meat_Organ,
            'AL_Meat_Ostrich': card.AL_Meat_Ostrich,
            'AL_Meat_Pigeon': card.AL_Meat_Pigeon,
            'AL_Meat_Quail': card.AL_Meat_Quail,
            'AL_Meat_Sheep': card.AL_Meat_Sheep,
            'AL_Meat_Veal': card.AL_Meat_Veal,
            'AL_Nuts_Acorn': card.AL_Nuts_Acorn,
            'AL_Nuts_Almonds': card.AL_Nuts_Almonds,
            'AL_Nuts_Cashew': card.AL_Nuts_Cashew,
            'AL_Nuts_Chestnut': card.AL_Nuts_Chestnut,
            'AL_Nuts_Hazelnuts': card.AL_Nuts_Hazelnuts,
            'AL_Nuts_Hemp': card.AL_Nuts_Hemp,
            'AL_Nuts_Mixed': card.AL_Nuts_Mixed,
            'AL_Nuts_Nutmeg': card.AL_Nuts_Nutmeg,
            'AL_Nuts_Peanuts': card.AL_Nuts_Peanuts,
            'AL_Nuts_Pecan': card.AL_Nuts_Pecan,
            'AL_Other_Sesame': card.AL_Other_Sesame,
            'AL_Poultry_Duck': card.AL_Poultry_Duck,
            'AL_Poultry_Offal': card.AL_Poultry_Offal,
            'AL_Seafood_Fish': card.AL_Seafood_Fish,
            'AL_Seeds_Cottonseed': card.AL_Seeds_Cottonseed,
            'AL_Seeds_Flax': card.AL_Seeds_Flax,
            'AL_Seeds_Flaxseed': card.AL_Seeds_Flaxseed,
            'AL_Seeds_Poppy': card.AL_Seeds_Poppy,
            'AL_Seeds_Pumpkin': card.AL_Seeds_Pumpkin,
            'AL_Seeds_Safflower': card.AL_Seeds_Safflower,
            'AL_Seeds_Sesame': card.AL_Seeds_Sesame,
            'AL_Seeds_Sunflower': card.AL_Seeds_Sunflower,
            'AL_Seeds_Watermelon': card.AL_Seeds_Watermelon,
            'AL_Sesame_Sesame': card.AL_Sesame_Sesame,
            'AL_Soy_Soy': card.AL_Soy_Soy,
            'AL_Soy_SoySauce': card.AL_Soy_SoySauce,
            'AL_Soy_Soybean': card.AL_Soy_Soybean,
            'AL_Soy_Soybeans': card.AL_Soy_Soybeans,
            'Diet_Low_Sugar': card.Diet_Low_Sugar,
            'Diet_Mediterranean': card.Diet_Mediterranean,
            'Diet_High_Protein': card.Diet_High_Protein,
            'Whole_Foods': card.Whole_Foods,
            'Diet_Low_Fat': card.Diet_Low_Fat,
            'Diet_Low_Calorie': card.Diet_Low_Calorie,
            'Diet_Anti_Inflammatory': card.Diet_Anti_Inflammatory,
            'Diet_Paleo': card.Diet_Paleo,
            'Diet_Diabetic': card.Diet_Diabetic,
            'Diet_Low_Carb': card.Diet_Low_Carb,
            'Diet_Bodybuilding': card.Diet_Bodybuilding,
            'Diet_Vegan': card.Diet_Vegan,
            'Diet_Vegetarian': card.Diet_Vegetarian,
            'Diet_Plant_Based': card.Diet_Plant_Based,
            'Diet_Gluten_Free': card.Diet_Gluten_Free,
            'Diet_Keto': card.Diet_Keto,
            'bread_rice_tag': card.bread_rice_tag,
            'cooked_rice': card.cooked_rice,
            'main_dish': card.main_dish,
            'GL_category': card.GL_category,
            'GL_value': card.GL_value,
            'GI_category': card.GI_category,
            'GI_value': card.GI_value,
            'Disease_Kidney': card.Disease_Kidney,
            'Disease_Kidney_stones': card.Disease_Kidney_stones,
            'Disease_Acid_reflux': card.Disease_Acid_reflux,
            'Disease_Gout': card.Disease_Gout,
            'Disease_Celiac': card.Disease_Celiac,
            'Disease_IBS': card.Disease_IBS,
            'Disease_Diverticulitis': card.Disease_Diverticulitis,
            'Disease_Hypothyroidism': card.Disease_Hypothyroidism,
            'Disease_Gastroesophageal_Reflux': card.Disease_Gastroesophageal_Reflux,
            'Disease_Foodborne_Illness': card.Disease_Foodborne_Illness,
            'Disease_High_blood_pressure': card.Disease_High_blood_pressure,
            'Disease_High_cholesterol': card.Disease_High_cholesterol,
            'Disease_Diabetes': card.Disease_Diabetes,
            'Disease_Hypercholesterolemia': card.Disease_Hypercholesterolemia,
            'Disease_G6PD_Deficiency': card.Disease_G6PD_Deficiency,
            'Disease_Constipation': card.Disease_Constipation,
            'Disease_Gallbladder': card.Disease_Gallbladder,
            'Disease_Latex_Fruit_Syndrome': card.Disease_Latex_Fruit_Syndrome,
            'Disease_Obesity': card.Disease_Obesity,
            'Disease_CVD': card.Disease_CVD,
            'Disease_IBD': card.Disease_IBD,
            'Disease_Lactose_Intolerance': card.Disease_Lactose_Intolerance,
            'Disease_NAFLD': card.Disease_NAFLD,
            'Disease_Anemia': card.Disease_Anemia,
            'Disease_Osteoporosis': card.Disease_Osteoporosis,
            'Disease_Arthritis': card.Disease_Arthritis,
            'Disease_Autoimmune': card.Disease_Autoimmune,
            'Disease_CancerPrevention': card.Disease_CancerPrevention,
            'Disease_MentalHealth': card.Disease_MentalHealth,
            'Allergens_Eggplant': card.Allergens_Eggplant,
            'Allergens_Mushroom': card.Allergens_Mushroom,
            'Allergens_Dried_fruits': card.Allergens_Dried_fruits
        }

        return JsonResponse({'card': card_data})

    except FoodCard.DoesNotExist:
        return JsonResponse({'error': 'Card not found'}, status=404)


def get_food_allergies(request, food_code):
    """
    API endpoint to get allergies for a specific food item
    """
    try:
        # Get the food item from database
        food = Food.objects.get(Data_Base_Number=food_code)
        
        # Extract allergies from AL_List field
        al_list = food.AL_List
        allergies = []
        
        if al_list and al_list.strip():
            # Split by comma and clean up each allergy
            allergies = [allergy.strip() for allergy in al_list.split(',') if allergy.strip()]
        
        return JsonResponse({
            'success': True,
            'food_code': food_code,
            'food_name': food.FA_Name,
            'allergies': allergies
        })
        
    except Food.DoesNotExist:
        return JsonResponse({
            'success': False,
            'error': 'Food not found'
        }, status=404)
    except Exception as e:
        return JsonResponse({
            'success': False,
            'error': str(e)
        }, status=500)


def update_card(request, card_id):
    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")
    if request.method == 'POST':
        try:
            data = json.loads(request.body)
            card_data = data.get('card_data', {})
            # Fetch the card from the database
            card = FoodCard.objects.get(id=card_id)
            # Update card fields
            card.FA_Name = card_data.get('FA_Name', card.FA_Name)
            card.Calories = card_data.get('Calories', card.Calories)
            card.carb_ratio = card_data.get('carbRatio', card.carb_ratio)
            card.fat_ratio = card_data.get('fatRatio', card.fat_ratio)
            card.protein_ratio = card_data.get('proteinRatio', card.protein_ratio)
            
            # Update main_dish_code
            main_dish_code = card_data.get('main_dish_code', '')
            if main_dish_code:
                try:
                    card.main_dish_code = int(main_dish_code)
                except (ValueError, TypeError):
                    card.main_dish_code = -1  # Default value if conversion fails
            else:
                card.main_dish_code = -1  # Default value if not provided
            card.FODMAP_Classification = card_data.get('FODMAP_Classification', card.FODMAP_Classification)
            card.bread_rice_tag = card_data.get('bread_rice_tag', card.bread_rice_tag)
            card.GL_category = card_data.get('GL_category', card.GL_category)
            card.GL_value = card_data.get('GL_value', card.GL_value)
            card.GI_category = card_data.get('GI_category', card.GI_category)
            card.GI_value = card_data.get('GI_value', card.GI_value)
            # Update tags
            tag_fields = [
                'is_keto', 'is_med', 'is_conditional_med', 'is_lowcarb',
                'is_mediumcarb', 'is_highcarb', 'is_highprotein', 'is_mediumprotein', 'is_lowprotein', 'is_mind',
                'is_low_fodmap', 'is_lowfat', 'is_mediumfat', 'is_highfat', 'is_breakfast', 'is_lunch',
                'is_dinner', 'is_snack', 'Dessert', 'Added_Meat', 'Ing_Main', 'Beverage', 'Sandwich', 'Condiment',
                'Lunch',
                'Snack', 'Bread', 'Entree', 'Soup', 'Side_Dish', 'Appetizer', 'Dinner', 'Pasta', 'Breakfast', 'Salad',
                'Ingredient', 'Bacteria_Acidaminococcus', 'Bacteria_Bacteroidetes', 'Bacteria_Alistipes',
                'Bacteria_Bifidobacterium', 'Bacteria_Akkermansia', 'Bacteria_Blautia', 'Bacteria_Fecalibacterium',
                'Bacteria_Lawsonibacter_asaccharolyticus', 'Bacteria_Lactobacillus', 'Vegan', 'Umami', 'Slow_cooked',
                'Spicy', 'Baked', 'Pickled', 'Low_calorie', 'Grilled', 'Roasted', 'High_protein', 'Fried', 'Sweet',
                'Salty',
                'Lactose_free', 'Low_carb', 'Cold', 'Vegetarian', 'Sour', 'Sweet_Sour', 'Raw', 'Gluten_free', 'Boiled',
                'Steamed', 'Low_fat', 'Prepared_Meals', 'Meats', 'Sweets', 'Beans_and_Lentils', 'Snacks',
                'Grains_and_Pasta', 'Nuts_and_Seeds', 'Fruits', 'Beverages', 'Fish', 'Baked_Foods', 'Breakfast_Cereals',
                'Vegetables', 'Spices_and_Herbs', 'Soups_and_Sauces', 'Dairy_and_Egg_Products', 'Fats_and_Oils',
                'Lactose_Intolerance', 'stones', 'reflux', 'Spices', 'Seafood', 'Poultry', 'Herbs', 'Processed_Meats',
                'Legumes', 'Dairy', 'Nuts', 'Mixed_Dishes', 'Sauces', 'Oils', 'Grains', 'Seeds', 'Pastries', 'Spreads',
                'Eggs', 'Prebiotic', 'Probiotic', 'Low_Inflammatory', 'Diverse', 'Polyphenol_Rich', 'Stews',
                'Mixed_Rice',
                'Soups_and_Aashes', 'Traditional_Foods', 'International_Foods', 'Vegetarian_Foods', 'Simple_Dishes',
                'Main_Salads', 'Fast_Foods', 'AL_Dairy_Butter', 'AL_Dairy_Cheese',
                'AL_Dairy_Coconut', 'AL_Dairy_CoconutMilk', 'AL_Dairy_Cream', 'AL_Dairy_CreamCheese',
                'AL_Dairy_IceCream', 'AL_Dairy_Mayonnaise', 'AL_Dairy_Milk', 'AL_Dairy_Milk_Chocolate',
                'AL_Dairy_Whey', 'AL_Dairy_Yogurt', 'AL_Egg', 'AL_Eggs_Egg', 'AL_Eggs_Mayonnaise', 'AL_Fish_Crab',
                'AL_Fish_Fish', 'AL_Fish_Shrimp', 'AL_Fish_Tuna', 'AL_Grain_Barley', 'AL_Grain_Bean',
                'AL_Grain_Buckwheat', 'AL_Grain_Chickpea', 'AL_Grain_Corn', 'AL_Grain_Lentil', 'AL_Grain_Mung',
                'AL_Grain_Oat', 'AL_Grain_Oats', 'AL_Grain_Quinoa', 'AL_Grain_Rice', 'AL_Grain_Rye', 'AL_Grain_Soy',
                'AL_Grain_Unspecified', 'AL_Grain_Wheat', 'AL_Grains_Wheat', 'AL_Legume_Beans', 'AL_Legume_Lentil',
                'AL_Legume_Pea', 'AL_Legume_RedBean', 'AL_Legume_Soy', 'AL_Legume_Split_Peas', 'AL_Legume_SplitPea',
                'AL_Meat_Bacon', 'AL_Meat_Beef', 'AL_Meat_Chicken', 'AL_Meat_Ham', 'AL_Meat_Lamb', 'AL_Meat_Pepperoni',
                'AL_Meat_Pork', 'AL_Meat_Processed', 'AL_Meat_Sausage', 'AL_Meat_Turkey', 'AL_Meat_Unspecified',
                'AL_Meat_Varies', 'AL_Meat_Various', 'AL_Nuts_Almond', 'AL_Nuts_Coconut', 'AL_Nuts_Hazelnut',
                'AL_Nuts_Nuts', 'AL_Nuts_Peanut', 'AL_Nuts_Pine_Nuts', 'AL_Nuts_Pistachio', 'AL_Nuts_Sesame',
                'AL_Nuts_Unspecified', 'AL_Nuts_Varies', 'AL_Nuts_Various', 'AL_Nuts_Walnut', 'AL_Poultry_Chicken',
                'AL_Seeds_Chia', 'AL_Shellfish_Shrimp', 'AL_Dairy_Cream_Cheese', 'AL_Dairy_Creamer',
                'AL_Dairy_Goat_Milk', 'AL_Dairy_Kashk', 'AL_Dairy_Kefir', 'AL_Eggs_Whole', 'AL_Eggs_Whole_Eggs',
                'AL_Eggs_Yolk', 'AL_Fish_Anchovy', 'AL_Fish_Carp', 'AL_Fish_Catfish', 'AL_Fish_Caviar',
                'AL_Fish_Croaker', 'AL_Fish_Flatfish', 'AL_Fish_Flounder', 'AL_Fish_Grouper', 'AL_Fish_Herring',
                'AL_Fish_KingMackerel', 'AL_Fish_Mackerel', 'AL_Fish_Mullet', 'AL_Fish_Pike', 'AL_Fish_Porgy',
                'AL_Fish_Roe', 'AL_Fish_Salmon', 'AL_Fish_Sardine', 'AL_Fish_Shark', 'AL_Fish_Snapper',
                'AL_Fish_Swordfish', 'AL_Fish_Tilapia', 'AL_Fish_Trout', 'AL_Fish_Unspecified', 'AL_Fish_Whitefish',
                'AL_Grain_Bulgur', 'AL_Grain_Cereal', 'AL_Grain_Millet', 'AL_Grain_Pea', 'AL_Grain_SplitPea',
                'AL_Grains_Barley', 'AL_Grains_Brown_Rice', 'AL_Grains_Bulgur', 'AL_Grains_Chickpea', 'AL_Grains_Corn',
                'AL_Grains_Millet', 'AL_Grains_Oats', 'AL_Grains_Rice', 'AL_Legume_Bean', 'AL_Legume_Chickpea',
                'AL_Legume_Fava', 'AL_Legume_FavaBean', 'AL_Legume_Peas', 'AL_Legume_Unspecified', 'AL_Meat_Camel',
                'AL_Meat_Duck', 'AL_Meat_Fat', 'AL_Meat_Fish', 'AL_Meat_Goose', 'AL_Meat_Ground', 'AL_Meat_Kidney',
                'AL_Meat_Offal', 'AL_Meat_Organ', 'AL_Meat_Ostrich', 'AL_Meat_Pigeon', 'AL_Meat_Quail', 'AL_Meat_Sheep',
                'AL_Meat_Veal', 'AL_Nuts_Acorn', 'AL_Nuts_Almonds', 'AL_Nuts_Cashew', 'AL_Nuts_Chestnut',
                'AL_Nuts_Hazelnuts', 'AL_Nuts_Hemp', 'AL_Nuts_Mixed', 'AL_Nuts_Nutmeg', 'AL_Nuts_Peanuts',
                'AL_Nuts_Pecan', 'AL_Other_Sesame', 'AL_Poultry_Duck', 'AL_Poultry_Offal', 'AL_Seafood_Fish',
                'AL_Seeds_Cottonseed', 'AL_Seeds_Flax', 'AL_Seeds_Flaxseed', 'AL_Seeds_Poppy', 'AL_Seeds_Pumpkin',
                'AL_Seeds_Safflower', 'AL_Seeds_Sesame', 'AL_Seeds_Sunflower', 'AL_Seeds_Watermelon',
                'AL_Sesame_Sesame', 'AL_Soy_Soy', 'AL_Soy_SoySauce', 'AL_Soy_Soybean', 'AL_Soy_Soybeans',
                'Diet_Low_Sugar', 'Diet_Mediterranean',
                'Diet_High_Protein', 'Whole_Foods', 'Diet_Low_Fat', 'Diet_Low_Calorie', 'Diet_Anti_Inflammatory',
                'Diet_Paleo', 'Diet_Diabetic', 'Diet_Low_Carb', 'Diet_Bodybuilding', 'Diet_Vegan', 'Diet_Vegetarian',
                'Diet_Plant_Based', 'Diet_Gluten_Free', 'Diet_Keto', 'cooked_rice', 'main_dish',
                'Disease_Kidney', 'Disease_Kidney_stones',
                'Disease_Acid_reflux', 'Disease_Gout', 'Disease_Celiac', 'Disease_IBS', 'Disease_Diverticulitis',
                'Disease_Hypothyroidism', 'Disease_Gastroesophageal_Reflux', 'Disease_Foodborne_Illness',
                'Disease_High_blood_pressure', 'Disease_High_cholesterol', 'Disease_Diabetes',
                'Disease_Hypercholesterolemia', 'Disease_G6PD_Deficiency', 'Disease_Constipation',
                'Disease_Gallbladder',
                'Disease_Latex_Fruit_Syndrome', 'Disease_Obesity', 'Disease_CVD', 'Disease_IBD',
                'Disease_Lactose_Intolerance', 'Disease_NAFLD', 'Disease_Anemia', 'Disease_Osteoporosis',
                'Disease_Arthritis', 'Disease_Autoimmune', 'Disease_CancerPrevention', 'Disease_MentalHealth',
            ]
            # Handle regular tags (non-allergen tags)
            regular_tags = [tag for tag in tag_fields if not tag.startswith('AL_')]
            for tag in regular_tags:
                setattr(card, tag, card_data.get(tag, False))
            
            # Handle allergen tags - Set all to True by default, then set found ones to False
            allergen_tags = [tag for tag in tag_fields if tag.startswith('AL_')]
            for allergen_tag in allergen_tags:
                setattr(card, allergen_tag, True)  # Default to True (unsafe)
            
            # Now set the specific allergens found in this card to False (safe)
            for allergen_tag in allergen_tags:
                if allergen_tag in card_data and card_data[allergen_tag] is False:
                    setattr(card, allergen_tag, False)
                    print(f"Setting {allergen_tag} to False (safe for this card)")
            # Update foods
            card.foods = card_data.get('foods', [])
            card.edited_by = django_user
            card.edited_at = datetime.now()
            card.save()
            return JsonResponse({'status': 'success'})
        except FoodCard.DoesNotExist:
            return JsonResponse({'error': 'Card not found'}, status=404)
        except Exception as e:
            return JsonResponse({'error': str(e)}, status=500)
    else:
        return JsonResponse({'error': 'Invalid request method'}, status=400)


@api_view(["POST", "GET"])
def manage_card_tags(request):
    if request.method == 'POST':
        selected_card_ids = request.POST.getlist('card_ids')
        selected_tags = request.POST.getlist('tags')

        try:
            # Update the selected cards with the selected tags
            # Assuming tags are Boolean fields on the FoodCard model
            update_fields = {tag: True for tag in selected_tags}
            FoodCard.objects.filter(id__in=selected_card_ids).update(**update_fields)
            return JsonResponse({'success': True, 'message': 'تگ‌ها با موفقیت به کارت‌های انتخاب شده اضافه شدند'})
        except Exception as e:
            # Log the error as needed
            print(f"Error updating tags: {e}")
            return JsonResponse({'success': False, 'message': 'خطا در اعمال تگ‌ها.'})
    else:
        if not request.user.is_authenticated:
            return redirect('panel-login')
        user = request.user
        assistant = Assistant.objects.get(django_user=user)
        # Assuming new_messages is a helper function
        zipped_data, total_new_messages, free_time = new_messages(request)
        total_notifications = len(free_time)

        tag_info_list = [
            {'name': 'is_keto', 'display_name': 'Keto'},
            {'name': 'is_med', 'display_name': 'Med'},
            {'name': 'is_conditional_med', 'display_name': 'Conditional Med'},
            {'name': 'is_lowcarb', 'display_name': 'Low Carb'},
            {'name': 'is_mediumcarb', 'display_name': 'Medium Carb'},
            {'name': 'is_highcarb', 'display_name': 'High Carb'},
            {'name': 'is_highprotein', 'display_name': 'High Protein'},
            {'name': 'is_mediumprotein', 'display_name': 'Medium Protein'},
            {'name': 'is_lowprotein', 'display_name': 'Low Protein'},
            {'name': 'is_lowfat', 'display_name': 'Low Fat'},
            {'name': 'is_mediumfat', 'display_name': 'Medium Fat'},
            {'name': 'is_highfat', 'display_name': 'High Fat'},
            {'name': 'is_mind', 'display_name': 'Mind'},
            {'name': 'is_low_fodmap', 'display_name': 'Low FODMAP'},
            {'name': 'is_breakfast', 'display_name': 'صبحانه'},
            {'name': 'is_lunch', 'display_name': 'ناهار'},
            {'name': 'is_dinner', 'display_name': 'شام'},
            {'name': 'is_snack', 'display_name': 'میان وعده'},
            {'name': 'Dessert', 'display_name': 'Dessert'},
            {'name': 'Added_Meat', 'display_name': 'Added_Meat'},
            {'name': 'Ing_Main', 'display_name': 'Ing_Main'},
            {'name': 'Beverage', 'display_name': 'Beverage'},
            {'name': 'Sandwich', 'display_name': 'Sandwich'},
            {'name': 'Condiment', 'display_name': 'Condiment'},
            {'name': 'Lunch', 'display_name': 'Lunch'},
            {'name': 'Snack', 'display_name': 'Snack'},
            {'name': 'Bread', 'display_name': 'Bread'},
            {'name': 'Entree', 'display_name': 'Entree'},
            {'name': 'Soup', 'display_name': 'Soup'},
            {'name': 'Side_Dish', 'display_name': 'Side_Dish'},
            {'name': 'Appetizer', 'display_name': 'Appetizer'},
            {'name': 'Dinner', 'display_name': 'Dinner'},
            {'name': 'Pasta', 'display_name': 'Pasta'},
            {'name': 'Breakfast', 'display_name': 'Breakfast'},
            {'name': 'Salad', 'display_name': 'Salad'},
            {'name': 'Ingredient', 'display_name': 'Ingredient'},
            {'name': 'Bacteria_Acidaminococcus', 'display_name': 'Bacteria_Acidaminococcus'},
            {'name': 'Bacteria_Bacteroidetes', 'display_name': 'Bacteria_Bacteroidetes'},
            {'name': 'Bacteria_Alistipes', 'display_name': 'Bacteria_Alistipes'},
            {'name': 'Bacteria_Bifidobacterium', 'display_name': 'Bacteria_Bifidobacterium'},
            {'name': 'Bacteria_Akkermansia', 'display_name': 'Bacteria_Akkermansia'},
            {'name': 'Bacteria_Blautia', 'display_name': 'Bacteria_Blautia'},
            {'name': 'Bacteria_Fecalibacterium', 'display_name': 'Bacteria_Fecalibacterium'},
            {'name': 'Bacteria_Lawsonibacter_asaccharolyticus',
             'display_name': 'Bacteria_Lawsonibacter_asaccharolyticus'},
            {'name': 'Bacteria_Lactobacillus', 'display_name': 'Bacteria_Lactobacillus'},
            {'name': 'Vegan', 'display_name': 'Vegan'},
            {'name': 'Umami', 'display_name': 'Umami'},
            {'name': 'Slow_cooked', 'display_name': 'Slow_cooked'},
            {'name': 'Spicy', 'display_name': 'Spicy'},
            {'name': 'Baked', 'display_name': 'Baked'},
            {'name': 'Pickled', 'display_name': 'Pickled'},
            {'name': 'Low_calorie', 'display_name': 'Low_calorie'},
            {'name': 'Grilled', 'display_name': 'Grilled'},
            {'name': 'Roasted', 'display_name': 'Roasted'},
            {'name': 'High_protein', 'display_name': 'High_protein'},
            {'name': 'Fried', 'display_name': 'Fried'},
            {'name': 'Sweet', 'display_name': 'Sweet'},
            {'name': 'Salty', 'display_name': 'Salty'},
            {'name': 'Lactose_free', 'display_name': 'Lactose_free'},
            {'name': 'Low_carb', 'display_name': 'Low_carb'},
            {'name': 'Cold', 'display_name': 'Cold'},
            {'name': 'Vegetarian', 'display_name': 'Vegetarian'},
            {'name': 'Sour', 'display_name': 'Sour'},
            {'name': 'Sweet_Sour', 'display_name': 'Sweet_Sour'},
            {'name': 'Raw', 'display_name': 'Raw'},
            {'name': 'Gluten_free', 'display_name': 'Gluten_free'},
            {'name': 'Boiled', 'display_name': 'Boiled'},
            {'name': 'Steamed', 'display_name': 'Steamed'},
            {'name': 'Low_fat', 'display_name': 'Low_fat'},
            {'name': 'Prepared_Meals', 'display_name': 'Prepared_Meals'},
            {'name': 'Meats', 'display_name': 'Meats'},
            {'name': 'Sweets', 'display_name': 'Sweets'},
            {'name': 'Beans_and_Lentils', 'display_name': 'Beans_and_Lentils'},
            {'name': 'Snacks', 'display_name': 'Snacks'},
            {'name': 'Grains_and_Pasta', 'display_name': 'Grains_and_Pasta'},
            {'name': 'Nuts_and_Seeds', 'display_name': 'Nuts_and_Seeds'},
            {'name': 'Fruits', 'display_name': 'Fruits'},
            {'name': 'Beverages', 'display_name': 'Beverages'},
            {'name': 'Fish', 'display_name': 'Fish'},
            {'name': 'Baked_Foods', 'display_name': 'Baked_Foods'},
            {'name': 'Breakfast_Cereals', 'display_name': 'Breakfast_Cereals'},
            {'name': 'Vegetables', 'display_name': 'Vegetables'},
            {'name': 'Spices_and_Herbs', 'display_name': 'Spices_and_Herbs'},
            {'name': 'Soups_and_Sauces', 'display_name': 'Soups_and_Sauces'},
            {'name': 'Dairy_and_Egg_Products', 'display_name': 'Dairy_and_Egg_Products'},
            {'name': 'Fats_and_Oils', 'display_name': 'Fats_and_Oils'},
            {'name': 'Lactose_Intolerance', 'display_name': 'Lactose_Intolerance'},
            {'name': 'stones', 'display_name': 'stones'},
            {'name': 'reflux', 'display_name': 'reflux'},
            {'name': 'Spices', 'display_name': 'Spices'},
            {'name': 'Seafood', 'display_name': 'Seafood'},
            {'name': 'Poultry', 'display_name': 'Poultry'},
            {'name': 'Herbs', 'display_name': 'Herbs'},
            {'name': 'Processed_Meats', 'display_name': 'Processed_Meats'},
            {'name': 'Legumes', 'display_name': 'Legumes'},
            {'name': 'Dairy', 'display_name': 'Dairy'},
            {'name': 'Nuts', 'display_name': 'Nuts'},
            {'name': 'Mixed_Dishes', 'display_name': 'Mixed_Dishes'},
            {'name': 'Sauces', 'display_name': 'Sauces'},
            {'name': 'Oils', 'display_name': 'Oils'},
            {'name': 'Grains', 'display_name': 'Grains'},
            {'name': 'Seeds', 'display_name': 'Seeds'},
            {'name': 'Pastries', 'display_name': 'Pastries'},
            {'name': 'Spreads', 'display_name': 'Spreads'},
            {'name': 'Eggs', 'display_name': 'Eggs'},
            {'name': 'Prebiotic', 'display_name': 'Prebiotic'},
            {'name': 'Probiotic', 'display_name': 'Probiotic'},
            {'name': 'Low_Inflammatory', 'display_name': 'Low_Inflammatory'},
            {'name': 'Diverse', 'display_name': 'Diverse'},
            {'name': 'Polyphenol_Rich', 'display_name': 'Polyphenol_Rich'},
            {'name': 'Stews', 'display_name': 'Stews'},
            {'name': 'Mixed_Rice', 'display_name': 'Mixed_Rice'},
            {'name': 'Soups_and_Aashes', 'display_name': 'Soups_and_Aashes'},
            {'name': 'Traditional_Foods', 'display_name': 'Traditional_Foods'},
            {'name': 'International_Foods', 'display_name': 'International_Foods'},
            {'name': 'Vegetarian_Foods', 'display_name': 'Vegetarian_Foods'},
            {'name': 'Simple_Dishes', 'display_name': 'Simple_Dishes'},
            {'name': 'Main_Salads', 'display_name': 'Main_Salads'},
            {'name': 'Fast_Foods', 'display_name': 'Fast_Foods'},
            {'name': 'FODMAP_Classification', 'display_name': 'FODMAP_Classification'},
            {'name': 'AL_Dairy_Butter', 'display_name': 'AL_Dairy_Butter'},
            {'name': 'AL_Dairy_Cheese', 'display_name': 'AL_Dairy_Cheese'},
            {'name': 'AL_Dairy_Coconut', 'display_name': 'AL_Dairy_Coconut'},
            {'name': 'AL_Dairy_CoconutMilk', 'display_name': 'AL_Dairy_CoconutMilk'},
            {'name': 'AL_Dairy_Cream', 'display_name': 'AL_Dairy_Cream'},
            {'name': 'AL_Dairy_CreamCheese', 'display_name': 'AL_Dairy_CreamCheese'},
            {'name': 'AL_Dairy_IceCream', 'display_name': 'AL_Dairy_IceCream'},
            {'name': 'AL_Dairy_Mayonnaise', 'display_name': 'AL_Dairy_Mayonnaise'},
            {'name': 'AL_Dairy_Milk', 'display_name': 'AL_Dairy_Milk'},
            {'name': 'AL_Dairy_Milk_Chocolate', 'display_name': 'AL_Dairy_Milk_Chocolate'},
            {'name': 'AL_Dairy_Whey', 'display_name': 'AL_Dairy_Whey'},
            {'name': 'AL_Dairy_Yogurt', 'display_name': 'AL_Dairy_Yogurt'},
            {'name': 'AL_Egg', 'display_name': 'AL_Egg'},
            {'name': 'AL_Eggs_Egg', 'display_name': 'AL_Eggs_Egg'},
            {'name': 'AL_Eggs_Mayonnaise', 'display_name': 'AL_Eggs_Mayonnaise'},
            {'name': 'AL_Fish_Crab', 'display_name': 'AL_Fish_Crab'},
            {'name': 'AL_Fish_Fish', 'display_name': 'AL_Fish_Fish'},
            {'name': 'AL_Fish_Shrimp', 'display_name': 'AL_Fish_Shrimp'},
            {'name': 'AL_Fish_Tuna', 'display_name': 'AL_Fish_Tuna'},
            {'name': 'AL_Grain_Barley', 'display_name': 'AL_Grain_Barley'},
            {'name': 'AL_Grain_Bean', 'display_name': 'AL_Grain_Bean'},
            {'name': 'AL_Grain_Buckwheat', 'display_name': 'AL_Grain_Buckwheat'},
            {'name': 'AL_Grain_Chickpea', 'display_name': 'AL_Grain_Chickpea'},
            {'name': 'AL_Grain_Corn', 'display_name': 'AL_Grain_Corn'},
            {'name': 'AL_Grain_Lentil', 'display_name': 'AL_Grain_Lentil'},
            {'name': 'AL_Grain_Mung', 'display_name': 'AL_Grain_Mung'},
            {'name': 'AL_Grain_Oat', 'display_name': 'AL_Grain_Oat'},
            {'name': 'AL_Grain_Oats', 'display_name': 'AL_Grain_Oats'},
            {'name': 'AL_Grain_Quinoa', 'display_name': 'AL_Grain_Quinoa'},
            {'name': 'AL_Grain_Rice', 'display_name': 'AL_Grain_Rice'},
            {'name': 'AL_Grain_Rye', 'display_name': 'AL_Grain_Rye'},
            {'name': 'AL_Grain_Soy', 'display_name': 'AL_Grain_Soy'},
            {'name': 'AL_Grain_Unspecified', 'display_name': 'AL_Grain_Unspecified'},
            {'name': 'AL_Grain_Wheat', 'display_name': 'AL_Grain_Wheat'},
            {'name': 'AL_Grains_Wheat', 'display_name': 'AL_Grains_Wheat'},
            {'name': 'AL_Legume_Beans', 'display_name': 'AL_Legume_Beans'},
            {'name': 'AL_Legume_Lentil', 'display_name': 'AL_Legume_Lentil'},
            {'name': 'AL_Legume_Pea', 'display_name': 'AL_Legume_Pea'},
            {'name': 'AL_Legume_RedBean', 'display_name': 'AL_Legume_RedBean'},
            {'name': 'AL_Legume_Soy', 'display_name': 'AL_Legume_Soy'},
            {'name': 'AL_Legume_Split_Peas', 'display_name': 'AL_Legume_Split_Peas'},
            {'name': 'AL_Legume_SplitPea', 'display_name': 'AL_Legume_SplitPea'},
            {'name': 'AL_Meat_Bacon', 'display_name': 'AL_Meat_Bacon'},
            {'name': 'AL_Meat_Beef', 'display_name': 'AL_Meat_Beef'},
            {'name': 'AL_Meat_Chicken', 'display_name': 'AL_Meat_Chicken'},
            {'name': 'AL_Meat_Ham', 'display_name': 'AL_Meat_Ham'},
            {'name': 'AL_Meat_Lamb', 'display_name': 'AL_Meat_Lamb'},
            {'name': 'AL_Meat_Pepperoni', 'display_name': 'AL_Meat_Pepperoni'},
            {'name': 'AL_Meat_Pork', 'display_name': 'AL_Meat_Pork'},
            {'name': 'AL_Meat_Processed', 'display_name': 'AL_Meat_Processed'},
            {'name': 'AL_Meat_Sausage', 'display_name': 'AL_Meat_Sausage'},
            {'name': 'AL_Meat_Turkey', 'display_name': 'AL_Meat_Turkey'},
            {'name': 'AL_Meat_Unspecified', 'display_name': 'AL_Meat_Unspecified'},
            {'name': 'AL_Meat_Varies', 'display_name': 'AL_Meat_Varies'},
            {'name': 'AL_Meat_Various', 'display_name': 'AL_Meat_Various'},
            {'name': 'AL_Nuts_Almond', 'display_name': 'AL_Nuts_Almond'},
            {'name': 'AL_Nuts_Coconut', 'display_name': 'AL_Nuts_Coconut'},
            {'name': 'AL_Nuts_Hazelnut', 'display_name': 'AL_Nuts_Hazelnut'},
            {'name': 'AL_Nuts_Nuts', 'display_name': 'AL_Nuts_Nuts'},
            {'name': 'AL_Nuts_Peanut', 'display_name': 'AL_Nuts_Peanut'},
            {'name': 'AL_Nuts_Pine_Nuts', 'display_name': 'AL_Nuts_Pine_Nuts'},
            {'name': 'AL_Nuts_Pistachio', 'display_name': 'AL_Nuts_Pistachio'},
            {'name': 'AL_Nuts_Sesame', 'display_name': 'AL_Nuts_Sesame'},
            {'name': 'AL_Nuts_Unspecified', 'display_name': 'AL_Nuts_Unspecified'},
            {'name': 'AL_Nuts_Varies', 'display_name': 'AL_Nuts_Varies'},
            {'name': 'AL_Nuts_Various', 'display_name': 'AL_Nuts_Various'},
            {'name': 'AL_Nuts_Walnut', 'display_name': 'AL_Nuts_Walnut'},
            {'name': 'AL_Poultry_Chicken', 'display_name': 'AL_Poultry_Chicken'},
            {'name': 'AL_Seeds_Chia', 'display_name': 'AL_Seeds_Chia'},
            {'name': 'AL_Shellfish_Shrimp', 'display_name': 'AL_Shellfish_Shrimp'},
            {'name': 'AL_Dairy_Cream_Cheese', 'display_name': 'AL_Dairy_Cream_Cheese'},
            {'name': 'AL_Dairy_Creamer', 'display_name': 'AL_Dairy_Creamer'},
            {'name': 'AL_Dairy_Goat_Milk', 'display_name': 'AL_Dairy_Goat_Milk'},
            {'name': 'AL_Dairy_Kashk', 'display_name': 'AL_Dairy_Kashk'},
            {'name': 'AL_Dairy_Kefir', 'display_name': 'AL_Dairy_Kefir'},
            {'name': 'AL_Eggs_Whole', 'display_name': 'AL_Eggs_Whole'},
            {'name': 'AL_Eggs_Whole_Eggs', 'display_name': 'AL_Eggs_Whole_Eggs'},
            {'name': 'AL_Eggs_Yolk', 'display_name': 'AL_Eggs_Yolk'},
            {'name': 'AL_Fish_Anchovy', 'display_name': 'AL_Fish_Anchovy'},
            {'name': 'AL_Fish_Carp', 'display_name': 'AL_Fish_Carp'},
            {'name': 'AL_Fish_Catfish', 'display_name': 'AL_Fish_Catfish'},
            {'name': 'AL_Fish_Caviar', 'display_name': 'AL_Fish_Caviar'},
            {'name': 'AL_Fish_Croaker', 'display_name': 'AL_Fish_Croaker'},
            {'name': 'AL_Fish_Flatfish', 'display_name': 'AL_Fish_Flatfish'},
            {'name': 'AL_Fish_Flounder', 'display_name': 'AL_Fish_Flounder'},
            {'name': 'AL_Fish_Grouper', 'display_name': 'AL_Fish_Grouper'},
            {'name': 'AL_Fish_Herring', 'display_name': 'AL_Fish_Herring'},
            {'name': 'AL_Fish_KingMackerel', 'display_name': 'AL_Fish_KingMackerel'},
            {'name': 'AL_Fish_Mackerel', 'display_name': 'AL_Fish_Mackerel'},
            {'name': 'AL_Fish_Mullet', 'display_name': 'AL_Fish_Mullet'},
            {'name': 'AL_Fish_Pike', 'display_name': 'AL_Fish_Pike'},
            {'name': 'AL_Fish_Porgy', 'display_name': 'AL_Fish_Porgy'},
            {'name': 'AL_Fish_Roe', 'display_name': 'AL_Fish_Roe'},
            {'name': 'AL_Fish_Salmon', 'display_name': 'AL_Fish_Salmon'},
            {'name': 'AL_Fish_Sardine', 'display_name': 'AL_Fish_Sardine'},
            {'name': 'AL_Fish_Shark', 'display_name': 'AL_Fish_Shark'},
            {'name': 'AL_Fish_Snapper', 'display_name': 'AL_Fish_Snapper'},
            {'name': 'AL_Fish_Swordfish', 'display_name': 'AL_Fish_Swordfish'},
            {'name': 'AL_Fish_Tilapia', 'display_name': 'AL_Fish_Tilapia'},
            {'name': 'AL_Fish_Trout', 'display_name': 'AL_Fish_Trout'},
            {'name': 'AL_Fish_Unspecified', 'display_name': 'AL_Fish_Unspecified'},
            {'name': 'AL_Fish_Whitefish', 'display_name': 'AL_Fish_Whitefish'},
            {'name': 'AL_Grain_Bulgur', 'display_name': 'AL_Grain_Bulgur'},
            {'name': 'AL_Grain_Cereal', 'display_name': 'AL_Grain_Cereal'},
            {'name': 'AL_Grain_Millet', 'display_name': 'AL_Grain_Millet'},
            {'name': 'AL_Grain_Pea', 'display_name': 'AL_Grain_Pea'},
            {'name': 'AL_Grain_SplitPea', 'display_name': 'AL_Grain_SplitPea'},
            {'name': 'AL_Grains_Barley', 'display_name': 'AL_Grains_Barley'},
            {'name': 'AL_Grains_Brown_Rice', 'display_name': 'AL_Grains_Brown_Rice'},
            {'name': 'AL_Grains_Bulgur', 'display_name': 'AL_Grains_Bulgur'},
            {'name': 'AL_Grains_Chickpea', 'display_name': 'AL_Grains_Chickpea'},
            {'name': 'AL_Grains_Corn', 'display_name': 'AL_Grains_Corn'},
            {'name': 'AL_Grains_Millet', 'display_name': 'AL_Grains_Millet'},
            {'name': 'AL_Grains_Oats', 'display_name': 'AL_Grains_Oats'},
            {'name': 'AL_Grains_Rice', 'display_name': 'AL_Grains_Rice'},
            {'name': 'AL_Legume_Bean', 'display_name': 'AL_Legume_Bean'},
            {'name': 'AL_Legume_Chickpea', 'display_name': 'AL_Legume_Chickpea'},
            {'name': 'AL_Legume_Fava', 'display_name': 'AL_Legume_Fava'},
            {'name': 'AL_Legume_FavaBean', 'display_name': 'AL_Legume_FavaBean'},
            {'name': 'AL_Legume_Peas', 'display_name': 'AL_Legume_Peas'},
            {'name': 'AL_Legume_Unspecified', 'display_name': 'AL_Legume_Unspecified'},
            {'name': 'AL_Meat_Camel', 'display_name': 'AL_Meat_Camel'},
            {'name': 'AL_Meat_Duck', 'display_name': 'AL_Meat_Duck'},
            {'name': 'AL_Meat_Fat', 'display_name': 'AL_Meat_Fat'},
            {'name': 'AL_Meat_Fish', 'display_name': 'AL_Meat_Fish'},
            {'name': 'AL_Meat_Goose', 'display_name': 'AL_Meat_Goose'},
            {'name': 'AL_Meat_Ground', 'display_name': 'AL_Meat_Ground'},
            {'name': 'AL_Meat_Kidney', 'display_name': 'AL_Meat_Kidney'},
            {'name': 'AL_Meat_Offal', 'display_name': 'AL_Meat_Offal'},
            {'name': 'AL_Meat_Organ', 'display_name': 'AL_Meat_Organ'},
            {'name': 'AL_Meat_Ostrich', 'display_name': 'AL_Meat_Ostrich'},
            {'name': 'AL_Meat_Pigeon', 'display_name': 'AL_Meat_Pigeon'},
            {'name': 'AL_Meat_Quail', 'display_name': 'AL_Meat_Quail'},
            {'name': 'AL_Meat_Sheep', 'display_name': 'AL_Meat_Sheep'},
            {'name': 'AL_Meat_Veal', 'display_name': 'AL_Meat_Veal'},
            {'name': 'AL_Nuts_Acorn', 'display_name': 'AL_Nuts_Acorn'},
            {'name': 'AL_Nuts_Almonds', 'display_name': 'AL_Nuts_Almonds'},
            {'name': 'AL_Nuts_Cashew', 'display_name': 'AL_Nuts_Cashew'},
            {'name': 'AL_Nuts_Chestnut', 'display_name': 'AL_Nuts_Chestnut'},
            {'name': 'AL_Nuts_Hazelnuts', 'display_name': 'AL_Nuts_Hazelnuts'},
            {'name': 'AL_Nuts_Hemp', 'display_name': 'AL_Nuts_Hemp'},
            {'name': 'AL_Nuts_Mixed', 'display_name': 'AL_Nuts_Mixed'},
            {'name': 'AL_Nuts_Nutmeg', 'display_name': 'AL_Nuts_Nutmeg'},
            {'name': 'AL_Nuts_Peanuts', 'display_name': 'AL_Nuts_Peanuts'},
            {'name': 'AL_Nuts_Pecan', 'display_name': 'AL_Nuts_Pecan'},
            {'name': 'AL_Other_Sesame', 'display_name': 'AL_Other_Sesame'},
            {'name': 'AL_Poultry_Duck', 'display_name': 'AL_Poultry_Duck'},
            {'name': 'AL_Poultry_Offal', 'display_name': 'AL_Poultry_Offal'},
            {'name': 'AL_Seafood_Fish', 'display_name': 'AL_Seafood_Fish'},
            {'name': 'AL_Seeds_Cottonseed', 'display_name': 'AL_Seeds_Cottonseed'},
            {'name': 'AL_Seeds_Flax', 'display_name': 'AL_Seeds_Flax'},
            {'name': 'AL_Seeds_Flaxseed', 'display_name': 'AL_Seeds_Flaxseed'},
            {'name': 'AL_Seeds_Poppy', 'display_name': 'AL_Seeds_Poppy'},
            {'name': 'AL_Seeds_Pumpkin', 'display_name': 'AL_Seeds_Pumpkin'},
            {'name': 'AL_Seeds_Safflower', 'display_name': 'AL_Seeds_Safflower'},
            {'name': 'AL_Seeds_Sesame', 'display_name': 'AL_Seeds_Sesame'},
            {'name': 'AL_Seeds_Sunflower', 'display_name': 'AL_Seeds_Sunflower'},
            {'name': 'AL_Seeds_Watermelon', 'display_name': 'AL_Seeds_Watermelon'},
            {'name': 'AL_Sesame_Sesame', 'display_name': 'AL_Sesame_Sesame'},
            {'name': 'AL_Soy_Soy', 'display_name': 'AL_Soy_Soy'},
            {'name': 'AL_Soy_SoySauce', 'display_name': 'AL_Soy_SoySauce'},
            {'name': 'AL_Soy_Soybean', 'display_name': 'AL_Soy_Soybean'},
            {'name': 'AL_Soy_Soybeans', 'display_name': 'AL_Soy_Soybeans'},
            {'name': 'Diet_Low_Sugar', 'display_name': 'Diet_Low_Sugar'},
            {'name': 'Diet_Mediterranean', 'display_name': 'Diet_Mediterranean'},
            {'name': 'Diet_High_Protein', 'display_name': 'Diet_High_Protein'},
            {'name': 'Whole_Foods', 'display_name': 'Whole_Foods'},
            {'name': 'Diet_Low_Fat', 'display_name': 'Diet_Low_Fat'},
            {'name': 'Diet_Low_Calorie', 'display_name': 'Diet_Low_Calorie'},
            {'name': 'Diet_Anti_Inflammatory', 'display_name': 'Diet_Anti_Inflammatory'},
            {'name': 'Diet_Paleo', 'display_name': 'Diet_Paleo'},
            {'name': 'Diet_Diabetic', 'display_name': 'Diet_Diabetic'},
            {'name': 'Diet_Low_Carb', 'display_name': 'Diet_Low_Carb'},
            {'name': 'Diet_Bodybuilding', 'display_name': 'Diet_Bodybuilding'},
            {'name': 'Diet_Vegan', 'display_name': 'Diet_Vegan'},
            {'name': 'Diet_Vegetarian', 'display_name': 'Diet_Vegetarian'},
            {'name': 'Diet_Plant_Based', 'display_name': 'Diet_Plant_Based'},
            {'name': 'Diet_Gluten_Free', 'display_name': 'Diet_Gluten_Free'},
            {'name': 'Diet_Keto', 'display_name': 'Diet_Keto'},
            {'name': 'bread_rice_tag', 'display_name': 'bread_rice_tag'},
            {'name': 'cooked_rice', 'display_name': 'cooked_rice'},
            {'name': 'main_dish', 'display_name': 'main_dish'},
            {'name': 'GL_category', 'display_name': 'GL_category'},
            {'name': 'GL_value', 'display_name': 'GL_value'},
            {'name': 'GI_category', 'display_name': 'GI_category'},
            {'name': 'GI_value', 'display_name': 'GI_value'},
            {'name': 'Disease_Kidney', 'display_name': 'Disease_Kidney'},
            {'name': 'Disease_Kidney_stones', 'display_name': 'Disease_Kidney_stones'},
            {'name': 'Disease_Acid_reflux', 'display_name': 'Disease_Acid_reflux'},
            {'name': 'Disease_Gout', 'display_name': 'Disease_Gout'},
            {'name': 'Disease_Celiac', 'display_name': 'Disease_Celiac'},
            {'name': 'Disease_IBS', 'display_name': 'Disease_IBS'},
            {'name': 'Disease_Diverticulitis', 'display_name': 'Disease_Diverticulitis'},
            {'name': 'Disease_Hypothyroidism', 'display_name': 'Disease_Hypothyroidism'},
            {'name': 'Disease_Gastroesophageal_Reflux', 'display_name': 'Disease_Gastroesophageal_Reflux'},
            {'name': 'Disease_Foodborne_Illness', 'display_name': 'Disease_Foodborne_Illness'},
            {'name': 'Disease_High_blood_pressure', 'display_name': 'Disease_High_blood_pressure'},
            {'name': 'Disease_High_cholesterol', 'display_name': 'Disease_High_cholesterol'},
            {'name': 'Disease_Diabetes', 'display_name': 'Disease_Diabetes'},
            {'name': 'Disease_Hypercholesterolemia', 'display_name': 'Disease_Hypercholesterolemia'},
            {'name': 'Disease_G6PD_Deficiency', 'display_name': 'Disease_G6PD_Deficiency'},
            {'name': 'Disease_Constipation', 'display_name': 'Disease_Constipation'},
            {'name': 'Disease_Gallbladder', 'display_name': 'Disease_Gallbladder'},
            {'name': 'Disease_Latex_Fruit_Syndrome', 'display_name': 'Disease_Latex_Fruit_Syndrome'},
            {'name': 'Disease_Obesity', 'display_name': 'Disease_Obesity'},
            {'name': 'Disease_CVD', 'display_name': 'Disease_CVD'},
            {'name': 'Disease_IBD', 'display_name': 'Disease_IBD'},
            {'name': 'Disease_Lactose_Intolerance', 'display_name': 'Disease_Lactose_Intolerance'},
            {'name': 'Disease_NAFLD', 'display_name': 'Disease_NAFLD'},
            {'name': 'Disease_Anemia', 'display_name': 'Disease_Anemia'},
            {'name': 'Disease_Osteoporosis', 'display_name': 'Disease_Osteoporosis'},
            {'name': 'Disease_Arthritis', 'display_name': 'Disease_Arthritis'},
            {'name': 'Disease_Autoimmune', 'display_name': 'Disease_Autoimmune'},
            {'name': 'Disease_CancerPrevention', 'display_name': 'Disease_CancerPrevention'},
            {'name': 'Disease_MentalHealth', 'display_name': 'Disease_MentalHealth'},
        ]

        context = {
            'tag_info_list': tag_info_list,
            'user': assistant,
            "message": zipped_data,
            "total_new_messages": total_new_messages,
            'free_time': free_time,
            'total_notifications': total_notifications,
        }
        return render(request, 'assistant/manage_card_tags.html', context=context)


@api_view(["GET"])
def manage_card_tags_data(request):
    if not request.user.is_authenticated:
        return JsonResponse({'error': 'Unauthorized'}, status=401)

    try:
        user = request.user
        assistant = Assistant.objects.get(django_user=user)

        # DataTables parameters
        draw = int(request.GET.get('draw', 1))
        start = int(request.GET.get('start', 0))
        length = int(request.GET.get('length', 10))
        search_value = request.GET.get('search[value]', '').strip()
        order_column_index = request.GET.get('order[0][column]', 1)  # default to 'نام کارت'
        order_dir = request.GET.get('order[0][dir]', 'asc')  # corrected to 'dir]'

        # Mapping of column index to model fields
        column_order_map = {
            1: 'FA_Name',
            2: 'Calories',
            # Add more mappings if necessary
        }

        # Determine the column to order by
        try:
            order_column_index = int(order_column_index)
        except ValueError:
            order_column_index = 1  # Default to 'FA_Name' if invalid

        order_field = column_order_map.get(order_column_index, 'FA_Name')
        if order_dir == 'desc':
            order_field = f'-{order_field}'

        # Initial queryset
        queryset = FoodCard.objects.all()

        # Get selected tags from GET parameters (if implementing tag-based filtering)
        selected_tags = request.GET.getlist('tags[]')  # DataTables sends array as 'tags[]'
        if selected_tags:
            tag_filters = Q()
            for tag in selected_tags:
                tag_filters &= Q(**{tag: True})
            queryset = queryset.filter(tag_filters)

        # Apply search filter
        if search_value:
            queryset = queryset.filter(
                Q(FA_Name__icontains=search_value) |
                Q(Calories__icontains=search_value)
            )

        # Total records before filtering
        records_total = FoodCard.objects.count()

        # Total records after filtering
        records_filtered = queryset.count()

        # Apply ordering
        queryset = queryset.order_by(order_field)

        # Apply pagination
        paginator = Paginator(queryset, length)
        page_number = (start // length) + 1
        try:
            page_obj = paginator.page(page_number)
        except:
            page_obj = paginator.page(paginator.num_pages)

        # Prepare data for DataTables
        data = []
        for card in page_obj:
            # Prepare tags display
            active_tags = []
            tag_fields = [
                'is_keto', 'is_med', 'is_conditional_med', 'is_lowcarb', 'is_mediumcarb',
                'is_highcarb', 'is_highprotein', 'is_mediumprotein', 'is_lowprotein',
                'is_lowfat', 'is_mediumfat', 'is_highfat', 'is_mind', 'is_low_fodmap',
                'is_breakfast', 'is_lunch', 'is_dinner', 'is_snack', 'Dessert',
                'Added_Meat', 'Ing_Main', 'Beverage', 'Sandwich', 'Condiment',
                'Lunch', 'Snack', 'Bread', 'Entree', 'Soup', 'Side_Dish', 'Appetizer',
                'Dinner', 'Pasta', 'Breakfast', 'Salad', 'Ingredient',
                'Bacteria_Acidaminococcus', 'Bacteria_Bacteroidetes', 'Bacteria_Alistipes',
                'Bacteria_Bifidobacterium', 'Bacteria_Akkermansia', 'Bacteria_Blautia',
                'Bacteria_Fecalibacterium', 'Bacteria_Lawsonibacter_asaccharolyticus',
                'Bacteria_Lactobacillus', 'Vegan', 'Umami', 'Slow_cooked', 'Spicy',
                'Baked', 'Pickled', 'Low_calorie', 'Grilled', 'Roasted', 'High_protein',
                'Fried', 'Sweet', 'Salty', 'Lactose_free', 'Low_carb', 'Cold',
                'Vegetarian', 'Sour', 'Sweet_Sour', 'Raw', 'Gluten_free', 'Boiled',
                'Steamed', 'Low_fat', 'Prepared_Meals', 'Meats', 'Sweets',
                'Beans_and_Lentils', 'Snacks', 'Grains_and_Pasta', 'Nuts_and_Seeds',
                'Fruits', 'Beverages', 'Fish', 'Baked_Foods', 'Breakfast_Cereals',
                'Vegetables', 'Spices_and_Herbs', 'Soups_and_Sauces',
                'Dairy_and_Egg_Products', 'Fats_and_Oils', 'Lactose_Intolerance',
                'stones', 'reflux', 'Spices', 'Seafood', 'Poultry', 'Herbs',
                'Processed_Meats', 'Legumes', 'Dairy', 'Nuts', 'Mixed_Dishes',
                'Sauces', 'Oils', 'Grains', 'Seeds', 'Pastries', 'Spreads', 'Eggs',
                'Prebiotic', 'Probiotic', 'Low_Inflammatory', 'Diverse',
                'Polyphenol_Rich', 'Stews', 'Mixed_Rice', 'Soups_and_Aashes',
                'Traditional_Foods', 'International_Foods', 'Vegetarian_Foods',
                'Simple_Dishes', 'Main_Salads', 'Fast_Foods',
                'FODMAP_Classification', 'AL_Dairy_Butter', 'AL_Dairy_Cheese',
                'AL_Dairy_Coconut', 'AL_Dairy_CoconutMilk', 'AL_Dairy_Cream', 'AL_Dairy_CreamCheese',
                'AL_Dairy_IceCream', 'AL_Dairy_Mayonnaise', 'AL_Dairy_Milk', 'AL_Dairy_Milk_Chocolate',
                'AL_Dairy_Whey', 'AL_Dairy_Yogurt', 'AL_Egg', 'AL_Eggs_Egg', 'AL_Eggs_Mayonnaise', 'AL_Fish_Crab',
                'AL_Fish_Fish', 'AL_Fish_Shrimp', 'AL_Fish_Tuna', 'AL_Grain_Barley', 'AL_Grain_Bean',
                'AL_Grain_Buckwheat', 'AL_Grain_Chickpea', 'AL_Grain_Corn', 'AL_Grain_Lentil', 'AL_Grain_Mung',
                'AL_Grain_Oat', 'AL_Grain_Oats', 'AL_Grain_Quinoa', 'AL_Grain_Rice', 'AL_Grain_Rye', 'AL_Grain_Soy',
                'AL_Grain_Unspecified', 'AL_Grain_Wheat', 'AL_Grains_Wheat', 'AL_Legume_Beans', 'AL_Legume_Lentil',
                'AL_Legume_Pea', 'AL_Legume_RedBean', 'AL_Legume_Soy', 'AL_Legume_Split_Peas', 'AL_Legume_SplitPea',
                'AL_Meat_Bacon', 'AL_Meat_Beef', 'AL_Meat_Chicken', 'AL_Meat_Ham', 'AL_Meat_Lamb', 'AL_Meat_Pepperoni',
                'AL_Meat_Pork', 'AL_Meat_Processed', 'AL_Meat_Sausage', 'AL_Meat_Turkey', 'AL_Meat_Unspecified',
                'AL_Meat_Varies', 'AL_Meat_Various', 'AL_Nuts_Almond', 'AL_Nuts_Coconut', 'AL_Nuts_Hazelnut',
                'AL_Nuts_Nuts', 'AL_Nuts_Peanut', 'AL_Nuts_Pine_Nuts', 'AL_Nuts_Pistachio', 'AL_Nuts_Sesame',
                'AL_Nuts_Unspecified', 'AL_Nuts_Varies', 'AL_Nuts_Various', 'AL_Nuts_Walnut', 'AL_Poultry_Chicken',
                'AL_Seeds_Chia', 'AL_Shellfish_Shrimp', 'AL_Dairy_Cream_Cheese', 'AL_Dairy_Creamer',
                'AL_Dairy_Goat_Milk', 'AL_Dairy_Kashk', 'AL_Dairy_Kefir', 'AL_Eggs_Whole', 'AL_Eggs_Whole_Eggs',
                'AL_Eggs_Yolk', 'AL_Fish_Anchovy', 'AL_Fish_Carp', 'AL_Fish_Catfish', 'AL_Fish_Caviar',
                'AL_Fish_Croaker', 'AL_Fish_Flatfish', 'AL_Fish_Flounder', 'AL_Fish_Grouper', 'AL_Fish_Herring',
                'AL_Fish_KingMackerel', 'AL_Fish_Mackerel', 'AL_Fish_Mullet', 'AL_Fish_Pike', 'AL_Fish_Porgy',
                'AL_Fish_Roe', 'AL_Fish_Salmon', 'AL_Fish_Sardine', 'AL_Fish_Shark', 'AL_Fish_Snapper',
                'AL_Fish_Swordfish', 'AL_Fish_Tilapia', 'AL_Fish_Trout', 'AL_Fish_Unspecified', 'AL_Fish_Whitefish',
                'AL_Grain_Bulgur', 'AL_Grain_Cereal', 'AL_Grain_Millet', 'AL_Grain_Pea', 'AL_Grain_SplitPea',
                'AL_Grains_Barley', 'AL_Grains_Brown_Rice', 'AL_Grains_Bulgur', 'AL_Grains_Chickpea', 'AL_Grains_Corn',
                'AL_Grains_Millet', 'AL_Grains_Oats', 'AL_Grains_Rice', 'AL_Legume_Bean', 'AL_Legume_Chickpea',
                'AL_Legume_Fava', 'AL_Legume_FavaBean', 'AL_Legume_Peas', 'AL_Legume_Unspecified', 'AL_Meat_Camel',
                'AL_Meat_Duck', 'AL_Meat_Fat', 'AL_Meat_Fish', 'AL_Meat_Goose', 'AL_Meat_Ground', 'AL_Meat_Kidney',
                'AL_Meat_Offal', 'AL_Meat_Organ', 'AL_Meat_Ostrich', 'AL_Meat_Pigeon', 'AL_Meat_Quail', 'AL_Meat_Sheep',
                'AL_Meat_Veal', 'AL_Nuts_Acorn', 'AL_Nuts_Almonds', 'AL_Nuts_Cashew', 'AL_Nuts_Chestnut',
                'AL_Nuts_Hazelnuts', 'AL_Nuts_Hemp', 'AL_Nuts_Mixed', 'AL_Nuts_Nutmeg', 'AL_Nuts_Peanuts',
                'AL_Nuts_Pecan', 'AL_Other_Sesame', 'AL_Poultry_Duck', 'AL_Poultry_Offal', 'AL_Seafood_Fish',
                'AL_Seeds_Cottonseed', 'AL_Seeds_Flax', 'AL_Seeds_Flaxseed', 'AL_Seeds_Poppy', 'AL_Seeds_Pumpkin',
                'AL_Seeds_Safflower', 'AL_Seeds_Sesame', 'AL_Seeds_Sunflower', 'AL_Seeds_Watermelon',
                'AL_Sesame_Sesame', 'AL_Soy_Soy', 'AL_Soy_SoySauce', 'AL_Soy_Soybean', 'AL_Soy_Soybeans',
                'Diet_Low_Sugar', 'Diet_Mediterranean', 'Diet_High_Protein',
                'Whole_Foods', 'Diet_Low_Fat', 'Diet_Low_Calorie',
                'Diet_Anti_Inflammatory', 'Diet_Paleo', 'Diet_Diabetic',
                'Diet_Low_Carb', 'Diet_Bodybuilding', 'Diet_Vegan',
                'Diet_Vegetarian', 'Diet_Plant_Based', 'Diet_Gluten_Free',
                'Diet_Keto', 'bread_rice_tag', 'cooked_rice', 'main_dish',
                'GL_category', 'GL_value', 'GI_category', 'GI_value',
                'Disease_Kidney', 'Disease_Kidney_stones', 'Disease_Acid_reflux',
                'Disease_Gout', 'Disease_Celiac', 'Disease_IBS',
                'Disease_Diverticulitis', 'Disease_Hypothyroidism',
                'Disease_Gastroesophageal_Reflux', 'Disease_Foodborne_Illness',
                'Disease_High_blood_pressure', 'Disease_High_cholesterol',
                'Disease_Diabetes', 'Disease_Hypercholesterolemia',
                'Disease_G6PD_Deficiency', 'Disease_Constipation',
                'Disease_Gallbladder', 'Disease_Latex_Fruit_Syndrome',
                'Disease_Obesity', 'Disease_CVD', 'Disease_IBD', 'Disease_Lactose_Intolerance', 'Disease_NAFLD',
                'Disease_Anemia', 'Disease_Osteoporosis', 'Disease_Arthritis', 'Disease_Autoimmune',
                'Disease_CancerPrevention', 'Disease_MentalHealth',
            ]
            for tag in tag_fields:
                if getattr(card, tag, False):
                    active_tags.append(tag)

            # Prepare operations (Delete button)
            delete_button = f'<button type="button" class="btn btn-danger btn-sm" onclick="deleteCard(event, \'{card.id}\')"><i class="bi bi-trash"></i> حذف</button>'

            data.append({
                'checkbox': f'<div class="form-check"><input type="checkbox" name="card_ids" value="{card.id}" class="form-check-input cardCheckbox" aria-label="انتخاب کارت {card.FA_Name}"></div>',
                'name': card.FA_Name,
                'calories': card.Calories,
                'tags': ' '.join([f'<span class="tag-card">{tag}</span>' for tag in active_tags]),
                'operations': delete_button
            })

        response = {
            'draw': draw,
            'recordsTotal': records_total,
            'recordsFiltered': records_filtered,
            'data': data
        }

        return JsonResponse(response)

    except Exception as e:
        return JsonResponse({'error': str(e)}, status=500)


def food_list(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)

    user = request.user

    if not user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")
    temp = AppUser.objects.all()
    if not temp.exists():
        return generate_error_page(request, 'invalid request')

    queryset = Food.objects.all()

    # Searching
    search_query = request.GET.get('search', '')
    if search_query:
        queryset = queryset.filter(FA_Name__icontains=search_query)

    # Sorting
    sort_field = request.GET.get('sort_field', 'FA_Name')
    sort_order = request.GET.get('sort_order', 'asc')
    if sort_field:
        if sort_order == 'desc':
            sort_field = '-' + sort_field
        queryset = queryset.order_by(sort_field)

    paginator = Paginator(queryset, 25)
    page_number = request.GET.get('page')
    page_obj = paginator.get_page(page_number)
    return render(request, 'assistant/food_list.html', {
        'page_obj': page_obj,
        'request': request,  # so we can access request.GET in template
    })




def food_list(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')

    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)

    user = request.user

    if not user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")
    temp = AppUser.objects.all()
    if not temp.exists():
        return generate_error_page(request, 'invalid request')

    queryset = Food.objects.all()

    # Searching
    search_query = request.GET.get('search', '')
    if search_query:
        queryset = queryset.filter(FA_Name__icontains=search_query)

    # Sorting
    sort_field = request.GET.get('sort_field', 'FA_Name')
    sort_order = request.GET.get('sort_order', 'asc')
    if sort_field:
        if sort_order == 'desc':
            sort_field = '-' + sort_field
        queryset = queryset.order_by(sort_field)

    paginator = Paginator(queryset, 25)
    page_number = request.GET.get('page')
    page_obj = paginator.get_page(page_number)
    return render(request, 'assistant/food_list.html', {
        'page_obj': page_obj,
        'request': request,  # so we can access request.GET in template
    })


def food_create(request):
    if request.method == 'POST':
        form = FoodForm(request.POST)
        if form.is_valid():
            form.save()
            return redirect('panel-assistant-food-list')
    else:
        unique_id = datetime.now().strftime("%Y%m%d%H%M%S")
        form = FoodForm(initial={'Data_Base_Number': unique_id})

    return render(request, 'assistant/food_form.html', {'form': form, 'title': 'Add New Food'})


def food_edit(request, pk):
    food = get_object_or_404(Food, pk=pk)
    if request.method == 'POST':
        form = FoodForm(request.POST, instance=food)
        if form.is_valid():
            form.save()
            return redirect('panel-assistant-food-list')
    else:
        form = FoodForm(instance=food)
    return render(request, 'assistant/food_form.html', {'form': form, 'title': f'ویرایش {food.FA_Name}'})


def food_duplicate(request, pk):
    # Retrieve the original food item
    original_food = get_object_or_404(Food, pk=pk)

    if request.method == 'POST':
        form = FoodForm(request.POST)
        if form.is_valid():
            form.save()  # Save the new Food instance
            return redirect('panel-assistant-food-list')
    else:
        # Generate a unique Data_Base_Number
        unique_id = datetime.now().strftime("%Y%m%d%H%M%S")

        # Convert original_food to a dictionary
        initial_data = model_to_dict(original_food)

        # Update fields for duplication
        initial_data.update({
            'FA_Name': f"کپی - {original_food.FA_Name}",
            'Data_Base_Number': unique_id,
        })

        # Initialize the form with the updated data
        form = FoodForm(initial=initial_data)

    return render(request, 'assistant/food_form.html', {'form': form, 'title': 'کپی غذا'})


def food_delete(request, pk):
    food = get_object_or_404(Food, pk=pk)
    if request.method == 'POST':
        # If confirmed delete
        food.delete()
        return redirect('panel-assistant-food-list')
    return redirect('panel-assistant-food-list')


def activity_list(request):
    # 1) Authentication & Permission Check (similar to your food_list)
    if not request.user.is_authenticated:
        return redirect('panel-login')

    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)

    user = request.user
    if not user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")

    # 2) Fetch all Activity objects
    queryset = Activity.objects.all()

    # 3) Searching
    search_query = request.GET.get('search', '')
    if search_query:
        queryset = queryset.filter(name__icontains=search_query)

    # 4) Sorting
    sort_field = request.GET.get('sort_field', 'name')
    sort_order = request.GET.get('sort_order', 'asc')
    if sort_field:
        if sort_order == 'desc':
            sort_field = '-' + sort_field
        queryset = queryset.order_by(sort_field)

    # 5) Pagination
    paginator = Paginator(queryset, 25)  # show 25 items per page
    page_number = request.GET.get('page')
    page_obj = paginator.get_page(page_number)

    # 6) Render Template
    return render(request, 'assistant/activities_list.html', {
        'page_obj': page_obj,
        'request': request,  # so the template can access request.GET
    })


def activity_create(request):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    if not request.user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")

    if request.method == 'POST':
        form = ActivityForm(request.POST)
        if form.is_valid():
            form.save()
            return redirect('panel-assistant-activity-list')
    else:
        form = ActivityForm()

    return render(request, 'assistant/activity_form.html', {
        'form': form,
        'title': 'افزودن فعالیت جدید'
    })


def activity_edit(request, pk):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    if not request.user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")

    activity = get_object_or_404(Activity, pk=pk)

    if request.method == 'POST':
        form = ActivityForm(request.POST, instance=activity)
        if form.is_valid():
            form.save()
            return redirect('panel-assistant-activity-list')
    else:
        form = ActivityForm(instance=activity)

    return render(request, 'assistant/activity_form.html', {
        'form': form,
        'title': f'ویرایش {activity.name}'
    })


def activity_delete(request, pk):
    if not request.user.is_authenticated:
        return redirect('panel-login')
    if not request.user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")

    activity = get_object_or_404(Activity, pk=pk)
    if request.method == 'POST':
        activity.delete()
    return redirect('panel-assistant-activity-list')


@api_view(["GET"])
def chat_inbox(request):
    user = request.user
    if not user.is_authenticated:
        return redirect('panel-login')

    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)

    # Step 1. Get all patients (AppUser instances) and map by their django_user.id.
    patients = AppUser.objects.select_related('django_user').all()
    patient_map = {p.django_user.id: p for p in patients}

    # Step 2. Get all messages where either the sender or receiver is one of our patients.
    messages = (Message.objects
                .filter(Q(receiver=user, sender__id__in=patient_map.keys()) | Q(sender=user, receiver__id__in=patient_map.keys()))
                .select_related('sender', 'receiver')
                .order_by('-time'))

    # Step 3. Group messages by conversation partner (the patient)
    conversations = {}
    for msg in messages:
        # Skip if sender or receiver is None
        if not msg.sender or not msg.receiver:
            continue  # ignore this message, move to the next
        # Determine which party is the patient.
        if msg.sender.id in patient_map:
            patient_id = msg.sender.id
        elif msg.receiver.id in patient_map:
            patient_id = msg.receiver.id
        else:
            continue  # Skip messages not involving a patient (should not happen)

        if patient_id not in conversations:
            conversations[patient_id] = msg

    # Step 4. Build the enriched conversation list for display.
    enriched_conversations = []
    now = timezone.now()
    for patient_id, last_msg in conversations.items():
        partner = patient_map.get(patient_id)
        if not partner:
            continue

        is_from_patient = (last_msg.sender.id == partner.django_user.id)
        prefix = f"{partner.profile.first_name}: " if is_from_patient else f"مربی ({partner.profile.therapist}): "
        message_author_class = "from-patient" if is_from_patient else "from-doctor"

        # Count unread messages
        unread_count = Message.objects.filter(
            sender=partner.django_user,
            has_been_seen=False
        ).count()

        # Format the message time:
        message_time = last_msg.time
        if message_time.date() == now.date():
            time_display = "Today"
        elif message_time.date() == (now - timedelta(days=1)).date():
            time_display = "Yesterday"
        else:
            time_display = message_time.strftime("%d/%m/%Y")
        # Combine date and time (hour:minute)
        time_with_hour = f"{time_display} {message_time.strftime('%H:%M')}"

        enriched_conversations.append({
            'app_user': partner,  # the patient AppUser record
            'last_message': last_msg,  # the latest message in that conversation
            'prefix': prefix,  # e.g. "John:" or "You:"
            'message_author_class': message_author_class,
            'formatted_time': time_with_hour,
            'unread_count': unread_count,
        })

    # (Optional) Implement search if needed.
    search_query = request.GET.get('search', '')
    if search_query:
        enriched_conversations = [
            conv for conv in enriched_conversations
            if (search_query.lower() in conv['app_user'].django_user.get_full_name().lower()
                or search_query.lower() in conv['last_message'].text.lower())
        ]

    # (Optional) Implement pagination.
    from django.core.paginator import Paginator
    paginator = Paginator(enriched_conversations, 20)  # 20 conversations per page
    page_number = request.GET.get('page')
    page_obj = paginator.get_page(page_number)

    context = {
        'conversations': page_obj,
        'search_query': search_query,
    }
    return render(request, 'assistant/chat_inbox.html', context)


def recent_points_view(request):
    # 1) Ensure user is authenticated
    if not request.user.is_authenticated:
        return redirect('panel-login')

    # 2) Language handling (if needed)
    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)

    # 3) Identify the assistant (coach) object, or handle if not found
    try:
        assistant = Assistant.objects.get(django_user=request.user)
    except Assistant.DoesNotExist:
        # Or handle it gracefully (e.g., show error / redirect)
        return redirect('panel-login')

    # 4) Define the date range for the recent 7 days
    end_date = date.today()
    start_date = end_date - timedelta(days=6)

    # 5) Get the search term for filtering by patient name
    search_query = request.GET.get('search', '').strip()

    # 6) Base queryset of Points limited to the coach's patients (via user.profile.therapist)
    points_qs = Points.objects.select_related('user', 'user__profile').filter(
        date__gte=start_date,
        date__lte=end_date,
        user__profile__therapist=request.user  # Key line: only those patients of this coach
    )

    # 7) If a search term is provided, filter by first name/last name/phone
    if search_query:
        points_qs = points_qs.filter(
            Q(user__profile__first_name__icontains=search_query) |
            Q(user__profile__last_name__icontains=search_query) |
            Q(user__phone_number__icontains=search_query)
        )

    # 8) Order by date descending
    points_qs = points_qs.order_by('-date')

    # 9) Paginate the results
    paginator = Paginator(points_qs, 20)
    page_number = request.GET.get('page')
    try:
        points_page = paginator.page(page_number)
    except PageNotAnInteger:
        points_page = paginator.page(1)
    except EmptyPage:
        points_page = paginator.page(paginator.num_pages)

    # 10) Render the template
    context = {
        'points_page': points_page,
        'search_query': search_query,
        'start_date': start_date,
        'end_date': end_date,
    }
    return render(request, 'assistant/recent_points.html', context)


@api_view(["GET"])
def purchased_sessions(request):
    """Therapist view for assigned purchased sessions"""
    if not request.user.is_authenticated:
        return redirect('panel-login')
    user = request.user
    if not user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")

    assistant = Assistant.objects.get(django_user=user)
    
    # Get assigned purchased sessions for this therapist
    from payment.models import TimeSlotPayment
    assigned_sessions = TimeSlotPayment.objects.filter(
        assigned_therapist=assistant,
        is_paid=True,
        status='assigned'
    ).order_by('time_slot__start_time')

    # Get upcoming sessions (next 7 days)
    from datetime import datetime, timedelta
    today = datetime.now()
    next_week = today + timedelta(days=7)
    
    upcoming_sessions = assigned_sessions.filter(
        time_slot__start_time__gte=today,
        time_slot__start_time__lte=next_week
    )

    # Get completed sessions
    completed_sessions = assigned_sessions.filter(
        status='completed'
    )

    # Get statistics
    total_sessions = assigned_sessions.count()
    upcoming_count = upcoming_sessions.count()
    completed_count = completed_sessions.count()

    zipped_data, total_new_messages, free_time = new_messages(request)
    total_notifications = len(free_time)

    context = {
        'user': assistant,
        'assigned_sessions': assigned_sessions,
        'upcoming_sessions': upcoming_sessions,
        'completed_sessions': completed_sessions,
        'total_sessions': total_sessions,
        'upcoming_count': upcoming_count,
        'completed_count': completed_count,
        "message": zipped_data,
        "total_new_messages": total_new_messages,
        'free_time': free_time,
        'total_notifications': total_notifications,
    }

    return render(request, template_name='assistant/purchased_sessions.html', context=context)


@api_view(['POST'])
def complete_session(request, session_id):
    """Mark a session as completed"""
    if not request.user.is_authenticated:
        return redirect('panel-login')
    user = request.user
    if not user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "you have not access")

    try:
        from payment.models import TimeSlotPayment
        session = TimeSlotPayment.objects.filter(
            id=session_id,
            assigned_therapist__django_user=user,
            is_paid=True,
            status='assigned'
        ).first()

        if not session:
            return myResponse.Error("Session not found", Errors.DataNotFound.code)

        # Mark session as completed
        session.status = 'completed'
        session.save()

        from django.contrib import messages
        messages.success(request, f"جلسه با {session.user.profile.first_name} {session.user.profile.last_name} با موفقیت تکمیل شد")
        
        return redirect('panel-assistant-purchased-sessions')

    except Exception as e:
        return myResponse.Error(f"Error completing session: {str(e)}", Errors.InvalidArgument.code)


@api_view(['POST'])
def set_call_link(request):
    """Set call link for a session and send notification to user"""
    if not request.user.is_authenticated:
        return JsonResponse({'success': False, 'error': 'User not authenticated'}, status=401)
    
    user = request.user
    if not user.groups.filter(name='Assistant').exists():
        return JsonResponse({'success': False, 'error': 'Access denied'}, status=403)

    try:
        # Get data from POST request instead of body
        session_id = request.POST.get('session_id')
        call_link = request.POST.get('call_link')
        call_notes = request.POST.get('call_notes', '')

        if not session_id or not call_link:
            return JsonResponse({'success': False, 'error': 'Missing required fields'}, status=400)

        # Get the session
        from payment.models import TimeSlotPayment
        session = TimeSlotPayment.objects.filter(
            id=session_id,
            assigned_therapist__django_user=user,
            is_paid=True,
            status='assigned'
        ).first()

        if not session:
            return JsonResponse({'success': False, 'error': 'Session not found or not assigned to you'}, status=404)

        # Check if session date matches today or future
        from datetime import datetime
        today = datetime.now().date()
        session_date = session.time_slot.start_time.date()
        
        if session_date < today:
            return JsonResponse({'success': False, 'error': 'Cannot set call link for past sessions'}, status=400)

        # Update the timeslot with call link
        session.time_slot.appointment_link = call_link
        session.time_slot.save()

        # Send notification to the user using the same system as message sending
        try:
            from panel.PushNotification import PushNotification
            from panel.models import FirebaseNotification
            
            # Prepare notification message
            session_time = session.time_slot.start_time.strftime("%Y/%m/%d ساعت %H:%M")
            therapist_name = f"{session.assigned_therapist.first_name} {session.assigned_therapist.last_name}"
            
            notification_title = "لینک تماس تنظیم شد"
            notification_body = f"لینک تماس برای جلسه شما در تاریخ {session_time} توسط {therapist_name} تنظیم شد."
            
            if call_notes:
                notification_body += f"\nیادداشت: {call_notes}"
            
            # Send notification using the same system as message sending
            push_notification = PushNotification(session.user)
            notification_sent = push_notification.send_notification_to_user(
                session.user, 
                notification_title, 
                notification_body
            )
            
            if notification_sent:
                return JsonResponse({
                    'success': True, 
                    'message': 'Call link set successfully and notification sent'
                })
            else:
                return JsonResponse({
                    'success': True, 
                    'message': 'Call link set successfully but notification failed'
                })
            
        except Exception as e:
            # Even if notification fails, still save the call link
            print(f"Debug: Notification system failed: {e}")
            return JsonResponse({
                'success': True, 
                'message': 'Call link set successfully (notification system failed)'
            })

    except Exception as e:
        return JsonResponse({'success': False, 'error': str(e)}, status=500)


@api_view(["GET"])
def diet_management(request):
    """
    Diet Management Dashboard for Assistants
    
    Categorizes users into three groups based on their diet status:
    1. Recently received diet (within 7 days)
    2. Active diet (8-23 days remaining)
    3. Expiring soon (0-7 days remaining)
    """
    # Authentication & Authorization
    if not request.user.is_authenticated:
        return redirect('panel-login')
    
    language = request.session.get(settings.LANGUAGE_COOKIE_NAME)
    if language:
        translation.activate(language)
    
    user = request.user
    if not user.groups.filter(name='Assistant').exists():
        return generate_error_page(request, "You don't have access to this page")
    
    try:
        assistant = Assistant.objects.get(django_user=user)
    except Assistant.DoesNotExist:
        return generate_error_page(request, "Assistant profile not found")
    
    # Update last login time
    assistant.last_login = datetime.now()
    assistant.save()
    
    # Get all patients assigned to this assistant
    patients_list = AppUser.objects.filter(profile__therapist=user)
    
    # Current date for calculations
    today = datetime.now().date()
    
    # Initialize categorized lists
    recently_received = []  # Diet created within last 7 days
    active_diets = []  # Diet with 8-23 days remaining
    expiring_soon = []  # Diet with 0-7 days remaining
    no_diet = []  # Users without any diet
    
    # Process each patient
    for patient in patients_list:
        # Get the most recent diet for this patient
        latest_diet = Diet.objects.filter(user=patient).order_by('-from_date').first()
        
        if latest_diet:
            # Calculate days since diet started
            days_since_start = (today - latest_diet.from_date).days
            
            # Calculate days remaining (30 days total validity)
            days_remaining = 30 - days_since_start
            
            # Prepare patient data
            patient_data = {
                'patient': patient,
                'diet': latest_diet,
                'days_since_start': days_since_start,
                'days_remaining': max(0, days_remaining),
                'from_date': latest_diet.from_date,
                'jalali_date': latest_diet.jalali_starting_date,
            }
            
            # Check if patient has valid TherapistSelection credit
            has_valid_credit = False
            if patient.django_user:
                therapist_selection = TherapistSelection.objects.filter(user=patient.django_user).first()
                if therapist_selection and therapist_selection.credit > datetime.now():
                    has_valid_credit = True
            
            # Categorize based on days since start
            if days_since_start <= 7:
                # Recently received (within first 7 days)
                recently_received.append(patient_data)
            elif days_remaining <= 7 and days_remaining >= 0:
                # Expiring soon (last 7 days of validity) - only if has valid credit
                if has_valid_credit:
                    expiring_soon.append(patient_data)
            elif days_remaining > 7:
                # Active diet (more than 7 days remaining)
                active_diets.append(patient_data)
            else:
                # Diet expired (more than 30 days passed) - only if has valid credit
                if has_valid_credit:
                    expiring_soon.append(patient_data)
        else:
            # No diet found for this patient
            patient_data = {
                'patient': patient,
                'diet': None,
                'days_since_start': None,
                'days_remaining': None,
                'from_date': None,
                'jalali_date': None,
            }
            no_diet.append(patient_data)
    
    # Sort each category
    recently_received.sort(key=lambda x: x['days_since_start'])
    expiring_soon.sort(key=lambda x: x['days_remaining'])
    active_diets.sort(key=lambda x: x['days_remaining'])
    
    # Prepare context
    context = {
        'assistant': assistant,
        'recently_received': recently_received,
        'active_diets': active_diets,
        'expiring_soon': expiring_soon,
        'no_diet': no_diet,
        'recently_received_count': len(recently_received),
        'active_diets_count': len(active_diets),
        'expiring_soon_count': len(expiring_soon),
        'no_diet_count': len(no_diet),
        'total_patients': patients_list.count(),
    }
    
    return render(request, 'assistant/diet_management.html', context)


# ============================================
# Body Analysis Report Views
# ============================================

@api_view(["POST"])
def upload_body_analysis(request, patient_id):
    """
    Upload and analyze a body composition image using Gemini AI
    """
    if not request.user.is_authenticated:
        return JsonResponse({'success': False, 'error': 'احراز هویت نشده'}, status=401)
    
    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return JsonResponse({'success': False, 'error': 'دسترسی غیرمجاز'}, status=403)
    
    # Validate patient
    temp = AppUser.objects.filter(id=patient_id)
    if not temp.exists():
        return JsonResponse({'success': False, 'error': 'بیمار یافت نشد'}, status=404)
    
    patient = temp[0]
    assistant = Assistant.objects.get(django_user=django_user)
    
    # Check access to patient
    if patient.profile.therapist != assistant.django_user:
        return JsonResponse({'success': False, 'error': 'شما به این بیمار دسترسی ندارید'}, status=403)
    
    # Check if image file is provided
    if 'image' not in request.FILES:
        return JsonResponse({'success': False, 'error': 'لطفا یک تصویر آپلود کنید'}, status=400)
    
    image_file = request.FILES['image']
    
    # Validate file type
    allowed_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp']
    file_extension = image_file.name.split('.')[-1].lower()
    if file_extension not in allowed_extensions:
        return JsonResponse({
            'success': False, 
            'error': 'فرمت فایل نامعتبر است. فقط jpg, jpeg, png, gif, webp پذیرفته می‌شود'
        }, status=400)
    
    try:
        # Create report record
        report = BodyAnalysisReport.objects.create(
            patient=patient,
            created_by=django_user,
            image=image_file,
            status='pending'
        )
        
        # Get absolute path to the saved image
        image_path = report.image.path
        
        # Import and use Gemini service
        from panel.services.gemini_service import analyze_image
        
        print(f"Starting analysis for patient {patient_id}...")
        success, report_data, error_message = analyze_image(image_path)
        
        if success:
            report.report_data = report_data
            report.status = 'confirmed'
            report.save()
            
            print(f"Analysis successful for patient {patient_id}")
            
            return JsonResponse({
                'success': True,
                'report_id': report.id,
                'report_data': report_data,
                'message': 'تحلیل با موفقیت انجام شد'
            })
        else:
            report.error_message = error_message
            report.save()
            
            print(f"Analysis failed for patient {patient_id}: {error_message}")
            
            return JsonResponse({
                'success': False,
                'error': error_message or 'خطا در تحلیل تصویر'
            }, status=500)
            
    except Exception as e:
        print(f"Error in upload_body_analysis: {str(e)}")
        return JsonResponse({
            'success': False,
            'error': f'خطای سرور: {str(e)}'
        }, status=500)


@api_view(["POST"])
def share_body_analysis_report(request, report_id):
    """
    Confirm and share the body analysis report with the patient
    """
    if not request.user.is_authenticated:
        return JsonResponse({'success': False, 'error': 'احراز هویت نشده'}, status=401)
    
    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return JsonResponse({'success': False, 'error': 'دسترسی غیرمجاز'}, status=403)
    
    # Get the report
    try:
        report = BodyAnalysisReport.objects.get(id=report_id)
    except BodyAnalysisReport.DoesNotExist:
        return JsonResponse({'success': False, 'error': 'گزارش یافت نشد'}, status=404)
    
    # Check access
    assistant = Assistant.objects.get(django_user=django_user)
    if report.patient.profile.therapist != assistant.django_user:
        return JsonResponse({'success': False, 'error': 'شما به این گزارش دسترسی ندارید'}, status=403)
    
    try:
        # Mark as shared
        report.mark_as_shared()
        
        # Generate share token if not exists
        share_url = report.get_public_share_url()
        
        # Create a medical note with the report
        report_summary = "گزارش آنالیز ترکیب بدنی"
        if report.report_data and 'report_title' in report.report_data:
            report_summary = report.report_data['report_title']
        
        note_text = f"""
        <h3>{report_summary}</h3>
        <p>گزارش تحلیل ترکیب بدنی در تاریخ {report.created_at.strftime('%Y/%m/%d - %H:%M')} ثبت شد.</p>
        <p><a href="/panel/assistant/body-analysis-report/{report.id}/" target="_blank">مشاهده گزارش کامل</a></p>
        <p><strong>لینک اشتراک‌گذاری عمومی:</strong> <a href="{share_url}" target="_blank">{share_url}</a></p>
        """
        
        medical_note = MedicalNote.objects.create(
            user=django_user,
            patient=report.patient,
            text=note_text,
            writer_name=str(assistant)
        )
        
        # TODO: Send notification to patient via Firebase
        # This can be implemented later to notify the patient that a new report is available
        
        print(f"Report {report_id} shared with patient {report.patient.id}")
        print(f"Public share URL: {share_url}")
        
        return JsonResponse({
            'success': True,
            'message': 'گزارش با موفقیت با بیمار به اشتراک گذاشته شد',
            'share_url': share_url
        })
        
    except Exception as e:
        print(f"Error in share_body_analysis_report: {str(e)}")
        return JsonResponse({
            'success': False,
            'error': f'خطا در اشتراک‌گذاری گزارش: {str(e)}'
        }, status=500)


@api_view(["GET"])
def view_body_analysis_report(request, report_id):
    """
    View a body analysis report (for both assistant and patient)
    """
    if not request.user.is_authenticated:
        return redirect('panel-login')
    
    try:
        report = BodyAnalysisReport.objects.get(id=report_id)
    except BodyAnalysisReport.DoesNotExist:
        return generate_error_page(request, 'گزارش یافت نشد')
    
    django_user = request.user
    
    # Check access - either the assistant who created it or the patient
    is_assistant = django_user.groups.filter(name='Assistant').exists()
    is_patient = False
    
    if is_assistant:
        assistant = Assistant.objects.get(django_user=django_user)
        if report.patient.profile.therapist != assistant.django_user:
            return generate_error_page(request, 'شما به این گزارش دسترسی ندارید')
    else:
        # Check if user is the patient
        try:
            app_user = AppUser.objects.get(django_user=django_user)
            if app_user.id != report.patient.id:
                return generate_error_page(request, 'شما به این گزارش دسترسی ندارید')
            is_patient = True
        except AppUser.DoesNotExist:
            return generate_error_page(request, 'شما به این گزارش دسترسی ندارید')
    
    context = {
        'report': report,
        'patient': report.patient,
        'is_patient': is_patient,
        'is_assistant': is_assistant,
    }
    
    return render(request, 'assistant/body_analysis_report.html', context)


@api_view(["POST"])
def edit_body_analysis_report(request, report_id):
    """
    Edit/update a body analysis report (only for assistants)
    """
    if not request.user.is_authenticated:
        return JsonResponse({'success': False, 'error': 'احراز هویت نشده'}, status=401)
    
    django_user = request.user
    if not django_user.groups.filter(name='Assistant').exists():
        return JsonResponse({'success': False, 'error': 'دسترسی غیرمجاز'}, status=403)
    
    # Get the report
    try:
        report = BodyAnalysisReport.objects.get(id=report_id)
    except BodyAnalysisReport.DoesNotExist:
        return JsonResponse({'success': False, 'error': 'گزارش یافت نشد'}, status=404)
    
    # Check access
    assistant = Assistant.objects.get(django_user=django_user)
    if report.patient.profile.therapist != assistant.django_user:
        return JsonResponse({'success': False, 'error': 'شما به این گزارش دسترسی ندارید'}, status=403)
    
    try:
        import json
        data = json.loads(request.body)
        
        # Validate required fields
        if 'report_title' not in data:
            return JsonResponse({'success': False, 'error': 'عنوان گزارش الزامی است'}, status=400)
        
        if 'sections' not in data or not isinstance(data['sections'], list):
            return JsonResponse({'success': False, 'error': 'بخش‌های گزارش نامعتبر است'}, status=400)
        
        # Update report data
        updated_report_data = {
            'report_title': data['report_title'],
            'sections': data['sections']
        }
        
        report.report_data = updated_report_data
        report.save()
        
        print(f"Report {report_id} updated by assistant {assistant.id}")
        
        return JsonResponse({
            'success': True,
            'message': 'گزارش با موفقیت به‌روزرسانی شد',
            'report_data': updated_report_data
        })
        
    except json.JSONDecodeError:
        return JsonResponse({'success': False, 'error': 'داده‌های ارسالی نامعتبر است'}, status=400)
    except Exception as e:
        print(f"Error in edit_body_analysis_report: {str(e)}")
        return JsonResponse({
            'success': False,
            'error': f'خطا در به‌روزرسانی گزارش: {str(e)}'
        }, status=500)


@api_view(["GET"])
def public_bia_report(request, share_token):
    """
    Public view for BIA report - accessible without authentication via share token
    """
    try:
        report = BodyAnalysisReport.objects.get(share_token=share_token)
    except BodyAnalysisReport.DoesNotExist:
        return render(request, 'assistant/bia_public_report.html', {
            'patient_name': 'نامشخص',
            'report_date': '-',
            'report_time': '-',
            'report_image': None,
            'report_data': None,
            'report_data_json': '{}',
            'error_message': 'گزارش یافت نشد یا لینک اشتراک‌گذاری نامعتبر است',
            'shared_date': None,
        })
    
    # Prepare patient name
    patient_name = f"{report.patient.profile.first_name} {report.patient.profile.last_name}".strip()
    if not patient_name:
        patient_name = "بیمار"
    
    # Format dates
    import jdatetime
    from django.utils import timezone
    
    # Convert created_at to Jalali
    if report.created_at:
        jalali_date = jdatetime.datetime.fromgregorian(datetime=report.created_at)
        report_date = jalali_date.strftime('%Y/%m/%d')
        report_time = report.created_at.strftime('%H:%M')
    else:
        report_date = '-'
        report_time = '-'
    
    # Convert shared_at to Jalali
    shared_date = None
    if report.shared_at:
        jalali_shared = jdatetime.datetime.fromgregorian(datetime=report.shared_at)
        shared_date = jalali_shared.strftime('%Y/%m/%d - %H:%M')
    
    # Get image URL
    report_image = None
    if report.image:
        report_image = report.image.url
    
    # Prepare report data as JSON string for JavaScript
    import json
    report_data_json = json.dumps(report.report_data if report.report_data else {}, ensure_ascii=False)
    
    context = {
        'patient_name': patient_name,
        'report_date': report_date,
        'report_time': report_time,
        'report_image': report_image,
        'report_data': report.report_data,
        'report_data_json': report_data_json,
        'error_message': report.error_message,
        'shared_date': shared_date,
    }
    
    return render(request, 'assistant/bia_public_report.html', context)
