#!/usr/bin/env python
"""
Script to migrate hospitals from hardcoded HOSPITAL_INFO to database Hospital model
Run this after creating the Hospital model and migrations
"""

import os
import django
import sys

# Setup Django environment
os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'ziluck_project.settings')
django.setup()

from django.contrib.auth.models import User as DjangoUser, Group
from panel.models import Hospital, Manager
from ziluck_project.hospital_config import HOSPITAL_INFO

def migrate_hospitals():
    """Migrate hospitals from HOSPITAL_INFO to Hospital model"""
    
    print("=" * 60)
    print("Hospital Migration Script")
    print("Migrating from hardcoded HOSPITAL_INFO to database")
    print("=" * 60)
    print()
    
    # Get or create Hospital group
    hospital_group, created = Group.objects.get_or_create(name='Hospital')
    if created:
        print("✓ Hospital group created")
    else:
        print("✓ Hospital group already exists")
    
    # Get a manager for created_by field
    manager = Manager.objects.first()
    if not manager:
        print("⚠ Warning: No manager found. Hospital will be created without created_by")
    
    migrated_count = 0
    skipped_count = 0
    
    for api_key, info in HOSPITAL_INFO.items():
        if api_key == "default":
            print(f"\n⊘ Skipping 'default' entry")
            skipped_count += 1
            continue
        
        hospital_code = info.get("code", "UNKNOWN")
        hospital_name = info.get("name", "Unknown Hospital")
        
        print(f"\n→ Processing: {hospital_name} ({hospital_code})")
        
        # Check if hospital already exists
        if Hospital.objects.filter(hospital_code=hospital_code).exists():
            print(f"  ⊘ Hospital with code '{hospital_code}' already exists, skipping")
            skipped_count += 1
            continue
        
        if Hospital.objects.filter(api_key=api_key).exists():
            print(f"  ⊘ Hospital with API key already exists, skipping")
            skipped_count += 1
            continue
        
        # Create Django user for hospital
        username = f"hospital_{hospital_code.lower()}"
        if DjangoUser.objects.filter(username=username).exists():
            print(f"  ⚠ Username '{username}' already exists, using modified username")
            username = f"hospital_{hospital_code.lower()}_{api_key[:8]}"
        
        try:
            # Create Django user
            django_user = DjangoUser.objects.create_user(
                username=username,
                password=f"Hospital{hospital_code}2024!",  # You should change this
                email=info.get("contact_email", f"{username}@hospital.com")
            )
            django_user.groups.add(hospital_group)
            
            # Create Hospital
            hospital = Hospital.objects.create(
                name=hospital_name,
                hospital_code=hospital_code,
                api_key=api_key,
                api_secret=f"HS_{api_key}",  # Create a secret based on key
                contact_email=info.get("contact_email", f"{username}@hospital.com"),
                contact_phone=info.get("phone", "N/A"),
                contact_person=info.get("contact_person", ""),
                address=info.get("location", ""),
                description=f"Migrated from HOSPITAL_INFO. Integration type: {info.get('integration_type', 'N/A')}",
                django_user=django_user,
                created_by=manager,
                is_active=info.get("enabled", True)
            )
            
            print(f"  ✓ Hospital created successfully!")
            print(f"    - ID: {hospital.id}")
            print(f"    - Username: {username}")
            print(f"    - Password: Hospital{hospital_code}2024! (CHANGE THIS!)")
            print(f"    - API Key: {api_key}")
            
            # Create promotion group and discount
            try:
                ziluck_group, discount = hospital.create_promotion_group_and_discount()
                promotion_code = f"{hospital_code}_PROMOTION"
                print(f"  ✓ Created promotion code: {promotion_code}")
                print(f"    - ZiLuckGroup: {ziluck_group.name}")
                print(f"    - Discount Code: {discount.code}")
            except Exception as promo_error:
                print(f"  ⚠ Warning: Could not create promotion group/discount: {str(promo_error)}")
            
            migrated_count += 1
            
        except Exception as e:
            print(f"  ✗ Error creating hospital: {str(e)}")
            if 'django_user' in locals():
                django_user.delete()
    
    print()
    print("=" * 60)
    print("Migration Summary")
    print("=" * 60)
    print(f"✓ Successfully migrated: {migrated_count} hospitals")
    print(f"⊘ Skipped: {skipped_count} hospitals")
    print()
    
    if migrated_count > 0:
        print("⚠ IMPORTANT SECURITY NOTES:")
        print("1. Change all default passwords immediately!")
        print("2. Review hospital credentials in the admin panel")
        print("3. Update any external systems with new credentials if needed")
        print("4. Consider regenerating API secrets for security")
        print()
        print("Next steps:")
        print("1. Login to manager panel: /panel/manager/hospitals/")
        print("2. Review and update hospital information")
        print("3. Change default passwords")
        print("4. Test hospital login for each migrated hospital")
    else:
        print("No hospitals were migrated. This could mean:")
        print("1. All hospitals already exist in database")
        print("2. HOSPITAL_INFO is empty or only has 'default' entry")
        print("3. There were errors during migration")
    
    print()

if __name__ == '__main__':
    try:
        migrate_hospitals()
    except Exception as e:
        print(f"\n✗ Fatal error during migration: {str(e)}")
        import traceback
        traceback.print_exc()
        sys.exit(1)

