# Server Requirements Guide for YOLO Food Detection System

This guide provides comprehensive information on measuring and planning server resources for your YOLO food detection system.

## 📊 How to Measure Current Usage

### Method 1: API Endpoints (Real-time)

#### Quick Resource Check
```bash
curl -X GET http://your-domain.com/api/v1/system/resources/
```

#### Comprehensive Analysis
```bash
curl -X GET http://your-domain.com/api/v1/system/analysis/
```

#### Model Memory Impact
```bash
curl -X GET http://your-domain.com/api/v1/system/model-impact/
```

#### Performance Benchmark
```bash
curl -X POST http://your-domain.com/api/v1/system/benchmark/ \
  -H 'Content-Type: application/json' \
  -d '{"num_tests": 20}'
```

### Method 2: Command Line Scripts

#### Complete Resource Analysis
```bash
source /Users/mohammadamintebianian/PycharmProjects/ziluck/venv/bin/activate
python resource_analysis_script.py --benchmark-tests 20 --save-report
```

#### Real-time Monitoring
```bash
python monitor_resources.py --interval 5
```

### Method 3: System Commands (Basic)

```bash
# Memory usage
free -h
cat /proc/meminfo

# Disk usage
df -h
du -sh /path/to/project

# CPU usage
top
htop
```

## 🧠 Memory (RAM) Requirements

### Baseline Requirements

| Component | Memory Usage |
|-----------|--------------|
| **Base System** | 512 MB |
| **Django Framework** | 200-300 MB |
| **YOLO Model** | 50-200 MB |
| **Per Detection Request** | 5-20 MB |
| **Safety Buffer** | 512 MB |

### Detailed Memory Analysis

#### Model Memory Impact
- **Small YOLO model**: 50-100 MB
- **Medium YOLO model**: 100-150 MB
- **Large YOLO model**: 150-200 MB
- **Your current model**: Run `python resource_analysis_script.py` to measure

#### Concurrent Request Memory
```
Memory per concurrent request = 5-20 MB
For 10 concurrent users = 50-200 MB
For 50 concurrent users = 250-1000 MB
```

### Memory Recommendations by Scale

| Scale | Concurrent Users | Recommended RAM |
|-------|------------------|-----------------|
| **Small** | 1-10 users | 2-4 GB |
| **Medium** | 10-50 users | 4-8 GB |
| **Large** | 50-200 users | 8-16 GB |
| **Enterprise** | 200+ users | 16+ GB |

## 💾 Storage Requirements

### Storage Components

| Component | Storage Usage |
|-----------|---------------|
| **Base Django Project** | 100-500 MB |
| **YOLO Model Files** | 50-200 MB each |
| **Model Versions** | 250-1000 MB (5 versions) |
| **User Images** | 1-5 MB per image |
| **Logs** | 100-500 MB |
| **System/OS** | 10-20 GB |

### Storage Calculations

#### Model Storage
```
Single model size: 50-200 MB
Keep 5 versions: 250-1000 MB
Backup models: +50% = 375-1500 MB
```

#### User Images Storage
```
Average image size: 2 MB
Images per day: 100-1000
Monthly storage: 6-60 GB
Annual storage: 72-720 GB
```

### Storage Recommendations

| Usage Level | Daily Images | Recommended Storage |
|-------------|--------------|-------------------|
| **Light** | <100 | 50 GB |
| **Medium** | 100-1000 | 100-200 GB |
| **Heavy** | 1000-10000 | 500 GB - 1 TB |
| **Enterprise** | >10000 | 1+ TB |

## ⚡ CPU Requirements

### CPU Performance Factors

1. **Model Size**: Larger models need more CPU
2. **Image Resolution**: Higher resolution = more processing
3. **Concurrent Requests**: More users = more CPU cores needed
4. **Detection Speed**: Target response time affects CPU needs

### CPU Benchmarking

Run this to measure your CPU performance:
```bash
python resource_analysis_script.py --benchmark-tests 50
```

### CPU Recommendations

| Performance Target | Concurrent Users | Recommended CPU |
|-------------------|------------------|-----------------|
| **Basic** | 1-5 users | 2-4 cores |
| **Standard** | 5-20 users | 4-8 cores |
| **High Performance** | 20-50 users | 8-16 cores |
| **Enterprise** | 50+ users | 16+ cores |

### Detection Time Targets

- **Acceptable**: 1-3 seconds per detection
- **Good**: 0.5-1 seconds per detection
- **Excellent**: <0.5 seconds per detection

## 🔧 How to Measure for Upgrades

### Step 1: Run Complete Analysis

```bash
# This will give you comprehensive measurements
python resource_analysis_script.py --benchmark-tests 20 --save-report
```

### Step 2: Monitor During Peak Usage

```bash
# Run this during your busiest hours
python monitor_resources.py --interval 10
```

### Step 3: Load Testing

```bash
# Simulate high load
for i in {1..50}; do
  curl -X POST http://your-domain.com/api/v1/system/benchmark/ \
    -H 'Content-Type: application/json' \
    -d '{"num_tests": 5}' &
done
wait
```

### Step 4: Analyze Results

The analysis script will tell you:
- ✅ Current resources are sufficient
- ⚠️ Upgrade recommended
- 🚨 Upgrade required immediately

## 📈 Scaling Scenarios

### Scenario 1: Current Usage
```bash
# Measure current baseline
python resource_analysis_script.py
```

### Scenario 2: 2x Traffic Growth
```bash
# Run benchmark with double the load
python resource_analysis_script.py --benchmark-tests 40
```

### Scenario 3: Peak Load Simulation
```bash
# Simulate peak usage
curl -X POST http://your-domain.com/api/v1/system/benchmark/ \
  -d '{"num_tests": 100}'
```

## 🎯 Specific Measurement Commands

### Memory Measurement
```bash
# Current memory usage
curl -X GET http://localhost:8000/api/v1/system/model-impact/

# Response shows:
# - Baseline memory: XXX MB
# - After model loading: XXX MB  
# - Model impact: XXX MB
```

### Performance Measurement
```bash
# Detection performance
curl -X POST http://localhost:8000/api/v1/system/benchmark/ \
  -H 'Content-Type: application/json' \
  -d '{"num_tests": 50}'

# Response shows:
# - Average detection time
# - Memory per detection
# - CPU usage
```

### Storage Measurement
```bash
# Storage analysis
curl -X GET http://localhost:8000/api/v1/system/resources/

# Response shows:
# - Project size
# - Models size  
# - Available disk space
```

## 🚀 Upgrade Decision Matrix

### When to Upgrade Memory
- ❌ System memory usage > 85%
- ❌ Frequent out-of-memory errors
- ❌ Slow response times
- ❌ Process gets killed by OS

### When to Upgrade Storage
- ❌ Disk usage > 90%
- ❌ Cannot store new model versions
- ❌ No space for user images
- ❌ Log rotation fails

### When to Upgrade CPU
- ❌ Detection time > 3 seconds
- ❌ CPU usage constantly > 90%
- ❌ Request timeouts
- ❌ Cannot handle concurrent users

## 📋 Recommended Server Specifications

### Minimum Requirements (Development)
- **RAM**: 4 GB
- **Storage**: 50 GB SSD
- **CPU**: 2-4 cores
- **Network**: 10 Mbps

### Production Requirements (Small Scale)
- **RAM**: 8 GB
- **Storage**: 100 GB SSD
- **CPU**: 4-8 cores
- **Network**: 100 Mbps

### Production Requirements (Medium Scale)
- **RAM**: 16 GB
- **Storage**: 500 GB SSD
- **CPU**: 8-16 cores
- **Network**: 1 Gbps

### Production Requirements (Large Scale)
- **RAM**: 32+ GB
- **Storage**: 1+ TB SSD
- **CPU**: 16+ cores
- **Network**: 1+ Gbps

## 🔍 Monitoring & Alerts

### Set Up Continuous Monitoring
```bash
# Run in background
nohup python monitor_resources.py --interval 60 > monitor.log 2>&1 &
```

### Alert Thresholds
- **Memory**: Alert at 80%, critical at 90%
- **Disk**: Alert at 85%, critical at 95%
- **CPU**: Alert at 80%, critical at 95%

### Health Check Endpoints
```bash
# Add to your monitoring system
curl -X GET http://your-domain.com/api/v1/system/resources/
curl -X GET http://your-domain.com/api/v1/model/status/
```

## 💡 Optimization Tips

1. **Use SSD storage** for better I/O performance
2. **Enable swap** if memory is limited (not recommended for production)
3. **Use Redis caching** to reduce memory pressure
4. **Implement request queuing** for high load
5. **Consider GPU acceleration** for faster detection
6. **Optimize image sizes** before detection
7. **Clean up old model versions** regularly
8. **Monitor and log** resource usage patterns

## 🔧 Automated Resource Management

### Auto-cleanup Script
```bash
# Clean old models (keep only 5 latest)
find detection_models/ -name "*.pt" -type f -printf '%T@ %p\n' | sort -n | head -n -5 | cut -d' ' -f2- | xargs rm -f
```

### Resource Alerts
```bash
# Add to crontab for hourly checks
0 * * * * curl -s http://localhost:8000/api/v1/system/resources/ | python check_thresholds.py
```

This comprehensive guide will help you measure your exact resource needs and plan server upgrades accordingly. Use the provided tools and scripts to get precise measurements for your specific usage patterns.


