# YOLO Model Update Guide

This guide explains how to update your food detection model without restarting the Django server.

## Overview

The food detection system now supports **hot-swapping** models, meaning you can update the model while your Django application is running, without any downtime.

## When Model Setup Code Triggers

The model loading code in `APIs/apps.py` triggers during:

1. **Django server startup** (`python manage.py runserver`)
2. **WSGI application startup** (production deployment)
3. **Django shell startup** (`python manage.py shell`)
4. **Application reload** (when Django detects code changes)

The model loads **once** during startup and stays in memory for all subsequent requests.

## Methods to Update the Model

### Method 1: API Endpoints (Recommended)

#### 1. Upload New Model File
```bash
curl -X POST \
  http://your-domain.com/api/v1/model/update/ \
  -H 'Content-Type: multipart/form-data' \
  -F 'model_file=@/path/to/new_model.pt'
```

#### 2. Switch to Existing Model
```bash
curl -X POST \
  http://your-domain.com/api/v1/model/switch/ \
  -H 'Content-Type: application/json' \
  -d '{"model_filename": "model_v2.pt"}'
```

#### 3. Check Model Status
```bash
curl -X GET http://your-domain.com/api/v1/model/status/
```

#### 4. List Available Models
```bash
curl -X GET http://your-domain.com/api/v1/model/list/
```

#### 5. Reload Current Model
```bash
curl -X POST http://your-domain.com/api/v1/model/reload/
```

#### 6. Validate Model Before Update
```bash
curl -X POST \
  http://your-domain.com/api/v1/model/validate/ \
  -H 'Content-Type: multipart/form-data' \
  -F 'model_file=@/path/to/model_to_validate.pt'
```

### Method 2: Command Line Script

#### Update Model
```bash
source /Users/mohammadamintebianian/PycharmProjects/ziluck/venv/bin/activate
python update_model_script.py update --model-path /path/to/new_model.pt
```

#### Check Status
```bash
python update_model_script.py status
```

#### List Available Models
```bash
python update_model_script.py list
```

#### Reload Current Model
```bash
python update_model_script.py reload
```

### Method 3: Manual File Replacement + Reload

1. **Replace the model file:**
   ```bash
   cp /path/to/new_model.pt /Users/mohammadamintebianian/PycharmProjects/ziluck/detection_models/best.pt
   ```

2. **Reload via API:**
   ```bash
   curl -X POST http://your-domain.com/api/v1/model/reload/
   ```

## API Response Examples

### Successful Model Update
```json
{
  "status": "success",
  "message": "Model updated successfully",
  "data": {
    "success": true,
    "model_path": "/path/to/model/best.pt",
    "load_time": "2025-01-19T10:30:45.123456",
    "backup_path": "/path/to/backup/best_backup_1642598445.pt",
    "message": "Model updated successfully"
  }
}
```

### Model Status Response
```json
{
  "status": "success",
  "message": "Model status retrieved",
  "data": {
    "model_loaded": true,
    "model_path": "/path/to/model/best.pt",
    "load_time": "2025-01-19T10:30:45.123456",
    "model_available": true
  }
}
```

### Available Models Response
```json
{
  "status": "success",
  "message": "Available models retrieved",
  "data": {
    "models": [
      {
        "filename": "best.pt",
        "path": "/path/to/model/best.pt",
        "size": 52428800,
        "modified": "2025-01-19T10:30:00.000000"
      },
      {
        "filename": "model_v2.pt",
        "path": "/path/to/model/model_v2.pt",
        "size": 54525952,
        "modified": "2025-01-18T15:20:00.000000"
      }
    ]
  }
}
```

## Automated Model Updates

### Using CI/CD Pipeline

```bash
#!/bin/bash
# deploy_model.sh

# Download new model
wget https://your-model-storage.com/latest_model.pt -O /tmp/new_model.pt

# Update model via API
curl -X POST \
  http://your-domain.com/api/v1/model/update/ \
  -H 'Content-Type: multipart/form-data' \
  -F 'model_file=@/tmp/new_model.pt'

# Clean up
rm /tmp/new_model.pt
```

### Using Cron Job

```bash
# Add to crontab for daily model updates
0 2 * * * /path/to/your/deploy_model.sh >> /var/log/model_update.log 2>&1
```

## Safety Features

1. **Validation**: Models are validated before loading
2. **Backup**: Old models are automatically backed up
3. **Graceful Fallback**: If update fails, the old model continues working
4. **Thread Safety**: Model updates are thread-safe
5. **Error Handling**: Comprehensive error handling and logging

## Monitoring Model Updates

### Check Logs
```bash
# Django logs will show model loading messages
tail -f /path/to/django.log | grep -i "model"
```

### Health Check Endpoint
```bash
# Use model status endpoint for health checks
curl -X GET http://your-domain.com/api/v1/model/status/
```

## Troubleshooting

### Model Not Loading
1. Check file permissions
2. Verify model file format (.pt)
3. Check available memory
4. Review Django logs

### Update Fails
1. Validate model file first
2. Check disk space
3. Ensure model is compatible with current YOLO version

### Performance Issues
1. Monitor memory usage during updates
2. Consider updating during low-traffic periods
3. Use validation endpoint before production updates

## Best Practices

1. **Always validate** models before updating in production
2. **Test updates** in staging environment first
3. **Monitor performance** after model updates
4. **Keep backups** of working models
5. **Document model versions** and changes
6. **Update during low-traffic** periods when possible

## Integration with Your Workflow

The model update system integrates seamlessly with your existing food detection API. When users upload food images via `insertFoodWithPic`, they automatically get results from the latest model version without any changes to your client applications.


