# Hospital User Upgrade Tracking Solution

## Overview

This document describes the implementation of Solution 1 (Hospital User Association Table) for tracking which hospital upgraded which user's subscription plan. This solution provides a reliable way to associate premium users with the hospital that paid for them.

## What Was Implemented

### 1. New Database Model: `HospitalUserUpgrade`

The `HospitalUserUpgrade` model tracks every hospital-initiated user upgrade with the following fields:

- **user**: Foreign key to the upgraded user
- **hospital_name**: Name of the hospital (e.g., "Nikan Hospital")
- **hospital_code**: Unique code identifying the hospital (e.g., "NIKAN")
- **plan**: Plan number that was purchased (1-8)
- **credit_days_added**: Number of credit days added to user account
- **upgrade_date**: When the upgrade was processed
- **payment_reference**: Hospital's internal payment reference
- **notes**: Additional notes about the upgrade

### 2. Enhanced Hospital Configuration

Updated `ziluck_project/hospital_config.py` to support:
- Multiple hospitals with unique API keys
- Hospital-specific information (name, code, contact details)
- Better validation and tracking capabilities

### 3. Modified API Endpoint

Updated `successful_payment_from_hospital` API to:
- Create `HospitalUserUpgrade` records automatically
- Include hospital tracking information in responses
- Provide better audit trail

### 4. New Statistics API Endpoint

Added `get_hospital_upgrade_statistics` endpoint for:
- Querying hospital upgrade statistics
- Generating reports on hospital performance
- Tracking user upgrade history

### 5. Django Admin Integration

Added comprehensive admin interface for:
- Viewing all hospital upgrades
- Filtering by hospital, plan, and date
- Managing upgrade records
- Generating reports

## How It Works

### 1. Hospital Payment Flow

```
Hospital System → API Call → successful_payment_from_hospital
                                    ↓
                            Validate API Key
                                    ↓
                            Process User Upgrade
                                    ↓
                            Create HospitalUserUpgrade Record
                                    ↓
                            Return Success Response
```

### 2. Data Tracking

Every time a hospital calls the API:
1. **User validation**: Verify user exists by phone number
2. **Therapist assignment**: Assign or reuse previous therapist
3. **Credit addition**: Add subscription credit based on plan
4. **Tracking record**: Create `HospitalUserUpgrade` record
5. **Welcome message**: Send personalized welcome message
6. **Notifications**: Send Telegram and email notifications

### 3. Database Relationships

```
User (AppUser) ←→ HospitalUserUpgrade ←→ Hospital Info
     ↓                    ↓
Profile              Plan Details
     ↓                    ↓
TherapistSelection   Credit Days
```

## API Usage Examples

### 1. Hospital Payment Simulation

```bash
POST /payment/hospital/successful-payment/
Content-Type: application/x-www-form-urlencoded

hospital_api_key=Z9xd3NikanhospitalQ8zYpR7G5wK2uHvM1R&phone_number=09123456789&plan=1
```

**Response:**
```json
{
    "status": "success",
    "message": "Payment simulation completed successfully",
    "data": {
        "user_id": 123,
        "phone_number": "09123456789",
        "plan": 1,
        "credit_days_added": 30,
        "hospital_name": "Nikan Hospital",
        "hospital_code": "NIKAN",
        "upgrade_tracking_id": "HOSPITAL_NIKAN_20250816_122340",
        "actions_performed": [
            "therapist_selection_created",
            "credit_added",
            "hospital_upgrade_tracked",
            "welcome_message_sent",
            "notifications_sent"
        ]
    }
}
```

### 2. Hospital Statistics Query

```bash
GET /payment/hospital/statistics/?hospital_code=NIKAN&days=30&format=summary
```

**Response:**
```json
{
    "status": "success",
    "data": {
        "hospital_code": "NIKAN",
        "hospital_name": "Nikan Hospital",
        "statistics": {
            "total_upgrades": 15,
            "total_users": 12,
            "plans_distribution": [
                {"plan": 1, "count": 8},
                {"plan": 2, "count": 4},
                {"plan": 3, "count": 3}
            ],
            "total_credit_days": 450,
            "period_days": 30
        }
    }
}
```

## Database Queries

### 1. Get All Upgrades for a Hospital

```python
from payment.models import HospitalUserUpgrade

# Get all upgrades for Nikan Hospital
upgrades = HospitalUserUpgrade.objects.filter(hospital_code='NIKAN')

# Get active upgrades (within credit period)
active_upgrades = HospitalUserUpgrade.objects.filter(
    hospital_code='NIKAN',
    upgrade_date__gte=datetime.now() - timedelta(days=365)
)
```

### 2. Get User's Hospital History

```python
# Get complete hospital upgrade history for a user
user_history = HospitalUserUpgrade.get_user_hospital_history(user)

# Check if user has active hospital upgrade
active_upgrade = user_history.filter(
    upgrade_date__gte=datetime.now() - timedelta(days=365)
).first()
```

### 3. Generate Hospital Reports

```python
# Get statistics for all hospitals
hospitals_summary = HospitalUserUpgrade.get_hospitals_summary()

# Get detailed statistics for specific hospital
nikan_stats = HospitalUserUpgrade.get_hospital_statistics('NIKAN', days=90)
```

## Benefits of This Solution

### 1. **Reliable Tracking**
- Every hospital upgrade is automatically recorded
- No manual intervention required
- Complete audit trail maintained

### 2. **Easy Reporting**
- Simple queries for hospital performance
- User upgrade history readily available
- Statistical analysis capabilities

### 3. **Scalable Design**
- Supports multiple hospitals
- Efficient database indexing
- Minimal performance impact

### 4. **Admin Management**
- Django admin interface for easy management
- Filtering and search capabilities
- Export functionality available

### 5. **API Integration**
- RESTful endpoints for external systems
- Hospital-specific authentication
- Comprehensive response data

## Future Enhancements

### 1. **Additional Tracking Fields**
- Hospital payment gateway information
- Payment amount and currency
- Hospital contact person details

### 2. **Advanced Analytics**
- Revenue tracking by hospital
- User retention analysis
- Plan popularity by hospital

### 3. **Integration Features**
- Hospital dashboard access
- Automated reporting
- Webhook notifications

### 4. **Multi-Hospital Support**
- Hospital-specific configurations
- Custom upgrade workflows
- Hospital branding options

## Maintenance and Monitoring

### 1. **Regular Checks**
- Monitor upgrade record creation
- Verify hospital API key usage
- Check for failed upgrade attempts

### 2. **Data Cleanup**
- Archive old upgrade records
- Clean up test data
- Optimize database queries

### 3. **Performance Monitoring**
- Database query performance
- API response times
- Memory usage patterns

## Conclusion

This solution successfully addresses the requirement to track which hospital upgraded which user. By implementing a dedicated tracking model with comprehensive API support, we now have:

- **Complete visibility** into hospital upgrade activities
- **Reliable data** for business intelligence and reporting
- **Scalable architecture** for future growth
- **Easy management** through Django admin interface
- **API access** for external system integration

The implementation follows Django best practices and provides a solid foundation for hospital integration tracking. 