# Hospital Panel Setup Guide

This guide provides instructions for setting up the Hospital user role and panel in your Django project.

## Overview

The Hospital Panel allows hospital/clinic accounts to:
- View their API credentials
- See users assigned to their hospital (filtered by `hospital_code`)
- Monitor subscription upgrades (expiring soon, recently upgraded, active subscriptions)

## Installation Steps

### 1. Create Database Migrations

Run the following commands to create and apply database migrations:

```bash
source /Users/mohammadamintebianian/PycharmProjects/ziluck/venv/bin/activate
cd /Users/mohammadamintebianian/PycharmProjects/ziluck
python manage.py makemigrations
python manage.py migrate
```

### 2. Create Hospital Group

You need to create a "Hospital" group in Django. Run this in Django shell:

```bash
python manage.py shell
```

Then execute:

```python
from django.contrib.auth.models import Group

# Create Hospital group
hospital_group, created = Group.objects.get_or_create(name='Hospital')
if created:
    print("Hospital group created successfully!")
else:
    print("Hospital group already exists.")
```

### 3. Create Your First Hospital (via Manager Panel)

1. Log in as a Manager
2. Navigate to "Hospital Management" in the sidebar
3. Click "Add Hospital"
4. Fill in the required information:
   - Hospital Name
   - Hospital Code (unique identifier)
   - Contact Email
   - Contact Phone
   - Username (for hospital login)
   - Password (for hospital login)
5. Click "Create Hospital"
6. API credentials will be generated automatically

### 4. Assign Users to Hospital

To assign users to a hospital, you need to set the `hospital_code` field in their profile:

```python
from APIs.models import User, Profile

# Get user
user = User.objects.get(phone_number="PHONE_NUMBER")

# Set hospital code
user.profile.hospital_code = "HOSPITAL_CODE"
user.profile.save()
```

Or via Django Admin:
1. Go to Django Admin
2. Find the user's Profile
3. Set the `hospital_code` field
4. Save

## Features

### Hospital Panel Pages

#### 1. Dashboard
- Overview of hospital statistics
- Total users, upgrades, and active subscriptions
- Recent upgrades list
- Quick links to other pages

**URL:** `/panel/hospital/`

#### 2. Credentials Page
- Hospital name and code
- API Key and API Secret (with copy functionality)
- Contact information
- Regenerate credentials button

**URL:** `/panel/hospital/credentials/`

#### 3. Users Page
- List of all users with matching `hospital_code`
- User details: name, phone, age, gender, assigned doctor
- Search functionality
- Pagination support

**URL:** `/panel/hospital/users/`

#### 4. Upgrades Page
- Subscription monitoring grouped by:
  - **Expiring Soon**: Users with < 7 days remaining
  - **Recently Upgraded**: Users upgraded within last 7 days
  - **Active Subscriptions**: All other active users
- Each section shows: User name, phone, upgrade date, days remaining, plan type

**URL:** `/panel/hospital/upgrades/`

### Manager Panel Features

Managers can:
- View all hospitals
- Create new hospitals
- Edit hospital information
- Delete hospitals
- Regenerate API credentials
- See user count per hospital

**URL:** `/panel/manager/hospitals/`

## Database Models

### Hospital Model

Located in: `panel/models.py`

Fields:
- `name`: Hospital name
- `hospital_code`: Unique hospital identifier
- `api_key`: Auto-generated API key
- `api_secret`: Auto-generated API secret
- `contact_email`: Contact email
- `contact_phone`: Contact phone
- `contact_person`: Optional contact person name
- `address`: Optional address
- `description`: Optional description
- `is_active`: Boolean status
- `django_user`: Linked Django user for login
- `created_at`: Creation timestamp
- `updated_at`: Last update timestamp
- `created_by`: Manager who created the hospital

### HospitalUserUpgrade Model

Located in: `payment/models.py` (already exists)

Fields:
- `user`: Foreign key to AppUser
- `hospital_name`: Name of hospital
- `hospital_code`: Hospital identifier
- `plan`: Plan number (1-8)
- `credit_days_added`: Number of days added
- `upgrade_date`: When upgrade was processed
- `payment_reference`: Optional payment reference
- `discount_code`: Optional discount code used
- `paid_price`: Final price paid
- `notes`: Optional notes

## Security & Access Control

### Permissions

- **Hospital users** can only see:
  - Their own hospital information
  - Users with matching `hospital_code`
  - Upgrades with matching `hospital_code`

- **Managers** can:
  - View all hospitals
  - Create/Edit/Delete hospitals
  - Regenerate API credentials
  - See all hospital data

### Login Flow

Hospital users are automatically redirected to their dashboard upon login:

1. User logs in with hospital credentials
2. System checks for "Hospital" group membership
3. User is redirected to `/panel/hospital/` (dashboard)

### Decorators

The `@hospital_required` decorator ensures:
- User is authenticated
- User belongs to "Hospital" group
- Hospital account exists
- Hospital object is attached to `request.hospital`

## API Integration

Hospitals receive API credentials for programmatic access:

- **API Key**: Format: `HK_{random_string}`
- **API Secret**: Format: `HS_{random_string}`

These can be regenerated at any time by:
- Hospital user: via Credentials page
- Manager: via Hospital Management page

## Testing

### Create a Test Hospital

```bash
python manage.py shell
```

```python
from django.contrib.auth.models import User as DjangoUser, Group
from panel.models import Hospital, Manager
import secrets

# Get manager (or use superuser)
manager = Manager.objects.first()

# Create Django user
django_user = DjangoUser.objects.create_user(
    username='test_hospital',
    password='test123',
    email='test@hospital.com'
)

# Add to Hospital group
hospital_group = Group.objects.get(name='Hospital')
django_user.groups.add(hospital_group)

# Create hospital
hospital = Hospital.objects.create(
    name='Test Hospital',
    hospital_code='TEST001',
    api_key=f"HK_{secrets.token_urlsafe(32)}",
    api_secret=f"HS_{secrets.token_urlsafe(48)}",
    contact_email='test@hospital.com',
    contact_phone='123-456-7890',
    django_user=django_user,
    created_by=manager,
    is_active=True
)

print(f"Hospital created: {hospital.name}")
print(f"API Key: {hospital.api_key}")
print(f"Login with: username=test_hospital, password=test123")
```

### Test Login

1. Go to `/panel/login/`
2. Enter credentials: `test_hospital` / `test123`
3. You should be redirected to `/panel/hospital/`

## Troubleshooting

### Hospital Not Found Error

**Problem:** "Hospital account not found" after login

**Solution:**
- Ensure Hospital group exists
- Ensure Hospital object is created and linked to Django user
- Check that `django_user` field is set on Hospital model

### No Users Showing

**Problem:** Users page is empty

**Solution:**
- Ensure users have `hospital_code` set in their Profile
- Hospital code must match exactly
- Check that `hospital_code` field is not null

### Migration Issues

**Problem:** Errors during migration

**Solution:**
```bash
python manage.py makemigrations panel
python manage.py migrate panel
```

## Files Modified/Created

### New Files
- `panel/hospital.py` - Hospital panel views
- `panel/templates/hospital/base.html` - Base template
- `panel/templates/hospital/dashboard.html` - Dashboard template
- `panel/templates/hospital/credentials.html` - Credentials template
- `panel/templates/hospital/users.html` - Users list template
- `panel/templates/hospital/upgrades.html` - Upgrades monitoring template
- `panel/templates/manager/hospitals.html` - Manager hospital list
- `panel/templates/manager/create_hospital.html` - Create hospital form
- `panel/templates/manager/edit_hospital.html` - Edit hospital form

### Modified Files
- `panel/models.py` - Added Hospital model
- `panel/urls.py` - Added hospital and manager URLs
- `panel/view.py` - Updated login flow for hospital users
- `panel/manager.py` - Added hospital management views

## Next Steps

1. Run migrations
2. Create Hospital group
3. Create your first hospital via Manager panel
4. Assign users to hospitals
5. Test hospital login
6. Monitor subscriptions

## Support

For questions or issues, refer to the inline code documentation or contact the development team.


