# Hospital Panel - Quick Start Guide

## 🚀 Get Started in 5 Minutes

### Step 1: Run Migrations (1 min)
```bash
cd /Users/mohammadamintebianian/PycharmProjects/ziluck
source venv/bin/activate
python manage.py makemigrations
python manage.py migrate
```

### Step 2: Create Hospital Group (30 seconds)
```bash
python create_hospital_group.py
```

**Or manually:**
```bash
python manage.py shell
```
```python
from django.contrib.auth.models import Group
Group.objects.get_or_create(name='Hospital')
exit()
```

### Step 2.5: Migrate Existing Hospitals (Optional - 1 min)
If you have existing hospitals in `HOSPITAL_INFO` config, migrate them:

```bash
python migrate_hospitals_to_db.py
```

This will:
- Read hospitals from `ziluck_project/hospital_config.py`
- Create database records for each
- Create login credentials
- Preserve existing API keys for backward compatibility

**⚠️ IMPORTANT:** Default passwords are generated. Change them immediately!

### Step 2.6: Create Promotion Groups for Existing Hospitals (Optional - 30 sec)
If you migrated hospitals and want to create promotion groups for them:

```bash
python create_hospital_promotion_groups.py
```

This creates:
- ZiLuckGroup named `[HOSPITAL_CODE]_PROMOTION`
- Discount code with same name
- Automatically linked together

### Step 3: Create Your First Hospital (2 min)

**Option A: Via Manager Panel (Recommended)**
1. Login as Manager: `/panel/login/`
2. Go to: **Hospital Management** (in sidebar)
3. Click: **"Add Hospital"**
4. Fill in:
   - Hospital Name: e.g., "City General Hospital"
   - Hospital Code: e.g., "CGH001" (must be unique)
   - Contact Email: e.g., "admin@cityhospital.com"
   - Contact Phone: e.g., "123-456-7890"
   - Username: e.g., "city_hospital"
   - Password: e.g., "SecurePass123!"
5. Click: **"Create Hospital"**
6. ✅ Done! API credentials generated automatically

**Option B: Via Django Shell**
```python
python manage.py shell
```
```python
from django.contrib.auth.models import User as DjangoUser, Group
from panel.models import Hospital, Manager
import secrets

# Get a manager
manager = Manager.objects.first()

# Create Django user
django_user = DjangoUser.objects.create_user(
    username='test_hospital',
    password='test123',
    email='test@hospital.com'
)

# Add to Hospital group
hospital_group = Group.objects.get(name='Hospital')
django_user.groups.add(hospital_group)

# Create hospital
hospital = Hospital.objects.create(
    name='Test Hospital',
    hospital_code='TEST001',
    api_key=f"HK_{secrets.token_urlsafe(32)}",
    api_secret=f"HS_{secrets.token_urlsafe(48)}",
    contact_email='test@hospital.com',
    contact_phone='123-456-7890',
    django_user=django_user,
    created_by=manager,
    is_active=True
)

print(f"✅ Hospital Created!")
print(f"Login: test_hospital / test123")
print(f"API Key: {hospital.api_key}")
```

### Step 4: Assign Users to Hospital (1 min)

```bash
python manage.py shell
```
```python
from APIs.models import User

# Get user by phone number
user = User.objects.get(phone_number="PHONE_NUMBER_HERE")

# Assign to hospital
user.profile.hospital_code = "CGH001"  # Use your hospital code
user.profile.save()

print(f"✅ User {user.profile.first_name} assigned to hospital")
```

**Or via Django Admin:**
1. Go to: `/admin/`
2. Find: **Profiles**
3. Search user and edit
4. Set: `hospital_code = "CGH001"`
5. Save

### Step 5: Login and Test (30 seconds)

1. Go to: `/panel/login/`
2. Enter hospital credentials
3. ✅ You're in! You should see:
   - Dashboard with statistics
   - Credentials page with API keys
   - Users page (filtered by your hospital)
   - Upgrades page (subscription monitoring)

## 📱 Hospital Panel URLs

| Page | URL | Description |
|------|-----|-------------|
| Dashboard | `/panel/hospital/` | Overview & statistics |
| Credentials | `/panel/hospital/credentials/` | API keys & info |
| Users | `/panel/hospital/users/` | Your hospital's users |
| Upgrades | `/panel/hospital/upgrades/` | Subscription monitoring |

## 👨‍💼 Manager Panel URLs

| Page | URL | Description |
|------|-----|-------------|
| Hospital List | `/panel/manager/hospitals/` | All hospitals |
| Create Hospital | `/panel/manager/hospitals/create/` | Add new hospital |
| Edit Hospital | `/panel/manager/hospitals/<id>/edit/` | Edit details |

## 🎯 Common Tasks

### Add Multiple Users to Hospital
```python
from APIs.models import User

# Get users by criteria
users = User.objects.filter(phone_number__startswith="091")

# Assign all to hospital
for user in users:
    user.profile.hospital_code = "CGH001"
    user.profile.save()

print(f"✅ {users.count()} users assigned")
```

### Check Hospital's User Count
```python
from panel.models import Hospital

hospital = Hospital.objects.get(hospital_code="CGH001")
print(f"Users: {hospital.get_users_count()}")
print(f"Upgrades: {hospital.get_active_upgrades_count()}")
```

### Regenerate API Credentials
**Via Panel:**
- Login as Manager
- Go to Hospital Management
- Click dropdown → "Regenerate Credentials"

**Via Code:**
```python
from panel.models import Hospital

hospital = Hospital.objects.get(hospital_code="CGH001")
api_key, api_secret = hospital.regenerate_api_credentials()
print(f"New API Key: {api_key}")
print(f"New API Secret: {api_secret}")
```

## 🐛 Troubleshooting

### "Hospital account not found"
**Problem:** Hospital group exists but Hospital object is missing

**Solution:**
```python
# Check if Hospital exists
from panel.models import Hospital
Hospital.objects.filter(django_user=user)

# If empty, create Hospital object
```

### Users Not Showing
**Problem:** `hospital_code` not set on user profiles

**Solution:**
```python
# Check user's hospital code
user.profile.hospital_code
# If None or different, update it
user.profile.hospital_code = "YOUR_CODE"
user.profile.save()
```

### Login Redirects to Wrong Page
**Problem:** User in multiple groups

**Solution:** Remove user from other groups, keep only "Hospital"
```python
user.groups.clear()
user.groups.add(Group.objects.get(name='Hospital'))
```

## 📚 More Information

- **Full Guide:** `HOSPITAL_SETUP_GUIDE.md`
- **Implementation:** `HOSPITAL_IMPLEMENTATION_SUMMARY.md`
- **Code:** Check inline comments in `panel/hospital.py`

## ✅ Verification Checklist

Before going live, verify:
- [ ] Migrations applied successfully
- [ ] Hospital group created
- [ ] At least one hospital created
- [ ] Test login works
- [ ] Dashboard loads
- [ ] Users filtered correctly
- [ ] Upgrades display properly
- [ ] Manager can create hospitals
- [ ] API credentials visible
- [ ] Regenerate works

## 🎉 You're All Set!

Your Hospital Panel is ready to use. If you encounter any issues:
1. Check the troubleshooting section above
2. Review the full setup guide
3. Check Django logs
4. Contact support

**Happy Hospital Managing! 🏥**

