# Hospital Promotion Groups & Discount Codes

## 🎯 Overview

When a hospital is created in the system, the following are automatically generated:

1. **ZiLuckGroup**: A user group named `[HOSPITAL_CODE]_PROMOTION`
2. **Discount Code**: A discount code with the same name `[HOSPITAL_CODE]_PROMOTION`

These are automatically linked together and can be used for hospital-specific promotions and user tracking.

## 📋 What Gets Created

### Example: Nikan Hospital (Code: NIKAN)

When Nikan Hospital is created, the system automatically generates:

#### 1. ZiLuckGroup
- **Name**: `NIKAN_PROMOTION`
- **Type**: Company group (`is_company=True`)
- **Visibility**: Private (`is_public=False`)
- **Group ID**: `nikan_promotion`
- **Creator/Admin**: Hospital's Django user

#### 2. Discount Code
- **Code**: `NIKAN_PROMOTION`
- **Linked to**: ZiLuckGroup `NIKAN_PROMOTION`
- **Expiry**: 10 years from creation
- **Discount**: 0% (can be modified by manager)
- **Usage Limit**: 999,999 (essentially unlimited)

## 🔄 When Are They Created?

Promotion groups and discount codes are created automatically in these scenarios:

### 1. New Hospital Creation via Manager Panel
When a manager creates a hospital through `/panel/manager/hospitals/create/`:
```python
hospital = Hospital.objects.create(...)
hospital.create_promotion_group_and_discount()
```

### 2. New Hospital Creation via Django Admin
When a hospital is created in Django admin at `/admin/panel/hospital/add/`:
```python
# Automatically triggered by HospitalAdmin.save_model()
```

### 3. Hospital Migration Script
When running `migrate_hospitals_to_db.py`:
```bash
python migrate_hospitals_to_db.py
```

### 4. Manual Creation for Existing Hospitals
For hospitals created before this feature:
```bash
python create_hospital_promotion_groups.py
```

## 🎨 How It Works

### Automatic Creation Flow

```
Manager Creates Hospital
        ↓
Hospital.objects.create()
        ↓
create_promotion_group_and_discount()
        ↓
    ┌───────────────┴───────────────┐
    ↓                               ↓
Create ZiLuckGroup            Create Discount
[CODE]_PROMOTION             [CODE]_PROMOTION
    ↓                               ↓
Link them together ←─────────────────┘
```

### Code Structure

**In `panel/models.py`:**
```python
class Hospital(models.Model):
    # ... fields ...
    
    def create_promotion_group_and_discount(self):
        """Auto-create promotion group and discount"""
        promotion_code = f"{self.hospital_code}_PROMOTION"
        
        # Create ZiLuckGroup
        ziluck_group = ZiLuckGroup.objects.create(
            name=promotion_code,
            is_public=False,
            is_company=True,
            ...
        )
        
        # Create Discount linked to group
        discount = Discount.objects.create(
            code=promotion_code,
            ziluck_group=ziluck_group,
            ...
        )
        
        return ziluck_group, discount
```

## 💼 Use Cases

### 1. Hospital Promotion Campaigns

**Scenario**: Nikan Hospital wants to offer 20% discount

**Steps**:
1. Manager goes to Django admin → Discounts
2. Find `NIKAN_PROMOTION`
3. Set `percentage = 20`
4. Save

**Result**: All users using code `NIKAN_PROMOTION` get 20% off

### 2. User Registration Tracking

**Scenario**: Track which users came from Nikan Hospital

**API Call**:
```bash
curl -X POST /payment/get_pay_link/ \
  -d "discount_code=NIKAN_PROMOTION" \
  -d "plan=1" \
  ...
```

**Result**: 
- User gets discount (if percentage > 0)
- User is added to `NIKAN_PROMOTION` group
- Can be tracked in hospital panel

### 3. Hospital-Specific Features

**Scenario**: Enable special features for Nikan users

**Code**:
```python
# Check if user is in Nikan promotion group
user = User.objects.get(...)
if user.django_user.groups.filter(name='NIKAN_PROMOTION').exists():
    # Enable special features
    enable_nikan_features(user)
```

### 4. Bulk Operations

**Scenario**: Send notification to all Nikan promotion users

**Code**:
```python
nikan_group = ZiLuckGroup.objects.get(name='NIKAN_PROMOTION')
users = nikan_group.user_set.all()

for user in users:
    send_notification(user, "Special Nikan offer!")
```

## 🔧 Management

### View Promotion Code

**Hospital Panel:**
1. Login as hospital user
2. Go to "Credentials" page
3. See "Promotion Code & Discount" section

**Manager Panel:**
1. Login as manager
2. Go to "Hospital Management"
3. Click hospital → View details

### Modify Discount Percentage

**Django Admin:**
1. Go to `/admin/payment/discount/`
2. Search for `[HOSPITAL_CODE]_PROMOTION`
3. Edit `percentage` field
4. Save

**Example**:
```python
from payment.models import Discount

discount = Discount.objects.get(code='NIKAN_PROMOTION')
discount.percentage = 15.0  # 15% off
discount.save()
```

### Check Group Members

```python
from APIs.models import ZiLuckGroup

group = ZiLuckGroup.objects.get(name='NIKAN_PROMOTION')
members = group.user_set.all()
print(f"Members: {members.count()}")
```

### Retrieve Hospital's Promotion Info

```python
from panel.models import Hospital

hospital = Hospital.objects.get(hospital_code='NIKAN')

# Get promotion group
group = hospital.get_promotion_group()
print(f"Group: {group.name}")

# Get discount
discount = hospital.get_promotion_discount()
print(f"Discount: {discount.code} - {discount.percentage}%")
```

## 📊 Database Schema

### ZiLuckGroup
| Field | Value |
|-------|-------|
| name | `[HOSPITAL_CODE]_PROMOTION` |
| is_public | False |
| is_company | True |
| group_id | `[hospital_code]_promotion` |
| creator | Hospital django_user |
| admin | Hospital django_user |

### Discount
| Field | Value |
|-------|-------|
| code | `[HOSPITAL_CODE]_PROMOTION` |
| ziluck_group | Link to ZiLuckGroup |
| expire_date | +10 years |
| percentage | 0 (configurable) |
| number_of_discounts | 999999 |

## 🛠️ Troubleshooting

### Issue: Promotion Group Not Created

**Check**:
```python
hospital = Hospital.objects.get(hospital_code='NIKAN')
group = hospital.get_promotion_group()
print(group)  # Should show group object
```

**Fix**:
```bash
python create_hospital_promotion_groups.py
```

### Issue: Discount Code Not Working

**Verify discount exists**:
```python
from payment.models import Discount

discount = Discount.objects.filter(code='NIKAN_PROMOTION').first()
print(f"Discount: {discount}")
print(f"Valid: {discount.is_valid()}")
print(f"Percentage: {discount.percentage}%")
```

**Common issues**:
1. Discount expired → Update `expire_date`
2. No uses left → Update `number_of_discounts`
3. Wrong code → Check spelling

### Issue: Users Not Added to Group

**Check discount-group link**:
```python
discount = Discount.objects.get(code='NIKAN_PROMOTION')
print(f"Linked group: {discount.ziluck_group}")
```

If `ziluck_group` is None, recreate:
```bash
python create_hospital_promotion_groups.py
```

## 🔄 Migration

### For Existing Hospitals

If you have hospitals created before this feature:

```bash
# Create promotion groups for all existing hospitals
python create_hospital_promotion_groups.py
```

**Output**:
```
Processing: Nikan Hospital (NIKAN)
  ✓ Created promotion code: NIKAN_PROMOTION
    - ZiLuckGroup: NIKAN_PROMOTION (ID: 123)
    - Discount Code: NIKAN_PROMOTION (ID: 456)
```

## 📈 Analytics

### Track Promotion Usage

```python
from payment.models import Discount, ZarinPalPayment

discount = Discount.objects.get(code='NIKAN_PROMOTION')

# Payments using this discount
payments = ZarinPalPayment.objects.filter(discount=discount)
print(f"Total payments: {payments.count()}")

# Successful payments
successful = payments.filter(is_payed=True)
print(f"Successful: {successful.count()}")

# Total revenue
revenue = sum([p.amount_after_discount for p in successful])
print(f"Revenue: {revenue}")
```

### Group Growth Over Time

```python
from APIs.models import ZiLuckGroup
from datetime import datetime, timedelta

group = ZiLuckGroup.objects.get(name='NIKAN_PROMOTION')
users = group.user_set.all()

# Users joined in last 30 days
thirty_days_ago = datetime.now() - timedelta(days=30)
recent = users.filter(date_joined__gte=thirty_days_ago)
print(f"New members (30d): {recent.count()}")
```

## 🔐 Security

### Access Control

- **Manager**: Full access to all promotion groups
- **Hospital**: Read-only access to their own promotion code
- **Users**: Can use codes but cannot see group details

### Best Practices

1. **Regular Audits**: Review discount percentages monthly
2. **Expiry Management**: Set realistic expiry dates
3. **Usage Limits**: Adjust `number_of_discounts` for limited campaigns
4. **Monitoring**: Track unusual discount usage patterns

## 📝 API Integration

### Payment with Promotion Code

```bash
curl -X POST /payment/get_pay_link/ \
  -H "Content-Type: application/json" \
  -d '{
    "token": "user_token",
    "plan": "1",
    "discount_code": "NIKAN_PROMOTION"
  }'
```

**Response**:
```json
{
  "status": "ok",
  "data": {
    "link": "payment_gateway_url",
    "main_amount": 100000,
    "amount_after_discount": 80000
  }
}
```

**Side Effects**:
- User charged discounted amount
- User added to `NIKAN_PROMOTION` group
- Hospital can see user in their panel

## 🎉 Summary

✅ **Auto-Created**: Promotion groups created automatically
✅ **Integrated**: Linked with discount system
✅ **Flexible**: Discount percentage configurable
✅ **Trackable**: Users auto-added to groups
✅ **Visible**: Shown in hospital credentials page
✅ **Manageable**: Full admin interface support

**Promotion codes make hospital integrations more powerful and trackable!**


