# Hospital Admin & Promotion Groups Update

## 🎉 New Features Added

### 1. Django Admin Integration ✅
Hospital model is now fully integrated into Django admin panel with a rich interface.

### 2. Automatic Promotion Groups ✅
When a hospital is created, a ZiLuckGroup and Discount code are automatically generated.

---

## 🔧 Feature 1: Django Admin Integration

### Access Django Admin
URL: `/admin/panel/hospital/`

### What You Can Do

#### List View
- View all hospitals in a searchable table
- Filter by: Active status, Created date, Created by
- Search by: Name, Code, Email, Phone, API Key
- Sort by any column
- See user count for each hospital

#### Add/Edit Hospital
- **Basic Information**: Name, Code, Status
- **API Credentials**: Key and Secret (collapsed section)
- **Contact Info**: Email, Phone, Person, Address
- **Additional Info**: Description
- **System Info**: Django user, Creator, Timestamps

#### Features
- ✅ Read-only fields: Created date, Updated date, Hospital code, API key
- ✅ Custom admin actions
- ✅ Auto-save triggers promotion group creation
- ✅ User count display in list
- ✅ Inline editing support

### How to Access

1. Login to Django admin: `/admin/`
2. Navigate to **Panel** section
3. Click **Hospitals**
4. Browse, search, filter, or create hospitals

### Admin Interface

```
/admin/panel/hospital/
├── List View
│   ├── Columns: Name, Code, Active, Email, Phone, Users, Created
│   ├── Filters: Active, Created Date, Created By
│   └── Search: Name, Code, Email, Phone, API Key
├── Add Hospital
│   ├── Basic Info
│   ├── API Credentials (collapsed)
│   ├── Contact Info
│   ├── Additional Info (collapsed)
│   └── System Info (collapsed)
└── Change Hospital
    └── Same as Add, with history
```

---

## 🎁 Feature 2: Automatic Promotion Groups

### What Gets Created

When a hospital with code `NIKAN` is created:

#### 1. ZiLuckGroup
- **Name**: `NIKAN_PROMOTION`
- **Type**: Company group
- **Visibility**: Private
- **Creator**: Hospital's user

#### 2. Discount Code
- **Code**: `NIKAN_PROMOTION`
- **Linked**: To ZiLuckGroup above
- **Expiry**: 10 years
- **Discount**: 0% (configurable)
- **Limit**: 999,999 uses

### When It Happens

Promotion groups are created automatically:

1. ✅ **Manager Panel**: When creating hospital via `/panel/manager/hospitals/create/`
2. ✅ **Django Admin**: When adding hospital via `/admin/panel/hospital/add/`
3. ✅ **Migration**: When running `migrate_hospitals_to_db.py`
4. ✅ **Manual**: When running `create_hospital_promotion_groups.py`

### Code Format

```python
promotion_code = f"{hospital_code}_PROMOTION"
```

**Examples:**
- NIKAN → `NIKAN_PROMOTION`
- CITY_GENERAL → `CITY_GENERAL_PROMOTION`
- TEST001 → `TEST001_PROMOTION`

### Where to See It

#### Hospital Credentials Page
Hospital users can see their promotion code at:
`/panel/hospital/credentials/`

**Display:**
```
┌─────────────────────────────────────┐
│ Promotion Code & Discount           │
├─────────────────────────────────────┤
│ Promotion Code: NIKAN_PROMOTION     │
│ [Copy button]                        │
│                                      │
│ ZiLuck Group: NIKAN_PROMOTION        │
│ (Auto-assignment for users)          │
└─────────────────────────────────────┘
```

#### Manager Panel
Managers see success message when creating hospital:
```
Hospital 'Nikan Hospital' created successfully!
Promotion code 'NIKAN_PROMOTION' and group created.
```

### How to Use

#### Use Case 1: Discount Campaign
```python
# Set 20% discount for Nikan users
from payment.models import Discount

discount = Discount.objects.get(code='NIKAN_PROMOTION')
discount.percentage = 20.0
discount.save()
```

Now when users use code `NIKAN_PROMOTION`, they get 20% off!

#### Use Case 2: User Tracking
```python
# Find all users from Nikan promotion
from APIs.models import ZiLuckGroup

group = ZiLuckGroup.objects.get(name='NIKAN_PROMOTION')
users = group.user_set.all()
print(f"Nikan users: {users.count()}")
```

#### Use Case 3: Bulk Operations
```python
# Send notification to all Nikan users
from panel.PushNotification import PushNotification

group = ZiLuckGroup.objects.get(name='NIKAN_PROMOTION')
for django_user in group.user_set.all():
    app_user = User.objects.get(django_user=django_user)
    PushNotification().send_push_notification(
        app_user,
        "Special Nikan Offer!",
        "Get 30% off this month"
    )
```

---

## 📁 Files Modified/Created

### Modified Files
```
panel/admin.py              - Added HospitalAdmin
panel/models.py             - Added promotion methods
panel/manager.py            - Updated create_hospital
migrate_hospitals_to_db.py  - Added promotion creation
panel/templates/hospital/credentials.html - Added promotion display
```

### New Files
```
create_hospital_promotion_groups.py      - Utility script
HOSPITAL_PROMOTION_GROUPS.md            - Full documentation
HOSPITAL_ADMIN_PROMOTION_UPDATE.md      - This file
```

---

## 🚀 Quick Start

### For New Hospitals
Nothing extra needed! Just create a hospital normally:

**Manager Panel:**
1. Go to `/panel/manager/hospitals/`
2. Click "Add Hospital"
3. Fill in details
4. Save
5. ✨ Promotion group auto-created!

**Django Admin:**
1. Go to `/admin/panel/hospital/`
2. Click "Add Hospital"
3. Fill in details
4. Save
5. ✨ Promotion group auto-created!

### For Existing Hospitals
Run this once to create promotion groups:

```bash
python create_hospital_promotion_groups.py
```

**Output:**
```
Processing: Nikan Hospital (NIKAN)
  ✓ Created promotion code: NIKAN_PROMOTION
    - ZiLuckGroup: NIKAN_PROMOTION (ID: 123)
    - Discount Code: NIKAN_PROMOTION (ID: 456)

Summary:
✓ Created: 3 promotion groups
⊘ Skipped: 0 (already exist)
✗ Errors: 0
```

---

## 🧪 Testing

### Test 1: Admin Access
```bash
# Login to Django admin
# Navigate to /admin/panel/hospital/
# Verify hospitals are listed
# Try searching, filtering
```

### Test 2: Create Hospital
```bash
# Create new hospital via admin or manager panel
# Verify promotion group created
# Check in /admin/apis/ziluckgroup/
# Check in /admin/payment/discount/
```

### Test 3: View Promotion Code
```bash
# Login as hospital user
# Go to /panel/hospital/credentials/
# Verify promotion code displayed
# Try copy button
```

### Test 4: Use Discount Code
```bash
# As app user, purchase subscription
# Use discount code: NIKAN_PROMOTION
# Verify discount applied
# Verify user added to group
```

### Test 5: Migration Script
```bash
python create_hospital_promotion_groups.py
# Verify no errors
# Check promotion groups created
```

---

## 📊 Database Structure

### Hospital → Promotion Group Relationship

```
Hospital (NIKAN)
    ↓ creates
ZiLuckGroup (NIKAN_PROMOTION)
    ↓ linked to
Discount (NIKAN_PROMOTION)
    ↓ used by
Users → Added to ZiLuckGroup
    ↓ visible in
Hospital Panel → Users Page
```

### Tables Updated

**panel_hospital**
- No new fields
- New methods: `create_promotion_group_and_discount()`, `get_promotion_group()`, `get_promotion_discount()`

**apis_ziluckgroup** (extends auth_group)
- New records with name `[CODE]_PROMOTION`

**payment_discount**
- New records with code `[CODE]_PROMOTION`
- Linked to ZiLuckGroup via `ziluck_group_id`

---

## 🔐 Permissions

### Django Admin Access
- **Superuser**: Full access to all hospitals
- **Staff**: Can view hospitals (if granted permission)
- **Manager**: Via manager panel, not admin
- **Hospital**: Cannot access admin

### Promotion Group Access
- **Manager**: Full access via Django admin
- **Hospital**: Read-only via credentials page
- **Users**: Cannot see groups directly

---

## 📈 Use Cases

### Marketing Campaigns
```
1. Create hospital: "City General"
2. Auto-created code: CITY_GENERAL_PROMOTION
3. Set discount: 25% off
4. Hospital shares code with patients
5. Track usage in admin
```

### User Segmentation
```
1. All Nikan users in NIKAN_PROMOTION group
2. Query group for bulk operations
3. Send targeted notifications
4. Generate custom reports
```

### Revenue Tracking
```
1. Check Discount model for payment counts
2. Calculate total revenue per hospital
3. Monitor discount usage trends
4. Adjust percentages based on data
```

### Hospital Analytics
```
1. Count users in promotion group
2. Compare with total hospital users
3. Calculate conversion rate
4. Identify top performing hospitals
```

---

## 🛠️ Troubleshooting

### Issue: Hospital Not in Admin
**Cause**: Admin not loaded or error in admin.py

**Fix**:
```bash
python manage.py shell
>>> from panel.admin import HospitalAdmin
>>> # Should import without errors
```

### Issue: Promotion Group Not Created
**Cause**: Error during hospital creation

**Fix**:
```bash
python create_hospital_promotion_groups.py
```

### Issue: Can't Edit Hospital in Admin
**Cause**: Permissions not set

**Fix**:
```bash
# Login as superuser or grant permissions
# User → Permissions → panel | hospital | Can change hospital
```

### Issue: Discount Code Not Working
**Cause**: Code doesn't exist or expired

**Check**:
```python
from payment.models import Discount

discount = Discount.objects.get(code='NIKAN_PROMOTION')
print(f"Valid: {discount.is_valid()}")
print(f"Expires: {discount.expire_date}")
print(f"Uses left: {discount.number_of_discounts}")
```

---

## 📚 Documentation

**Complete guides available:**
- `HOSPITAL_PROMOTION_GROUPS.md` - Full promotion groups documentation
- `HOSPITAL_SETUP_GUIDE.md` - Complete hospital setup
- `HOSPITAL_QUICK_START.md` - 5-minute quick start
- `HOSPITAL_IMPLEMENTATION_SUMMARY.md` - Technical details
- `API_INTEGRATION_UPDATE.md` - API changes

---

## ✅ Summary

### What's New
✅ Hospital model in Django admin with rich interface
✅ Auto-creation of promotion groups and discount codes
✅ Format: `[HOSPITAL_CODE]_PROMOTION`
✅ Utility script for existing hospitals
✅ Display in hospital credentials page
✅ Full documentation

### Benefits
🎯 Easier hospital management via admin
🎯 Automatic promotion code generation
🎯 Better user tracking and segmentation
🎯 Simplified discount campaigns
🎯 No manual setup required

### Next Steps
1. Access Django admin at `/admin/`
2. Navigate to Hospitals section
3. Create or view hospitals
4. Check promotion codes in credentials
5. Use codes for discount campaigns

**Everything is production-ready and fully documented! 🚀**


