# Deep Linking Implementation Guide for Lacto Mobile App

## Overview
This guide explains how to implement deep linking in your mobile app to handle payment callbacks from the ZarinPal payment gateway for timeslot purchases.

## Deep Link Format

### Success Payment
```
lacto://payment/callback?status=success&timeslot_id={timeslot_id}&transaction_id={transaction_id}&payment_id={payment_id}
```

### Failed Payment
```
lacto://payment/callback?status=failed&timeslot_id={timeslot_id}&error_message={error_message}
```

### Cancelled Payment
```
lacto://payment/callback?status=cancelled&timeslot_id=0&error_message={error_message}
```

## Parameters

| Parameter | Type | Description |
|-----------|------|-------------|
| `status` | string | Payment status: `success`, `failed`, or `cancelled` |
| `timeslot_id` | integer | ID of the booked timeslot (0 for cancelled/failed payments) |
| `transaction_id` | string | Payment gateway transaction ID (only for successful payments) |
| `payment_id` | integer | Internal payment ID (only for successful payments) |
| `error_message` | string | URL-encoded error description (only for failed/cancelled payments) |

## Mobile App Implementation

### 1. Register Custom URL Scheme

#### iOS (Info.plist)
```xml
<key>CFBundleURLTypes</key>
<array>
    <dict>
        <key>CFBundleURLName</key>
        <string>com.lacto.app</string>
        <key>CFBundleURLSchemes</key>
        <array>
            <string>lacto</string>
        </array>
    </dict>
</array>
```

#### Android (AndroidManifest.xml)
```xml
<activity android:name=".MainActivity">
    <intent-filter>
        <action android:name="android.intent.action.VIEW" />
        <category android:name="android.intent.category.DEFAULT" />
        <category android:name="android.intent.category.BROWSABLE" />
        <data android:scheme="lacto" />
    </intent-filter>
</activity>
```

### 2. Handle Deep Link Callbacks

#### iOS (Swift)
```swift
import UIKit

class AppDelegate: UIResponder, UIApplicationDelegate {
    
    func application(_ app: UIApplication, open url: URL, options: [UIApplication.OpenURLOptionsKey : Any] = [:]) -> Bool {
        handleDeepLink(url: url)
        return true
    }
    
    func application(_ application: UIApplication, continue userActivity: NSUserActivity, restorationHandler: @escaping ([UIUserActivityRestoring]?) -> Void) -> Bool {
        if userActivity.activityType == NSUserActivityTypeBrowsingWeb {
            if let url = userActivity.webpageURL {
                handleDeepLink(url: url)
            }
        }
        return true
    }
    
    private func handleDeepLink(url: URL) {
        guard url.scheme == "lacto" else { return }
        
        if url.host == "payment" && url.path == "/callback" {
            handlePaymentCallback(url: url)
        }
    }
    
    private func handlePaymentCallback(url: URL) {
        guard let components = URLComponents(url: url, resolvingAgainstBaseURL: false),
              let queryItems = components.queryItems else { return }
        
        let status = queryItems.first { $0.name == "status" }?.value ?? ""
        let timeslotId = Int(queryItems.first { $0.name == "timeslot_id" }?.value ?? "0") ?? 0
        let transactionId = queryItems.first { $0.name == "transaction_id" }?.value
        let paymentId = Int(queryItems.first { $0.name == "payment_id" }?.value ?? "0") ?? 0
        let errorMessage = queryItems.first { $0.name == "error_message" }?.value?.removingPercentEncoding
        
        DispatchQueue.main.async {
            self.showPaymentResult(status: status, timeslotId: timeslotId, transactionId: transactionId, paymentId: paymentId, errorMessage: errorMessage)
        }
    }
    
    private func showPaymentResult(status: String, timeslotId: Int, transactionId: String?, paymentId: Int, errorMessage: String?) {
        switch status {
        case "success":
            showSuccessPayment(timeslotId: timeslotId, transactionId: transactionId ?? "", paymentId: paymentId)
        case "failed":
            showFailedPayment(timeslotId: timeslotId, errorMessage: errorMessage ?? "Unknown error")
        case "cancelled":
            showCancelledPayment(errorMessage: errorMessage ?? "Payment was cancelled")
        default:
            showUnknownPaymentStatus()
        }
    }
}
```

#### Android (Kotlin)
```kotlin
import android.content.Intent
import android.net.Uri
import androidx.appcompat.app.AppCompatActivity

class MainActivity : AppCompatActivity() {
    
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.activity_main)
        
        // Handle deep link if app was opened via deep link
        handleIntent(intent)
    }
    
    override fun onNewIntent(intent: Intent?) {
        super.onNewIntent(intent)
        handleIntent(intent)
    }
    
    private fun handleIntent(intent: Intent?) {
        val data: Uri? = intent?.data
        if (data != null) {
            handleDeepLink(data)
        }
    }
    
    private fun handleDeepLink(uri: Uri) {
        if (uri.scheme == "lacto" && uri.host == "payment" && uri.path == "/callback") {
            handlePaymentCallback(uri)
        }
    }
    
    private fun handlePaymentCallback(uri: Uri) {
        val status = uri.getQueryParameter("status") ?: ""
        val timeslotId = uri.getQueryParameter("timeslot_id")?.toIntOrNull() ?: 0
        val transactionId = uri.getQueryParameter("transaction_id")
        val paymentId = uri.getQueryParameter("payment_id")?.toIntOrNull() ?: 0
        val errorMessage = uri.getQueryParameter("error_message")?.let { 
            java.net.URLDecoder.decode(it, "UTF-8") 
        }
        
        runOnUiThread {
            showPaymentResult(status, timeslotId, transactionId, paymentId, errorMessage)
        }
    }
    
    private fun showPaymentResult(status: String, timeslotId: Int, transactionId: String?, paymentId: Int, errorMessage: String?) {
        when (status) {
            "success" -> showSuccessPayment(timeslotId, transactionId ?: "", paymentId)
            "failed" -> showFailedPayment(timeslotId, errorMessage ?: "Unknown error")
            "cancelled" -> showCancelledPayment(errorMessage ?: "Payment was cancelled")
            else -> showUnknownPaymentStatus()
        }
    }
}
```

### 3. UI Implementation

#### Success Payment Screen
```swift
// iOS
func showSuccessPayment(timeslotId: Int, transactionId: String, paymentId: Int) {
    let storyboard = UIStoryboard(name: "Main", bundle: nil)
    let successVC = storyboard.instantiateViewController(withIdentifier: "PaymentSuccessViewController") as! PaymentSuccessViewController
    
    successVC.timeslotId = timeslotId
    successVC.transactionId = transactionId
    successVC.paymentId = paymentId
    
    // Present the success screen
    if let windowScene = UIApplication.shared.connectedScenes.first as? UIWindowScene,
       let window = windowScene.windows.first {
        window.rootViewController = successVC
        window.makeKeyAndVisible()
    }
}
```

```kotlin
// Android
private fun showSuccessPayment(timeslotId: Int, transactionId: String, paymentId: Int) {
    val intent = Intent(this, PaymentSuccessActivity::class.java).apply {
        putExtra("timeslot_id", timeslotId)
        putExtra("transaction_id", transactionId)
        putExtra("payment_id", paymentId)
        flags = Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK
    }
    startActivity(intent)
    finish()
}
```

#### Error Payment Screen
```swift
// iOS
func showFailedPayment(timeslotId: Int, errorMessage: String) {
    let alert = UIAlertController(title: "Payment Failed", message: errorMessage, preferredStyle: .alert)
    alert.addAction(UIAlertAction(title: "OK", style: .default) { _ in
        // Navigate to timeslot selection or main screen
        self.navigateToMainScreen()
    })
    
    if let windowScene = UIApplication.shared.connectedScenes.first as? UIWindowScene,
       let window = windowScene.windows.first,
       let rootVC = window.rootViewController {
        rootVC.present(alert, animated: true)
    }
}
```

```kotlin
// Android
private fun showFailedPayment(timeslotId: Int, errorMessage: String) {
    AlertDialog.Builder(this)
        .setTitle("Payment Failed")
        .setMessage(errorMessage)
        .setPositiveButton("OK") { _, _ ->
            // Navigate to timeslot selection or main screen
            navigateToMainScreen()
        }
        .show()
}
```

## Testing Deep Links

### 1. iOS Simulator
```bash
# Open app with deep link
xcrun simctl openurl booted "lacto://payment/callback?status=success&timeslot_id=123&transaction_id=ZP123456789&payment_id=456"
```

### 2. Android Emulator
```bash
# Open app with deep link
adb shell am start -W -a android.intent.action.VIEW -d "lacto://payment/callback?status=success&timeslot_id=123&transaction_id=ZP123456789&payment_id=456" com.lacto.app
```

### 3. Web Browser Testing
```html
<!-- Test deep link in browser -->
<a href="lacto://payment/callback?status=success&timeslot_id=123&transaction_id=ZP123456789&payment_id=456">
    Test Deep Link
</a>
```

## Security Considerations

1. **Validate Parameters**: Always validate the incoming parameters before processing
2. **Authentication**: Ensure the user is authenticated before showing payment results
3. **Rate Limiting**: Implement rate limiting for deep link callbacks
4. **Logging**: Log all deep link attempts for security monitoring

## Error Handling

1. **Invalid URLs**: Handle malformed deep link URLs gracefully
2. **Missing Parameters**: Provide default values for missing parameters
3. **Network Errors**: Handle cases where the app needs to fetch additional data
4. **User Not Logged In**: Redirect to login screen if user is not authenticated

## Best Practices

1. **Fallback URLs**: Always provide fallback URLs for cases where deep linking fails
2. **User Experience**: Ensure smooth transitions between payment gateway and app
3. **Loading States**: Show appropriate loading states while processing deep links
4. **Analytics**: Track deep link usage for user behavior analysis

## Troubleshooting

### Common Issues

1. **Deep Link Not Opening App**
   - Verify URL scheme registration
   - Check app bundle identifier
   - Ensure app is installed

2. **Parameters Not Parsed Correctly**
   - Verify URL encoding for special characters
   - Check parameter names match exactly
   - Validate parameter types

3. **App Crashes on Deep Link**
   - Add proper error handling
   - Validate all parameters before use
   - Test with various parameter combinations

### Debug Tips

1. **Enable Logging**: Add comprehensive logging for deep link handling
2. **Test Different Scenarios**: Test success, failure, and cancellation flows
3. **Validate on Real Devices**: Test on actual devices, not just simulators
4. **Check App State**: Ensure deep links work in all app states (foreground, background, terminated)

## Support

For technical support or questions about this implementation, please contact the development team or refer to the project documentation.
