# Blood Test Report Generation API

## Overview
This API endpoint processes blood test reports (PDF or images) and generates comprehensive health reports with structured data analysis.

## Endpoint Details

### URL
```
POST /api/report/blood/test/generate/
```

### Authentication
Required: User token

### Request Parameters

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `token` | string | Yes | User authentication token (30 characters) |
| `file` | file | Yes | Blood test report file (PDF, PNG, JPG, or JPEG) |

### Request Example

#### Using cURL
```bash
curl -X POST http://your-domain.com/api/report/blood/test/generate/ \
  -F "token=your_authentication_token_here" \
  -F "file=@/path/to/blood_test_report.pdf"
```

#### Using Python (requests)
```python
import requests

url = "http://your-domain.com/api/report/blood/test/generate/"
files = {
    'file': open('blood_test_report.pdf', 'rb')
}
data = {
    'token': 'your_authentication_token_here'
}

response = requests.post(url, files=files, data=data)
result = response.json()
print(result)
```

#### Using Postman
1. Set method to `POST`
2. URL: `http://your-domain.com/api/report/blood/test/generate/`
3. In Body tab, select `form-data`
4. Add key `token` with your token value
5. Add key `file` and select type as `File`, then upload your PDF/image

### Response Format

#### Success Response (200 OK)
```json
{
    "status": 1,
    "message": "گزارش با موفقیت ایجاد شد",
    "data": {
        "structured_data": {
            "general_information": {
                "patient_name": "نام بیمار",
                "age": "35",
                "gender": "مرد",
                "date_of_birth": "1368/05/15",
                "collection_date": "1403/08/10",
                "confirm_date": "1403/08/11"
            },
            "test_results": [
                {
                    "category": "Biochemistry Test Result",
                    "tests": [
                        {
                            "test_name": "FBS",
                            "result": 95,
                            "unit": "mg/dL",
                            "normal_range": "70-100",
                            "method": "Enzymatic"
                        }
                        // ... more tests
                    ]
                }
                // ... more categories
            ],
            "urine_analysis": {
                // ... urine analysis data if present
            }
        },
        "test_analysis": [
            {
                "test_name": "FBS",
                "status": "normal"
            },
            {
                "test_name": "Cholesterol",
                "status": "high"
            }
            // ... more test analyses
        ],
        "health_report_markdown": "# گزارش جامع وضعیت سلامت\n\n## مقدمه گزارش\n...",
        "user_id": 123,
        "processed_at": "2025-10-13T10:30:00"
    }
}
```

#### Error Responses

**Invalid Token (401)**
```json
{
    "status": -1,
    "message": "توکن اشتباه وارد شده است",
    "data": null
}
```

**Missing File (400)**
```json
{
    "status": -1,
    "message": "فایل آزمایش خون الزامی است",
    "data": null
}
```

**Invalid File Format (400)**
```json
{
    "status": -1,
    "message": "فرمت فایل باید PDF یا تصویر باشد",
    "data": null
}
```

**Processing Error (500)**
```json
{
    "status": -1,
    "message": "خطا در پردازش گزارش: [error details]",
    "data": null
}
```

## Processing Pipeline

The API uses a sophisticated 5-stage pipeline:

1. **OCR Stage**: Extracts text from PDF/images using Google Cloud Vision
2. **Structuring Stage**: Uses AI (Vertex AI/Gemini/OpenAI fallback) to structure raw text into JSON
3. **Analysis Stage**: Analyzes test results against normal ranges
4. **Report Generation**: Creates a comprehensive Persian health report
5. **Cleanup**: Removes temporary files

## Response Data Explanation

### structured_data
Contains the complete parsed medical data including:
- General patient information
- All test results organized by category
- Normal ranges and units for each test
- Test methods used

### test_analysis
Array of test result analyses with statuses:
- `normal`: Result is within normal range
- `high`: Result is above normal range
- `low`: Result is below normal range
- `abnormal`: Qualitative result differs from normal
- `not_applicable`: No range available
- `not_analyzed`: Range format not recognized

### health_report_markdown
A beautifully formatted Persian health report including:
- Introductory health overview
- Body identity at a glance
- Overall health snapshot
- CBC & basic chemistry analysis
- Metabolic & cardiovascular landscape
- Vital organs health (liver, thyroid, etc.)

## Supported File Formats

- **PDF**: Multi-page blood test reports
- **Images**: PNG, JPG, JPEG formats
- **Resolution**: Recommended 300 DPI or higher for best OCR results

## AI Provider Fallback

The system uses a robust 3-tier fallback mechanism:
1. **Google Vertex AI** (Primary - skipped if not available)
2. **Google AI Studio (Gemini)** (Primary fallback - highly reliable)
3. **OpenAI GPT-4** (Final fallback)

This ensures high availability even if one provider fails. If Vertex AI is not properly configured or has version incompatibilities, the system automatically skips to Google AI Studio without errors.

## Best Practices

1. **File Quality**: Use high-resolution scans for better OCR accuracy
2. **File Size**: Keep files under 10MB for optimal processing
3. **Language**: Currently optimized for Persian/Farsi reports
4. **Format**: Standard pathology lab formats are best supported

## Example Use Cases

1. **Mobile App Integration**: Allow users to upload blood test photos for instant analysis
2. **Health Dashboard**: Display structured test results in a clean UI
3. **Health Coaching**: Use the Markdown report for personalized guidance
4. **Trend Analysis**: Store structured data for longitudinal health tracking

## Configuration Requirements

The following environment variables/files must be configured:

1. **Google Cloud Credentials**:
   - Service account JSON: `key-DocumentAI/healiomenew-94ee24f33f41.json`
   - Project ID: `healiomenew`
   - Location: `us-central1`

2. **API Keys**:
   - Gemini API Key (configured in BloodTestReport_generat.py)
   - OpenAI API Key (configured in BloodTestReport_generat.py)

## Troubleshooting

### Common Issues

**1. ImportError: cannot import name 'GenerativeModel' from 'vertexai.generative_models'**
- This is a known version compatibility issue with Vertex AI
- **Solution**: The system will automatically skip Vertex AI and use Google AI Studio (Gemini) instead
- No action needed - the API will work normally with the fallback providers

**2. OCR fails to detect text**
- Ensure image quality is high enough
- Check if the image is properly oriented
- Verify file is not corrupted

**3. All AI providers fail**
- Check internet connectivity
- Verify API keys are valid (especially Gemini API key)
- Check Google Cloud credentials for OCR

**4. Timeout errors**
- Large files may take longer to process
- Consider implementing async processing for production

## Security Considerations

1. All uploaded files are stored temporarily and deleted after processing
2. User authentication is required via token
3. Files are processed in isolated temporary directories
4. No persistent storage of uploaded files

## Performance Notes

- Average processing time: 30-60 seconds for a typical 2-page PDF
- Memory usage: ~200MB per request
- Concurrent requests: Limited by AI provider rate limits

## Support

For issues or questions, please contact the development team or create an issue in the project repository.

