# Automated Diet Generation System

## Overview

This document describes the automated diet generation system that creates 14-day diets with 5 meals per day when users complete the LifestyleQuestionnaire through the mobile app.

## Architecture

### Service Layer (`panel/services/diet_generation.py`)

The core diet generation logic has been extracted into a reusable service layer:

- **`generate_meal_plan(user, meal_type, selected_tags=None, count=14)`**: Generates meal plans for a specific meal type
- **`generate_complete_diet_for_user(user_id, questionnaire_event_id=None)`**: Creates a complete 14×5 diet
- **`start_diet_generation_background(user_id, questionnaire_event_id=None)`**: Starts diet generation in a background thread

### Refactored API Endpoint

The existing `assistant/column/refresh/` endpoint now uses the service layer while maintaining the same functionality and authentication requirements.

### Background Task Integration

The questionnaire completion logic now automatically triggers diet generation for users who:
1. Complete a LifestyleQuestionnaire
2. Submit the request from the mobile app (`is_from_app=1`)
3. Have no existing diets

## Features

### 1. **Idempotency**
- Each questionnaire completion generates a unique event ID
- Duplicate diet generation requests for the same event return the existing diet
- Prevents data duplication on retry scenarios

### 2. **Asynchronous Processing**
- Diet generation runs in background threads
- Questionnaire API responds immediately (non-blocking)
- Uses the existing threading approach (compatible with current infrastructure)

### 3. **Error Handling**
- Graceful handling of meal generation failures
- Partial diet creation if some meals fail
- Comprehensive logging for debugging

### 4. **Schema Compliance**
- Generated diets follow the canonical `Diet.diet_json` schema
- Compatible with existing diet display and management systems
- Proper nutritional information calculation

## Data Flow

```
1. User completes LifestyleQuestionnaire via mobile app
2. Questionnaire API validates and saves data
3. System checks if user has existing diets
4. If no diets exist, background diet generation is triggered
5. Service generates 5 meal types × 14 days = 70 meal cards
6. Complete diet is assembled and saved
7. Diet becomes available immediately with from_date = now + 24h
```

## Diet JSON Structure

```json
{
  "week1": {
    "breakfast": [
      [{"title": "Card Name", "ingredient": [...]}],
      // ... 13 more days
    ],
    "lunch": [...],
    "dinner": [...],
    "snack1": [...],
    "snack2": [...]
  },
  "type": "Auto-Generated Diet",
  "generated_at": "2024-01-XX...",
  "questionnaire_event_id": "lifestyle_questionnaire_123_456_abc12345"
}
```

## Configuration

### Meal Types and Calorie Distribution

The system uses the existing calorie distribution logic:
- Determines user's CR (Calorie Requirement) from profile
- Uses coach CR if available, otherwise patient CR
- Maps to predefined calorie distributions for each meal type

### Food Card Selection

Cards are filtered based on:
- User's dietary restrictions and allergies (from questionnaire)
- Disease-specific exclusions
- Meal-type specific filters (breakfast/lunch/dinner/snack)
- Calorie targets (±10% tolerance)
- Uniqueness constraints (no duplicate main_dish_codes or FA_Names)

## Testing

### Unit Tests (`panel/tests/test_diet_generation.py`)

- Service function testing with mocked dependencies
- Idempotency verification
- Error handling scenarios
- Background task initialization

### Integration Tests

- End-to-end diet generation flow
- Database integration testing
- Schema validation

### Running Tests

```bash
python manage.py test panel.tests.test_diet_generation
```

## Monitoring and Debugging

### Logging

All diet generation activities are logged with detailed information:
- User identification
- Meal generation progress
- Error conditions
- Final diet creation status

### Key Log Messages

- `"Started background diet generation for user {user_id}"`
- `"Generated {count} cards for {meal_type}"`
- `"Successfully created diet {diet_id} for user {user_id}"`
- `"Failed to generate diet for user {user_id}: {error}"`

## Deployment Notes

### Dependencies

No additional dependencies required. The system uses:
- Django ORM for data access
- Python threading for background processing
- Existing assistant module functions
- Current FoodCard and Diet models

### Database Considerations

- No schema changes required
- Uses existing `Diet.diet_json` field
- Leverages current indexing on user and date fields

### Performance

- Background processing prevents API blocking
- Memory-efficient meal generation (processes one meal type at a time)
- Minimal database queries per meal generation

## Future Enhancements

1. **Queue System**: Replace threads with Celery for better scalability
2. **Advanced Scheduling**: Smarter timing for diet start dates
3. **Preference Learning**: Incorporate user feedback into generation
4. **Batch Processing**: Generate diets for multiple users simultaneously
5. **Caching**: Cache common meal combinations for faster generation

## Troubleshooting

### Common Issues

1. **No cards generated**: Check user's CR value and dietary restrictions
2. **Partial diet creation**: Review food card availability for specific meal types
3. **Generation timeouts**: Monitor background thread execution

### Debug Commands

```python
# Test diet generation for a specific user
from panel.services.diet_generation import generate_complete_diet_for_user
success, diet_id, error = generate_complete_diet_for_user(user_id)

# Test individual meal generation
from panel.services.diet_generation import generate_meal_plan
cards = generate_meal_plan(user, 'breakfast')
```

## API Changes Summary

### New Functionality
- Automatic diet generation on LifestyleQuestionnaire completion
- Background processing with idempotency
- Service layer for reusable diet generation

### Backward Compatibility
- Existing API endpoints maintain full compatibility
- No changes to request/response formats
- Coach workflow remains unchanged
