# Hospital API Integration Update

## 🔄 What Changed

The hospital API key integration has been updated from a hardcoded configuration system to a dynamic database-driven system.

### Before (Old System)
Hospital API keys were stored in `ziluck_project/hospital_config.py` as a hardcoded dictionary:

```python
HOSPITAL_INFO = {
    "Z9xd3NikanhospitalQ8zYpR7G5wK2uHvM1R": {
        "name": "Nikan Hospital",
        "code": "NIKAN",
        ...
    }
}
```

### After (New System)
Hospital API keys are now stored in the database using the `Hospital` model with full CRUD management via the Manager panel.

## 📊 Benefits of the New System

1. **Dynamic Management**: Add/edit/delete hospitals without code changes
2. **Scalable**: No limit on number of hospitals
3. **Secure**: Regenerate API credentials on demand
4. **Auditable**: Track who created hospitals and when
5. **Integrated**: Hospital panel for each hospital to view their users

## 🔧 API Changes

### Profile Info API (`/profile_info/`)

#### Parameter
The `hospital_api_key` parameter works the same way:

```json
POST /profile_info/
{
    "token": "user_token_here",
    "hospital_api_key": "HK_xxx...",
    ...other profile fields...
}
```

#### What Happens Now

1. **Database Lookup** (Primary):
   ```python
   hospital = Hospital.objects.get(api_key=hospital_api_key, is_active=True)
   user.profile.hospital_code = hospital.hospital_code
   ```

2. **Legacy Fallback** (Backward Compatibility):
   If not found in database, falls back to old `HOSPITAL_INFO` dictionary
   ```python
   if hospital_api_key in HOSPITAL_INFO:
       hospital_code = HOSPITAL_INFO[hospital_api_key]["code"]
       user.profile.hospital_code = hospital_code
   ```

3. **Error Handling**:
   If API key not found in either place, error is logged but request continues

## 🔑 API Key Format

### Old Format (Legacy)
- Custom strings like: `Z9xd3NikanhospitalQ8zYpR7G5wK2uHvM1R`

### New Format (Recommended)
- Prefix-based: `HK_{random_secure_token}`
- Example: `HK_xJ8kL2mN9pQ5rT3vW7yZ1aB4cD6eF0gH2iK5lM8n`

Both formats work! The system checks the database first, then falls back to legacy config.

## 🚀 Migration Guide

### Step 1: Run Database Migration
```bash
python manage.py makemigrations
python manage.py migrate
```

### Step 2: Migrate Existing Hospitals
```bash
python migrate_hospitals_to_db.py
```

This script will:
- ✓ Read hospitals from `HOSPITAL_INFO`
- ✓ Create `Hospital` database records
- ✓ Create Django users for hospital login
- ✓ Set default passwords (must be changed!)

### Step 3: Update External Systems (If Needed)

If you have external hospital systems calling the API:

#### Option A: Keep Using Same API Keys (Recommended)
- No changes needed!
- Old API keys migrated to database automatically
- Everything works as before

#### Option B: Generate New API Keys
1. Login as Manager → Hospital Management
2. Select hospital → Regenerate Credentials
3. Update external system with new API key

### Step 4: Test Integration

```bash
# Test with curl
curl -X POST https://your-domain.com/profile_info/ \
  -H "Content-Type: application/json" \
  -d '{
    "token": "user_token",
    "hospital_api_key": "HK_your_api_key_here",
    "first_name": "Test",
    "last_name": "User",
    ...
  }'
```

## 🔐 Security Improvements

### Regenerate API Credentials

**Via Manager Panel:**
1. Go to `/panel/manager/hospitals/`
2. Click dropdown → "Regenerate Credentials"
3. New API key and secret generated

**Via Django Shell:**
```python
from panel.models import Hospital

hospital = Hospital.objects.get(hospital_code="NIKAN")
api_key, api_secret = hospital.regenerate_api_credentials()
print(f"New API Key: {api_key}")
print(f"New API Secret: {api_secret}")
```

### API Key Validation

The system now validates:
- ✓ API key exists in database
- ✓ Hospital is active (`is_active=True`)
- ✓ API key format is correct

## 📝 API Response Changes

### Success Response
Same as before - profile is created/updated with `hospital_code` set:

```json
{
    "status": "ok",
    "message": "پروفایل ساخته شد",
    "data": {
        ...profile data including hospital_code...
    }
}
```

### User Profile Includes Hospital Code
When fetching profile (GET `/profile_info/`):

```json
{
    "status": "ok",
    "data": {
        "first_name": "John",
        "last_name": "Doe",
        "hospital_code": "NIKAN",
        ...
    }
}
```

## 🔍 Debugging

### Check If API Key Exists
```python
from panel.models import Hospital

# Check in database
hospital = Hospital.objects.filter(api_key="YOUR_API_KEY").first()
if hospital:
    print(f"Found: {hospital.name} ({hospital.hospital_code})")
    print(f"Active: {hospital.is_active}")
else:
    print("Not found in database")

# Check users with this hospital
from APIs.models import Profile
users = Profile.objects.filter(hospital_code=hospital.hospital_code)
print(f"Users: {users.count()}")
```

### View Hospital Assignment Logs
Check Django logs for messages like:
```
User 09123456789 assigned to hospital: Nikan Hospital (NIKAN)
```

Or for legacy:
```
User 09123456789 assigned to hospital using legacy config: NIKAN
```

Or for errors:
```
Invalid hospital API key: invalid_key_here
```

## 🛠 Troubleshooting

### Issue: "Invalid hospital API key" Error

**Causes:**
1. API key doesn't exist in database
2. API key doesn't exist in `HOSPITAL_INFO`
3. Hospital is inactive (`is_active=False`)

**Solutions:**
1. Check if hospital exists: `Hospital.objects.filter(api_key="KEY")`
2. Create hospital via Manager panel
3. Activate hospital: `hospital.is_active = True; hospital.save()`

### Issue: User Not Assigned to Hospital

**Causes:**
1. `hospital_api_key` parameter not sent
2. API key validation failed
3. Exception during lookup

**Solutions:**
1. Verify `hospital_api_key` is in POST data
2. Check API key validity
3. Review Django error logs

### Issue: Legacy API Keys Not Working

**Solution:**
Run migration script to move them to database:
```bash
python migrate_hospitals_to_db.py
```

## 📊 Monitoring

### Track Hospital API Usage

```python
from APIs.models import Profile
from django.db.models import Count

# Users per hospital
stats = Profile.objects.filter(
    hospital_code__isnull=False
).values('hospital_code').annotate(
    count=Count('id')
).order_by('-count')

for stat in stats:
    print(f"{stat['hospital_code']}: {stat['count']} users")
```

### Recent Hospital Registrations

```python
from APIs.models import User
from datetime import datetime, timedelta

# Users registered in last 7 days with hospital code
recent = User.objects.filter(
    profile__hospital_code__isnull=False,
    created_at__gte=datetime.now() - timedelta(days=7)
).select_related('profile')

for user in recent:
    print(f"{user.phone_number} -> {user.profile.hospital_code}")
```

## 🎯 Best Practices

### For Hospital Integrations

1. **Use Descriptive Hospital Codes**
   - Good: `NIKAN`, `TEHRAN_HEART`, `CITY_GENERAL`
   - Bad: `H1`, `HOSP`, `TEST`

2. **Keep API Keys Secure**
   - Don't commit to version control
   - Use environment variables
   - Regenerate if compromised

3. **Test Before Production**
   - Test with one user first
   - Verify hospital_code is set
   - Check hospital panel shows user

4. **Monitor Usage**
   - Track users per hospital
   - Monitor failed API key attempts
   - Review hospital upgrade counts

### For Platform Administrators

1. **Regular Audits**
   - Review hospital list monthly
   - Deactivate unused hospitals
   - Update contact information

2. **Security**
   - Regenerate credentials annually
   - Use strong passwords for hospital accounts
   - Enable 2FA (when implemented)

3. **Communication**
   - Notify hospitals before credential changes
   - Provide migration support
   - Document API changes

## 📞 Support

For questions or issues:
1. Check this documentation
2. Review Django error logs
3. Test with `migrate_hospitals_to_db.py`
4. Contact platform development team

## 🎉 Summary

✅ **Backward Compatible**: Old API keys still work via legacy fallback
✅ **Easy Migration**: One-command script to migrate existing hospitals
✅ **Better Security**: Regenerate credentials on demand
✅ **Full Management**: Create/edit/delete via admin panel
✅ **Scalable**: No code changes needed for new hospitals

The new system provides a solid foundation for hospital integrations while maintaining compatibility with existing implementations!


