## User Creation and Login Flow

This document describes the end-to-end flow for user verification, registration, and login in the Lacto API.

### Overview

- Step 1 — Get verification code: call `user/activation_code/` with a `phone_number`. You receive a `temp_token` via API and a one-time code via SMS.
- Step 2 — Check registration status: call `v2/user/check/registration/` with the same `phone_number` (or `email`) to see whether the user is already registered (`isRegistered` boolean).
- Step 3a — If not registered, complete signup: call `user/sign/` with `phone_number`, `temp_token`, and `activation_code` to create the account and receive the main `token`.
- Step 3b — If registered, log in: call `user/login/` with `phone_number`, `temp_token`, and `activation_code` to receive the main `token`.

Notes:
- SMS resend is rate-limited (cooldown). The code validity window and cooldown are enforced server-side.
- Responses contain localized Persian messages. The examples below show the shape and critical fields.

### Endpoints Summary

- Activation code (v1): `POST /api/user/activation_code/`
- Registration status (v2): `POST /api/v2/user/check/registration/`
- Sign up (v1): `POST /api/user/sign/`
- Log in (v1): `POST /api/user/login/`

Depending on deployment, the base prefixes (`/api`, `/api/v2`) may differ; paths above reflect the modules:
- `APIs/urls.py` (v1) and `APIs/views/v2/url.py` (v2).

---

### 1) Request verification code — activation_code

Endpoint: `POST /api/user/activation_code/`

Body:
```json
{
  "phone_number": "09XXXXXXXXX"
}
```

Validations and behavior:
- `phone_number` must match regex `^09[0-9]{9}$`.
- If a recent request exists for the same phone, a cooldown is enforced before another SMS is sent.
- The server issues or reuses a `temp_token` (15 chars) and a 4-digit `activation_code`, and sends the code via SMS.

Success response (shape):
```json
{
  "message": "کد فعال سازی با موفقیت ارسال شد",
  "data": {
    "temp_token": "<15-char-temp-token>"
  }
}
```

Possible errors (examples):
- SMS provider error → `ErrorInSMS`.
- Cooldown not elapsed → `Try_a_few_seconds_later`.
- Invalid argument/format errors.

Example curl:
```bash
curl -X POST \
  -H "Content-Type: application/x-www-form-urlencoded" \
  -d "phone_number=09123456789" \
  https://<host>/api/user/activation_code/
```

Relevant implementation references:
```45:106:APIs/views/user.py
def activation_code(request):
    try:
        RequsetChecker(request.POST, [
            {
                "name": "phone_number",
                "format": "^09[0-9]{9}$"
            }
        ], request)

        phone = str(request.POST["phone_number"])

        xx = TempToken.objects.filter(phone_number=phone)
        now = datetime.now()
        # ... logic for expiry, cooldown, token/code generation ...
        if not SMS().send_lookup(code, phone):
            return myResponse.Error(Errors.ErrorInSMS.message, Errors.ErrorInSMS.code)

        data = { "temp_token": token }
        return myResponse.OK("کد فعال سازی با موفقیت ارسال شد", data)
```

---

### 2) Check registration status — v2/registration

Endpoint: `POST /api/v2/user/check/registration/`

Body (one of):
```json
{ "phone_number": "09XXXXXXXXX" }
```
or
```json
{ "email": "user@example.com" }
```

Behavior:
- If `phone_number` is provided, looks up the corresponding `User` and returns `isRegistered` (true only when user exists and has a non-null profile and is marked registered).
- If `email` is provided, looks up by `profile__email` with similar logic.

Success response (shape):
```json
{
  "message": "وضعیت کاربری",
  "data": { "isRegistered": true }
}
```

Error response:
- If neither `phone_number` nor `email` provided → `InvalidRequest`.

Relevant implementation references:
```135:177:APIs/views/v2/user.py
def registration(request):
    RequsetChecker(request.POST, [
        { "name": "phone_number", "format": "^09[0-9]{9}$", "required": False },
        { "name": "email", "format": "", "required": False }
    ], request)

    if request.POST.get("phone_number"):
        # ... compute isRegistered by phone ...
        return myResponse.OK("وضعیت کاربری", {"isRegistered": isRegistered})
    elif request.POST.get("email"):
        # ... compute isRegistered by email ...
        return myResponse.OK("وضعیت کاربری", {"isRegistered": isRegistered})
    else:
        return myResponse.Error(Errors.InvalidRequest.message, Errors.InvalidRequest.code)
```

---

### 3a) Complete signup — sign

Endpoint: `POST /api/user/sign/`

Body:
```json
{
  "phone_number": "09XXXXXXXXX",
  "temp_token": "<15-char-temp-token>",
  "activation_code": "\d{4}"
}
```

Validations and behavior:
- Verifies the `TempToken` record by `phone_number` + `temp_token` and checks the `activation_code` (4 digits).
- If a fully registered user already exists for the phone number, returns an error (`RegisteredLast`).
- Otherwise creates or updates the application `User`, creates related `Profile` and `Address` if needed, provisions a corresponding `DjangoUser` (username = phone, password = first 8 chars of the new main token), adds user to `Patient` group, marks `isLogedin = True`, and returns the main session `token` (30 chars).
- The used temp token is deleted on success.

Success response (shape):
```json
{
  "message": "دریافت توکن با موفقیت انجام شد",
  "data": { "token": "<30-char-token>" }
}
```

Possible errors:
- `TempTokenNotExist` (no matching temp token for phone).
- `RegisteredLast` (user already registered; must use login).
- `InvalidToken` (temp token mismatch).
- `InvalidActivationCode` (code mismatch).

Relevant implementation references:
```179:250:APIs/views/user.py
def sign(request):
    RequsetChecker(request.POST, [
        { "name": "temp_token", "format": "^(\S){15}$" },
        { "name": "phone_number", "format": "^09[0-9]{9}$" },
        { "name": "activation_code", "format": "^[0-9]{4}$" }
    ], request)
```

---

### 3b) Login — LogIn

Endpoint: `POST /api/user/login/`

Body:
```json
{
  "phone_number": "09XXXXXXXXX",
  "temp_token": "<15-char-temp-token>",
  "activation_code": "\d{4}"
}
```

Validations and behavior:
- Verifies the `TempToken` record and checks the `activation_code`.
- Requires an existing, previously registered user with a non-null profile.
- Issues a new `token` (30 chars), updates the linked `DjangoUser` password to the first 8 chars of the token, sets `isLogedin = True`, and deletes the temp token.

Success response (shape):
```json
{
  "message": "ورود با موفقیت انجام شد",
  "data": { "token": "<30-char-token>" }
}
```

Possible errors:
- `InvalidToken` (no matching temp token for phone, or mismatch).
- `InvalidActivationCode`.
- `NotFoundUser` (no such registered user).
- `NotFoundProfile` (user exists but profile missing).

Relevant implementation references:
```269:341:APIs/views/user.py
def LogIn(request):
    RequsetChecker(request.POST, [
        { "name": "temp_token", "format": "^(\S){15}$" },
        { "name": "phone_number", "format": "^09[0-9]{9}$" },
        { "name": "activation_code", "format": "." }
    ], request)
```

---

### Typical Client Flow

1) User enters `phone_number`.
2) Client calls `POST /api/user/activation_code/` and receives `temp_token` (SMS delivers the 4-digit code to the phone).
3) Client calls `POST /api/v2/user/check/registration/` with the same `phone_number` to fetch `isRegistered`.
   - If `isRegistered = false`: call `POST /api/user/sign/` with `phone_number`, `temp_token`, and `activation_code` to create the account and obtain the main `token`.
   - If `isRegistered = true`: call `POST /api/user/login/` with `phone_number`, `temp_token`, and `activation_code` to obtain the main `token`.
