# updateTotalPointDay Function Documentation

## Overview
The `updateTotalPointDay` function is a comprehensive scoring system that calculates and updates a user's daily health points based on multiple wellness metrics including diet, physical activity, water intake, sleep quality, and user engagement.

## Function Signature

```python
def updateTotalPointDay(user, record, isCall=0, is_Diet=False, is_Activity=False, 
                        is_Step=False, is_Sleep=False, is_Message=False, 
                        is_Water=False, is_Mid_Day_Sleep=False)
```

## Parameters

| Parameter | Type | Default | Description |
|-----------|------|---------|-------------|
| `user` | User object | Required | The user object for whom points are being calculated |
| `record` | Record object | Required | The daily record object that stores point values |
| `isCall` | int | 0 | Special flag for diet score handling. `-1` triggers normal calculation, other values override the diet score |
| `is_Diet` | bool | False | Flag to trigger diet and fiber point calculations |
| `is_Activity` | bool | False | Flag to trigger activity/exercise point calculations |
| `is_Step` | bool | False | Flag to trigger step count point calculations |
| `is_Sleep` | bool | False | Flag to trigger sleep quality point calculations |
| `is_Message` | bool | False | Flag to trigger messaging engagement point calculations |
| `is_Water` | bool | False | Flag to trigger water intake point calculations |
| `is_Mid_Day_Sleep` | bool | False | Flag to identify mid-day naps (prevents sleep point calculation) |

## Return Value

Returns a dictionary with the following structure:

```python
{
    "DayPoint": int,              # Total points for the day (0-100)
    "totalCooperate": int,        # Cooperation score (currently always 0)
    "Diet_score": float,          # Total diet points (0-40)
    "Fiber_score": float,         # Fiber intake points (0-15)
    "Activity_score": float,      # Combined activity + step points (0-10)
    "Water_score": float,         # Water intake points (0-10)
    "Sleep_score": float,         # Sleep quality points (0-10)
    "Engagement_score": float,    # User engagement points (0-10)
    "Message_score": int,         # Message interaction points (0-5)
    "points_awarded": dict        # Breakdown of points gained in this call
}
```

### points_awarded Dictionary Structure

```python
{
    'Diet_points_gained': int,
    'Fiber_points_gained': int,
    'Activity_points_gained': int,
    'Step_points_gained': int,
    'Water_points_gained': int,
    'Sleep_points_gained': int,
    'Engagement_points_gained': int,
    'Message_points_gained': int
}
```

## Scoring Breakdown

### Total Points: 100 Maximum

| Category | Maximum Points | Description |
|----------|----------------|-------------|
| Diet | 40 | Based on meal adherence (breakfast, lunch, dinner, 2 snacks) |
| Fiber | 15 | Based on daily fiber intake vs. gender-based targets |
| Exercise | 10 | Combined score from activities and steps |
| Water | 10 | Based on daily water intake (target: 8 glasses) |
| Sleep | 10 | Based on sleep duration and timing |
| Engagement | 10 | Data entry frequency and challenge participation |
| Messages | 5 | Communication/interaction points |

## Detailed Scoring Logic

### 1. Diet Scoring (is_Diet=True)

**Maximum Points: 40**

- **Meal Breakdown:**
  - Breakfast: 0-10 points
  - Lunch: 0-10 points
  - Dinner: 0-10 points
  - Snack 1: 0-10 points
  - Snack 2: 0-10 points

- **Logic:**
  - Retrieves all eating items for the day
  - Calculates nutrition facts per meal
  - Scores each meal based on mismatch between actual and target nutrition
  - Each meal can score up to 10 points
  - Total diet score is the sum of all meals (capped at 40)

- **Data Entry Bonus:**
  - Increments `this_day_DataEntry_score` by 1 (max 10) when diet data is entered

### 2. Fiber Scoring (calculated with Diet)

**Maximum Points: 15**

- **Gender-based targets:**
  - Female (gender=1): 28g per day
  - Male: 38g per day

- **Formula:**
  ```
  Fiber_points = (actual_fiber_intake / target_fiber) * 15
  Capped at 15 points maximum
  ```

### 3. Step Scoring (is_Step=True)

**Maximum Points: 10** (combined with Activity)

- **Logic:**
  - 1 point per 1,000 steps
  - Maximum 10 points at 10,000 steps
  - Combined with Activity score (total cannot exceed 10)

- **Point Award Logic:**
  - First-time entry: Award full step points
  - If steps equal previous and not maxed: -1 (no change)
  - If activity points exist and combined > 10: Award only to reach 10
  - If total exercise already at 10: Award -2 (no gain)

### 4. Activity Scoring (is_Activity=True)

**Maximum Points: 5** (contributes to 10-point Exercise total)

- **Logic:**
  - If any activity logged for the day: 5 points
  - If no activity: 0 points
  - Combined with step points (total cannot exceed 10)

- **Data Entry Bonus:**
  - Increments `this_day_DataEntry_score` by 1 (max 10)

### 5. Water Scoring (is_Water=True)

**Maximum Points: 10**

- **Target:** 8 glasses of water per day

- **Formula:**
  ```
  Water_points = (water_intake / 8) * 10
  Capped at 10 points maximum
  ```

- **Data Entry Bonus:**
  - Increments `this_day_DataEntry_score` by 1 (max 10)

### 6. Sleep Scoring (is_Sleep=True, is_Mid_Day_Sleep=False)

**Maximum Points: 10**

- **Components:**
  - Sleep Duration: 0-5 points
  - Sleep Start Time: 0-5 points

#### Nap Detection
Sleeps are classified as naps if:
- Start time is between 11:00 AM and 8:00 PM, AND
- Duration is less than 2 hours

**Naps are excluded from scoring.**

#### Sleep Duration Points (5 max)
- **Ideal:** 8 hours
- **Formula:**
  ```
  duration_difference = |actual_hours - 8|
  duration_points = max(5 - duration_difference, 0)
  ```

#### Sleep Start Time Points (5 max)
- **Ideal start time:** 10:00 PM (22:00)
- **Formula:**
  ```
  hours_late = max(start_time - 22:00, 0)
  start_time_points = max(5 - hours_late, 0)
  ```
- No penalty for sleeping before 10 PM

#### Total Sleep Points
```
Total = duration_points + start_time_points (max 10)
```

- **Data Entry Bonus:**
  - Increments `this_day_DataEntry_score` by 1 (max 10)

### 7. Message Scoring (is_Message=True)

**Maximum Points: 5**

- **Logic:**
  - If any messages sent or received during the day: +1 point
  - Accumulates throughout the day (max 5)

### 8. Engagement Scoring (automatic)

**Maximum Points: 10**

- **Components:**
  - Data Entry Score: 0-10 (incremented when diet, activity, water, or sleep data is entered)
  - Challenge Participation: +5 if user is in an active challenge

- **Formula:**
  ```
  Engagement_score = min(data_entry_score + challenge_score, 10)
  ```

## Important Notes

### Point Award Tracking
The function tracks incremental points gained in the `points_awarded` dictionary. However, there's a **bug** in the diet scoring section:

```python
# Line 1070: Should be 'Diet_points_gained' not 'Water_points_gained'
points_awarded['Water_points_gained'] = max(points_gained, 0)
```

### Time Range
All queries filter data within the same day:
- Start: 00:00:00 of the record date
- End: 23:59:59 of the record date

### Idempotency Considerations
The function can be called multiple times for different metrics:
- Previous scores are retrieved from the record
- New points are calculated incrementally
- Some calculations prevent duplicate point awards

### Record Updates
The function modifies the `record` object and calls `record.save()` at the end, persisting all calculated scores.

## Usage Examples

### Example 1: Update Diet and Fiber Points
```python
result = updateTotalPointDay(
    user=current_user,
    record=daily_record,
    is_Diet=True
)
print(f"Total points: {result['DayPoint']}")
print(f"Diet score: {result['Diet_score']}")
```

### Example 2: Update Multiple Metrics
```python
result = updateTotalPointDay(
    user=current_user,
    record=daily_record,
    is_Diet=True,
    is_Water=True,
    is_Step=True
)
```

### Example 3: Override Diet Score
```python
result = updateTotalPointDay(
    user=current_user,
    record=daily_record,
    isCall=25,  # Override with specific score
    is_Diet=True
)
```

## Dependencies

This function relies on the following:
- `get_user_info()`: Retrieves user profile information
- `get_calory_per_day()`: Calculates daily caloric needs
- `get_meal_values()`: Calculates meal percentage distribution
- `cal_eaten_nutritionFacts()`: Calculates nutrition facts from eating items
- `cal_diet_mistmatchScore()`: Scores meal adherence

### Models Used
- `Eating`: Food consumption records
- `Walking`: Step count records
- `Activities_log`: Exercise activity records
- `Water`: Water intake records
- `Sleep`: Sleep tracking records
- `co_message`: Communication records
- `Challenge`: User challenge participation

## Known Issues

1. **Bug in Line 1070**: `points_awarded['Water_points_gained']` should be `points_awarded['Diet_points_gained']`

2. **Inconsistent Point Award Logic**: The step scoring has complex conditional logic that may result in unexpected `-1` or `-2` values in points_awarded

3. **Sleep Query Efficiency**: Uses `.last()` on sleep records which may not be efficient; consider filtering by date range

4. **Thread Safety**: No locking mechanism; concurrent calls could lead to race conditions

## Related Documentation

- [Scoring System Documentation](./SCORING_SYSTEM_DOCUMENTATION.md)
- [Scoring Module](./scoring_system.md)
- User Profile and Health Metrics
- Daily Record Management

---

*Last Updated: October 9, 2025*

