# Hospital Integration Feature

## Overview

The Hospital Integration feature allows hospital systems to seamlessly create user accounts in the ZiLuck application for their patients without requiring SMS verification. This enables a smooth user experience where patients can access the ZiLuck app directly from their hospital's application.

## How It Works

### Traditional User Registration Flow
1. User calls `activation_code` API
2. User receives SMS verification code
3. User calls `sign` API with verification code
4. User account is created with profile

### Hospital Integration Flow
1. Hospital system calls `create_hospital_user` API
2. Basic user account is created immediately without SMS verification
3. User can access ZiLuck app using the returned token
4. Profile information can be added later through regular profile APIs

### Phase 1: Basic User Creation
This implementation focuses on creating only the essential user instance:
- User record with phone number and token
- Django user account
- Patient group assignment
- No profile information (first_name, last_name, etc.)

Profile information will be collected later when the user completes their profile setup.

## API Endpoint

**URL:** `POST /api/user/create/hospital/`

### Required Parameters
- `phone_number`: Iranian mobile number (format: 09XXXXXXXXX)
- `hospital_api_key`: Secure API key for hospital authentication

## Configuration

### Setting Up Hospital API Keys

1. Edit `ziluck_project/hospital_config.py`
2. Add your hospital API keys to the `HOSPITAL_API_KEYS` dictionary:

```python
# Single API key for all hospital integrations
HOSPITAL_API_KEY = "ziluck_hospital_integration_key_2024"
```
```

3. Add hospital information to `HOSPITAL_INFO`:

```python
HOSPITAL_INFO = {
    "default": {
        "name": "Hospital Integration",
        "code": "HOSP",
        "contact_email": "integration@ziluck.com",
        "enabled": True,
        "created_date": "2024-01-01",
        "max_users_per_day": 10000
    }
}
```
```

### Security Considerations

1. **API Key Security**: Use strong, unique API keys for each hospital
2. **HTTPS**: Always use HTTPS in production
3. **Rate Limiting**: Consider implementing rate limiting
4. **Logging**: All API calls are logged for security monitoring

## Testing

### Using the Test Script

1. Run the test script to verify the integration:

```bash
python script/test_hospital_integration.py
```

2. Update the test script with your actual API keys and server URL

### Manual Testing

You can test the API using curl or any HTTP client:

```bash
curl -X POST http://localhost:8000/api/user/create/hospital/ \
  -H "Content-Type: application/x-www-form-urlencoded" \
  -d "phone_number=09123456789" \
  -d "hospital_api_key=ziluck_hospital_integration_key_2024"
```

## Response Examples

### Successful User Creation

```json
{
    "status": "success",
    "message": "کاربر با موفقیت ایجاد شد",
    "data": {
        "user_id": 12345,
        "token": "abc123def456ghi789jkl012mno345pqr678stu901vwx234yz",
        "message": "User created successfully"
    }
}
```

### User Already Exists

```json
{
    "status": "success",
    "message": "کاربر قبلاً ثبت شده است",
    "data": {
        "user_id": 12345,
        "token": "abc123def456ghi789jkl012mno345pqr678stu901vwx234yz",
        "message": "User already exists and is registered"
    }
}
```

### Error Response

```json
{
    "status": "error",
    "message": "Invalid hospital API key",
    "error_code": 1005
}
```

## Integration Steps for Hospitals

1. **Get API Key**: Contact ZiLuck team to get your hospital API key
2. **Test Integration**: Use the test script to verify the integration works
3. **Implement in Hospital System**: Add the API call to your hospital application
4. **Go Live**: Deploy the integration to production
5. **Monitor**: Monitor API usage and logs

## Example Hospital Integration Code

### Python Example

```python
import requests

def create_ziluck_user(patient_phone):
    url = "https://your-ziluck-domain.com/api/user/create/hospital/"
    
    payload = {
        "phone_number": patient_phone,
        "hospital_api_key": "ziluck_hospital_integration_key_2024"
    }
    
    response = requests.post(url, data=payload)
    
    if response.status_code == 200:
        result = response.json()
        if result["status"] == "success":
            return result["data"]["token"]
        else:
            raise Exception(f"API Error: {result['message']}")
    else:
        raise Exception(f"HTTP Error: {response.status_code}")
```

### JavaScript Example

```javascript
async function createZiluckUser(patientPhone) {
    const url = 'https://your-ziluck-domain.com/api/user/create/hospital/';
    
    const formData = new FormData();
    formData.append('phone_number', patientPhone);
    formData.append('hospital_api_key', 'ziluck_hospital_integration_key_2024');
    
    const response = await fetch(url, {
        method: 'POST',
        body: formData
    });
    
    const result = await response.json();
    
    if (result.status === 'success') {
        return result.data.token;
    } else {
        throw new Error(`API Error: ${result.message}`);
    }
}
```

## Support

For questions or issues with the hospital integration:

1. Check the logs for error details
2. Review the API documentation
3. Contact the ZiLuck development team
4. Monitor the hospital integration dashboard (if available)

## Future Enhancements

- Webhook notifications for user creation events
- Bulk user creation API
- Hospital-specific user settings
- Integration analytics dashboard
- Automated user synchronization 