# مستندات پایگاه داده

## نمای کلی

برنامه uDiet از SQLite به عنوان پایگاه داده اصلی در محیط توسعه استفاده می‌کند و از MySQL برای محیط تولید پشتیبانی می‌کند. ساختار پایگاه داده برای پشتیبانی از ویژگی‌های نظارت بر سلامت، مدیریت رژیم غذایی، مدیریت کاربر و گزارش‌دهی طراحی شده است.

## نمودار ارتباط موجودیت‌ها

```mermaid
erDiagram
    User ||--o{ Activity : ثبت می‌کند
    User ||--o{ FoodLog : ثبت می‌کند
    User ||--o{ BloodSugar : ثبت می‌کند
    User ||--o{ Medication : مصرف می‌کند
    User ||--o{ Sleep : ثبت می‌کند
    User ||--o{ Weight : ثبت می‌کند
    User ||--o{ WaterIntake : ثبت می‌کند
    User ||--o{ Message : ارسال می‌کند
    User ||--o{ Message : دریافت می‌کند
    User ||--o{ Notification : دریافت می‌کند
    User ||--o{ UserSettings : دارد
    User ||--o{ Achievement : کسب می‌کند
    User ||--o{ Points : کسب می‌کند
    User ||--o{ DietPlan : دنبال می‌کند
    User ||--o{ FavoriteFood : دارد
    User ||--o{ FavoriteActivity : دارد
    User ||--o{ FavoriteDrug : دارد
    User ||--o{ Questionnaire : پاسخ می‌دهد
    Food ||--o{ FoodLog : در آن استفاده می‌شود
    Food ||--o{ DietPlan : در آن گنجانده شده
    Activity ||--o{ ActivityLog : نوع آن است
    Drug ||--o{ MedicationLog : نوع آن است
    Insulin ||--o{ InsulinLog : نوع آن است
    GLP1 ||--o{ GLP1Log : نوع آن است
    DietPlan ||--o{ Meal : شامل
    Meal ||--o{ MealFood : شامل
```

## مدل‌های اصلی

### کاربر
```sql
CREATE TABLE user (
    id INTEGER PRIMARY KEY,
    phone_number VARCHAR(20) UNIQUE NOT NULL,
    password_hash VARCHAR(255) NOT NULL,
    name VARCHAR(100),
    email VARCHAR(255),
    profile_image VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    last_login TIMESTAMP,
    is_active BOOLEAN DEFAULT TRUE,
    role VARCHAR(20) DEFAULT 'user'
);
```

### تنظیمات کاربر
```sql
CREATE TABLE user_settings (
    id INTEGER PRIMARY KEY,
    user_id INTEGER REFERENCES user(id),
    notifications_enabled BOOLEAN DEFAULT TRUE,
    language VARCHAR(10) DEFAULT 'fa',
    timezone VARCHAR(50),
    measurement_system VARCHAR(10) DEFAULT 'metric',
    theme VARCHAR(10) DEFAULT 'light',
    quiet_hours_start TIME,
    quiet_hours_end TIME,
    FOREIGN KEY (user_id) REFERENCES user(id)
);
```

### معیارهای سلامت

#### فعالیت
```sql
CREATE TABLE activity (
    id INTEGER PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    category VARCHAR(50),
    calories_per_hour FLOAT,
    description TEXT
);

CREATE TABLE activity_log (
    id INTEGER PRIMARY KEY,
    user_id INTEGER REFERENCES user(id),
    activity_id INTEGER REFERENCES activity(id),
    duration INTEGER NOT NULL, -- به دقیقه
    calories_burned FLOAT,
    date TIMESTAMP NOT NULL,
    notes TEXT,
    FOREIGN KEY (user_id) REFERENCES user(id),
    FOREIGN KEY (activity_id) REFERENCES activity(id)
);
```

#### غذا
```sql
CREATE TABLE food (
    id INTEGER PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    category VARCHAR(50),
    calories FLOAT,
    protein FLOAT,
    carbs FLOAT,
    fat FLOAT,
    fiber FLOAT,
    sugar FLOAT,
    serving_size FLOAT,
    serving_unit VARCHAR(20),
    description TEXT
);

CREATE TABLE food_log (
    id INTEGER PRIMARY KEY,
    user_id INTEGER REFERENCES user(id),
    food_id INTEGER REFERENCES food(id),
    quantity FLOAT NOT NULL,
    unit VARCHAR(20),
    meal_type VARCHAR(20),
    date TIMESTAMP NOT NULL,
    notes TEXT,
    FOREIGN KEY (user_id) REFERENCES user(id),
    FOREIGN KEY (food_id) REFERENCES food(id)
);
```

#### قند خون
```sql
CREATE TABLE blood_sugar_log (
    id INTEGER PRIMARY KEY,
    user_id INTEGER REFERENCES user(id),
    value FLOAT NOT NULL,
    unit VARCHAR(10),
    measurement_type VARCHAR(20),
    date TIMESTAMP NOT NULL,
    notes TEXT,
    image_url VARCHAR(255),
    FOREIGN KEY (user_id) REFERENCES user(id)
);
```

#### دارو
```sql
CREATE TABLE drug (
    id INTEGER PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    category VARCHAR(50),
    description TEXT
);

CREATE TABLE medication_log (
    id INTEGER PRIMARY KEY,
    user_id INTEGER REFERENCES user(id),
    drug_id INTEGER REFERENCES drug(id),
    dosage FLOAT NOT NULL,
    unit VARCHAR(20),
    date TIMESTAMP NOT NULL,
    notes TEXT,
    FOREIGN KEY (user_id) REFERENCES user(id),
    FOREIGN KEY (drug_id) REFERENCES drug(id)
);
```

#### خواب
```sql
CREATE TABLE sleep_log (
    id INTEGER PRIMARY KEY,
    user_id INTEGER REFERENCES user(id),
    start_time TIMESTAMP NOT NULL,
    end_time TIMESTAMP NOT NULL,
    quality VARCHAR(20),
    notes TEXT,
    FOREIGN KEY (user_id) REFERENCES user(id)
);
```

#### وزن
```sql
CREATE TABLE weight_log (
    id INTEGER PRIMARY KEY,
    user_id INTEGER REFERENCES user(id),
    weight FLOAT NOT NULL,
    unit VARCHAR(10),
    date TIMESTAMP NOT NULL,
    notes TEXT,
    FOREIGN KEY (user_id) REFERENCES user(id)
);
```

#### مصرف آب
```sql
CREATE TABLE water_intake_log (
    id INTEGER PRIMARY KEY,
    user_id INTEGER REFERENCES user(id),
    amount FLOAT NOT NULL,
    unit VARCHAR(10),
    date TIMESTAMP NOT NULL,
    FOREIGN KEY (user_id) REFERENCES user(id)
);
```

### مدیریت رژیم غذایی

#### برنامه رژیم غذایی
```sql
CREATE TABLE diet_plan (
    id INTEGER PRIMARY KEY,
    user_id INTEGER REFERENCES user(id),
    name VARCHAR(100) NOT NULL,
    type VARCHAR(50),
    description TEXT,
    start_date DATE NOT NULL,
    end_date DATE,
    total_calories FLOAT,
    total_protein FLOAT,
    total_carbs FLOAT,
    total_fat FLOAT,
    FOREIGN KEY (user_id) REFERENCES user(id)
);

CREATE TABLE meal (
    id INTEGER PRIMARY KEY,
    diet_plan_id INTEGER REFERENCES diet_plan(id),
    name VARCHAR(100) NOT NULL,
    time TIME NOT NULL,
    FOREIGN KEY (diet_plan_id) REFERENCES diet_plan(id)
);

CREATE TABLE meal_food (
    id INTEGER PRIMARY KEY,
    meal_id INTEGER REFERENCES meal(id),
    food_id INTEGER REFERENCES food(id),
    quantity FLOAT NOT NULL,
    unit VARCHAR(20),
    FOREIGN KEY (meal_id) REFERENCES meal(id),
    FOREIGN KEY (food_id) REFERENCES food(id)
);
```

### پیام‌رسانی و اعلان‌ها

#### پیام
```sql
CREATE TABLE message (
    id INTEGER PRIMARY KEY,
    sender_id INTEGER REFERENCES user(id),
    recipient_id INTEGER REFERENCES user(id),
    message TEXT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    is_read BOOLEAN DEFAULT FALSE,
    FOREIGN KEY (sender_id) REFERENCES user(id),
    FOREIGN KEY (recipient_id) REFERENCES user(id)
);
```

#### اعلان
```sql
CREATE TABLE notification (
    id INTEGER PRIMARY KEY,
    user_id INTEGER REFERENCES user(id),
    title VARCHAR(100) NOT NULL,
    message TEXT NOT NULL,
    type VARCHAR(50),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    is_read BOOLEAN DEFAULT FALSE,
    FOREIGN KEY (user_id) REFERENCES user(id)
);
```

### دستاوردها و امتیازات

#### دستاورد
```sql
CREATE TABLE achievement (
    id INTEGER PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    description TEXT,
    icon_url VARCHAR(255),
    points INTEGER DEFAULT 0
);

CREATE TABLE user_achievement (
    id INTEGER PRIMARY KEY,
    user_id INTEGER REFERENCES user(id),
    achievement_id INTEGER REFERENCES achievement(id),
    earned_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES user(id),
    FOREIGN KEY (achievement_id) REFERENCES achievement(id)
);
```

#### امتیازات
```sql
CREATE TABLE points (
    id INTEGER PRIMARY KEY,
    user_id INTEGER REFERENCES user(id),
    amount INTEGER NOT NULL,
    reason VARCHAR(100),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES user(id)
);
```

### پرسشنامه‌ها

```sql
CREATE TABLE questionnaire (
    id INTEGER PRIMARY KEY,
    title VARCHAR(100) NOT NULL,
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE question (
    id INTEGER PRIMARY KEY,
    questionnaire_id INTEGER REFERENCES questionnaire(id),
    text TEXT NOT NULL,
    type VARCHAR(50),
    options TEXT,
    FOREIGN KEY (questionnaire_id) REFERENCES questionnaire(id)
);

CREATE TABLE user_answer (
    id INTEGER PRIMARY KEY,
    user_id INTEGER REFERENCES user(id),
    question_id INTEGER REFERENCES question(id),
    answer TEXT NOT NULL,
    answered_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES user(id),
    FOREIGN KEY (question_id) REFERENCES question(id)
);
```

## ایندکس‌ها

### ایندکس‌های اصلی
```sql
-- ایندکس‌های کاربر
CREATE INDEX idx_user_phone ON user(phone_number);
CREATE INDEX idx_user_email ON user(email);

-- ایندکس‌های لاگ‌ها
CREATE INDEX idx_activity_log_user_date ON activity_log(user_id, date);
CREATE INDEX idx_food_log_user_date ON food_log(user_id, date);
CREATE INDEX idx_blood_sugar_log_user_date ON blood_sugar_log(user_id, date);
CREATE INDEX idx_medication_log_user_date ON medication_log(user_id, date);
CREATE INDEX idx_sleep_log_user_date ON sleep_log(user_id, date);
CREATE INDEX idx_weight_log_user_date ON weight_log(user_id, date);
CREATE INDEX idx_water_intake_log_user_date ON water_intake_log(user_id, date);

-- ایندکس‌های پیام‌رسانی
CREATE INDEX idx_message_sender ON message(sender_id);
CREATE INDEX idx_message_recipient ON message(recipient_id);
CREATE INDEX idx_notification_user ON notification(user_id);
```

## مهاجرت‌ها

### ایجاد جداول
```sql
-- ایجاد جداول اصلی
python manage.py makemigrations

-- اعمال مهاجرت‌ها
python manage.py migrate
```

### به‌روزرسانی ساختار
```sql
-- ایجاد مهاجرت جدید
python manage.py makemigrations

-- بررسی وضعیت مهاجرت‌ها
python manage.py showmigrations

-- اعمال مهاجرت‌های جدید
python manage.py migrate
```

## پشتیبان‌گیری

### پشتیبان‌گیری خودکار
```bash
# پشتیبان‌گیری روزانه
0 0 * * * /usr/bin/python3 manage.py dumpdata > backup_$(date +\%Y\%m\%d).json

# پشتیبان‌گیری هفتگی
0 0 * * 0 /usr/bin/python3 manage.py dumpdata > backup_weekly_$(date +\%Y\%m\%d).json
```

### بازیابی
```bash
# بازیابی از پشتیبان
python manage.py loaddata backup_20240315.json
```

## بهینه‌سازی

### بهینه‌سازی پرس‌وجوها
```sql
-- استفاده از EXPLAIN برای تحلیل پرس‌وجوها
EXPLAIN SELECT * FROM food_log WHERE user_id = 1 AND date >= '2024-03-01';

-- بهینه‌سازی پرس‌وجوهای پیچیده
CREATE INDEX idx_food_log_complex ON food_log(user_id, date, meal_type);
```

### نگهداری
```sql
-- بهینه‌سازی جداول
VACUUM;

-- تحلیل جداول
ANALYZE;

-- بازسازی ایندکس‌ها
REINDEX;
```

## امنیت

### رمزنگاری
```sql
-- رمزنگاری داده‌های حساس
ALTER TABLE user ADD COLUMN encrypted_data BLOB;

-- مدیریت کلیدها
CREATE TABLE encryption_keys (
    id INTEGER PRIMARY KEY,
    key_name VARCHAR(100) NOT NULL,
    key_value BLOB NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

### دسترسی
```sql
-- ایجاد کاربر با دسترسی محدود
CREATE USER 'udiet_readonly'@'localhost' IDENTIFIED BY 'password';
GRANT SELECT ON udiet.* TO 'udiet_readonly'@'localhost';

-- محدودیت دسترسی به جداول حساس
REVOKE ALL ON user FROM 'udiet_readonly'@'localhost';
```

## مانیتورینگ

### پرس‌وجوهای مانیتورینگ
```sql
-- بررسی حجم داده‌ها
SELECT table_name, COUNT(*) as row_count 
FROM information_schema.tables 
WHERE table_schema = 'udiet';

-- بررسی عملکرد ایندکس‌ها
SELECT * FROM pg_stat_user_indexes 
WHERE schemaname = 'udiet';

-- بررسی پرس‌وجوهای کند
SELECT * FROM pg_stat_activity 
WHERE state = 'active' 
AND query NOT ILIKE '%pg_stat_activity%';
```

### هشدارها
```sql
-- تنظیم هشدار برای حجم داده
CREATE OR REPLACE FUNCTION check_table_size()
RETURNS trigger AS $$
BEGIN
    IF (SELECT COUNT(*) FROM food_log) > 1000000 THEN
        RAISE NOTICE 'هشدار: حجم جدول food_log از حد مجاز فراتر رفته است';
    END IF;
    RETURN NEW;
END;
$$ LANGUAGE plpgsql;
``` 