# Check for Discount Hospital API Documentation

## Overview

The `check_for_discount_hospital` API endpoint validates discount codes for hospital integrations and calculates discounted prices for payment plans. This endpoint is specifically designed for hospital systems to check discount eligibility before processing payments.

## Endpoint Information

- **URL**: `/payment/check_for_discount_hospital/`
- **Method**: `POST`
- **Content-Type**: `application/x-www-form-urlencoded`

## Request Parameters

### Required Parameters

| Parameter | Type | Description |
|-----------|------|-------------|
| `hospital_api_key` | string | Valid hospital API key for authentication |
| `plan_id` | integer | Database ID of the payment plan |

### Optional Parameters

| Parameter | Type | Description |
|-----------|------|-------------|
| `discount_code` | string | Discount code to validate and apply (case-insensitive) |

## Request Example

```bash
curl -X POST https://api.ziluck.info/payment/hospital/check/discount/ \
  -d "hospital_api_key=Z9xd3NikanhospitalQ8zYpR7G5wK2uHvM1R" \
  -d "plan_id=1" \
  -d "discount_code=HOSPITAL20"
```


## Response Format

### Success Response (No Discount Code)

```json
{
  "status": "OK",
  "message": "discount code",
  "error_code": "1000",
  "data": [{
    "message": "کد تخفیفی وارد نشده است.",
    "original_price": 500000,
    "price_with_promotion": 500000
  }]
}
```

### Success Response (Valid Discount Code)

```json
{
  "status": "OK",
  "message": "discount code",
  "error_code": "1000",
  "data": [{
    "message": "تبریک! خرید شما شامل 20 درصد تخفیف خواهد شد!",
    "original_price": 500000,
    "price_with_promotion": 400000
  }]
}
```

### Error Response

```json
{
  "status": "Error",
  "message": "Error description",
  "error_code": "1308",
  "data": []
}
```

## Error Codes

| Error Code | Description | Possible Causes |
|------------|-------------|-----------------|
| `1003` | Invalid Argument | Missing required parameters (`hospital_api_key` or `plan_id`) |
| `1308` | Invalid Hospital API Key | Invalid or disabled hospital API key |
| `1309` | Invalid Plan ID | Plan ID is not a valid integer or plan doesn't exist |
| `1310` | Discount Code Not Defined for Hospital | Discount code exists but is not associated with the hospital's group |
| `1000` | Internal Error | Unexpected server error |

## Business Logic

### 1. Authentication
- Validates the hospital API key using `validate_hospital_api_key()`
- Retrieves hospital information including the discount group code

### 2. Plan Validation
- Converts `plan_id` to integer
- Queries `PaymentPlans` model using `plan_database_id`
- Returns error if plan doesn't exist

### 3. Discount Code Processing

#### Without Discount Code
- Returns original plan price
- Message: "کد تخفیفی وارد نشده است."

#### With Discount Code
1. **Code Validation**: Searches for discount code in `Discount` model (case-insensitive)
2. **Group Matching**: Validates that the discount code's `ziluck_group` matches the hospital's `nikan_discount_group_code`
3. **Price Calculation**: Applies percentage discount to original price
4. **Success Message**: Returns congratulatory message with discount percentage

### 4. Price Calculation Formula

```python
price_with_promotion = float(plan.price) * (100 - percentage) / 100
```

## Database Models

### PaymentPlans
- **Field**: `plan_database_id` (used for lookup)
- **Field**: `price` (original plan price)

### Discount
- **Field**: `code` (discount code, case-insensitive)
- **Field**: `percentage` (discount percentage)
- **Field**: `ziluck_group` (foreign key to ZiLuckGroup)

### ZiLuckGroup
- **Field**: `name` (group name, must match hospital's `nikan_discount_group_code`)

## Hospital Configuration

The API uses hospital configuration from `ziluck_project/hospital_config.py`:

```python
HOSPITAL_INFO = {
    "Z9xd3NikanhospitalQ8zYpR7G5wK2uHvM1R": {
        "name": "Nikan Hospital",
        "code": "NIKAN",
        "nikan_discount_group_code": "NIKAN_PROMOTION",
        "enabled": True
    }
}
```
