# Blood Test Report API - Server Setup Guide

## Overview
The Blood Test Report API (`generate_blood_test_report`) processes PDF and image files of blood test reports using OCR and AI analysis. This document outlines the required dependencies and setup steps for deployment on a live server.

## Required Python Dependencies

Install these packages on your live server:

```bash
pip install PyMuPDF>=1.23.0  # For PDF processing (fitz module)
pip install Pillow>=10.0.0   # For image processing
pip install google-cloud-vision>=3.4.0  # For OCR
pip install google-cloud-aiplatform  # For Vertex AI
pip install google-generativeai  # For Gemini API fallback
pip install openai>=1.0.0  # For OpenAI fallback
```

Or add to your `requirements.txt`:
```
PyMuPDF>=1.23.0
Pillow>=10.0.0
google-cloud-vision>=3.4.0
google-cloud-aiplatform>=1.34.0
google-generativeai>=0.3.0
openai>=1.0.0
```

## System Requirements

### 1. Temporary Directory Access
The API creates temporary directories for processing files. Ensure:
- Write permissions for the application user in the temp directory
- Sufficient disk space (recommend at least 1GB free)

### 2. Google Cloud Credentials
The API requires Google Cloud credentials for OCR and AI services.

**Location**: `APIs/modules/healiomenew-94ee24f33f41.json`

Ensure this file:
- Exists on the live server
- Has correct file permissions (readable by application user)
- Contains valid service account credentials with these APIs enabled:
  - Cloud Vision API
  - Vertex AI API

### 3. Environment Variables (Optional)
You can override credential paths with environment variables:

```bash
export GOOGLE_APPLICATION_CREDENTIALS="/path/to/service-account.json"
```

## Testing the Setup

### 1. Check Dependencies
Test if all dependencies are installed:

```python
# Run in Django shell (python manage.py shell)
import fitz
from PIL import Image
from google.cloud import vision
import vertexai
import google.generativeai as genai
import openai

print("✅ All dependencies installed successfully")
```

### 2. Test PDF Processing
```python
from APIs.modules.BloodTestReport_generat import convert_pdf_to_images
import tempfile

# Test with a sample PDF
temp_dir = tempfile.mkdtemp()
try:
    images = convert_pdf_to_images("path/to/test.pdf", temp_dir)
    print(f"✅ PDF processing works: {len(images)} pages converted")
except Exception as e:
    print(f"❌ Error: {e}")
```

### 3. Test OCR
```python
from APIs.modules.BloodTestReport_generat import run_ocr_on_images

# Test with sample images
text = run_ocr_on_images(["path/to/test_image.jpg"])
print(f"✅ OCR works: {len(text)} characters extracted")
```

## Common Issues and Solutions

### Issue 1: "خطا در پردازش فایل" (Error processing file)
**Cause**: PDF conversion failed (empty image_paths)

**Solutions**:
1. Check if PyMuPDF is installed: `pip show PyMuPDF`
2. Verify temp directory has write permissions
3. Check server logs for detailed error messages

### Issue 2: "PyMuPDF library not available"
**Cause**: PyMuPDF not installed or import failing

**Solution**:
```bash
pip uninstall PyMuPDF pymupdf
pip install PyMuPDF --no-cache-dir
```

### Issue 3: Google Cloud Vision Errors
**Cause**: Invalid or missing credentials

**Solutions**:
1. Verify credentials file exists: `ls -l APIs/modules/healiomenew-94ee24f33f41.json`
2. Check file permissions: `chmod 644 APIs/modules/healiomenew-94ee24f33f41.json`
3. Verify APIs are enabled in Google Cloud Console:
   - https://console.cloud.google.com/apis/library/vision.googleapis.com
   - https://console.cloud.google.com/apis/library/aiplatform.googleapis.com

### Issue 4: Image File Validation Errors
**Cause**: Pillow not installed or corrupted image files

**Solutions**:
1. Install Pillow: `pip install Pillow --upgrade`
2. Test image validation:
   ```python
   from PIL import Image
   img = Image.open("test.jpg")
   img.verify()
   ```

## Deployment Checklist

- [ ] All Python dependencies installed
- [ ] Google Cloud credentials file in place
- [ ] Credentials file has correct permissions
- [ ] Vision API enabled in GCP project
- [ ] Vertex AI API enabled in GCP project
- [ ] Temp directory is writable
- [ ] Sufficient disk space available
- [ ] Test with sample PDF and image files
- [ ] Monitor server logs for detailed error messages

## Server Log Monitoring

The API logs detailed information. Monitor these logs:

```bash
# Watch Django logs
tail -f /path/to/django/logs/django.log

# Or check system logs
journalctl -u your-django-service -f
```

Look for log entries starting with:
- `[API]` - Main API flow
- `[INFO]` - PDF/OCR processing steps
- `[ERROR]` - Error details

## Performance Considerations

- PDF conversion can take 2-5 seconds per page
- OCR processing takes 1-3 seconds per page
- AI analysis takes 5-15 seconds
- Total processing time: 10-60 seconds depending on document size

Consider:
- Implementing request timeouts (60-120 seconds)
- Adding progress indicators for users
- Implementing async processing for large documents

## API Response Examples

### Success Response
```json
{
    "status": "Success",
    "message": "گزارش با موفقیت ایجاد شد",
    "data": {
        "structured_data": {...},
        "test_analysis": {...},
        "health_report_markdown": "...",
        "user_id": 123,
        "processed_at": "2025-11-03T10:30:00"
    }
}
```

### Error Responses with Improved Messages

**Missing Dependencies:**
```json
{
    "status": "Error",
    "message": "کتابخانه‌های زیر روی سرور نصب نشده‌اند: PyMuPDF, Pillow",
    "error_code": "-1"
}
```

**PDF Conversion Error:**
```json
{
    "status": "Error",
    "message": "خطا: کتابخانه PyMuPDF روی سرور نصب نشده است",
    "error_code": "-1"
}
```

**Invalid Image:**
```json
{
    "status": "Error",
    "message": "فایل تصویری معتبر نیست یا خراب است",
    "error_code": "-1"
}
```

## Support

If issues persist after following this guide:
1. Check the detailed logs on the server
2. Verify all dependencies are correctly installed
3. Test each component individually using the test scripts above
4. Ensure Google Cloud APIs are enabled and credentials are valid

## Security Notes

- Keep the Google Cloud credentials file secure
- Don't commit credentials to version control
- Rotate service account keys periodically
- Monitor API usage in Google Cloud Console


