# Hospital Payment Simulation API

## Overview

The Hospital Payment Simulation API allows hospital systems to simulate successful payment actions for users without requiring actual payment verification. This API performs the exact same actions that occur when a user's payment is successfully verified through the standard payment gateways.

## Endpoint

```
POST /payment/simulate/successful-payment/
```

## Purpose

This API is designed for hospital integration scenarios where:
- Users have already paid through hospital systems
- Hospital needs to activate user accounts in ZiLuck
- No actual payment gateway verification is required
- Same user experience as successful payment verification

## Authentication

The API uses hospital API key authentication. The API key must be provided in the request parameters.

## Request Parameters

| Parameter | Type | Required | Description | Example |
|-----------|------|----------|-------------|---------|
| `hospital_api_key` | string | Yes | Valid hospital API key for authentication | `"Z9xd3NikanhospitalQ8zYpR7G5wK2uHvM1R"` |
| `phone_number` | string | Yes | User's phone number (format: 09XXXXXXXXX) | `"09123456789"` |
| `plan` | integer | Yes | Plan number (1-8) | `1` |

### Plan Numbers

| Plan | Type | Credit Days | Description |
|------|------|-------------|-------------|
| 1 | Silver | 30 | Monthly plan |
| 2 | Golden | 90 | Three month plan |
| 3 | Platinum | 30 | Monthly plan |
| 4 | Bronze | 90 | Three month plan |
| 5 | Monthly | 30 | Monthly plan |
| 6 | Quarterly | 90 | Three month plan |
| 7 | Special Monthly | 30 | Monthly plan with special features |
| 8 | Special Quarterly | 90 | Three month plan with special features |

## Request Example

### cURL
```bash
curl -X POST \
  https://your-domain.com/payment/simulate/successful-payment/ \
  -H 'Content-Type: application/x-www-form-urlencoded' \
  -d 'hospital_api_key=Z9xd3NikanhospitalQ8zYpR7G5wK2uHvM1R&phone_number=09123456789&plan=1'
```

### Python
```python
import requests

url = "https://your-domain.com/payment/simulate/successful-payment/"
data = {
    "hospital_api_key": "Z9xd3NikanhospitalQ8zYpR7G5wK2uHvM1R",
    "phone_number": "09123456789",
    "plan": 1
}

response = requests.post(url, data=data)
print(response.json())
```

### JavaScript
```javascript
const formData = new FormData();
formData.append('hospital_api_key', 'Z9xd3NikanhospitalQ8zYpR7G5wK2uHvM1R');
formData.append('phone_number', '09123456789');
formData.append('plan', '1');

fetch('/payment/simulate/successful-payment/', {
    method: 'POST',
    body: formData
})
.then(response => response.json())
.then(data => console.log(data));
```

## Response Format

### Success Response (200)

```json
{
    "status": "success",
    "message": "Payment simulation completed successfully",
    "data": {
        "user_id": 12345,
        "phone_number": "09123456789",
        "plan": 1,
        "credit_days_added": 30,
        "therapist_assigned": 67890,
        "hospital_name": "Hospital Integration",
        "actions_performed": [
            "therapist_selection_created",
            "credit_added",
            "welcome_message_sent",
            "notifications_sent"
        ]
    }
}
```

### Error Responses

#### Missing Parameter (400)
```json
{
    "error": "Missing required parameter: hospital_api_key",
    "status": "error"
}
```

#### Invalid API Key (401)
```json
{
    "error": "Invalid hospital API key",
    "status": "error"
}
```

#### Invalid Plan (400)
```json
{
    "error": "Invalid plan number. Must be between 1-8",
    "status": "error"
}
```

#### User Not Found (404)
```json
{
    "error": "User not found with the provided phone number",
    "status": "error"
}
```

#### Server Error (500)
```json
{
    "error": "Error processing payment simulation: [error details]",
    "status": "error"
}
```

## Actions Performed

When the API is called successfully, it performs the following actions (identical to successful payment verification):

### 1. Therapist Selection
- Creates or updates `TherapistSelection` record
- Assigns the same therapist if user had previous sessions
- Assigns default doctor if no previous therapist exists
- Updates user profile with assigned therapist

### 2. Credit Addition
- Adds credit days based on plan type
- Monthly plans (1, 3, 5, 7): 30 days
- Quarterly plans (2, 4, 6, 8): 90 days

### 3. Welcome Message
- Sends personalized welcome message to user
- Special message for plans 7 and 8 (diet plan notification)
- Message is sent from default doctor to user

### 4. Notifications
- Sends Telegram notification about successful purchase
- Sends receipt email if user has email address
- Reference number set to "HOSPITAL_INTEGRATION"

## Security Considerations

1. **API Key Validation**: Hospital API key must be valid and enabled
2. **User Verification**: User must exist in the system with provided phone number
3. **Plan Validation**: Plan number must be within valid range (1-8)
4. **Audit Trail**: All actions are logged and can be tracked

## Rate Limiting

The API follows the same rate limiting as other hospital integration endpoints:
- 60 requests per minute
- 1000 requests per hour
- 10000 requests per day

## Error Handling

The API includes comprehensive error handling:
- Parameter validation
- User existence verification
- Database operation error handling
- Graceful fallbacks for optional operations

## Integration Notes

1. **Idempotency**: The API can be called multiple times safely
2. **User Experience**: Identical to actual payment verification
3. **Hospital Tracking**: All requests are logged with hospital information
4. **Fallback Handling**: Continues operation even if some optional features fail

## Testing

### Test Cases

1. **Valid Request**: All parameters correct
2. **Invalid API Key**: Wrong or expired key
3. **Missing Parameters**: One or more required parameters missing
4. **Invalid Plan**: Plan number outside valid range
5. **Non-existent User**: Phone number not in system
6. **User Without Profile**: User exists but no profile
7. **User Without Email**: User exists but no email address

### Test Data

```bash
# Valid test case
curl -X POST \
  https://your-domain.com/payment/simulate/successful-payment/ \
  -d 'hospital_api_key=Z9xd3NikanhospitalQ8zYpR7G5wK2uHvM1R&phone_number=09999999999&plan=1'

# Invalid API key test
curl -X POST \
  https://your-domain.com/payment/simulate/successful-payment/ \
  -d 'hospital_api_key=INVALID_KEY&phone_number=09999999999&plan=1'
```

## Support

For technical support or questions about this API:
- Contact: integration@ziluck.com
- Documentation: [Hospital Integration Guide](../hospital_integration.md)
- API Status: [Status Page](https://status.ziluck.com) 