# Hospital Integration API

## Overview

The Hospital Integration API allows hospital systems to create user instances in the ZiLuck application without requiring SMS verification. This enables seamless integration for patients who are already registered in the hospital system.

## API Endpoint

### Create Hospital User

**Endpoint:** `POST /api/user/create/hospital/`

**Description:** Creates a new user instance for hospital patients without SMS verification.

#### Request Parameters

| Parameter | Type | Required | Format | Description |
|-----------|------|----------|--------|-------------|
| `phone_number` | string | Yes | `^09[0-9]{9}$` | Iranian mobile phone number |
| `hospital_api_key` | string | Yes | `^[a-zA-Z0-9_]{20,64}$` | Hospital system API key for authentication |

#### Request Example

```json
{
    "phone_number": "09123456789",
    "hospital_api_key": "ziluck_hospital_integration_key_2024"
}
```

#### Response Format

**Success Response (200 OK):**

```json
{
    "status": "success",
    "message": "کاربر با موفقیت ایجاد شد",
    "data": {
        "user_id": 12345,
        "token": "abc123def456ghi789jkl012mno345pqr678stu901vwx234yz",
        "message": "User created successfully"
    }
}
```

**User Already Exists Response:**

```json
{
    "status": "success",
    "message": "کاربر قبلاً ثبت شده است",
    "data": {
        "user_id": 12345,
        "token": "abc123def456ghi789jkl012mno345pqr678stu901vwx234yz",
        "message": "User already exists and is registered"
    }
}
```

**Error Response (400 Bad Request):**

```json
{
    "status": "error",
    "message": "Invalid hospital API key",
    "error_code": 1005
}
```

#### Error Codes

| Code | Message | Description |
|------|---------|-------------|
| 1005 | Invalid input, not authorized | Invalid parameters or API key |
| 1007 | Internal server error | Server-side error |

## Security Considerations

1. **API Key Validation**: The hospital API key must be validated against your hospital system's authentication mechanism.
2. **HTTPS**: All API calls should be made over HTTPS.
3. **Rate Limiting**: Consider implementing rate limiting to prevent abuse.
4. **Logging**: All API calls are logged for security and debugging purposes.

## Implementation Notes

1. **User Creation Flow**: The API creates only the basic user instance:
   - Main User record (no profile information)
   - Django User account
   - Patient group assignment
   - Authentication token

2. **Existing User Handling**: If a user with the same phone number already exists:
   - If registered: Returns existing user information
   - If not registered: Marks as registered and generates new token

3. **Token Generation**: Uses the same algorithm as the `sign` API:
   - 30-character random string using letters and digits
   - Generated using `generateRandomString(token_alphabet, 30)`

4. **Hospital Tracking**: All hospitals use a single API key for tracking:
   - All requests are logged with hospital integration information
   - Rate limiting is configured globally for all hospitals
   - Simple and unified authentication for all hospital integrations

5. **Profile Creation**: Profile information is not created at this stage.
   Users will need to complete their profile through the regular profile update APIs.

## Integration Steps

1. **Configure API Key**: Replace `"your_hospital_api_key_here"` in the code with your actual hospital API key validation logic.

2. **Test Integration**: Use the API endpoint to create test users and verify the integration works correctly.

3. **Production Deployment**: Deploy the updated code to production environment.

4. **Monitor Usage**: Monitor API usage and logs to ensure proper functioning.

## Example Integration Code

```python
import requests

def create_hospital_patient(patient_phone):
    url = "https://your-domain.com/api/user/create/hospital/"
    
    payload = {
        "phone_number": patient_phone,
        "hospital_api_key": "ziluck_hospital_integration_key_2024"
    }
    
    response = requests.post(url, data=payload)
    
    if response.status_code == 200:
        result = response.json()
        if result["status"] == "success":
            return result["data"]["token"]
        else:
            raise Exception(f"API Error: {result['message']}")
    else:
        raise Exception(f"HTTP Error: {response.status_code}")
``` 