# Authentication Flow API Documentation

## Overview

The authentication system follows a 3-step flow to determine whether a user should login or register, then authenticate them accordingly. This flow ensures secure user authentication while providing a seamless experience.

## Authentication Flow

```
1. Request Activation Code → 2. Check Registration Status → 3. Login or Register
```

---

## API Endpoints

### 1. Request Activation Code

**Endpoint:**
```
POST /api/v2/user/activation_code/
```

**Description:**
Requests a verification code to be sent to the user's phone number. This is the first step in the authentication flow.

**Headers:**
```
Content-Type: application/json
```

**Request Body:**
| Name          | Type   | Required | Description           |
|---------------|--------|----------|-----------------------|
| phone_number  | string | Yes      | User's phone number   |

**Request Example:**
```json
{
  "phone_number": "+989123456789"
}
```

**Response Example:**
```json
{
  "status": "success",
  "message": "Activation code sent successfully",
  "data": {
    "temp_token": "eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9...",
    "expires_in": 300
  }
}
```

**Error Codes:**
| Code | Description                    |
|------|--------------------------------|
| 400  | Invalid phone number format    |
| 429  | Too many requests (rate limit) |
| 500  | SMS service unavailable        |

---

### 2. Check Registration Status

**Endpoint:**
```
POST /api/v2/user/check/registration/
```

**Description:**
Checks whether a user with the given phone number has been previously registered. This determines the next step in the authentication flow.

**Headers:**
```
Content-Type: application/json
```

**Request Body:**
| Name          | Type   | Required | Description           |
|---------------|--------|----------|-----------------------|
| phone_number  | string | Yes      | User's phone number   |

**Request Example:**
```json
{
  "phone_number": "+989123456789"
}
```

**Response Example:**
```json
{
  "status": "success",
  "data": {
    "is_registered": true,
    "user_exists": true
  }
}
```

**Response Values:**
- `is_registered: true` → User exists, proceed to **Login**
- `is_registered: false` → User doesn't exist, proceed to **Register**

**Error Codes:**
| Code | Description                    |
|------|--------------------------------|
| 400  | Invalid phone number format    |
| 404  | Phone number not found        |

---

### 3. User Login

**Endpoint:**
```
POST /api/v2/user/login/
```

**Description:**
Authenticates an existing user using the verification code, temporary token, and phone number. Returns a permanent user token for application access.

**Headers:**
```
Content-Type: application/json
```

**Request Body:**
| Name              | Type   | Required | Description                    |
|-------------------|--------|----------|--------------------------------|
| verification_code | string | Yes      | SMS verification code          |
| temp_token        | string | Yes      | Temporary token from step 1    |
| phone_number      | string | Yes      | User's phone number            |

**Request Example:**
```json
{
  "verification_code": "123456",
  "temp_token": "eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9...",
  "phone_number": "+989123456789"
}
```

**Response Example:**
```json
{
  "status": "success",
  "message": "Login successful",
  "data": {
    "token": "eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9...",
    "user_id": 12345,
    "expires_in": 86400,
    "refresh_token": "refresh_token_here"
  }
}
```

**Error Codes:**
| Code | Description                    |
|------|--------------------------------|
| 400  | Invalid verification code      |
| 401  | Invalid temporary token        |
| 404  | User not found                |
| 422  | Verification code expired      |

---

### 4. User Registration

**Endpoint:**
```
POST /api/v2/user/sign/
```

**Description:**
Registers a new user using the verification code, temporary token, and phone number. Returns a permanent user token for application access.

**Headers:**
```
Content-Type: application/json
```

**Request Body:**
| Name              | Type   | Required | Description                    |
|-------------------|--------|----------|--------------------------------|
| verification_code | string | Yes      | SMS verification code          |
| temp_token        | string | Yes      | Temporary token from step 1    |
| phone_number      | string | Yes      | User's phone number            |

**Request Example:**
```json
{
  "verification_code": "123456",
  "temp_token": "eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9...",
  "phone_number": "+989123456789"
}
```

**Response Example:**
```json
{
  "status": "success",
  "message": "Registration successful",
  "data": {
    "token": "eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9...",
    "user_id": 12345,
    "expires_in": 86400,
    "refresh_token": "refresh_token_here"
  }
}
```

**Error Codes:**
| Code | Description                    |
|------|--------------------------------|
| 400  | Invalid verification code      |
| 401  | Invalid temporary token        |
| 409  | User already exists            |
| 422  | Verification code expired      |

---

## Complete Flow Implementation

### Step-by-Step Process

1. **Request Activation Code**
   ```bash
   curl -X POST "https://api.example.com/api/v2/user/activation_code/" \
     -H "Content-Type: application/json" \
     -d '{"phone_number": "+989123456789"}'
   ```

2. **Check Registration Status**
   ```bash
   curl -X POST "https://api.example.com/api/v2/user/check/registration/" \
     -H "Content-Type: application/json" \
     -d '{"phone_number": "+989123456789"}'
   ```

3. **Based on Response, Call Login or Register**
   
   **If user exists (login):**
   ```bash
   curl -X POST "https://api.example.com/api/v2/user/login/" \
     -H "Content-Type: application/json" \
     -d '{
       "verification_code": "123456",
       "temp_token": "temp_token_from_step_1",
       "phone_number": "+989123456789"
     }'
   ```
   
   **If user doesn't exist (register):**
   ```bash
   curl -X POST "https://api.example.com/api/v2/user/sign/" \
     -H "Content-Type: application/json" \
     -d '{
       "verification_code": "123456",
       "temp_token": "temp_token_from_step_1",
       "phone_number": "+989123456789"
     }'
   ```

### Flow Logic

```javascript
// Pseudo-code for the complete flow
async function authenticateUser(phoneNumber) {
  // Step 1: Request activation code
  const activationResponse = await requestActivationCode(phoneNumber);
  const tempToken = activationResponse.data.temp_token;
  
  // Step 2: Check if user is registered
  const registrationCheck = await checkRegistration(phoneNumber);
  
  // Step 3: Login or Register based on result
  if (registrationCheck.data.is_registered) {
    // User exists - Login
    const loginResponse = await loginUser(verificationCode, tempToken, phoneNumber);
    return loginResponse.data.token;
  } else {
    // User doesn't exist - Register
    const registerResponse = await registerUser(verificationCode, tempToken, phoneNumber);
    return registerResponse.data.token;
  }
}
```

## Important Notes

- **Temporary Token Expiry**: The `temp_token` expires after 5 minutes (300 seconds)
- **Verification Code**: SMS verification codes typically expire after 10-15 minutes
- **User Token**: The final token returned from login/register is valid for 24 hours (86400 seconds)
- **Rate Limiting**: Activation code requests are rate-limited to prevent abuse
- **Phone Number Format**: Use international format with country code (e.g., +989123456789)

## Security Considerations

- All API calls must be made over HTTPS
- Temporary tokens are single-use and expire quickly
- Verification codes are time-sensitive
- User tokens should be stored securely and refreshed before expiry
- Phone number verification prevents unauthorized access 