# User Acquisition Tracking API

## Overview
Track where users heard about your application to improve marketing analytics and user acquisition strategies.

## Endpoints

### 1. Get Available Acquisition Sources
**Endpoint:** `GET /user/acquisition/sources/`

**Description:** Returns a list of all available acquisition sources with their codes and Persian names.

**Parameters:** None required

**Response:**
```json
{
    "result": true,
    "message": "Available acquisition sources",
    "data": {
        "sources": [
            {"code": "instagram", "name": "اینستاگرام"},
            {"code": "telegram", "name": "تلگرام"},
            {"code": "google", "name": "گوگل"},
            {"code": "friend_referral", "name": "معرفی دوستان"},
            {"code": "doctor", "name": "پزشک/بیمارستان"},
            {"code": "podcast", "name": "پادکست"},
            {"code": "tv_radio", "name": "تلویزیون/رادیو"},
            {"code": "website", "name": "وبسایت"},
            {"code": "other", "name": "سایر"}
        ]
    }
}
```

---

### 2. Update User Acquisition Source
**Endpoint:** `POST /user/acquisition/update/`

**Description:** Update where the user heard about your application.

**Parameters:**
- `token` (required): User's authentication token
- `acquisition_source` (required): One of the source codes from the list above
- `acquisition_details` (conditional): 
  - **REQUIRED** when `acquisition_source` is `"other"`
  - Optional for all other sources (can provide additional context)

**Request Examples:**

#### Example 1: User heard from Instagram
```bash
POST /user/acquisition/update/
{
    "token": "user_token_here",
    "acquisition_source": "instagram",
    "acquisition_details": "@ziluck_official"  # Optional but helpful
}
```

#### Example 2: User heard from "Other" source (details REQUIRED)
```bash
POST /user/acquisition/update/
{
    "token": "user_token_here",
    "acquisition_source": "other",
    "acquisition_details": "از دوست صمیمی‌ام در محل کار شنیدم"  # REQUIRED!
}
```

#### Example 3: Attempting "other" without details (ERROR)
```bash
POST /user/acquisition/update/
{
    "token": "user_token_here",
    "acquisition_source": "other"
    # Missing acquisition_details!
}
```

**Success Response:**
```json
{
    "result": true,
    "message": "Acquisition source updated successfully",
    "data": {
        "acquisition_source": "instagram",
        "acquisition_source_name": "اینستاگرام",
        "acquisition_details": "@ziluck_official",
        "acquisition_date": 1672531200  # Unix timestamp
    }
}
```

**Error Response (when "other" selected without details):**
```json
{
    "result": false,
    "message": "acquisition_details is required when selecting 'other' as the source. Please specify where you heard about us.",
    "code": 400
}
```

---

## Frontend Implementation Guide

### UI Flow

1. **Show Acquisition Source Selection**
   - Display a list or dropdown with all available sources
   - Fetch sources from `GET /user/acquisition/sources/`

2. **Conditional Text Input**
   - When user selects any source EXCEPT "other":
     - Show optional text input for additional details
     - Placeholder: "جزئیات اضافی (اختیاری)"
   
   - When user selects "other":
     - Show **required** text input
     - Placeholder: "لطفاً مشخص کنید از کجا با ما آشنا شدید *"
     - Add validation to prevent submission if empty

3. **Submit**
   - Send POST request to `/user/acquisition/update/`
   - Handle success/error responses

### Example React/React Native Component

```jsx
import React, { useState, useEffect } from 'react';

function AcquisitionSourcePicker({ userToken }) {
    const [sources, setSources] = useState([]);
    const [selectedSource, setSelectedSource] = useState('');
    const [details, setDetails] = useState('');
    const [error, setError] = useState('');

    useEffect(() => {
        // Fetch available sources
        fetch('/user/acquisition/sources/')
            .then(res => res.json())
            .then(data => {
                if (data.result) {
                    setSources(data.data.sources);
                }
            });
    }, []);

    const handleSubmit = async () => {
        // Validate "other" requires details
        if (selectedSource === 'other' && !details.trim()) {
            setError('لطفاً مشخص کنید از کجا با ما آشنا شدید');
            return;
        }

        const formData = new FormData();
        formData.append('token', userToken);
        formData.append('acquisition_source', selectedSource);
        formData.append('acquisition_details', details);

        const response = await fetch('/user/acquisition/update/', {
            method: 'POST',
            body: formData
        });

        const result = await response.json();
        if (result.result) {
            // Success! Navigate to next screen
            console.log('Acquisition source saved!');
        } else {
            setError(result.message);
        }
    };

    return (
        <div>
            <h2>از کجا با ما آشنا شدید؟</h2>
            
            <select 
                value={selectedSource} 
                onChange={(e) => setSelectedSource(e.target.value)}
            >
                <option value="">انتخاب کنید</option>
                {sources.map(source => (
                    <option key={source.code} value={source.code}>
                        {source.name}
                    </option>
                ))}
            </select>

            {selectedSource && (
                <input
                    type="text"
                    value={details}
                    onChange={(e) => setDetails(e.target.value)}
                    placeholder={
                        selectedSource === 'other' 
                            ? 'لطفاً مشخص کنید از کجا با ما آشنا شدید *' 
                            : 'جزئیات اضافی (اختیاری)'
                    }
                    required={selectedSource === 'other'}
                />
            )}

            {error && <p style={{color: 'red'}}>{error}</p>}

            <button onClick={handleSubmit}>ذخیره</button>
        </div>
    );
}
```

---

## Best Practices

1. **When to Ask**: Ask during onboarding/registration, not after login
2. **Make it Optional**: Don't force users to complete this to use the app
3. **Clear Labels**: Use clear Persian labels explaining what you're asking
4. **Additional Context**: Even for predefined sources, allow optional details:
   - Instagram → Which account?
   - Friend Referral → Which friend? (could match with user_code)
   - Doctor → Which doctor/hospital?

---

## Analytics & Reporting

### Admin Panel
You can now filter and analyze users by acquisition source in the Django admin:
- Go to Profile admin
- Filter by `acquisition_source`
- Search by `acquisition_details`
- View `acquisition_date` to see when they heard about you

### Database Queries
```python
from APIs.models import Profile
from django.db.models import Count

# Count users by acquisition source
acquisition_stats = Profile.objects.values('acquisition_source').annotate(
    count=Count('id')
).order_by('-count')

# Get all users who heard from Instagram
instagram_users = Profile.objects.filter(acquisition_source='instagram')

# Get "other" sources with details
other_sources = Profile.objects.filter(
    acquisition_source='other'
).values('acquisition_details', 'acquisition_date')
```

---

## Migration

The database migration has been created. To apply it:

```bash
python manage.py migrate APIs
```

This will add the new fields without affecting existing data.









