# Diet Generation - Telegram Debugging Guide 📱

## Overview

All diet generation events will now be sent directly to your Telegram account (Chat ID: 76335794) in real-time! No need for server log access.

## What You'll Receive on Telegram

When a user submits a **LifestyleQuestionnaire** from the app with `is_from_app=1`, you'll receive a series of messages tracking the entire process:

---

### 🔍 Step 1: Initial Check
```
🔍 Diet Gen Check
User ID: 123
Questionnaire submitted from app
```

---

### ✅ Step 2: Diet Eligibility Check

**Scenario A: No existing diet**
```
✅ User 123: No existing diets
Will generate new diet
```

**Scenario B: Old diet (14+ days)**
```
✅ User 123: 15 days since last diet
Will generate new diet
```

**Scenario C: Recent diet (skip generation)**
```
⏭️ User 123: Only 5 days since last diet
Skipping (need 14+ days)
```

---

### 🚀 Step 3: Generation Started
```
🚀 Starting Diet Generation
User ID: 123
Event ID: lifestyle_questionnai...
```

---

### ✅ Step 4: Thread Status
```
✅ Background thread started
User ID: 123
Result: True
```

---

### 🔄 Step 5: Background Task
```
🔄 Background Task Started
User ID: 123
Generating diet...
```

---

### 🎉 Step 6: Final Result

**Success:**
```
✅ SUCCESS!
User ID: 123
Diet ID: 789
Diet generated successfully
```

**Failure:**
```
❌ FAILED
User ID: 123
Error: Failed to generate any meal plans for user 123 (0/5 meals generated)
```

**Exception:**
```
❌ EXCEPTION in Background Task
User ID: 123
Error: User has no profile with CR value set
```

---

## Common Issues & Solutions

### Issue 1: Not Receiving Messages at All
**Possible Causes:**
- `is_from_app` parameter not set to `1` in the app
- Questionnaire type is not "LifestyleQuestionnaire"
- Telegram bot is blocked

**Solution:**
- Ensure app sends `is_from_app=1` in POST request
- Check that questionnaire_type is exactly "LifestyleQuestionnaire"

---

### Issue 2: "Skipping (need 14+ days)" Message
**Cause:** User has a diet created within the last 14 days

**Solution:** This is expected behavior. Diet will only auto-generate if:
- User has NO existing diets, OR
- Latest diet is 14+ days old

---

### Issue 3: "Background thread started, Result: False"
**Cause:** Threading failed to start

**Solution:** 
- Check server supports threading
- Consider using Celery for production environments

---

### Issue 4: "FAILED - 0/5 meals generated"
**Possible Causes:**
- User profile missing CR (Calorie Requirement) value
- User questionnaire data missing or invalid
- No suitable food cards in database matching criteria

**Solution:**
- Ensure user profile has `CR` or `CR_coach` set
- Verify user has completed questionnaire properly
- Check FoodCard database has appropriate meals

---

### Issue 5: Exception Messages
**Common Exceptions:**

**"User has no profile with CR value set"**
- User.profile.CR and User.profile.CR_coach are both None
- Solution: Set calorie requirement in user profile

**"'NoneType' object has no attribute..."**
- Missing data in user profile or questionnaire
- Solution: Ensure all required profile fields are populated

**"Database error"**
- Permission or connection issues
- Solution: Check database connectivity and permissions

---

## Testing Steps

### Test 1: Submit Test Questionnaire
1. Have a test user submit LifestyleQuestionnaire from the mobile app
2. Ensure `is_from_app=1` is sent in the request
3. Watch your Telegram for messages

### Test 2: Check Different Scenarios
- Test with user who has no diets
- Test with user who has old diet (14+ days)
- Test with user who has recent diet (< 14 days)

### Test 3: Verify Success
- Check if final message shows "✅ SUCCESS!"
- Verify Diet ID is created
- Check database for new Diet entry

---

## What Happens If Telegram Fails?

Don't worry! If Telegram notification fails for any reason:
- ✅ Diet generation will continue normally
- ✅ Error is silently caught and logged
- ✅ App functionality is NOT affected

The Telegram notifications are for monitoring only and won't break your app if they fail.

---

## Quick Debugging Checklist

When diet generation doesn't work, check these in order:

1. ☐ Did you receive "🔍 Diet Gen Check" message?
   - NO → App not sending `is_from_app=1`
   - YES → Continue

2. ☐ Did you receive eligibility check message?
   - "⏭️ Skipping" → User has recent diet (expected)
   - "✅ Will generate" → Continue

3. ☐ Did you receive "🚀 Starting Diet Generation"?
   - NO → Check eligibility criteria
   - YES → Continue

4. ☐ Did you receive "✅ Background thread started"?
   - "Result: False" → Threading issue
   - "Result: True" → Continue

5. ☐ Did you receive "🔄 Background Task Started"?
   - NO → Thread didn't execute
   - YES → Continue

6. ☐ What was the final result?
   - "✅ SUCCESS" → Everything worked!
   - "❌ FAILED" → Check error message
   - "❌ EXCEPTION" → Check exception details

---

## Manual Testing Command

To manually test diet generation for a specific user:

```python
# In Django shell or management command
from panel.services.diet_generation import generate_complete_diet_for_user

user_id = 123  # Replace with actual user ID
success, diet_id, error = generate_complete_diet_for_user(user_id)

print(f"Success: {success}")
print(f"Diet ID: {diet_id}")
print(f"Error: {error}")
```

---

## Need More Details?

If you need more detailed information in the Telegram messages, we can:
1. Add more notification points
2. Include additional debug information
3. Add meal-by-meal generation tracking
4. Include user profile data in messages

Just let me know what additional information would be helpful!

