# Same Lunch and Dinner Feature - Logic Explanation

## 📋 Overview
This feature ensures that lunch and dinner meals have similar main dishes by matching their `main_dish_code` values on a **position-by-position basis** (lunch1 ↔ dinner1, lunch2 ↔ dinner2, etc.).

---

## 🔄 Complete Flow Diagram

```
Page Load
    ↓
┌─────────────────────────────────────┐
│ 1. INITIALIZATION PHASE             │
└─────────────────────────────────────┘
    ↓
checkMealVariety() runs automatically
    ↓
Check: Does MEAL_VARIETY contain "[بله]"?
    ├─ YES → Auto-check "Same lunch and dinner" checkbox
    └─ NO  → Leave checkbox unchecked (assistant can manually enable)
    ↓
┌─────────────────────────────────────┐
│ 2. USER INTERACTION PHASE           │
└─────────────────────────────────────┘
    ↓
Assistant clicks refresh button for lunch OR dinner
    ↓
refreshMeal('lunch') or refreshMeal('dinner') called
    ↓
┌─────────────────────────────────────┐
│ 3. FRONTEND DECISION PHASE          │
└─────────────────────────────────────┘
    ↓
Check: Is "Same lunch and dinner" checkbox checked?
    ├─ NO → Generate meals normally (skip to step 6)
    └─ YES → Continue to step 4
    ↓
Check: Is meal 'lunch' or 'dinner'?
    ├─ NO (breakfast/snack) → Generate normally (skip to step 6)
    └─ YES → Continue to step 5
    ↓
┌─────────────────────────────────────┐
│ 4. COLLECT OTHER MEAL DATA          │
└─────────────────────────────────────┘
    ↓
Determine other meal:
    • If refreshing LUNCH → other meal = DINNER
    • If refreshing DINNER → other meal = LUNCH
    ↓
Loop through all 14 table rows (days 1-14)
    ↓
For each row:
    • Get the cell from OTHER meal column
    • Extract cellData from that cell
    • Store in otherMealCells array (maintains order!)
    ↓
Check: Does other meal have any foods?
    ├─ NO → mainDishCodes = null (skip to step 6)
    └─ YES → Continue to step 5
    ↓
┌─────────────────────────────────────┐
│ 5. EXTRACT main_dish_codes          │
└─────────────────────────────────────┘
    ↓
Create empty mainDishCodes array
    ↓
Loop through otherMealCells (index 0 to 13):
    ↓
    For each cellData at position [index]:
        ↓
        Check main_dish_code value:
        ├─ Is it > 0 AND !== -1 ?
        │   ├─ YES → mainDishCodes[index] = main_dish_code
        │   │        (✅ Valid code, will match)
        │   └─ NO  → mainDishCodes[index] = null
        │            (❌ Invalid, will generate random)
        └─ Result: mainDishCodes array has 14 elements
    ↓
Example result:
mainDishCodes = [2052, 3041, null, 2105, -1→null, 1234, ...]
                 ↑     ↑     ↑     ↑      ↑         ↑
               Day1  Day2  Day3  Day4   Day5      Day6
    ↓
┌─────────────────────────────────────┐
│ 6. SEND REQUEST TO BACKEND          │
└─────────────────────────────────────┘
    ↓
AJAX POST to: /panel/assistant/column/refresh/
    ↓
Data sent:
{
    patient_id: "123",
    meal_type: "lunch",
    count: 14,
    tags: {...},
    months: [...],
    main_dish_codes: [2052, 3041, null, 2105, null, 1234, ...] // or null if feature disabled
}
    ↓
┌─────────────────────────────────────┐
│ 7. BACKEND PROCESSING               │
└─────────────────────────────────────┘
    ↓
refresh_column_cards() in panel/assistant.py receives request
    ↓
Extract main_dish_codes from request
    ↓
Call generate_meal_plan() with all parameters including main_dish_codes
    ↓
┌─────────────────────────────────────┐
│ 8. MEAL PLAN GENERATION             │
└─────────────────────────────────────┘
    ↓
generate_meal_plan() in panel/services/diet_generation.py
    ↓
Apply all normal filters first:
    • Calorie range (target ±10%)
    • Allergen exclusions
    • Disease exclusions
    • Meal type (is_lunch or is_dinner)
    • Special tags (bread, rice, salad, etc.)
    • Month filters
    ↓
Result: cards_query (filtered QuerySet)
    ↓
Check: Is main_dish_codes provided and not empty?
    ├─ NO → Use standard generation (_generate_standard_meal_cards)
    └─ YES → Continue to MATCHING ALGORITHM
    ↓
┌─────────────────────────────────────┐
│ 9. MATCHING ALGORITHM               │
└─────────────────────────────────────┘
    ↓
Create empty selected_cards list
    ↓
FOR idx from 0 to 13 (14 iterations):
    ↓
    Get main_dish_code at position idx
    ↓
    ┌──────────────────────────────────┐
    │ DECISION TREE FOR EACH ROW       │
    └──────────────────────────────────┘
    ↓
    Is main_dish_code None, -1, or 0?
    ├─ YES (Invalid code)
    │   ↓
    │   📍 Print: "Row X: No valid main_dish_code, generating random"
    │   ↓
    │   Call _generate_standard_meal_cards(user, cards_query, 1)
    │   ↓
    │   Add result to selected_cards[idx]
    │   ↓
    │   Continue to next row
    │
    └─ NO (Valid code, e.g., 2052)
        ↓
        Filter: cards_query.filter(main_dish_code=2052)
        ↓
        Do matching cards exist?
        ├─ YES
        │   ↓
        │   Select random card from matching cards
        │   ↓
        │   ✅ Print: "Row X: Matched main_dish_code=2052 → Card: کوفته تبریزی"
        │   ↓
        │   Add card to selected_cards[idx]
        │
        └─ NO
            ↓
            ⚠️ Print: "Row X: No card found with main_dish_code=2052, generating random"
            ↓
            Call _generate_standard_meal_cards(user, cards_query, 1)
            ↓
            Add result to selected_cards[idx]
    ↓
    Continue to next row (idx++)
    ↓
END FOR LOOP
    ↓
Result: selected_cards = [card1, card2, card3, ..., card14]
        Each card at position matches the main_dish_code from other meal
    ↓
┌─────────────────────────────────────┐
│ 10. FORMAT AND RETURN               │
└─────────────────────────────────────┘
    ↓
For each card in selected_cards:
    • Extract nutritional info (calories, fat, protein, carbs)
    • Format as JSON object with main_dish_code included
    ↓
Return response:
{
    "cards": [
        {
            "FA_Name": "کوفته تبریزی با سبزیجات",
            "main_dish_code": 2052,
            "Calories": 350,
            "foods": [...]
        },
        ...
    ],
    "excluded_meals": {...}
}
    ↓
┌─────────────────────────────────────┐
│ 11. FRONTEND DISPLAY                │
└─────────────────────────────────────┘
    ↓
AJAX success callback receives cards array
    ↓
Loop through rows (0 to 13):
    • Get cell for current meal at row[idx]
    • Call populateCell(cell, cards[idx])
    • Display card info in table cell
    ↓
Result: Table shows 14 lunch cards with matching main_dish_codes to dinner
```

---

## 🎯 Key Logic Points

### 1. **Position-Based Matching is Strict**
```javascript
// Frontend collects in order
otherMealCells.forEach((cellData, index) => {
    mainDishCodes[index] = cellData.main_dish_code;  // Position preserved!
});

// Backend processes in order
for idx, main_dish_code in enumerate(main_dish_codes):
    selected_cards[idx] = find_card_with_code(main_dish_code)  // Position preserved!
```

This means:
- `mainDishCodes[0]` (from dinner1) → generates lunch1
- `mainDishCodes[1]` (from dinner2) → generates lunch2
- ...and so on

### 2. **Invalid Code Handling**
```javascript
// Codes considered INVALID:
- null (no data in cellData)
- 0 (no main dish)
- -1 (explicitly marked as "no main dish")

// Action for invalid codes:
→ Generate random card for that position
→ Other positions still use matching
```

### 3. **Fallback Strategy**
```python
# Three-tier fallback:
1. Try to match main_dish_code → Success: Use matched card
2. No match found → Fallback: Generate random appropriate card
3. Random generation fails → Error logged, position may be empty
```

### 4. **Independence of Rows**
Each of the 14 rows is processed **independently**:
- If row 1 matches successfully but row 2 fails → row 1 still has matched card
- Each row's success/failure doesn't affect others
- This allows partial matching when some main_dish_codes don't exist

### 5. **Order-Independent Generation**
The feature works regardless of which meal is generated first:

**Scenario A: Dinner first, then Lunch**
1. Generate dinner normally (14 cards with various main_dish_codes)
2. Enable "Same lunch and dinner"
3. Refresh lunch → Collects main_dish_codes from dinner → Matches

**Scenario B: Lunch first, then Dinner**
1. Generate lunch normally (14 cards with various main_dish_codes)
2. Enable "Same lunch and dinner"
3. Refresh dinner → Collects main_dish_codes from lunch → Matches

---

## 🔍 Example Walkthrough

### Initial State:
```
Dinner column already has:
Row 1: کوفته تبریزی (main_dish_code = 2052)
Row 2: خورش قیمه (main_dish_code = 3041)
Row 3: سالاد (main_dish_code = -1)    ← No main dish
Row 4: مرغ با سبزیجات (main_dish_code = 2105)
...
```

### User Action:
1. Assistant checks "Same lunch and dinner" checkbox
2. Assistant clicks "Refresh Lunch" button

### Frontend Processing:
```javascript
sameLunchDinnerEnabled = true
meal = 'lunch'
otherMeal = 'dinner'

// Collect from dinner column
mainDishCodes = [
    2052,  // Row 1: valid
    3041,  // Row 2: valid
    null,  // Row 3: -1 → converted to null
    2105,  // Row 4: valid
    ...
]
```

### Backend Processing:
```python
# Row 1: main_dish_code = 2052
matching_cards = FoodCard.objects.filter(main_dish_code=2052, is_lunch=True, ...)
# Found: ["کوفته تبریزی", "کوفته قزوینی", ...]
selected_cards[0] = random.choice(matching_cards)  # e.g., "کوفته قزوینی"

# Row 2: main_dish_code = 3041
matching_cards = FoodCard.objects.filter(main_dish_code=3041, is_lunch=True, ...)
selected_cards[1] = random.choice(matching_cards)  # e.g., "خورش قیمه با برنج"

# Row 3: main_dish_code = null (was -1)
selected_cards[2] = _generate_standard_meal_cards(...)  # Random lunch card

# Row 4: main_dish_code = 2105
matching_cards = FoodCard.objects.filter(main_dish_code=2105, is_lunch=True, ...)
selected_cards[3] = random.choice(matching_cards)  # Matching card
```

### Result:
```
Lunch column now has:
Row 1: کوفته قزوینی (main_dish_code = 2052) ✅ MATCHES dinner row 1
Row 2: خورش قیمه (main_dish_code = 3041) ✅ MATCHES dinner row 2
Row 3: میگو با سبزیجات (main_dish_code = 1234) 🎲 RANDOM (dinner had no main dish)
Row 4: مرغ شکم پر (main_dish_code = 2105) ✅ MATCHES dinner row 4
```

---

## 🚨 Edge Cases Handled

1. **Both meals empty**: Feature does nothing, generates normally
2. **One meal empty**: Feature waits until other meal is filled, then matches
3. **All main_dish_codes are -1**: All rows generate random cards
4. **Mix of valid/invalid codes**: Valid codes match, invalid generate random
5. **main_dish_code exists but no matching cards**: Fallback to random for that position
6. **Assistant unchecks checkbox mid-way**: Next refresh uses normal generation
7. **Feature enabled but neither lunch nor dinner refreshed**: No effect on breakfast/snacks

---

## 📊 Performance Considerations

- **Database Queries**: One filter query per valid main_dish_code (max 14)
- **Optimization**: Invalid codes (null, -1, 0) skip the filter query entirely
- **Caching**: Not implemented (each refresh queries fresh cards)
- **Randomization**: Even with same main_dish_code, different cards may be selected

---

## 🎓 Summary

The logic ensures:
1. ✅ **Position correspondence**: lunch[i] ↔ dinner[i] for all i ∈ [0,13]
2. ✅ **Invalid code exclusion**: -1, 0, null are not used for matching
3. ✅ **Graceful fallback**: Missing matches don't break the generation
4. ✅ **Bi-directional**: Works regardless of which meal is generated first
5. ✅ **Optional**: Can be toggled on/off by assistant
6. ✅ **Automatic**: Auto-enables based on patient preference (MEAL_VARIETY)

