# Same Lunch and Dinner Feature Implementation

## Overview
This document describes the implementation of the "Same lunch and dinner" feature that allows assistants to generate similar meals for lunch and dinner based on matching `main_dish_code` values.

## Feature Description
The feature adds a new toggle checkbox labeled "ناهار و شام مشابه" (Same lunch and dinner) that:
1. Automatically checks when a patient's `MEAL_VARIETY` questionnaire value is "[بله]"
2. Can be manually toggled by the assistant at any time
3. When enabled, ensures that lunch and dinner meals share the same `main_dish_code` values

## Implementation Details

### 1. Frontend Changes (diets2.html)

#### A. New Toggle Checkbox
Added a new toggle switch after the seasonal snack checkboxes:
```html
<label class="toggle-switch">
    <input type="checkbox" id="sameLunchDinner">
    <span class="toggle-slider"></span>
    <span class="toggle-label">{% trans "ناهار و شام مشابه" %}</span>
</label>
```

#### B. MEAL_VARIETY Checker Function
Added JavaScript function to check questionnaire data and auto-enable checkbox:
```javascript
function checkMealVariety() {
    try {
        const rawQuestionnaire = {{ raw_questionnaire|safe }};
        const mealVariety = rawQuestionnaire.MEAL_VARIETY;
        
        if (mealVariety && String(mealVariety).includes('بله')) {
            document.getElementById('sameLunchDinner').checked = true;
        }
    } catch (error) {
        console.error("Error checking MEAL_VARIETY:", error);
    }
}
```

#### C. Modified refreshMeal Function
Enhanced the `refreshMeal()` function to:
1. Check if "Same lunch and dinner" checkbox is enabled
2. When refreshing lunch or dinner, collect `main_dish_code` values from the other meal
3. Send these codes to the backend for matching

Key logic:
```javascript
// Check if "Same lunch and dinner" is enabled
let mainDishCodes = null;
const sameLunchDinnerEnabled = document.getElementById('sameLunchDinner').checked;

if (sameLunchDinnerEnabled && (meal === 'lunch' || meal === 'dinner')) {
    const otherMeal = meal === 'lunch' ? 'dinner' : 'lunch';
    
    // Check if the other meal has foods
    const otherMealHasFoods = otherMealCells.some(cellData => 
        cellData && cellData.foods && cellData.foods.length > 0
    );
    
    if (otherMealHasFoods) {
        // Collect main_dish_codes from the other meal
        mainDishCodes = [];
        otherMealCells.forEach(cellData => {
            if (cellData && cellData.main_dish_code) {
                mainDishCodes.push(cellData.main_dish_code);
            } else {
                mainDishCodes.push(null);
            }
        });
    }
}
```

### 2. Backend Changes

#### A. panel/assistant.py
Modified `refresh_column_cards()` to:
1. Accept `main_dish_codes` parameter from request
2. Pass it to the `generate_meal_plan()` function

```python
main_dish_codes = request.data.get('main_dish_codes', None)
print(f"Main dish codes: {main_dish_codes}")

result = generate_meal_plan(
    patient, 
    meal_type, 
    selected_tags, 
    count, 
    months_filter, 
    return_metadata=True, 
    bypass_zero_calorie=bypass_zero_calorie,
    bypassed_meals=bypassed_meals,
    main_dish_codes=main_dish_codes
)
```

#### B. panel/services/diet_generation.py
Enhanced `generate_meal_plan()` function to:
1. Accept `main_dish_codes` parameter
2. Filter food cards by matching `main_dish_code` when provided
3. Fall back to random selection if no matching card is found

```python
def generate_meal_plan(..., main_dish_codes=None):
    # Check if main_dish_codes filtering is needed
    if main_dish_codes and len(main_dish_codes) > 0:
        print(f"Filtering by main_dish_codes: {main_dish_codes}")
        selected_cards = []
        
        for idx, main_dish_code in enumerate(main_dish_codes):
            if main_dish_code is None:
                # Generate a random card
                card_for_row = _generate_standard_meal_cards(user, cards_query, 1)
                if card_for_row:
                    selected_cards.extend(card_for_row)
            else:
                # Filter cards by main_dish_code
                matching_cards = cards_query.filter(main_dish_code=main_dish_code)
                
                if matching_cards.exists():
                    card = random.choice(list(matching_cards))
                    selected_cards.append(card)
                else:
                    # Fallback to random card
                    card_for_row = _generate_standard_meal_cards(user, cards_query, 1)
                    if card_for_row:
                        selected_cards.extend(card_for_row)
```

### 3. Localization

Added translations in all locale files:

**Persian (fa):**
```
msgid "ناهار و شام مشابه"
msgstr "ناهار و شام مشابه"
```

**English (en):**
```
msgid "ناهار و شام مشابه"
msgstr "Same lunch and dinner"
```

**Arabic (ar):**
```
msgid "ناهار و شام مشابه"
msgstr "نفس الغداء والعشاء"
```

## Usage Scenarios

### Scenario 1: MEAL_VARIETY is False
- Checkbox is not automatically checked
- System behaves normally (no changes)

### Scenario 2: Initial Page Load with MEAL_VARIETY = "[بله]"
1. Assistant opens diets2 page
2. Checkbox is automatically checked
3. Assistant clicks "refresh dinner"
4. Dinner meals are generated normally (14 cards)
5. Assistant clicks "refresh lunch"
6. System collects `main_dish_code` values from all 14 dinner cards
7. For each lunch card (1-14), system finds food cards with matching `main_dish_code`
8. Result: lunch1 has same `main_dish_code` as dinner1, lunch2 matches dinner2, etc.

### Scenario 3: Manual Toggle by Assistant
1. Patient didn't select MEAL_VARIETY, but assistant prefers similar meals
2. Assistant manually checks "Same lunch and dinner" toggle
3. System detects that dinner already has foods
4. When assistant clicks "refresh lunch":
   - System collects `main_dish_code` from existing dinner cards
   - Generates lunch cards with matching `main_dish_code` values

### Scenario 4: One Meal is Empty
- If assistant enables toggle and clicks refresh on a meal
- System checks if the other meal (lunch/dinner) has foods
- If other meal is empty, system generates cards normally (no matching)
- This allows flexibility in generation order

### Scenario 5: Both Meals are Empty
- System generates meals normally
- No main_dish_code matching occurs
- Behaves like the toggle is disabled

## Benefits

1. **Patient Preference Support**: Automatically respects patient's meal variety preferences
2. **Flexibility**: Assistants can manually enable/disable at any time
3. **Smart Matching**: Maintains nutritional balance while ensuring similar main dishes
4. **Position-Based Accuracy**: Ensures lunch1 matches dinner1, lunch2 matches dinner2, etc. (not random positions)
5. **Invalid Code Handling**: Properly excludes main_dish_code values of -1 (no main dish) from matching
6. **Fallback Logic**: If no matching card exists, system gracefully falls back to random selection
7. **Order Independence**: Works regardless of which meal is generated first

## Improvements

### Version 1.1 Updates:
1. **Excluded main_dish_code = -1**: Cards with main_dish_code of -1 (indicating no main dish or not applicable) are now excluded from matching logic
2. **Position-based matching enforced**: The system now explicitly maintains position correspondence:
   - If lunch1 has main_dish_code=2052, dinner1 will have main_dish_code=2052
   - If lunch2 has main_dish_code=3041, dinner2 will have main_dish_code=3041
   - And so on for all 14 rows
3. **Enhanced logging**: Added detailed console logging with emojis to track the matching process:
   - ✅ Successful matches
   - ⚠️ Fallback to random when no match found
   - 📍 Random generation for invalid codes
   - ❌ Failed card generation

### Version 1.2 Updates (Randomization Improvements):
1. **Timestamp-based seeding**: Random seed now includes microsecond timestamp instead of just date
   - **Before**: User A at 10am and User B at 2pm with 1200 kcal → Identical results
   - **After**: Each generation uses unique timestamp → Different results every time
   
2. **Position-specific seeding**: Each row (1-14) gets its own random seed
   - Seed includes: `user_id + meal_type + row_index + main_dish_code + timestamp`
   - This ensures variation even when matching the same main_dish_code
   
3. **Why this matters**:
   ```
   Scenario: 10 cards exist with main_dish_code=2052
   
   OLD BEHAVIOR (date-only seed):
   - User A (1200 kcal, Monday 10am) → Card #3 from the 10 options
   - User B (1200 kcal, Monday 2pm) → Card #3 (SAME!)
   - User C (1200 kcal, Monday 5pm) → Card #3 (SAME!)
   
   NEW BEHAVIOR (timestamp + position seed):
   - User A (1200 kcal, Monday 10am) → Card #3
   - User B (1200 kcal, Monday 2pm) → Card #7 (DIFFERENT!)
   - User C (1200 kcal, Monday 5pm) → Card #2 (DIFFERENT!)
   ```

4. **Maintains similarity while adding variety**:
   - Lunch and dinner still have matching main_dish_codes (same main dish type)
   - But the specific card variation changes between users
   - Example: Both users get "کوفته" (meatballs), but User A gets "کوفته تبریزی" and User B gets "کوفته قزوینی"

## Technical Notes

- `main_dish_code` is a field in the `FoodCard` model that categorizes food cards by their main dish type
- The feature maintains all existing constraints (calories, allergens, diseases, tags)
- Cards are filtered by `main_dish_code` only after applying all other filters
- **Position-based matching**: Each row is matched by index (lunch1 → dinner1, lunch2 → dinner2, ..., lunch14 → dinner14)
- **Exclusion of invalid codes**: `main_dish_code` values of -1, 0, or null are treated as "no main dish" and trigger random card generation
- If a specific `main_dish_code` has no matching cards, the system falls back to random selection for that position

## Files Modified

1. `/panel/templates/assistant/diets2.html` - Added checkbox and JavaScript logic
2. `/panel/assistant.py` - Modified `refresh_column_cards()` function
3. `/panel/services/diet_generation.py` - Enhanced `generate_meal_plan()` function
4. `/locale/fa/LC_MESSAGES/django.po` - Added Persian translation
5. `/locale/en/LC_MESSAGES/django.po` - Added English translation
6. `/locale/ar/LC_MESSAGES/django.po` - Added Arabic translation

## Testing Recommendations

1. **Test with MEAL_VARIETY = "[بله]"**: Verify checkbox auto-checks
2. **Test manual toggle**: Check that checkbox can be toggled on/off
3. **Test generation order**: Try generating dinner first, then lunch, and vice versa
4. **Test with empty meals**: Verify system handles empty lunch/dinner gracefully
5. **Test main_dish_code matching**: Verify lunch and dinner have matching main_dish_codes
6. **Test fallback**: Verify system works even if some main_dish_codes have no matches

## Future Enhancements

1. Add visual indicator when meals are matched
2. Show warning if main_dish_code matching fails for many rows
3. Add option to match specific days rather than all 14 days
4. Extend feature to support snacks similarity

