# Lab Test User Hierarchy Example

This document shows the complete hierarchy when a user is created through the lab test upload process.

## Model Hierarchy Structure

```
DjangoUser (django.contrib.auth.models.User)
    │
    ├─── User (APIs.models.User)
    │       │
    │       ├─── Profile (APIs.models.Profile)  ← Patient information stored here
    │       │
    │       └─── LabTest (APIs.models.LabTest)  ← Multiple lab tests per user
    │               │
    │               └─── LabTestResult (APIs.models.LabTestResult)  ← Multiple results per lab test
    │                       │
    │                       └─── BiomarkerDefinition (APIs.models.BiomarkerDefinition)  ← Optional link
```

## Concrete Example

### Scenario
A doctor uploads an Excel file with blood test results for:
- **Patient Name**: "علی احمدی" (Ali Ahmadi)
- **National ID**: "1234567890"
- **Test Date**: "1403/08/15"
- **Lab Results**: Hb: 14.5, FBS: 95, LDL: 120, etc.

---

### Step 1: DjangoUser (Authentication User)
**Created automatically** when a new user is found/created

```python
DjangoUser {
    id: 1001
    username: "lab_1234567890_1734600000"  # Auto-generated
    password: None  # Will be set later if needed
    email: ""
    is_active: True
    date_joined: "2024-12-19 12:00:00"
}
```

---

### Step 2: User (APIs User Model)
**Created** and linked to DjangoUser

```python
User {
    id: 501
    phone_number: ""  # Empty initially
    token: ""  # Empty initially
    django_user: DjangoUser(id=1001)  ← Links to Step 1
    profile: Profile(id=301)  ← Links to Step 3
    created_at: "2024-12-19"
    isLogedin: False
    isRegistered: False
}
```

---

### Step 3: Profile (Patient Information)
**Created/Updated** with patient data from lab test

```python
Profile {
    id: 301
    first_name: "علی"  ← From lab test Excel
    last_name: "احمدی"  ← From lab test Excel
    national_id: "1234567890"  ← From lab test Excel
    gender: 0  # 0=male, 1=female (defaults to 0 if not specified)
    birth_date: "1994-01-01"  # Calculated from age if provided
    weight: 70.0  # Default, can be updated later
    height: 170  # Default, can be updated later
    # ... other profile fields with defaults
}
```

**Note**: If a user with the same `national_id` already exists, the existing Profile is **updated** with new information from the lab test.

---

### Step 4: LabTest (Blood Test Record)
**Created** for each lab test uploaded, linked to User

```python
LabTest {
    id: 201
    user: User(id=501)  ← REQUIRED: Links to Step 2
    test_date: "2024-12-15"  ← From lab test Excel
    lab_center: "آزمایشگاه نوو"  ← From lab test Excel
    registration_number: "LAB-12345"  ← From lab test Excel
    source_file: "lab_tests/source_files/novo_lab_test.xlsx"
    source_type: "excel"
    
    # Analysis Results (stored as JSON)
    overall_health_score: 78
    indices: {
        "Cardiometabolic": 75,
        "Hematology": 82,
        "Organ Function": 80,
        "Thyroid": 70,
        "Hormonal": 65
    }
    index_details: { ... }  # Detailed analysis
    risks: {
        "CVD": {"value": 12.5, "label": "Low", "badge": "badge-success"},
        "Diabetes": {"value": 8.2, "label": "Low", "badge": "badge-success"},
        "Liver": {"value": 15.3, "label": "Low", "badge": "badge-success"}
    }
    raw_data: {
        "Hb": 14.5,
        "Fasting Glucose": 95,
        "LDL": 120,
        "HDL": 45,
        # ... all biomarkers from Excel
    }
    
    is_analyzed: True
    analyzed_at: "2024-12-19 12:05:00"
    created_by: DjangoUser(id=50)  ← The doctor who uploaded
    created_at: "2024-12-19 12:00:00"
}
```

**Key Point**: LabTest does **NOT** store duplicate patient info (name, national_id, gender, age). It gets this from `user.profile`.

---

### Step 5: LabTestResult (Individual Biomarker Results)
**Created** for each biomarker in the lab test

```python
# Example 1: Hemoglobin (Hb)
LabTestResult {
    id: 1001
    lab_test: LabTest(id=201)  ← Links to Step 4
    biomarker: BiomarkerDefinition(id=15)  ← Optional: Links to Hb definition
    biomarker_name: "Hb"
    value: 14.5
    value_string: ""
    unit: "g/dL"
    method: "Automated"
    ref_min: 13.5
    ref_max: 17.5
    fuzzy_score: 85
    status: "طبیعی"
    lifestyle_feedback: "سطح هموگلوبین شما در محدوده طبیعی است..."
    test_code: "CBC-001"
}

# Example 2: Fasting Blood Sugar (FBS)
LabTestResult {
    id: 1002
    lab_test: LabTest(id=201)
    biomarker: BiomarkerDefinition(id=25)  ← Links to FBS definition
    biomarker_name: "Fasting Glucose"
    value: 95.0
    value_string: ""
    unit: "mg/dL"
    method: "Enzymatic"
    ref_min: 70.0
    ref_max: 100.0
    fuzzy_score: 92
    status: "ایده‌آل"
    lifestyle_feedback: "قند خون ناشتای شما در محدوده ایده‌آل است..."
    test_code: "GLU-001"
}

# Example 3: LDL Cholesterol
LabTestResult {
    id: 1003
    lab_test: LabTest(id=201)
    biomarker: BiomarkerDefinition(id=30)
    biomarker_name: "LDL"
    value: 120.0
    value_string: ""
    unit: "mg/dL"
    method: "Direct"
    ref_min: 0.0
    ref_max: 100.0
    fuzzy_score: 65
    status: "نیازمند بررسی"
    lifestyle_feedback: "سطح LDL شما کمی بالاتر از حد مطلوب است. توصیه می‌شود..."
    test_code: "LIP-001"
}

# ... and so on for each biomarker in the lab test
```

---

## Database Relationships Summary

### One-to-One Relationships:
- `DjangoUser` ↔ `User` (one-to-one via `django_user` ForeignKey)
- `User` ↔ `Profile` (one-to-one via `profile` ForeignKey)

### One-to-Many Relationships:
- `User` → `LabTest` (one user can have many lab tests)
- `LabTest` → `LabTestResult` (one lab test has many biomarker results)
- `BiomarkerDefinition` → `LabTestResult` (one biomarker definition can appear in many results)

### Many-to-One Relationships:
- `LabTest` ← `User` (many lab tests belong to one user)
- `LabTestResult` ← `LabTest` (many results belong to one lab test)
- `LabTestResult` ← `BiomarkerDefinition` (many results can reference one biomarker definition)

---

## Key Design Principles

1. **No Duplicate Patient Data**: 
   - Patient information (name, national_id, gender, age) is stored **only** in `Profile`
   - `LabTest` does **NOT** duplicate this information
   - Access patient info via: `lab_test.user.profile.first_name`

2. **User Matching Logic**:
   - When uploading a lab test, the system first tries to find an existing user by `national_id`
   - If not found, tries to find by `first_name + last_name`
   - If still not found, creates a new user
   - If found, **updates** the existing profile with any new/missing information

3. **Data Flow**:
   ```
   Excel File → Extract Patient Info → Find/Create User → Update Profile → 
   Create LabTest → Analyze → Create LabTestResults → Store in Database
   ```

4. **Accessing Data**:
   ```python
   # Get patient name from a lab test
   lab_test = LabTest.objects.get(id=201)
   patient_name = lab_test.get_patient_name()  # "علی احمدی"
   
   # Or directly
   patient_name = f"{lab_test.user.profile.first_name} {lab_test.user.profile.last_name}"
   
   # Get all lab tests for a user
   user = User.objects.get(id=501)
   all_lab_tests = user.lab_tests.all()  # Related name: 'lab_tests'
   
   # Get all results for a lab test
   lab_test = LabTest.objects.get(id=201)
   all_results = lab_test.results.all()  # Related name: 'results'
   
   # Get a specific biomarker result
   hb_result = lab_test.results.filter(biomarker_name="Hb").first()
   ```

---

## Example: Multiple Lab Tests for Same User

If the same patient (Ali Ahmadi) uploads another lab test 3 months later:

```
User(id=501)  ← Same user (found by national_id)
    │
    ├─── LabTest(id=201)  ← First lab test (Dec 2024)
    │       └─── LabTestResult(id=1001-1050)  ← 50 biomarkers
    │
    └─── LabTest(id=202)  ← Second lab test (Mar 2025)
            └─── LabTestResult(id=2001-2050)  ← 50 biomarkers
```

**No duplicate user created!** The system finds the existing user by `national_id` and links the new lab test to the same user.

---

## Benefits of This Structure

1. ✅ **No Data Duplication**: Patient info stored once in Profile
2. ✅ **Historical Tracking**: Multiple lab tests per user for trend analysis
3. ✅ **Data Integrity**: Foreign key constraints ensure data consistency
4. ✅ **Easy Queries**: Can easily find all lab tests for a user, or all users with a specific biomarker value
5. ✅ **Profile Updates**: Lab test data can update missing profile information
6. ✅ **Scalability**: Can handle thousands of lab tests per user efficiently

