# HR Lab Tests Panel - How It Works

## How Users Are Linked to Companies

### 1. Company Structure
- **Company** = `ZiLuckGroup` with `is_company=True`
- Each company is a Django `Group` (ZiLuckGroup extends Group)
- Companies have `admin` and `creator` fields (DjangoUser references)

### 2. User-Company Relationship
Users are linked to companies via **Django's built-in Group membership**:

```python
# When a user joins a company:
company_group = ZiLuckGroup.objects.get(id=company_id, is_company=True)
user.django_user.groups.add(company_group)  # Add user to the group
# OR
company_group.user_set.add(user.django_user)  # Same thing, different direction
```

### 3. How HR Panel Identifies Employees

**Current Logic:**
```python
# Step 1: Find companies HR has access to
company_groups = ZiLuckGroup.objects.filter(
    is_company=True,
    admin=request.user  # OR creator=request.user
)

# Step 2: Get all employees in selected company
employees = AppUser.objects.filter(
    django_user__groups=selected_company  # Users whose django_user is in this group
)

# Step 3: Get lab tests for those employees
lab_tests = LabTest.objects.filter(
    user__django_user__groups=selected_company
)
```

## Current Access Control

**HR can see lab tests for:**
- Companies where they are the `admin` of the ZiLuckGroup
- Companies where they are the `creator` of the ZiLuckGroup
- All companies (if superuser)

**Employees shown:**
- All users whose `django_user` is a member of the selected company's ZiLuckGroup

## How to Assign Users to Companies

### Method 1: Via Django Admin
1. Go to Django Admin → Groups (ZiLuckGroup)
2. Select the company group
3. Add users to the group

### Method 2: Via Code
```python
from APIs.models import User as AppUser, ZiLuckGroup

# Get company
company = ZiLuckGroup.objects.get(id=company_id, is_company=True)

# Get user
user = AppUser.objects.get(id=user_id)

# Add user to company
company.user_set.add(user.django_user)
```

### Method 3: Via Registration/Discount Code
When users register with a company discount code (e.g., `COMPANY_PROMOTION`), they can be automatically added to the company group.

## Example Flow

1. **Company Created:**
   ```
   Company: "ABC Corp"
   ZiLuckGroup: id=5, name="ABC_CORP", is_company=True
   Admin: HR User (django_user_id=10)
   ```

2. **Employee Joins:**
   ```
   Employee: John Doe
   AppUser: id=100, django_user_id=50
   
   # Add to company
   company.user_set.add(john_doe.django_user)
   # Now: john_doe.django_user.groups contains company group
   ```

3. **Lab Test Uploaded:**
   ```
   LabTest: id=200
   User: john_doe (AppUser id=100)
   ```

4. **HR Views Dashboard:**
   ```
   HR User (django_user_id=10) logs in
   → Sees "ABC Corp" (because they're admin)
   → Selects "ABC Corp"
   → System finds: AppUser.objects.filter(django_user__groups=company)
   → Finds: john_doe (because his django_user is in the group)
   → Shows: LabTest id=200
   ```

## Potential Improvements

1. **HR could also see companies they're members of** (not just admin/creator)
2. **Add a direct company field to User/Profile** for easier querying
3. **Add company assignment during lab test upload** (if user not in company, prompt to assign)

