# Food Recommendations Cache - Complete Flow Documentation

## Overview
This document describes the complete flow of the food recommendations caching system. The system consists of two APIs:
1. **Generation API** (`get_food_recommendations_by_disease`): Generates recommendations and saves them to cache
2. **Cache Retrieval API** (`get_cached_food_recommendations`): Returns cached recommendations extremely fast (< 1 second)

## Data Model

### FoodRecommendationsCache Model
Located in `APIs/models.py`, this model stores all food recommendations for a user in one record:

```python
class FoodRecommendationsCache(models.Model):
    user = models.OneToOneField(User, on_delete=models.CASCADE, related_name='food_recommendations_cache')
    recommendations = models.JSONField(default=list)  # Complete list of all recommendations
    diseases_string = models.TextField(blank=True, null=True)
    lab_results = models.JSONField(default=list, blank=True, null=True)
    total_foods = models.IntegerField(default=0)
    created_at = models.DateTimeField(default=datetime.now)
    updated_at = models.DateTimeField(auto_now=True)
```

**Key Features:**
- One-to-One relationship with User (one cache record per user)
- All recommendations stored in a single JSONField for fast retrieval
- Indexed on `user` and `updated_at` for optimal query performance

## API Endpoints

### 1. Generation API: `get_food_recommendations_by_disease`

**Endpoint:** `POST /api/v1/report/food/recommendations/disease/`

**Parameters:**
- `token`: User authentication token (required)
- `diseases_string`: String of diseases separated by '&&' (optional)
- `lab_results`: JSON array of lab test results (optional)

**Purpose:** Generates food recommendations and saves them to cache

**Complete Flow:**

#### Step 1: Authentication
```
User sends POST request with token
↓
API validates token
↓
Retrieves User object
```

#### Step 2: Generate Recommendations
```
Parse diseases_string and lab_results
↓
Call process_recommendations(diseases_string, lab_results)
  - Fetches all foods from Neo4j database
  - Calculates color (Red/Yellow/Green) for each food
  - Processes based on diseases and lab results
↓
Generate complete recommendations list
```

#### Step 3: Save to Cache
```
Create or update FoodRecommendationsCache record
  - Store all recommendations in JSONField
  - Store diseases_string and lab_results
  - Store total_foods count
  - Set created_at and updated_at timestamps
↓
Save to database
```

#### Step 4: Send Notifications & Response
```
Send Firebase and Push notifications
↓
Return response with recommendations
  - recommendations: Complete list
  - total_foods: Count
  - user_id: User ID
  - diseases_processed: Diseases string
  - lab_tests_processed: Count
```

### 2. Cache Retrieval API: `get_cached_food_recommendations`

**Endpoint:** `POST /api/v1/report/food/recommendations/cached/`

**Parameters:**
- `token`: User authentication token (required)

**Purpose:** Returns cached recommendations extremely fast if they exist

**Complete Flow:**

#### Step 1: Authentication
```
User sends POST request with token
↓
API validates token
↓
Retrieves User object
```

#### Step 2: Retrieve from Cache
```
Check if FoodRecommendationsCache exists for user
↓
YES → Return cached recommendations immediately (< 100ms)
  - Reads single JSONField from database
  - No computation needed
  - No Neo4j queries
  - Returns immediately with all recommendations
↓
NO → Return error message
```

## Performance Characteristics

### Cache Retrieval API (Fast Path)
- **Response Time**: < 1 second (typically < 100ms)
- **Database Queries**: 1 query (SELECT on indexed user field)
- **Computation**: None
- **Network Calls**: None (no Neo4j queries)

### Generation API (Slow Path)
- **Response Time**: Variable (depends on Neo4j query and processing)
- **Database Queries**: 
  - 1 query to save cache (INSERT/UPDATE)
  - Multiple queries to Neo4j (fetch all foods)
- **Computation**: High (processing all foods, calculating colors)
- **Network Calls**: Neo4j database queries

## Response Formats

### Cache Retrieval API - Success Response
```json
{
  "status": "success",
  "message": "توصیه‌های غذایی با موفقیت دریافت شد",
  "data": {
    "recommendations": [
      {
        "FoodCode": 12345,
        "Foodname": "غذا",
        "color": "Green",
        "Calories": 100,
        "protein": 10,
        "carbohydrate": 20,
        "Fat": 5,
        "EnglishFoodName": "Food"
      },
      // ... more recommendations
    ],
    "total_foods": 500,
    "user_id": 123,
    "diseases_processed": "PCOS&&دیابت",
    "lab_tests_processed": 2,
    "cache_created_at": "2024-01-01T12:00:00",
    "cache_updated_at": "2024-01-01T12:00:00"
  }
}
```

### Cache Retrieval API - Error Response (No Cache)
```json
{
  "status": "error",
  "message": "هیچ توصیه غذایی ذخیره‌شده‌ای یافت نشد. لطفا ابتدا توصیه‌ها را ایجاد کنید.",
  "code": -1
}
```

### Generation API - Success Response
```json
{
  "status": "success",
  "message": "توصیه‌های غذایی با موفقیت ایجاد شد",
  "data": {
    "recommendations": [...],
    "total_foods": 500,
    "user_id": 123,
    "diseases_processed": "PCOS&&دیابت",
    "lab_tests_processed": 2
  }
}
```

## Database Schema

### Table: APIs_foodrecommendationscache

| Column | Type | Description |
|--------|------|-------------|
| id | Integer (PK) | Primary key |
| user_id | Integer (FK) | Foreign key to User table (UNIQUE) |
| recommendations | JSON | Complete list of all food recommendations |
| diseases_string | Text | Diseases string used for generation |
| lab_results | JSON | Lab results used for generation |
| total_foods | Integer | Total count of recommendations |
| created_at | DateTime | When cache was first created |
| updated_at | DateTime | Last update timestamp |

### Indexes
- Index on `user_id` (for fast lookups)
- Index on `updated_at DESC` (for sorting/querying)

## Cache Invalidation

Currently, the cache is **not automatically invalidated**. The cache will be updated when:
- User calls the API with new parameters
- `update_or_create` is called with new recommendations

### Manual Cache Clearance
To clear a user's cache manually:
```python
from APIs.models import FoodRecommendationsCache

# Clear specific user's cache
FoodRecommendationsCache.objects.filter(user=user).delete()

# Clear all caches
FoodRecommendationsCache.objects.all().delete()
```

## Admin Interface

The model is registered in Django Admin (`APIs/admin.py`) with:
- List display: id, user, total_foods, created_at, updated_at
- Search fields: phone number, first name, last name, diseases_string
- Filters: created_at, updated_at
- Read-only fields: created_at, updated_at
- Optimized queryset with select_related for user and profile

## Migration

To apply the database migration:

```bash
source venv/bin/activate
python manage.py makemigrations APIs
python manage.py migrate
```

## Usage Example

### Step 1: Generate Recommendations (Generation API)
```python
# POST /api/v1/report/food/recommendations/disease/
{
    "token": "user_token_here",
    "diseases_string": "PCOS (سندروم تخمدان پلی‌کیستیک)&&دیابت",
    "lab_results": "[{\"test_name\": \"MCHC\", \"status\": \"low\"}]"
}

# Response: Generated recommendations, saved to cache, ~5-10 seconds
# This API generates recommendations and automatically saves them to cache
```

### Step 2: Retrieve Cached Recommendations (Cache Retrieval API)
```python
# POST /api/v1/report/food/recommendations/cached/
{
    "token": "user_token_here"
}

# Response: Returns cached recommendations immediately, < 100ms
# This API only reads from cache - extremely fast!
```

### Workflow
```
1. User calls Generation API once → Generates and saves to cache
2. User calls Cache Retrieval API multiple times → Returns cached data instantly
3. To regenerate recommendations → Call Generation API again
```

## Benefits

1. **Extremely Fast Response**: Cached responses return in < 1 second
2. **Single Database Query**: Only one query needed for cached data
3. **Reduced Neo4j Load**: No need to query Neo4j on every request
4. **Cost Efficient**: Reduces computation and database queries
5. **Better User Experience**: Fast, responsive API
6. **Scalable**: Can handle many concurrent requests efficiently

## Future Enhancements

Potential improvements:
1. Add cache expiration/invalidation based on time or data changes
2. Add versioning to cache (invalidate when diseases/lab results change)
3. Add background job to pre-generate recommendations
4. Add cache warming on user registration
5. Add metrics/monitoring for cache hit rate

