# Diet Generation Debugging Guide - Telegram Notifications

## What Was Added

I've added **Telegram notifications** that send debug messages directly to your Telegram account so you can monitor diet generation in real-time without needing server access!

## Changes Made

### 1. Added Telegram Debug Method to `panel/telegram_notification.py`
- New `send_debug_log()` method for sending debug messages
- Automatically truncates long messages (Telegram 4096 char limit)
- Silently fails if Telegram is unavailable (won't break the app)

### 2. Updated `APIs/views/user.py` with Telegram Notifications
- Sends notification when questionnaire is submitted from app
- Tracks existing diet checks and date calculations
- Notifies when diet generation starts
- **Sends error messages if anything fails** (previously silently ignored)

### 3. Updated `panel/services/diet_generation.py` with Telegram Notifications
- Notifies when background thread starts
- Sends success message with Diet ID when complete
- Sends detailed error messages if generation fails
- Tracks all exceptions in background tasks

## How It Works - You'll Receive Telegram Messages!

### Option 1: Django Logs (if configured)
```bash
# Check your Django application logs
tail -f /path/to/your/django/logs/application.log
```

### Option 2: System Logs
```bash
# For systemd service
sudo journalctl -u your-django-service -f

# For Gunicorn
tail -f /var/log/gunicorn/error.log
```

### Option 3: Check Settings
Add this to your `settings.py` if not already present:
```python
LOGGING = {
    'version': 1,
    'disable_existing_loggers': False,
    'handlers': {
        'file': {
            'level': 'INFO',
            'class': 'logging.FileHandler',
            'filename': '/path/to/your/logs/diet_generation.log',
        },
        'console': {
            'level': 'INFO',
            'class': 'logging.StreamHandler',
        },
    },
    'loggers': {
        'APIs.views.user': {
            'handlers': ['file', 'console'],
            'level': 'INFO',
            'propagate': True,
        },
        'panel.services.diet_generation': {
            'handlers': ['file', 'console'],
            'level': 'INFO',
            'propagate': True,
        },
    },
}
```

## What to Look For in Logs

### Successful Flow:
```
INFO: LifestyleQuestionnaire submitted - User: 123, is_from_app: True
INFO: User 123: No existing diets found, will generate diet
INFO: User 123: Starting background diet generation with event_id: lifestyle_questionnaire_123_456_abc123
INFO: User 123: Background diet generation started, result: True
INFO: Background thread started successfully for user 123
INFO: Background task started for user 123, event_id: lifestyle_questionnaire_123_456_abc123
INFO: Starting generate_complete_diet_for_user for user_id: 123, event_id: lifestyle_questionnaire_123_456_abc123
INFO: User 123 found successfully
INFO: User 123: Generated 5/5 meal types
INFO: User 123: Successfully created diet 789 with start_date: 2025-11-06
INFO: Successfully generated diet 789 for user 123
```

## Common Issues to Check

### Issue 1: `is_from_app` Parameter Not Set
**Log to look for:**
```
INFO: LifestyleQuestionnaire submitted - User: 123, is_from_app: False
```
**Solution:** Ensure the mobile app sends `is_from_app=1` in the POST request

### Issue 2: Recent Diet Exists
**Log to look for:**
```
INFO: User 123: Latest diet from 2025-11-01, 4 days ago
INFO: User 123: Only 4 days since last diet, skipping generation
```
**Solution:** This is expected behavior - diet is only generated if 14+ days have passed

### Issue 3: Background Thread Not Starting
**Log to look for:**
```
ERROR: Failed to start background thread for user 123: [error message]
```
**Solution:** Check server threading support, consider using Celery

### Issue 4: User Profile Missing CR Value
**Log to look for:**
```
ERROR: User 123: Error in diet generation: [something about CR or profile]
```
**Solution:** Ensure user profile has calorie requirement (CR) set

### Issue 5: Database Permission Issues
**Log to look for:**
```
ERROR: User 123: Error saving diet: [database error]
```
**Solution:** Check database permissions and connection

## Testing Steps

1. **Submit a LifestyleQuestionnaire from the app** with `is_from_app=1`
2. **Check logs immediately** for the flow described above
3. **Look for any ERROR or WARNING messages**
4. **Share the log output** with specific user ID for analysis

## Quick Test Command

To test if a user can have a diet generated manually:
```python
from panel.services.diet_generation import generate_complete_diet_for_user
success, diet_id, error = generate_complete_diet_for_user(USER_ID)
print(f"Success: {success}, Diet ID: {diet_id}, Error: {error}")
```

## Next Steps

1. Deploy these changes to the live server
2. Submit a test questionnaire from the app
3. Check the logs for the messages above
4. Share the log output to identify the exact issue

