#!/usr/bin/env python
import os
import sys
import django
import requests
import pandas as pd
import ast
import json
import logging
from datetime import datetime
from tqdm import tqdm

from django.db import transaction

# ---------------------------------------------------------
# 1. Add Project Directory to PYTHONPATH
#    (Adjust this path to your project's root folder)
# ---------------------------------------------------------
sys.path.append('/Users/parsa/Lacto/ziluck/')  # <-- Adjust as needed

# ---------------------------------------------------------
# 2. Setup Django Environment
#    (Replace with your actual settings module)
# ---------------------------------------------------------
os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'ziluck_project.settings')
django.setup()

# ---------------------------------------------------------
# 3. Imports from your Django app
#    (Adjust to the actual location of your DietTemplate model)
# ---------------------------------------------------------
from APIs.models import DietTemplate

# ---------------------------------------------------------
# 4. Configure Logging
# ---------------------------------------------------------
logging.basicConfig(
    filename=os.path.join(os.path.dirname(__file__), '../../processing_debug.log'),
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s'
)

# ---------------------------------------------------------
# 5. Helper Functions
# ---------------------------------------------------------

def download_file(url, local_path):
    """
    Downloads a file from a URL to a local path with a progress bar.
    """
    response = requests.get(url, stream=True)
    response.raise_for_status()  # Raise if there's an HTTP error

    total_size = int(response.headers.get('content-length', 0))
    block_size = 1024  # 1 Kibibyte

    with open(local_path, 'wb') as file, tqdm(
        total=total_size, unit='iB', unit_scale=True, desc="Downloading Excel file"
    ) as t:
        for data in response.iter_content(block_size):
            t.update(len(data))
            file.write(data)

def excel_to_bool(value):
    """
    Converts an Excel cell value to a boolean.
    Expects integer-like (e.g., 0 or 1), but can be extended as needed.
    Returns False for any non-1 value.
    """
    try:
        return bool(int(value)) if pd.notna(value) else False
    except (ValueError, TypeError):
        return False

# ---------------------------------------------------------
# 6. Core Processing Function
# ---------------------------------------------------------

def process_excel_file(df):
    """
    Processes the Excel DataFrame to create DietTemplate objects.
    Always creates new templates (no 'update' logic here by default).
    """
    created_count = 0
    skipped_count = 0
    skipped_rows = []

    # Bacteria field name mapping
    BACTERIA_MAP = {
        'Bacteria_Akkermansia': 'akkermansia',
        'Bacteria_Bacteroidetes': 'bacteroides',
        'Bacteria_Bifidobacterium': 'bifidobacterium',
        'Bacteria_Fecalibacterium': 'faecalibacterium',
        'Bacteria_Alistipes': 'alistipes',
        'eubacterium': 'eubacterium',
        'Bacteria_Lactobacillus': 'lactobacillus',
        'Bacteria_Blautia': 'blautia',
        'Bacteria_Acidaminococcus': 'Bacteria_Acidaminococcus',
        'Bacteria_Lawsonibacter_asaccharolyticus': 'Bacteria_Lawsonibacter_asaccharolyticus',
    }

    map_fodmap = {'low': 1, 'variable': 2, 'high': 3}
    map_gl_category = {'low': 1, 'medium': 2, 'high': 3}

    with transaction.atomic():
        for index, row in tqdm(df.iterrows(), total=df.shape[0], desc="Processing rows"):
            try:
                # ------------------------------------------------
                # a) Create a new DietTemplate
                # ------------------------------------------------
                dt = DietTemplate()

                # ------------------------------------------------
                # b) Fill in various fields
                # ------------------------------------------------

                # 1) Basic text / numeric fields
                if 'diet_name' in row and pd.notna(row['diet_name']):
                    dt.diet_name = str(row['diet_name'])

                if 'diet_cal' in row and pd.notna(row['diet_cal']):
                    try:
                        dt.calorie_range = int(row['diet_cal'])
                    except ValueError:
                        dt.calorie_range = 0

                if 'template_id' in row and pd.notna(row['template_id']):
                    try:
                        dt.template_id = int(row['template_id'])
                    except ValueError:
                        dt.template_id = 0

                if 'stewes_lunch' in row and pd.notna(row['stewes_lunch']):
                    try:
                        dt.lunch_Stews = int(row['stewes_lunch'])
                    except ValueError:
                        dt.lunch_Stews = 0

                if 'stews_dinner' in row and pd.notna(row['stews_dinner']):
                    try:
                        dt.dinner_Stews = int(row['stews_dinner'])
                    except ValueError:
                        dt.dinner_Stews = 0

                if 'mixed_rice_lunch' in row and pd.notna(row['mixed_rice_lunch']):
                    try:
                        dt.lunch_Mixed_Rice = int(row['mixed_rice_lunch'])
                    except ValueError:
                        dt.lunch_Mixed_Rice = 0

                if 'mixed_rice_dinner' in row and pd.notna(row['mixed_rice_dinner']):
                    try:
                        dt.dinner_Mixed_Rice = int(row['mixed_rice_dinner'])
                    except ValueError:
                        dt.dinner_Mixed_Rice = 0

                if 'company_name' in row and pd.notna(row['company_name']):
                    dt.company_name = str(row['company_name'])

                if 'city' in row and pd.notna(row['city']):
                    dt.city = str(row['city'])

                if 'is_user_template' in row:
                    dt.is_user_template = excel_to_bool(row['is_user_template'])

                # 2) JSON data
                if 'json_data' in row and pd.notna(row['json_data']):
                    try:
                        diet_data = ast.literal_eval(str(row['json_data']))
                        dt.diet_json = diet_data
                        # Example: override 'diet_name' if "type" key is present
                        if 'type' in diet_data:
                            dt.diet_name = diet_data['type']
                    except (ValueError, SyntaxError, json.JSONDecodeError):
                        dt.diet_json = {}

                # 3) Bacteria fields
                bacteria_value = row.get('bacteria_type', '')
                if pd.notna(bacteria_value):
                    bacteria_str = str(bacteria_value).strip()
                    if bacteria_str in BACTERIA_MAP:
                        field_name = BACTERIA_MAP[bacteria_str]
                        setattr(dt, field_name, True)
                    else:
                        # Not mapped => skip or log
                        skipped_count += 1
                        skipped_rows.append(f"Row {index}: Unmapped bacteria_type '{bacteria_str}'")

                # 4) Diet booleans
                dt.med = excel_to_bool(row.get('Diet_Mediterranean', False))
                dt.low_gi = excel_to_bool(row.get('low_gi', False))
                dt.fasting = excel_to_bool(row.get('fasting', False))
                dt.keto = excel_to_bool(row.get('Diet_Keto', False))
                dt.mind = excel_to_bool(row.get('mind', False))
                dt.sport = excel_to_bool(row.get('sport', False))
                dt.liquid_or_pureed_diet = excel_to_bool(row.get('liquid_or_pureed_diet', False))
                dt.eliminate_diet = excel_to_bool(row.get('eliminate_diet', False))
                dt.low_fodmap = excel_to_bool(row.get('low_fodmap', False))
                dt.pregnant = excel_to_bool(row.get('pregnant', False))
                dt.lactating = excel_to_bool(row.get('lactating', False))
                dt.Diet_Low_Sugar = excel_to_bool(row.get('Diet_Low_Sugar', False))
                dt.Diet_High_Protein = excel_to_bool(row.get('Diet_High_Protein', False))
                dt.Whole_Foods = excel_to_bool(row.get('Whole_Foods', False))
                dt.Diet_Low_Fat = excel_to_bool(row.get('Diet_Low_Fat', False))
                dt.Diet_Low_Calorie = excel_to_bool(row.get('Diet_Low_Calorie', False))
                dt.Diet_Anti_Inflammatory = excel_to_bool(row.get('Diet_Anti_Inflammatory', False))
                dt.Diet_Paleo = excel_to_bool(row.get('Diet_Paleo', False))
                dt.Diet_Diabetic = excel_to_bool(row.get('Diet_Diabetic', False))
                dt.Diet_Low_Carb = excel_to_bool(row.get('Diet_Low_Carb', False))
                dt.Diet_Bodybuilding = excel_to_bool(row.get('Diet_Bodybuilding', False))
                dt.Diet_Vegan = excel_to_bool(row.get('Diet_Vegan', False))
                dt.Diet_Vegetarian = excel_to_bool(row.get('Diet_Vegetarian', False))
                dt.Diet_Plant_Based = excel_to_bool(row.get('Diet_Plant_Based', False))
                dt.Diet_Gluten_Free = excel_to_bool(row.get('Diet_Gluten_Free', False))

                # 5) Disease booleans
                dt.fatty_liver = excel_to_bool(row.get('fatty_liver', False))
                dt.diabetes = excel_to_bool(row.get('Disease_Diabetes', False))
                dt.ibs = excel_to_bool(row.get('Disease_IBS', False))
                dt.pcos = excel_to_bool(row.get('pcos', False))
                dt.neghres = excel_to_bool(row.get('Disease_Gout', False))
                dt.ms = excel_to_bool(row.get('ms', False))
                dt.obesity_surgery = excel_to_bool(row.get('obesity_surgery', False))
                dt.renal = excel_to_bool(row.get('renal', False))
                dt.cvd = excel_to_bool(row.get('cvd', False))
                dt.migraine = excel_to_bool(row.get('migraine', False))
                dt.cancer = excel_to_bool(row.get('cancer', False))
                dt.nervous_disease = excel_to_bool(row.get('nervous_disease', False))
                dt.gi_disease = excel_to_bool(row.get('gi_disease', False))
                dt.healthy = excel_to_bool(row.get('healthy', False))
                dt.Disease_Kidney = excel_to_bool(row.get('Disease_Kidney', False))
                dt.Disease_Kidney_stones = excel_to_bool(row.get('Disease_Kidney_stones', False))
                dt.Disease_Acid_reflux = excel_to_bool(row.get('Disease_Acid_reflux', False))
                dt.Disease_Celiac = excel_to_bool(row.get('Disease_Celiac', False))
                dt.Disease_Diverticulitis = excel_to_bool(row.get('Disease_Diverticulitis', False))
                dt.Disease_Hypothyroidism = excel_to_bool(row.get('Disease_Hypothyroidism', False))
                dt.Disease_Gastroesophageal_Reflux = excel_to_bool(row.get('Disease_Gastroesophageal_Reflux', False))
                dt.Disease_Foodborne_Illness = excel_to_bool(row.get('Disease_Foodborne_Illness', False))
                dt.Disease_High_blood_pressure = excel_to_bool(row.get('Disease_High_blood_pressure', False))
                dt.Disease_High_cholesterol = excel_to_bool(row.get('Disease_High_cholesterol', False))
                dt.Disease_Hypercholesterolemia = excel_to_bool(row.get('Disease_Hypercholesterolemia', False))
                dt.Disease_G6PD_Deficiency = excel_to_bool(row.get('Disease_G6PD_Deficiency', False))
                dt.Disease_Constipation = excel_to_bool(row.get('Disease_Constipation', False))
                dt.Disease_Gallbladder = excel_to_bool(row.get('Disease_Gallbladder', False))
                dt.Disease_Latex_Fruit_Syndrome = excel_to_bool(row.get('Disease_Latex_Fruit_Syndrome', False))
                dt.Disease_Obesity = excel_to_bool(row.get('Disease_Obesity', False))

                # 6) Various booleans
                dt.cooked_rice = excel_to_bool(row.get('cooked_rice', False))
                dt.main_dish = excel_to_bool(row.get('main_dish', False))
                dt.Allergens_Peanut = excel_to_bool(row.get('Allergens_Peanut', False))
                dt.Allergens_Wheat = excel_to_bool(row.get('Allergens_Wheat', False))
                dt.Allergens_Shrimp = excel_to_bool(row.get('Allergens_Shrimp', False))
                dt.Allergens_Mushroom = excel_to_bool(row.get('Allergens_Mushroom', False))
                dt.Allergens_Eggplant = excel_to_bool(row.get('Allergens_Eggplant', False))
                dt.Allergens_Dried_fruits = excel_to_bool(row.get('Allergens_Dried_fruits', False))
                dt.Allergens_Gluten = excel_to_bool(row.get('Allergens_Gluten', False))
                dt.Allergens_Fish = excel_to_bool(row.get('Allergens_Fish', False))
                dt.Allergens_Sesame = excel_to_bool(row.get('Allergens_Sesame', False))
                dt.Allergens_Shellfish = excel_to_bool(row.get('Allergens_Shellfish', False))
                dt.Allergens_Nuts = excel_to_bool(row.get('Allergens_nuts', False))
                dt.Allergens_Soy = excel_to_bool(row.get('Allergens_Soy', False))
                dt.Allergens_Eggs = excel_to_bool(row.get('Allergens_Eggs', False))
                dt.Allergens_Peanuts = excel_to_bool(row.get('Allergens_Peanuts', False))
                dt.Allergens_Dairy = excel_to_bool(row.get('Allergens_Dairy', False))
                dt.Allergens_Legumes = excel_to_bool(row.get('Allergens_Legumes', False))
                dt.Allergens_Depends_on_filling_ingredients = excel_to_bool(
                    row.get('Allergens_Depends_on_filling_ingredients', False)
                )
                dt.Allergens_Milk = excel_to_bool(row.get('Allergens_Milk', False))
                dt.Dessert = excel_to_bool(row.get('Dessert', False))
                dt.Added_Meat = excel_to_bool(row.get('Added_Meat', False))
                dt.Ing_Main = excel_to_bool(row.get('Ing_Main', False))
                dt.Beverage = excel_to_bool(row.get('Beverage', False))
                dt.Sandwich = excel_to_bool(row.get('Sandwich', False))
                dt.Condiment = excel_to_bool(row.get('Condiment', False))
                dt.Lunch = excel_to_bool(row.get('Lunch', False))
                dt.Snack = excel_to_bool(row.get('Snack', False))
                dt.Bread = excel_to_bool(row.get('Bread', False))
                dt.Entree = excel_to_bool(row.get('Entree', False))
                dt.Soup = excel_to_bool(row.get('Soup', False))
                dt.Side_Dish = excel_to_bool(row.get('Side_Dish', False))
                dt.Appetizer = excel_to_bool(row.get('Appetizer', False))
                dt.Dinner = excel_to_bool(row.get('Dinner', False))
                dt.Pasta = excel_to_bool(row.get('Pasta', False))
                dt.Breakfast = excel_to_bool(row.get('Breakfast', False))
                dt.Salad = excel_to_bool(row.get('Salad', False))
                dt.Ingredient = excel_to_bool(row.get('Ingredient', False))
                dt.Vegan = excel_to_bool(row.get('Vegan', False))
                dt.Umami = excel_to_bool(row.get('Umami', False))
                dt.Slow_cooked = excel_to_bool(row.get('Slow_cooked', False))
                dt.Spicy = excel_to_bool(row.get('Spicy', False))
                dt.Baked = excel_to_bool(row.get('Baked', False))
                dt.Pickled = excel_to_bool(row.get('Pickled', False))
                dt.Low_calorie = excel_to_bool(row.get('Low_calorie', False))
                dt.Grilled = excel_to_bool(row.get('Grilled', False))
                dt.Roasted = excel_to_bool(row.get('Roasted', False))
                dt.High_protein = excel_to_bool(row.get('High_protein', False))
                dt.Fried = excel_to_bool(row.get('Fried', False))
                dt.Sweet = excel_to_bool(row.get('Sweet', False))
                dt.Salty = excel_to_bool(row.get('Salty', False))
                dt.Lactose_free = excel_to_bool(row.get('Lactose_free', False))
                dt.Low_carb = excel_to_bool(row.get('Low_carb', False))
                dt.Cold = excel_to_bool(row.get('Cold', False))
                dt.Vegetarian = excel_to_bool(row.get('Vegetarian', False))
                dt.Sour = excel_to_bool(row.get('Sour', False))
                dt.Sweet_Sour = excel_to_bool(row.get('Sweet_Sour', False))
                dt.Raw = excel_to_bool(row.get('Raw', False))
                dt.Gluten_free = excel_to_bool(row.get('Gluten_free', False))
                dt.Boiled = excel_to_bool(row.get('Boiled', False))
                dt.Steamed = excel_to_bool(row.get('Steamed', False))
                dt.Low_fat = excel_to_bool(row.get('Low_fat', False))
                dt.Prepared_Meals = excel_to_bool(row.get('Prepared_Meals', False))
                dt.Meats = excel_to_bool(row.get('Meats', False))
                dt.Sweets = excel_to_bool(row.get('Sweets', False))
                dt.Beans_and_Lentils = excel_to_bool(row.get('Beans_and_Lentils', False))
                dt.Snacks = excel_to_bool(row.get('Snacks', False))
                dt.Grains_and_Pasta = excel_to_bool(row.get('Grains_and_Pasta', False))
                dt.Nuts_and_Seeds = excel_to_bool(row.get('Nuts_and_Seeds', False))
                dt.Fruits = excel_to_bool(row.get('Fruits', False))
                dt.Beverages = excel_to_bool(row.get('Beverages', False))
                dt.Fish = excel_to_bool(row.get('Fish', False))
                dt.Baked_Foods = excel_to_bool(row.get('Baked_Foods', False))
                dt.Breakfast_Cereals = excel_to_bool(row.get('Breakfast_Cereals', False))
                dt.Vegetables = excel_to_bool(row.get('Vegetables', False))
                dt.Spices_and_Herbs = excel_to_bool(row.get('Spices_and_Herbs', False))
                dt.Soups_and_Sauces = excel_to_bool(row.get('Soups_and_Sauces', False))
                dt.Dairy_and_Egg_Products = excel_to_bool(row.get('Dairy_and_Egg_Products', False))
                dt.Fats_and_Oils = excel_to_bool(row.get('Fats_and_Oils', False))
                dt.Lactose_Intolerance = excel_to_bool(row.get('Lactose_Intolerance', False))
                dt.stones = excel_to_bool(row.get('stones', False))
                dt.reflux = excel_to_bool(row.get('reflux', False))
                dt.Spices = excel_to_bool(row.get('Spices', False))
                dt.Seafood = excel_to_bool(row.get('Seafood', False))
                dt.Poultry = excel_to_bool(row.get('Poultry', False))
                dt.Herbs = excel_to_bool(row.get('Herbs', False))
                dt.Processed_Meats = excel_to_bool(row.get('Processed_Meats', False))
                dt.Legumes = excel_to_bool(row.get('Legumes', False))
                dt.Dairy = excel_to_bool(row.get('Dairy', False))
                dt.Nuts = excel_to_bool(row.get('Nuts', False))
                dt.Mixed_Dishes = excel_to_bool(row.get('Mixed_Dishes', False))
                dt.Sauces = excel_to_bool(row.get('Sauces', False))
                dt.Oils = excel_to_bool(row.get('Oils', False))
                dt.Grains = excel_to_bool(row.get('Grains', False))
                dt.Seeds = excel_to_bool(row.get('Seeds', False))
                dt.Pastries = excel_to_bool(row.get('Pastries', False))
                dt.Spreads = excel_to_bool(row.get('Spreads', False))
                dt.Eggs = excel_to_bool(row.get('Eggs', False))
                dt.Prebiotic = excel_to_bool(row.get('Prebiotic', False))
                dt.Probiotic = excel_to_bool(row.get('Probiotic', False))
                dt.Low_Inflammatory = excel_to_bool(row.get('Low_Inflammatory', False))
                dt.Diverse = excel_to_bool(row.get('Diverse', False))
                dt.Polyphenol_Rich = excel_to_bool(row.get('Polyphenol_Rich', False))
                dt.Stews = excel_to_bool(row.get('Stews', False))
                dt.Mixed_Rice = excel_to_bool(row.get('Mixed_Rice', False))
                dt.Soups_and_Aashes = excel_to_bool(row.get('Soups_and_Aashes', False))
                dt.Traditional_Foods = excel_to_bool(row.get('Traditional_Foods', False))
                dt.International_Foods = excel_to_bool(row.get('International_Foods', False))
                dt.Vegetarian_Foods = excel_to_bool(row.get('Vegetarian_Foods', False))
                dt.Simple_Dishes = excel_to_bool(row.get('Simple_Dishes', False))
                dt.Main_Salads = excel_to_bool(row.get('Main_Salads', False))
                dt.Fast_Foods = excel_to_bool(row.get('Fast_Foods', False))

                # FODMAP classification
                if 'FODMAP_Classification' in row and pd.notna(row['FODMAP_Classification']):
                    fodmap_key = str(row['FODMAP_Classification']).strip().lower()
                    dt.FODMAP_Classification = map_fodmap.get(fodmap_key, 0)

                # GL and GI
                if 'GL_category' in row and pd.notna(row['GL_category']):
                    gl_cat_key = str(row['GL_category']).strip().lower()
                    dt.GL_category = map_gl_category.get(gl_cat_key, 0)

                if 'GL_value' in row and pd.notna(row['GL_value']):
                    try:
                        dt.GL_value = int(row['GL_value'])
                    except (ValueError, TypeError):
                        dt.GL_value = 0

                if 'GI_category' in row and pd.notna(row['GI_category']):
                    gi_cat_key = str(row['GI_category']).strip().lower()
                    dt.GI_category = map_gl_category.get(gi_cat_key, 0)

                if 'GI_value' in row and pd.notna(row['GI_value']):
                    try:
                        dt.GI_value = int(row['GI_value'])
                    except (ValueError, TypeError):
                        dt.GI_value = 0

                # Example of integer fields for lunch/dinner categories
                for fld in [
                    'lunch_Grilled', 'lunch_Soups_and_Aashes',
                    'lunch_Pasta', 'lunch_Seafood', 'lunch_Traditional_Foods',
                    'lunch_International_Foods', 'lunch_Vegetarian_Foods', 'lunch_Simple_Dishes',
                    'lunch_Main_Salads', 'lunch_Fast_Foods',
                    'dinner_Grilled', 'dinner_Soups_and_Aashes', 'dinner_Pasta', 'dinner_Seafood',
                    'dinner_Traditional_Foods', 'dinner_International_Foods',
                    'dinner_Vegetarian_Foods', 'dinner_Simple_Dishes', 'dinner_Main_Salads',
                    'dinner_Fast_Foods'
                ]:
                    if fld in row and pd.notna(row[fld]):
                        try:
                            setattr(dt, fld, int(row[fld]))
                        except ValueError:
                            setattr(dt, fld, 0)

                # ------------------------------------------------
                # c) Save the object
                # ------------------------------------------------
                dt.save()
                created_count += 1

            except Exception as ex:
                # If any unexpected error occurs for this row, log and skip
                logging.error(f"Error processing row {index}: {ex}")
                skipped_count += 1
                skipped_rows.append(f"Row {index}: {ex}")

    # Return summary
    return {
        "created_count": created_count,
        "skipped_count": skipped_count,
        "skipped_rows": skipped_rows
    }

# ---------------------------------------------------------
# 7. Main Execution Flow
# ---------------------------------------------------------

def main():
    """
    Main entry point for the script.
    1) Optionally download the Excel from a URL
    2) Read into a DataFrame
    3) Process and create DietTemplate objects
    4) Cleanup
    """
    # -----------------------------------------------------
    # (A) Define the URL of the Excel file
    #     or comment out if you prefer local-only approach
    # -----------------------------------------------------
    excel_url = "https://lacto.ir/updated_first_dataset_stews.xlsx"
    local_excel_path = "downloaded_updated_first_dataset_stews.xlsx"

    # (B) Download file from URL
    print(f"Starting download from {excel_url}...")
    try:
        download_file(excel_url, local_excel_path)
        print(f"Downloaded Excel file to {local_excel_path}")
        logging.info(f"Downloaded Excel file from {excel_url} to {local_excel_path}")
    except Exception as e:
        logging.error(f"Failed to download the Excel file: {e}")
        print(f"Failed to download the Excel file: {e}")
        return

    # (C) Read Excel into a DataFrame
    try:
        print("Loading and processing dataset...")
        df = pd.read_excel(local_excel_path)
    except Exception as e:
        logging.error(f"Failed to read the Excel file: {e}")
        print(f"Failed to read the Excel file: {e}")
        return

    # (D) Process the DataFrame
    try:
        result = process_excel_file(df)
        print("DietTemplate objects created successfully.")
        logging.info("DietTemplate objects created successfully.")
        print(f"Created: {result['created_count']}, Skipped: {result['skipped_count']}")
        if result['skipped_rows']:
            print("Skipped details:")
            for skip_info in result['skipped_rows']:
                print(f" - {skip_info}")
    except Exception as e:
        logging.error(f"Failed to process the Excel file: {e}")
        print(f"Failed to process the Excel file: {e}")

    # (E) Optionally remove the downloaded file
    try:
        os.remove(local_excel_path)
        print(f"Removed temporary file {local_excel_path}")
        logging.info(f"Removed temporary file {local_excel_path}")
    except OSError as e:
        logging.warning(f"Failed to remove temporary file {local_excel_path}: {e}")
        print(f"Failed to remove temporary file {local_excel_path}: {e}")

# ---------------------------------------------------------
# 8. Entry Point
# ---------------------------------------------------------
if __name__ == "__main__":
    main()
