import platform

def init():
    server = platform.version()
    print("running in", server)

    if "#52-Ubuntu" in server:
        import os
        import django
        os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'ziluck_project.settings')
        django.setup()
    else:
        import os
        import sys
        import django
        sys.path.append('/home/idiair/.ziluck310')
        os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'ziluck_project.settings')
        django.setup()

init()



# update_foodcards.py

import os
import django
import requests
import pandas as pd
import ast
import json
import logging
from datetime import datetime
from tqdm import tqdm
import sys

# --------------------------------------------
# 1. Add Project Directory to PYTHONPATH
# --------------------------------------------

# Adjust the path to point to your project's root directory
# Ensure that this path is correct based on your project structure
sys.path.append('/Users/parsa/Lacto/ziluck/')

# --------------------------------------------
# 2. Setup Django Environment
# --------------------------------------------

# Replace 'ziluck_project.settings' with your actual Django settings module if different
os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'ziluck_project.settings')
django.setup()

from APIs.models import FoodCard  # Ensure 'APIs' is your correct app name
from django.db import transaction

# --------------------------------------------
# 3. Configure Logging
# --------------------------------------------

logging.basicConfig(
    filename=os.path.join(os.path.dirname(__file__), '../../processing_debug.log'),
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s'
)

# --------------------------------------------
# 4. Define Helper Functions
# --------------------------------------------

def download_file(url, local_path):
    """
    Downloads a file from a URL to a local path with a progress bar.
    """
    response = requests.get(url, stream=True)
    response.raise_for_status()  # Ensure we notice bad responses

    total_size = int(response.headers.get('content-length', 0))
    block_size = 1024  # 1 Kibibyte

    with open(local_path, 'wb') as file, tqdm(
        total=total_size, unit='iB', unit_scale=True, desc="Downloading Excel file"
    ) as t:
        for data in response.iter_content(block_size):
            t.update(len(data))
            file.write(data)

def excel_to_bool(value):
    """
    Converts Excel cell value to boolean.
    Expects '0' or '1'. Returns False for any non-1 value.
    """
    try:
        return bool(int(value)) if pd.notna(value) else False
    except (ValueError, TypeError):
        return False

# --------------------------------------------
# 5. Define the Processing Function
# --------------------------------------------

def process_excel_file(df):
    """
    Processes the Excel DataFrame to update/create FoodCard objects.
    For new FoodCards, all fields including 'foods' are set.
    For existing FoodCards, only tag-related fields are updated; 'foods' remains unchanged.
    """
    # Setup counters
    updated_count = 0
    created_count = 0
    skipped_count = 0
    skipped_rows = []

    # Define mappings
    map_fodmap = {
        'low': 1,
        'variable': 2,
        'high': 3
    }
    map_gl_category = {
        'low': 1,
        'medium': 2,
        'high': 3
    }

    # Define field sets
    tag_fields = {
        'is_keto': 'Diet_Keto',
        'is_med': 'Diet_Mediterranean',
        'is_conditional_med': 'is_conditional_med',
        'is_lowcarb': 'Diet_Low_Carb',
        'is_mediumcarb': 'is_mediumcarb',
        'is_highcarb': 'is_highcarb',
        'is_highprotein': 'Diet_High_Protein',
        'is_lowprotein': 'is_lowprotein',
        'is_mediumprotein': 'is_mediumprotein',
        'is_mind': 'is_mind',
        'is_low_fodmap': 'is_low_fodmap',
        'is_lowfat': 'Diet_Low_Fat',
        'is_mediumfat': 'is_mediumfat',
        'is_highfat': 'is_highfat',
        'is_breakfast': 'is_breakfast',
        'is_lunch': 'is_lunch',
        'is_dinner': 'is_dinner',
        'is_snack': 'is_snack',
        # Add other tag fields as needed
    }

    additional_boolean_fields = [
        'Dessert', 'Added_Meat', 'Ing_Main', 'Beverage', 'Sandwich',
        'Condiment', 'Snack', 'Bread', 'Entree', 'Soup', 'Side_Dish',
        'Appetizer', 'Dinner', 'Pasta', 'Breakfast', 'Salad',
        'Ingredient', 'Vegan', 'Umami', 'Slow_cooked', 'Spicy',
        'Baked', 'Pickled', 'Low_calorie', 'Grilled', 'Roasted',
        'High_protein', 'Fried', 'Salty', 'Lactose_free',
        'Low_carb', 'Cold', 'Vegetarian', 'Sour', 'Sweet_Sour',
        'Raw', 'Gluten_free', 'Boiled', 'Steamed',
        'Prepared_Meals', 'Meats', 'Sweets', 'Beans_and_Lentils',
        'Snacks', 'Grains_and_Pasta', 'Nuts_and_Seeds', 'Fruits',
        'Beverages', 'Fish', 'Baked_Foods',
        'Breakfast_Cereals', 'Vegetables', 'Spices_and_Herbs',
        'Soups_and_Sauces', 'Dairy_and_Egg_Products',
        'Fats_and_Oils', 'Lactose_Intolerance',
        'stones', 'reflux', 'cooked_rice', 'main_dish',
        'Spices', 'Seafood', 'Poultry', 'Herbs',
        'Processed_Meats', 'Legumes', 'Dairy', 'Nuts',
        'Mixed_Dishes', 'Sauces', 'Oils', 'Grains',
        'Seeds', 'Pastries', 'Spreads', 'Eggs',
        'Prebiotic', 'Probiotic', 'Low_Inflammatory',
        'Diverse', 'Polyphenol_Rich', 'Stews',
        'Mixed_Rice', 'Soups_and_Aashes', 'Traditional_Foods',
        'International_Foods', 'Vegetarian_Foods',
        'Simple_Dishes', 'Main_Salads', 'Fast_Foods',
        # Add more as needed
    ]

    # JSON fields to skip updating for existing FoodCards
    json_fields = ['foods']

    # Start atomic transaction
    with transaction.atomic():
        for index, row in tqdm(df.iterrows(), total=df.shape[0], desc="Processing rows"):
            # -----------------------------
            # a) Get the 'cart_code_final'
            # -----------------------------
            cart_code = row.get('cart_code_final', None)
            if pd.isna(cart_code):
                skipped_count += 1
                skipped_rows.append(f"Row {index} has no cart_code_final")
                continue

            # Attempt to convert to integer (if it's numeric)
            try:
                cart_code = int(cart_code)
            except (ValueError, TypeError):
                skipped_count += 1
                skipped_rows.append(f"Row {index}: Invalid cart_code_final={cart_code}")
                continue

            # -----------------------------
            # b) Get or create FoodCard by PK
            # -----------------------------
            try:
                fc = FoodCard.objects.get(pk=cart_code)
                is_new = False
            except FoodCard.DoesNotExist:
                # Create a brand-new FoodCard with that ID
                fc = FoodCard(id=cart_code)
                is_new = True

            # ------------------------------------------------------------------
            # c) Set fields based on whether the FoodCard is new or existing
            # ------------------------------------------------------------------

            # Update boolean fields
            for attr, excel_col in tag_fields.items():
                setattr(fc, attr, excel_to_bool(row.get(excel_col, 0)))

            # Update additional boolean fields
            for field in additional_boolean_fields:
                setattr(fc, field, excel_to_bool(row.get(field, 0)))

            # Update Allergens
            allergens_fields = [
                'Allergens_Shrimp', 'Allergens_Mushroom', 'Allergens_Eggplant',
                'Allergens_Dried_fruits', 'Allergens_Gluten', 'Allergens_Fish',
                'Allergens_Sesame', 'Allergens_Shellfish', 'Allergens_Nuts',
                'Allergens_Soy', 'Allergens_Eggs', 'Allergens_Peanuts',
                'Allergens_Dairy', 'Allergens_Legumes',
                'Allergens_Depends_on_filling_ingredients', 'Allergens_Milk'
            ]

            for field in allergens_fields:
                setattr(fc, field, excel_to_bool(row.get(field, 0)))

            # Update Diets
            diets_fields = [
                'Diet_Low_Sugar', 'Whole_Foods', 'Diet_Low_Calorie',
                'Diet_Anti_Inflammatory', 'Diet_Paleo', 'Diet_Diabetic',
                'Diet_Bodybuilding', 'Diet_Vegan', 'Diet_Vegetarian',
                'Diet_Plant_Based', 'Diet_Gluten_Free'
            ]

            for field in diets_fields:
                setattr(fc, field, excel_to_bool(row.get(field, 0)))

            # Update Diseases
            diseases_fields = [
                'Disease_Kidney', 'Disease_Kidney_stones', 'Disease_Acid_reflux',
                'Disease_Gout', 'Disease_Celiac', 'Disease_IBS',
                'Disease_Diverticulitis', 'Disease_Hypothyroidism',
                'Disease_Gastroesophageal_Reflux', 'Disease_Foodborne_Illness',
                'Disease_High_blood_pressure', 'Disease_High_cholesterol',
                'Disease_Diabetes', 'Disease_Hypercholesterolemia',
                'Disease_G6PD_Deficiency', 'Disease_Constipation',
                'Disease_Gallbladder', 'Disease_Latex_Fruit_Syndrome',
                'Disease_Obesity'
            ]

            for field in diseases_fields:
                setattr(fc, field, excel_to_bool(row.get(field, 0)))

            # Update Bacteria fields
            bacteria_fields = [
                'Bacteria_Acidaminococcus', 'Bacteria_Bacteroidetes',
                'Bacteria_Alistipes', 'Bacteria_Bifidobacterium',
                'Bacteria_Akkermansia', 'Bacteria_Blautia',
                'Bacteria_Fecalibacterium',
                'Bacteria_Lawsonibacter_asaccharolyticus',
                'Bacteria_Lactobacillus'
            ]

            for field in bacteria_fields:
                setattr(fc, field, excel_to_bool(row.get(field, 0)))

            # Update FODMAP_Classification
            if 'FODMAP_Classification' in row and not pd.isna(row['FODMAP_Classification']):
                fodmap_key = str(row['FODMAP_Classification']).strip().lower()
                fc.FODMAP_Classification = map_fodmap.get(fodmap_key, 0)  # default 0 if not found

            # Update GL_category and GI_category
            if 'GL_category' in row and not pd.isna(row['GL_category']):
                gl_cat_key = str(row['GL_category']).strip().lower()
                fc.GL_category = map_gl_category.get(gl_cat_key, 0)

            if 'GL_value' in row and not pd.isna(row['GL_value']):
                try:
                    fc.GL_value = int(row['GL_value'])
                except (ValueError, TypeError):
                    fc.GL_value = 0

            if 'GI_category' in row and not pd.isna(row['GI_category']):
                gi_cat_key = str(row['GI_category']).strip().lower()
                fc.GI_category = map_gl_category.get(gi_cat_key, 0)

            if 'GI_value' in row and not pd.isna(row['GI_value']):
                try:
                    fc.GI_value = int(row['GI_value'])
                except (ValueError, TypeError):
                    fc.GI_value = 0

            # Update text fields
            text_fields = ['FA_Name', 'EN_Name', 'bread_rice_tag']
            for field in text_fields:
                if field in row and not pd.isna(row[field]):
                    setattr(fc, field, str(row[field]))

            # Update numeric fields
            numeric_fields = ['Calories', 'carb_ratio', 'fat_ratio', 'protein_ratio']
            for field in numeric_fields:
                if field in row and not pd.isna(row[field]):
                    try:
                        setattr(fc, field, float(row[field]))
                    except (ValueError, TypeError):
                        setattr(fc, field, 0)

            # --------------------------------------------
            # **Modify: Update 'foods' only if FoodCard is new**
            # --------------------------------------------
            if is_new:
                # Update foods (assuming 'data_json' is a JSON string)
                if 'data_json' in row and not pd.isna(row['data_json']):
                    try:
                        fc.foods = ast.literal_eval(row['data_json'])
                    except (ValueError, SyntaxError):
                        fc.foods = []
                else:
                    fc.foods = []
            # --------------------------------------------

            # Save the FoodCard object
            fc.save()

            # Update counters
            if is_new:
                created_count += 1
            else:
                updated_count += 1

# --------------------------------------------
# 6. Main Execution Flow
# --------------------------------------------

def main():
    # Define the URL of the Excel file
    excel_url = "https://lacto.ir/Last_Final_updated_valid_cart_json.xlsx"  # Your file URL

    # Define the local path to save the downloaded Excel file
    local_excel_path = "downloaded_excel.xlsx"

    print(f"Starting download from {excel_url}...")
    try:
        download_file(excel_url, local_excel_path)
        print(f"Downloaded Excel file to {local_excel_path}")
        logging.info(f"Downloaded Excel file from {excel_url} to {local_excel_path}")
    except Exception as e:
        logging.error(f"Failed to download the Excel file: {e}")
        print(f"Failed to download the Excel file: {e}")
        return

    # Read the Excel file into a DataFrame
    try:
        print("Loading and processing datasets...")
        df = pd.read_excel(local_excel_path)
    except Exception as e:
        logging.error(f"Failed to read the Excel file: {e}")
        print(f"Failed to read the Excel file: {e}")
        return

    # Process the DataFrame
    try:
        process_excel_file(df)
        print("FoodCard objects updated successfully.")
        logging.info("FoodCard objects updated successfully.")
    except Exception as e:
        logging.error(f"Failed to process the Excel file: {e}")
        print(f"Failed to process the Excel file: {e}")
    finally:
        # Optionally, remove the downloaded file after processing
        try:
            os.remove(local_excel_path)
            print(f"Removed temporary file {local_excel_path}")
            logging.info(f"Removed temporary file {local_excel_path}")
        except OSError as e:
            logging.warning(f"Failed to remove temporary file {local_excel_path}: {e}")
            print(f"Failed to remove temporary file {local_excel_path}: {e}")

if __name__ == "__main__":
    main()
